//***********************************************************************************************
//Impromptu Modular: Modules for VCV Rack by Marc Boulé
//
//This is code from the Fundamental plugin by Andrew Belt 
//See ./LICENSE.md for all licenses
//***********************************************************************************************


#include "FundamentalUtil.hpp"


// From Fundamental VCF.cpp

inline float clip(float x) {
	return tanhf(x);
};

void LadderFilter::process(float input, float dt) {
	dsp::stepRK4(0.f, dt, state, 4, [&](float t, const float x[], float dxdt[]) {
		float inputc = clip(input - resonance * x[3]);
		float yc0 = clip(x[0]);
		float yc1 = clip(x[1]);
		float yc2 = clip(x[2]);
		float yc3 = clip(x[3]);

		dxdt[0] = omega0 * (inputc - yc0);
		dxdt[1] = omega0 * (yc0 - yc1);
		dxdt[2] = omega0 * (yc1 - yc2);
		dxdt[3] = omega0 * (yc2 - yc3);
	});

	lowpass = state[3];
	// TODO This is incorrect when `resonance > 0`. Is the math wrong?
	highpass = clip((input - resonance*state[3]) - 4 * state[0] + 6*state[1] - 4*state[2] + state[3]);
};



// From Fundamental VCO.cpp

void VoltageControlledOscillator::setPitch(float pitchKnob, float pitchCv) {
	// Compute frequency
	pitch = pitchKnob;
	if (analog) {
		// Apply pitch slew
		const float pitchSlewAmount = 3.0f;
		pitch += pitchSlew * pitchSlewAmount;
	}
	else {
		// Quantize coarse knob if digital mode
		pitch = std::round(pitch);
	}
	pitch += pitchCv;
	// Note C4
	freq = 261.626f * std::pow(2.0f, pitch / 12.0f);
};

void VoltageControlledOscillator::setPulseWidth(float pulseWidth) {
	const float pwMin = 0.01f;
	pw = clamp(pulseWidth, pwMin, 1.0f - pwMin);
};

void VoltageControlledOscillator::process(float deltaTime, float syncValue) {
	if (analog) {
		// Adjust pitch slew
		if (++pitchSlewIndex > 32) {
			const float pitchSlewTau = 100.0f; // Time constant for leaky integrator in seconds
			pitchSlew += (random::normal() - pitchSlew / pitchSlewTau) * deltaTime;
			pitchSlewIndex = 0;
		}
	}

	// Advance phase
	float deltaPhase = clamp(freq * deltaTime, 1e-6, 0.5f);

	// Detect sync
	int syncIndex = -1; // Index in the oversample loop where sync occurs [0, OVERSAMPLE)
	float syncCrossing = 0.0f; // Offset that sync occurs [0.0f, 1.0f)
	if (syncEnabled) {
		syncValue -= 0.01f;
		if (syncValue > 0.0f && lastSyncValue <= 0.0f) {
			float deltaSync = syncValue - lastSyncValue;
			syncCrossing = 1.0f - syncValue / deltaSync;
			syncCrossing *= OVERSAMPLE;
			syncIndex = (int)syncCrossing;
			syncCrossing -= syncIndex;
		}
		lastSyncValue = syncValue;
	}

	if (syncDirection)
		deltaPhase *= -1.0f;

	sqrFilter.setCutoff(40.0f * deltaTime);

	for (int i = 0; i < OVERSAMPLE; i++) {
		if (syncIndex == i) {
			if (soft) {
				syncDirection = !syncDirection;
				deltaPhase *= -1.0f;
			}
			else {
				// phase = syncCrossing * deltaPhase / OVERSAMPLE;
				phase = 0.0f;
			}
		}

		if (analog) {
			// Quadratic approximation of sine, slightly richer harmonics
			if (phase < 0.5f)
				sinBuffer[i] = 1.f - 16.f * std::pow(phase - 0.25f, 2);
			else
				sinBuffer[i] = -1.f + 16.f * std::pow(phase - 0.75f, 2);
			sinBuffer[i] *= 1.08f;
		}
		else {
			sinBuffer[i] = std::sin(2.f*float(M_PI) * phase);
		}
		if (analog) {
			triBuffer[i] = 1.25f * interpolateLinear(triTable, phase * 2047.f);
		}
		else {
			if (phase < 0.25f)
				triBuffer[i] = 4.f * phase;
			else if (phase < 0.75f)
				triBuffer[i] = 2.f - 4.f * phase;
			else
				triBuffer[i] = -4.f + 4.f * phase;
		}
		if (analog) {
			sawBuffer[i] = 1.66f * interpolateLinear(sawTable, phase * 2047.f);
		}
		else {
			if (phase < 0.5f)
				sawBuffer[i] = 2.f * phase;
			else
				sawBuffer[i] = -2.f + 2.f * phase;
		}
		sqrBuffer[i] = (phase < pw) ? 1.f : -1.f;
		if (analog) {
			// Simply filter here
			sqrFilter.process(sqrBuffer[i]);
			sqrBuffer[i] = 0.71f * sqrFilter.highpass();
		}

		// Advance phase
		phase += deltaPhase / OVERSAMPLE;
		phase = eucMod(phase, 1.0f);
	}
};

	
	
// From Fundamental VCO.cpp

float sawTable[2048] = {
	0.00221683, 0.00288535, 0.00382874, 0.00493397, 0.006088, 0.00717778, 0.0080962, 0.00887097, 0.00957292, 0.0102536, 0.0109644, 0.0117569, 0.0126765, 0.0137091, 0.014811, 0.0159388, 0.0170492, 0.0180987, 0.0190781, 0.0200243, 0.0209441, 0.0218439, 0.0227299, 0.023608, 0.0244664, 0.0253042, 0.0261303, 0.0269533, 0.0277823, 0.0286234, 0.0294678, 0.0303142, 0.0311635, 0.0320168, 0.0328749, 0.0337369, 0.0346001, 0.0354672, 0.0363411, 0.0372246, 0.0381206, 0.0390346, 0.039964, 0.0409017, 0.0418407, 0.042774, 0.0436961, 0.0446189, 0.0455409, 0.0464563, 0.047359, 0.0482433, 0.0491029, 0.049938, 0.0507563, 0.0515658, 0.0523744, 0.0531901, 0.0540067, 0.0548165, 0.0556269, 0.0564456, 0.0572799, 0.0581378, 0.0590252, 0.059934, 0.0608529, 0.0617707, 0.0626761, 0.0635623, 0.0644415, 0.0653142, 0.0661789, 0.0670339, 0.0678778, 0.0687058, 0.0695165, 0.070317, 0.0711147, 0.0719168, 0.0727304, 0.0735539, 0.0743828, 0.075215, 0.0760485, 0.0768811, 0.07771, 0.0785292, 0.0793449, 0.0801652, 0.080998, 0.0818513, 0.082737, 0.0836581, 0.0845968, 0.0855346, 0.0864532, 0.0873342, 0.0881769, 0.0889945, 0.0897932, 0.0905797, 0.0913603, 0.0921398, 0.0929034, 0.0936544, 0.0944023, 0.0951566, 0.0959267, 0.0967196, 0.0975297, 0.0983516, 0.09918, 0.10001, 0.100836, 0.101666, 0.102504, 0.103342, 0.104172, 0.104985, 0.105773, 0.106533, 0.107272, 0.107999, 0.108722, 0.109449, 0.110184, 0.110909, 0.11163, 0.112359, 0.113107, 0.113883, 0.114701, 0.115559, 0.116438, 0.117321, 0.118187, 0.119019, 0.11982, 0.120602, 0.121371, 0.122131, 0.122889, 0.123648, 0.124405, 0.125157, 0.125905, 0.126649, 0.127387, 0.128116, 0.128828, 0.129533, 0.130244, 0.130971, 0.131727, 0.132522, 0.133348, 0.134189, 0.13503, 0.135856, 0.136654, 0.137426, 0.138182, 0.13893, 0.13968, 0.140439, 0.141215, 0.142006, 0.142805, 0.143602, 0.144389, 0.145159, 0.145905, 0.146633, 0.147351, 0.148066, 0.148785, 0.149516, 0.150256, 0.151, 0.151747, 0.152495, 0.153242, 0.15399, 0.154742, 0.155496, 0.156248, 0.156992, 0.157725, 0.158443, 0.15915, 0.159849, 0.160542, 0.16123, 0.161916, 0.162592, 0.163258, 0.163921, 0.164589, 0.165267, 0.165962, 0.166664, 0.167374, 0.168094, 0.168827, 0.169575, 0.170347, 0.171149, 0.171967, 0.172788, 0.173595, 0.174376, 0.175127, 0.17586, 0.17658, 0.177293, 0.178003, 0.178717, 0.179439, 0.180163, 0.180879, 0.181579, 0.182254, 0.182887, 0.183461, 0.184006, 0.184553, 0.185136, 0.185786, 0.186519, 0.187312, 0.188146, 0.189001, 0.189859, 0.190701, 0.191556, 0.192426, 0.193296, 0.194149, 0.194968, 0.195739, 0.19647, 0.197172, 0.197852, 0.198518, 0.199177, 0.199827, 0.200455, 0.20107, 0.201683, 0.202303, 0.202941, 0.203594, 0.204255, 0.204924, 0.205599, 0.206279, 0.206967, 0.207672, 0.208389, 0.209106, 0.209812, 0.210497, 0.211154, 0.211791, 0.212413, 0.213022, 0.213622, 0.214216, 0.214789, 0.215338, 0.21588, 0.216432, 0.21701, 0.217632, 0.218309, 0.219022, 0.219748, 0.220466, 0.221152, 0.221789, 0.222391, 0.22297, 0.223536, 0.224097, 0.224665, 0.225227, 0.225769, 0.22631, 0.226866, 0.227457, 0.2281, 0.228809, 0.229566, 0.230349, 0.231134, 0.231897, 0.232619, 0.233316, 0.233998, 0.234672, 0.235345, 0.236023, 0.236715, 0.237415, 0.238118, 0.238815, 0.239499, 0.240162, 0.240802, 0.241424, 0.242035, 0.242638, 0.24324, 0.243846, 0.244452, 0.245055, 0.245654, 0.24625, 0.246841, 0.247425, 0.248001, 0.248572, 0.249141, 0.249711, 0.250285, 0.250867, 0.251454, 0.252043, 0.252627, 0.253202, 0.253762, 0.254314, 0.254859, 0.255395, 0.255918, 0.256428, 0.256916, 0.257372, 0.257809, 0.258245, 0.258695, 0.259177, 0.259689, 0.260217, 0.260762, 0.26133, 0.261924, 0.262549, 0.263221, 0.263934, 0.264667, 0.2654, 0.266114, 0.266792, 0.267451, 0.268097, 0.268732, 0.269357, 0.269973, 0.270576, 0.271158, 0.271729, 0.2723, 0.272879, 0.273479, 0.274107, 0.274757, 0.275414, 0.276063, 0.27669, 0.277282, 0.277837, 0.278368, 0.278887, 0.279406, 0.279937, 0.280489, 0.281057, 0.281633, 0.282207, 0.282772, 0.283319, 0.283845, 0.284355, 0.284855, 0.285351, 0.285848, 0.286353, 0.286861, 0.287368, 0.287877, 0.288389, 0.288904, 0.289424, 0.289941, 0.290461, 0.290987, 0.291524, 0.292076, 0.292648, 0.293238, 0.293841, 0.294449, 0.295058, 0.295663, 0.296276, 0.296899, 0.297519, 0.298127, 0.29871, 0.299257, 0.299765, 0.300247, 0.300716, 0.301185, 0.301669, 0.302178, 0.302707, 0.303244, 0.303774, 0.304286, 0.304765, 0.305192, 0.305574, 0.305941, 0.306323, 0.306748, 0.307245, 0.307827, 0.308466, 0.309123, 0.309764, 0.310351, 0.310859, 0.311312, 0.311727, 0.312122, 0.312512, 0.312915, 0.313329, 0.313739, 0.314148, 0.314558, 0.314972, 0.315393, 0.315812, 0.316231, 0.316656, 0.317095, 0.317554, 0.318043, 0.318572, 0.319125, 0.319683, 0.320227, 0.320739, 0.321207, 0.321641, 0.322057, 0.322467, 0.322886, 0.323327, 0.32378, 0.324239, 0.324707, 0.325188, 0.325686, 0.326205, 0.326755, 0.327327, 0.327907, 0.328484, 0.329045, 0.329585, 0.330116, 0.33064, 0.331156, 0.331664, 0.332165, 0.332653, 0.33313, 0.3336, 0.334068, 0.334539, 0.33502, 0.33552, 0.33603, 0.336536, 0.337024, 0.337481, 0.337893, 0.338261, 0.3386, 0.338927, 0.339258, 0.33961, 0.339982, 0.340358, 0.340742, 0.341136, 0.341541, 0.34196, 0.342407, 0.342874, 0.343348, 0.343815, 0.344261, 0.344675, 0.345061, 0.345429, 0.345787, 0.346144, 0.346507, 0.346879, 0.347253, 0.347628, 0.348002, 0.348377, 0.34875, 0.349118, 0.349481, 0.349845, 0.350216, 0.350597, 0.350995, 0.351411, 0.351838, 0.352273, 0.352708, 0.353139, 0.353563, 0.353984, 0.354404, 0.354824, 0.355243, 0.355661, 0.356077, 0.356489, 0.356901, 0.357316, 0.357737, 0.358168, 0.358608, 0.359055, 0.359506, 0.35996, 0.360414, 0.360869, 0.361328, 0.361789, 0.36225, 0.362706, 0.363154, 0.363596, 0.364034, 0.364466, 0.364893, 0.365313, 0.365724, 0.366133, 0.366538, 0.366935, 0.367317, 0.367681, 0.36802, 0.368324, 0.368606, 0.36888, 0.369159, 0.369458, 0.369786, 0.370135, 0.370494, 0.370854, 0.371206, 0.371541, 0.371852, 0.372145, 0.372432, 0.372726, 0.373037, 0.373377, 0.373753, 0.374151, 0.374555, 0.374948, 0.375316, 0.375645, 0.375944, 0.376225, 0.376497, 0.376772, 0.377061, 0.377367, 0.377681, 0.377998, 0.378314, 0.378622, 0.378916, 0.379197, 0.379469, 0.379735, 0.380002, 0.380274, 0.380552, 0.380831, 0.38111, 0.381395, 0.381687, 0.381992, 0.382311, 0.382642, 0.382983, 0.383329, 0.383676, 0.384021, 0.384367, 0.384716, 0.385066, 0.385417, 0.385767, 0.386116, 0.38647, 0.386824, 0.387177, 0.387522, 0.387858, 0.388187, 0.388517, 0.388841, 0.389149, 0.389433, 0.389684, 0.389883, 0.390037, 0.390171, 0.39031, 0.390476, 0.390693, 0.390945, 0.391221, 0.391516, 0.391825, 0.392141, 0.392465, 0.392808, 0.393164, 0.393524, 0.393881, 0.394226, 0.39456, 0.394891, 0.395217, 0.395538, 0.395851, 0.396157, 0.396451, 0.396736, 0.397016, 0.397296, 0.39758, 0.397874, 0.39818, 0.39849, 0.398797, 0.399095, 0.399375, 0.399634, 0.399877, 0.400107, 0.400331, 0.400554, 0.40078, 0.401001, 0.401216, 0.401431, 0.401652, 0.401885, 0.402136, 0.402411, 0.402699, 0.402992, 0.403279, 0.403551, 0.403804, 0.404051, 0.404289, 0.404515, 0.404728, 0.404924, 0.405086, 0.405215, 0.405334, 0.405465, 0.405629, 0.405848, 0.406118, 0.406424, 0.406751, 0.407085, 0.407412, 0.407728, 0.408054, 0.408384, 0.408707, 0.409011, 0.409286, 0.40952, 0.409718, 0.409896, 0.410074, 0.410268, 0.410495, 0.410756, 0.411038, 0.411329, 0.411618, 0.411891, 0.412139, 0.41237, 0.412589, 0.412802, 0.413013, 0.413229, 0.413447, 0.413663, 0.413878, 0.414097, 0.414321, 0.414556, 0.4148, 0.415051, 0.415309, 0.41557, 0.415834, 0.416101, 0.416385, 0.416677, 0.416965, 0.417238, 0.417483, 0.417691, 0.417871, 0.418029, 0.418173, 0.418311, 0.418451, 0.418583, 0.418702, 0.418816, 0.418934, 0.419064, 0.419214, 0.419382, 0.419563, 0.419752, 0.419942, 0.42013, 0.420312, 0.420491, 0.420669, 0.420849, 0.42103, 0.421215, 0.421399, 0.421577, 0.421758, 0.421949, 0.42216, 0.422397, 0.422673, 0.422978, 0.423295, 0.423607, 0.4239, 0.424162, 0.424418, 0.424664, 0.424892, 0.425091, 0.425255, 0.425365, 0.425417, 0.425437, 0.425453, 0.425491, 0.425579, 0.425726, 0.425913, 0.426116, 0.426313, 0.426481, 0.426598, 0.42666, 0.426688, 0.426709, 0.426745, 0.42682, 0.426953, 0.427125, 0.427325, 0.427539, 0.427754, 0.427957, 0.428158, 0.428365, 0.428571, 0.42877, 0.428954, 0.429116, 0.429254, 0.429373, 0.429483, 0.429592, 0.429707, 0.429833, 0.429956, 0.43008, 0.430211, 0.430356, 0.43052, 0.430711, 0.430923, 0.431151, 0.431385, 0.431618, 0.431843, 0.432075, 0.432315, 0.43255, 0.432772, 0.432969, 0.43313, 0.43326, 0.433365, 0.433455, 0.433539, 0.433624, 0.4337, 0.433748, 0.43379, 0.433848, 0.433947, 0.434111, 0.434373, 0.434713, 0.435083, 0.435437, 0.435726, 0.435907, 0.435996, 0.436022, 0.436011, 0.435989, 0.435982, 0.435999, 0.436012, 0.436022, 0.436035, 0.436054, 0.436083, 0.436125, 0.436177, 0.436234, 0.436294, 0.436352, 0.436403, 0.436434, 0.436453, 0.436478, 0.436526, 0.436614, 0.43676, 0.436966, 0.437211, 0.43747, 0.43772, 0.437936, 0.438113, 0.438274, 0.43842, 0.438554, 0.438679, 0.438796, 0.438898, 0.438986, 0.439067, 0.439145, 0.439229, 0.439323, 0.439421, 0.439522, 0.439625, 0.43973, 0.439836, 0.439943, 0.440053, 0.440164, 0.440277, 0.440389, 0.440502, 0.440621, 0.440748, 0.440873, 0.440989, 0.441088, 0.441163, 0.441213, 0.441245, 0.441266, 0.441282, 0.441302, 0.441323, 0.441334, 0.441341, 0.441354, 0.441383, 0.441436, 0.44152, 0.441628, 0.441752, 0.441883, 0.442013, 0.442135, 0.44226, 0.44239, 0.442518, 0.442638, 0.442745, 0.44283, 0.442892, 0.442939, 0.442982, 0.443033, 0.4431, 0.443186, 0.443279, 0.443381, 0.443494, 0.443621, 0.443764, 0.443936, 0.444132, 0.444338, 0.444539, 0.44472, 0.444871, 0.445005, 0.445125, 0.445231, 0.445322, 0.445397, 0.445451, 0.445479, 0.445491, 0.445496, 0.445505, 0.445526, 0.445556, 0.445586, 0.445621, 0.445663, 0.445715, 0.445779, 0.445858, 0.445947, 0.446044, 0.446145, 0.446248, 0.44635, 0.446452, 0.446557, 0.446667, 0.446784, 0.446909, 0.447046, 0.447194, 0.447348, 0.447505, 0.447661, 0.447814, 0.447987, 0.448173, 0.448351, 0.4485, 0.448599, 0.448623, 0.448552, 0.448426, 0.448289, 0.448186, 0.448161, 0.448239, 0.448392, 0.44859, 0.448805, 0.449007, 0.449168, 0.449302, 0.449424, 0.449536, 0.449637, 0.449728, 0.44981, 0.449878, 0.449934, 0.449982, 0.450025, 0.450064, 0.450101, 0.45013, 0.450154, 0.450176, 0.450199, 0.450225, 0.450252, 0.450279, 0.450307, 0.450338, 0.450373, 0.450413, 0.450461, 0.450514, 0.45057, 0.450625, 0.450677, 0.450725, 0.450775, 0.450825, 0.45087, 0.450906, 0.450929, 0.450938, 0.450934, 0.450921, 0.450901, 0.450877, 0.45085, 0.450811, 0.450761, 0.45071, 0.450664, 0.450633, 0.450621, 0.45062, 0.450628, 0.450649, 0.450683, 0.450732, 0.450802, 0.450892, 0.450997, 0.45111, 0.451225, 0.451334, 0.451451, 0.451579, 0.451705, 0.451819, 0.451909, 0.451964, 0.451985, 0.451981, 0.451963, 0.451939, 0.451919, 0.451906, 0.451887, 0.451865, 0.451845, 0.451832, 0.45183, 0.451841, 0.451861, 0.451888, 0.451919, 0.451949, 0.451977, 0.452007, 0.452039, 0.452071, 0.452102, 0.452129, 0.452153, 0.452177, 0.452199, 0.452217, 0.452231, 0.452237, 0.452233, 0.452219, 0.452199, 0.452179, 0.452163, 0.452155, 0.452151, 0.452149, 0.452152, 0.452159, 0.452173, 0.452194, 0.452227, 0.452267, 0.45231, 0.452352, 0.452388, 0.452422, 0.45246, 0.452497, 0.452525, 0.452538, 0.452531, 0.452507, 0.452471, 0.45242, 0.452351, 0.452262, 0.452146, 0.451988, 0.451802, 0.451605, 0.451418, 0.45126, 0.451143, 0.451054, 0.450984, 0.450924, 0.450866, 0.450801, 0.450725, 0.450645, 0.450566, 0.450496, 0.45044, 0.450406, 0.450404, 0.450422, 0.450449, 0.450469, 0.450469, 0.450438, 0.450381, 0.450308, 0.450229, 0.450154, 0.450093, 0.450049, 0.450013, 0.449983, 0.449957, 0.449933, 0.449908, 0.449892, 0.449885, 0.449877, 0.449858, 0.449822, 0.44976, 0.449688, 0.449602, 0.449498, 0.449371, 0.449217, 0.449034, 0.448826, 0.448593, 0.448337, 0.448059, 0.447758, 0.447436, 0.44709, 0.446721, 0.446331, 0.445918, 0.445483, 0.445027, 0.444548, 0.444047, 0.443524, 0.44298, 0.442413, 0.441818, 0.441202, 0.440567, 0.439919, 0.439262, 0.438614, 0.437974, 0.437319, 0.436626, 0.43587, 0.435049, 0.434402, 0.433835, 0.433123, 0.432039, 0.430358, 0.428368, 0.427503, 0.426601, 0.424216, 0.418905, 0.409225, 0.396402, 0.382576, 0.365547, 0.34307, 0.312906, 0.272788, 0.221013, 0.159633, 0.091601, 0.0198945, -0.0525338, -0.124172, -0.201123, -0.28145, -0.361514, -0.437677, -0.506323, -0.565947, -0.61981, -0.668566, -0.712743, -0.752877, -0.789483, -0.821637, -0.849041, -0.872562, -0.893078, -0.911459, -0.928371, -0.94253, -0.953999, -0.963373, -0.971244, -0.978205, -0.984408, -0.988963, -0.992177, -0.994451, -0.996186, -0.997786, -0.999106, -0.999791, -1.0, -0.999892, -0.999627, -0.99935, -0.998924, -0.998293, -0.997489, -0.996544, -0.99549, -0.994324, -0.992938, -0.9914, -0.989797, -0.988217, -0.98675, -0.985459, -0.984295, -0.983177, -0.982026, -0.98076, -0.9793, -0.977576, -0.975661, -0.973667, -0.971707, -0.969891, -0.968331, -0.967043, -0.965911, -0.964815, -0.963632, -0.962241, -0.960532, -0.958548, -0.956417, -0.954273, -0.952245, -0.950466, -0.948995, -0.947739, -0.946587, -0.94543, -0.944156, -0.942662, -0.940966, -0.939147, -0.937271, -0.935407, -0.933622, -0.931964, -0.930389, -0.928864, -0.927356, -0.925834, -0.924266, -0.922637, -0.920969, -0.919284, -0.917603, -0.915947, -0.914338, -0.912774, -0.91124, -0.909724, -0.908215, -0.9067, -0.905174, -0.903657, -0.902144, -0.900623, -0.899085, -0.897519, -0.895907, -0.894246, -0.892562, -0.890886, -0.889245, -0.887668, -0.886183, -0.884763, -0.883367, -0.881952, -0.880476, -0.878893, -0.877185, -0.875402, -0.873605, -0.871856, -0.870213, -0.868734, -0.867403, -0.866151, -0.864909, -0.863607, -0.862175, -0.86057, -0.85884, -0.857056, -0.855289, -0.853607, -0.852081, -0.850725, -0.849477, -0.848268, -0.847031, -0.845696, -0.844208, -0.842597, -0.840914, -0.839205, -0.83752, -0.835907, -0.834385, -0.83292, -0.831484, -0.830055, -0.828605, -0.827111, -0.825575, -0.824013, -0.82244, -0.820868, -0.81931, -0.817777, -0.816267, -0.814769, -0.813273, -0.811771, -0.810251, -0.808705, -0.807134, -0.805552, -0.803974, -0.802414, -0.800886, -0.799393, -0.797924, -0.796471, -0.795025, -0.793575, -0.792118, -0.790676, -0.78924, -0.787798, -0.786332, -0.784828, -0.78326, -0.781618, -0.77994, -0.778267, -0.776639, -0.775098, -0.773676, -0.772344, -0.771056, -0.769763, -0.768417, -0.766973, -0.765427, -0.763815, -0.762179, -0.760556, -0.758986, -0.757505, -0.756108, -0.754755, -0.753405, -0.752018, -0.750554, -0.748967, -0.747275, -0.745539, -0.743822, -0.742186, -0.740694, -0.739378, -0.738183, -0.737041, -0.735882, -0.734639, -0.73325, -0.731736, -0.730147, -0.728531, -0.726936, -0.725409, -0.723977, -0.722606, -0.721272, -0.719951, -0.718621, -0.717259, -0.715868, -0.714464, -0.713052, -0.711638, -0.710226, -0.708824, -0.707431, -0.706042, -0.704652, -0.703255, -0.701847, -0.700422, -0.698983, -0.697535, -0.696081, -0.694629, -0.693182, -0.691736, -0.690286, -0.688838, -0.687396, -0.685967, -0.684557, -0.683176, -0.681815, -0.680459, -0.679094, -0.677703, -0.676271, -0.674794, -0.673291, -0.671787, -0.670306, -0.668872, -0.66751, -0.666213, -0.664942, -0.663664, -0.66234, -0.660935, -0.659418, -0.657823, -0.656203, -0.654607, -0.653087, -0.651693, -0.650422, -0.649228, -0.648061, -0.64687, -0.645608, -0.64424, -0.642798, -0.641314, -0.639821, -0.638352, -0.636938, -0.635586, -0.634273, -0.63298, -0.631688, -0.630378, -0.629033, -0.627661, -0.626273, -0.624876, -0.623477, -0.622085, -0.620705, -0.619334, -0.617966, -0.616596, -0.615218, -0.613827, -0.612416, -0.61099, -0.609555, -0.608122, -0.606697, -0.60529, -0.603898, -0.602516, -0.60114, -0.599766, -0.598392, -0.597014, -0.595638, -0.594262, -0.592886, -0.59151, -0.590132, -0.588747, -0.587351, -0.585953, -0.584565, -0.583196, -0.581857, -0.580561, -0.579296, -0.578045, -0.576789, -0.575509, -0.574187, -0.572823, -0.571433, -0.570038, -0.568655, -0.567304, -0.566, -0.564732, -0.563486, -0.562244, -0.560992, -0.559712, -0.558394, -0.557048, -0.555694, -0.55435, -0.553035, -0.551767, -0.550554, -0.549377, -0.548213, -0.547039, -0.545833, -0.544579, -0.543296, -0.541991, -0.540673, -0.539347, -0.538023, -0.536698, -0.535368, -0.534033, -0.532691, -0.531345, -0.529992, -0.528625, -0.527247, -0.525867, -0.524493, -0.523133, -0.521796, -0.520479, -0.519176, -0.517878, -0.51658, -0.515273, -0.513957, -0.512638, -0.511316, -0.509991, -0.508665, -0.507338, -0.506007, -0.504673, -0.503337, -0.502003, -0.500672, -0.499347, -0.498028, -0.496714, -0.495401, -0.494085, -0.492762, -0.491429, -0.490084, -0.488734, -0.487386, -0.486047, -0.484726, -0.483416, -0.482114, -0.480822, -0.479543, -0.478281, -0.477039, -0.475822, -0.474626, -0.473441, -0.47226, -0.471073, -0.469878, -0.468688, -0.467499, -0.466305, -0.465101, -0.463881, -0.462642, -0.461385, -0.460117, -0.458846, -0.457581, -0.456327, -0.455092, -0.453867, -0.452643, -0.451408, -0.450154, -0.448868, -0.447541, -0.44619, -0.444838, -0.443506, -0.442214, -0.44098, -0.439791, -0.438628, -0.437473, -0.436306, -0.43511, -0.433885, -0.432645, -0.431396, -0.430143, -0.428894, -0.427652, -0.426416, -0.425183, -0.423949, -0.422711, -0.421466, -0.420212, -0.418951, -0.417684, -0.416414, -0.415141, -0.413868, -0.412589, -0.411302, -0.410015, -0.408732, -0.407459, -0.406204, -0.404969, -0.403749, -0.402535, -0.401316, -0.400084, -0.398829, -0.397551, -0.396262, -0.394973, -0.393697, -0.392445, -0.391224, -0.390027, -0.388843, -0.387665, -0.386484, -0.385291, -0.384089, -0.382883, -0.381676, -0.380469, -0.379266, -0.378068, -0.37687, -0.375674, -0.374482, -0.373293, -0.372111, -0.370936, -0.369768, -0.368605, -0.367445, -0.366287, -0.365128, -0.36397, -0.362812, -0.361657, -0.360503, -0.359351, -0.358203, -0.357061, -0.355922, -0.354784, -0.353643, -0.352495, -0.351336, -0.350168, -0.348994, -0.34782, -0.346649, -0.345486, -0.344334, -0.343189, -0.342048, -0.340906, -0.339762, -0.338611, -0.337458, -0.336305, -0.335148, -0.333984, -0.33281, -0.331624, -0.330426, -0.329218, -0.328005, -0.326791, -0.32558, -0.324371, -0.323158, -0.321945, -0.320736, -0.319535, -0.318345, -0.317165, -0.315991, -0.314825, -0.313666, -0.312516, -0.311376, -0.310249, -0.309132, -0.30802, -0.30691, -0.305796, -0.304683, -0.30358, -0.302479, -0.301367, -0.300236, -0.299074, -0.297859, -0.296599, -0.295325, -0.294068, -0.292855, -0.291717, -0.290659, -0.289654, -0.28867, -0.287676, -0.286641, -0.28555, -0.284431, -0.28329, -0.282128, -0.280946, -0.279746, -0.278513, -0.277246, -0.27596, -0.274674, -0.273405, -0.272167, -0.270948, -0.26974, -0.268546, -0.26737, -0.266215, -0.265084, -0.263983, -0.262904, -0.261837, -0.260774, -0.259706, -0.258637, -0.257578, -0.256522, -0.255458, -0.254379, -0.253276, -0.252151, -0.251008, -0.249851, -0.248684, -0.247511, -0.246333, -0.245133, -0.243922, -0.242709, -0.241508, -0.240329, -0.239178, -0.238042, -0.236921, -0.235815, -0.234725, -0.233651, -0.232605, -0.231587, -0.230582, -0.229575, -0.228551, -0.227496, -0.226409, -0.225303, -0.22419, -0.223083, -0.221996, -0.22094, -0.219911, -0.218897, -0.217885, -0.216863, -0.215818, -0.214744, -0.213651, -0.212546, -0.211438, -0.210337, -0.209249, -0.208177, -0.207112, -0.20605, -0.204984, -0.203909, -0.202819, -0.201716, -0.200605, -0.19949, -0.198377, -0.197271, -0.196173, -0.195078, -0.193985, -0.192896, -0.191809, -0.190725, -0.189653, -0.188591, -0.187529, -0.186458, -0.185368, -0.18425, -0.183097, -0.181922, -0.180741, -0.179569, -0.17842, -0.1773, -0.17619, -0.175094, -0.174015, -0.172958, -0.171927, -0.170938, -0.169986, -0.169054, -0.168122, -0.167172, -0.166186, -0.165179, -0.164157, -0.163123, -0.162077, -0.161021, -0.159952, -0.158857, -0.157749, -0.15664, -0.155542, -0.15447, -0.153432, -0.152424, -0.151428, -0.150429, -0.149413, -0.148364, -0.147283, -0.146183, -0.145069, -0.14395, -0.142834, -0.141724, -0.140608, -0.139489, -0.138373, -0.137265, -0.136171, -0.135093, -0.134025, -0.132968, -0.131919, -0.130878, -0.129844, -0.128821, -0.127809, -0.126804, -0.125801, -0.124797, -0.123787, -0.122773, -0.121758, -0.120744, -0.119733, -0.118728, -0.117733, -0.11675, -0.115771, -0.114792, -0.113806, -0.112807, -0.111794, -0.110771, -0.109741, -0.108707, -0.107672, -0.106636, -0.105591, -0.104541, -0.103491, -0.102448, -0.10142, -0.100412, -0.0994198, -0.0984399, -0.0974673, -0.0964976, -0.0955265, -0.0945581, -0.0935982, -0.0926402, -0.0916777, -0.0907045, -0.0897139, -0.0887001, -0.0876694, -0.0866321, -0.0855982, -0.0845778, -0.0835811, -0.0826095, -0.0816517, -0.0806952, -0.0797277, -0.0787369, -0.0777107, -0.0766526, -0.075579, -0.0745061, -0.0734503, -0.0724281, -0.0714473, -0.0704955, -0.0695569, -0.0686153, -0.067655, -0.0666595, -0.0656228, -0.0645626, -0.0634995, -0.0624539, -0.0614458, -0.0604904, -0.0595764, -0.0586898, -0.0578174, -0.0569453, -0.0560601, -0.0551719, -0.0542917, -0.0534099, -0.0525163, -0.0516014, -0.0506555, -0.0496757, -0.0486708, -0.0476511, -0.0466261, -0.0456061, -0.044596, -0.0435836, -0.0425703, -0.0415587, -0.040552, -0.0395533, -0.0385637, -0.0375815, -0.0366045, -0.0356314, -0.0346604, -0.0336902, -0.0327247, -0.0317635, -0.0308037, -0.0298422, -0.0288762, -0.0279032, -0.026925, -0.025943, -0.0249582, -0.0239714, -0.0229839, -0.0219896, -0.0209822, -0.0199722, -0.0189704, -0.0179881, -0.0170357, -0.0161152, -0.0152188, -0.0143404, -0.0134736, -0.0126123, -0.0117534, -0.0109179, -0.0100977, -0.00927668, -0.00843894, -0.0075681, -0.00664104, -0.0056472, -0.00462252, -0.00360524, -0.00263304, -0.00174417
};

float triTable[2048] = {
	0.00205034, 0.00360933, 0.00578754, 0.00835358, 0.0110761, 0.0137236, 0.0160981, 0.0183687, 0.020617, 0.0228593, 0.0251122, 0.027392, 0.0297206, 0.0321003, 0.034499, 0.0368839, 0.0392222, 0.0414812, 0.0436497, 0.0457532, 0.0478204, 0.0498797, 0.0519597, 0.0540801, 0.0562156, 0.0583599, 0.0605111, 0.0626673, 0.0648265, 0.0669885, 0.069155, 0.0713261, 0.0735021, 0.0756831, 0.0778692, 0.0800601, 0.0822559, 0.0844568, 0.086663, 0.0888749, 0.091096, 0.0933308, 0.095572, 0.097812, 0.100043, 0.102257, 0.104463, 0.106663, 0.108853, 0.111028, 0.113181, 0.115309, 0.11741, 0.11949, 0.121555, 0.123613, 0.125667, 0.127712, 0.129738, 0.131757, 0.133783, 0.135828, 0.137904, 0.140002, 0.142116, 0.144248, 0.146397, 0.148564, 0.150756, 0.152981, 0.155227, 0.157478, 0.159719, 0.161935, 0.164142, 0.166348, 0.16854, 0.170706, 0.172831, 0.174903, 0.176915, 0.178884, 0.180829, 0.18277, 0.184725, 0.186695, 0.188658, 0.190622, 0.192593, 0.194577, 0.196582, 0.198611, 0.200659, 0.202715, 0.204773, 0.206824, 0.208863, 0.210895, 0.212924, 0.214952, 0.216985, 0.219024, 0.221064, 0.223103, 0.225147, 0.227201, 0.229273, 0.231368, 0.23349, 0.235633, 0.237784, 0.239934, 0.242072, 0.244201, 0.246336, 0.248466, 0.250583, 0.252677, 0.254738, 0.256759, 0.258749, 0.260722, 0.26269, 0.264667, 0.26666, 0.268659, 0.27066, 0.272662, 0.274664, 0.276664, 0.278666, 0.280672, 0.282676, 0.284675, 0.286663, 0.288636, 0.290596, 0.292546, 0.294487, 0.296422, 0.298354, 0.300276, 0.30218, 0.304077, 0.305982, 0.307905, 0.309861, 0.311873, 0.313923, 0.315983, 0.318021, 0.320007, 0.321913, 0.323747, 0.325536, 0.327304, 0.329078, 0.330883, 0.33272, 0.334568, 0.336425, 0.338291, 0.340166, 0.342047, 0.34393, 0.345818, 0.347717, 0.349633, 0.351571, 0.353541, 0.355551, 0.357581, 0.359612, 0.361626, 0.363604, 0.365556, 0.367494, 0.369414, 0.371311, 0.373181, 0.375019, 0.376812, 0.378576, 0.380325, 0.382078, 0.38385, 0.385646, 0.387452, 0.389264, 0.391078, 0.39289, 0.394698, 0.396507, 0.398318, 0.400125, 0.401927, 0.403717, 0.405494, 0.407256, 0.40901, 0.41076, 0.41251, 0.414267, 0.416024, 0.41778, 0.419538, 0.421301, 0.423072, 0.424857, 0.426659, 0.428472, 0.430287, 0.432097, 0.433893, 0.435678, 0.437459, 0.439233, 0.440997, 0.442746, 0.444477, 0.446184, 0.44787, 0.449548, 0.451226, 0.452915, 0.454622, 0.456343, 0.45807, 0.459797, 0.461518, 0.463227, 0.464921, 0.466606, 0.468285, 0.469962, 0.47164, 0.473324, 0.475015, 0.476707, 0.478398, 0.480083, 0.481759, 0.483424, 0.485082, 0.486734, 0.488379, 0.490016, 0.491647, 0.493268, 0.49488, 0.496487, 0.49809, 0.499692, 0.501294, 0.502892, 0.504487, 0.506082, 0.507679, 0.509283, 0.510886, 0.512485, 0.514089, 0.515705, 0.51734, 0.519007, 0.520729, 0.522484, 0.524239, 0.52596, 0.527614, 0.529175, 0.530657, 0.532094, 0.533515, 0.534954, 0.53644, 0.537952, 0.539475, 0.541016, 0.542579, 0.544171, 0.545804, 0.547497, 0.549225, 0.550958, 0.552666, 0.554319, 0.555907, 0.557451, 0.558968, 0.560474, 0.561985, 0.563516, 0.565063, 0.566615, 0.568164, 0.569701, 0.571217, 0.572711, 0.574192, 0.575659, 0.577106, 0.578532, 0.579932, 0.581296, 0.582629, 0.583943, 0.585252, 0.586569, 0.5879, 0.589228, 0.590557, 0.591892, 0.593238, 0.594602, 0.595985, 0.597385, 0.598797, 0.600218, 0.601643, 0.603069, 0.604501, 0.605938, 0.60738, 0.608827, 0.610277, 0.61173, 0.613187, 0.614649, 0.616114, 0.617584, 0.619058, 0.620545, 0.622044, 0.623545, 0.625037, 0.62651, 0.627958, 0.629403, 0.630838, 0.632249, 0.633624, 0.634949, 0.636201, 0.63738, 0.63852, 0.639657, 0.640828, 0.642065, 0.643361, 0.644693, 0.646047, 0.647405, 0.648753, 0.650083, 0.651415, 0.652746, 0.654072, 0.655389, 0.656694, 0.65799, 0.659279, 0.660559, 0.661826, 0.663078, 0.664312, 0.665526, 0.666724, 0.667911, 0.66909, 0.670267, 0.671435, 0.672582, 0.673724, 0.674874, 0.676046, 0.677253, 0.678497, 0.679767, 0.681053, 0.682348, 0.683643, 0.684935, 0.686242, 0.687554, 0.688863, 0.690158, 0.691429, 0.692672, 0.693896, 0.695105, 0.696306, 0.697506, 0.698708, 0.699909, 0.701106, 0.7023, 0.703489, 0.704673, 0.705857, 0.707048, 0.708236, 0.709408, 0.710553, 0.711661, 0.712721, 0.713743, 0.714746, 0.715747, 0.716765, 0.71781, 0.718864, 0.719926, 0.720996, 0.722076, 0.723167, 0.724272, 0.725391, 0.726519, 0.727651, 0.728784, 0.729915, 0.731064, 0.732222, 0.733373, 0.734501, 0.73559, 0.736625, 0.737607, 0.738559, 0.739503, 0.740458, 0.741448, 0.742462, 0.74349, 0.744529, 0.745577, 0.746634, 0.747701, 0.748797, 0.749906, 0.751012, 0.752098, 0.753145, 0.754144, 0.755107, 0.756048, 0.756979, 0.757913, 0.758861, 0.759812, 0.760762, 0.761714, 0.76267, 0.763634, 0.76461, 0.765599, 0.766594, 0.76759, 0.768579, 0.769555, 0.770513, 0.77146, 0.772402, 0.773347, 0.774301, 0.775274, 0.776266, 0.777269, 0.778272, 0.779266, 0.780239, 0.781189, 0.782124, 0.783047, 0.783961, 0.784869, 0.785775, 0.786678, 0.787576, 0.788467, 0.78935, 0.790223, 0.791082, 0.791926, 0.79276, 0.793589, 0.794419, 0.795256, 0.796098, 0.79694, 0.797783, 0.79863, 0.799482, 0.80034, 0.801205, 0.802075, 0.802947, 0.80382, 0.804691, 0.805566, 0.806452, 0.807338, 0.808211, 0.80906, 0.809874, 0.810645, 0.811386, 0.81211, 0.812829, 0.813558, 0.814304, 0.815056, 0.815811, 0.816566, 0.81732, 0.818071, 0.818818, 0.819561, 0.820303, 0.821046, 0.821791, 0.82254, 0.823293, 0.824047, 0.824803, 0.825561, 0.82632, 0.827082, 0.827848, 0.828615, 0.829381, 0.830142, 0.830896, 0.83163, 0.832351, 0.833072, 0.833807, 0.83457, 0.835373, 0.83622, 0.837092, 0.837972, 0.838841, 0.839683, 0.840509, 0.841335, 0.842146, 0.842929, 0.84367, 0.844352, 0.844957, 0.845509, 0.846038, 0.846575, 0.847151, 0.847775, 0.848416, 0.849075, 0.849755, 0.850456, 0.851183, 0.85195, 0.852752, 0.85357, 0.854387, 0.855184, 0.855947, 0.856677, 0.857391, 0.858105, 0.858834, 0.859596, 0.860406, 0.861255, 0.862121, 0.86298, 0.863811, 0.864592, 0.86533, 0.86604, 0.866731, 0.867413, 0.868096, 0.868783, 0.869464, 0.87014, 0.870811, 0.871481, 0.872149, 0.872817, 0.873483, 0.874146, 0.874806, 0.875459, 0.876105, 0.876739, 0.877367, 0.877993, 0.878622, 0.879259, 0.879906, 0.880561, 0.88122, 0.881879, 0.882535, 0.883182, 0.883819, 0.884449, 0.885079, 0.885715, 0.886363, 0.887038, 0.887757, 0.888489, 0.889203, 0.889868, 0.890451, 0.890953, 0.891393, 0.891786, 0.892143, 0.892477, 0.8928, 0.89311, 0.893397, 0.893649, 0.893858, 0.894011, 0.894122, 0.894207, 0.894245, 0.894213, 0.894089, 0.893852, 0.893506, 0.893064, 0.892538, 0.891937, 0.891272, 0.890547, 0.889748, 0.888879, 0.887944, 0.886946, 0.885891, 0.884773, 0.883588, 0.882343, 0.881043, 0.879694, 0.878299, 0.876848, 0.875342, 0.87379, 0.872197, 0.87057, 0.868908, 0.867204, 0.86546, 0.863682, 0.861874, 0.860039, 0.858177, 0.856286, 0.854363, 0.852406, 0.850414, 0.848385, 0.846323, 0.844227, 0.842095, 0.839925, 0.837715, 0.835457, 0.83315, 0.830807, 0.82844, 0.826061, 0.82368, 0.821277, 0.818856, 0.816424, 0.81399, 0.811562, 0.80915, 0.806747, 0.804345, 0.801937, 0.799514, 0.797067, 0.7946, 0.792119, 0.789623, 0.787112, 0.784586, 0.782038, 0.779443, 0.776826, 0.774214, 0.771636, 0.76912, 0.766686, 0.764314, 0.761978, 0.759652, 0.757309, 0.754927, 0.752535, 0.750137, 0.747724, 0.745288, 0.742821, 0.740305, 0.737727, 0.73512, 0.732517, 0.72995, 0.727452, 0.725032, 0.722667, 0.72033, 0.717997, 0.715644, 0.713254, 0.710851, 0.708438, 0.706014, 0.703575, 0.701121, 0.698641, 0.696136, 0.693616, 0.691095, 0.688585, 0.686098, 0.683641, 0.6812, 0.678761, 0.676308, 0.673827, 0.671306, 0.668755, 0.666184, 0.663599, 0.661012, 0.65843, 0.655845, 0.653253, 0.65066, 0.64807, 0.645488, 0.64292, 0.640369, 0.637827, 0.635287, 0.632742, 0.630183, 0.6276, 0.624998, 0.622388, 0.619784, 0.617199, 0.614645, 0.612122, 0.60962, 0.607132, 0.60465, 0.602168, 0.599682, 0.597203, 0.594727, 0.592252, 0.589774, 0.58729, 0.584793, 0.582286, 0.579778, 0.577277, 0.57479, 0.572324, 0.569874, 0.567436, 0.56501, 0.562592, 0.560181, 0.557782, 0.555401, 0.553029, 0.550655, 0.548272, 0.545869, 0.543444, 0.541004, 0.538559, 0.536116, 0.533683, 0.531266, 0.52886, 0.52646, 0.524064, 0.521671, 0.519279, 0.516893, 0.514518, 0.512143, 0.50976, 0.507362, 0.504938, 0.502484, 0.50001, 0.497529, 0.495055, 0.4926, 0.490175, 0.487775, 0.485386, 0.482995, 0.480589, 0.478154, 0.475672, 0.473159, 0.47064, 0.468141, 0.46569, 0.463311, 0.460999, 0.458729, 0.456477, 0.454215, 0.451918, 0.449584, 0.44723, 0.444865, 0.442494, 0.440126, 0.437766, 0.43542, 0.433079, 0.430733, 0.428371, 0.425982, 0.423555, 0.421087, 0.418597, 0.416105, 0.413627, 0.411182, 0.408763, 0.406361, 0.403972, 0.401595, 0.399227, 0.396868, 0.394526, 0.392195, 0.389871, 0.387547, 0.385218, 0.382886, 0.380556, 0.378225, 0.375891, 0.373548, 0.371195, 0.368828, 0.366452, 0.364071, 0.361691, 0.359317, 0.356948, 0.354579, 0.352212, 0.349851, 0.347498, 0.345159, 0.342839, 0.340534, 0.338236, 0.335938, 0.333632, 0.331313, 0.328992, 0.326667, 0.324335, 0.321992, 0.319635, 0.317258, 0.314862, 0.312455, 0.310044, 0.307636, 0.305238, 0.30284, 0.300442, 0.298047, 0.29566, 0.293284, 0.290916, 0.288546, 0.286183, 0.283838, 0.281522, 0.279246, 0.277034, 0.274878, 0.272745, 0.270602, 0.268418, 0.266164, 0.263851, 0.261499, 0.259129, 0.256758, 0.254405, 0.25208, 0.249766, 0.247457, 0.245146, 0.242826, 0.240491, 0.238132, 0.235759, 0.233382, 0.231014, 0.228664, 0.22634, 0.224037, 0.221748, 0.219468, 0.217193, 0.214919, 0.212651, 0.210391, 0.208136, 0.205878, 0.203614, 0.201338, 0.19905, 0.196756, 0.194458, 0.192161, 0.189868, 0.187578, 0.185288, 0.182999, 0.180713, 0.178434, 0.176164, 0.173903, 0.171648, 0.1694, 0.167155, 0.164912, 0.162674, 0.16045, 0.158231, 0.156008, 0.153768, 0.151502, 0.149211, 0.146903, 0.144578, 0.142234, 0.139871, 0.137483, 0.135038, 0.132556, 0.130074, 0.127626, 0.125247, 0.12297, 0.120783, 0.118646, 0.116519, 0.114362, 0.112133, 0.109827, 0.107474, 0.105099, 0.102728, 0.100388, 0.0980987, 0.0958451, 0.0936141, 0.0913946, 0.0891749, 0.0869439, 0.084699, 0.0824483, 0.0801958, 0.0779458, 0.0757025, 0.073469, 0.0712418, 0.06902, 0.066804, 0.0645945, 0.0623925, 0.0602031, 0.0580333, 0.0558718, 0.0537071, 0.0515269, 0.0493195, 0.0470875, 0.0448382, 0.0425741, 0.0402977, 0.0380118, 0.0357137, 0.0333861, 0.0310419, 0.028698, 0.0263721, 0.0240815, 0.0218298, 0.019603, 0.0173978, 0.015211, 0.013039, 0.0108804, 0.00875302, 0.00665008, 0.00455578, 0.00245417, 0.000329488, -0.00183022, -0.00401578, -0.00621704, -0.00842485, -0.0106295, -0.0128215, -0.0149995, -0.0171705, -0.019339, -0.0215093, -0.0236866, -0.0258755, -0.0280772, -0.0302866, -0.0324974, -0.0347034, -0.0368983, -0.0390758, -0.0412378, -0.0433932, -0.0455518, -0.0477225, -0.0499145, -0.0521294, -0.0543603, -0.056599, -0.0588375, -0.0610684, -0.0632876, -0.0655037, -0.0677167, -0.0699247, -0.0721264, -0.0743201, -0.0765108, -0.0786993, -0.0808802, -0.0830487, -0.0851995, -0.0873284, -0.089439, -0.0915334, -0.0936121, -0.0956753, -0.0977242, -0.0997517, -0.101752, -0.103736, -0.105717, -0.107708, -0.10972, -0.111741, -0.113769, -0.115806, -0.117857, -0.119925, -0.122017, -0.124138, -0.126278, -0.128425, -0.130568, -0.132695, -0.134806, -0.136907, -0.139004, -0.141099, -0.143195, -0.145297, -0.147412, -0.149533, -0.151649, -0.15375, -0.155824, -0.157859, -0.159851, -0.161824, -0.163796, -0.16579, -0.167825, -0.169907, -0.172019, -0.174147, -0.176275, -0.178386, -0.18047, -0.182542, -0.184604, -0.186657, -0.188702, -0.19074, -0.192767, -0.19478, -0.196787, -0.198791, -0.200797, -0.202811, -0.204826, -0.206843, -0.208861, -0.210884, -0.212911, -0.214946, -0.216992, -0.219043, -0.221093, -0.223135, -0.225165, -0.227174, -0.229169, -0.231159, -0.233156, -0.235167, -0.237203, -0.239262, -0.241337, -0.243416, -0.245491, -0.247553, -0.249605, -0.251657, -0.253703, -0.255735, -0.257747, -0.25973, -0.261673, -0.263588, -0.265491, -0.267398, -0.269325, -0.271286, -0.27327, -0.275268, -0.277268, -0.279258, -0.281228, -0.283188, -0.285145, -0.28709, -0.289016, -0.290914, -0.292773, -0.294573, -0.296337, -0.298095, -0.299876, -0.301711, -0.303622, -0.305594, -0.307596, -0.309599, -0.311574, -0.313491, -0.315348, -0.31717, -0.318975, -0.320785, -0.322621, -0.324499, -0.326405, -0.328328, -0.330257, -0.332182, -0.334093, -0.33599, -0.337882, -0.339769, -0.341653, -0.343533, -0.345412, -0.347287, -0.34916, -0.351029, -0.352894, -0.354757, -0.356615, -0.358471, -0.360323, -0.362171, -0.364016, -0.365857, -0.367695, -0.36953, -0.371361, -0.373187, -0.375008, -0.376822, -0.378625, -0.380423, -0.382218, -0.384017, -0.385823, -0.387639, -0.389461, -0.391286, -0.393111, -0.39493, -0.396744, -0.398568, -0.400394, -0.402209, -0.404001, -0.405756, -0.407455, -0.409092, -0.410699, -0.412308, -0.41395, -0.415656, -0.417438, -0.419271, -0.421126, -0.422972, -0.424782, -0.426532, -0.428234, -0.429909, -0.431572, -0.433239, -0.434928, -0.436646, -0.438382, -0.440124, -0.44186, -0.443579, -0.445267, -0.446914, -0.448535, -0.450152, -0.451785, -0.453454, -0.455174, -0.456935, -0.458721, -0.460517, -0.462308, -0.464079, -0.465839, -0.467598, -0.46935, -0.471092, -0.472818, -0.474526, -0.476213, -0.477884, -0.479547, -0.481207, -0.482871, -0.484546, -0.486228, -0.487908, -0.489577, -0.491226, -0.492845, -0.49444, -0.496014, -0.497571, -0.499113, -0.500644, -0.502158, -0.503643, -0.505112, -0.506581, -0.508065, -0.509579, -0.511122, -0.512685, -0.514262, -0.515847, -0.517436, -0.519026, -0.520626, -0.522233, -0.523842, -0.525448, -0.527046, -0.528634, -0.530216, -0.531793, -0.533369, -0.534945, -0.536525, -0.538114, -0.539707, -0.541298, -0.542879, -0.544443, -0.545986, -0.54751, -0.549021, -0.550522, -0.552019, -0.553515, -0.555017, -0.556521, -0.558019, -0.559502, -0.56096, -0.562383, -0.563768, -0.565125, -0.566468, -0.567813, -0.569172, -0.57054, -0.571898, -0.573261, -0.574643, -0.576059, -0.577525, -0.579052, -0.580624, -0.582221, -0.583824, -0.585413, -0.586978, -0.588546, -0.590111, -0.591667, -0.593204, -0.594717, -0.596205, -0.597674, -0.599125, -0.600559, -0.601977, -0.60338, -0.604764, -0.606132, -0.607485, -0.608824, -0.610153, -0.611468, -0.612764, -0.614047, -0.615324, -0.616602, -0.617887, -0.619173, -0.620458, -0.621743, -0.623032, -0.624325, -0.625623, -0.626915, -0.628209, -0.629514, -0.630837, -0.632189, -0.633581, -0.635008, -0.636455, -0.637902, -0.639332, -0.640731, -0.642122, -0.643505, -0.644871, -0.646211, -0.647516, -0.648774, -0.649981, -0.651157, -0.652322, -0.653495, -0.654696, -0.655923, -0.657163, -0.658409, -0.659653, -0.660887, -0.662105, -0.66331, -0.664506, -0.6657, -0.666896, -0.668101, -0.669321, -0.670557, -0.671796, -0.673025, -0.674233, -0.675407, -0.676535, -0.677633, -0.678721, -0.679822, -0.680957, -0.682138, -0.683351, -0.684586, -0.685833, -0.687084, -0.688329, -0.689583, -0.69085, -0.692118, -0.69337, -0.694595, -0.69578, -0.696928, -0.698049, -0.699154, -0.700253, -0.701358, -0.702475, -0.703598, -0.704718, -0.705824, -0.70691, -0.707963, -0.708967, -0.709937, -0.710897, -0.711871, -0.712884, -0.713953, -0.715065, -0.716205, -0.717357, -0.718507, -0.719638, -0.720763, -0.721891, -0.723013, -0.724122, -0.725208, -0.726265, -0.72729, -0.728292, -0.72928, -0.730265, -0.731255, -0.732248, -0.73323, -0.734211, -0.7352, -0.736205, -0.737238, -0.738311, -0.739412, -0.740521, -0.741619, -0.742686, -0.743708, -0.744696, -0.745661, -0.746613, -0.747561, -0.748516, -0.749476, -0.750434, -0.75139, -0.752341, -0.753286, -0.754223, -0.75515, -0.756071, -0.756987, -0.757902, -0.758819, -0.759739, -0.760661, -0.761581, -0.762501, -0.763418, -0.764331, -0.765241, -0.76615, -0.767056, -0.767958, -0.768856, -0.769747, -0.770623, -0.771492, -0.772365, -0.773251, -0.77416, -0.775104, -0.776077, -0.777062, -0.778043, -0.779001, -0.779922, -0.780812, -0.781681, -0.782536, -0.783385, -0.784234, -0.785089, -0.785946, -0.7868, -0.787644, -0.788474, -0.789283, -0.790058, -0.790807, -0.791546, -0.792295, -0.793069, -0.793885, -0.79474, -0.795619, -0.796505, -0.797383, -0.798236, -0.799069, -0.799894, -0.800708, -0.801505, -0.802282, -0.803032, -0.80375, -0.804442, -0.80512, -0.805793, -0.806474, -0.807167, -0.807859, -0.808552, -0.809247, -0.809944, -0.810644, -0.811352, -0.812064, -0.812778, -0.813491, -0.814198, -0.814899, -0.815595, -0.816289, -0.816978, -0.817663, -0.818343, -0.819017, -0.819684, -0.820347, -0.821007, -0.821667, -0.822327, -0.822983, -0.823637, -0.824291, -0.824947, -0.82561, -0.826279, -0.826953, -0.827631, -0.828313, -0.829, -0.82969, -0.830403, -0.831135, -0.831867, -0.832579, -0.833252, -0.833867, -0.834421, -0.834934, -0.835429, -0.835926, -0.836447, -0.836999, -0.837564, -0.838138, -0.838714, -0.839288, -0.839852, -0.840397, -0.840932, -0.841471, -0.842025, -0.842609, -0.843234, -0.843893, -0.844576, -0.845274, -0.845979, -0.846681, -0.847395, -0.848127, -0.848861, -0.849583, -0.850275, -0.850926, -0.851554, -0.85216, -0.852739, -0.853287, -0.853801, -0.85427, -0.854693, -0.855083, -0.855456, -0.855826, -0.856207, -0.85659, -0.856966, -0.857339, -0.857712, -0.858088, -0.858469, -0.858855, -0.859243, -0.85963, -0.860011, -0.860383, -0.86074, -0.861083, -0.861421, -0.861764, -0.862121, -0.862499, -0.862889, -0.863289, -0.863701, -0.86413, -0.864576, -0.865053, -0.865568, -0.866101, -0.866632, -0.867139, -0.867603, -0.86803, -0.868434, -0.868816, -0.86918, -0.869529, -0.869881, -0.870272, -0.870662, -0.871001, -0.871238, -0.871324, -0.87118, -0.870828, -0.870378, -0.869937, -0.869614, -0.869502, -0.869488, -0.869554, -0.869727, -0.870037, -0.870513, -0.871247, -0.872318, -0.873557, -0.874789, -0.875836, -0.876525, -0.876945, -0.877194, -0.877256, -0.877115, -0.876754, -0.87615, -0.875272, -0.874164, -0.872877, -0.871464, -0.869974, -0.868396, -0.866678, -0.864844, -0.862921, -0.860934, -0.858903, -0.856797, -0.854614, -0.852368, -0.850075, -0.84775, -0.845392, -0.842974, -0.840512, -0.838024, -0.83553, -0.833047, -0.830567, -0.828078, -0.825584, -0.823086, -0.820586, -0.818085, -0.815569, -0.813046, -0.810528, -0.808027, -0.805556, -0.803131, -0.800752, -0.798393, -0.796028, -0.79363, -0.791176, -0.788669, -0.786125, -0.783558, -0.780983, -0.778414, -0.775859, -0.773306, -0.770751, -0.768194, -0.765633, -0.763064, -0.76048, -0.757883, -0.755282, -0.752691, -0.750121, -0.747581, -0.745068, -0.742576, -0.740095, -0.737621, -0.735144, -0.732669, -0.730205, -0.727744, -0.725278, -0.722802, -0.720306, -0.717784, -0.715244, -0.712698, -0.710156, -0.707631, -0.70513, -0.702646, -0.700175, -0.697711, -0.695248, -0.692784, -0.69032, -0.687859, -0.6854, -0.682941, -0.680482, -0.67802, -0.675555, -0.673088, -0.670622, -0.668159, -0.665703, -0.663256, -0.660822, -0.658392, -0.655958, -0.653513, -0.651048, -0.64856, -0.646055, -0.643541, -0.641026, -0.63852, -0.636029, -0.633553, -0.631083, -0.628611, -0.626129, -0.62363, -0.62111, -0.618576, -0.616031, -0.613479, -0.610924, -0.608366, -0.605796, -0.603217, -0.600638, -0.598067, -0.59551, -0.592974, -0.590451, -0.58794, -0.585435, -0.582935, -0.580436, -0.577928, -0.575415, -0.572911, -0.570425, -0.567972, -0.565563, -0.563209, -0.560889, -0.55858, -0.55626, -0.553906, -0.551515, -0.549106, -0.546682, -0.544246, -0.541801, -0.53935, -0.536897, -0.534438, -0.531967, -0.529476, -0.526961, -0.524413, -0.521828, -0.519219, -0.516602, -0.513988, -0.511392, -0.508796, -0.506192, -0.503597, -0.501024, -0.498489, -0.496007, -0.493576, -0.491182, -0.488812, -0.486449, -0.484079, -0.481706, -0.479345, -0.476989, -0.474628, -0.472254, -0.46986, -0.467445, -0.465015, -0.462576, -0.460135, -0.457698, -0.455269, -0.452846, -0.450424, -0.448, -0.445567, -0.443123, -0.440673, -0.438221, -0.435759, -0.433282, -0.430783, -0.428251, -0.425661, -0.423039, -0.420415, -0.417822, -0.415292, -0.412848, -0.410474, -0.40814, -0.405816, -0.40347, -0.401075, -0.39863, -0.396155, -0.393671, -0.391196, -0.388748, -0.386341, -0.38396, -0.381597, -0.379251, -0.376914, -0.374584, -0.372274, -0.369988, -0.367709, -0.365422, -0.36311, -0.360759, -0.358377, -0.355973, -0.353552, -0.35112, -0.348683, -0.346232, -0.343757, -0.341272, -0.338791, -0.33633, -0.333904, -0.331509, -0.329135, -0.326778, -0.324435, -0.322102, -0.319783, -0.317494, -0.315222, -0.312948, -0.310658, -0.308332, -0.305972, -0.303591, -0.301191, -0.298771, -0.296331, -0.293869, -0.291362, -0.288824, -0.286279, -0.28375, -0.281261, -0.278825, -0.276424, -0.274049, -0.27169, -0.269337, -0.266981, -0.264622, -0.262268, -0.259921, -0.257585, -0.255261, -0.252955, -0.250671, -0.248402, -0.246138, -0.243872, -0.241594, -0.239308, -0.237026, -0.234738, -0.232437, -0.230116, -0.227767, -0.225386, -0.222981, -0.220562, -0.218139, -0.215721, -0.213308, -0.210888, -0.208467, -0.206051, -0.203648, -0.201264, -0.198897, -0.196543, -0.194201, -0.191873, -0.18956, -0.187264, -0.184994, -0.182742, -0.180497, -0.178248, -0.175985, -0.173709, -0.17143, -0.169145, -0.16685, -0.164542, -0.162215, -0.159862, -0.15749, -0.15511, -0.152733, -0.150369, -0.148029, -0.145705, -0.143391, -0.141079, -0.138761, -0.136432, -0.134085, -0.131728, -0.129367, -0.127009, -0.124661, -0.122327, -0.119997, -0.117672, -0.115358, -0.113058, -0.110777, -0.108528, -0.106311, -0.10411, -0.101907, -0.0996853, -0.097427, -0.0951316, -0.0928122, -0.0904801, -0.0881476, -0.0858269, -0.0835248, -0.0812303, -0.0789414, -0.0766578, -0.074378, -0.0721014, -0.0698264, -0.0675536, -0.0652851, -0.0630235, -0.0607712, -0.0585312, -0.0563106, -0.0541026, -0.0518976, -0.0496861, -0.0474588, -0.0452096, -0.0429452, -0.040671, -0.0383926, -0.0361152, -0.0338438, -0.0315732, -0.0293009, -0.0270301, -0.024764, -0.0225054, -0.0202499, -0.0179787, -0.0157089, -0.0134595, -0.011251, -0.00910325
};
