#!/usr/bin/ruby -w

require "xmlrpc/client"
require "fileutils"
require "singleton"
require "timeout"
require "pp"

require "glib2"

def debug(*args)
	return if false
	pp args
end



module Eat
end



class Eat::Aria2 < GLib::Object

	DEFAULT_PORT = 6801
	DEFAULT_HOST = '127.0.0.1'

	attr_accessor :user
	attr_accessor :password
	attr_accessor :hostname
	attr_accessor :port

	attr_reader :is_connected
	attr_reader :version

	private

	type_register

	signal_new("download_status", GLib::Signal::RUN_FIRST, nil,
			nil,					# Return type: void
			String)					# Parameters: gid
	signal_new("download_completed", GLib::Signal::RUN_FIRST, nil,
			nil,					# Return type: void
			String)					# Parameters: gid
	signal_new("download_removed", GLib::Signal::RUN_FIRST, nil,
			nil,					# Return type: void
			String)					# Parameters: gid
	signal_new("download_stopped", GLib::Signal::RUN_FIRST, nil,
			nil,					# Return type: void
			String)					# Parameters: gid

	def signal_do_download_status(gid)
	end
	def signal_do_download_completed(gid)
	end
	def signal_do_download_removed(gid)
	end
	def signal_do_download_stopped(gid)
	end

	def initialize(hostname = DEFAULT_HOST, port = DEFAULT_PORT, user = nil, password = nil)
		super(nil)

		@new_downloads = []

		@confdir = ENV['XDG_CONFIG_HOME']
		@confdir = ENV['HOME']+"/.config" if @confdir == nil
		@confdir += "/eatmonkey"

		@hostname, @port, @user, @password = hostname, port, user, password
		@is_connected = false
		@use_local_server = true
		@version = "n/a"
		@@client = nil if !defined? @@client

		if !defined? @@pid
			begin
				@@pid = File.open(@confdir+"/aria2.pid").readline.strip.to_i
				Process.kill("CONT", @@pid)
			rescue
				@@pid = 0
			end
		end
	end

=begin
	start_server:
	Returns negative values on error, -1 if a service is already listening
	on the port and -2 if there was a problem during the creaton of the
	server. It returns the pid of the server on success or 0 if no server
	should be run (see use_local_server).
=end
	def start_server()
		return 0 if @use_local_server == false
		begin
			TCPSocket.new(@hostname, @port).close
			return -1
		rescue
			begin
				# Create config file
				FileUtils.mkdir_p(@confdir, :mode => 0700)
				FileUtils.touch(@confdir+"/aria2.conf")
				# Cleanup log file
				FileUtils.rm(@confdir+"/aria2.log", :force => true)
				# Launch aria2c process
				command = "aria2c --disable-ipv6=true --enable-xml-rpc --xml-rpc-listen-port=#{@port} " \
					"--conf-path=#{@confdir}/aria2.conf --log=#{@confdir}/aria2.log --log-level=notice " \
					"--dht-file-path=#{@confdir}/dht.dat --lowest-speed-limit=1K"
				debug("start server", command)
				@@pid = Process.spawn(command, :pgroup=>true, :chdir=>ENV['HOME'],
						STDOUT=>"/dev/null", STDIN=>"/dev/null")
				Process.detach(@@pid)
				# Wait for the server to respond properly to requests
				begin
					debug("test request")
					begin
						result = @@client.call("aria2.getVersion")
					rescue
					end
					sleep 1 if result == nil
				end while result == nil
				# Store pid in a file
				begin
					File.open(@confdir+"/aria2.pid", "w").puts(@@pid)
				rescue
					p $!
				end
				puts "Started aria2 XML-RPC Server (pid #{@@pid.to_s})..."
				return @@pid
			rescue
				return -2
			end
		end
	end

	def call(method, *args)
		begin
			result = @@client.call(method, *args)
		rescue XMLRPC::FaultException => e
			# Unsupported/Bad XMLRPC request
			debug("XMLRPC server didn't support the request")
			puts e.message
		rescue Errno::EPIPE => e
			# Connection interrupted/timed out/server shutdown
			return nil if !@is_connected
			connect(true)
			result = call(method, args)
		rescue Errno::ECONNREFUSED => e
			# Connection refused
			return nil if start_server < 0
			result = call(method, args)
		rescue Exception => e
			# Unhandled exception
			puts $!, e.message
		end
		result
	end

	public

=begin
	connect:
	Creates an XMLRPC::Client instance and sends a first request that will
	make sure we can talk to the server.
=end
	def connect(force=false)
		return if @@client and force == false
		@is_connected = false
		begin
			@@client = XMLRPC::Client.new3({:host => @hostname, :path => "/rpc",
					:port => @port, :user => @user, :password => @password,
					:timeout => 60})
			debug("call")
			result = nil
			Timeout::timeout(5) do
				result = call("aria2.getVersion")
			end
			raise Exception.new("Unable to get an appropriate response to our request") if result == nil
			@version = result["version"]
			@is_connected = true
		rescue Exception => e
			@@client = nil
			puts "Can't establish a connection"
			debug(e.message)
		end
	end

=begin
	use_custom_server:
	Sets aria2 to connect to a custom server.
=end
	def use_custom_server(hostname, port, user, password)
		@use_local_server = false
		@hostname = hostname
		@port = port
		@user = user.to_s if !user.empty?
		@password = password.to_s if !password.empty?
	end

=begin
	use_local_server:
	Sets aria2 to connect to a local (self-managed) server.
=end
	def use_local_server()
		@use_local_server = true
		@hostname = DEFAULT_HOST
		@port = DEFAULT_PORT
		@user = nil
		@password = nil
	end

	def use_local_server?()
		@use_local_server and @@pid > 0
	end

=begin
	shutdown:
	Disconnect and terminates the local server if it exists.
=end
	def shutdown()
		debug("shutdown", @@pid)
		begin
			if @@pid > 0
				FileUtils.rm(@confdir+"/aria2.pid", :force => true)
				Process.kill("TERM", @@pid)
				sleep 1 while Process.kill("CONT", @@pid)
			end
		rescue Errno::ESRCH
			# Process does no more exist
		rescue
			p $!
		end
		@@pid = 0
		@@client = nil
		@is_connected = false
	end

	# Adds new HTTP(S)/FTP/BitTorrent/Magnet URI.
	def add_uri(uris, options = nil, position = nil)
		uris = [ uris ] if uris.class == String
		gid = call("aria2.addUri", uris)
		@new_downloads << gid if gid != nil
		gid
	end

	# Adds BitTorrent download by uploading .torrent file.
	def add_torrent(torrent, uris = nil, options = nil, position = nil)
		data = read(torrent)
		gid = call("aria2.addTorrent", XMLRPC::Base64.new(data))
		@new_downloads << gid if gid != nil
		gid
	end

	# Adds Metalink download by uploading .metalink file.
	def add_metalink(metalink, options = nil, position = nil)
		data = read(metalink)
		gid = call("aria2.addMetalink", XMLRPC::Base64.new(data))
		@new_downloads << gid if gid != nil
		gid
	end

	# Removes the download denoted by @gid.
	def remove(gid)
		call("aria2.remove", gid)
	end

	# Returns download progress of the download denoted by @gid.
	def tell_status(gid)
		result = call("aria2.tellStatus", gid)
		result = [] if result == nil
		result
	end

	# Returns URIs used in the download denoted by @gid.
	def get_uris(gid)
		call("aria2.getUris", gid)
	end

	# Returns file list of the download denoted by @gid.
	def get_files(gid)
		call("aria2.getFiles", gid)
	end

	# Returns peer list of the download denoted by @gid.
	def get_peers(gid)
		call("aria2.getPeers", gid)
	end

	# Returns the list of active downloads.
	def tell_active()
		result = call("aria2.tellActive")
		result = [] if !result
		result
	end

	# Returns the list of waiting download.
	def tell_waiting(offset, num)
		result = call("aria2.tellWaiting", offset, num)
		result = [] if !result
		result
	end

	# Returns the list of stopped download.
	def tell_stopped(offset, num)
		return nil if @version < "1.8"
		result = call("aria2.tellStopped", offset, num)
		result = [] if result == nil
		result
	end

	# Changes the position of the download denoted by @gid.
	def change_position(gid, pos, from_current = false)
		return nil if @version < "1.8"
		how = from_current ? "POS_CUR" : "POS_RET"
		call("aria2.changePosition", gid, pos, how)
	end

	# Returns options of the download denoted by @gid.
	def get_options(gid)
		return nil if @version < "1.8"
		call("aria2.getOption", gid)
	end

	# Changes options of the download denoted by @gid dynamically.
	def change_options(gid, options)
		call("aria2.changeOption", gid, options)
	end

	# Returns global options.
	def get_global_options()
		return nil if @version < "1.8"
		call("aria2.getGlobalOption")
	end

	# Changes global options dynamically.
	def change_global_options(options)
		call("aria2.changeGlobalOption", options)
	end

	# Purges completed/error/removed downloads to free memory.
	def purge_download_result()
		call("aria2.purgeDownloadResult")
	end

end



class Eat::Aria2Listener < Eat::Aria2

	include Singleton

	def initialize()
		super
		@downloads = []
		start
	end

	def start()
		GLib::Timeout.add_seconds(1) do

			active_downloads = []

			if @is_connected

				# Emit status signal for each active download
				tell_active.each do |res|
					gid = res["gid"]
					@downloads << gid if !@downloads.find_index(gid)
					active_downloads << gid
					signal_emit("download_status", gid)
				end

				# Check/Cleanup unhandled new downloads
				(@new_downloads - active_downloads).each do |gid|
					@downloads << gid
				end
				@new_downloads.clear

				# Emit specific signal for each inactive download
				(@downloads - active_downloads).each do |gid|
					# Get status
					status = tell_status(gid)
					status_msg = status["status"]
					case status["status"]
						when "complete" then signal_emit("download_completed", gid)
						when "removed" then signal_emit("download_removed", gid)
						when "error"
							if status["errorCode"] == "5"
								# Download was aborted due to "pause"
								debug "pause lasted too long, download %s dropped" % gid
							else
								puts "Unhandled error (%s: %s)" % [gid, status["errorCode"]]
							end
						else puts "Unhandled status (%s: %s)" % [gid, status_msg]
					end
				end

			end

			@downloads = active_downloads
			true
		end
	end

end



class Eat::Aria2Config

	attr_reader :filename
	attr_accessor :values

	def initialize(filename="aria2.conf")
		@filename = filename
		@values = {}
		parse
	end

	private

	def default_values()
		# Basic options
		#@values["dir"] = "." # directory to store downloaded files
		@values["log"] = "-" # file name of the log file, - means STDOUT
		@values["max-concurrent-downloads"] = 5 # maximum number of parallel downloads for every static URI
		@values["check-integrity"] = false # check file integrity, effect only in BitTorrent and Metalink

		# HTTP/FTP options
		#@values["all-proxy"] = "" # proxy server for all protocols
		@values["connect-timeout"] = 60 # timeout to establish a connection
		@values["lowest-speed-limit"] = 0 # close connection if download speed is lower than this value
		@values["max-file-not-found"] = 0 # number of times to try a file reporting 404 before giving up
		@values["max-tries"] = 5 # number of tries
		@values["proxy-method"] = "get" # specify method to use in proxy request, get or tunnel
		@values["remote-time"] = false # applies the file timestamp from the remote server
		@values["split"] = 5 # number of connections to use for downloading a file
		@values["timeout"] = 60 # timeout in seconds

		# HTTP options
		#@values["ca-certificate"] = "" # file name of certificate authorities to use
		#@values["certificate"] = "" # file name of client certificate
		@values["check-certificate"] = true # verify the peer using certificates
		@values["http-auth-challenge"] = false # send auth header only when it is requested by the server
		@values["http-user"] = "" # HTTP user
		@values["http-passwd"] = "" # HTTP password
		#@values["http-proxy"] = "" # proxy server for HTTP
		#@values["https-proxy"] = "" # proxy server for HTTPS
		#@values["private-key"] = "" # file name of private key to use
		@values["referer"] = "" # set referer
		@values["enable-http-keep-alive"] = true # enable HTTP/1.1 persistent connection
		@values["enable-http-pipelining"] = false # enable HTTP/1.1 pipelining
		@values["header"] = "" # header to append to HTTP request, can be used repeatedly
		@values["load-cookies"] = "" # cookies file name to use, supports Firefox3 (sqlite) and Netscape
		@values["save-cookies"] = "" # save cookies to file name in Netscape format
		@values["use-head"] = false # use HEAD method for first request
		@values["user-agent"] = "Eatmonkey/0.1.0 aria2/1.8.0"

		# FTP options
		@values["ftp-user"] = "anonymous" # FTP user
		@values["ftp-passwd"] = "ARIA2USER@" # FTP password
		@values["ftp-pasv"] = true # use passive mode
		#@values["ftp-proxy"] = "" # proxy server for FTP
		@values["ftp-type"] = "binary" # transfer type
		@values["ftp-reuse-connection"] = true # reuse connection

		# BitTorrent options
		@values["bt-external-ip"] = "" # report IP address to tracker
		@values["bt-hash-check-seed"] = true # continue to seed after file is complete and check integrity
		@values["bt-max-open-files"] = 100 # number of files to open in each download
		@values["bt-max-peers"] = 55 # number of maximum peers, 0 means unlimited
		@values["bt-min-crypto-level"] = "plain" # minimum level of encryption, plain or arc4
		@values["bt-prioritize-piece"] = "" # can contain head,tail useful for previewing files
		@values["bt-require-crypto"] = false # accept and establish connection with crypto only
		@values["bt-request-peer-speed-limit"] = "50K" # tries more peers if download speed is under value
		@values["bt-save-metadata"] = false # save metadata to .torrent file
		@values["bt-seed-unverified"] = false # seed without verifying piece hashes
		@values["bt-stop-timeout"] = 0 # stop download if speed is 0 during given seconds, 0 does nothing
		@values["bt-tracker-interval"] = 0 # interval in seconds between tracker requests, 0 does nothing
		#@values["dht-entry-point"] = "" # set HOST:PORT as entry point to DHT network
		#@values["dht-file-path"] = "" # file name of DHT routing table, $XDG_CONFIG_HOME/eatmonkey/dht.dat
		@values["dht-listen-port"] = "6881-6999" # UDP listening port for DHT
		@values["enable-dht"] = true # enable DHT functionality
		@values["enable-peer-exchange"] = true # enable PEX extension
		@values["follow-torrent"] = true # parse .torrent file and downloads mentionned files, can be mem
		@values["listen-port"] = "6881-6999" # TCP listening port for BitTorrent
		@values["max-overall-upload-limit"] = 0 # max overall upload speed in B/s, 0 means unlimited
		@values["max-upload-limit"] = 0 # max upload speed per each download, 0 means unlimited
		@values["peer-id-prefix"] = "Eatmonkey/0.1.0" # set peer ID, max 20 bytes
		@values["seed-ratio"] = 1.0 # specify share ratio

		# Metalink options
		@values["follow-metalink"] = true # parse .metalink file and downloads mentionned files, can be mem
		@values["metalink-servers"] = 5 # number of servers to connect to simultaneously
		@values["metalink-language"] = "" # language of the file to download
		@values["metalink-location"] = "" # preferred location of the server to use, can be JP,US
		@values["metalink-os"] = "" # operating system of the file to download
		@values["metalink-version"] = "" # version of the file to download
		@values["metalink-preferred-protocol"] = "none" # preferred protocol to use, can be http,https,ftp
		@values["metalink-enable-unique-protocol"] = true # use only one type of protocol for download

		# XML-RPC options
		@values["enable-xml-rpc"] = false
		@values["xml-rpc-listen-all"] = false
		@values["xml-rpc-listen-port"] = 6800
		@values["xml-rpc-max-request-size"] = "2M"
		@values["xml-rpc-passwd"] = ""
		@values["xml-rpc-user"] = ""

		# Advanced options
		@values["allow-overwrite"] = false # re-download the file even if file exists
		@values["allow-piece-length-change"] = false # continue download even if piece length is different from control file
		@values["async-dns"] = true # asynchronous DNS
		@values["auto-file-renaming"] = true # rename filename if file already exists
		@values["auto-save-interval"] = 60 # save a .aria2 control file every SEC seconds
		@values["conf-path"] = "" # change the configuration file path
		@values["daemon"] = false # run as daemon
		@values["disable-ipv6"] = false # disable IPv6
		@values["enable-direct-io"] = true # enable direct I/O to lower CPU cycles while allocating/checking files
		@values["file-allocation"] = "prealloc" # file allocation method, can be none, prealloc, falloc
		@values["interface"] = "" # bind sockets to given interface, can be interface name, IP and hostname
		@values["log-level"] = "debug" # log level, can be debug,info,notice,warn,error
		@values["on-download-complete"] = "" # command to execute when download completes, cf. start
		@values["on-download-error"] = "" # command to execute when download aborts, cf. start
		@values["on-download-start"] = "" # command to execute when download starts, GID is passed to command
		@values["on-download-stop"] = "" # command to execute when download stops, not executed if complete/error is set
		@values["summary-interval"] = 60 # interval in seconds to output summary, 0 to supress the output
		@values["force-sequential"] = false # download one URI after another
		@values["max-overall-download-limit"] = 0 # max overall download speed in B/s, 0 means unlimited
		@values["max-download-limit"] = 0 # max download speed per each download in B/s, 0 means unlimited
		@values["no-file-allocation-limit"] = "5K" # no file allocation for files smaller than value
		@values["parameterized-uri"] = false # enable parameterized URI support, e.g. {file1,file2} file[0-9]
		@values["quiet"] = false # no console output
		@values["realtime-chunk-checksum"] = true # validate chunk of data while downloading if checksums are provided
	end

	def parse()
		@values.clear
		default_values
		begin
			file = File.open(@filename)
			while !file.eof and line = file.readline
				next if line !~ /^([a-zA-Z0-9-]+)=(.*)$/
				key = $1
				val = $2.strip
				if val == "true"
					val = true
				elsif val == "false"
					val = false
				elsif val =~ /^[0-9]+$/
					val = val.to_i
				elsif val =~ /^[0-9]+\.[0-9]+$/
					val = val.to_f
				end
				@values[key] = val
			end
			file.close
		rescue
		end
	end

	public

	def [](key)
		@values[key]
	end

	def []=(key, val)
		@values[key] = val
	end

	def filename=(filename)
		@filename = filename
		parse
	end

	def save()
		begin
			file = File.open(@filename, "w")
			@values.each do |key,val|
				file.puts("#{key}=#{val}")
			end
			file.close
		rescue
			p $!
		end
	end

end



if __FILE__ == $0
	aria2 = Eat::Aria2.new
	exit if !aria2.is_connected
	puts "Version: aria2 "+aria2.version
	#aria2.add_uri(["http://dlc.sun.com/torrents/info/osol-0906-x86.iso.torrent"])
	begin
		aria2.tell_active.each do |res|
			gid = res["gid"]
			puts " *** Set download max rate to 16KB ***"
			aria2.change_options(gid, "max-download-limit" => "16000")
			puts " *** Get status ***"
			pp aria2.tell_status(gid)
			puts " *** Get uris ***"
			pp aria2.get_uris(gid)
			puts " *** Get files ***"
			pp aria2.get_files(gid)
			puts " *** Get peers ***"
			pp aria2.get_peers(gid).length
		end
		sleep 5
	end while true
end

