#!/usr/bin/ruby -w

require "gtk2"
require "pango"

require "config"
require "eataria2"
require "eatsettings"

include Gtk



module Eat
end



class Eat::Manager

	def initialize()
		# Setup aria2 listener
		@aria2 = Eat::Aria2Listener.instance
		@aria2.signal_connect("connected") { set_sensitive(true) }
		@aria2.signal_connect("disconnected") { set_sensitive(false) }
		@aria2.signal_connect("download_status") { |this, gid| update_row(gid) }
		@aria2.signal_connect("download_completed") { |this, gid| update_row(gid) }
		@aria2.signal_connect("download_removed") { |this, gid| update_row(gid) }

		# Retrieve GUI widgets
		builder = Builder.new
		builder << "manager.ui"

		@window = builder["window-manager"]
		@treeview = builder["treeview-manager"]
		@liststore = builder["liststore-manager"]
		@statusbar = builder["statusbar-manager"]

		@newdl_dialog = builder["dialog-new-download"]
		@custom_uri = builder["custom-uri"]
		@select_file = builder["select-file"]
		@select_file_dialog = builder["filechooserdialog-new-download"]

		@action_add = builder["action-add"]
		@action_pause = builder["action-pause"]
		@action_resume = builder["action-resume"]
		@action_remove = builder["action-remove"]

		# Setup actions
		@action_add.signal_connect("activate") { action_add }
		@action_pause.signal_connect("activate") { action_pause }
		@action_resume.signal_connect("activate") { action_resume }
		@action_remove.signal_connect("activate") { action_remove }
		builder["action-settings"].signal_connect("activate") { show_settings_dialog }
		builder["action-quit"].signal_connect("activate") { action_quit }
		builder["action-about"].signal_connect("activate") { show_about_dialog }

		# Setup window
		@window.signal_connect('delete-event') { action_quit }

		# Setup info bar
=begin
		# ruby-gnome2-0.19.3 doesn't know about Gtk::InfoBar
		# The info bar will be used at least in one case where the connection gets
		# refused. It will provide a visible message with an action button to manually
		# try to reconnect.
		@infobar = InfoBar.new
		@infobar.set_no_show_all
		@infolabel = Label.new
		@infobar.add(@infolabel)
		hbox = builder["hbox-info-bar"]
		hbox.add(@infobar)
=end

		# Setup tree view
		cell = CellRendererText.new
		cell.ellipsize = Pango::ELLIPSIZE_END
		column = builder["treeviewcolumn-completed"]
		column.pack_start(cell, true)
		column.add_attribute(cell, :text, 8)
		cell = CellRendererText.new
		cell.ellipsize = Pango::ELLIPSIZE_END
		column = builder["treeviewcolumn-filepath"]
		column.pack_start(cell, true)
		column.add_attribute(cell, :text, 9)

		# Setup statusbar
		@statuscontext = { "main" => @statusbar.get_context_id("main context"),
				"menu" => @statusbar.get_context_id("menu context") }
		@statusbar.push(@statuscontext["main"], "#{PACKAGE_STRING} - aria2 #{@aria2.version}")

#		builder["menuitem-add"].signal_connect("select") do
#			@statusbar.push(@statuscontext["menu"], "Add a new download")
#		end
#		builder["menuitem-add"].signal_connect("deselect") do
#			@statusbar.pop(@statuscontext["menu"])
#		end

		# Set initial sensitivity
		set_sensitive(@aria2.is_connected)

		# Setup new download dialog
		file_filter = builder["filefilter-new-download"]
		file_filter.name = "Torrents, Metalinks"
		file_filter.add_pattern("*.torrent")
		file_filter.add_pattern("*.metalink")
		@select_file_dialog.filter = file_filter
		@select_file.signal_connect('clicked') do
			@select_file_dialog.unselect_all
			res = @select_file_dialog.run
			@custom_uri.text = @select_file_dialog.filename if res == Gtk::Dialog::RESPONSE_ACCEPT
			@select_file_dialog.hide
		end
	end

	def show()
		@window.show_all
	end

	def set_sensitive(sensitive)
		@treeview.set_sensitive(sensitive)
		@action_add.set_sensitive(sensitive)
		@action_pause.set_sensitive(sensitive)
		@action_resume.set_sensitive(sensitive)
		@action_remove.set_sensitive(sensitive)
		@statusbar.push(@statuscontext["main"], "#{PACKAGE_STRING} - aria2 #{@aria2.version}")
	end

	def update_row(gid)
		row_iter = nil
		gid_i = gid.to_i

		result = @aria2.tell_status(gid)
		return if result.empty?

		@liststore.each do |model, path, iter|
			next unless iter.get_value(0) == gid_i
			row_iter = iter
			break
		end

		if !row_iter
			# Avoid adding rows with incomplete information
			return if result["totalLength"] == "0" or result["status"] != "active"

			# Add unknown active download to liststore
			row_iter = @liststore.append
			@liststore.set_value(row_iter, 0, gid_i)
			uris = @aria2.get_uris(gid)
			p uris
			begin
				# TODO Torrent downloads don't have a URI
				@liststore.set_value(row_iter, 10, uris[0]["uri"])
			rescue
			end
		end

		case result["status"]
		when "active" then
			@liststore.set_value(row_iter, 1, result["connections"].to_i)
			@liststore.set_value(row_iter, 2, result["completedLength"].to_i)
			@liststore.set_value(row_iter, 3, result["uploadLength"].to_i)
			@liststore.set_value(row_iter, 4, result["totalLength"].to_i)
			@liststore.set_value(row_iter, 5, result["downloadSpeed"].to_i)
			@liststore.set_value(row_iter, 6, result["uploadSpeed"].to_i)
			@liststore.set_value(row_iter, 7, result["infoHash"])

			completed = result["completedLength"].to_i
			total = result["totalLength"].to_i
			percent = total > 0 ? 100 * completed / total : 0
			@liststore.set_value(row_iter, 8, percent)

			result = @aria2.get_files(gid.to_s)
			if result != nil and !result[0]["path"].empty?
				@liststore.set_value(row_iter, 9, File.basename(result[0]["path"]))
			end
		when "complete" then
			@liststore.set_value(row_iter, 8, 100)
			# Update the name, useful for very small files for which
			# this callback didn't run with the "active" state.
			result = @aria2.get_files(gid.to_s)
			if result != nil and !result[0]["path"].empty?
				@liststore.set_value(row_iter, 9, File.basename(result[0]["path"]))
			end
		when "removed" then
			# TODO mark as stopped/inactive
		when "error" then
			@liststore.set_value(row_iter, 8, -1)
		end
	end

	def show_about_dialog()
		logo = nil
		begin
			logo = Gdk::Pixbuf.new("/usr/share/pixmaps/eatmonkey-logo.png")
		rescue
		begin
			logo = Gdk::Pixbuf.new("/usr/local/share/pixmaps/eatmonkey-logo.png")
		rescue
		end
		end
		AboutDialog.show(@window,
				"program-name" => "Eatmonkey",
				"version" => PACKAGE_VERSION,
				"copyright" => "Copyright 2010 \xC2\xA9 Mike Massonnet",
				"logo" => logo,
				"comments" => "Stupid download manager for monkeys and Capuchins!" \
				"\n\n" \
				"Eatmonkey is a download manager that works exclusively with " \
				"aria2, the ultra fast download utility. It has support for " \
				"HTTP(s)/FTP, BitTorrent and Metalink files.")
	end

	def show_settings_dialog()
		used_local_server = @aria2.use_local_server?

		dialog = Eat::SettingsDialog.new(@window)
		res = dialog.run

		if res == Dialog::RESPONSE_OK

			settings = Eat::Settings.instance

			if settings["custom-server"]
				# Switch to custom server
				debug("switch to custom server")
				@aria2.use_custom_server(settings["xmlrpc-host"], settings["xmlrpc-port"],
						settings["xmlrpc-user"], settings["xmlrpc-passwd"])
				@aria2.shutdown if used_local_server # TODO prevent stopping unfinished downloads
				@aria2.connect(true)
			else
				# Switch to local server
				if used_local_server
					# TODO need to store active downloads to effectively restart them
					# and avoid gid conflicts between old and new/readded downloads
					res = 0
					count = @aria2.tell_active.count
					if count > 0
						dialog = MessageDialog.new(@window,
								Dialog::DESTROY_WITH_PARENT | Dialog::NO_SEPARATOR,
								MessageDialog::QUESTION, MessageDialog::BUTTONS_YES_NO,
								"Restart the aria2 XML-RPC Server?")
						dialog.secondary_text = "To take the new changes into account aria2 has to be " \
								"restarted, this will stop/resume the current downloads."
						res = dialog.run
						dialog.destroy
					end
					if res == Dialog::RESPONSE_YES or count == 0
						debug("restart local server")
						@aria2.shutdown
						@aria2.connect(true)
					end
				else
					debug("switch to local server")
					@aria2.use_local_server
					@aria2.connect(true)
				end
			end

		end
	end

	private

	def action_quit()
		@aria2.shutdown # TODO make this optional
		main_quit
	end

	def action_add()
		@custom_uri.text = ""
		@custom_uri.grab_focus
		res = @newdl_dialog.run
		@newdl_dialog.hide
		uri = @custom_uri.text
		if res == Dialog::RESPONSE_OK and !uri.empty?
			# TODO check if it is a uri or a torrent/metalink file and use the right
			# method addUri/addTorrent/addMetalink
			puts "download file %s" % uri
			gid = @aria2.add_uri(uri)
			if gid != nil
				puts "gid: %s" % gid

				# Add new row to liststore
				row_iter = @liststore.append
				@liststore.set_value(row_iter, 0, gid.to_i)
				@liststore.set_value(row_iter, 9, File.basename(uri))
				@liststore.set_value(row_iter, 10, uri)
			end
		end
	end

	def action_pause()
		@treeview.selection.selected_each do |model, path, iter|
			gid = iter.get_value(0)
			@aria2.remove(gid.to_s)
		end
	end

	def action_resume()
		@treeview.selection.selected_each do |model, path, iter|
			gid = iter.get_value(0).to_s
			status = @aria2.tell_status(gid)
			case status["status"]
			when "removed"
				# Restart the download queued at position 0 and delete current row
				# as new gid will be created
				uri = iter.get_value(10)
				if !uri.empty?
					@aria2.add_uri(uri, nil, 0)
					@liststore.remove(iter)
				end
			end
		end
	end

	def action_remove()
		@treeview.selection.selected_each do |model, path, iter|
			gid = iter.get_value(0)
			@aria2.remove(gid.to_s)
			@liststore.remove(iter)
		end
	end

end



if __FILE__ == $0
	manager = Eat::Manager.new
	manager.show
	main
end

