#!/usr/bin/ruby -w

require "gtk2"
require "pango"

require "config"
require "eataria2"
require "eatsettings"

include Gtk



module Eat
end



class Eat::Manager

	def initialize()
		# Store manually removed downloads to avoid readding them into the tree store
		@removed_downloads = Array.new

		# Setup aria2 listener
		@aria2 = Eat::Aria2Listener.instance
		@aria2.signal_connect("connected") {
			@removed_downloads.clear
			update_newdl_dialog
			set_sensitive(true)
			@infobar.hide_all
			@infobar.set_no_show_all(true)
		}
		@aria2.signal_connect("disconnected") {
			set_sensitive(false)
			if !@aria2.use_local_server?
				@infobar.set_no_show_all(false)
				@infobar.show_all
			end
		}
		@aria2.signal_connect("download_status") { |this, gid| update_row(gid) }
		@aria2.signal_connect("download_completed") { |this, gid| update_row(gid) }
		@aria2.signal_connect("download_removed") { |this, gid| update_row(gid) }

		# Retrieve GUI widgets
		builder = Builder.new
		builder << "manager.ui"

		@window = builder["window-manager"]
		@treeview = builder["treeview-manager"]
		@liststore = builder["liststore-manager"]
		@statusbar = builder["statusbar-manager"]

		@newdl_dialog = builder["dialog-new-download"]
		@file_uri = builder["file-uri"]
		@file_select_file = builder["file-select-file"]
		@file_select_file_dialog = builder["filechooserdialog-new-download"]
		@file_max_download_speed = builder["file-max-download-speed"]
		@file_split = builder["file-split"]
		@hbox_download_dir = builder["hbox-download-dir"]
		@file_download_dir = builder["file-download-dir-button"]
		@file_max_upload_speed = builder["file-max-upload-speed"]
		@file_seed_ratio = builder["file-seed-ratio"]

		@action_add = builder["action-add"]
		@action_pause = builder["action-pause"]
		@action_resume = builder["action-resume"]
		@action_remove = builder["action-remove"]
		@action_settings = builder["action-settings"]

		# Setup actions
		@action_add.signal_connect("activate") { action_add }
		@action_pause.signal_connect("activate") { action_pause }
		@action_resume.signal_connect("activate") { action_resume }
		@action_remove.signal_connect("activate") { action_remove }
		@action_settings.signal_connect("activate") { show_settings_dialog }
		builder["action-quit"].signal_connect("activate") { action_quit }
		builder["action-about"].signal_connect("activate") { show_about_dialog }

		# Setup window
		@window.signal_connect('delete-event') { action_quit }

		# Setup info bar
=begin
		# ruby-gnome2-0.19.3 doesn't know about Gtk::InfoBar
		# The info bar will be used at least in one case where the connection gets
		# refused. It will provide a visible message with an action button to manually
		# try to reconnect.
		@infobar = InfoBar.new
=end
		@infobar = HBox.new(false, 4)
		@infobar.set_no_show_all(true)
		infobutton = Button.new("Reconnect")
		infobutton.signal_connect('clicked') { @aria2.connect(true) }
		@infobar.pack_end(infobutton, false, false, 0)
		infolabel = Label.new("Disconnected from aria2")
		@infobar.pack_end(infolabel, false, false, 0)
		infoimg = Image.new(Stock::DIALOG_ERROR, IconSize::MENU)
		@infobar.pack_end(infoimg, false, false, 0)
		hbox = builder["hbox-info-bar"]
		hbox.add(@infobar)

		# Setup tree view
		cell = CellRendererText.new
		cell.ellipsize = Pango::ELLIPSIZE_END
		column = builder["treeviewcolumn-completed"]
		column.pack_start(cell, true)
		column.add_attribute(cell, :text, 8)
		cell = CellRendererText.new
		cell.ellipsize = Pango::ELLIPSIZE_END
		column = builder["treeviewcolumn-filepath"]
		column.pack_start(cell, true)
		column.add_attribute(cell, :text, 9)

		# Setup DND on tree view
		Drag.dest_set(@treeview, Drag::DEST_DEFAULT_ALL, nil, Gdk::DragContext::ACTION_COPY)
		Drag.dest_add_uri_targets(@treeview)
		@treeview.signal_connect("drag-data-received") { |this, context, x, y, selection_data, info, time|
			selection_data.uris.each do |uri|
				add_new_download(uri)
			end
			Drag.finish(context, true, false, time)
		}

		# Setup statusbar
		@statuscontext = { "main" => @statusbar.get_context_id("main context"),
				"menu" => @statusbar.get_context_id("menu context") }
		@statusbar.push(@statuscontext["main"], "#{PACKAGE_STRING} - aria2 #{@aria2.version}")

#		builder["menuitem-add"].signal_connect("select") do
#			@statusbar.push(@statuscontext["menu"], "Add a new download")
#		end
#		builder["menuitem-add"].signal_connect("deselect") do
#			@statusbar.pop(@statuscontext["menu"])
#		end

		# Set initial sensitivity
		set_sensitive(@aria2.is_connected)

		# Setup new download dialog
		file_filter = builder["filefilter-new-download"]
		file_filter.name = "Torrents, Metalinks"
		file_filter.add_pattern("*.torrent")
		file_filter.add_pattern("*.metalink")
		@file_select_file_dialog.filter = file_filter
		@file_select_file.signal_connect('clicked') do
			@file_select_file_dialog.unselect_all
			res = @file_select_file_dialog.run
			@file_uri.text = @file_select_file_dialog.filename if res == Gtk::Dialog::RESPONSE_ACCEPT
			@file_select_file_dialog.hide
		end
	end

	def show()
		@window.show_all
	end

	def set_sensitive(sensitive)
		# Set/Unset sensitivity of widgets that need aria2
		@treeview.set_sensitive(sensitive)
		@action_add.set_sensitive(sensitive)
		@action_pause.set_sensitive(sensitive)
		@action_resume.set_sensitive(sensitive)
		@action_remove.set_sensitive(sensitive)
		@statusbar.push(@statuscontext["main"], "#{PACKAGE_STRING} - aria2 #{@aria2.version}")
	end

	def update_row(gid)
		row_iter = nil
		gid_i = gid.to_i

		# Find gid in removed downloads
		return if @removed_downloads.include? gid_i

		# Retrieve status information on gid
		status = @aria2.tell_status(gid)
		return if status.empty?

		# Find gid in model
		@liststore.each do |model, path, iter|
			next unless iter[0] == gid_i
			row_iter = iter
			break
		end

		# Add inexistent gid
		if !row_iter
			# Avoid adding rows with incomplete information
			return if status["totalLength"] == "0" or status["status"] != "active"

			# Add unknown active download to liststore
			row_iter = @liststore.append
			row_iter[0] = gid_i
			uris = @aria2.get_uris(gid)
			p "uris:", uris
			begin
				# TODO Torrent downloads don't have a URI -- try to read from parent
				row_iter[11] = uris[0]["uri"]
			rescue
			end
		end

		# Update status of gid in the model
		case status["status"]
		when "active" then
			row_iter[1] = status["connections"].to_i
			row_iter[2] = status["completedLength"].to_i
			row_iter[3] = status["uploadLength"].to_i
			row_iter[4] = status["totalLength"].to_i
			row_iter[5] = status["downloadSpeed"].to_i
			row_iter[6] = status["uploadSpeed"].to_i
			row_iter[7] = status["infoHash"]

			completed = status["completedLength"].to_i
			total = status["totalLength"].to_i
			percent = total > 0 ? 100 * completed / total : 0
			row_iter[8] = percent

			set_download_name_for_iter(row_iter, status)
		when "complete" then
			row_iter[8] = 100
			# Update the name, useful for very small files for which
			# this callback didn't run with the "active" state.
			set_download_name_for_iter(row_iter, status)
		when "removed" then
			# TODO mark as stopped/inactive
		when "error" then
			row_iter[8] = -1
		end
	end

	def show_about_dialog()
		logo = nil
		begin
			logo = Gdk::Pixbuf.new("/usr/share/pixmaps/eatmonkey-logo.png")
		rescue
		begin
			logo = Gdk::Pixbuf.new("/usr/local/share/pixmaps/eatmonkey-logo.png")
		rescue
		end
		end
		AboutDialog.show(@window,
				"program-name" => "Eatmonkey",
				"version" => PACKAGE_VERSION,
				"copyright" => "Copyright 2010 \xC2\xA9 Mike Massonnet",
				"logo" => logo,
				"comments" => "Stupid download manager for monkeys and Capuchins!" \
				"\n\n" \
				"Eatmonkey is a download manager that works exclusively with " \
				"aria2, the ultra fast download utility. It has support for " \
				"HTTP(s)/FTP, BitTorrent and Metalink files.")
	end

	def show_settings_dialog()
		used_local_server = @aria2.use_local_server?

		dialog = Eat::SettingsDialog.new(@window)
		res = dialog.run

		if res == Dialog::RESPONSE_OK

			settings = Eat::Settings.instance
			update_newdl_dialog
			set_sensitive(false)

			if settings["custom-server"]
				# Switch to custom server
				debug("switch to custom server")
				@aria2.use_custom_server(settings["xmlrpc-host"], settings["xmlrpc-port"],
						settings["xmlrpc-user"], settings["xmlrpc-passwd"])
				@aria2.shutdown if used_local_server # TODO prevent stopping unfinished downloads
				@aria2.connect(true)
			else
				# Switch to local server
				if used_local_server
					# TODO need to store active downloads to effectively restart them
					# and avoid gid conflicts between old and new/readded downloads
					res = 0
					count = @aria2.tell_active.count
					if count > 0
						dialog = MessageDialog.new(@window,
								Dialog::DESTROY_WITH_PARENT | Dialog::NO_SEPARATOR,
								MessageDialog::QUESTION, MessageDialog::BUTTONS_YES_NO,
								"Restart the aria2 XML-RPC Server?")
						dialog.secondary_text = "To take the new changes into account aria2 has to be " \
								"restarted, this will stop/resume the current downloads."
						res = dialog.run
						dialog.destroy
					end
					if res == Dialog::RESPONSE_YES or count == 0
						debug("restart local server")
						@aria2.shutdown
						@aria2.connect(true)
					else
						# If the server is not restarted directly then reset the control buttons sensitive
						set_sensitive(true)
					end
				else
					debug("switch to local server")
					@aria2.use_local_server
					@aria2.connect(true)
				end
			end

		end
	end

	def add_new_download(uri=nil)
		update_newdl_dialog
		@file_uri.text = uri == nil ? "" : uri
		@file_uri.grab_focus
		res = @newdl_dialog.run
		@newdl_dialog.hide
		uri = @file_uri.text
		if res == Dialog::RESPONSE_OK and !uri.empty?
			# TODO check if it is a uri or a torrent/metalink file and use the right
			# method addUri/addTorrent/addMetalink
			puts "download file %s" % uri
			options = {
				"max-download-limit" => @file_max_download_speed.value_as_int.to_s,
				"split" => @file_split.value_as_int.to_s,
				"max-upload-limit" => @file_max_upload_speed.value_as_int.to_s,
				"seed-ratio" => @file_seed_ratio.value.to_s,
			}
			options["dir"] = @file_download_dir.current_folder if @aria2.use_local_server?
			gid = @aria2.add_uri(uri, options)
			if gid != nil
				puts "gid: %s" % gid
				# Add new row to liststore
				row_iter = @liststore.append
				row_iter[0] = gid.to_i
				row_iter[9] = File.basename(uri)
				row_iter[11] = uri
			end
		end
	end

	private

	def set_download_name_for_iter(iter, status)
		return if iter[10] == true
		name = get_download_name(status)
		if name != nil
			iter[9] = name
			iter[10] = true
		end
	end

	def get_download_name(status)
		return if status == nil

		# Use the bittorrent array
		if status['bittorrent'] != nil
			return status['bittorrent']['info']['name']
		# Use the files array
		elsif status['files'] != nil and !status['files'][0]['path'].empty?
			return File.basename(status['files'][0]['path'])
		# Fallback on the aria2.getFile XML-RPC method for older versions of aria2
		else
			result = @aria2.get_files(status['gid'])
			if result != nil and !result[0]["path"].empty?
				return File.basename(result[0]["path"])
			end
		end
	end

	def update_newdl_dialog()
		settings = Eat::Settings.instance
		@file_max_download_speed.value = settings.aria2["max-download-limit"].to_i
		@file_split.value = settings.aria2["split"]
		if @aria2.use_local_server?
			@hbox_download_dir.show_all
			@file_download_dir.current_folder = settings.aria2["dir"]
		else
			@hbox_download_dir.hide_all
		end
		@file_max_upload_speed.value = settings.aria2["max-upload-limit"].to_i
		@file_seed_ratio.value = settings.aria2["seed-ratio"].to_f
	end

	def action_quit()
		# TODO make this optional e.g. keep active downloads running
		set_sensitive(false)
		@action_settings.set_sensitive(false)
		@aria2.signal_connect("shutdown") { main_quit }
		@aria2.shutdown
	end

	def action_add()
		add_new_download
	end

	def action_pause()
		# Remove gid from aria2 but keep them in model
		@treeview.selection.selected_each do |model, path, iter|
			gid = iter[0]
			@aria2.remove(gid.to_s)
		end
	end

	def action_resume()
		# Resume gid that have been removed
		@treeview.selection.selected_each do |model, path, iter|
			gid = iter[0].to_s
			status = @aria2.tell_status(gid)
			case status["status"]
			when "removed"
				# Restart the download queued at position 0 and delete current row
				# as new gid will be created
				uri = iter[11]
				if !uri.empty?
					percent = iter[8]
					@liststore.remove(iter)
					gid = @aria2.add_uri(uri, nil, 0)
					if gid != nil
						# TODO this is a little better than nothing, when
						# the download is resumed it will reappear immediately
						# but with incomplete information
						iter = @liststore.append
						iter[0] = gid.to_i
						iter[8] = percent
						iter[9] = File.basename(uri)
						iter[11] = uri
					end
				end
			end
		end
	end

	def action_remove()
		@treeview.selection.selected_each do |model, path, iter|
			gid = iter[0]
			@aria2.remove(gid.to_s)
			@liststore.remove(iter)
			@removed_downloads << gid
		end
	end

end



if __FILE__ == $0
	manager = Eat::Manager.new
	manager.show
	main
end

