/*
 *  ghal
 *
 *  Copyright (c) 2007 Brian Tarricone <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <dbus/dbus-glib-lowlevel.h>

#include "ghal-device.h"
#include "ghal-context.h"
#include "ghal-private.h"
#include "ghal-enum-types.h"
#include "ghal-marshal.h"

struct _GHalDevicePrivate
{
    GHalContext *context;
    gchar *udi;
};

enum
{
    SIG_NEW_CAPABILITY = 0,
    SIG_LOST_CAPABILITY,
    SIG_PROPERTY_CHANGED,
    SIG_CONDITION,
    N_SIGS,
};

static void ghal_device_class_init(GHalDeviceClass *klass);
static void ghal_device_init(GHalDevice *device);

static void ghal_device_finalize(GObject *obj);

static guint ghd_signals[N_SIGS] = { 0, };


G_DEFINE_TYPE(GHalDevice, ghal_device, G_TYPE_OBJECT)


static void
ghal_device_class_init(GHalDeviceClass *klass)
{
    GObjectClass *object_class = (GObjectClass *)klass;
    
    g_type_class_add_private(klass, sizeof(GHalDevicePrivate));
    
    object_class->finalize = ghal_device_finalize;
    
    ghd_signals[SIG_NEW_CAPABILITY] = g_signal_new("new-capability",
                                                   GHAL_TYPE_DEVICE,
                                                   G_SIGNAL_RUN_LAST,
                                                   G_STRUCT_OFFSET(GHalDeviceClass,
                                                                   new_capability),
                                                   NULL, NULL,
                                                   g_cclosure_marshal_VOID__STRING,
                                                   G_TYPE_NONE, 1,
                                                   G_TYPE_STRING);
    
    ghd_signals[SIG_LOST_CAPABILITY] = g_signal_new("lost-capability",
                                                    GHAL_TYPE_DEVICE,
                                                    G_SIGNAL_RUN_LAST,
                                                    G_STRUCT_OFFSET(GHalDeviceClass,
                                                                    lost_capability),
                                                    NULL, NULL,
                                                    g_cclosure_marshal_VOID__STRING,
                                                    G_TYPE_NONE, 1,
                                                    G_TYPE_STRING);
    
    ghd_signals[SIG_PROPERTY_CHANGED] = g_signal_new("property-changed",
                                                     GHAL_TYPE_DEVICE,
                                                     G_SIGNAL_RUN_LAST,
                                                     G_STRUCT_OFFSET(GHalDeviceClass,
                                                                     property_changed),
                                                     NULL, NULL,
                                                     ghal_marshal_VOID__STRING_ENUM,
                                                     G_TYPE_NONE, 2,
                                                     G_TYPE_STRING,
                                                     GHAL_TYPE_PROPERTY_CHANGE_TYPE);
    
    ghd_signals[SIG_CONDITION] = g_signal_new("condition",
                                              GHAL_TYPE_DEVICE,
                                              G_SIGNAL_RUN_LAST,
                                              G_STRUCT_OFFSET(GHalDeviceClass,
                                                              condition),
                                              NULL, NULL,
                                              ghal_marshal_VOID__STRING_STRING,
                                              G_TYPE_NONE, 2,
                                              G_TYPE_STRING, G_TYPE_STRING);
}

static void
ghal_device_init(GHalDevice *device)
{
    device->priv = G_TYPE_INSTANCE_GET_PRIVATE(device, GHAL_TYPE_DEVICE,
                                                GHalDevicePrivate);
}

static void
ghal_device_finalize(GObject *obj)
{
    GHalDevice *device = GHAL_DEVICE(obj);
    
    g_free(device->priv->udi);
    
    G_OBJECT_CLASS(ghal_device_parent_class)->finalize(obj);
}



/*
 * internal private stuff
 */

void
_ghal_device_set_context(GHalDevice *device,
                         GHalContext *context)
{
    device->priv->context = context;
}

GHalContext *
_ghal_device_peek_context(GHalDevice *device)
{
    return device->priv->context;
}

void
_ghal_device_set_udi(GHalDevice *device,
                     const gchar *udi)
{
    device->priv->udi = g_strdup(udi);
}

void
_ghal_device_new_capability(GHalDevice *device,
                            const gchar *capability)
{
    g_signal_emit(G_OBJECT(device), ghd_signals[SIG_NEW_CAPABILITY], 0,
                  capability);
}

void
_ghal_device_lost_capability(GHalDevice *device,
                             const gchar *capability)
{
    g_signal_emit(G_OBJECT(device), ghd_signals[SIG_LOST_CAPABILITY], 0,
                  capability);
}

void
_ghal_device_property_changed(GHalDevice *device,
                              const gchar *property,
                              GhalPropertyChangeType change_type)
{
    g_signal_emit(G_OBJECT(device), ghd_signals[SIG_PROPERTY_CHANGED], 0,
                  property, change_type);
}

void
_ghal_device_condition(GHalDevice *device,
                       const gchar *condition_name,
                       const gchar *condition_detail)
{
    g_signal_emit(G_OBJECT(device), ghd_signals[SIG_CONDITION], 0,
                  condition_name, condition_detail);
}



/*
 * public api
 */

const gchar *
ghal_device_peek_udi(GHalDevice *device)
{
    g_return_val_if_fail(GHAL_IS_DEVICE(device), NULL);
    return device->priv->udi;
}

gboolean
ghal_device_exists(GHalDevice *device,
                   GError **error)
{
    LibHalContext *hal_ctx = _ghal_context_peek_libhal_context(device->priv->context);
    gboolean ret = FALSE;
    DBusError derror;
    
    dbus_error_init(&derror);
    ret = libhal_device_exists(hal_ctx, device->priv->udi, &derror);
    if(dbus_error_is_set(&derror)) {
        if(error)
            dbus_set_g_error(error, &derror);
    }
    dbus_error_free(&derror);
    
    return ret;
}

gboolean
ghal_device_property_exists(GHalDevice *device,
                            const gchar *property,
                            GError **error)
{
    LibHalContext *hal_ctx = _ghal_context_peek_libhal_context(device->priv->context);
    gboolean ret = FALSE;
    DBusError derror;
    
    dbus_error_init(&derror);
    ret = libhal_device_property_exists(hal_ctx, device->priv->udi, property,
                                        &derror);
    if(dbus_error_is_set(&derror)) {
        if(error)
            dbus_set_g_error(error, &derror);
    }
    dbus_error_free(&derror);
    
    return ret;
}

#define GHAL_GET_PROPERTY_TMPL(rettype, nametype) \
rettype \
ghal_device_get_property_##nametype(GHalDevice *device, \
                                    const gchar *property, \
                                    GError **error) \
{ \
    LibHalContext *hal_ctx = _ghal_context_peek_libhal_context(device->priv->context); \
    rettype ret; \
    DBusError derror; \
    \
    dbus_error_init(&derror); \
    ret = libhal_device_get_property_##nametype(hal_ctx, \
                                                device->priv->udi, \
                                                property, \
                                                &derror); \
    if(dbus_error_is_set(&derror)) { \
        if(error) \
            dbus_set_g_error(error, &derror); \
    } \
    dbus_error_free(&derror); \
    \
    return ret; \
}

GHAL_GET_PROPERTY_TMPL(gint, int)
GHAL_GET_PROPERTY_TMPL(guint64, uint64)
GHAL_GET_PROPERTY_TMPL(gdouble, double)
GHAL_GET_PROPERTY_TMPL(gboolean, bool)
#undef GHAL_GET_PROPERTY_TMPL

/* string and strlist need to be handled differently */

gchar *
ghal_device_get_property_string(GHalDevice *device,
                                const gchar *property,
                                GError **error)
{
    LibHalContext *hal_ctx = _ghal_context_peek_libhal_context(device->priv->context);
    gchar *ret = NULL;
    char *tmp;
    DBusError derror;
    
    dbus_error_init(&derror);
    tmp = libhal_device_get_property_string(hal_ctx,
                                            device->priv->udi,
                                            property,
                                            &derror);
    if(dbus_error_is_set(&derror)) {
        if(error)
            dbus_set_g_error(error, &derror);
    }
    dbus_error_free(&derror);
    
    if(tmp) {
        ret = g_strdup(tmp);
        libhal_free_string(tmp);
    }
    
    return ret;
}

GList *
ghal_device_get_property_strlist(GHalDevice *device,
                                 const gchar *property,
                                 GError **error)
{
    LibHalContext *hal_ctx = _ghal_context_peek_libhal_context(device->priv->context);
    GList *ret = NULL;
    gchar **tmp;
    DBusError derror;
    
    dbus_error_init(&derror);
    tmp = libhal_device_get_property_strlist(hal_ctx,
                                             device->priv->udi,
                                             property,
                                             &derror);
    if(dbus_error_is_set(&derror)) {
        if(error)
            dbus_set_g_error(error, &derror);
    }
    dbus_error_free(&derror);
    
    if(tmp) {
        gint i;
        for(i = 0; tmp[i]; ++i)
            ret = g_list_prepend(ret, g_strdup(tmp[i]));
        libhal_free_string_array(tmp);
    }
    
    return g_list_reverse(ret);
}

#define GHAL_SET_PROPERTY_TMPL(settype, nametype) \
gboolean \
ghal_device_set_property_##nametype(GHalDevice *device, \
                                    const gchar *property, \
                                    settype value, \
                                    GError **error) \
{ \
    LibHalContext *hal_ctx = _ghal_context_peek_libhal_context(device->priv->context); \
    gboolean ret; \
    DBusError derror; \
    \
    dbus_error_init(&derror); \
    ret = libhal_device_set_property_##nametype(hal_ctx, \
                                                device->priv->udi, \
                                                property, \
                                                value, \
                                                &derror); \
    if(dbus_error_is_set(&derror)) { \
        if(error) \
            dbus_set_g_error(error, &derror); \
    } \
    dbus_error_free(&derror); \
    \
    return ret; \
}

GHAL_SET_PROPERTY_TMPL(const gchar *, string)
GHAL_SET_PROPERTY_TMPL(gint, int)
GHAL_SET_PROPERTY_TMPL(guint64, uint64)
GHAL_SET_PROPERTY_TMPL(gdouble, double)
GHAL_SET_PROPERTY_TMPL(gboolean, bool)
#undef GHAL_SET_PROPERTY_TMPL

#define GHAL_SET_STRLIST_TMPL(op, settype) \
gboolean \
ghal_device_property_strlist_##op(GHalDevice *device, \
                                 const gchar *property, \
                                 settype value, \
                                 GError **error) \
{ \
    LibHalContext *hal_ctx = _ghal_context_peek_libhal_context(device->priv->context); \
    gboolean ret; \
    DBusError derror; \
    \
    dbus_error_init(&derror); \
    ret = libhal_device_property_strlist_##op(hal_ctx, \
                                             device->priv->udi, \
                                             property, \
                                             value, \
                                             &derror); \
    if(dbus_error_is_set(&derror)) { \
        if(error) \
            dbus_set_g_error(error, &derror); \
    } \
    dbus_error_free(&derror); \
    \
    return ret; \
}
GHAL_SET_STRLIST_TMPL(append, const gchar *)
GHAL_SET_STRLIST_TMPL(prepend, const gchar *)
GHAL_SET_STRLIST_TMPL(remove_index, guint)
GHAL_SET_STRLIST_TMPL(remove, const gchar *)
#undef GHAL_SET_STRLIST_TMPL

gboolean
ghal_device_remove_property(GHalDevice *device,
                            const gchar *property,
                            GError **error)
{
    LibHalContext *hal_ctx = _ghal_context_peek_libhal_context(device->priv->context);
    gboolean ret = FALSE;
    DBusError derror;
    
    dbus_error_init(&derror);
    ret = libhal_device_remove_property(hal_ctx, device->priv->udi, property,
                                        &derror);
    if(dbus_error_is_set(&derror)) {
        if(error)
            dbus_set_g_error(error, &derror);
    }
    dbus_error_free(&derror);
    
    return ret;
}

gboolean
ghal_device_query_capability(GHalDevice *device,
                             const gchar *capability,
                             GError **error)
{
    LibHalContext *hal_ctx = _ghal_context_peek_libhal_context(device->priv->context);
    gboolean ret = FALSE;
    DBusError derror;
    
    dbus_error_init(&derror);
    ret = libhal_device_query_capability(hal_ctx, device->priv->udi,
                                         capability, &derror);
    if(dbus_error_is_set(&derror)) {
        if(error)
            dbus_set_g_error(error, &derror);
    }
    dbus_error_free(&derror);
    
    return ret;
}

DBusGProxy *
ghal_device_get_dbus_proxy(GHalDevice *device,
                           const gchar *dbus_interface)
{
    DBusGConnection *dbus_gconn = _ghal_context_peek_dbus_g_connection(device->priv->context);
    
    g_return_val_if_fail(dbus_gconn && dbus_interface, NULL);
    
    return dbus_g_proxy_new_for_name(dbus_gconn,
                                     "org.freedesktop.Hal",
                                     device->priv->udi,
                                     dbus_interface);
}

guint
ghal_device_hash(gconstpointer device)
{
    return g_str_hash(GHAL_DEVICE(device)->priv->udi);
}

gboolean
ghal_device_equal(gconstpointer a,
                  gconstpointer b)
{
    return g_str_equal(GHAL_DEVICE(a)->priv->udi, GHAL_DEVICE(b)->priv->udi);
}

gint
ghal_device_compare(gconstpointer a,
                    gconstpointer b)
{
    return strcmp(GHAL_DEVICE(a)->priv->udi, GHAL_DEVICE(b)->priv->udi);
}
