/*
 *  xfce4-volstatus-icon
 *
 *  Copyright (c) 2006 Brian Tarricone <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libxfcegui4/libxfcegui4.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include <exo-hal/exo-hal.h>

#include "xfce-volstatus-common.h"

gboolean
xfce_volstatus_remove_drive(GHalDrive *drive)
{
    GList *volumes, *l;
    gboolean ret = TRUE;
    GError *error = NULL;
    gint exit_status = 0;
    
    g_return_val_if_fail(GHAL_IS_DRIVE(drive), FALSE);
    
    volumes = ghal_drive_list_volumes(drive);
    for(l = volumes; l; l = l->next) {
        GHalVolume *volume = l->data;
        
        if(ghal_volume_is_mounted(volume)) {
            gchar *udi_quoted = g_shell_quote(ghal_device_peek_udi(GHAL_DEVICE(volume)));
            gchar *cmd = g_strdup_printf("exo-unmount -n -h %s", udi_quoted);
            
            ret = g_spawn_command_line_sync(cmd, NULL, NULL, &exit_status,
                                            &error);
            
            g_free(udi_quoted);
            g_free(cmd);
            
            if(!ret || exit_status)
                break;
        }
    }
    g_list_foreach(volumes, (GFunc)g_object_unref, NULL);
    g_list_free(volumes);
    
    if(ghal_drive_requires_eject(drive)) {
        gchar *udi_quoted = g_shell_quote(ghal_device_peek_udi(GHAL_DEVICE(drive)));
        gchar *cmd = g_strdup_printf("exo-eject -n -h %s", udi_quoted);
        
        ret = g_spawn_command_line_sync(cmd, NULL, NULL, &exit_status, &error);
        
        g_free(udi_quoted);
        g_free(cmd);
    }
    
    if(!ret) {
        gchar *name = ghal_drive_get_display_name(drive);
        gchar *primary = g_strdup_printf(_("Unable to remove \"%s\""), name);
        xfce_message_dialog(NULL, _("Removal Failure"),
                            GTK_STOCK_DIALOG_WARNING, primary,
                            error ? error->message : _("Unknown error"),
                            GTK_STOCK_CLOSE, GTK_RESPONSE_ACCEPT, NULL);
        g_free(name);
        g_free(primary);
        if(error)
            g_error_free(error);
    }
    
    return ret;
}

const gchar *
xfce_volstatus_get_icon_name()
{
    const gchar *icon_name = NULL;
    GtkIconTheme *icon_theme = gtk_icon_theme_get_default();
    
    if(gtk_icon_theme_has_icon(icon_theme, "drive-removable-media"))
        icon_name = "drive-removable-media";
    else if(gtk_icon_theme_has_icon(icon_theme, "gnome-dev-removable"))
        icon_name = "gnome-dev-removable";
    
    return icon_name;
}
