/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif	/* !HAVE_CONFIG_H */

#include <gtk/gtk.h>

#include <libxfce4util/i18n.h>

#include "input_page.h"

static void input_page_class_init(InputPageClass *);
static void input_page_init(InputPage *);
static void input_page_finalize(GObject *);

/* signal handlers */
static void toggled_cb(GtkWidget *, InputPage *);
static void valuechanged_cb(GtkWidget *, InputPage *);

static GtkWidgetClass *parent_class = NULL;

GtkType
input_page_get_type(void)
{
	static GtkType input_page_type = 0;

	if (!input_page_type) {
		static const GTypeInfo input_page_info = {
			sizeof(InputPageClass),
			NULL,
			NULL,
			(GClassInitFunc)input_page_class_init,
			NULL,
			NULL,
			sizeof(InputPage),
			0,
			(GInstanceInitFunc)input_page_init
		};

		input_page_type = g_type_register_static(GTK_TYPE_VBOX,
			"InputPage", &input_page_info, 0);
	}

	return(input_page_type);
}

static void
input_page_class_init(InputPageClass *klass)
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS(klass);
	object_class->finalize = input_page_finalize;

	parent_class = gtk_type_class(gtk_vbox_get_type());
}

static void
input_page_init(InputPage *pg)
{
	GtkWidget *label;
	GtkWidget *hbox;

	gtk_container_set_border_width(GTK_CONTAINER(pg), 6);
	gtk_box_set_spacing(GTK_BOX(pg), 4);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(pg), hbox, FALSE, TRUE, 0);

	label = gtk_label_new(_("Encoding:"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);

	pg->encoding = gtk_combo_new();
	gtk_widget_show(pg->encoding);
	gtk_box_pack_start(GTK_BOX(hbox), pg->encoding, TRUE, TRUE, 0);

	pg->all = gtk_radio_button_new_with_label(NULL, _("Print all pages"));
	gtk_widget_show(pg->all);
	g_signal_connect(G_OBJECT(pg->all), "toggled",
		G_CALLBACK(toggled_cb), pg);
	gtk_box_pack_start(GTK_BOX(pg), pg->all, FALSE, TRUE, 0);

	pg->range = gtk_radio_button_new_with_label_from_widget(
		GTK_RADIO_BUTTON(pg->all), _("Print pages in range from"));
	gtk_widget_show(pg->range);
	gtk_box_pack_start(GTK_BOX(pg), pg->range, FALSE, TRUE, 0);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(pg), hbox, FALSE, TRUE, 0);

	pg->from = gtk_spin_button_new_with_range(1.0, 1000.0, 1.0);
	gtk_widget_set_sensitive(pg->from, FALSE);
	gtk_widget_show(pg->from);
	g_signal_connect(G_OBJECT(pg->from), "value-changed",
		G_CALLBACK(valuechanged_cb), pg);
	gtk_box_pack_start(GTK_BOX(hbox), pg->from, FALSE, TRUE, 0);

	label = gtk_label_new("to");
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);

	pg->to = gtk_spin_button_new_with_range(1.0, 1000.0, 1.0);
	gtk_widget_set_sensitive(pg->to, FALSE);
	gtk_widget_show(pg->to);
	g_signal_connect(G_OBJECT(pg->to), "value-changed",
		G_CALLBACK(valuechanged_cb), pg);
	gtk_box_pack_start(GTK_BOX(hbox), pg->to, FALSE, TRUE, 0);

	pg->cut = gtk_check_button_new_with_label(_("Cut long lines"));
	gtk_widget_show(pg->cut);
	gtk_box_pack_start(GTK_BOX(pg), pg->cut, FALSE, TRUE, 0);

	pg->interpret = gtk_check_button_new_with_label(
		_("Interpret TAB, BS and FF characters"));
	gtk_widget_show(pg->interpret);
	gtk_box_pack_start(GTK_BOX(pg), pg->interpret, FALSE, TRUE, 0);

	pg->binary = gtk_check_button_new_with_label(
			_("Force binary printing"));
	gtk_widget_show(pg->binary);
	gtk_box_pack_start(GTK_BOX(pg), pg->binary, FALSE, TRUE, 0);
}

static void
input_page_finalize(GObject *object)
{
	g_return_if_fail(object != NULL);
	g_return_if_fail(INPUT_IS_PAGE(object));

	G_OBJECT_CLASS(parent_class)->finalize(object);
}

GtkWidget *
input_page_new(void)
{
	InputPage *pg;
	GList *list;

	pg = INPUT_PAGE(g_object_new(input_page_get_type(), NULL));

	if ((list = xfce_option_list(encodings)) != NULL) {
		gtk_combo_set_popdown_strings(GTK_COMBO(pg->encoding), list);
		g_list_free(list);
	}

	return(GTK_WIDGET(pg));
}

void
input_page_set_settings(InputPage *pg, const XfPrintSettingsInput *ip)
{
	g_return_if_fail(pg != NULL);
	g_return_if_fail(ip != NULL);

	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(pg->encoding)->entry),
		xfce_option_alias(encodings, ip->encoding));

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(
			ip->all ? pg->all : pg->range), TRUE);

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(pg->from), ip->from);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(pg->to), ip->to);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pg->cut), ip->cut);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pg->interpret),
		ip->interpret);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pg->binary), ip->binary);
}

void
input_page_get_settings(InputPage *pg, XfPrintSettingsInput *ip)
{
	g_return_if_fail(pg != NULL);
	g_return_if_fail(ip != NULL);

	ip->encoding = xfce_option_name(encodings, gtk_entry_get_text(GTK_ENTRY(
				GTK_COMBO(pg->encoding)->entry)));

	ip->all = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pg->all));
	ip->from = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(pg->from));
	ip->to = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(pg->to));
	
	ip->cut = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pg->cut));
	ip->interpret = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
			pg->interpret));
	ip->binary = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
			pg->binary));
}

static void
toggled_cb(GtkWidget *btn, InputPage *pg)
{
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pg->all))) {
		gtk_widget_set_sensitive(pg->from, FALSE);
		gtk_widget_set_sensitive(pg->to, FALSE);
	}
	else {
		gtk_widget_set_sensitive(pg->from, TRUE);
		gtk_widget_set_sensitive(pg->to, TRUE);
	}
}

static void
valuechanged_cb(GtkWidget *btn, InputPage *pg)
{
	gint from;
	gint to;

	from = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(pg->from));
	to = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(pg->to));

	if (from <= to)
		return;

	if (btn == pg->from)
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(pg->to), from);
	else if (btn == pg->to)
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(pg->from), to);
}

