/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* !HAVE_CONFIG_H */

#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <glib.h>
#include <libxfce4util/i18n.h>
#include <libxfcegui4/dialogs.h>

#include <filter.h>
#include <xfprinter.h>
#include <xfprintsettings.h>

static XfceFilter *a2ps_filter(const XfPrintSettings *, const gchar *);

gboolean
xfprint_toprinter(const XfPrinter *pr, const gchar *ifile,
		  const XfPrintSettings *settings)
{
	XfceFilterList *filters;
	XfceFilter *filter;
	gboolean ret;
	gchar *lp;

	filters = xfce_filterlist_new();

	if (pr->postscript) {
		filter = a2ps_filter(settings, ifile);
		xfce_filterlist_append(filters, filter);

		if (settings->sheets.reverse) {
			filter = xfce_filter_new("psselect");
			xfce_filter_add(filter, "-r");
			xfce_filterlist_append(filters, filter);
		}
	}

	if ((lp = g_find_program_in_path("lp")) != NULL) {
		filter = xfce_filter_new(lp);
		xfce_filter_add(filter, "-d");
		xfce_filter_add(filter, "%s", pr->name);
		xfce_filter_add(filter, "-n");
		xfce_filter_add(filter, "%d", settings->copies);
	}
	else if ((lp = g_find_program_in_path("lpr")) != NULL) {
		filter = xfce_filter_new(lp);
		xfce_filter_add(filter, "-P%s", pr->name);
		xfce_filter_add(filter, "-#%d", settings->copies);
	}
	else {
		show_error(
			_("Neither lp nor lpr was found in PATH, therefore "
			"xfprint was unable to print the file."));
		xfce_filterlist_free(filters);
		return(FALSE);
	}

	if (!pr->postscript)
		xfce_filter_add(filter, "%s", ifile);
	xfce_filterlist_append(filters, filter);

	ret = !xfce_filterlist_execute(filters, STDIN_FILENO, STDOUT_FILENO,
		STDERR_FILENO);

	xfce_filterlist_free(filters);

	if (!ret)
		show_error(_("Unable to sent job to printer"));
	else
		show_info(_("Job sent to printer"));

	return(ret);
}

gboolean
xfprint_tofile(const gchar *ofile, const gchar *ifile,
	       const XfPrintSettings *settings)
{
	XfceFilterList *filters;
	XfceFilter *filter;
	gboolean ret;
	int output;

	filters = xfce_filterlist_new();
	ret = FALSE;

#ifdef O_EXLOCK
	if ((output = open(ofile, O_CREAT|O_EXLOCK|O_TRUNC|O_WRONLY, 0750)) < 0)
#else
	if ((output = open(ofile, O_CREAT | O_TRUNC | O_WRONLY, 0750)) < 0)
#endif
		goto fail;

	filter = a2ps_filter(settings, ifile);
	xfce_filterlist_append(filters, filter);

	if (settings->sheets.reverse) {
		filter = xfce_filter_new("psselect");
		xfce_filter_add(filter, "-r");
		xfce_filterlist_append(filters, filter);
	}

	ret = !xfce_filterlist_execute(filters, STDIN_FILENO, output,
		STDERR_FILENO);

	(void)close(output);

fail:
	xfce_filterlist_free(filters);

	if (!ret)
		show_error(_("Unable to print to file"));
	else
		show_info(_("Job sent to file"));

	return(ret);
}

static XfceFilter *
a2ps_filter(const XfPrintSettings *settings, const gchar *ifile)
{
	XfceFilter *filter;

	filter = xfce_filter_new("a2ps");

	/* sheet settings */
	xfce_filter_add(filter, "--medium=%s", settings->sheets.papersize);
	if (settings->sheets.landscape)
		xfce_filter_add(filter, "--landscape");
	else
		xfce_filter_add(filter, "--portrait");
	xfce_filter_add(filter, "--columns=%d", settings->sheets.cols);
	xfce_filter_add(filter, "--rows=%d", settings->sheets.rows);
	if (settings->sheets.fillcols)
		xfce_filter_add(filter, "--major=columns");
	else
		xfce_filter_add(filter, "--major=rows");
	if (settings->sheets.borders)
		xfce_filter_add(filter, "--borders=yes");
	else
		xfce_filter_add(filter, "--borders=no");

	/* virtual pages settings */
	xfce_filter_add(filter, "--line-numbers=%d",
		settings->vpages.linenumbers);
	if (settings->vpages.lpp > 0)
		xfce_filter_add(filter, "--lines-per-page=%d",
			settings->vpages.lpp);
	if (settings->vpages.cpl > 0)
		xfce_filter_add(filter, "--chars-per-line=%d",
			settings->vpages.cpl);
	xfce_filter_add(filter, "--tabsize=%d", settings->vpages.tabsize);
	xfce_filter_add(filter, "--non-printable-format=%s",
		settings->vpages.nonprtfmt);

	/* pretty print settings */
	if (strcmp(settings->pprint.language, "__auto__") != 0)
		xfce_filter_add(filter, "--pretty-print=%s",
			settings->pprint.language);
	xfce_filter_add(filter, "--highlight-level=%s",
		settings->pprint.highlight);
	xfce_filter_add(filter, "--strip-level=%d", settings->pprint.strip);

	/* input settings */
	xfce_filter_add(filter, "--encoding=%s", settings->input.encoding);
	if (!settings->input.all) {
		xfce_filter_add(filter, "--pages=%d-%d", settings->input.from,
			settings->input.to);
	}
	if (settings->input.cut)
		xfce_filter_add(filter, "--truncate-lines=yes");
	else
		xfce_filter_add(filter, "--truncate-lines=no");
	if (settings->input.interpret)
		xfce_filter_add(filter, "--interpret=yes");
	else
		xfce_filter_add(filter, "--interpret=no");
	if (settings->input.binary)
		xfce_filter_add(filter, "--print-anyway=yes");
	else
		xfce_filter_add(filter, "--print-anyway=no");

	/* headings settings */
	if (!settings->headings.headers) {
		xfce_filter_add(filter, "--no-header");
	}
	else {
		xfce_filter_add(filter,"--header=%s",settings->headings.header);
		xfce_filter_add(filter, "--underlay=%s",
			settings->headings.underlay);
		xfce_filter_add(filter, "--center-title=%s",
			settings->headings.ctitle);
		xfce_filter_add(filter, "--left-title=%s",
			settings->headings.ltitle);
		xfce_filter_add(filter, "--right-title=%s",
			settings->headings.rtitle);
		xfce_filter_add(filter, "--left-footer=%s",
			settings->headings.lfooter);
		xfce_filter_add(filter, "--footer=%s",
			settings->headings.cfooter);
		xfce_filter_add(filter, "--right-footer=%s",
			settings->headings.rfooter);
	}

	xfce_filter_add(filter, "--output=-");
	xfce_filter_add(filter, "%s", ifile);

	return(filter);
}

