/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif	/* !HAVE_CONFIG_H */

#include <gtk/gtk.h>

#include <libxfce4util/i18n.h>

#include "vpages_page.h"

static void vpages_page_class_init(VpagesPageClass *);
static void vpages_page_init(VpagesPage *);
static void vpages_page_finalize(GObject *);

/* signal handlers */
static void checked_cb(GtkWidget *, VpagesPage *);

static GtkWidgetClass *parent_class = NULL;

GtkType
vpages_page_get_type(void)
{
	static GtkType vpages_page_type = 0;

	if (!vpages_page_type) {
		static const GTypeInfo vpages_page_info = {
			sizeof(VpagesPageClass),
			NULL,
			NULL,
			(GClassInitFunc)vpages_page_class_init,
			NULL,
			NULL,
			sizeof(VpagesPage),
			0,
			(GInstanceInitFunc)vpages_page_init
		};

		vpages_page_type = g_type_register_static(GTK_TYPE_TABLE,
			"VpagesPage", &vpages_page_info, 0);
	}

	return(vpages_page_type);
}

static void
vpages_page_class_init(VpagesPageClass *klass)
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS(klass);
	object_class->finalize = vpages_page_finalize;

	parent_class = gtk_type_class(gtk_table_get_type());
}

static void
vpages_page_init(VpagesPage *pg)
{
	GtkWidget *label;

	gtk_container_set_border_width(GTK_CONTAINER(pg), 6);
	gtk_table_resize(GTK_TABLE(pg), 6, 2);
	gtk_table_set_col_spacings(GTK_TABLE(pg), 4);
	gtk_table_set_row_spacings(GTK_TABLE(pg), 4);
	gtk_table_set_homogeneous(GTK_TABLE(pg), FALSE);

	pg->clineno = gtk_check_button_new_with_label(_("Line numbers:"));
	gtk_widget_show(pg->clineno);
	g_signal_connect(G_OBJECT(pg->clineno), "toggled",
		G_CALLBACK(checked_cb), pg);
	gtk_table_attach(GTK_TABLE(pg), pg->clineno, 0, 1, 0, 1,
		GTK_FILL, 0, 0, 0);

	pg->lineno = gtk_spin_button_new_with_range(1.0, 100.0, 1.0);
	gtk_widget_set_sensitive(pg->lineno, FALSE);
	gtk_widget_show(pg->lineno);
	gtk_table_attach(GTK_TABLE(pg), pg->lineno, 1, 2, 0, 1, 
		GTK_EXPAND | GTK_FILL, 0, 0, 0);

	pg->clpp = gtk_check_button_new_with_label(_("Lines per page:"));
	gtk_widget_show(pg->clpp);
	g_signal_connect(G_OBJECT(pg->clpp), "toggled",
		G_CALLBACK(checked_cb), pg);
	gtk_table_attach(GTK_TABLE(pg), pg->clpp, 0, 1, 1, 2,
		GTK_FILL, 0, 0, 0);

	pg->lpp = gtk_spin_button_new_with_range(1.0, 100.0, 1.0);
	gtk_widget_set_sensitive(pg->lpp, FALSE);
	gtk_widget_show(pg->lpp);
	gtk_table_attach(GTK_TABLE(pg), pg->lpp, 1, 2, 1, 2,
		GTK_EXPAND | GTK_FILL, 0, 0, 0);

	pg->ccpl = gtk_check_button_new_with_label(_("Chars per line:"));
	gtk_widget_show(pg->ccpl);
	g_signal_connect(G_OBJECT(pg->ccpl), "toggled",
		G_CALLBACK(checked_cb), pg);
	gtk_table_attach(GTK_TABLE(pg), pg->ccpl, 0, 1, 2, 3,
		GTK_FILL, 0, 0, 0);

	pg->cpl = gtk_spin_button_new_with_range(1.0, 100.0, 1.0);
	gtk_widget_set_sensitive(pg->cpl, FALSE);
	gtk_widget_show(pg->cpl);
	gtk_table_attach(GTK_TABLE(pg), pg->cpl, 1, 2, 2, 3,
		GTK_EXPAND | GTK_FILL, 0, 0, 0);

	label = gtk_label_new(_("Tabulation size:"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_widget_show(label);
	gtk_table_attach(GTK_TABLE(pg), label, 0, 1, 3, 4,
		GTK_FILL, 0, 0, 0);

	pg->tabsize = gtk_spin_button_new_with_range(1.0, 32.0, 1.0);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(pg->tabsize), 8.0);
	gtk_widget_show(pg->tabsize);
	gtk_table_attach(GTK_TABLE(pg), pg->tabsize, 1, 2, 3, 4,
		GTK_EXPAND | GTK_FILL, 0, 0, 0);

	label = gtk_label_new(_("Non-printable format:"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_widget_show(label);
	gtk_table_attach(GTK_TABLE(pg), label, 0, 1, 4, 5,
		GTK_FILL, 0, 0, 0);

	pg->nprfmt = gtk_combo_new();
	gtk_widget_show(pg->nprfmt);
	gtk_table_attach(GTK_TABLE(pg), pg->nprfmt, 1, 2, 4, 5,
		GTK_EXPAND | GTK_FILL, 0, 0, 0);
}

static void
vpages_page_finalize(GObject *object)
{
	g_return_if_fail(object != NULL);
	g_return_if_fail(VPAGES_IS_PAGE(object));
	
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

GtkWidget *
vpages_page_new(void)
{
	VpagesPage *pg;
	GList *list;

	pg = VPAGES_PAGE(g_object_new(vpages_page_get_type(), NULL));

	if ((list = xfce_option_list(non_printable_fmts)) != NULL) {
		gtk_combo_set_popdown_strings(GTK_COMBO(pg->nprfmt), list);
		g_list_free(list);
	}

	return(GTK_WIDGET(pg));
}

void
vpages_page_set_settings(VpagesPage *pg, const XfPrintSettingsVirtualPages *vp)
{
	g_return_if_fail(pg != NULL);
	g_return_if_fail(vp != NULL);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pg->clineno),
		(vp->linenumbers > 0));
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(pg->lineno), vp->linenumbers);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pg->clpp),
		(vp->lpp > 0));
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(pg->lpp), vp->lpp);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pg->ccpl),
		(vp->cpl > 0));
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(pg->cpl), vp->cpl);

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(pg->tabsize), vp->tabsize);
	
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(pg->nprfmt)->entry),
		xfce_option_alias(non_printable_fmts, vp->nonprtfmt));
}

void
vpages_page_get_settings(VpagesPage *pg, XfPrintSettingsVirtualPages *vp)
{
	g_return_if_fail(pg != NULL);
	g_return_if_fail(vp != NULL);

	vp->linenumbers = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
			pg->clineno))
		? gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(pg->lineno))
		: 0;

	vp->lpp = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pg->clpp))
		? gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(pg->lpp))
		: 0;

	vp->cpl = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pg->ccpl))
		? gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(pg->cpl))
		: 0;

	vp->tabsize = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(
			pg->tabsize));

	vp->nonprtfmt = xfce_option_name(non_printable_fmts,
		gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(pg->nprfmt)->entry)));
}

static void
checked_cb(GtkWidget *btn, VpagesPage *pg)
{
	gboolean active;
	
	active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(btn));

	if (btn == pg->clineno)
		gtk_widget_set_sensitive(pg->lineno, active);
	else if (btn == pg->clpp)
		gtk_widget_set_sensitive(pg->lpp, active);
	else if (btn == pg->ccpl)
		gtk_widget_set_sensitive(pg->cpl, active);
}

