/*
 * Program name:  equake
 * Version:	  1.03.6
 * Source file:	  equake_dat.h  
 * Description:	  Equake pulls data about earthquakes and displays it in interesting ways
 *
 * Copyright (C) 2012-2014 Jeroen van Aart
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */


#define QUAKESERVER "earthquake.usgs.gov" /* should never change unless hostname changes */
#define PORTNO 80
#define PORTNAME "http"
#define QUAKEPATH_WW "/earthquakes/eventpage/" /* path to worldwide earthquakes */
#define QUAKEPATH_AF "#summary" /* is last part of url */
#define QUAKEPATH_DYFI_BE "/earthquakes/eventpage/" /* path to did you feel it maps */
#define QUAKEPATH_DYFI_AF "#dyfi" /* path to did you feel it maps */
#define QUAKEFILE_HOURLY "earthquakes/feed/v1.0/summary/1.0_hour.csv"
#define QUAKEFILE_DAILY "earthquakes/feed/v1.0/summary/1.0_day.csv"
#define QUAKEFILE_WEEKLY "earthquakes/feed/v1.0/summary/1.0_week.csv"
/* 
 * the format string is the string that comes before the actual data
 * it's used to search for the start of data and to see if data is valid
 */
#define FORMATSTRING "time,latitude,longitude,depth,mag,magType,nst,gap,dmin,rms,net,id,updated,place"

/* maximum amount of earthquakes that can be remembered */
#define MAXQUAKES_HR 20
#define MAXQUAKES_DAY 1000
#define MAXQUAKES_WEEK 10000
/* maximum size the buffer will be, one can reasonably expect it to not overflow (yeah right) */
#define BUFFERSIZE 40960
#define BUFFERSIZEDAILY 409600
#define BUFFERSIZEWEEKLY 409600
/* max size of urls and tooltip text */
#define URLSIZE 512
/* max size of label text */
#define LABELSIZE 256
/* max size of http requests */
#define HTTPREQ 256
/* default time to poll for updates is 5 minutes */
#define POLLTIME 300
/* this decides whether to save data as signature file */
#define SAVESIG true
/* default location of signature */
#define SIGFILE "/.equake_sigfile.eq" /* this will be appeneded to user's home if found */
/* filename of config file, path will be derived elsewhere but normally is user's $HOME */
#define CONFIGFILE "/.equake_appletrc"
/* first string in config file to identify it */
#define CONFIGHEADER "equake_applet_cfg_v01.jva"
/* 
 * time out for tcp socket, set to 2 seconds so we don't have to wait too long since the call to recv is set to block 
 * it also appears to fix some weird bug or another where the whole applet would disappear from the panel
 */
#define SOCKTIMEOUT 2
#define SOCKTIMEOUTWEEK 4 /* doubled in case of weekly download */
/* the minimum magnitude to cause a special alert */
#define ALERTHEAVY 6
/* the minimum magnitude to make it bold in dialog */
#define DAILYHEAVY 5 
/* the minimum magnitude to store in the signature file, can be overridden through config file parameter */
#define SIGFILEMAG 4
/* minimum magnitude earthquake to monitor, default is 1 */
#define MONITORMAG 1
/* 
 * periods for which to pull data, daily and weekly, hourly is default but defined as 0 for clarity 
 * FIRST is a special case to reflect the start aof the applet and first time  use of functions to get and process data
 */
#define FIRST 10
#define HOURLY 0
#define DAILY 1
#define WEEKLY 2
/* defaul timezone string, normally is UTC */
#define TIMEZONE "UTC"
/* default text when busy */
#define BUSYTXT "Downloading and processing data..."
#define BUSYTXTLONG "Equake is downloading and processing data.\nThis should only take a few seconds..."
/* default text of label */
#define LABELTXT "Equake"
/* default text of the tooltip of the button that visualises the magnitude of the earthquake */
#define BUTTONTXT "Click to go to: http://earthquake.usgs.gov/"
/* title of hourly and daily recorded earthquakes window */
#define RECHOURTXT "Recorded earthquakes in the last hour"
#define RECDAYTXT "Recorded earthquakes in the last day"
#define RECWEEKTXT "Recorded earthquakes in the last week"
/* usgs website */
#define USGS_WEBSITE "http://earthquake.usgs.gov/"
/* map url, we'll put the proper coordinates and label in between */
#define MAPURL1 "maps.google.com/maps?q="
#define MAPURL2 ",+"
#define MAPURL3 "+("
#define MAPURL4 ")&amp;iwloc=A&amp;hl=en"

/* information for about window */
#define EQVERSION "1.03.6"
#define COPYRIGHT "©2012-2014 Jeroen van Aart"
#define LICENSE "GNU General Public License"
#define COMMENTS "Equake pulls data about earthquakes and displays it in interesting ways"
#define WEBSITE "http://www.e-quake.org"
#define WEBSITELABEL "equake plugin website"
#define AUTHOR "Jeroen van Aart <android@e-quake.org>"

#define PREFSTEXT "The configuration is saved in a file in your home directory (~/.equake_appletrc). You can also read the file for an explanation about how to configure equake.\n\nThe following options can be configured:\n\n- polltime in seconds, will only allow (2, 3, 4, 5, 10, 15, 20, 30) *60 seconds, any other value will default to 5*60 seconds\n\n- alert about high magnitude earthquakes, 1=yes, 0=no (if not set or anything else=yes)\n\n- minimum magnitude to display alert dialog, default is 6 (can be 3, 4, 5, 6, 7, 8, anything else defaults to 6)\n\n- minimum alert to store in signature file (~/.equake_sigfile.eq can be used with email client for example), default is 4 (can be 2, 3, 4, 5, 6, 7, 8, anything else defaults to 4, 0 turns it off)\n\n- minimum magnitude earthquake to monitor, default is 1 (can be 1, 2 or 3, anything else defaults to 1)\n\nThe program will automatically notice any changes."
#define PREFSWINDOW "Change equake preferences"
#define PREFPOLLTEXT "Polling Time (in minutes)"
#define PREFALERTEXT "Will alert"
#define PREFALERTHEAVYTEXT "If Yes, minimum magnitude to alert"
#define PREFSIGFILETEXT "Minimun magnitude to store in signature file"
#define PREFMONITORMAGTEXT "Minimum magnitude to monitor"

/* error message to be displayed when no data is available for whatever reason */
#define NODATA "No data available at this time"

/* error message to be displayed when failed to load a web browser */
#define NOBROWSER "Failed to load web browser"

/* 
 * earthquake.usgs.gov description:
 * http://earthquake.usgs.gov/earthquakes/feed/v1.0/csv.php
 *
 * Note: files are not inclusive (the past day file does not include 
 * past hour events, for example).
 */
struct Equake_Format /* structure to hold Earthquake data */
{
  char *version;
  char *datetime; /* Time when the event occurred. Times are reported in milliseconds since the epoch . In certain output formats, the date is formatted for readability. */
  char *datetime_local;
  char *lat; /* lattitude */
  char *lon; /* longtitude */
  char *magnitude;
  char *magnitude_type;
  char *depth;
  char *nst; /* number of reporting stations */
  char *gap; /* The largest azimuthal gap between azimuthally adjacent stations (in degrees). In general, the smaller this number, the more reliable is the calculated horizontal position of the earthquake. */
  char *dmin; /* Horizontal distance from the epicenter to the nearest station (in degrees). 1 degree is approximately 111.2 kilometers. In general, the smaller this number, the more reliable is the calculated depth of the earthquake. */
  char *rms; /* The root-mean-square (RMS) travel time residual, in sec, using all weights. This parameter provides a measure of the fit of the observed arrival times to the predicted arrival times for this location. Smaller numbers reflect a better fit of the data. */
  char *src; /* (source network) The ID of a data contributor. Identifies the network considered to be the preferred source of information for this event. */
  char *eqid; /* ID, assumed to be unique */
  char *updated; /* Time when the event was most recently updated. Times are reported in milliseconds since the epoch. In certain output formats, the date is formatted for readability. */
  char *region;
};


struct Equake_Data /* holds all data */ 
{
  int sockfd;
  int portno;
  int l_received;
  char portname[5];
  int properties_index[5]; /* these are the indexes of the property combo boxes */
  int poll_time;
  int will_alert; /* default set to 1, so we will by default display alert about high magnitude earthquakes */
  int alert_heavy; /* minimum magnitude upon which a alert dialog is displayed, default is 6 */
  int monitormag; /* minimum magnitude earthquake to monitor, default is 1 */
  int pos; /* number position of earthquake in array which is being displayed */
  int posperiod; /* period data that had to be loaded to find current earthquake, normally it's HOURLY */
  int quake_history_count;
  int quake_history_countdaily;
  int quake_history_countweekly;
  int sigfilemag; /* minimum magnitude displayed in signature file */
  char home[URLSIZE+1]; /* user's home directory, we guess URLSIZE should be large enough, we use strncpy() and strncat() ... */
  char configfile[URLSIZE+1];
  char sigfile[URLSIZE+1]; /* location of sigfile if HOME is found */
  char *report_prev_eqid;
  char *sigfile_prev_eqid;
  char quakeserver[256];
  char quakepath_us[256];
  char quakepath_ww[256];
  char quakepath_dyfi_be[128];
  char quakepath_dyfi_af[128];
  char quakefile_hourly[256];
  char quakefile_daily[256];
  char quakefile_weekly[256];
  char httprequest[HTTPREQ];
  char httprequestdaily[HTTPREQ];
  char httprequestweekly[HTTPREQ];
  char buffer[BUFFERSIZE];
  char bufferdaily[BUFFERSIZEDAILY];
  char bufferweekly[BUFFERSIZEWEEKLY];
  char formatstring[256];
  char tooltiptext_label[URLSIZE];
  char lbl[LABELSIZE]; /* text the label will hold */
  struct Equake_Format equakeformat[MAXQUAKES_HR];
  struct Equake_Format equakeformatday[MAXQUAKES_DAY];
  struct Equake_Format equakeformatweek[MAXQUAKES_WEEK];
  struct sockaddr_in server;
  struct hostent *he;
  GdkPixbuf *equake_about;
  XfcePanelPlugin *applet;
  GtkWidget *label;
  GtkWidget *logo;
  GtkWidget *prefsdialog;
  GtkWidget *helpdialog;
  GtkWidget *alert_heavy_dialog;
  GtkWidget *hbox;
}; 
