#include <signal.h>
#include <string.h>

#include <glib.h>
#include <glib-object.h>

#include <libxfcegui4/libxfcegui4.h>
#include <libxfcegui4/netk-screen.h>
#include <libxfcegui4/netk-window.h>
#include <libxfce4util/debug.h>

#include "ls_icon.h"

struct _LSIconPrivate 
{
  NetkScreen		*pScreen;
  NetkWindow		*pWindow;
  /* tooltips for the app name							*/
  GtkTooltips		*pToolTips;
  /* factory for the popup menu							*/
  GtkItemFactory	*pItemFactory;					
  /* a pointer to ((GtkWidget *)this)->style					*/
  GtkStyle		*pStyle;
  LSIconBox		*pIconBox;
  glong			lXID;
  gboolean		bIsVisible;
  gboolean		bFallbackIcon;
  gboolean		bApplyOwnColor;
  XfceIconbutton	*pButton;
  GdkColor		aaColor[LS_ICON_WIN_SIZE][LS_ICON_STATE_SIZE];
 
  /* popup menu for the window options						*/
  GtkWidget		*pMenu;
  
  struct _LSIconDisplay {
    LSDisplayIcon	sameWorkspace;
    LSDisplayIcon	otherWorkspace;
  } display;
  
  struct _LSIconPrivateSignals
  {
    gulong		active_workspace_changed;
    gulong		state_changed;
    gulong		clicked;
    gulong		event;
    gulong 		workspace_changed;
    gulong		icon_changed[2];
    gulong		style_set;
    gulong		size_allocate;
    gulong		name_changed;
  } signals;
};

static void 		ls_icon_init(LSIcon *plsi);
static void 		ls_icon_class_init(LSIconClass *plsic);
static void 		ls_icon_finalize(GObject *pgo);
static void 		ls_icon_set_tooltip(LSIcon *plsi);
static void 		ls_icon_hide(LSIcon *plsi);
static void 		ls_icon_show(LSIcon *plsi);
static void 		ls_icon_init_colors(LSIcon *plsi);
/* CALLBACKS									*/
static void		cb_lsi_active_workspace_changed(NetkScreen *pScreen,
    			  void *pVoid);
static void 		cb_lsi_window_state_changed(NetkWindow *pWin, 
    			  NetkWindowState netkwinOldState, 
			  NetkWindowState netkwinNewState, 
			  void *pVoid);
static void 		cb_lsi_button_clicked(GtkButton *pButton, gpointer *pUserData);
static gboolean 	cb_lsi_button_button2_pressed(GtkWidget *pWidget, 
    			  GdkEvent *pEvent, gpointer *pUserData);
static void 		cb_lsi_workspace_changed(NetkWindow *pWin, void *pVoid);
static void 		cb_lsi_icon_changed(NetkWindow *pWin, void *pVoid);
static void 		cb_lsi_name_changed(NetkWindow *pWin, void *pVoid);
static void		cb_lsi_realize(GtkWidget *pWidget, void *pUserData);
static void		cb_lsi_style_set(GtkWidget *pWidget,GtkStyle *pOldStyle,
    			  void *pUserData);
  
static gpointer parent_class = NULL;

/********************************************************************************/
/* GLIB OBJECT FUNCTIONS							*/
/********************************************************************************/
GType ls_icon_get_type()
{
  static GType ls_icon_type = 0;

  if(!ls_icon_type)
  {
    static const GTypeInfo ls_icon_info = {
      sizeof(LSIconClass),
      (GBaseInitFunc)NULL,
      (GBaseFinalizeFunc)NULL,
      (GClassInitFunc)ls_icon_class_init,
      NULL,				/* class_finalize */
      NULL,				/* class_data */
      sizeof(LSIcon),
      0,				/* n_preallocs */
      (GInstanceInitFunc)ls_icon_init
    };

    ls_icon_type = g_type_register_static(/* GTK_TYPE_BUTTON */
      GTK_TYPE_EVENT_BOX, "LSIcon", &ls_icon_info, 0);
  }

  return ls_icon_type;
}

static void ls_icon_class_init(LSIconClass *plsic) 
{
  GObjectClass *object_class = G_OBJECT_CLASS(plsic);
  
  object_class			= G_OBJECT_CLASS(plsic);
  parent_class 			= gtk_type_class(GTK_TYPE_EVENT_BOX);
  object_class->finalize 	= ls_icon_finalize;

  return;
}

static void ls_icon_init(LSIcon *plsi) 
{
  return;
}

static void ls_icon_finalize(GObject *pgo) 
{
  LSIcon *plsi;

  g_return_if_fail(pgo!=NULL);
  g_return_if_fail(LS_IS_ICON(pgo));

  plsi = LS_ICON(pgo);

  g_signal_handler_disconnect(G_OBJECT(plsi->priv->pScreen), 
    plsi->priv->signals.active_workspace_changed);

  if(ls_icon_get_window(plsi)!=NULL)
  {
    g_signal_handler_disconnect(G_OBJECT(ls_icon_get_window(plsi)), 
      plsi->priv->signals.state_changed);
  }

  gtk_object_destroy(GTK_OBJECT(plsi->priv->pToolTips));
  if(plsi->priv->pMenu!=NULL)
    gtk_object_destroy(GTK_OBJECT(plsi->priv->pMenu));
  if(plsi->priv->pItemFactory!=NULL)
    gtk_object_destroy(GTK_OBJECT(plsi->priv->pItemFactory));
  g_free(plsi->priv);
  plsi->priv = NULL;
  
  G_OBJECT_CLASS(parent_class)->finalize(pgo);

  return;
}

/********************************************************************************/
/* CONSTRUCTORS									*/
/********************************************************************************/
LSIcon *ls_icon_new(NetkWindow *pWin, NetkScreen *pScreen) 
{
  LSIcon *plsi;
  
  /* INIT */
  plsi 				= LS_ICON(g_object_new(LS_TYPE_ICON, NULL));
  plsi->priv			= (LSIconPrivate *)g_malloc(sizeof(LSIconPrivate));
  plsi->priv->pWindow		= pWin;
  plsi->priv->pScreen		= pScreen;
  plsi->priv->lXID		= netk_window_get_xid(plsi->priv->pWindow);
  plsi->priv->pToolTips		= gtk_tooltips_new();
  plsi->priv->pToolTips->delay	= 0;
  plsi->priv->pButton		= XFCE_ICONBUTTON(xfce_iconbutton_new());
  plsi->priv->pStyle		= NULL;
/*
  plsi->priv->pStyle		= gtk_widget_get_default_style();
 */
  plsi->priv->bApplyOwnColor    = FALSE;
  plsi->priv->pItemFactory	= NULL;
  plsi->priv->pMenu		= NULL;

  /* default settings								*/
  plsi->priv->display.otherWorkspace.minimized   	= TRUE;
  plsi->priv->display.otherWorkspace.notMinimized	= TRUE;
  plsi->priv->display.otherWorkspace.skipTasklist      	= FALSE;
  plsi->priv->display.sameWorkspace.minimized          	= TRUE;
  plsi->priv->display.sameWorkspace.notMinimized	= TRUE;
  plsi->priv->display.sameWorkspace.skipTasklist       	= FALSE;

  gtk_tooltips_enable(plsi->priv->pToolTips);

  gtk_container_set_border_width(GTK_CONTAINER(plsi), 0);

  gtk_button_set_relief(GTK_BUTTON(plsi->priv->pButton), GTK_RELIEF_NONE);
  gtk_container_set_border_width(GTK_CONTAINER(plsi->priv->pButton), 0);

  if(ls_icon_re_evaluate_icon(plsi)==FALSE)
    plsi->priv->bFallbackIcon=TRUE;
  else
    plsi->priv->bFallbackIcon=FALSE;
  
  plsi->priv->signals.clicked =
    g_signal_connect_object(G_OBJECT(plsi->priv->pButton), "clicked",
    G_CALLBACK(cb_lsi_button_clicked), plsi, 0);
  plsi->priv->signals.event =
    g_signal_connect_object(G_OBJECT(plsi->priv->pButton), "event", 
    G_CALLBACK(cb_lsi_button_button2_pressed), (gpointer)plsi, 0);
  
  gtk_container_add(GTK_CONTAINER(plsi), GTK_WIDGET(plsi->priv->pButton));

  gtk_widget_show(GTK_WIDGET(plsi->priv->pButton));
  
  ls_icon_set_tooltip(plsi);
  /* skip creating the menu, is it really usefull or just memory waste
  ls_icon_create_menu(plsi);
   */
  plsi->priv->signals.icon_changed[0] =
    g_signal_connect_object(G_OBJECT(plsi->priv->pWindow), 
    "icon_changed", G_CALLBACK(cb_lsi_icon_changed), plsi, 0);
  plsi->priv->signals.workspace_changed =
    g_signal_connect_object(G_OBJECT(
    ls_icon_get_window(plsi)),
    "workspace_changed", G_CALLBACK(cb_lsi_workspace_changed), plsi, 0);
  plsi->priv->signals.name_changed =
    g_signal_connect_object(G_OBJECT(
    ls_icon_get_window(plsi)),
    "name_changed", G_CALLBACK(cb_lsi_name_changed), plsi, 0);
  plsi->priv->signals.icon_changed[1] =
    g_signal_connect_object(G_OBJECT(
    netk_window_get_application(ls_icon_get_window(plsi))),
    "icon_changed", G_CALLBACK(cb_lsi_icon_changed), plsi, 0);
  plsi->priv->signals.active_workspace_changed =
    g_signal_connect_object(G_OBJECT(plsi->priv->pScreen), 
    "active_workspace_changed", 
    G_CALLBACK(cb_lsi_active_workspace_changed), plsi, 0);  
  plsi->priv->signals.state_changed =
    g_signal_connect_object(G_OBJECT(pWin), "state_changed",
    G_CALLBACK(cb_lsi_window_state_changed), plsi, 0);
  /*  
  recursion error!!!
   */
  plsi->priv->signals.style_set =
    g_signal_connect_object(G_OBJECT(plsi), "style-set",
    G_CALLBACK(cb_lsi_style_set), plsi, 0);
  g_signal_connect_object(G_OBJECT(plsi), "realize",
    G_CALLBACK(cb_lsi_realize), plsi, 0);
/*
  plsi->priv->signals.size_allocate =
    g_signal_connect_object(G_OBIECT(plsi), "size-allocate", 
    G_CALLBACK(cb_lsi_size_allocate), plsi, 0);
  ls_icon_init_colors(plsi);
*/  
  
  return plsi;
}

/********************************************************************************/
/* FUNCTIONS                                                                    */
/********************************************************************************/ 
void ls_icon_init_colors(LSIcon *plsi)
{
  GdkColor	*pC;
 
  if(ls_icon_get_parent(plsi)!=NULL)
  {
    TRACE("use parent style");
    plsi->priv->pStyle		= gtk_widget_get_style(GTK_WIDGET(
 				    ls_icon_get_parent(plsi)));
  }
  else if (plsi->priv->pStyle==NULL)
  {
    TRACE("ls_icong_get_parent()==NULL use own style, may brake style change!");
    plsi->priv->pStyle		= gtk_widget_get_style(GTK_WIDGET(
 				    plsi->priv->pButton));
  }
 
  /* default color								*/
  pC = &(plsi->priv->aaColor[LS_ICON_WIN_DEFAULT][LS_ICON_STATE_NORMAL]);
  pC->red = plsi->priv->pStyle->bg[GTK_STATE_NORMAL].red;
  pC->blue = plsi->priv->pStyle->bg[GTK_STATE_NORMAL].blue;
  pC->green = plsi->priv->pStyle->bg[GTK_STATE_NORMAL].green;

  pC = &(plsi->priv->aaColor[LS_ICON_WIN_DEFAULT][LS_ICON_STATE_PRELIGHT]);
  pC->red = plsi->priv->pStyle->bg[GTK_STATE_PRELIGHT].red;
  pC->blue = plsi->priv->pStyle->bg[GTK_STATE_PRELIGHT].blue;
  pC->green = plsi->priv->pStyle->bg[GTK_STATE_PRELIGHT].green;
  
  pC = &(plsi->priv->aaColor[LS_ICON_WIN_DEFAULT][LS_ICON_STATE_ACTIVE]);
  pC->red = plsi->priv->pStyle->bg[GTK_STATE_ACTIVE].red;
  pC->blue = plsi->priv->pStyle->bg[GTK_STATE_ACTIVE].blue;
  pC->green = plsi->priv->pStyle->bg[GTK_STATE_ACTIVE].green;

  /* minimized color								*/
  pC = &(plsi->priv->aaColor[LS_ICON_WIN_MINIMIZED][LS_ICON_STATE_NORMAL]);
  pC->red = 0.85*plsi->priv->pStyle->bg[GTK_STATE_NORMAL].red;
  pC->blue = 0.85*plsi->priv->pStyle->bg[GTK_STATE_NORMAL].blue;
  pC->green = plsi->priv->pStyle->bg[GTK_STATE_NORMAL].green;
  
  pC = &(plsi->priv->aaColor[LS_ICON_WIN_MINIMIZED][LS_ICON_STATE_PRELIGHT]);
  pC->red = 0.85*plsi->priv->pStyle->bg[GTK_STATE_PRELIGHT].red;
  pC->blue = 0.85*plsi->priv->pStyle->bg[GTK_STATE_PRELIGHT].blue;
  pC->green = plsi->priv->pStyle->bg[GTK_STATE_PRELIGHT].green;
  
  pC = &(plsi->priv->aaColor[LS_ICON_WIN_MINIMIZED][LS_ICON_STATE_ACTIVE]);
  pC->red = 0.85*plsi->priv->pStyle->bg[GTK_STATE_ACTIVE].red;
  pC->blue = 0.85*plsi->priv->pStyle->bg[GTK_STATE_ACTIVE].blue;
  pC->green = plsi->priv->pStyle->bg[GTK_STATE_ACTIVE].green;

  /* win skip									*/
  pC = &(plsi->priv->aaColor[LS_ICON_WIN_SKIP][LS_ICON_STATE_NORMAL]);
  pC->red = plsi->priv->pStyle->bg[GTK_STATE_NORMAL].red;
  pC->blue = 0.85*plsi->priv->pStyle->bg[GTK_STATE_NORMAL].blue;
  pC->green = 0.85*plsi->priv->pStyle->bg[GTK_STATE_NORMAL].green;
  
  pC = &(plsi->priv->aaColor[LS_ICON_WIN_SKIP][LS_ICON_STATE_PRELIGHT]);
  pC->red = plsi->priv->pStyle->bg[GTK_STATE_PRELIGHT].red;
  pC->blue = 0.85*plsi->priv->pStyle->bg[GTK_STATE_PRELIGHT].blue;
  pC->green = 0.85*plsi->priv->pStyle->bg[GTK_STATE_PRELIGHT].green;

  pC = &(plsi->priv->aaColor[LS_ICON_WIN_SKIP][LS_ICON_STATE_ACTIVE]);
  pC->red = plsi->priv->pStyle->bg[GTK_STATE_ACTIVE].red;
  pC->blue = 0.85*plsi->priv->pStyle->bg[GTK_STATE_ACTIVE].blue;
  pC->green = 0.85*plsi->priv->pStyle->bg[GTK_STATE_ACTIVE].green;
  
  /* win other ws								*/
  pC = &(plsi->priv->aaColor[LS_ICON_WIN_OTHER_WS][LS_ICON_STATE_NORMAL]);
  pC->red = 0.85*plsi->priv->pStyle->bg[GTK_STATE_NORMAL].red;
  pC->blue = plsi->priv->pStyle->bg[GTK_STATE_NORMAL].blue;
  pC->green = 0.85*plsi->priv->pStyle->bg[GTK_STATE_NORMAL].green;
  
  pC = &(plsi->priv->aaColor[LS_ICON_WIN_OTHER_WS][LS_ICON_STATE_PRELIGHT]);
  pC->red = 0.85*plsi->priv->pStyle->bg[GTK_STATE_PRELIGHT].red;
  pC->blue = plsi->priv->pStyle->bg[GTK_STATE_PRELIGHT].blue;
  pC->green = 0.85*plsi->priv->pStyle->bg[GTK_STATE_PRELIGHT].green;
  
  pC = &(plsi->priv->aaColor[LS_ICON_WIN_OTHER_WS][LS_ICON_STATE_ACTIVE]);
  pC->red = 0.85*plsi->priv->pStyle->bg[GTK_STATE_ACTIVE].red;
  pC->blue = plsi->priv->pStyle->bg[GTK_STATE_ACTIVE].blue;
  pC->green = 0.85*plsi->priv->pStyle->bg[GTK_STATE_ACTIVE].green;

  return;
}
  
void ls_icon_set_color(LSIcon *plsi, glong lWin)
{
  
  if(GTK_WIDGET_REALIZED(plsi)==FALSE)
    return;

  if(plsi->priv->bApplyOwnColor==FALSE)
    return;
  
  /* !ATTENTION! no return before g_signal_handler_unblock			*/
  g_signal_handler_block(plsi, plsi->priv->signals.style_set);
  
  if(lWin<LS_ICON_WIN_DEFAULT||lWin>LS_ICON_WIN_OTHER_WS)
  {
    DBG("reset color");
    
    gtk_widget_modify_bg(GTK_WIDGET(plsi), GTK_STATE_NORMAL, NULL);
    gtk_widget_modify_bg(GTK_WIDGET(plsi->priv->pButton), GTK_STATE_ACTIVE, NULL);
    gtk_widget_modify_bg(GTK_WIDGET(plsi->priv->pButton), GTK_STATE_PRELIGHT, NULL);
  } 
  else
  {
    gtk_widget_modify_bg(GTK_WIDGET(plsi), GTK_STATE_NORMAL, 
      &(plsi->priv->aaColor[lWin][LS_ICON_STATE_NORMAL]));
    gtk_widget_modify_bg(GTK_WIDGET(plsi->priv->pButton), GTK_STATE_ACTIVE, 
      &(plsi->priv->aaColor[lWin][LS_ICON_STATE_ACTIVE]));
    gtk_widget_modify_bg(GTK_WIDGET(plsi->priv->pButton), GTK_STATE_PRELIGHT,
      &(plsi->priv->aaColor[lWin][LS_ICON_STATE_PRELIGHT]));
  }
  
  g_signal_handler_unblock(plsi, plsi->priv->signals.style_set);

  return;
}

void ls_icon_set_use_color(LSIcon *plsi, gint nUseColor)
{
  g_return_if_fail(plsi!=NULL);
  g_return_if_fail(LS_IS_ICON(plsi));
  
  if(nUseColor==1)
  {
    plsi->priv->bApplyOwnColor = TRUE;
    ls_icon_re_evaluate_visibility(plsi);
  }
  else if(nUseColor==0)
  {
    /* should be before bApplyOwnColor = FALSE, so that it don't return after	*
     * doing nothing!								*/
    ls_icon_set_color(plsi, -1);
    plsi->priv->bApplyOwnColor = FALSE;
    ls_icon_re_evaluate_visibility(plsi);
  }

  return;
}

NetkWindow *ls_icon_get_window(LSIcon *plsi)
{
  g_return_val_if_fail(plsi!=NULL, NULL);
  g_return_val_if_fail(LS_IS_ICON(plsi), NULL);
   
  return plsi->priv->pWindow; 
}

static void ls_icon_set_tooltip(LSIcon *plsi)
{
  g_return_if_fail(ls_icon_get_window(plsi)!=NULL);
  g_return_if_fail(NETK_IS_WINDOW(ls_icon_get_window(plsi)));
  
  ls_icon_set_ftooltip(plsi, "%s", netk_window_get_name(ls_icon_get_window(plsi)));
}

void ls_icon_set_ftooltip(LSIcon *plsi, const char *szFormat, ...) 
{
  g_return_if_fail(GTK_IS_TOOLTIPS(plsi->priv->pToolTips));

  if(szFormat != NULL)
  {
    va_list 	vaArgs;
    char    	*szToolTip;
    
    va_start(vaArgs, szFormat);
    szToolTip = g_strdup_vprintf(szFormat, vaArgs);
    va_end(vaArgs);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(plsi->priv->pToolTips),
      GTK_WIDGET(plsi->priv->pButton), szToolTip, szToolTip 
      /* szutf8ToolTip, szutf8ToolTip */);
  }
  else
  {
    gtk_tooltips_set_tip (GTK_TOOLTIPS(plsi->priv->pToolTips),
      GTK_WIDGET(plsi->priv->pButton), NULL, NULL);
  }

  return;    
}

void ls_icon_set_parent(LSIcon *plsi, LSIconBox *plsib)
{
  plsi->priv->pIconBox = plsib;
}

LSIconBox *ls_icon_get_parent(LSIcon *plsi)
{
  return plsi->priv->pIconBox;
}

glong ls_icon_get_window_xid(LSIcon *plsi)
{
  return plsi->priv->lXID;
}

gboolean ls_icon_re_evaluate_visibility(LSIcon *plsi)
{
  NetkWindow		*pWin;
  NetkWorkspace		*pWS;
  gint			nwsAckt;
  gint			nwsWin;
  NetkWindowState	nwst;
  gboolean		bVisible;
  
  g_return_val_if_fail(plsi!=NULL, FALSE);
  g_return_val_if_fail(LS_IS_ICON(plsi), FALSE);

  bVisible 	= TRUE;
  
  pWin		= ls_icon_get_window(plsi);

  g_return_val_if_fail(pWin!=NULL, FALSE);
  g_return_val_if_fail(NETK_IS_WINDOW(pWin), FALSE);

  pWS		= netk_window_get_workspace(pWin);
  if(NETK_IS_WORKSPACE(pWS))
    nwsWin	= netk_workspace_get_number(netk_window_get_workspace(pWin));
  else
    nwsWin	= -1;

  pWS		= netk_screen_get_active_workspace(NETK_SCREEN(
		    ls_icon_get_screen(plsi)));
  nwsAckt	= netk_workspace_get_number(pWS);
  nwst		= netk_window_get_state(pWin);

  if (plsi->priv->bApplyOwnColor == TRUE)
  {
    ls_icon_set_color(plsi, LS_ICON_WIN_DEFAULT);
  }
  
  if(nwsWin == nwsAckt || netk_window_is_sticky(pWin)) 		/* SAME WORKSPACE			*/
  {
    if (
      nwst & NETK_WINDOW_STATE_MINIMIZED &&
      plsi->priv->display.sameWorkspace.minimized == TRUE 
    ) {
      if (plsi->priv->bApplyOwnColor)
        ls_icon_set_color(plsi, LS_ICON_WIN_MINIMIZED);

      bVisible = TRUE;
    }
    else if (
      !(nwst & NETK_WINDOW_STATE_MINIMIZED) &&
      plsi->priv->display.sameWorkspace.notMinimized == TRUE	
    ) {
      if (plsi->priv->bApplyOwnColor)
        ls_icon_set_color(plsi, LS_ICON_WIN_DEFAULT);

      bVisible = TRUE;
    }
    else
    {
      bVisible = FALSE;
    }

    if (
      nwst & NETK_WINDOW_STATE_SKIP_TASKLIST &&
      plsi->priv->display.sameWorkspace.skipTasklist == TRUE
    ) {
      if (plsi->priv->bApplyOwnColor)
        ls_icon_set_color(plsi, LS_ICON_WIN_SKIP);

      bVisible = TRUE;
    }
    else if (
      nwst & NETK_WINDOW_STATE_SKIP_TASKLIST &&
      plsi->priv->display.sameWorkspace.skipTasklist == FALSE
    ) {
      bVisible = FALSE;
    }
    
  }
  else /* OTHER WORKSPACE							*/
  {
    if (plsi->priv->bApplyOwnColor)
      ls_icon_set_color(plsi, LS_ICON_WIN_OTHER_WS);
    
    if (
      nwst & NETK_WINDOW_STATE_MINIMIZED &&
      plsi->priv->display.otherWorkspace.minimized == TRUE 
    ) {
      bVisible = TRUE;
    }
    else if (
      !(nwst & NETK_WINDOW_STATE_MINIMIZED) &&
      plsi->priv->display.otherWorkspace.notMinimized == TRUE	
    ) {
      bVisible = TRUE;
    }
    else
    {
      bVisible = FALSE;
    }

    if (
      nwst & NETK_WINDOW_STATE_SKIP_TASKLIST &&
      plsi->priv->display.otherWorkspace.skipTasklist == TRUE
    ) {
      bVisible = TRUE;
    }
    else if (
      nwst & NETK_WINDOW_STATE_SKIP_TASKLIST &&
      plsi->priv->display.otherWorkspace.skipTasklist == FALSE
    ) {
      bVisible = FALSE;
    }
  }
  
  if(bVisible)
  {
    ls_icon_show(LS_ICON(plsi));
  }
  else
  {
    ls_icon_hide(LS_ICON(plsi));
  }

  return bVisible;
}

static void ls_icon_hide(LSIcon *plsi)
{
  gtk_widget_hide(GTK_WIDGET(plsi));

  return;
}

static void ls_icon_show(LSIcon *plsi)
{
  gtk_widget_show(GTK_WIDGET(plsi));

  return;
}

NetkScreen *ls_icon_get_screen(LSIcon *plsi)
{
  return plsi->priv->pScreen;
}


void ls_icon_set_icon(LSIcon *plsi, GdkPixbuf *pPixBuf)
{
  g_return_if_fail(LS_IS_ICON(plsi));
  
  xfce_iconbutton_set_pixbuf(LS_ICON(plsi)->priv->pButton, pPixBuf);
}

gboolean ls_icon_re_evaluate_icon(LSIcon *plsi)
{
  if(netk_window_get_icon_is_fallback(ls_icon_get_window(plsi))!=TRUE)
  {
    ls_icon_set_icon(plsi, 
      netk_window_get_icon(
	ls_icon_get_window(plsi)));
  }
  else
  {
    NetkApplication *app;
    
    app = netk_window_get_application(ls_icon_get_window(plsi));
    
    if (app)
    {
        ls_icon_set_icon(plsi, netk_application_get_icon(app));

        if(netk_application_get_icon_is_fallback(app)) {
            return FALSE;
        }
    }
  }
  
  return TRUE;
}

void ls_icon_set_show_same_workspace(LSIcon *plsi, LSDisplayIcon *pdi)
{
  memcpy((void *)&(plsi->priv->display.sameWorkspace), (const void *)pdi, 
    sizeof(LSDisplayIcon));
  ls_icon_re_evaluate_visibility(plsi);
  
  return;
}

void ls_icon_set_show_other_workspace(LSIcon *plsi, LSDisplayIcon *pdi)
{
  memcpy((void *)&(plsi->priv->display.otherWorkspace), (const void *)pdi,
    sizeof(LSDisplayIcon));
  ls_icon_re_evaluate_visibility(plsi);

  return;
}

/*******************************************************************************/
/* CALLBACKS                                                                   */
/*******************************************************************************/
static void cb_lsi_workspace_changed (
  NetkWindow *pWin,
  void *pVoid
) {
  g_return_if_fail(LS_IS_ICON(pVoid));
  ls_icon_re_evaluate_visibility(LS_ICON(pVoid));

  return;
}

static void cb_lsi_name_changed (
  NetkWindow *pWin,
  void *pVoid
) {
  g_return_if_fail(LS_IS_ICON(pVoid));
  ls_icon_set_tooltip(LS_ICON(pVoid));
  return;
}

static void cb_lsi_icon_changed (
  NetkWindow *pWin,
  void *pVoid
) {
  g_return_if_fail(LS_IS_ICON(pVoid));
  
  if(ls_icon_re_evaluate_icon(LS_ICON(pVoid))==FALSE)
    LS_ICON(pVoid)->priv->bFallbackIcon=TRUE;

  return;
}

static void cb_lsi_active_workspace_changed (
  NetkScreen *pScreen, 
  void *pVoid
) {
  g_return_if_fail(LS_IS_ICON(pVoid));
  
  ls_icon_re_evaluate_visibility(LS_ICON(pVoid));
 
  return;
}

static void cb_lsi_window_state_changed (
  NetkWindow *pWin,
  NetkWindowState netkwinOldState,
  NetkWindowState netkwinNewState,
  void *pVoid
) {
  g_return_if_fail(pWin!=NULL);
  g_return_if_fail(LS_IS_ICON(pVoid));
  
  ls_icon_re_evaluate_visibility(LS_ICON(pVoid));
 
  return;
}

static void cb_lsi_realize(
  GtkWidget *pWidget,
  void *pUserData
) {
  g_return_if_fail(LS_IS_ICON(pUserData));
  
  ls_icon_init_colors(LS_ICON(pUserData));
  ls_icon_re_evaluate_visibility(LS_ICON(pUserData));

  return;
}

static void cb_lsi_style_set(
  GtkWidget	*pWidget,
  GtkStyle	*pOldStyle,
  void 		*pUserData
) {
  LSIcon	*plsi;

  g_return_if_fail(LS_IS_ICON(pUserData));
  plsi = LS_ICON(pUserData);
  
  ls_icon_init_colors(plsi);
  ls_icon_set_color(plsi, -1);
  ls_icon_re_evaluate_visibility(plsi);

  return;
}

static gboolean cb_lsi_button_button2_pressed( 
  GtkWidget *pWidget,
  GdkEvent *pEvent,
  gpointer *pUserData 
) {
  if(pEvent->type!=GDK_BUTTON_PRESS)
    return FALSE;
  if(((GdkEventButton *)pEvent)->button!=3)
    return FALSE;

  g_return_val_if_fail(pUserData!=NULL, FALSE);
  g_return_val_if_fail(LS_IS_ICON(pUserData), FALSE);

  /* do we use the menu or not?							*/
  if(LS_ICON(pUserData)->priv->pMenu==NULL)
  {
    NetkWindow *pWin;
    NetkWorkspace *pWS;
    
    pWS = netk_screen_get_active_workspace(ls_icon_get_screen(LS_ICON(pUserData)));
    pWin = ls_icon_get_window(LS_ICON(pUserData));

    g_return_val_if_fail(NETK_IS_WINDOW(pWin), FALSE);
    
    if(
      netk_workspace_get_number(pWS) !=
      netk_workspace_get_number(netk_window_get_workspace(pWin))
    ) {
      netk_window_move_to_workspace(pWin, pWS);
    }
    
    if(netk_window_is_minimized(pWin))
      netk_window_unminimize(pWin);

    netk_window_activate(pWin);
  }
  else
  {
    gtk_menu_popup(GTK_MENU(LS_ICON(pUserData)->priv->pMenu), NULL, NULL,
      NULL, NULL, ((GdkEventButton *)pEvent)->button, 
      ((GdkEventButton *)pEvent)->time);
  }

  return TRUE;
}

static void cb_lsi_button_clicked(GtkButton *pButton, gpointer *pUserData)
{
  NetkWindow 	*pWin;
  LSIcon	*pIcon;

  g_return_if_fail(pUserData!=NULL);
  g_return_if_fail(LS_IS_ICON(pUserData));

  pIcon = LS_ICON(pUserData);
  pWin = ls_icon_get_window(pIcon);
  if(netk_window_is_minimized(pWin))
  {
    NetkWorkspace *pWS;
    pWS = netk_screen_get_active_workspace(ls_icon_get_screen(pIcon));
    if(
      netk_workspace_get_number(pWS) ==
      netk_workspace_get_number(netk_window_get_workspace(pWin)) ||
      netk_window_is_sticky(pWin)) {
      netk_window_unminimize(pWin);
      netk_window_activate(pWin);
    } 
    else
    {
      netk_window_move_to_workspace(pWin, pWS);
      netk_window_unminimize(pWin);
      netk_window_activate(pWin);
    }
  }
  else
  {
    netk_window_minimize(pWin);
  }
  
  return;
}

#if 0
static void cb_lsi_size_allocate (
  GtkWidget *widget, 
  GtkAllocation *allocatio,  
  gpointer user_data
) {
  LSIcon *plsi;

  g_return_if_fail(LS_IS_ICON(user_data));
  plsi = user_data;

  g_signal_handler_block(G_OBJECT(plsi), plsi->priv->signals.size_allocate);

  
  
  g_signal_handler_unblock(G_OBJECT(plsi), plsi->priv->signals.size_allocate);
}
#endif

/********************************************************************************/
/* MENU										*/
/********************************************************************************/
static void 	cb_lsim_close_window(gpointer pUserData, gint nAction, 
    		  GtkWidget *pWidget);
static void 	cb_lsim_maxi_demaximize_horizontally (gpointer pUserData, 
    		  gint nAction, GtkWidget *pWidget);
static void	cb_lsim_maxi_demaximize_vertically (gpointer pUserData, gint nAction, 
    		  GtkWidget *pWidget);
static void 	cb_lsim_mini_deminimize (gpointer pUserData, gint nAction, 
    		  GtkWidget *pWidget);
static void 	cb_lsim_maxi_demaximize (gpointer pUserData, gint nAction, 
    		  GtkWidget *pWidget);
static void     cb_lsim_kill_app(gpointer pUserData, gint nAction, GtkWidget *pWidget);

static GtkItemFactoryEntry lsIconMenu[] = 
{
  {"/Mini-Deminimize",			NULL,	cb_lsim_mini_deminimize,
    1, "<Item>"},
  {"/Sep0",				NULL,	NULL, 		0, "<Separator>"},
  {"/Maxi-Demaximize Window",		NULL,	cb_lsim_maxi_demaximize, 
    1, "<Item>"},
  {"/Maxi-Demaximize Vertical",		NULL,	cb_lsim_maxi_demaximize_vertically, 
    1, "<Item>"},
  {"/Maxi-Demaximize Horizontal",	NULL,	cb_lsim_maxi_demaximize_horizontally,
    1, "<Item>"},
  {"/Sep1",				NULL,	NULL,		0, "<Separator>"},
  {"/Close Window",			NULL,	cb_lsim_close_window,	
    1, "<Item>"},
  {"/Kill Application",			NULL,	cb_lsim_kill_app,	
    1, "<Item>"}
};

static gint nLsIconMenu = sizeof (lsIconMenu) / sizeof (lsIconMenu[0]);

void ls_icon_create_menu(LSIcon *plsi)
{
  /*
    GtkItemFactory	*pItemFactory;
   */
  GtkAccelGroup		*pAccelGroup;

  pAccelGroup	= gtk_accel_group_new();
  plsi->priv->pItemFactory =
    gtk_item_factory_new(GTK_TYPE_MENU, "<main>", pAccelGroup);
  gtk_item_factory_create_items(plsi->priv->pItemFactory, 
    nLsIconMenu, lsIconMenu, (gpointer)plsi);
  
  plsi->priv->pMenu = gtk_item_factory_get_widget(plsi->priv->pItemFactory, 
    "<main>");

  return;
}					/* ls_icon_creat_menu()			*/

/********************************************************************************/
/* MENU - CALLBACKS								*/
/********************************************************************************/
static void cb_lsim_kill_app(gpointer pUserData, gint nAction, GtkWidget *pWidget)
{
  gint	nPid;
  
  g_return_if_fail(LS_IS_ICON(pUserData));
  g_return_if_fail(ls_icon_get_window(LS_ICON(pUserData))!=NULL);

  nPid		= netk_window_get_pid(ls_icon_get_window(LS_ICON(pUserData)));
  if(nPid==0)
  {
    nPid	= netk_application_get_pid(
		    netk_window_get_application(
		      ls_icon_get_window(LS_ICON(pUserData))));
    if(nPid==0)
    {
      g_warning("unable to get pid of application for window XID:%ld",
	netk_window_get_xid(ls_icon_get_window(LS_ICON(pUserData))));
      return;
    }
  }
  
  kill((pid_t)nPid, 9);
  
  return;
}

static void cb_lsim_close_window(gpointer pUserData, gint nAction, GtkWidget *pWidget)
{
  g_return_if_fail(LS_IS_ICON(pUserData));
  g_return_if_fail(ls_icon_get_window(LS_ICON(pUserData))!=NULL);

  netk_window_close(ls_icon_get_window(LS_ICON(pUserData)));

  return;
}

static void cb_lsim_maxi_demaximize_horizontally (
  gpointer pUserData, 
  gint nAction, 
  GtkWidget *pWidget)
{
  g_return_if_fail(LS_IS_ICON(pUserData));
  g_return_if_fail(ls_icon_get_window(LS_ICON(pUserData))!=NULL);

  if(netk_window_is_maximized_horizontally(ls_icon_get_window(LS_ICON(pUserData))))
    netk_window_unmaximize_horizontally(ls_icon_get_window(LS_ICON(pUserData)));
  else
    netk_window_maximize_horizontally(ls_icon_get_window(LS_ICON(pUserData)));

  return;
}

static void cb_lsim_maxi_demaximize_vertically (
  gpointer pUserData, 
  gint nAction, 
  GtkWidget *pWidget
) {
  g_return_if_fail(LS_IS_ICON(pUserData));
  g_return_if_fail(ls_icon_get_window(LS_ICON(pUserData))!=NULL);

  if(netk_window_is_maximized_vertically(ls_icon_get_window(LS_ICON(pUserData))))
    netk_window_unmaximize_vertically(ls_icon_get_window(LS_ICON(pUserData)));
  else
    netk_window_maximize_vertically(ls_icon_get_window(LS_ICON(pUserData)));

  return;
}

static void cb_lsim_mini_deminimize (
  gpointer pUserData, 
  gint nAction, 
  GtkWidget *pWidget
) {
  g_return_if_fail(LS_IS_ICON(pUserData));
  g_return_if_fail(ls_icon_get_window(LS_ICON(pUserData))!=NULL);

  if(netk_window_is_minimized(ls_icon_get_window(LS_ICON(pUserData))))
    netk_window_unminimize(ls_icon_get_window(LS_ICON(pUserData)));
  else
    netk_window_minimize(ls_icon_get_window(LS_ICON(pUserData)));

  return;
}

static void cb_lsim_maxi_demaximize (
  gpointer pUserData, 
  gint nAction, 
  GtkWidget *pWidget)
{
  g_return_if_fail(LS_IS_ICON(pUserData));
  g_return_if_fail(ls_icon_get_window(LS_ICON(pUserData))!=NULL);

  if(netk_window_is_maximized(ls_icon_get_window(LS_ICON(pUserData))))
    netk_window_unmaximize(ls_icon_get_window(LS_ICON(pUserData)));
  else
    netk_window_maximize(ls_icon_get_window(LS_ICON(pUserData)));

  return;
}
