#include <stdlib.h>

#include "mfsk.h"
#include "filter.h"
#include "sfft.h"
#include "varicode.h"
#include "misc.h"

static void recvbit(struct mfsk *m, int bit)
{
	int c;

	m->datashreg = (m->datashreg << 1) | !!bit;

	/* search for "001" */
	if ((m->datashreg & 7) == 1) {
		/* the "1" belongs to the next symbol */
		c = varidec(m->datashreg >> 1);

		if (c != -1)
			trx_put_rx_char(c);

		/* we already received this */
		m->datashreg = 1;
	}
}

static void decodesymbol(struct trx *trx, unsigned char symbol)
{
	struct mfsk *m = (struct mfsk *) trx->modem;
	int c, met;

	m->symbolpair[0] = m->symbolpair[1];
	m->symbolpair[1] = symbol;

	m->symcounter = m->symcounter ? 0 : 1;

	/* MFSK16 doesn't need a vote */
	if (trx->mode == MODE_MFSK16 && m->symcounter)
		return;

	if (m->symcounter) {
		if ((c = viterbi_decode(m->dec1, m->symbolpair, &met)) == -1)
			return;

		m->metric1 = decayavg(m->metric1, met / 20.0, 0.125);

		if (m->metric1 < m->metric2)
			return;

		trx->metric = m->metric1;
	} else {
		if ((c = viterbi_decode(m->dec2, m->symbolpair, &met)) == -1)
			return;

		m->metric2 = decayavg(m->metric2, met / 20.0, 0.125);

		if (m->metric2 < m->metric1)
			return;

		trx->metric = m->metric2;
	}

	if (trx->squelchon && trx->metric < trx->sqval)
		return;

	recvbit(m, c & 0x80);
	recvbit(m, c & 0x40);
	recvbit(m, c & 0x20);
	recvbit(m, c & 0x10);
	recvbit(m, c & 0x08);
	recvbit(m, c & 0x04);
	recvbit(m, c & 0x02);
	recvbit(m, c & 0x01);

//	fprintf(stderr, "met1=%-5.1f met2=%-5.1f %d\n", 
//		m->metric1, m->metric2,
//		m->metric1 > m->metric2 ? 1 : 2);
}

static void softdecode(struct trx *trx, complex *bins)
{
	struct mfsk *m = (struct mfsk *) trx->modem;
	float tone, sum, *b;
	unsigned char *symbols;
	int i, j, k;

	b = alloca(m->symbits * sizeof(float));
	symbols = alloca(m->symbits * sizeof(unsigned char));

	for (i = 0; i < m->symbits; i++)
		b[i] = 0.0;

	/* avoid divide by zero later */
	sum = 1e-10;

	/* gray decode and form soft decision samples */
	for (i = 0; i < m->numtones; i++) {
		j = graydecode(i);

		if (trx->reverse)
			k = (m->numtones - 1) - i;
		else
			k = i;

		tone = cmod(bins[k + m->basetone]);

		for (k = 0; k < m->symbits; k++)
			b[k] += (j & (1 << (m->symbits - k - 1))) ? tone : -tone;

		sum += tone;
	}

	/* shift to range 0...255 */
	for (i = 0; i < m->symbits; i++)
		symbols[i] = clamp(128.0 + (b[i] / sum * 128.0), 0, 255);

	interleave_syms(m->rxinlv, symbols);

	for (i = 0; i < m->symbits; i++)
		decodesymbol(trx, symbols[i]);
}

static complex mixer(struct trx *trx, complex in)
{
	struct mfsk *m = (struct mfsk *) trx->modem;
	complex z;
	float f;

	f = trx->frequency - trx->bandwidth / 2;

	/* Basetone is always 1000 Hz */
	f -= 1000.0;

	z.re = cos(m->phaseacc);
	z.im = sin(m->phaseacc);

	z = cmul(z, in);

	m->phaseacc -= 2.0 * M_PI * f / SampleRate;

	if (m->phaseacc > M_PI)
		m->phaseacc -= 2.0 * M_PI;
	else if (m->phaseacc < M_PI)
		m->phaseacc += 2.0 * M_PI;

	return z;
}

static int harddecode(struct trx *trx, complex *in)
{
	struct mfsk *m = (struct mfsk *) trx->modem;

	int i, symbol = 0;
	double x, max = 0.0;

	in += m->basetone;

	for (i = 0; i < m->numtones; i++) {
		if ((x = cmod(in[i])) > max) {
			max = x;
			symbol = i;
		}
	}

	return symbol;
}

static void update_syncscope(struct mfsk *m)
{
	float *data;
	int i, j;

	data = alloca(2 * m->symlen * sizeof(float));

	for (i = 0; i < 2 * m->symlen; i++) {
		j = (i + m->pipeptr) % (2 * m->symlen);
		data[i] = cmod(m->pipe[j].vector[m->prev1symbol]);
	}

	trx_set_scope(data, 2 * m->symlen, TRUE);
}

static void synchronize(struct mfsk *m)
{
	int i, j, syn = -1;
	float val, max = 0.0;

	if (m->currsymbol == m->prev1symbol)
		return;
	if (m->prev1symbol == m->prev2symbol)
		return;

	for (i = 0; i < 2 * m->symlen; i++) {
		j = (i + m->pipeptr) % (2 * m->symlen);
		val = cmod(m->pipe[j].vector[m->prev1symbol]);
		if (val > max) {
			syn = i;
			max = val;
		}
	}

	m->syncaverage += (syn - m->symlen) / 16.0;

	if (m->syncaverage < 0.0)
		m->syncaverage += m->symlen;
	if (m->syncaverage >= m->symlen)
		m->syncaverage -= m->symlen;

	m->symboltime = (unsigned int) floor(m->syncaverage);
}

static void afc(struct trx *trx)
{
	struct mfsk *m = (struct mfsk *) trx->modem;
	complex z;
	float x;

	if (trx->afcon == FALSE || trx->metric < trx->sqval)
		return;

	if (m->currsymbol != m->prev1symbol)
		return;

	z = ccor(m->prev1vector, m->currvector);
	x = carg(z) / m->symlen / (2.0 * M_PI / SampleRate);

	if (x > -m->tonespacing / 2.0 &&  x < m->tonespacing / 2.0)
		trx_set_freq(trx->frequency + (x / 8.0));
}

int mfsk_rxprocess(struct trx *trx, float *buf, int len)
{
	struct mfsk *m = (struct mfsk *) trx->modem;
	complex z, *bins;
	int i;

	while (len-- > 0) {
		/* create analytic signal... */
		z.re = z.im = *buf++;
		filter_run(m->hilbert, z, &z);

		/* ...so it can be shifted in frequency */
		z = mixer(trx, z);

		/* feed it to the sliding FFT */
		bins = sfft_run(m->sfft, z);

		/* copy current vector to the pipe */
		for (i = 0; i < m->numtones; i++)
			m->pipe[m->pipeptr].vector[i] = bins[i + m->basetone];

		if (m->synccounter > 0) {
			m->synccounter--;
		} else if (m->symbolphase == m->symboltime) {
			m->synccounter = m->symlen / 2;

			m->currsymbol = harddecode(trx, bins);
			m->currvector = bins[m->currsymbol + m->basetone];

			/* decode symbol */
			softdecode(trx, bins);

			/* update the scope */
			update_syncscope(m);

			/* symbol sync */
			synchronize(m);

			/* frequency tracking */
			afc(trx);

			m->prev2symbol = m->prev1symbol;
			m->prev2vector = m->prev1vector;
			m->prev1symbol = m->currsymbol;
			m->prev1vector = m->currvector;
		}

		m->pipeptr = (m->pipeptr + 1) % (2 * m->symlen);
		m->symbolphase = (m->symbolphase + 1) % m->symlen;
	}

	return 0;
}
