/*
 * xconvers - GTK+ convers client for amateur radio
 * Copyright (C) 2000-2003 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * callback.c - functions that connect to a widget.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "history.h"
#include "preferences.h"
#include "net.h"
#include "color.h"
#include "types.h"
#include "utils.h"

/************************ DEFINES AND GLOBAL VARIABLES ***********************/

#define HOSTNAMEHISTORY 10
#define PORTHISTORY     10

extern GtkWidget *mainwindow;
extern gchar *savedline;
gchar *preferencesdir, *logfilename;
gpointer cvhost, cvport;
GdkFont *textfont;
GtkWidget *preferencesdialog;
GtkButton *selectedcolorbutton;
gint rxmonitor, buttonnumber = -1, mousebutton = 0;
gboolean connected = FALSE, locked = FALSE;
GList *txhistory, *hostnamehistory, *porthistory;
FILE *logfp;
preferencestype preferences;
gboolean coloryes;

/**********************************MAIN WINDOW********************************/

/*
 * This function is called when the main window is created. It is used to set
 * the background, font and the different colors for the user messages. It also
 * calls loadpreferences for retrieving default settings, sets a greeting and
 * the window title.
 */

void on_mainwindow_show(GtkWidget *widget, gpointer user_data)
{
  GtkWidget *menuclose, *mainstatusbar, *mainscrolledwindow;
  GString *greeting = g_string_new("");

  menuclose = lookup_widget(mainwindow, "close");
  gtk_widget_set_sensitive(menuclose, 0);

  textfont = gdk_font_load(preferences.font->str);
  if (!preferences.menu) widget_visible("mainmenubar", 0);
  if (!preferences.statusbar)
  {
    widget_visible("progressbar", 0);
    widget_visible("mainstatusbar", 0);
    widget_visible("statushbox", 0);
  }
  if (!preferences.scrollbar)
  {
    mainscrolledwindow = lookup_widget(mainwindow, "mainscrolledwindow");
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mainscrolledwindow), 
      GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  }
  g_string_sprintf(greeting, _("Welcome to %s"), PACKAGE);
  mainstatusbar = lookup_widget (mainwindow, "mainstatusbar");
  gtk_statusbar_push (GTK_STATUSBAR(mainstatusbar), 1, greeting->str);
  if (!preferences.statusbar)
  {
    g_string_prepend(greeting, "--> ");
    g_string_append(greeting, "\n");
    maintext_add(greeting->str, MESSAGE_TX);
  }
  g_string_free(greeting, TRUE);
}

/*
 * Try to catch PgUp/PgDn key so the receive window can scroll. Up and Down
 * arrow will show command line history. Alt key will activate menu. Typing
 * any other key will give focus to the transmit widget.
 */

gboolean on_mainwindow_key_press_event(GtkWidget *widget, GdkEventKey *event,
  gpointer user_data)
{
  GtkWidget *maintext, *mainmenubar;
  GtkAdjustment *adj;

  maintext = lookup_widget(mainwindow, "maintext");
  adj = GTK_ADJUSTMENT(GTK_TEXT(maintext)->vadj);
  switch (event->keyval)
  {
    case GDK_Up:
      gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
      tx_previous();
    break;
    case GDK_Down:
      gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
      tx_next();
    break;
    case GDK_Page_Up:
      gtk_adjustment_set_value(adj,adj->value-adj->page_size);
    break;
    case GDK_Page_Down:
      if(adj->value < adj->upper - adj->page_size)
        gtk_adjustment_set_value(adj, adj->value+adj->page_size);
    break;
    case GDK_Alt_L:
      mainmenubar = lookup_widget(mainwindow, "mainmenubar");
      gtk_widget_grab_focus(mainmenubar);
    break;
    default:
    break;
  }
  return FALSE;
}

/*
 * When a message is entered into the one-line entry at the bottom of the main
 * window, this function is called. The whole entry is grabbed, sent to the 
 * socket and cleared again.
 */

void on_mainentry_activate(GtkEditable *editable, gpointer user_data)
{
  GString *entry;
  GtkWidget *mainentry;

  mainentry = lookup_widget(mainwindow, "mainentry");
  entry = g_string_new(gtk_editable_get_chars(GTK_EDITABLE(mainentry), 0, -1));
  gtk_entry_select_region(GTK_ENTRY(mainentry), 0, 
    GTK_ENTRY(mainentry)->text_length);
  tx(entry);
  g_string_free(entry, TRUE);
  gtk_entry_set_text(GTK_ENTRY(mainentry),"");
  gtk_widget_grab_focus(GTK_WIDGET(mainentry));
}

/*
 * Cleanup, save history and exit the gtk_main loop nicely if the main window
 * is deleted.
 */

gboolean on_mainwindow_delete_event(GtkWidget *widget, GdkEvent *event,
  gpointer user_data)
{
  savehistory();

  if (txhistory) g_list_free(txhistory);
  if (hostnamehistory) g_list_free(hostnamehistory);
  if (porthistory) g_list_free(porthistory);
  if (savedline) g_free(savedline);

  gdk_window_get_size(mainwindow->window, &preferences.width, &preferences.height);
  gdk_window_get_root_origin(mainwindow->window, &preferences.x, &preferences.y);

  savepreferences(coloryes);
  gtk_exit(0);
  return FALSE;
}

/********************************* MENUS *************************************/

/*
 * Exit is activated from the menu. This emits a delete event.
 */

void on_exit_activate(GtkMenuItem *menuitem, gpointer user_data)
{
  gtk_signal_emit_by_name(GTK_OBJECT(mainwindow),"delete_event");
}

/*
 * Open is selected, just show the open dialog.
 */

void on_open_activate(GtkMenuItem *menuitem, gpointer user_data)
{
  GtkWidget *opendialog;

  opendialog = create_opendialog();
  set_title(opendialog, _("%s - open connection"));
  set_icon(opendialog, _("open"));
  gtk_widget_show(opendialog);
}

/*
 * After close is selected, a dialog will show up, asking if you really want
 * to close the connection.
 */

void on_close_activate(GtkMenuItem *menuitem, gpointer user_data)
{
  GtkWidget *closedialog, *closelabel;
  GString *labeltext = g_string_new("");

  closedialog = create_closedialog ();
  closelabel = lookup_widget (closedialog, "closelabel");
  g_string_sprintf(labeltext, _("Close connection to %s ?"), (gchar *)cvhost);
  gtk_label_set_text(GTK_LABEL(closelabel), labeltext->str);
  g_string_free(labeltext, TRUE);
  set_title(closedialog, _("%s - close connection"));
  set_icon(closedialog, _("close"));
  gtk_widget_set_sensitive(mainwindow, 0);
  gtk_widget_show(closedialog);
}

/*
 * Activate preferences dialog.
 */
 
 
void on_preferences_activate(GtkMenuItem *menuitem, gpointer user_data)
{
  preferencesdialog = create_preferencesdialog();
  set_title(preferencesdialog, _("%s - preferences"));
  set_icon(preferencesdialog, _("preferences"));
  gtk_widget_show(preferencesdialog);
}

/*
 * Activate the about window.
 */

void on_about_activate(GtkMenuItem *menuitem, gpointer user_data)
{
  GtkWidget *aboutdialog;

  aboutdialog = create_aboutdialog();
  set_title(aboutdialog, _("%s - about"));
  set_icon(aboutdialog, _("about"));
  gtk_widget_show(aboutdialog);
}

/******************************* OPEN DIALOG *********************************/

/*
 * When the open dialog appears, memory is allocated for the host and port
 * to connect to. The combo widgets are filled with their history.
 */

void on_opendialog_show(GtkWidget *widget, gpointer user_data)
{
  GtkWidget *opendialog, *hostnamecombo, *portcombo;
  
  if (!cvhost) cvhost = g_malloc0(100*sizeof(gchar));
  if (!cvport) cvport = g_malloc0(20*sizeof(gchar));
  opendialog = gtk_widget_get_toplevel(widget);
  hostnamecombo = lookup_widget(opendialog, "hostnamecombo");
  portcombo = lookup_widget(opendialog, "portcombo");
  if (hostnamehistory) gtk_combo_set_popdown_strings(GTK_COMBO(hostnamecombo), hostnamehistory);
  if (porthistory) gtk_combo_set_popdown_strings(GTK_COMBO(portcombo), porthistory);
  gtk_widget_set_sensitive(mainwindow, 0);
}

/*
 * When OK is clicked in the open dialog, the dialog is destroyed and the
 * actual connecting should start. If nothing is entered, values default to
 * 'localhost' and '3600'. Hostname and port are saved to a linked list,
 * up to a maximum of {CONNECT|PORT}HISTORY.
 * If a host or portname is selected from the combobox and it is not the first
 * one, it is moved to up. Before we actually connect, we do some cleanups.
 */

void on_openOKbutton_clicked(GtkButton *button, gpointer user_data)
{
  GtkWidget *opendialog, *hostnamecombo, *portcombo, *hostnamecomboentry,
    *portcomboentry, *menuclose, *menuopen;
  GList *node;

  opendialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
  hostnamecombo = lookup_widget(opendialog, "hostnamecombo");
  portcombo = lookup_widget(opendialog, "portcombo");
  hostnamecomboentry = GTK_COMBO(hostnamecombo)->entry;
  portcomboentry = GTK_COMBO(portcombo)->entry;
  strcpy(cvhost, gtk_editable_get_chars(GTK_EDITABLE(hostnamecomboentry), 0, -1));
  strcpy(cvport, gtk_editable_get_chars(GTK_EDITABLE(portcomboentry), 0, -1));
  if (!strcmp(cvhost, "")) strcpy(cvhost, "localhost");
  if (!strcmp(cvport, "")) strcpy(cvport, "3600");

  node = g_list_find_custom(hostnamehistory, g_strdup(cvhost), (GCompareFunc)g_strncasecmp);
  if (!node) hostnamehistory = g_list_prepend(hostnamehistory, g_strdup(cvhost));
  else
  {
    if (g_list_position(hostnamehistory, node) != 0)
    {
      hostnamehistory = g_list_remove_link(hostnamehistory, node);
      hostnamehistory = g_list_prepend(hostnamehistory, g_strdup(cvhost));
    }
  }

  if (g_list_length(hostnamehistory) > HOSTNAMEHISTORY)
    hostnamehistory = g_list_remove(hostnamehistory, g_list_last(hostnamehistory)->data);

  node = g_list_find_custom(porthistory, g_strdup(cvport), (GCompareFunc)g_strncasecmp);
  if (!node) porthistory = g_list_prepend(porthistory, g_strdup(cvport));
  else
  {
    if (g_list_position(porthistory, node) != 0)
    {
      porthistory = g_list_remove_link(porthistory, node);
      porthistory = g_list_prepend(porthistory, g_strdup(cvport));
    }
  }

  if (g_list_length(porthistory) > PORTHISTORY)
    porthistory = g_list_remove(porthistory, g_list_last(porthistory)->data);

  if (rxmonitor) gdk_input_remove(rxmonitor);
  if (connected) connected = FALSE;

  menuclose = lookup_widget(mainwindow, "close");
  menuopen = lookup_widget(mainwindow, "open");
  gtk_widget_set_sensitive(menuopen, 0);
  gtk_widget_set_sensitive(menuclose, 0);

  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
  cvresolve();
}

/*
 * Cancel is clicked, just destroy the dialog window.
 */

void on_opencancelbutton_clicked(GtkButton *button, gpointer user_data)
{
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET (button)));
}

/*
 * When the dialog is destroyed, the tx widget should get focus.
 */

void on_opendialog_destroy(GtkObject *object, gpointer user_data)
{
  GtkWidget *mainentry;
  
  gtk_widget_set_sensitive(mainwindow, 1);
  mainentry = lookup_widget(mainwindow, "mainentry");
  gtk_widget_grab_focus(mainentry);
}

/***************************** CLOSE DIALOG **********************************/

/*
 * If OK is clicked on the close dialog, the socket is closed. I know this is
 * not nice to the server, I get a 'link failed'. The server wrongly assumes
 * that the link is down. Sending '/quit' won't work here, because the other
 * end could be down and I can wait for ages until the connection closes.
 * Maybe there exists some kind of socket timeout?
 */

void on_closeOKbutton_clicked(GtkButton *button, gpointer user_data)
{
  cvdisconnect();
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/*
 *  Just destroy the close dialog when cancel is clicked.
 */

void on_closecancelbutton_clicked(GtkButton *button, gpointer user_data)
{
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/*
 * When the dialog is destroyed, the tx widget should get focus.
 */

void on_closedialog_destroy(GtkObject *object, gpointer user_data)
{
  GtkWidget *mainentry;

  gtk_widget_set_sensitive(mainwindow, 1);
  mainentry = lookup_widget(mainwindow, "mainentry");
  gtk_widget_grab_focus(mainentry);
}

/**************************** PREFERENCES DIALOG *****************************/

/*
 * Check or uncheck autologin and logging boxes. 
 * The preferences struct is checked to see if the name, channel
 * and command strings are assigned. If so, the entries will be filled in. The 
 * background button gets the same color as the main text window and the font
 * entry is filled.
 */
 
void on_preferencesdialog_show(GtkWidget *widget, gpointer user_data)
{
  GtkWidget *loggingcheckbutton, *autologincheckbutton, *autologinnamelabel, 
    *autologinchannellabel, *autologincommandslabel, *autologinnameentry,
    *autologinchannelentry, *autologincommandsentry, *fontentry,
    *menucheckbutton, *statusbarcheckbutton, *scrollbarcheckbutton,
    *backgroundcolorbutton, *pixmapentry, *pixmapselectbutton,
    *colorradiobutton, *pixmapradiobutton, *poscheck, *sizecheck;

  /* set state of buttons */
  loggingcheckbutton = lookup_widget(widget, "loggingcheckbutton");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(loggingcheckbutton), preferences.logging);
  autologincheckbutton = lookup_widget(widget, "autologincheckbutton");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(autologincheckbutton), preferences.autologin);
  sizecheck = lookup_widget(widget, "sizecheckbutton");
  poscheck = lookup_widget(widget, "positioncheckbutton");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(sizecheck), preferences.size);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(poscheck), preferences.origin);

  /* set autologin entries */
  autologinnameentry = lookup_widget(widget, "autologinnameentry");
  autologinchannelentry = lookup_widget(widget, "autologinchannelentry");
  autologincommandsentry = lookup_widget(widget, "autologincommandsentry");
  if (preferences.name)
    gtk_entry_set_text(GTK_ENTRY(autologinnameentry), preferences.name->str);
  if (preferences.channel)
    gtk_entry_set_text(GTK_ENTRY(autologinchannelentry), 
      preferences.channel->str);
  if (preferences.commands)
    gtk_entry_set_text(GTK_ENTRY(autologincommandsentry), 
      preferences.commands->str);

  /* set sensitivity of widgets in autologin frame */
  if (!preferences.autologin)
  {
    autologinnamelabel = lookup_widget(widget, "autologinnamelabel");
    autologinchannellabel = lookup_widget(widget, "autologinchannellabel");
    autologincommandslabel = lookup_widget(widget, "autologincommandslabel");
    gtk_widget_set_sensitive(autologinnamelabel, 0);
    gtk_widget_set_sensitive(autologinchannellabel, 0);
    gtk_widget_set_sensitive(autologincommandslabel, 0);
    gtk_widget_set_sensitive(autologinnameentry, 0);
    gtk_widget_set_sensitive(autologinchannelentry, 0);
    gtk_widget_set_sensitive(autologincommandsentry, 0);
    gtk_editable_set_editable(GTK_EDITABLE(autologinnameentry), FALSE);
    gtk_editable_set_editable(GTK_EDITABLE(autologinchannelentry), FALSE);
    gtk_editable_set_editable(GTK_EDITABLE(autologincommandsentry), FALSE);
  }

  /* set state of widgets*/
  menucheckbutton = lookup_widget(widget, "menucheckbutton");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(menucheckbutton), preferences.menu);
  statusbarcheckbutton = lookup_widget(widget, "statusbarcheckbutton");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(statusbarcheckbutton), preferences.statusbar);
  scrollbarcheckbutton = lookup_widget(widget, "scrollbarcheckbutton");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(scrollbarcheckbutton), preferences.scrollbar);

  /* set sensitivity of widgets in background frame*/
  backgroundcolorbutton = lookup_widget(preferencesdialog, "backgroundcolorbutton");
  pixmapentry = lookup_widget(preferencesdialog, "pixmapentry");
  pixmapselectbutton = lookup_widget(preferencesdialog, "pixmapselectbutton");
  if (preferences.backgroundpixmap) gtk_entry_set_text(GTK_ENTRY(pixmapentry), preferences.backgroundpixmap->str);
  colorradiobutton = lookup_widget(preferencesdialog, "colorradiobutton");
  pixmapradiobutton = lookup_widget(preferencesdialog, "pixmapradiobutton");
  if (preferences.backgroundpixmap) 
  { 
  if (preferences.backgroundpixmap->len > 0)
    {
      gtk_widget_set_sensitive(backgroundcolorbutton, 0);
      gtk_widget_set_sensitive(pixmapentry, 1);
      gtk_widget_set_sensitive(pixmapselectbutton, 1);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(colorradiobutton), FALSE);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pixmapradiobutton), TRUE);
    }
  }
  else
  {
    gtk_widget_set_sensitive(backgroundcolorbutton, 1);
    gtk_widget_set_sensitive(pixmapentry, 0);
    gtk_widget_set_sensitive(pixmapselectbutton, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(colorradiobutton), TRUE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pixmapradiobutton), FALSE);
  }

  /* update the preview window (argument does not really matter) */
  updatepreview(preferences.backgroundcolor);
  
  /* set colors of all the buttons */
  setbuttoncolor("backgroundcolorbutton", preferences.backgroundcolor);
  setbuttoncolor("txmessagebutton", preferences.txfgcolor);
  setbuttoncolor("rxmessagebutton", preferences.rxfgcolor);
  setbuttoncolor("statusmessagebutton", preferences.statusfgcolor);
  setbuttoncolor("privatemessagebutton", preferences.privatefgcolor);
  setbuttoncolor("user1button", preferences.userfgcolor[0]);
  setbuttoncolor("user2button", preferences.userfgcolor[1]);
  setbuttoncolor("user3button", preferences.userfgcolor[2]);
  setbuttoncolor("user4button", preferences.userfgcolor[3]);
  setbuttoncolor("user5button", preferences.userfgcolor[4]);
  setbuttoncolor("user6button", preferences.userfgcolor[5]);
  setbuttoncolor("user7button", preferences.userfgcolor[6]);
  setbuttoncolor("user8button", preferences.userfgcolor[7]);
  setbuttoncolor("user9button", preferences.userfgcolor[8]);
  setbuttoncolor("user10button", preferences.userfgcolor[9]);

  /* find font and set it */
  fontentry = lookup_widget(widget, "fontentry");
  gtk_entry_set_text(GTK_ENTRY(fontentry), preferences.font->str);

  /* finally, make the main window insensitive */
  gtk_widget_set_sensitive(mainwindow, 0);
}

/*
 * The autologin checkbutton is toggled, we have to update the visibility and 
 * the editability of the name and channel labels and entries.
 */

void on_autologincheckbutton_toggled(GtkToggleButton *togglebutton,
  gpointer user_data)
{
  GtkWidget *autologinnamelabel, *autologinchannellabel,
    *autologincommandslabel,*autologinnameentry, *autologinchannelentry, 
    *autologincommandsentry;
  gboolean autologincheckbuttonstatus;

  autologincheckbuttonstatus = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON(togglebutton));
  preferencesdialog = gtk_widget_get_toplevel(GTK_WIDGET(togglebutton));
  autologinnamelabel = lookup_widget(preferencesdialog, "autologinnamelabel");
  autologinchannellabel = lookup_widget(preferencesdialog, "autologinchannellabel");
  autologincommandslabel = lookup_widget(preferencesdialog, "autologincommandslabel");
  autologinnameentry = lookup_widget(preferencesdialog, "autologinnameentry");
  autologinchannelentry = lookup_widget(preferencesdialog, "autologinchannelentry");
  autologincommandsentry = lookup_widget(preferencesdialog, "autologincommandsentry");
  gtk_widget_set_sensitive(autologinnamelabel, autologincheckbuttonstatus);
  gtk_widget_set_sensitive(autologinchannellabel, autologincheckbuttonstatus);
  gtk_widget_set_sensitive(autologincommandslabel, autologincheckbuttonstatus);
  gtk_widget_set_sensitive(autologinnameentry, autologincheckbuttonstatus);
  gtk_widget_set_sensitive(autologinchannelentry, autologincheckbuttonstatus);
  gtk_widget_set_sensitive(autologincommandsentry, autologincheckbuttonstatus);
  gtk_editable_set_editable(GTK_EDITABLE(autologinnameentry),
    autologincheckbuttonstatus);
  gtk_editable_set_editable(GTK_EDITABLE(autologinchannelentry),
    autologincheckbuttonstatus);
  gtk_editable_set_editable(GTK_EDITABLE(autologincommandsentry),
    autologincheckbuttonstatus);
}

/*
 * We look up the color of the clicked button and apply this to the
 * colorselection dialog. The preferences dialog is made insensitive and the
 * colorselectiondialog is shown. I use the button_pressed_event here because
 * I want to distinguish between left and right mouse clicks.
 */

gboolean
on_colorbutton_pressed_event           (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
  GtkWidget *colorselectiondialog;
  gdouble rgb[3];
  GdkColor buttoncolor;
  GtkStyle *buttonstyle;

  /* Save button and mouse button pressed */
  buttonnumber = atoi(user_data);
  mousebutton = event->button;

  /* Stop the event */
  gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "button_press_event");

  colorselectiondialog = create_colorselectiondialog();
  gtk_widget_destroy(GTK_COLOR_SELECTION_DIALOG(colorselectiondialog)->help_button);

  /* Get the buttoncolor before applying it to the color selection dialog. */
  /*Special case for private color button: right-click should select the background color*/
  buttonstyle = gtk_widget_get_style(widget);
  if (!(buttonnumber == 4)) buttoncolor = buttonstyle->bg[GTK_STATE_NORMAL];
    else 
    {
      if (mousebutton == 1) buttoncolor = preferences.privatefgcolor;
      else buttoncolor = preferences.privatebgcolor;
    }

  rgb[0] = buttoncolor.red / 65535.0;
  rgb[1] = buttoncolor.green / 65535.0;
  rgb[2] = buttoncolor.blue / 65535.0;
  gtk_color_selection_set_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(colorselectiondialog)->colorsel), rgb);

  set_title(colorselectiondialog, _("%s - select color"));
  set_icon(colorselectiondialog, _("color"));
  gtk_widget_set_sensitive(preferencesdialog, 0);
  gtk_widget_show(colorselectiondialog);

  return FALSE;
}

/*
 * Either background color or pixmap is selected here. Set the sensitivity of
 * the associated widgets accordingly.
 */

void on_radiobutton_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
  gint data;
  GtkWidget *backgroundcolorbutton, *pixmapentry, *pixmapselectbutton;

  data = atoi(user_data);
  backgroundcolorbutton = lookup_widget(preferencesdialog, "backgroundcolorbutton");
  pixmapentry = lookup_widget(preferencesdialog, "pixmapentry");
  pixmapselectbutton = lookup_widget(preferencesdialog, "pixmapselectbutton");

  switch (data)
  {
    case 1:
      gtk_widget_set_sensitive(backgroundcolorbutton, 1);
      gtk_widget_set_sensitive(pixmapentry, 0);
      gtk_widget_set_sensitive(pixmapselectbutton, 0);
    break;
    case 2:
      gtk_widget_set_sensitive(backgroundcolorbutton, 0);
      gtk_widget_set_sensitive(pixmapentry, 1);
      gtk_widget_set_sensitive(pixmapselectbutton, 1);
    break;
    default:
    break;
  }
}

/*
 * Select a background pixmap. Show the openfile dialog.
 */

void on_pixmapselectbutton_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *fileselectiondialog;

  fileselectiondialog = create_fileselectiondialog();
  
  /* Don't need these buttons */
  gtk_widget_destroy(GTK_FILE_SELECTION(fileselectiondialog)->fileop_c_dir);
  gtk_widget_destroy(GTK_FILE_SELECTION(fileselectiondialog)->fileop_del_file);
  gtk_widget_destroy(GTK_FILE_SELECTION(fileselectiondialog)->fileop_ren_file);

  set_title(fileselectiondialog, _("%s - select pixmap"));
  set_icon(fileselectiondialog, _("pixmap"));
  gtk_widget_set_sensitive(preferencesdialog, 0);
  gtk_widget_show(fileselectiondialog);
}

/*
 * Here we change the font of the main text window. The font is looked up in
 * the preferences struct and applied to the dialog.
 */

void on_fontbutton_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *fontselectiondialog;

  fontselectiondialog = create_fontselectiondialog();
  gtk_font_selection_dialog_set_preview_text(GTK_FONT_SELECTION_DIALOG(fontselectiondialog), _("How about this font?"));
  gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(fontselectiondialog), preferences.font->str);
  set_title(fontselectiondialog, _("%s - select font"));
  set_icon(fontselectiondialog, _("font"));
  gtk_widget_set_sensitive(preferencesdialog, 0);
  gtk_widget_show(fontselectiondialog);
}

/*
 * Save setting, update preferences struct and destroy the dialog.
 * If logging is checked and we are connected, the log file should be created.
 * If logging is unchecked and there is a logfile, we should close it.
 * The autologin settings are applied and the background is set.
 */

void on_preferencesOKbutton_clicked(GtkButton *button, gpointer user_data)
{
  GtkWidget *loggingcheckbutton, *autologincheckbutton, *autologinnameentry,
    *autologinchannelentry, *autologincommandsentry, *maintext,
    *menucheckbutton, *statusbarcheckbutton, *scrollbarcheckbutton, 
    *mainscrolledwindow, *colorradiobutton, *sizecheck, *poscheck;
  GtkStyle *maintextstyle;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *mask;

  /* Save state of logging */
  loggingcheckbutton = lookup_widget(preferencesdialog, "loggingcheckbutton");
  preferences.logging = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON(loggingcheckbutton));

  /* save position and size */
  sizecheck = lookup_widget(preferencesdialog, "sizecheckbutton");
  poscheck = lookup_widget(preferencesdialog, "positioncheckbutton");
  preferences.size = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sizecheck));
  preferences.origin = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(poscheck));

  /* Save the entries for autologin */
  autologincheckbutton = lookup_widget(preferencesdialog, 
    "autologincheckbutton");
  preferences.autologin = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON(autologincheckbutton));
  if (preferences.autologin)
  {
    autologinnameentry = lookup_widget(preferencesdialog, "autologinnameentry");
    autologinchannelentry = lookup_widget(preferencesdialog, "autologinchannelentry");
    autologincommandsentry = lookup_widget(preferencesdialog, "autologincommandsentry");
    preferences.name = g_string_new(gtk_editable_get_chars(GTK_EDITABLE(autologinnameentry), 0, -1));
    preferences.channel = g_string_new(gtk_editable_get_chars(GTK_EDITABLE(autologinchannelentry), 0, -1));
    preferences.commands = g_string_new(gtk_editable_get_chars(GTK_EDITABLE(autologincommandsentry), 0, -1));
  }

  /* Save visibility of some widgets */
  menucheckbutton = lookup_widget(preferencesdialog, "menucheckbutton");
  preferences.menu = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(menucheckbutton));
  widget_visible("mainmenubar", preferences.menu);

  statusbarcheckbutton = lookup_widget(preferencesdialog, "statusbarcheckbutton");
  preferences.statusbar = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(statusbarcheckbutton));
  widget_visible("progressbar", preferences.statusbar);
  widget_visible("mainstatusbar", preferences.statusbar);
  widget_visible("statushbox", preferences.statusbar);
  scrollbarcheckbutton = lookup_widget(preferencesdialog, "scrollbarcheckbutton");
  preferences.scrollbar = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(scrollbarcheckbutton));
  mainscrolledwindow = lookup_widget(mainwindow, "mainscrolledwindow");
  if (preferences.scrollbar)
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mainscrolledwindow), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
  else
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mainscrolledwindow), GTK_POLICY_NEVER, GTK_POLICY_NEVER);

  /* Check if we use a pixmap. If yes, see if we can create it. If we can't, revert to background color */
  colorradiobutton = lookup_widget(preferencesdialog, "colorradiobutton");
  coloryes = (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(colorradiobutton)));
  maintext = lookup_widget(mainwindow, "maintext");
  if (!(coloryes))
  {
    pixmap = gdk_pixmap_create_from_xpm(maintext->window, &mask, NULL, preferences.backgroundpixmap->str);
    if (!(pixmap)) coloryes = TRUE;
  }

  /* Turn logging on/off if it is set/unset and we are connected */
  if (preferences.logging && connected)
  {
    logfilename = g_strconcat(preferencesdir, "/", cvhost, ".log", NULL);
    if ((logfp = fopen(logfilename, "w")) == NULL) 
      g_error(_("Creating %s for writing."), logfilename);
  }
  else 
  { 
    if (!preferences.logging && connected) if (logfp != NULL) fclose(logfp);
  }

  /* Change the background. Either make it a color or a pixmap */
  if (coloryes)
  {
    maintextstyle = gtk_style_new();
    maintextstyle->base[GTK_STATE_NORMAL] = preferences.backgroundcolor;
    maintextstyle->base[GTK_STATE_INSENSITIVE] = preferences.backgroundcolor;
    gtk_widget_set_style(maintext, maintextstyle);
    preferences.backgroundpixmap = g_string_new("");
  }
  else
  {
    maintextstyle = gtk_style_copy(maintext->style);
    gtk_style_ref(maintextstyle);
    maintextstyle->bg_pixmap[GTK_STATE_NORMAL] = pixmap;
    gtk_widget_set_style(maintext, maintextstyle);
    gtk_style_unref(maintextstyle);
  }

  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET (button)));
}

/*
 * Just destroy the preferences dialog when cancel is clicked.
 */

void on_preferencescancelbutton_clicked(GtkButton *button, gpointer user_data)
{
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/*
 * When the preferences window is deleted the tx widget gets focus.
 */
 
void on_preferencesdialog_destroy(GtkObject *object, gpointer user_data)
{
  GtkWidget *mainentry;

  gtk_widget_set_sensitive(mainwindow, 1);
  mainentry = lookup_widget(mainwindow, "mainentry");
  gtk_widget_grab_focus(mainentry);
}

/****************************** ABOUT DIALOG *********************************/

/*
 * Update the about dialog with some information before it is shown.
 */

void on_aboutdialog_show(GtkWidget *widget, gpointer user_data)
{
  GString *info = g_string_new("");
  GtkWidget *aboutcreditslabel, *aboutlicenselabel, *abouthelptext;

  aboutcreditslabel = lookup_widget(widget, "aboutcreditslabel");
  abouthelptext = lookup_widget(widget, "abouthelptext");
  aboutlicenselabel = lookup_widget(widget, "aboutlicenselabel");

  g_string_sprintf(info, _("\n%s version %s\nCopyright (C) 2000-2002 Joop Stakenborg\n\
E-mail: <pa4tu@amsat.org>\nhttp://people.debian.org/~pa3aba/xconvers.html\n\n"), PACKAGE, VERSION);
  gtk_label_set_text(GTK_LABEL(aboutcreditslabel), info->str);

  g_string_sprintf(info, _("PgUp: Scroll the receive window up.\nPgDn: Scroll the receive window down.\n\
Up Arrow: Recall previously transmitted text.\nDn Arrow: Recall next transmitted text.\n\n\
Alt+P: Program menu (exit).\nAlt+H: Host menu (open/close a connection).\n\
Alt+S: Settings menu (preferences).\nAlt+E: Help menu (credits and help).\n"));
  gtk_text_insert(GTK_TEXT(abouthelptext), NULL, NULL, NULL, info->str, -1);

  g_string_sprintf(info, _("Xconvers comes with ABSOLUTELY NO WARRANTY.\nThis is free software, and you are welcome to\n\
redistribute it under certain conditions.\nFor details see the GNU General Public License.\n"));
  gtk_label_set_text(GTK_LABEL(aboutlicenselabel), info->str);

  g_string_free(info, TRUE);
}

/*
 * Clicking OK just destroys this widget and gives focus to the tx widget.
 */

void on_aboutOKbutton_clicked(GtkButton *button, gpointer user_data)
{
  GtkWidget *mainentry;
  
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
  mainentry = lookup_widget(mainwindow, "mainentry");
  gtk_widget_grab_focus(mainentry);  
}

/************************** COLOR SELECTION DIALOG ***************************/

/*
 * We have selected a color, so now we can set the color of the button in the 
 * preferences dialog and update the preview window.
 */

void on_colorok_button_clicked(GtkButton *button, gpointer user_data)
{
  GtkWidget *colorselectiondialog;
  GdkColor newcolor;
  gdouble rgb[3];
  
  colorselectiondialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
  gtk_color_selection_get_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(colorselectiondialog)->colorsel), rgb);
  newcolor.red = rgb[0] * 65535.0;
  newcolor.green = rgb[1] * 65535.0;
  newcolor.blue = rgb[2] * 65535.0;
  gtk_widget_set_sensitive(preferencesdialog, 1);
  updatepreview(newcolor);
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/*
 * Just destroy the dialog when cancel is clicked.
 */

void on_colorcancel_button_clicked (GtkButton *button, gpointer user_data)
{
  gtk_widget_set_sensitive(preferencesdialog, 1);
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/************************* FONT SELECTION DIALOG *****************************/

/*
 * A font is selected. Update the preferences struct, change the text font
 * for the main window and change the preview window.
 */

void on_fontok_button_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *fontselectiondialog, *fontentry;
  
  fontselectiondialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
  preferences.font = g_string_new(gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(fontselectiondialog)));
  fontentry = lookup_widget(preferencesdialog, "fontentry");
  gtk_entry_set_text(GTK_ENTRY(fontentry), preferences.font->str);
  textfont = gdk_font_load(preferences.font->str);
  updatepreview(preferences.backgroundcolor);
  gtk_widget_set_sensitive(preferencesdialog, 1);
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/*
 * Just destroy the dialog when cancel is clicked.
 */

void on_fontcancel_button_clicked (GtkButton *button, gpointer user_data)
{
  gtk_widget_set_sensitive(preferencesdialog, 1);
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/************************* FILE SELECTION DIALOG *****************************/

/*
 * Select a pixmap for the text background and fill the entry in the
 * preferences dialog.
 */

void
on_fileok_button_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *fileselectiondialog, *pixmapentry;
  
  fileselectiondialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
  preferences.backgroundpixmap = g_string_new(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileselectiondialog)));
  pixmapentry = lookup_widget(preferencesdialog, "pixmapentry");
  gtk_entry_set_text(GTK_ENTRY(pixmapentry), preferences.backgroundpixmap->str);
  gtk_widget_set_sensitive(preferencesdialog, 1);
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/*
 * Just destroy the dialog when cancel is clicked.
 */

void
on_filecancel_button_clicked (GtkButton *button, gpointer user_data)
{
  gtk_widget_set_sensitive(preferencesdialog, 1);
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}
