-----------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                E I N F O                                 --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.194 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;   use Atree;
with Namet;   use Namet;
with Nlists;  use Nlists;
with Sinfo;   use Sinfo;
with Snames;  use Snames;
with Stand;   use Stand;
with Output;  use Output;

package body Einfo is

   use Atree.Unchecked_Access;
   --  This is one of the packages that is allowed direct untyped access to
   --  the fields in a node, since it provides the next level abstraction
   --  which incorporates appropriate checks.

   ----------------------------------------------
   -- Usage of Fields in Defining Entity Nodes --
   ----------------------------------------------

   --  The first five of these fields are defined in Sinfo, since they in
   --  the base part of the node. The access routines for these fields and
   --  the corresponding set procedures are defined in Sinfo. The are all
   --  present in all entities.

   --    Chars                          Name1
   --    Next_Entity                    Node2
   --    Scope                          Node3
   --    Homonym                        Node4
   --    Etype                          Node5

   --  The remaining fields are in the node extension and are present only
   --  in entities. The usage of each field depends on the particular entity
   --  kind (see Einfo spec for details).

   --    Discriminant_Constraint        Elist6
   --    Small_Value                    Node6

   --    Alias                          Node7
   --    Corresponding_Concurrent_Type  Node7
   --    Delta_Value                    Node7
   --    Entry_Parameters_Type          Node7
   --    Equivalent_Type                Node7
   --    Lit_Name_Table                 Node7
   --    Renamed_Object                 Node7
   --    Corresponding_Record_Type      Node7

   --    Entry_Index_Constant           Node8
   --    Init_Proc                      Node8
   --    Original_Record_Component      Node8

   --    Digits_Value                   Uint9
   --    Discriminal                    Node9
   --    First_Entity                   Node9
   --    First_Index                    Node9
   --    First_Literal                  Node9
   --    Master_Id                      Node9
   --    Modulus                        Uint9

   --    Component_Type                 Node10
   --    Default_Value                  Node10
   --    Directly_Designated_Type       Node10
   --    Discriminant_Checking_Func     Node10
   --    Discriminant_Default_Value     Node10
   --    Last_Entity                    Node10
   --    Scalar_Range                   Node10

   --    Accept_Address                 Node11
   --    Direct_Full_Declaration        Node11
   --    Entry_Component                Node11
   --    Enumeration_Pos                Uint11
   --    First_Private_Entity           Node11
   --    Parent_Subtype                 Node11
   --    Slice_Range                    Node11
   --    String_Literal_Length          Uint11
   --    Table_High_Bound               Node11

   --    Enumeration_Rep                Uint12
   --    Esize                          Uint12
   --    Interface_Name                 Node12

   --    Corresponding_Integer_Type     Node12
   --    Storage_Size_Variable          Node13
   --    Finalization_Chain_Entity      Node13
   --    Primitive_Operations           Elist13

   --    Controlled_Component_Iterator  Node14
   --    Storage_Size_Variable          Node14
   --    Task_Activation_Chain_Entity   Node14

   --    Access_Disp_Table              Node15
   --    Number_Simple_Entries          Uint15
   --    Private_Subtype_List           Node15

   --    Next_Itype                     Field16

   --    Class_Wide_Type                FIeld17

   ---------------------------------------------
   -- Usage of Flags in Defining Entity Nodes --
   ---------------------------------------------

   --  All flags are unique, there is no overlaying, so each flag is physically
   --  present in every entity. However, for many of the flags, it only makes
   --  sense for them to be set true for certain subsets of entity kinds. See
   --  the spec of Einfo for futher details.

   --    Is_Generic_Type                Flag1
   --    Comes_From_Source              Flag2
   --    Is_Constrained                 Flag3
   --    Is_Frozen                      Flag4
   --    Has_Discriminants              Flag5
   --    Is_Dispatching_Operation       Flag6
   --    Is_Directly_Visible            Flag7
   --    In_Use                         Flag8
   --    Is_Use_Visible                 Flag9
   --    Is_Public                      Flag10
   --    Is_Inlined                     Flag11
   --    Analyzed                       Flag12
   --    Error_Posted                   Flag13
   --    Is_Private_Type                Flag14
   --    Is_Aliased                     Flag15
   --    Is_Volatile                    Flag16
   --    Is_Internal                    Flag17
   --    Is_Delayed                     Flag18
   --    Is_Abstract                    Flag19
   --    Is_Task_Record_Type            Flag20
   --    Has_Master_Entity              Flag21
   --    Needs_No_Actuals               Flag22
   --    Has_Storage_Size_Clause        Flag23
   --    Is_Imported                    Flag24
   --    Is_Limited_Type                Flag25
   --    Has_Completion                 Flag26
   --    Has_Convention_Intrinsic       Flag27
   --    Has_Address_Clause             Flag28
   --    Has_Size_Clause                Flag29
   --    Has_Tasks                      Flag30
   --    Suppress_Access_Checks         Flag31
   --    Suppress_Accessibility_Checks  Flag32
   --    Suppress_Discriminant_Checks   Flag33
   --    Suppress_Division_Checks       Flag34
   --    Suppress_Elaboration_Checks    Flag35
   --    Suppress_Index_Checks          Flag36
   --    Suppress_Length_Checks         Flag37
   --    Suppress_Overflow_Checks       Flag38
   --    Suppress_Range_Checks          Flag39
   --    Suppress_Storage_Checks        Flag40
   --    Suppress_Tag_Checks            Flag41
   --    Is_Controlled                  Flag42
   --    Has_Controlled                 Flag43
   --    Is_Pure                        Flag44
   --    In_Private_Part                Flag45
   --    Has_Subprogram_Body            Flag46
   --    Has_Exit                       Flag47
   --    Is_Package_Body                Flag48
   --    Reachable                      Flag49
   --    Needs_Discr_Check              Flag50
   --    Is_Packed                      Flag51
   --    Is_Entry_Formal                Flag52
   --    Is_Private_Descendant          Flag53
   --    Return_Present                 Flag54
   --    Is_Tagged_Type                 Flag55
   --    Has_Homonym                    Flag56
   --    Is_Private                     Flag57
   --    Non_Binary_Modulus             Flag58
   --    Is_Preelaborable               Flag59
   --    Is_Shared_Passive              Flag60
   --    Is_Remote_Types                Flag61
   --    Is_Remote_Call_Interface       Flag62
   --    Is_Character_Type              Flag63
   --    Is_Intrinsic_Subprogram        Flag64

   --------------------------------
   -- Attribute Access Functions --
   --------------------------------

   function Accept_Address (Id : E) return E is
   begin
      return Node11 (Id);
   end Accept_Address;

   function Access_Disp_Table (Id : E) return E is
   begin
      pragma Assert (Is_Tagged_Type (Id));
      return Node15 (Id);
   end Access_Disp_Table;

   function Alias (Id : E) return E is
   begin
      pragma Assert
        (Is_Overloadable (Id) or else Ekind (Id) = E_Subprogram_Type);
      return Node7 (Id);
   end Alias;

   function Associated_Storage_Pool (Id : E) return E is
   begin
      pragma Assert (Is_Access_Type (Id));
      return Node13 (Id);
   end Associated_Storage_Pool;

   function Class_Wide_Type (Id : E) return E is
   begin
      return Node17 (Id);
   end Class_Wide_Type;

   function Component_Type (Id : E) return E is
   begin
      return Node10 (Id);
   end Component_Type;

   function Controlled_Component_Iterator (Id : E) return E is
   begin
      return Node14 (Id);
   end Controlled_Component_Iterator;

   function Corresponding_Concurrent_Type (Id : E) return E is
   begin
      pragma Assert (Is_Record_Type (Id));
      return Node7 (Id);
   end Corresponding_Concurrent_Type;

   function Corresponding_Integer_Type (Id : E) return E is
   begin
      pragma Assert (Is_Fixed_Type (Id));
      return Node13 (Id);
   end Corresponding_Integer_Type;

   function Corresponding_Record_Type (Id : E) return E is
   begin
      pragma Assert (Is_Concurrent_Type (Id));
      return Node7 (Id);
   end Corresponding_Record_Type;

   function Default_Value (Id : E) return N is
   begin
      pragma Assert (Ekind (Id) = E_In_Parameter);
      return Node10 (Id);
   end Default_Value;

   function Delta_Value (Id : E) return N is
   begin
      pragma Assert (Is_Fixed_Type (Id));
      return Node7 (Id);
   end Delta_Value;

   function Digits_Value (Id : E) return U is
   begin
      pragma Assert (Is_Float_Type (Id) or else Is_Decimal_Fixed_Type (Id));
      return Uint9 (Id);
   end Digits_Value;

   function Direct_Full_Declaration (Id : E) return E is
   begin
      return Node11 (Id);
   end Direct_Full_Declaration;

   function Directly_Designated_Type (Id : E) return E is
   begin
      return Node10 (Id);
   end Directly_Designated_Type;

   function Discriminal (Id : E) return N is
   begin
      pragma Assert (Ekind (Id) = E_Discriminant);
      return Node9 (Id);
   end Discriminal;

   function Discriminant_Checking_Func (Id : E) return E is
   begin
      pragma Assert (Ekind (Id) = E_Component);
      return Node10 (Id);
   end Discriminant_Checking_Func;

   function Discriminant_Constraint (Id : E) return Elist_Id is
   begin
      pragma Assert
        (Is_Composite_Type (Id) and then not Is_Array_Type (Id));
      return Elist6 (Id);
   end Discriminant_Constraint;

   function Discriminant_Default_Value (Id : E) return N is
   begin
      pragma Assert (Ekind (Id) = E_Discriminant);
      return Node10 (Id);
   end Discriminant_Default_Value;

   function Entry_Component (Id : E) return E is
   begin
      return Node11 (Id);
   end Entry_Component;

   function Entry_Index_Constant (Id : E) return E is
   begin
      return Node8 (Id);
   end Entry_Index_Constant;

   function Entry_Parameters_Type (Id : E) return E is
   begin
      return Node7 (Id);
   end Entry_Parameters_Type;

   function Enumeration_Pos (Id : E) return Uint is
   begin
      return Uint11 (Id);
   end Enumeration_Pos;

   function Enumeration_Rep (Id : E) return Uint is
   begin
      return Uint12 (Id);
   end Enumeration_Rep;

   function Equivalent_Type (Id : E) return E is
   begin
      pragma Assert (Ekind (Id) = E_Class_Wide_Subtype);
      return Node7 (Id);
   end Equivalent_Type;

   function Esize (Id : E) return Uint is
   begin
      return Uint12 (Id);
   end Esize;

   function Finalization_Chain_Entity (Id : E) return E is
   begin
      return Node13 (Id);
   end Finalization_Chain_Entity;

   function First_Entity (Id : E) return E is
   begin
      return Node9 (Id);
   end First_Entity;

   function First_Index (Id : E) return N is
   begin
      return Node9 (Id);
   end First_Index;

   function First_Literal (Id : E) return E is
   begin
      return Node9 (Id);
   end First_Literal;

   function First_Private_Entity (Id : E) return E is
   begin
      return Node11 (Id);
   end First_Private_Entity;

   function Has_Address_Clause (Id : E) return B is
   begin
      return Flag28 (Id);
   end Has_Address_Clause;

   function Has_Completion (Id : E) return B is
   begin
      return Flag26 (Id);
   end Has_Completion;

   function Has_Controlled (Id : E) return B is
   begin
      return Flag43 (Id);
   end Has_Controlled;

   function Has_Convention_Intrinsic (Id : E) return B is
   begin
      return Flag27 (Id);
   end Has_Convention_Intrinsic;

   function Has_Discriminants (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag5 (Id);
   end Has_Discriminants;

   function Has_Exit (Id : E) return B is
   begin
      return Flag47 (Id);
   end Has_Exit;

   function Has_Homonym (Id : E) return B is
   begin
      return Flag56 (Id);
   end Has_Homonym;

   function Has_Master_Entity (Id : E) return B is
   begin
      return Flag21 (Id);
   end Has_Master_Entity;

   function Has_Size_Clause (Id : E) return B is
   begin
      return Flag29 (Id);
   end Has_Size_Clause;

   function Has_Storage_Size_Clause (Id : E) return B is
   begin
      return Flag23 (Id);
   end Has_Storage_Size_Clause;

   function Has_Subprogram_Body (Id : E) return B is
   begin
      return Flag46 (Id);
   end Has_Subprogram_Body;

   function Has_Tasks (Id : E) return B is
   begin
      return Flag30 (Id);
   end Has_Tasks;

   function In_Private_Part (Id : E) return B is
   begin
      return Flag45 (Id);
   end In_Private_Part;

   function In_Use (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag8 (Id);
   end In_Use;

   function Init_Proc (Id : E) return E is
   begin
      return Node8 (Id);
   end Init_Proc;

   function Interface_Name (Id : E) return N is
   begin
      return Node12 (Id);
   end Interface_Name;

   function Is_Abstract (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag19 (Id);
   end Is_Abstract;

   function Is_Aliased (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag15 (Id);
   end Is_Aliased;

   function Is_Character_Type (Id : E) return B is
   begin
      return Flag63 (Id);
   end Is_Character_Type;

   function Is_Constrained (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag3 (Id);
   end Is_Constrained;

   function Is_Controlled (Id : E) return B is
   begin
      return Flag42 (Id);
   end Is_Controlled;

   function Is_Delayed (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag18 (Id);
   end Is_Delayed;

   function Is_Directly_Visible (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag7 (Id);
   end Is_Directly_Visible;

   function Is_Dispatching_Operation (Id : E) return B is
   begin
      pragma Assert
        (Is_Overloadable (Id) or else Ekind (Id) = E_Subprogram_Type);
      return Flag6 (Id);
   end Is_Dispatching_Operation;

   function Is_Entry_Formal (Id : E) return B is
   begin
      return Flag52 (Id);
   end Is_Entry_Formal;

   function Is_Frozen (Id : E) return B is
   begin
      return Flag4 (Id);
   end Is_Frozen;

   function Is_Imported (Id : E) return B is
   begin
      return Flag24 (Id);
   end Is_Imported;

   function Is_Inlined (Id : E) return B is
   begin
      pragma Assert
        (Is_Overloadable (Id) or else Ekind (Id) = E_Subprogram_Type);
      return Flag11 (Id);
   end Is_Inlined;

   function Is_Internal (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag17 (Id);
   end Is_Internal;

   function Is_Intrinsic_Subprogram (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag64 (Id);
   end Is_Intrinsic_Subprogram;

   function Is_Named_Number (Id : E) return B is
   begin
      return Ekind (Id) in Named_Kind;
   end Is_Named_Number;

   function Is_Overloadable (Id : E) return B is
   begin
      return Ekind (Id) in Overloadable_Kind;
   end Is_Overloadable;

   function Is_Package_Body (Id : E) return B is
   begin
      return Flag48 (Id);
   end Is_Package_Body;

   function Is_Packed (Id : E) return B is
   begin
      return Flag51 (Id);
   end Is_Packed;

   function Is_Preelaborable (Id : E) return B is
   begin
      return Flag59 (Id);
   end Is_Preelaborable;

   function Is_Private (Id : E) return B is
   begin
      return Flag57 (Id);
   end Is_Private;

   function Is_Private_Descendant (Id : E) return B is
   begin
      return Flag53 (Id);
   end Is_Private_Descendant;

   function Is_Public (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag10 (Id);
   end Is_Public;

   function Is_Pure (Id : E) return B is
   begin
      return Flag44 (Id);
   end Is_Pure;

   function Is_Remote_Call_Interface (Id : E) return B is
   begin
      return Flag62 (Id);
   end Is_Remote_Call_Interface;

   function Is_Remote_Types (Id : E) return B is
   begin
      return Flag61 (Id);
   end Is_Remote_Types;

   function Is_Shared_Passive (Id : E) return B is
   begin
      return Flag60 (Id);
   end Is_Shared_Passive;

   function Is_Subprogram (Id : E) return B is
   begin
      return Ekind (Id) in Subprogram_Kind;
   end Is_Subprogram;

   function Is_Use_Visible (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag9 (Id);
   end Is_Use_Visible;

   function Is_Volatile (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag16 (Id);
   end Is_Volatile;

   function Last_Entity (Id : E) return E is
   begin
      return Node10 (Id);
   end Last_Entity;

   function Lit_Name_Table (Id : E) return E is
   begin
      return Node7 (Id);
   end Lit_Name_Table;

   function Master_Id (Id : E) return E is
   begin
      return Node9 (Id);
   end Master_Id;

   function Modulus (Id : E) return Uint is
   begin
      return Uint9 (Id);
   end Modulus;

   function Needs_Discr_Check (Id : E) return B is
   begin
      return Flag50 (Id);
   end Needs_Discr_Check;

   function Needs_No_Actuals (Id : E) return B is
   begin
      pragma Assert
        (Is_Overloadable (Id) or else Ekind (Id) = E_Subprogram_Type);
      return Flag22 (Id);
   end Needs_No_Actuals;

   function Next_Itype (Id : E) return E is
   begin
      return Node16 (Id);
   end Next_Itype;

   function Next_Literal (Id : E) return E is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Next (Id);
   end Next_Literal;

   function Non_Binary_Modulus (Id : E) return B is
   begin
      pragma Assert (Is_Modular_Integer_Type (Id));
      return Flag58 (Id);
   end Non_Binary_Modulus;

   function Number_Simple_Entries (Id : E) return U is
   begin
      pragma Assert (Is_Concurrent_Type (Id));
      return Uint15 (Id);
   end Number_Simple_Entries;

   function Original_Record_Component (Id : E) return E is
   begin
      --  needs double check E_Component or E_Discriminant ???
      return Node8 (Id);
   end Original_Record_Component;

   function Parent_Subtype (Id : E) return E is
   begin
      pragma Assert (Ekind (Id) in Scalar_Kind);
      return Node11 (Id);
   end Parent_Subtype;

   function Primitive_Operations (Id : E) return Elist_Id is
   begin
      pragma Assert (Is_Tagged_Type (Id));
      return Elist13 (Id);
   end Primitive_Operations;

   function Private_Subtype_List (Id : E) return E is
   begin
      pragma Assert (Ekind (Id) in Private_Kind);
      return Node15 (Id);
   end Private_Subtype_List;

   function Reachable (Id : E) return B is
   begin
      return Flag49 (Id);
   end Reachable;

   function Renamed_Object (Id : E) return N is
   begin
      return Node7 (Id);
   end Renamed_Object;

   function Return_Present (Id : E) return B is
   begin
      return Flag54 (Id);
   end Return_Present;

   function Small_Value (Id : E) return N is
   begin
      return Node6 (Id);
   end Small_Value;

   function Scalar_Range (Id : E) return N is
   begin
      return Node10 (Id);
   end Scalar_Range;

   function Slice_Range (Id : E) return N is
   begin
      return Node11 (Id);
   end Slice_Range;

   function Storage_Size_Variable (Id : E) return E is
   begin
      pragma Assert (Is_Access_Type (Id));
      return Node14 (Id);
   end Storage_Size_Variable;

   function String_Literal_Length (Id : E) return Uint is
   begin
      return Uint11 (Id);
   end String_Literal_Length;

   function Suppress_Access_Checks (Id : E) return B is
   begin
      return Flag31 (Id);
   end Suppress_Access_Checks;

   function Suppress_Accessibility_Checks (Id : E) return B is
   begin
      return Flag32 (Id);
   end Suppress_Accessibility_Checks;

   function Suppress_Discriminant_Checks (Id : E) return B is
   begin
      return Flag33 (Id);
   end Suppress_Discriminant_Checks;

   function Suppress_Division_Checks (Id : E) return B is
   begin
      return Flag34 (Id);
   end Suppress_Division_Checks;

   function Suppress_Elaboration_Checks (Id : E) return B is
   begin
      return Flag35 (Id);
   end Suppress_Elaboration_Checks;

   function Suppress_Index_Checks (Id : E) return B is
   begin
      return Flag36 (Id);
   end Suppress_Index_Checks;

   function Suppress_Length_Checks (Id : E) return B is
   begin
      return Flag37 (Id);
   end Suppress_Length_Checks;

   function Suppress_Overflow_Checks (Id : E) return B is
   begin
      return Flag38 (Id);
   end Suppress_Overflow_Checks;

   function Suppress_Range_Checks (Id : E) return B is
   begin
      return Flag39 (Id);
   end Suppress_Range_Checks;

   function Suppress_Storage_Checks (Id : E) return B is
   begin
      return Flag40 (Id);
   end Suppress_Storage_Checks;

   function Suppress_Tag_Checks (Id : E) return B is
   begin
      return Flag41 (Id);
   end Suppress_Tag_Checks;

   function Table_High_Bound (Id : E) return N is
   begin
      return Node11 (Id);
   end Table_High_Bound;

   function Task_Activation_Chain_Entity (Id : E) return E is
   begin
      return Node14 (Id);
   end Task_Activation_Chain_Entity;

   -----------------------------------
   -- Type Classification Functions --
   -----------------------------------

   function Is_Access_Type (Id : E) return B is
   begin
      return Ekind (Id) in Access_Kind;
   end Is_Access_Type;

   function Is_Array_Type (Id : E) return B is
   begin
      return Ekind (Id) in Array_Kind;
   end Is_Array_Type;

   function Is_Class_Wide_Type (Id : E) return B is
   begin
      return Ekind (Id) in Class_Wide_Kind;
   end Is_Class_Wide_Type;

   function Is_Composite_Type (Id : E) return B is
   begin
      return Ekind (Id) in Composite_Kind;
   end Is_Composite_Type;

   function Is_Concurrent_Type (Id : E) return B is
   begin
      return Ekind (Id) in Concurrent_Kind;
   end Is_Concurrent_Type;

   function Is_Decimal_Fixed_Type (Id : E) return B is
   begin
      return Ekind (Id) in Decimal_Fixed_Kind;
   end Is_Decimal_Fixed_Type;

   function Is_Digits_Type (Id : E) return B is
   begin
      return Ekind (Id) in Digits_Kind;
   end Is_Digits_Type;

   function Is_Discrete_Type (Id : E) return B is
   begin
      return Ekind (Id) in Discrete_Kind;
   end Is_Discrete_Type;

   function Is_Elementary_Type (Id : E) return B is
   begin
      return Ekind (Id) in Elementary_Kind;
   end Is_Elementary_Type;

   function Is_Enumeration_Type (Id : E) return B is
   begin
      return Ekind (Id) in Enumeration_Kind;
   end Is_Enumeration_Type;

   function Is_Fixed_Type (Id : E) return B is
   begin
      return Ekind (Id) in Fixed_Kind;
   end Is_Fixed_Type;

   function Is_Float_Type (Id : E) return B is
   begin
      return Ekind (Id) in Float_Kind;
   end Is_Float_Type;

   function Is_Generic_Type (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag1 (Id);
   end Is_Generic_Type;

   function Is_Incomplete_Type (Id : E) return B is
   begin
      return Ekind (Id) in Incomplete_Kind;
   end Is_Incomplete_Type;

   function Is_Integer_Type (Id : E) return B is
   begin
      return Ekind (Id) in Integer_Kind;
   end Is_Integer_Type;

   function Is_Limited_Type (Id : E) return B is
   begin
      return Flag25 (Id);
   end Is_Limited_Type;

   function Is_Modular_Integer_Type (Id : E) return B is
   begin
      return Ekind (Id) in Modular_Integer_Kind;
   end Is_Modular_Integer_Type;

   function Is_Numeric_Type (Id : E) return B is
   begin
      return Ekind (Id) in Numeric_Kind;
   end Is_Numeric_Type;

   function Is_Ordinary_Fixed_Type (Id : E) return B is
   begin
      return Ekind (Id) in Ordinary_Fixed_Kind;
   end Is_Ordinary_Fixed_Type;

   function Is_Private_Type (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag14 (Id);
   end Is_Private_Type;

   function Is_Protected_Type (Id : E) return B is
   begin
      return Ekind (Id) in Protected_Kind;
   end Is_Protected_Type;

   function Is_Real_Type (Id : E) return B is
   begin
      return Ekind (Id) in Real_Kind;
   end Is_Real_Type;

   function Is_Record_Type (Id : E) return B is
   begin
      return Ekind (Id) in Record_Kind;
   end Is_Record_Type;

   function Is_Scalar_Type (Id : E) return B is
   begin
      return Ekind (Id) in Scalar_Kind;
   end Is_Scalar_Type;

   function Is_Signed_Integer_Type (Id : E) return B is
   begin
      return Ekind (Id) in Signed_Integer_Kind;
   end Is_Signed_Integer_Type;

   function Is_Tagged_Type (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag55 (Id);
   end Is_Tagged_Type;

   function Is_Task_Record_Type (Id : E) return B is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      return Flag20 (Id);
   end Is_Task_Record_Type;

   function Is_Task_Type (Id : E) return B is
   begin
      return Ekind (Id) in Task_Kind;
   end Is_Task_Type;

   function Is_Type (Id : E) return B is
   begin
      return Ekind (Id) in Type_Kind;
   end Is_Type;

   ------------------------------
   -- Attribute Set Procedures --
   ------------------------------

   procedure Set_Accept_Address (Id : E; V : E) is
   begin
      Set_Node11 (Id, V);
   end Set_Accept_Address;

   procedure Set_Access_Disp_Table (Id : E; V : E) is
   begin
      pragma Assert (Is_Tagged_Type (Id));
      Set_Node15 (Id, V);
   end Set_Access_Disp_Table;

   procedure Set_Alias (Id : E; V : E) is
   begin
      pragma Assert
        (Is_Overloadable (Id) or else Ekind (Id) = E_Subprogram_Type);
      Set_Node7 (Id, V);
   end Set_Alias;

   procedure Set_Associated_Storage_Pool (Id : E; V : E) is
   begin
      pragma Assert (Is_Access_Type (Id));
      Set_Node13 (Id, V);
   end Set_Associated_Storage_Pool;

   procedure Set_Class_Wide_Type (Id : E; V : E) is
   begin
      pragma Assert (Is_Type (Id));
      Set_Node17 (Id, V);
   end Set_Class_Wide_Type;

   procedure Set_Component_Type (Id : E; V : E) is
   begin
      Set_Node10 (Id, V);
   end Set_Component_Type;

   procedure Set_Controlled_Component_Iterator (Id : E; V : E) is
   begin
      Set_Node14 (Id, V);
   end Set_Controlled_Component_Iterator;

   procedure Set_Corresponding_Concurrent_Type (Id : E; V : E) is
   begin
      pragma Assert (Is_Record_Type (Id) and then Is_Task_Type (V));
      Set_Node7 (Id, V);
   end Set_Corresponding_Concurrent_Type;

   procedure Set_Corresponding_Integer_Type (Id : E; V : E) is
   begin
      pragma Assert (Is_Fixed_Type (Id));
      Set_Node13 (Id, V);
   end Set_Corresponding_Integer_Type;

   procedure Set_Corresponding_Record_Type (Id : E; V : E) is
   begin
      pragma Assert (Is_Concurrent_Type (Id));
      Set_Node7 (Id, V);
   end Set_Corresponding_Record_Type;

   procedure Set_Default_Value (Id : E; V : N) is
   begin
      pragma Assert (Ekind (Id) = E_In_Parameter);
      Set_Node10 (Id, V);
   end Set_Default_Value;

   procedure Set_Delta_Value (Id : E; V : N) is
   begin
      pragma Assert (Is_Fixed_Type (Id));
      Set_Node7 (Id, V);
   end Set_Delta_Value;

   procedure Set_Digits_Value (Id : E; V : U) is
   begin
      pragma Assert (Is_Float_Type (Id) or else Is_Decimal_Fixed_Type (Id));
      Set_Uint9 (Id, V);
   end Set_Digits_Value;

   procedure Set_Direct_Full_Declaration (Id : E; V : E) is
   begin
      Set_Node11 (Id, V);
   end Set_Direct_Full_Declaration;

   procedure Set_Directly_Designated_Type (Id : E; V : E) is
   begin
      Set_Node10 (Id, V);
   end Set_Directly_Designated_Type;

   procedure Set_Discriminal (Id : E; V : E) is
   begin
      pragma Assert (Ekind (Id) = E_Discriminant);
      Set_Node9 (Id, V);
   end Set_Discriminal;

   procedure Set_Discriminant_Checking_Func (Id  : E; V : E) is
   begin
      pragma Assert
        (Ekind (Id) = E_Component and Ekind (Scope (Id)) in Record_Kind);
      Set_Node10 (Id, V);
   end Set_Discriminant_Checking_Func;

   procedure Set_Discriminant_Constraint (Id : E; V : L) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Elist6 (Id, V);
   end Set_Discriminant_Constraint;

   procedure Set_Discriminant_Default_Value (Id : E; V : N) is
   begin
      Set_Node10 (Id, V);
   end Set_Discriminant_Default_Value;

   procedure Set_Entry_Component (Id : E; V : E) is
   begin
      Set_Node11 (Id, V);
   end Set_Entry_Component;

   procedure Set_Entry_Index_Constant (Id : E; V : E) is
   begin
      Set_Node8 (Id, V);
   end Set_Entry_Index_Constant;

   procedure Set_Entry_Parameters_Type (Id : E; V : E) is
   begin
      Set_Node7 (Id, V);
   end Set_Entry_Parameters_Type;

   procedure Set_Enumeration_Pos (Id : E; V : U) is
   begin
      Set_Uint11 (Id, V);
   end Set_Enumeration_Pos;

   procedure Set_Enumeration_Rep (Id : E; V : U) is
   begin
      Set_Uint12 (Id, V);
   end Set_Enumeration_Rep;

   procedure Set_Equivalent_Type (Id : E; V : E) is
   begin
      pragma Assert (Ekind (Id) = E_Class_Wide_Subtype);
      Set_Node7 (Id, V);
   end Set_Equivalent_Type;

   procedure Set_Esize (Id : E; V : U) is
   begin
      Set_Uint12 (Id, V);
   end Set_Esize;

   procedure Set_Finalization_Chain_Entity (Id : E; V : E) is
   begin
      Set_Node13 (Id, V);
   end Set_Finalization_Chain_Entity;

   procedure Set_First_Entity (Id : E; V : E) is
   begin
      Set_Node9 (Id, V);
   end Set_First_Entity;

   procedure Set_First_Index (Id : E; V : N) is
   begin
      Set_Node9 (Id, V);
   end Set_First_Index;

   procedure Set_First_Literal (Id : E; V : E) is
   begin
      Set_Node9 (Id, V);
   end Set_First_Literal;

   procedure Set_First_Private_Entity (Id : E; V : E) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Node11 (Id, V);
   end Set_First_Private_Entity;

   procedure Set_Has_Address_Clause (Id : E; V : B := True) is
   begin
      Set_Flag28 (Id, V);
   end Set_Has_Address_Clause;

   procedure Set_Has_Completion (Id : E; V : B := True) is
   begin
      Set_Flag26 (Id, V);
   end Set_Has_Completion;

   procedure Set_Has_Controlled (Id : E; V : B := True) is
   begin
      Set_Flag43 (Id, V);
   end Set_Has_Controlled;

   procedure Set_Has_Convention_Intrinsic (Id : E; V : B := True) is
   begin
      Set_Flag27 (Id, V);
   end Set_Has_Convention_Intrinsic;

   procedure Set_Has_Discriminants (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag5 (Id, V);
   end Set_Has_Discriminants;

   procedure Set_Has_Exit (Id : E; V : B := True) is
   begin
      Set_Flag47 (Id, V);
   end Set_Has_Exit;

   procedure Set_Has_Homonym (Id : E; V : B := True) is
   begin
      Set_Flag56 (Id, V);
   end Set_Has_Homonym;

   procedure Set_Has_Master_Entity (Id : E; V : B := True) is
   begin
      Set_Flag21 (Id, V);
   end Set_Has_Master_Entity;

   procedure Set_Has_Size_Clause (Id : E; V : B := True) is
   begin
      Set_Flag29 (Id, V);
   end Set_Has_Size_Clause;

   procedure Set_Has_Storage_Size_Clause (Id : E; V : B := True) is
   begin
      Set_Flag23 (Id, V);
   end Set_Has_Storage_Size_Clause;

   procedure Set_Has_Subprogram_Body (Id : E; V : B := True) is
   begin
      Set_Flag46 (Id, V);
   end Set_Has_Subprogram_Body;

   procedure Set_Has_Tasks (Id : E; V : B := True) is
   begin
      Set_Flag30 (Id, V);
   end Set_Has_Tasks;

   procedure Set_In_Private_Part (Id : E; V : B := True) is
   begin
      Set_Flag45 (Id, V);
   end Set_In_Private_Part;

   procedure Set_In_Use (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag8 (Id, V);
   end Set_In_Use;

   procedure Set_Init_Proc (Id : E; V : E) is
   begin
      Set_Node8 (Id, V);
   end Set_Init_Proc;

   procedure Set_Interface_Name (Id : E; V : N) is
   begin
      Set_Node12 (Id, V);
   end Set_Interface_Name;

   procedure Set_Is_Abstract (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag19 (Id, V);
   end Set_Is_Abstract;

   procedure Set_Is_Aliased (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag15 (Id, V);
   end Set_Is_Aliased;

   procedure Set_Is_Character_Type (Id : E; V : B := True) is
   begin
      Set_Flag63 (Id, V);
   end Set_Is_Character_Type;

   procedure Set_Is_Constrained (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag3 (Id, V);
   end Set_Is_Constrained;

   procedure Set_Is_Controlled (Id : E; V : B := True) is
   begin
      Set_Flag42 (Id, V);
   end Set_Is_Controlled;

   procedure Set_Is_Delayed (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag18 (Id, V);
   end Set_Is_Delayed;

   procedure Set_Is_Directly_Visible (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag7 (Id, V);
   end Set_Is_Directly_Visible;

   procedure Set_Is_Dispatching_Operation (Id : E; V : B := True) is
   begin
      pragma Assert
        (Is_Overloadable (Id) or else Ekind (Id) = E_Subprogram_Type);
      Set_Flag6 (Id, V);
   end Set_Is_Dispatching_Operation;

   procedure Set_Is_Entry_Formal (Id : E; V : B := True) is
   begin
      Set_Flag52 (Id, V);
   end Set_Is_Entry_Formal;

   procedure Set_Is_Frozen (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag4 (Id, V);
   end Set_Is_Frozen;

   procedure Set_Is_Generic_Type (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag1 (Id, V);
   end Set_Is_Generic_Type;

   procedure Set_Is_Imported (Id : E; V : B := True) is
   begin
      Set_Flag24 (Id, V);
   end Set_Is_Imported;

   procedure Set_Is_Inlined (Id : E; V : B := True) is
   begin
      pragma Assert
        (Is_Overloadable (Id) or else Ekind (Id) = E_Subprogram_Type);
      Set_Flag11 (Id, V);
   end Set_Is_Inlined;

   procedure Set_Is_Internal (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag17 (Id, V);
   end Set_Is_Internal;

   procedure Set_Is_Intrinsic_Subprogram (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag64 (Id, V);
   end Set_Is_Intrinsic_Subprogram;

   procedure Set_Is_Limited_Type (Id : E; V : B := True) is
   begin
      Set_Flag25 (Id, V);
   end Set_Is_Limited_Type;

   procedure Set_Is_Preelaborable (Id : E; V : B := True) is
   begin
      Set_Flag59 (Id, V);
   end Set_Is_Preelaborable;

   procedure Set_Is_Package_Body (Id : E; V : B := True) is
   begin
      Set_Flag48 (Id, V);
   end Set_Is_Package_Body;

   procedure Set_Is_Packed (Id : E; V : B := True) is
   begin
      Set_Flag51 (Id, V);
   end Set_Is_Packed;

   procedure Set_Is_Private (Id : E; V : B := True) is
   begin
      Set_Flag57 (Id, V);
   end Set_Is_Private;

   procedure Set_Is_Private_Descendant (Id : E; V : B := True) is
   begin
      Set_Flag53 (Id, V);
   end Set_Is_Private_Descendant;

   procedure Set_Is_Private_Type (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag14 (Id, V);
   end Set_Is_Private_Type;

   procedure Set_Is_Public (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag10 (Id, V);
   end Set_Is_Public;

   procedure Set_Is_Pure (Id : E; V : B := True) is
   begin
      Set_Flag44 (Id, V);
   end Set_Is_Pure;

   procedure Set_Is_Remote_Call_Interface (Id : E; V : B := True) is
   begin
      Set_Flag62 (Id, V);
   end Set_Is_Remote_Call_Interface;

   procedure Set_Is_Remote_Types (Id : E; V : B := True) is
   begin
      Set_Flag61 (Id, V);
   end Set_Is_Remote_Types;

   procedure Set_Is_Shared_Passive (Id : E; V : B := True) is
   begin
      Set_Flag60 (Id, V);
   end Set_Is_Shared_Passive;

   procedure Set_Is_Tagged_Type (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag55 (Id, V);
   end Set_Is_Tagged_Type;

   procedure Set_Is_Task_Record_Type (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag20 (Id, V);
   end Set_Is_Task_Record_Type;

   procedure Set_Is_Use_Visible (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag9 (Id, V);
   end Set_Is_Use_Visible;

   procedure Set_Is_Volatile (Id : E; V : B := True) is
   begin
      pragma Assert (Nkind (Id) in N_Entity);
      Set_Flag16 (Id, V);
   end Set_Is_Volatile;

   procedure Set_Last_Entity (Id : E; V : E) is
   begin
      Set_Node10 (Id, V);
   end Set_Last_Entity;

   procedure Set_Lit_Name_Table (Id : E; V : E) is
   begin
      Set_Node7 (Id, V);
   end Set_Lit_Name_Table;

   procedure Set_Master_Id (Id : E; V : E) is
   begin
      Set_Node9 (Id, V);
   end Set_Master_Id;

   procedure Set_Modulus (Id : E; V : U) is
   begin
      Set_Uint9 (Id, V);
   end Set_Modulus;

   procedure Set_Needs_Discr_Check (Id : E; V : B := True) is
   begin
      pragma Assert (Ekind (Id) = E_Component);
      Set_Flag50 (Id, V);
   end Set_Needs_Discr_Check;

   procedure Set_Needs_No_Actuals (Id : E; V : B := True) is
   begin
      pragma Assert
        (Is_Overloadable (Id) or else Ekind (Id) = E_Subprogram_Type);
      Set_Flag22 (Id, V);
   end Set_Needs_No_Actuals;

   procedure Set_Next_Itype (Id : E; V : E) is
   begin
      Set_Node16 (Id, V);
   end Set_Next_Itype;

   procedure Set_Non_Binary_Modulus (Id : E; V : B := True) is
   begin
      pragma Assert (Is_Modular_Integer_Type (Id));
      Set_Flag58 (Id, V);
   end Set_Non_Binary_Modulus;

   procedure Set_Number_Simple_Entries (Id : E; V : U) is
   begin
      pragma Assert (Is_Concurrent_Type (Id));
      Set_Uint15 (Id, V);
   end Set_Number_Simple_Entries;

   procedure Set_Original_Record_Component (Id : E; V : E) is
   begin
      --  Needs double check E_Component or E_Discriminant ???
      Set_Node8 (Id, V);
   end Set_Original_Record_Component;

   procedure Set_Parent_Subtype (Id : E; V : E) is
   begin
      pragma Assert (Ekind (Id) in Scalar_Kind);
      Set_Node11 (Id, V);
   end Set_Parent_Subtype;

   procedure Set_Primitive_Operations (Id : E; V : L) is
   begin
      pragma Assert (Is_Tagged_Type (Id));
      Set_Elist13 (Id, V);
   end Set_Primitive_Operations;

   procedure Set_Private_Subtype_List (Id : E; V : E) is
   begin
      pragma Assert (Ekind (Id) in Private_Kind);
      Set_Node15 (Id, V);
   end Set_Private_Subtype_List;

   procedure Set_Reachable (Id : E; V : B := True) is
   begin
      Set_Flag49 (Id, V);
   end Set_Reachable;

   procedure Set_Renamed_Object (Id : E; V : N) is
   begin
      Set_Node7 (Id, V);
   end Set_Renamed_Object;

   procedure Set_Return_Present (Id : E; V : B := True) is
   begin
      Set_Flag54 (Id, V);
   end Set_Return_Present;

   procedure Set_Small_Value (Id : E; V : N) is
   begin
      Set_Node6 (Id, V);
   end Set_Small_Value;

   procedure Set_Scalar_Range (Id : E; V : N) is
   begin
      Set_Node10 (Id, V);
   end Set_Scalar_Range;

   procedure Set_Slice_Range (Id : E; V : N) is
   begin
      pragma Assert (Ekind (Id) = E_Slice_Subtype);
      Set_Node11 (Id, V);
   end Set_Slice_Range;

   procedure Set_Storage_Size_Variable (Id : E; V : E) is
   begin
      pragma Assert (Is_Access_Type (Id));
      Set_Node14 (Id, V);
   end Set_Storage_Size_Variable;

   procedure Set_String_Literal_Length (Id : E; V : U) is
   begin
      pragma Assert (Ekind (Id) = E_String_Literal_Subtype);
      Set_Uint11 (Id, V);
   end Set_String_Literal_Length;

   procedure Set_Suppress_Access_Checks (Id : E; V : B := True) is
   begin
      Set_Flag31 (Id, V);
   end Set_Suppress_Access_Checks;

   procedure Set_Suppress_Accessibility_Checks (Id : E; V : B := True) is
   begin
      Set_Flag32 (Id, V);
   end Set_Suppress_Accessibility_Checks;

   procedure Set_Suppress_Discriminant_Checks (Id : E; V : B := True) is
   begin
      Set_Flag33 (Id, V);
   end Set_Suppress_Discriminant_Checks;

   procedure Set_Suppress_Division_Checks (Id : E; V : B := True) is
   begin
      Set_Flag34 (Id, V);
   end Set_Suppress_Division_Checks;

   procedure Set_Suppress_Elaboration_Checks (Id : E; V : B := True) is
   begin
      Set_Flag35 (Id, V);
   end Set_Suppress_Elaboration_Checks;

   procedure Set_Suppress_Index_Checks (Id : E; V : B := True) is
   begin
      Set_Flag36 (Id, V);
   end Set_Suppress_Index_Checks;

   procedure Set_Suppress_Length_Checks (Id : E; V : B := True) is
   begin
      Set_Flag37 (Id, V);
   end Set_Suppress_Length_Checks;

   procedure Set_Suppress_Overflow_Checks (Id : E; V : B := True) is
   begin
      Set_Flag38 (Id, V);
   end Set_Suppress_Overflow_Checks;

   procedure Set_Suppress_Range_Checks (Id : E; V : B := True) is
   begin
      Set_Flag39 (Id, V);
   end Set_Suppress_Range_Checks;

   procedure Set_Suppress_Storage_Checks (Id : E; V : B := True) is
   begin
      Set_Flag40 (Id, V);
   end Set_Suppress_Storage_Checks;

   procedure Set_Suppress_Tag_Checks (Id : E; V : B := True) is
   begin
      Set_Flag41 (Id, V);
   end Set_Suppress_Tag_Checks;

   procedure Set_Table_High_Bound (Id : E; V : N) is
   begin
      pragma Assert (Ekind (Id) = E_Enum_Table_Type);
      Set_Node11 (Id, V);
   end Set_Table_High_Bound;

   procedure Set_Task_Activation_Chain_Entity (Id : E; V : E) is
   begin
      Set_Node14 (Id, V);
   end Set_Task_Activation_Chain_Entity;

   -------------------
   -- Append_Entity --
   -------------------

   procedure Append_Entity (Id : Entity_Id; V : Entity_Id) is
   begin
      if Last_Entity (V) = Empty then
         Set_First_Entity (V, Id);
      else
         Set_Next_Entity (Last_Entity (V), Id);
      end if;

      Set_Next_Entity (Id, Empty);
      Set_Scope (Id, V);
      Set_Last_Entity (V, Id);
   end Append_Entity;

   --------------------
   -- Base_Init_Proc --
   --------------------

   function Base_Init_Proc (Id : E) return E is
      Full_Type : E;

   begin
      pragma Assert (Ekind (Id) in Type_Kind);

      if Ekind (Id) in Private_Kind then
         Full_Type := Full_Declaration (Id);
      else
         Full_Type := Id;
      end if;

      if No (Full_Type) then
         return Empty;
      elsif Is_Task_Type (Full_Type) then
         return Init_Proc (Corresponding_Record_Type (Base_Type (Full_Type)));
      else
         return Init_Proc (Base_Type (Full_Type));
      end if;
   end Base_Init_Proc;

   ---------------
   -- Base_Type --
   ---------------

   function Base_Type (Id : E) return E is
   begin
      case Ekind (Id) is
         when E_Enumeration_Subtype     |
              E_Signed_Integer_Subtype  |
              E_Modular_Integer_Subtype |
              E_Float_Subtype           |
              E_Ordinary_Fixed_Subtype  |
              E_Decimal_Fixed_Subtype   |
              E_Array_Subtype           |
              E_String_Subtype          |
              E_Record_Subtype          |
              E_Private_Subtype         |
              E_Limited_Private_Subtype |
              E_Access_Subtype          |
              E_Protected_Subtype       |
              E_Task_Subtype            |
              E_String_Literal_Subtype  |
              E_Class_Wide_Subtype      =>
            return Etype (Id);

         when E_Slice_Subtype =>
            return Base_Type (Etype (Id));

         when others =>
            return Id;
      end case;
   end Base_Type;

   --------------------
   -- Constant_Value --
   --------------------

   function Constant_Value (Id : E) return N is
   begin
      pragma Assert (Nkind (Id) in N_Entity);

      if Nkind (Parent (Id)) = N_Object_Renaming_Declaration then
         return Renamed_Object (Id);
      else
         if Present (Expression (Parent (Id))) then
            return (Expression (Parent (Id)));
         elsif Present (Full_Declaration (Id)) then
            return (Expression (Parent (Full_Declaration (Id))));
         else
            return Empty;
         end if;
      end if;
   end Constant_Value;

   ----------------------
   -- Declaration_Node --
   ----------------------

   function Declaration_Node (Id : E) return N is
      P : Node_Id;

   begin
      P := Parent (Id);

      loop
         if Nkind (P) /= N_Selected_Component
           and then Nkind (P) /= N_Expanded_Name  -- necessary ???
         then
            return P;
         else
            P := Parent (P);
         end if;
      end loop;

   end Declaration_Node;

   ---------------------
   -- Designated_Type --
   ---------------------

   function Designated_Type (Id : E) return E is
      Desig_Type : E;

   begin
      Desig_Type := Directly_Designated_Type (Id);

      if (Ekind (Desig_Type) = E_Incomplete_Type
        and then Present (Full_Declaration (Desig_Type)))
      then
         return Full_Declaration (Desig_Type);

      elsif Is_Class_Wide_Type (Desig_Type)
        and then Ekind (Etype (Desig_Type)) = E_Incomplete_Type
        and then Present (Full_Declaration (Etype (Desig_Type)))
      then
         return Class_Wide_Type (Full_Declaration (Etype (Desig_Type)));

      else
         return Desig_Type;
      end if;
   end Designated_Type;

   ---------------------
   -- First_Component --
   ---------------------

   function First_Component (Id : E) return E is
      Comp_Id : E;

   begin
      pragma Assert
        (Is_Record_Type (Id) or else Is_Incomplete_Type (Id));

      Comp_Id := First_Entity (Id);

      while Present (Comp_Id) loop
         exit when Ekind (Comp_Id) = E_Component;
         Comp_Id := Next_Entity (Comp_Id);
      end loop;

      return Comp_Id;
   end First_Component;

   ------------------------
   -- First_Discriminant --
   ------------------------

   function First_Discriminant (Id : E) return E is
      Ent : Entity_Id;

   begin
      pragma Assert (Has_Discriminants (Id));

      Ent := First_Entity (Id);

      if Chars (Ent) = Name_uTag then
         pragma Assert (Is_Tagged_Type (Id));
         return Next_Entity (Ent);
      else
         return Ent;
      end if;
   end First_Discriminant;

   ------------------
   -- First_Formal --
   ------------------

   function First_Formal (Id : E) return E is
      Formal : E;

   begin
      pragma Assert
        (Is_Overloadable (Id)
          or else Ekind (Id) = E_Entry_Family
          or else Ekind (Id) = E_Subprogram_Type);

      if Ekind (Id) = E_Enumeration_Literal then
         return Empty;

      else
         Formal := First_Entity (Id);

         if Present (Formal) and then Ekind (Formal) in Formal_Kind then
            return Formal;
         else
            return Empty;
         end if;
      end if;
   end First_Formal;

   ----------------------
   -- Full_Declaration --
   ----------------------

   function Full_Declaration (Id : E) return E is
      Decl : E;

   begin
      Decl := Direct_Full_Declaration (Id);

      if Present (Decl)
        and then Ekind (Decl) in Private_Kind
      then
         Decl := Direct_Full_Declaration (Decl);
      end if;

      return Decl;
   end Full_Declaration;

   ---------------------
   -- Is_Boolean_Type --
   ---------------------

   function Is_Boolean_Type (Id : E) return B is
   begin
      return Root_Type (Id) = Standard_Boolean;
   end Is_Boolean_Type;

   --------------------
   -- Is_String_Type --
   --------------------

   function Is_String_Type (Id : E) return B is
   begin
      return Ekind (Id) in String_Kind
        or else (Ekind (Id) = E_Slice_Subtype
                  and then Is_String_Type (Base_Type (Id)))
        or else (Is_Array_Type (Id)
                  and then Number_Dimensions (Id) = 1
                  and then Is_Character_Type (Component_Type (Id)));
   end Is_String_Type;

   --------------------
   -- Next_Component --
   --------------------

   function Next_Component (Id : E) return E is
      Comp_Id : E;

   begin
      Comp_Id := Next_Entity (Id);

      while Present (Comp_Id) loop
         exit when Ekind (Comp_Id) = E_Component;
         Comp_Id := Next_Entity (Comp_Id);
      end loop;

      return Comp_Id;
   end Next_Component;

   -----------------------
   -- Next_Discriminant --
   -----------------------

   function Next_Discriminant (Id : E) return E is
      D : constant E := Next_Entity (Id);

   begin
      pragma Assert (Ekind (Id) = E_Discriminant);

      if Present (D) and then Ekind (D) = E_Discriminant then
         return D;
      else
         return Empty;
      end if;
   end Next_Discriminant;

   -----------------
   -- Next_Formal --
   -----------------

   function Next_Formal (Id : E) return E is
      P : E;

   begin
      --  Follow the chain of declared entities as long as the kind of
      --  the entity corresponds to a formal parameter. Skip internal
      --  entities that may have been created for implicit subtypes,
      --  in the process of analyzing default expressions.

      P := Id;

      loop
         P := Next_Entity (P);

         if No (P) or else Ekind (P) in Formal_Kind then
            return P;
         elsif not Is_Internal (P) then
            return Empty;
         end if;
      end loop;
   end Next_Formal;

   ----------------
   -- Next_Index --
   ----------------

   function Next_Index (Id : Node_Id) return Node_Id is
   begin
      return Next (Id);
   end Next_Index;

   --------------------
   -- Next_Overloads --
   --------------------

   function Next_Overloads (Id : E) return E is
   begin
      pragma Assert
        (Is_Overloadable (Id) or else Ekind (Id) = E_Subprogram_Type);
      return Homonym (Id);
   end Next_Overloads;

   -----------------------
   -- Number_Dimensions --
   -----------------------

   function Number_Dimensions (Id : E) return Pos is
      N : Int;
      T : Node_Id;

   begin
      N := 0;
      T := First_Index (Id);

      while Present (T) loop
         N := N + 1;
         T := Next (T);
      end loop;

      return N;
   end Number_Dimensions;

   --------------------
   -- Parameter_Mode --
   --------------------

   function Parameter_Mode (Id : E) return Formal_Kind is
   begin
      return Ekind (Id);
   end Parameter_Mode;

   ---------------
   -- Root_Type --
   ---------------

   function Root_Type (Id : E) return E is
      T : E;

   begin
      pragma Assert (Nkind (Id) in N_Entity);
      T := Id;

      while T /= Etype (T) loop
         T := Etype (T);
      end loop;

      return T;
   end Root_Type;

   ------------------
   -- Subtype_Kind --
   ------------------

   function Subtype_Kind (K : Entity_Kind) return Entity_Kind is
      Kind : Entity_Kind;

   begin
      case K is
         when Access_Kind               => Kind := E_Access_Subtype;

         when E_Array_Type              |
              E_Array_Subtype           => Kind := E_Array_Subtype;

         when E_Class_Wide_Type         |
              E_Class_Wide_Subtype      => Kind := E_Class_Wide_Subtype;

         when E_Decimal_Fixed_Type      |
              E_Decimal_Fixed_Subtype   => Kind := E_Decimal_Fixed_Subtype;

         when E_Ordinary_Fixed_Type     |
              E_Ordinary_Fixed_Subtype  => Kind := E_Ordinary_Fixed_Subtype;

         when E_Private_Type            |
              E_Private_Subtype         => Kind := E_Private_Subtype;

         when E_Limited_Private_Type    |
              E_Limited_Private_Subtype => Kind := E_Limited_Private_Subtype;

         when E_Record_Type             |
              E_Record_Subtype          => Kind := E_Record_Subtype;

         when E_String_Type             |
              E_String_Subtype          => Kind := E_String_Subtype;

         when Enumeration_Kind          => Kind := E_Enumeration_Subtype;
         when Float_Kind                => Kind := E_Float_Subtype;
         when Signed_Integer_Kind       => Kind := E_Signed_Integer_Subtype;
         when Modular_Integer_Kind      => Kind := E_Modular_Integer_Subtype;
         when Protected_Kind            => Kind := E_Protected_Subtype;
         when Task_Kind                 => Kind := E_Task_Subtype;

         when others =>
            pragma Assert (False); null;
      end case;

      return Kind;
   end Subtype_Kind;

   -------------------
   -- Tag_Component --
   -------------------

   function Tag_Component (Id : E) return E is
      Ent : Entity_Id;

   begin
      pragma Assert (Is_Tagged_Type (Id));

      Ent := First_Entity (Id);
      while Present (Ent) loop
         if Chars (Ent) = Name_uTag then
            return Ent;
         end if;

         Ent := Next_Entity (Ent);
      end loop;

      --  Here if no _Tag component found (some kind of compiler error!)

      pragma Assert (False);
   end Tag_Component;

   ---------------------
   -- Type_High_Bound --
   ---------------------

   function Type_High_Bound (Id : E) return Node_Id is
   begin
      return High_Bound (Scalar_Range (Id));
   end Type_High_Bound;

   --------------------
   -- Type_Low_Bound --
   --------------------

   function Type_Low_Bound (Id : E) return Node_Id is
   begin
      return Low_Bound (Scalar_Range (Id));
   end Type_Low_Bound;

   ------------------------
   -- Write_Entity_Flags --
   ------------------------

   procedure Write_Entity_Flags (Id : Entity_Id; Prefix : String) is

      procedure W (Flag_Name : String; Flag : Boolean) is
      begin
         if Flag then
            Write_Str (Prefix);
            Write_Str (Flag_Name);
            Write_Str (" = True");
            Write_Eol;
         end if;
      end W;

   --  Start of processing for Write_Entity_Flags

   begin
      W ("Comes_From_Source",             Flag2  (Id));
      W ("Has_Address_Clause",            Flag28 (Id));
      W ("Has_Completion",                Flag26 (Id));
      W ("Has_Controlled",                Flag43 (Id));
      W ("Has_Convention_Intrinsic",      Flag27 (Id));
      W ("Has_Discriminants",             Flag5  (Id));
      W ("Has_Exit",                      Flag47 (Id));
      W ("Has_Homonym",                   Flag56 (Id));
      W ("Has_Master_Entity",             Flag21 (Id));
      W ("Has_Size_Clause",               Flag29 (Id));
      W ("Has_Storage_Size_Clause",       Flag23 (Id));
      W ("Has_Subprogram_Body",           Flag46 (Id));
      W ("Has_Tasks",                     Flag30 (Id));
      W ("In_Private_Part",               Flag45 (Id));
      W ("In_Use",                        Flag8  (Id));
      W ("Is_Abstract",                   Flag19 (Id));
      W ("Is_Aliased",                    Flag15 (Id));
      W ("Is_Character_Type",             Flag63 (Id));
      W ("Is_Constrained",                Flag3  (Id));
      W ("Is_Controlled",                 Flag42 (Id));
      W ("Is_Delayed",                    Flag18 (Id));
      W ("Is_Directly_Visible",           Flag7  (Id));
      W ("Is_Dispatching_Operation",      Flag6  (Id));
      W ("Is_Entry_Formal",               Flag52 (Id));
      W ("Is_Frozen",                     Flag4  (Id));
      W ("Is_Generic_Type",               Flag1  (Id));
      W ("Is_Imported",                   Flag24 (Id));
      W ("Is_Inlined",                    Flag11 (Id));
      W ("Is_Internal",                   Flag17 (Id));
      W ("Is_Limited_Type",               Flag25 (Id));
      W ("Is_Package_Body",               Flag48 (Id));
      W ("Is_Packed",                     Flag51 (Id));
      W ("Is_Preelaborable",              Flag59 (Id));
      W ("Is_Private",                    Flag57 (Id));
      W ("Is_Private_Descendant",         Flag53 (Id));
      W ("Is_Private_Type",               Flag14 (Id));
      W ("Is_Public",                     Flag10 (Id));
      W ("Is_Pure",                       Flag44 (Id));
      W ("Is_Remote_Call_Interface",      Flag62 (Id));
      W ("Is_Remote_Types",               Flag61 (Id));
      W ("Is_Shared_Passive",             Flag60 (Id));
      W ("Is_Tagged_Type",                Flag55 (Id));
      W ("Is_Task_Record_Type",           Flag20 (Id));
      W ("Is_Use_Visible",                Flag9  (Id));
      W ("Is_Volatile",                   Flag16 (Id));
      W ("Needs_Discr_Check",             Flag50 (Id));
      W ("Needs_No_Actuals",              Flag22 (Id));
      W ("Non_Binary_Modulus",            Flag58 (Id));
      W ("Reachable",                     Flag49 (Id));
      W ("Return_Present",                Flag54 (Id));
      W ("Suppress_Access_Checks",        Flag31 (Id));
      W ("Suppress_Accessibility_Checks", Flag32 (Id));
      W ("Suppress_Discriminant_Checks",  Flag33 (Id));
      W ("Suppress_Division_Checks",      Flag34 (Id));
      W ("Suppress_Elaboration_Checks",   Flag35 (Id));
      W ("Suppress_Index_Checks",         Flag36 (Id));
      W ("Suppress_Length_Checks",        Flag37 (Id));
      W ("Suppress_Overflow_Checks",      Flag38 (Id));
      W ("Suppress_Range_Checks",         Flag39 (Id));
      W ("Suppress_Storage_Checks",       Flag40 (Id));
      W ("Suppress_Tag_Checks",           Flag41 (Id));

   end Write_Entity_Flags;

   -----------------------
   -- Write_Entity_Info --
   -----------------------

   procedure Write_Entity_Info (Id : Entity_Id; Prefix : String) is

      procedure Write_Kind (Id : Entity_Id) is
         K : constant String := Entity_Kind'Image (Ekind (Id));

      begin
         Write_Str (Prefix);
         Write_Str ("   Kind    ");

         if Is_Type (Id) and then Is_Tagged_Type (Id) then
            Write_Str ("TAGGED ");
         end if;

         Write_Str (K (3 .. K'Length));
         Write_Str (" ");

         if Is_Type (Id) and then Is_Private_Type (Id) then
            Write_Str ("Is_Private_Type ");
         end if;
      end Write_Kind;

      procedure Write_Attribute (Which : String; Nam : E) is
      begin
         Write_Str (Prefix);
         Write_Str (Which);
         Write_Int (Int (Nam));
         Write_Str (" ");
         Write_Name (Chars (Nam));
         Write_Str (" ");
      end Write_Attribute;

   begin
      Write_Eol;
      Write_Attribute ("Name ", Id);
      Write_Int (Int (Id));
      Write_Eol;
      Write_Kind (Id);
      Write_Eol;
      Write_Attribute ("   Type    ", Etype (Id));
      Write_Eol;
      Write_Attribute ("   Scope   ", Scope (Id));
      Write_Eol;

      case Ekind (Id) is

         when Discrete_Kind =>
            Write_Str ("Bounds: Id = ");

            if Present (Scalar_Range (Id)) then
               Write_Int (Int (Type_Low_Bound (Id)));
               Write_Str (" .. Id = ");
               Write_Int (Int (Type_High_Bound (Id)));
            else
               Write_Str ("Empty");
            end if;

            Write_Eol;

         when Array_Kind =>
            declare
               Index : E;

            begin
               Write_Attribute ("   Component Type    ",
                                                   Component_Type (Id));
               Write_Eol;
               Write_Str (Prefix);
               Write_Str ("   Indices ");

               Index := First_Index (Id);

               while Present (Index) loop
                  Write_Attribute (" ", Etype (Index));
                  Index := Next_Index (Index);
               end loop;

               Write_Eol;
            end;

         when Access_Kind =>
               Write_Attribute
                 ("   Directly Designated Type ",
                  Directly_Designated_Type (Id));
               Write_Eol;

         when Overloadable_Kind =>
            if Present (Homonym (Id)) then
               Write_Str ("   Homonym   ");
               Write_Name (Chars (Homonym (Id)));
               Write_Str ("   ");
               Write_Int (Int (Homonym (Id)));
               Write_Eol;
            end if;

            Write_Eol;

         when E_Component =>
            if Ekind (Scope (Id)) in Record_Kind then
               Write_Attribute (
                  "   Original_Record_Component   ",
                  Original_Record_Component (Id));
               Write_Int (Int (Original_Record_Component (Id)));
               Write_Eol;
            end if;

         when others => null;
      end case;
   end Write_Entity_Info;

   -----------------------
   -- Write_Field6_Name --
   -----------------------

   procedure Write_Field6_Name  (Id : Entity_Id) is
   begin
      case Ekind (Id) is
         when Concurrent_Kind |
              Incomplete_Kind |
              Record_Kind     =>
            Write_Str ("Discriminant_Constraint");

         when Fixed_Kind =>
            Write_Str ("Small_Value");

         when others =>
            Write_Str ("Field6");
      end case;
   end Write_Field6_Name;

   -----------------------
   -- Write_Field7_Name --
   -----------------------

   procedure Write_Field7_Name  (Id : Entity_Id) is
   begin
      case Ekind (Id) is
         when E_Enumeration_Literal |
              E_Function            |
              E_Procedure           =>
            Write_Str ("Alias");

         when E_Record_Type =>
            Write_Str ("Corresponding_Concurrent_Type");

         when E_Entry |
              E_Entry_Family =>
            Write_Str ("Entry_Parameters_Type");

         when E_Class_Wide_Subtype =>
            Write_Str ("Equivalent_Type");

         when Enumeration_Kind =>
            Write_Str ("Lit_Name_Table");

         when Fixed_Kind =>
            Write_Str ("Delta_Value");

         when E_Constant |
              E_Variable =>
            Write_Str ("Renamed_Object");

         when Task_Kind =>
            Write_Str ("Corresponding_Record_Type");

         when others =>
            Write_Str ("FIeld7");
      end case;
   end Write_Field7_Name;

   -----------------------
   -- Write_Field8_Name --
   -----------------------

   procedure Write_Field8_Name  (Id : Entity_Id) is
   begin
      case Ekind (Id) is
         when E_Entry |
              E_Entry_Family =>
            Write_Str ("Entry_Index_Constant");

         when Type_Kind =>
            Write_Str ("Init_Proc");

         when E_Component =>
            Write_Str ("Original_Record_Component");

         when others =>
            Write_Str ("Field8");
      end case;
   end Write_Field8_Name;

   -----------------------
   -- Write_Field9_Name --
   -----------------------

   procedure Write_Field9_Name  (Id : Entity_Id) is
   begin
      case Ekind (Id) is
         when Digits_Kind =>
            Write_Str ("Digits_Value");

         when E_Discriminant =>
            Write_Str ("Discriminal");

         when E_Block          |
              Class_Wide_Kind  |
              Concurrent_Kind  |
              E_Entry          |
              E_Entry_Family   |
              E_Function       |
              E_Loop           |
              E_Package        |
              E_Procedure      |
              E_Record_Type    |
              E_Record_Subtype =>

            Write_Str ("First_Entity");

         when Array_Kind =>
            Write_Str ("First_Index");

         when Enumeration_Kind =>
            Write_Str ("First_Literal");

         when Access_Kind =>
            Write_Str ("Master_Id");

         when Modular_Integer_Kind =>
            Write_Str ("Modulus");

         when others =>
            Write_Str ("Field9");
      end case;
   end Write_Field9_Name;

   ------------------------
   -- Write_Field10_Name --
   ------------------------

   procedure Write_Field10_Name  (Id : Entity_Id) is
   begin
      case Ekind (Id) is
         when Array_Kind =>
            Write_Str ("Component_Type");

         when E_In_Parameter         |
              E_Generic_In_Parameter =>
            Write_Str ("Default_Value");

         when Access_Kind =>
            Write_Str ("Directly_Designated_Type");

         when E_Component =>
            Write_Str ("Discriminant_Checking_Func");

         when E_Discriminant =>
            Write_Str ("Discriminant_Default_Value");

         when E_Block          |
              Class_Wide_Kind  |
              Concurrent_Kind  |
              E_Entry          |
              E_Entry_Family   |
              E_Function       |
              E_Loop           |
              E_Package        |
              E_Procedure      |
              E_Record_Type    |
              E_Record_Subtype =>
            Write_Str ("Last_Entity");

         when Scalar_Kind =>
            Write_Str ("Scalar_Kind");

         when others =>
            Write_Str ("Field10");
      end case;
   end Write_Field10_Name;

   ------------------------
   -- Write_Field11_Name --
   ------------------------

   procedure Write_Field11_Name  (Id : Entity_Id) is
   begin
      case Ekind (Id) is
         when E_Entry        |
              E_Entry_Family =>
            Write_Str ("Accept_Address");

         when Formal_Kind =>
            Write_Str ("Entry_Component");

         when E_Enumeration_Literal =>
            Write_Str ("Enumeration_Rep");

         when E_String_Literal_Subtype =>
            Write_Str ("String_Literal_Length");

         when E_Enum_Table_Type =>
            Write_Str ("Table_High_Bound");

         when Scalar_Kind =>
            Write_Str ("Parent_Subtype");

         when E_Slice_Subtype =>
            Write_Str ("Slice_Range");

         when Incomplete_Kind |
              E_Constant      =>
            Write_Str ("Directly_Designated_Type");

         when E_Package       |
              Concurrent_Kind =>
            Write_Str ("First_Private_Entity");

         when others =>
            Write_Str ("Field11");
      end case;
   end Write_Field11_Name;

   ------------------------
   -- Write_Field12_Name --
   ------------------------

   procedure Write_Field12_Name  (Id : Entity_Id) is
   begin
      case Ekind (Id) is
         when E_Enumeration_Literal =>
            Write_Str ("Enumeration_Rep");

         when Type_Kind   |
              E_Component |
              E_Constant  |
              E_Variable  =>
            Write_Str ("Esize");

         when E_Function | E_Procedure =>
            Write_Str ("Interface_Name");

         when others =>
            Write_Str ("Field12");
      end case;
   end Write_Field12_Name;

   ------------------------
   -- Write_Field13_Name --
   ------------------------

   procedure Write_Field13_Name  (Id : Entity_Id) is
   begin
      case Ekind (Id) is
         when Access_Kind =>
            Write_Str ("Associated_Storage_Pool");

         when Fixed_Kind =>
            Write_Str ("Corresponding_Integer_Type");

         when Record_Kind | Private_Kind =>
            Write_Str ("Primitive_Operations");

         when E_Block         |
              Concurrent_Kind |
              E_Function      |
              E_Procedure     |
              E_Entry         |
              E_Entry_Family  =>
            Write_Str ("Finalization_Chain_Entity");

         when others =>
            Write_Str ("FIeld13");
      end case;
   end Write_Field13_Name;

   ------------------------
   -- Write_Field14_Name --
   ------------------------

   procedure Write_Field14_Name  (Id : Entity_Id) is
   begin
      case Ekind (Id) is
         when Access_Kind =>
            Write_Str ("Storage_Size_Variable");

         when Array_Kind      |
              Private_Kind    |
              Protected_Kind  |
              Record_Kind     =>
            Write_Str ("Controlled_Component_Iterator");

         when E_Block         |
              Task_Kind       |
              E_Entry         |
              E_Entry_Family  |
              E_Function      |
              E_Package       |
              E_Procedure     =>
            Write_Str ("Task_Activation_Chain_Entity");

         when others =>
            Write_Str ("Field14");
      end case;
   end Write_Field14_Name;

   ------------------------
   -- Write_Field15_Name --
   ------------------------

   procedure Write_Field15_Name  (Id : Entity_Id) is
   begin
      case Ekind (Id) is
         when Record_Kind =>
            Write_Str ("Access_Disp_Table");

         when Concurrent_Kind =>
            Write_Str ("Number_Simple_Entries");

         when Private_Kind =>
            Write_Str ("Private_Subtype_List");

         when others =>
            Write_Str ("FIeld15");
      end case;
   end Write_Field15_Name;

   ------------------------
   -- Write_Field16_Name --
   ------------------------

   procedure Write_Field16_Name  (Id : Entity_Id) is
   begin
      case Ekind (Id) is
         when Type_Kind =>
            Write_Str ("Next_Itype");

         when others =>
            Write_Str ("Field16");
      end case;
   end Write_Field16_Name;

   ------------------------
   -- Write_Field17_Name --
   ------------------------

   procedure Write_Field17_Name  (Id : Entity_Id) is
   begin
      case Ekind (Id) is
         when Type_Kind =>
            Write_Str ("Class_Wide_Type");

         when others =>
            Write_Str ("Field17");
      end case;
   end Write_Field17_Name;

end Einfo;
