------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ A T T R                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.112 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Errout;   use Errout;
with Excep;    use Excep;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Opt;      use Opt;
with Output;   use Output;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Ch4;  use Sem_Ch4;
with Sem_Ch6;  use Sem_Ch6;
with Sem_Ch8;  use Sem_Ch8;
with Sem_Eval; use Sem_Eval;
with Sem_Res;  use Sem_Res;
with Sem_Type; use Sem_Type;
with Sem_Util; use Sem_Util;
with Stand;    use Stand;
with Sinfo;    use Sinfo;
with Sinput;   use Sinput;
with Snames;   use Snames;
with Stand;
with Stringt;  use Stringt;
with Table;
with Ttypes;   use Ttypes;
with Ttypef;   use Ttypef;
with Tbuild;   use Tbuild;
with Uintp;    use Uintp;

package body Sem_Attr is

   -----------------------
   -- Analyze_Attribute --
   -----------------------

   procedure Analyze_Attribute (N : Node_Id) is
      Loc         : constant Source_Ptr := Sloc (N);
      Aname       : constant Name_Id    := Attribute_Name (N);
      P           : constant Node_Id    := Prefix (N);
      Exprs       : constant List_Id    := Expressions (N);
      E1          : Node_Id;
      E2          : Node_Id;

      P_Type      : Entity_Id;
      --  Type of prefix after analysis

      P_Root_Type : Entity_Id;
      --  Root type of prefix after analysis

      -----------------------
      -- Local Subprograms --
      -----------------------

      procedure Access_Attribute;
      --  Used for Access, Unchecked_Access, Unrestricted_Access attributes.
      --  Internally, Id distinguishes which of the three cases is involved.

      procedure Check_Array_Or_Scalar_Type;
      --  Common procedure used by First, Last, Range attribute to check
      --  that the prefix is an array or scalar type, or a name of an
      --  array object, and that an argument appears only if appropriate
      --  (i.e. only in the array case).

      procedure Check_Array_Type;
      --  Common semantic checks for all array attributes

      procedure Check_Discrete_Attribute;
      --  Common processing for attributes operating on discrete types

      procedure Check_Discrete_Type;
      --  Verify that prefix of attribute N is a discrete type

      procedure Check_E0;
      --  Check that no attribute arguments are present

      procedure Check_E0_Or_E1;
      --  Check that at most one attribute argument is present

      procedure Check_E1;
      --  Check that exactly one attribute argument is present

      procedure Check_E2;
      --  Check that two attribute arguments are present

      procedure Check_Fixed_Type;
      --  Verify that prefix of attribute N is a fixed type

      procedure Check_Fixed_Type_0;
      --  Verify that prefix of attribute N is a fixed type and that
      --  no attribute expressions are present

      procedure Check_Float_Type;
      --  Verify that prefix of attribute N is a float type

      procedure Check_Float_Type_0;
      --  Verify that prefix of attribute N is a float type and that
      --  no attribute expressions are present

      procedure Check_Float_Type_1;
      --  Verify that prefix of attribute N is a float type and that
      --  exactly one attribute expression is present

      procedure Check_Float_Type_2;
      --  Verify that prefix of attribute N is a float type and that
      --  two attribute expressions are present

      procedure Check_Real_Type;
      --  Verify that prefix of attribute N is fixed or float type

      procedure Check_Scalar_Type;
      --  Verify that prefix of attribute N is a scalar type

      procedure Check_Standard_Prefix;
      --  Verify that prefix of attribute N is package Standard

      procedure Check_Task_Prefix;
      --  Verify that prefix of attribute N is a task or task type

      procedure Check_Type;
      --  Verify that the prefix of attribute N is a type

      procedure Error_Attr (Msg : String; Error_Node : Node_Id);
      --  Posts error using Error_Msg_N at given node, sets type of attribute
      --  node to Any_Type, and then raises Error_Resync to avoid any further
      --  semantic processing.

      procedure Float_Attribute_Boolean
        (Short_Float_Val     : Boolean;
         Float_Val           : Boolean;
         Long_Float_Val      : Boolean;
         Long_Long_Float_Val : Boolean);
      --  This procedure processes a float attribute with no arguments that
      --  returns a Boolean result. The four parameters are the Boolean result
      --  values for the four possible floating-point root types.

      procedure Float_Attribute_Universal_Integer
        (Short_Float_Val     : Int;
         Float_Val           : Int;
         Long_Float_Val      : Int;
         Long_Long_Float_Val : Int);
      --  This procedure processes a float attribute with no arguments that
      --  returns a universal integer result. All such results are easily
      --  within Int range, and the four parameters are the result values
      --  for the four possible floating-point root types.

      procedure Float_Attribute_Universal_Real
        (Short_Float_Val     : String;
         Float_Val           : String;
         Long_Float_Val      : String;
         Long_Long_Float_Val : String);
      --  This procedure processes a float attribute with no arguments that
      --  returns a universal real result. The four parameters are strings
      --  that contain representations of the values required in normal
      --  real literal format with a possible leading minus sign.

      procedure Standard_Attribute (Val : Int);
      --  Used to process attributes whose prefix is package Standard which
      --  yield values of type Universal_Integer. The attribute reference
      --  node is rewritten with an integer literal of the given value.

      procedure Unexpected_Argument (En : Node_Id);
      --  Signal unexpected attribute argument (En is the argument)

      procedure Unimplemented_Attribute;
      --  Give error message for unimplemented attribute

      ----------------------
      -- Access_Attribute --
      ----------------------

      procedure Access_Attribute is
         Index    : Interp_Index;
         It       : Interp;
         Acc_Type : Entity_Id;

         function Valid_Aliased_View (Obj : Node_Id) return Boolean is
            E : Entity_Id;

         begin
            if Is_Entity_Name (Obj) then
               E := Entity (Obj);

               return Is_Aliased (E)
                 or else (Present (Renamed_Object (E))
                           and then Valid_Aliased_View (Renamed_Object (E)))

                 or else ((Ekind (E) = E_In_Out_Parameter
                             or else Ekind (E) = E_Generic_In_Out_Parameter)
                           and then Is_Tagged_Type (Etype (E)))

                 or else ((Ekind (E) = E_Task_Type
                             or else Ekind (E) = E_Protected_Type)
                           and then In_Open_Scopes (E))

                  --  Access discriminant constraint

                 or else (Is_Type (E) and then E = Current_Scope)
                 or else (Ekind (E) in Incomplete_Kind
                           and then Full_Declaration (E) = Current_Scope);


            elsif Nkind (Obj) = N_Selected_Component then
               return Is_Aliased (Entity (Selector_Name (Obj)));

            elsif Nkind (Obj) = N_Indexed_Component then
               return (Is_Aliased (Etype (Prefix (Obj)))
                 or else Is_Access_Type (Etype (Prefix (Obj))));

            elsif Nkind (Obj) = N_Unchecked_Type_Conversion
              or else Nkind (Obj) = N_Type_Conversion
            then
               return Is_Tagged_Type (Etype (Obj));

            elsif Nkind (Obj) = N_Explicit_Dereference then
               return True;  --  more precise test needed???

            else
               return False;
            end if;
         end Valid_Aliased_View;

      --  Start of processing for Access_Attribute

      begin
         Check_E0;

         --  In the case of an access to subprogram, use the name of the
         --  subprogram itself as the designated type. Type-checking in
         --  this case compares the signatures of the designated types.

         if Is_Entity_Name (P) and then Is_Overloadable (Entity (P)) then
            if not Is_Overloaded (P) then
               Acc_Type :=
                 New_Internal_Entity
                   (E_Access_Subprogram_Type, Current_Scope, Loc, 'A');
               Set_Etype (Acc_Type,  Acc_Type);
               Set_Directly_Designated_Type (Acc_Type, Entity (P));
               Set_Etype (N, Acc_Type);

            else
               Get_First_Interp (P, Index, It);
               Set_Etype (N, Any_Type);

               while Present (It.Nam) loop
                  Acc_Type :=
                    New_Internal_Entity
                      (E_Access_Subprogram_Type, Current_Scope, Loc, 'A');
                  Set_Etype (Acc_Type,  Acc_Type);
                  Set_Directly_Designated_Type (Acc_Type, It.Nam);
                  Add_One_Interp (N,  Acc_Type,  Acc_Type);
                  Get_Next_Interp (Index, It);
               end loop;
            end if;

         --  Case of access to object

         else
            Acc_Type :=
              New_Internal_Entity (E_Allocator_Type, Current_Scope, Loc, 'A');
            Set_Etype (Acc_Type,  Acc_Type);
            Set_Directly_Designated_Type (Acc_Type, P_Type);
            Set_Etype (N, Acc_Type);

            --  Check for aliased view unless unrestricted case

            if Aname /= Name_Unrestricted_Access
              and then not Valid_Aliased_View (P)
            then
               Error_Attr ("prefix of % attribute must be aliased view", P);
            end if;
         end if;

      end Access_Attribute;

      --------------------------------
      -- Check_Array_Or_Scalar_Type --
      --------------------------------

      procedure Check_Array_Or_Scalar_Type is
         Index_Type : Entity_Id;

         D : Int;
         --  Dimension number for array attributes.

      begin
         if Is_Scalar_Type (P_Type) then
            Check_Type;

            if Present (E1) then
               Error_Attr ("invalid argument in % attribute", E1);
            else
               Set_Etype (N, Base_Type (P_Type));
               return;
            end if;

         else
            Check_Array_Type;

            --  We know prefix is an array type, or the name of an array
            --  object, and that the expression, if present, is static
            --  and within the range of the dimensions of the type.

            if Is_Array_Type (P_Type) then
               Index_Type := First_Index (P_Type);

            elsif Is_Access_Type (P_Type) then
               Index_Type := First_Index (Designated_Type (P_Type));
            end if;

            if No (E1) then

               --  First dimension assumed

               Set_Etype (N, Etype (Index_Type));

            else
               D := UI_To_Int (Intval (E1));

               for I in 1 .. D - 1 loop
                  Index_Type := Next_Index (Index_Type);
               end loop;

               Set_Etype (N, Etype (Index_Type));
               Set_Etype (E1, Standard_Integer);
            end if;
         end if;
      end Check_Array_Or_Scalar_Type;

      ----------------------
      -- Check_Array_Type --
      ----------------------

      procedure Check_Array_Type is
         D : Int;
         --  Dimension number for array attributes.

      begin
         Check_E0_Or_E1;

         if Is_Array_Type (P_Type) then
            if not Is_Constrained (P_Type)
              and then Is_Entity_Name (P)
              and then Is_Type (Entity (P))
            then
               Error_Attr
                 ("prefix for % attribute must be constrained array", P);
            end if;

            D := Number_Dimensions (P_Type);

         elsif Is_Access_Type (P_Type)
           and then Is_Array_Type (Designated_Type (P_Type))
         then
            if Is_Entity_Name (P) and then Is_Type (Entity (P)) then
               Error_Attr ("prefix of % attribute cannot be access type", P);
            end if;

            D := Number_Dimensions (Designated_Type (P_Type));

         else
            Error_Attr ("prefix for % attribute must be array", P);
         end if;

         if Present (E1) then
            Resolve (E1, Any_Integer);
            Set_Etype (E1, Standard_Integer);

            if not Is_Static (E1) then
               Error_Attr ("expression for dimension must be static", E1);

            elsif  UI_To_Int (Intval (E1)) > D
              or else UI_To_Int (Intval (E1)) < 1
            then
               Error_Attr ("invalid dimension number for array type", E1);
            end if;
         end if;
      end Check_Array_Type;

      ------------------------------
      -- Check_Discrete_Attribute --
      ------------------------------

      procedure Check_Discrete_Attribute is
      begin
         Check_Discrete_Type;
         Check_E1;
         Resolve (E1, P_Type);
      end Check_Discrete_Attribute;

      -------------------------
      -- Check_Discrete_Type --
      -------------------------

      procedure Check_Discrete_Type is
      begin
         Check_Type;

         if not Is_Discrete_Type (P_Type) then
            Error_Attr ("prefix of % attribute must be discrete type", P);
         end if;
      end Check_Discrete_Type;

      --------------
      -- Check_E0 --
      --------------

      procedure Check_E0 is
      begin
         if Present (E1) then
            Unexpected_Argument (E1);
         end if;
      end Check_E0;

      --------------------
      -- Check_E0_Or_E1 --
      --------------------

      procedure Check_E0_Or_E1 is
      begin
         if Present (E2) then
            Unexpected_Argument (E2);
         end if;
      end Check_E0_Or_E1;

      --------------
      -- Check_E1 --
      --------------

      procedure Check_E1 is
      begin
         Check_E0_Or_E1;

         if No (E1) then
            Error_Attr ("missing argument for % attribute", N);
         end if;
      end Check_E1;

      --------------
      -- Check_E2 --
      --------------

      procedure Check_E2 is
      begin
         if No (E1) then
            Error_Attr ("missing arguments for % attribute (2 required)", N);
         elsif No (E2) then
            Error_Attr ("missing argument for % attribute (2 required)", N);
         end if;
      end Check_E2;

      ----------------------
      -- Check_Fixed_Type --
      ----------------------

      procedure Check_Fixed_Type is
      begin
         Check_Type;

         if not Is_Fixed_Type (P_Type) then
            Error_Attr ("prefix of % attribute must be fixed type", P);
         end if;
      end Check_Fixed_Type;

      ------------------------
      -- Check_Fixed_Type_0 --
      ------------------------

      procedure Check_Fixed_Type_0 is
      begin
         Check_Fixed_Type;
         Check_E0;
      end Check_Fixed_Type_0;

      ----------------------
      -- Check_Float_Type --
      ----------------------

      procedure Check_Float_Type is
      begin
         Check_Type;

         if not Is_Float_Type (P_Type) then
            Error_Attr ("prefix of % attribute must be float type", P);
         end if;
      end Check_Float_Type;

      ------------------------
      -- Check_Float_Type_0 --
      ------------------------

      procedure Check_Float_Type_0 is
      begin
         Check_Float_Type;
         Check_E0;
      end Check_Float_Type_0;

      ------------------------
      -- Check_Float_Type_1 --
      ------------------------

      procedure Check_Float_Type_1 is
      begin
         Check_Float_Type;
         Check_E1;
      end Check_Float_Type_1;

      ------------------------
      -- Check_Float_Type_2 --
      ------------------------

      procedure Check_Float_Type_2 is
      begin
         Check_Float_Type;
         Check_E2;
      end Check_Float_Type_2;

      ---------------------
      -- Check_Real_Type --
      ---------------------

      procedure Check_Real_Type is
      begin
         Check_Type;

         if not Is_Real_Type (P_Type) then
            Error_Attr ("prefix of % attribute must be real type", P);
         end if;
      end Check_Real_Type;

      -----------------------
      -- Check_Scalar_Type --
      -----------------------

      procedure Check_Scalar_Type is
      begin
         Check_Type;

         if not Is_Scalar_Type (P_Type) then
            Error_Attr ("prefix of % attribute must be scalar type", P);
         end if;
      end Check_Scalar_Type;

      ---------------------------
      -- Check_Standard_Prefix --
      ---------------------------

      procedure Check_Standard_Prefix is
      begin
         Check_E0;

         if Nkind (P) /= N_Identifier or else Chars (P) /= Name_Standard then
            Error_Attr ("only allowed prefix for % attribute is Standard", P);
         end if;
      end Check_Standard_Prefix;

      -----------------------
      -- Check_Task_Prefix --
      -----------------------

      procedure Check_Task_Prefix is
      begin
         Analyze (P);

         if Is_Task_Type (Etype (P))
           or else (Is_Access_Type (Etype (P))
              and then Is_Task_Type (Designated_Type (Etype (P))))
         then
            Resolve (P, Etype (P));
         else
            Error_Attr ("prefix of % attribute must be a task", P);
         end if;
      end Check_Task_Prefix;

      ----------------
      -- Check_Type --
      ----------------

      --  The possibilities are an entity name denoting a type, or an
      --  attribute reference that denotes a type (Base or Class)

      procedure Check_Type is
      begin
         if (Nkind (P) in N_Entity_Name
               or else (Nkind (P) = N_Attribute_Reference
                          and then Present (Entity (P))))
           and then Is_Type (Entity (P))
         then
            null;
         else
            Error_Attr ("prefix of % attribute must be a type", P);
         end if;
      end Check_Type;

      ----------------
      -- Error_Attr --
      ----------------

      procedure Error_Attr (Msg : String; Error_Node : Node_Id) is
      begin
         Error_Msg_N (Msg, Error_Node);
         Set_Etype (N, Any_Type);
         Set_Entity (N, Any_Type);
         raise Error_Resync;
      end Error_Attr;

      -----------------------------
      -- Float_Attribute_Boolean --
      -----------------------------

      procedure Float_Attribute_Boolean
        (Short_Float_Val     : Boolean;
         Float_Val           : Boolean;
         Long_Float_Val      : Boolean;
         Long_Long_Float_Val : Boolean)
      is
         Val    : Boolean;
         Result : Node_Id;

      begin
         Check_Float_Type_0;

         if Is_Generic_Type (P_Type) then
            Set_Etype (N, Standard_Boolean);

         else
            if P_Root_Type = Standard_Short_Float then
               Val := Short_Float_Val;
            elsif P_Root_Type = Standard_Float then
               Val := Float_Val;
            elsif P_Root_Type = Standard_Long_Float then
               Val := Long_Float_Val;
            else
               pragma Assert (P_Root_Type = Standard_Long_Long_Float);
               Val := Long_Long_Float_Val;
            end if;

            if Val then
               Result := New_Reference_To (Standard_True, Loc);
            else
               Result := New_Reference_To (Standard_False, Loc);
            end if;

            Rewrite_Substitute_Tree (N, Result);
            Analyze (N);
         end if;
      end Float_Attribute_Boolean;

      ---------------------------------------
      -- Float_Attribute_Universal_Integer --
      ---------------------------------------

      procedure Float_Attribute_Universal_Integer
        (Short_Float_Val     : Int;
         Float_Val           : Int;
         Long_Float_Val      : Int;
         Long_Long_Float_Val : Int)
      is
         Val    : Int;
         Result : constant Node_Id := New_Node (N_Integer_Literal, Loc);

      begin
         Check_Float_Type_0;

         if Is_Generic_Type (P_Type) then
            Set_Etype (N, Universal_Integer);

         else
            if P_Root_Type = Standard_Short_Float then
               Val := Short_Float_Val;
            elsif P_Root_Type = Standard_Float then
               Val := Float_Val;
            elsif P_Root_Type = Standard_Long_Float then
               Val := Long_Float_Val;
            else
               pragma Assert (P_Root_Type = Standard_Long_Long_Float);
               Val := Long_Long_Float_Val;
            end if;

            Set_Intval (Result, UI_From_Int (Val));
            Rewrite_Substitute_Tree (N, Result);
            Analyze (N);
         end if;
      end Float_Attribute_Universal_Integer;

      ------------------------------------
      -- Float_Attribute_Universal_Real --
      ------------------------------------

      procedure Float_Attribute_Universal_Real
        (Short_Float_Val     : String;
         Float_Val           : String;
         Long_Float_Val      : String;
         Long_Long_Float_Val : String)
      is
         Result : Node_Id;

      begin
         Check_Float_Type_0;

         if Is_Generic_Type (P_Type) then
            Set_Etype (N, Universal_Real);

         else
            if P_Root_Type = Standard_Short_Float then
               Result := Real_Convert (Short_Float_Val);
            elsif P_Root_Type = Standard_Float then
               Result := Real_Convert (Float_Val);
            elsif P_Root_Type = Standard_Long_Float then
               Result := Real_Convert (Long_Float_Val);
            else
               pragma Assert (P_Root_Type = Standard_Long_Long_Float);
               Result := Real_Convert (Long_Long_Float_Val);
            end if;

            Rewrite_Substitute_Tree (N, Result);
            Analyze (N);
         end if;
      end Float_Attribute_Universal_Real;

      ------------------------
      -- Standard_Attribute --
      ------------------------

      procedure Standard_Attribute (Val : Int) is
      begin
         Check_Standard_Prefix;
         Rewrite_Substitute_Tree (N,
           Make_Integer_Literal (Loc, UI_From_Int (Val)));
         Analyze (N);
      end Standard_Attribute;

      -------------------------
      -- Unexpected Argument --
      -------------------------

      procedure Unexpected_Argument (En : Node_Id) is
      begin
         Error_Attr ("unexpected argument for % attribute", En);
      end Unexpected_Argument;

      -----------------------------
      -- Unimplemented_Attribute --
      -----------------------------

      procedure Unimplemented_Attribute is
      begin
         Unimplemented (N, "attribute");
      end Unimplemented_Attribute;

   ------------------------------------------------
   --  Start of Processing for Analyze_Attribute --
   ------------------------------------------------

   begin
      Error_Msg_Name_1 := Aname;

      --  Immediate return if unrecognized attribute (already diagnosed
      --  by parser, so there is nothing more that we need to do)

      if not Is_Attribute_Name (Aname) then
         raise Error_Resync;
      end if;

      --  Analyze prefix and exit if error in analysis

      Analyze (P);
      P_Type := Etype (P);

      if P_Type = Any_Type then
         raise Error_Resync;
      else
         P_Root_Type := Root_Type (P_Type);
      end if;

      --  Analyze expressions that may be present, exiting if an error occurs

      if No (Exprs) then
         E1 := Empty;
         E2 := Empty;

      else
         E1 := First (Exprs);
         Analyze (E1);

         if Etype (E1) = Any_Type then
            raise Error_Resync;
         end if;

         E2 := Next (E1);

         if Present (E2) then
            Analyze (E2);

            if Etype (E2) = Any_Type then
               raise Error_Resync;
            end if;

            if Present (Next (E2)) then
               Unexpected_Argument (Next (E2));
            end if;
         end if;
      end if;

      if Is_Overloaded (P)
        and then Aname /= Name_Access
        and then Aname /= Name_Address
        and then Aname /= Name_Unchecked_Access
      then
         Error_Attr ("ambiguous prefix for % attribute", P);
      end if;

      Error_Msg_Name_1 := Aname;

      case Get_Attribute_Id (Aname) is

      ------------------
      -- Abort_Signal --
      ------------------

      when Attribute_Abort_Signal =>
         Check_Standard_Prefix;
         Rewrite_Substitute_Tree (N,
           New_Reference_To (Stand.Abort_Signal, Loc));
         Analyze (N);

      ------------
      -- Access --
      ------------

      when Attribute_Access =>
         Access_Attribute;

      -------------
      -- Address --
      -------------

      when Attribute_Address =>
         Check_E0;
         Set_Etype (N, RTE (RE_Address));

      ------------------
      -- Address_Size --
      ------------------

      when Attribute_Address_Size =>
         Standard_Attribute (Ttypes.System_Address_Size);

      --------------
      -- Adjacent --
      --------------

      when Attribute_Adjacent =>
         Check_Float_Type_2;
         Unimplemented_Attribute;

      ---------
      -- Aft --
      ---------

      when Attribute_Aft =>
         Check_Fixed_Type_0;
         Set_Etype (N, Universal_Integer);
         Unimplemented (N, "attribute");

      ---------------
      -- Alignment --
      ---------------

      when Attribute_Alignment =>
         Check_E0;
         Unimplemented_Attribute;

      ----------
      -- Base --
      ----------

      when Attribute_Base => Base :
      begin
         Check_E0_Or_E1;
         Find_Type (P);
         Set_Etype (N, Base_Type (Entity (P)));

         if Present (Exprs) then

            --  Attribute is the subtype mark of a conversion.

            declare
               New_N : Node_Id;

            begin
               New_N :=
                 Make_Type_Conversion (Loc,
                   Subtype_Mark => New_Occurrence_Of (Etype (N), Loc),
                   Expression => New_Copy (E1));
               Rewrite_Substitute_Tree (N,  New_N);
               Analyze (N);
            end;

         else
            Set_Entity (N, Base_Type (Entity (P)));
         end if;
      end Base;

      ---------------
      -- Bit_Order --
      ---------------

      when Attribute_Bit_Order =>
         Check_E0;
         Unimplemented_Attribute;

      ------------------
      -- Body_Version --
      ------------------

      when Attribute_Body_Version =>
         Check_E0;
         Unimplemented_Attribute;

      --------------
      -- Callable --
      --------------

      when Attribute_Callable =>
         Check_E0;
         Set_Etype (N, Standard_Boolean);
         Check_Task_Prefix;

      ------------
      -- Caller --
      ------------

      when Attribute_Caller =>
         Check_E0;
         Unimplemented_Attribute;

      -------------
      -- Ceiling --
      -------------

      when Attribute_Ceiling =>
         Check_Float_Type_1;
         Unimplemented_Attribute;

      -----------
      -- Class --
      -----------

      when Attribute_Class => Class :
      begin
         Check_E0_Or_E1;
         Find_Type (N);

         if Present (E1) then

            --  This is a conversion not an attribute : T'Class (X)

            Rewrite_Substitute_Tree (N, Make_Type_Conversion (Loc,
              Subtype_Mark => New_Occurrence_Of (Etype (N), Loc),
              Expression => New_Copy (E1)));

            Analyze (N);
         end if;

      end Class;

      --------------------
      -- Component_Size --
      --------------------

      when Attribute_Component_Size =>
         Check_E0;
         Unimplemented_Attribute;

      -------------
      -- Compose --
      -------------

      when Attribute_Compose =>
         Check_Float_Type_2;
         Unimplemented_Attribute;

      -----------------
      -- Constrained --
      -----------------

      when Attribute_Constrained => Constrained :
      begin
         Check_E0;
         Set_Etype (N, Standard_Boolean);
         Unimplemented_Attribute;
      end Constrained;

      ---------------
      -- Copy_Sign --
      ---------------

      when Attribute_Copy_Sign =>
         Check_Float_Type_2;
         Unimplemented_Attribute;

      -----------
      -- Count --
      -----------

      when Attribute_Count => Count :
      declare
         Ent : Entity_Id;
         S   : Entity_Id;

      begin
         Check_E0;

         if Nkind (P) = N_Identifier
           or else Nkind (P) = N_Expanded_Name
         then
            Ent := Entity (P);

            if Ekind (Ent) /= E_Entry then
               Error_Attr ("invalid entry name",  N);
            end if;

         elsif Nkind (P) = N_Indexed_Component then
            Ent := Entity (Prefix (P));

            if Ekind (Ent) /= E_Entry_Family then
               Error_Attr ("invalid entry family name",  P);
               return;
            end if;

         else
            Error_Attr ("invalid entry name",  N);
            return;
         end if;

         for J in reverse 0 .. Scope_Stack.Last loop
            S := Scope_Stack.Table (J).Entity;

            if S = Scope (Ent) then
               exit;

            elsif Ekind (Scope (Ent)) in Task_Kind
              and then Ekind (S) /= E_Loop
              and then Ekind (S) /= E_Block
              and then Ekind (S) /= E_Entry
              and then Ekind (S) /= E_Entry_Family
            then
               Error_Attr ("Count cannot appear in inner unit", N);
            end if;
         end loop;

         Set_Etype (N, Universal_Integer);
      end Count;

      --------------
      -- Definite --
      --------------

      when Attribute_Definite =>
         Check_E0;
         Unimplemented_Attribute;

      -----------
      -- Delta --
      -----------

      when Attribute_Delta =>
         Check_Fixed_Type_0;
         Set_Etype (N, Universal_Real);
         Unimplemented_Attribute;

      ------------
      -- Denorm --
      ------------

      when Attribute_Denorm =>
         Float_Attribute_Boolean (
           Short_Float_Attr_Denorm,
           Float_Attr_Denorm,
           Long_Float_Attr_Denorm,
           Long_Long_Float_Attr_Denorm);

      ------------
      -- Digits --
      ------------

      when Attribute_Digits =>
         Check_Float_Type_0;  --  ??? what about decimal
         Set_Etype (N, Universal_Integer);

      ----------
      -- Emax --
      ----------

      when Attribute_Emax =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Emax,
           Float_Attr_Emax,
           Long_Float_Attr_Emax,
           Long_Long_Float_Attr_Emax);

      -------------
      -- Epsilon --
      -------------

      when Attribute_Epsilon =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Epsilon'Universal_Literal_String,
           Float_Attr_Epsilon'Universal_Literal_String,
           Long_Float_Attr_Epsilon'Universal_Literal_String,
           Long_Long_Float_Attr_Epsilon'Universal_Literal_String);

      --------------
      -- Exponent --
      --------------

      when Attribute_Exponent =>
         Check_Float_Type_1;
         Unimplemented_Attribute;

      ------------------
      -- External_Tag --
      ------------------

      when Attribute_External_Tag =>
         Check_E0;
         Unimplemented_Attribute;

      -----------
      -- First --
      -----------

      when Attribute_First =>
         Check_Array_Or_Scalar_Type;

      ---------------
      -- First_Bit --
      ---------------

      when Attribute_First_Bit =>
         Check_E0;
         Set_Etype (N, Universal_Integer);
         Unimplemented_Attribute;

      -----------
      -- Floor --
      -----------

      when Attribute_Floor =>
         Check_Float_Type_1;
         Unimplemented_Attribute;

      ----------
      -- Fore --
      ----------

      when Attribute_Fore =>
         Check_Fixed_Type_0;
         Set_Etype (N, Universal_Integer);
         Unimplemented_Attribute;

      --------------
      -- Fraction --
      --------------

      when Attribute_Fraction =>
         Check_Float_Type_1;
         Unimplemented_Attribute;

      --------------
      -- Identity --
      --------------

      when Attribute_Identity =>
         Check_E0;
         Unimplemented_Attribute;

      -----------
      -- Image --
      -----------

      when Attribute_Image => Image :
      begin
         Set_Etype (N, Standard_String);

         if Is_Real_Type (P_Type) then
            Check_Type;

            if Ada_83 then
               Error_Msg_N
                 ("% attribute not allowed for real types in Ada 83",
                  N);
            end if;
         else
            Check_Discrete_Attribute;
         end if;
      end Image;

      -----------
      -- Input --
      -----------

      when Attribute_Input =>
         Check_E2;
         Unimplemented_Attribute;

      -----------
      -- Large --
      -----------

      when Attribute_Large =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Large'Universal_Literal_String,
           Float_Attr_Large'Universal_Literal_String,
           Long_Float_Attr_Large'Universal_Literal_String,
           Long_Long_Float_Attr_Large'Universal_Literal_String);

      ----------
      -- Last --
      ----------

      when Attribute_Last =>
         Check_Array_Or_Scalar_Type;

      --------------
      -- Last_Bit --
      --------------

      when Attribute_Last_Bit =>
         Check_E0;
         Set_Etype (N, Universal_Integer);
         Unimplemented_Attribute;

      ------------------
      -- Leading_Part --
      ------------------

      when Attribute_Leading_Part =>
         Check_Float_Type_2;
         Unimplemented_Attribute;

      ------------
      -- Length --
      ------------

      when Attribute_Length =>
         Check_Array_Type;
         Set_Etype (N, Universal_Integer);

      -------------
      -- Machine --
      -------------

      when Attribute_Machine =>
         Check_Float_Type_1;
         Unimplemented_Attribute;

      ------------------
      -- Machine_Emax --
      ------------------

      when Attribute_Machine_Emax =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Machine_Emax,
           Float_Attr_Machine_Emax,
           Long_Float_Attr_Machine_Emax,
           Long_Long_Float_Attr_Machine_Emax);

      ------------------
      -- Machine_Emin --
      ------------------

      when Attribute_Machine_Emin =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Machine_Emin,
           Float_Attr_Machine_Emin,
           Long_Float_Attr_Machine_Emin,
           Long_Long_Float_Attr_Machine_Emin);

      ----------------------
      -- Machine_Mantissa --
      ----------------------

      when Attribute_Machine_Mantissa =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Machine_Mantissa,
           Float_Attr_Machine_Mantissa,
           Long_Float_Attr_Machine_Mantissa,
           Long_Long_Float_Attr_Machine_Mantissa);

      -----------------------
      -- Machine_Overflows --
      -----------------------

      when Attribute_Machine_Overflows =>
         Float_Attribute_Boolean (
           Short_Float_Attr_Machine_Overflows,
           Float_Attr_Machine_Overflows,
           Long_Float_Attr_Machine_Overflows,
           Long_Long_Float_Attr_Machine_Overflows);

      -------------------
      -- Machine_Radix --
      -------------------

      when Attribute_Machine_Radix =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Machine_Radix,
           Float_Attr_Machine_Radix,
           Long_Float_Attr_Machine_Radix,
           Long_Long_Float_Attr_Machine_Radix);

      --------------------
      -- Machine_Rounds --
      --------------------

      when Attribute_Machine_Rounds =>
         Float_Attribute_Boolean (
           Short_Float_Attr_Machine_Rounds,
           Float_Attr_Machine_Rounds,
           Long_Float_Attr_Machine_Rounds,
           Long_Long_Float_Attr_Machine_Rounds);

      --------------
      -- Mantissa --
      --------------

      when Attribute_Mantissa =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Mantissa,
           Float_Attr_Mantissa,
           Long_Float_Attr_Mantissa,
           Long_Long_Float_Attr_Mantissa);

      ---------
      -- Max --
      ---------

      when Attribute_Max =>
         Check_E2;
         Check_Scalar_Type;
         Resolve (E1, Base_Type (P_Type));
         Resolve (E2, Base_Type (P_Type));
         Set_Etype (N, Base_Type (P_Type));

      ----------------------------
      -- Max_Interrupt_Priority --
      ----------------------------

      when Attribute_Max_Interrupt_Priority =>
         Standard_Attribute (Ttypes.System_Max_Interrupt_Priority);

      ------------------
      -- Max_Priority --
      ------------------

      when Attribute_Max_Priority =>
         Standard_Attribute (Ttypes.System_Max_Priority);

      ----------------------------------
      -- Max_Size_In_Storage_Elements --
      ----------------------------------

      when Attribute_Max_Size_In_Storage_Elements =>
         Check_E0;
         Unimplemented_Attribute;

      ---------
      -- Min --
      ---------

      when Attribute_Min =>
         Check_E2;
         Check_Scalar_Type;
         Resolve (E1, Base_Type (P_Type));
         Resolve (E2, Base_Type (P_Type));
         Set_Etype (N, Base_Type (P_Type));

      -----------
      -- Model --
      -----------

      when Attribute_Model =>
         Check_Float_Type_1;
         Unimplemented_Attribute;

      ----------------
      -- Model_Emin --
      ----------------

      when Attribute_Model_Emin =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Model_Emin,
           Float_Attr_Model_Emin,
           Long_Float_Attr_Model_Emin,
           Long_Long_Float_Attr_Model_Emin);

      -------------------
      -- Model_Epsilon --
      -------------------

      when Attribute_Model_Epsilon =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Model_Epsilon'Universal_Literal_String,
           Float_Attr_Model_Epsilon'Universal_Literal_String,
           Long_Float_Attr_Model_Epsilon'Universal_Literal_String,
           Long_Long_Float_Attr_Model_Epsilon'Universal_Literal_String);

      --------------------
      -- Model_Mantissa --
      --------------------

      when Attribute_Model_Mantissa =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Model_Mantissa,
           Float_Attr_Model_Mantissa,
           Long_Float_Attr_Model_Mantissa,
           Long_Long_Float_Attr_Model_Mantissa);

      -----------------
      -- Model_Small --
      -----------------

      when Attribute_Model_Small =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Model_Small'Universal_Literal_String,
           Float_Attr_Model_Small'Universal_Literal_String,
           Long_Float_Attr_Model_Small'Universal_Literal_String,
           Long_Long_Float_Attr_Model_Small'Universal_Literal_String);

      ------------------
      -- Storage_Unit --
      ------------------

      when Attribute_Storage_Unit =>
         Standard_Attribute (Ttypes.System_Storage_Unit);

      ------------
      -- Output --
      ------------

      when Attribute_Output =>
         Check_E2;
         Unimplemented_Attribute;

      ---------
      -- Pos --
      ---------

      when Attribute_Pos =>
         Check_Discrete_Attribute;
         Set_Etype (N, Universal_Integer);

      --------------
      -- Position --
      --------------

      when Attribute_Position =>
         Check_E0;
         Set_Etype (N, Universal_Integer);

      ----------
      -- Pred --
      ----------

      when Attribute_Pred =>
         Check_Discrete_Attribute;
         Set_Etype (N, Base_Type (P_Type));

      ---------------------
      -- Range_Attribute --
      ---------------------

      when Attribute_Range =>
         Check_Array_Or_Scalar_Type;

      ----------
      -- Read --
      ----------

      when Attribute_Read =>
         Check_E2;
         Unimplemented_Attribute;

      ---------------
      -- Remainder --
      ---------------

      when Attribute_Remainder =>
         Check_Float_Type_2;
         Unimplemented_Attribute;

      -----------
      -- Round --
      -----------

      when Attribute_Round =>
         Check_E1;
         Unimplemented_Attribute;

      --------------
      -- Rounding --
      --------------

      when Attribute_Rounding =>
         Check_Float_Type_1;
         Unimplemented_Attribute;

      ---------------
      -- Safe_Emax --
      ---------------

      when Attribute_Safe_Emax =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Safe_Emax,
           Float_Attr_Safe_Emax,
           Long_Float_Attr_Safe_Emax,
           Long_Long_Float_Attr_Safe_Emax);

      ----------------
      -- Safe_First --
      ----------------

      when Attribute_Safe_First =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Safe_First'Universal_Literal_String,
           Float_Attr_Safe_First'Universal_Literal_String,
           Long_Float_Attr_Safe_First'Universal_Literal_String,
           Long_Long_Float_Attr_Safe_First'Universal_Literal_String);

      ----------------
      -- Safe_Large --
      ----------------

      when Attribute_Safe_Large =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Safe_Large'Universal_Literal_String,
           Float_Attr_Safe_Large'Universal_Literal_String,
           Long_Float_Attr_Safe_Large'Universal_Literal_String,
           Long_Long_Float_Attr_Safe_Large'Universal_Literal_String);

      ---------------
      -- Safe_Last --
      ---------------

      when Attribute_Safe_Last =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Safe_Last'Universal_Literal_String,
           Float_Attr_Safe_Last'Universal_Literal_String,
           Long_Float_Attr_Safe_Last'Universal_Literal_String,
           Long_Long_Float_Attr_Safe_Last'Universal_Literal_String);

      ----------------
      -- Safe_Small --
      ----------------

      when Attribute_Safe_Small =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Safe_Small'Universal_Literal_String,
           Float_Attr_Safe_Small'Universal_Literal_String,
           Long_Float_Attr_Safe_Small'Universal_Literal_String,
           Long_Long_Float_Attr_Safe_Small'Universal_Literal_String);

      -----------
      -- Scale --
      -----------

      when Attribute_Scale =>
         Check_E0;
         Unimplemented_Attribute;

      -------------
      -- Scaling --
      -------------

      when Attribute_Scaling =>
         Check_Float_Type_2;
         Unimplemented_Attribute;

      ------------------
      -- Signed_Zeros --
      ------------------

      when Attribute_Signed_Zeros =>
         Float_Attribute_Boolean (
           Short_Float_Attr_Signed_Zeros,
           Float_Attr_Signed_Zeros,
           Long_Float_Attr_Signed_Zeros,
           Long_Long_Float_Attr_Signed_Zeros);

      ----------
      -- Size --
      ----------

      when Attribute_Size =>
         Check_E0;
         Set_Etype (N, Universal_Integer);

      -----------
      -- Small --
      -----------

      when Attribute_Small =>
         Check_Real_Type;
         Set_Etype (N, Universal_Real);
         Unimplemented_Attribute;

      ------------------
      -- Storage_Pool --
      ------------------

      when Attribute_Storage_Pool =>
         Check_E0;
         Unimplemented_Attribute;

      ------------------
      -- Storage_Size --
      ------------------

      when Attribute_Storage_Size =>
         Check_E0;
         Set_Etype (N, Universal_Integer);

         if Is_Task_Type (P_Type) then
            null;

         elsif Is_Access_Type (P_Type) then
            Check_Type;
            Unimplemented_Attribute;

         else
            Error_Attr
              ("prefix of % attribute must be access or task type", P);
         end if;

      ----------
      -- Succ --
      ----------

      when Attribute_Succ =>
         Check_Discrete_Attribute;
         Set_Etype (N, Base_Type (P_Type));

      ---------
      -- Tag --
      ---------

      when Attribute_Tag =>
         Check_E0;
         Unimplemented_Attribute;
         Set_Etype (N, RTE (RE_Tag));

      ----------------
      -- Terminated --
      ----------------

      when Attribute_Terminated =>
         Check_E0;
         Set_Etype (N, Standard_Boolean);
         Check_Task_Prefix;

      ----------
      -- Tick --
      ----------

      when Attribute_Tick =>
         Check_Standard_Prefix;
         Rewrite_Substitute_Tree (N,
           Make_Real_Literal (Loc,
             Numerator   => UI_From_Int (Ttypes.System_Tick_Nanoseconds),
             Denominator => UI_From_Int (9),
             Decimal     => True));
         Analyze (N);

      ----------------
      -- Truncation --
      ----------------

      when Attribute_Truncation =>
         Check_Float_Type_1;
         Unimplemented_Attribute;

      -----------------------
      -- Unbiased_Rounding --
      -----------------------

      when Attribute_Unbiased_Rounding =>
         Check_Float_Type_1;
         Unimplemented_Attribute;

      ----------------------
      -- Unchecked_Access --
      ----------------------

      when Attribute_Unchecked_Access =>
         Access_Attribute;

      ------------------------------
      -- Universal_Literal_String --
      ------------------------------

      --  This is a GNAT specific attribute whose prefix must be a named
      --  number where the expression is either a single numeric literal,
      --  or a numeric literal immediately preceded by a minus sign. The
      --  result is equivalent to a string literal containing the text of
      --  the literal as it appeared in the source program with a possible
      --  leading minus sign.

      when Attribute_Universal_Literal_String => Universal_Literal_String :
      begin
         Check_E0;

         if not Is_Entity_Name (P)
           or else Ekind (Entity (P)) not in Named_Kind
         then
            Error_Attr ("prefix for % attribute must be named number", P);

         else
            declare
               Expr     : Node_Id;
               Negative : Boolean;
               S        : Source_Ptr;
               Src      : Source_Buffer_Ptr;

            begin
               Expr := Original_Node (Expression (Parent (Entity (P))));

               if Nkind (Expr) = N_Op_Minus then
                  Negative := True;
                  Expr := Original_Node (Right_Opnd (Expr));
               else
                  Negative := False;
               end if;

               if Nkind (Expr) /= N_Integer_Literal
                 and then Nkind (Expr) /= N_Real_Literal
               then
                  Error_Attr
                    ("named number for % attribute must be simple literal", N);
               end if;

               --  Build string literal corresponding to source literal text

               Start_String;

               if Negative then
                  Store_String_Char (Get_Char_Code ('-'));
               end if;

               S := Sloc (Expr);
               Src := Source_Text (Get_Source_File_Index (S));

               while Src (S) /= ';' and then Src (S) /= ' ' loop
                  Store_String_Char (Get_Char_Code (Src (S)));
                  S := S + 1;
               end loop;

               --  Now we rewrite the attribute with the string literal

               Rewrite_Substitute_Tree (N,
                 Make_String_Literal (Loc, End_String));
               Analyze (N);
            end;
         end if;
      end Universal_Literal_String;

      -------------------------
      -- Unrestricted_Access --
      -------------------------

      --  This is a GNAT specific attribute which is like Access except that
      --  all scope checks and checks for aliased views are omitted.

      when Attribute_Unrestricted_Access =>
         Access_Attribute;

      ---------
      -- Val --
      ---------

      when Attribute_Val => Val : declare
      begin
         Check_E1;
         Check_Discrete_Type;

         if not Is_Integer_Type (Etype (E1)) then
            Error_Attr ("argument of % attribute is not integer type", N);

         else
            Resolve (E1, Etype (E1));
         end if;

         Set_Etype (N, P_Type);
      end Val;

      -----------
      -- Valid --
      -----------

      when Attribute_Valid =>
         Check_E0;
         Unimplemented_Attribute;

      -----------
      -- Value --
      -----------

      when Attribute_Value =>
         Check_E1;
         Check_Discrete_Type;
         Resolve (E1, Standard_String);
         Set_Etype (N, P_Type);

      -------------
      -- Version --
      -------------

      when Attribute_Version =>
         Check_E0;
         Unimplemented_Attribute;

      ----------------
      -- Wide_Image --
      ----------------

      when Attribute_Wide_Image =>
         Check_E1;
         Unimplemented_Attribute;

      ----------------
      -- Wide_Value --
      ----------------

      when Attribute_Wide_Value =>
         Check_E1;
         Unimplemented_Attribute;

      -----------
      -- Width --
      -----------

      when Attribute_Width =>
         Check_E1;
         Check_Discrete_Type;
         Set_Etype (N, Universal_Integer);

      ---------------
      -- Word_Size --
      ---------------

      when Attribute_Word_Size =>
         Standard_Attribute (System_Word_Size);

      -----------
      -- Write --
      -----------

      when Attribute_Write =>
         Check_E2;
         Unimplemented_Attribute;

      end case;

   --  All errors raise Error_Resync, so that we get out before any further
   --  damage occurs when an error is detected (for example, if we check for
   --  one attribute expression, and the check succeeds, we want to be able
   --  to proceed securely assuming that an expression is in fact present.

   exception
      when Error_Resync =>
         Set_Etype (N, Any_Type);
         return;

   end Analyze_Attribute;

   -----------------------
   -- Resolve_Attribute --
   -----------------------

   procedure Resolve_Attribute (N : Node_Id; Typ : Entity_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      P     : constant Node_Id    := Prefix (N);
      Index : Interp_Index;
      It    : Interp;

   begin
      if Etype (N) = Universal_Integer or else Etype (N) = Universal_Real then
         Set_Etype (N, Typ);
      end if;

      case Get_Attribute_Id (Attribute_Name (N)) is

         --  For these attributes, if the prefix denotes an entity, it is
         --  interpreted as a name, never as a call. It may be overloaded,
         --  in which case resolution uses the profile of the context type.
         --  Otherwise prefix must be resolved.

         when Attribute_Access
            | Attribute_Address
            | Attribute_Unchecked_Access =>

            if Is_Entity_Name (P) then
               if Is_Overloaded (P) then
                  Get_First_Interp (P, Index, It);

                  while Present (It.Nam) loop

                     if Type_Conformant (It.Nam, Designated_Type (Typ)) then
                        Set_Entity (P, It.Nam);
                        exit;
                     end if;

                     Get_Next_Interp (Index, It);
                  end loop;
               end if;

               if Is_Abstract (Entity (P)) then
                  Error_Msg_N
                    ("prefix of % attribute cannot be abstract subprogram",
                      P);
                  Set_Etype (N, Any_Type);
                  raise Error_Resync;
               end if;

            else
               Resolve (P, Etype (P));
            end if;

            Set_Etype (N, Typ);

         when Attribute_Range =>

            if not Is_Entity_Name (P)
              or else not Is_Type (Entity (P)) then
               Resolve (P, Etype (P));
            end if;

            --  We now replace the Range attribute node with a range
            --  expression whose bounds are the 'First and 'Last attributes
            --  applied to the same prefix. The reason that we do this
            --  transformation here instead of in the expander is that it
            --  simplifies other parts of the semantic analysis (note that
            --  the RM specifically mentions this equivalence, we take care
            --  that the prefix is only evaluated once).

            Set_Evaluate_Once (P, True);
            Rewrite_Substitute_Tree (N,
              Make_Range (Loc,
                Low_Bound =>
                  Make_Attribute_Reference (Loc,
                    Prefix         => P,
                    Attribute_Name => Name_First,
                    Expressions    => Expressions (N)),
                High_Bound =>
                  Make_Attribute_Reference (Loc,
                    Prefix         => P,
                    Attribute_Name => Name_Last,
                    Expressions    => Expressions (N))));
            Analyze (N);

            --  Normally after resolving attribute nodes, Eval_Attribute
            --  is called to do any possible static evaluation of the node.
            --  However, here since the Range attribute has just been
            --  transformed into a range expression it is no longer an
            --  attribute node and therefore the call needs to be avoided
            --  and is accomplished by simply returning from the procedure.

            return;

         --  For other attributes, resolve prefix if it is not a type.

         when others =>
            if not Is_Entity_Name (P)
              or else not Is_Type (Entity (P)) then
               Resolve (P, Etype (P));
            end if;

      end case;

      Eval_Attribute (N);
   end Resolve_Attribute;

end Sem_Attr;
