%{
/*
Copyright (c) 1993 Jason Patterson.

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted.

Be aware, however, that using this software to extract icons from commercial
Macintosh applications may violate the copyright on these applications.

JASON PATTERSON DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO
EVENT SHALL JASON PATTERSON BE LIABLE FOR ANY SPECIAL, INDIRECT OR
CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM THE LOSS
OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.
*/
/************************************************************************/
/*									*/
/* macrestoxpm: extract icons from mac reources and write out xpm icons	*/
/*									*/
/************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include "macres.h"

static char *infile_name;
static int shortnames;
int type;
static int resnum;

#define FRAMES 32    /* max frames of a SICN animation */

static unsigned char image[FRAMES/2*32][32],mask[32][32];
static int hotx,hoty;

static int row,col;

%}

%union
{
  int type;			/* icon type */
  int num;			/* resource number */
  unsigned char bytestr[16];	/* byte string */
};

%token DATA CLOSE
%token <num> OPEN
%token <type> TYPE
%token <bytestr> STR

%%

file:		resources
	;

resources:	/* empty */
	|	resources resource
	;

resource:	resopen strlist resclose
	;

resopen:	DATA type OPEN
{
  if (type != other) /* found an icon! */
  {
    row = col = 0;
    resnum = $3;
  }
}
	;

type:	TYPE	{ type = $1; }
	;

strlist:	/* empty */
	|	strlist STR
{
  if (type != other)
  {
    int i;
    switch (type)
    {
     case ICN:  /* 32x32 + 32x32: large monochrome icon with mask */
     case ics:  /* 16x16 + 16x16: small monochrome icon with mask */
     case ICON: /* 32x32: large monochrome icon */
     case SICN: /* 16x16: small monochrome icon or animation */
     case CURS: /* 16x16 + 16x16 + 2 + 2: pointer with mask and hotspot */
      {
	int size = ((type == ICN || type == ICON) ? 32 : 16);
	int bpl = size/8; /* bytes per line */

	if (type == SICN) size *= FRAMES/2;  /* SICN animation */

	for (i = 0; i < 16; ++i)
	{
	  if (row < size)  /* image */
	  {
	    image[row+i/bpl][i%bpl*8+0] = ($2[i] & 0x80) ? 1 : 0;
	    image[row+i/bpl][i%bpl*8+1] = ($2[i] & 0x40) ? 1 : 0;
	    image[row+i/bpl][i%bpl*8+2] = ($2[i] & 0x20) ? 1 : 0;
	    image[row+i/bpl][i%bpl*8+3] = ($2[i] & 0x10) ? 1 : 0;
	    image[row+i/bpl][i%bpl*8+4] = ($2[i] & 0x08) ? 1 : 0;
	    image[row+i/bpl][i%bpl*8+5] = ($2[i] & 0x04) ? 1 : 0;
	    image[row+i/bpl][i%bpl*8+6] = ($2[i] & 0x02) ? 1 : 0;
	    image[row+i/bpl][i%bpl*8+7] = ($2[i] & 0x01) ? 1 : 0;
	  }
	  else if (row < 2*size)  /* mask */
	  {
	    mask[row+i/bpl-size][i%bpl*8+0] = $2[i] & 0x80;
	    mask[row+i/bpl-size][i%bpl*8+1] = $2[i] & 0x40;
	    mask[row+i/bpl-size][i%bpl*8+2] = $2[i] & 0x20;
	    mask[row+i/bpl-size][i%bpl*8+3] = $2[i] & 0x10;
	    mask[row+i/bpl-size][i%bpl*8+4] = $2[i] & 0x08;
	    mask[row+i/bpl-size][i%bpl*8+5] = $2[i] & 0x04;
	    mask[row+i/bpl-size][i%bpl*8+6] = $2[i] & 0x02;
	    mask[row+i/bpl-size][i%bpl*8+7] = $2[i] & 0x01;
	  }
	  else if (i == 0)  /* hotspot */
	  {
	    hotx = ($2[0] << 16) + $2[1];
	    hoty = ($2[2] << 16) + $2[3];
	  }
	}
	row += 16/bpl;
      }
      break;
      
     case icl4: /* 32x32x4: large 4-bit color icon */
      for (i = 0; i < 16; ++i)
      {
	image[row][2*i] = ($2[i] & 0xF0) >> 4;
	image[row][2*i+1] = ($2[i] & 0x0F);
      }
      ++row;
      break;
      
     case ics4: /* 16x16x4: small 4-bit color icon */
      for (i = 0; i < 8; ++i)
      {
	image[row][2*i] = ($2[i] & 0xF0) >> 4;
	image[row][2*i+1] = ($2[i] & 0x0F);
      }
      ++row;
      for (i = 0; i < 8; ++i)
      {
	image[row][2*i] = ($2[i+8] & 0xF0) >> 4;
	image[row][2*i+1] = ($2[i+8] & 0x0F);
      }
      ++row;
      break;
      
     case icl8: /* 32x32x8: large 8-bit color icon */
      for (i = 0; i < 16; ++i) image[row][col+i] = $2[i];
      if (col == 16) { ++row; col = 0; }
      else col = 16;
      break;

     case ics8: /* 16x16x8: small 8-bit color icon */
      for (i = 0; i < 16; ++i) image[row][i] = $2[i];
      ++row;
      break;
    }
  }
}
	;

resclose:	CLOSE
{
  if (type != other)
  {
    static const char *colors_8bit[256] =
    {
      "#FFFFFF","#FFFECB","#FFFE9A","#FFFE66","#FFFE33","#FEFE00","#FFCBFE",
      "#FFCBCB","#FFCC9A","#FFCC66","#FFCC33","#FECB00","#FF9AFE","#FF9ACC",
      "#FF9A9A","#FF9966","#FF9933","#FE9800","#FF66FE","#FF66CC","#FF6699",
      "#FF6666","#FF6633","#FE6500","#FF33FE","#FF33CC","#FF3399","#FF3366",
      "#FF3333","#FE3200","#FE00FE","#FE00CB","#FE0098","#FE0065","#FE0032",
      "#FE0000","#CBFFFF","#CBFFCB","#CCFF9A","#CCFF66","#CCFF33","#CBFE00",
      "#CBCBFF","#CCCCCC","#CCCC99","#CCCC66","#CBCB32","#CDCD00","#CC9AFF",
      "#CC99CC","#CC9999","#CC9966","#CB9832","#CD9A00","#CC66FF","#CC66CC",
      "#CC6699","#CC6666","#CB6532","#CD6600","#CC33FF","#CB32CB","#CB3298",
      "#CB3265","#CB3232","#CD3300","#CB00FE","#CD00CD","#CD009A","#CD0066",
      "#CD0033","#CD0000","#9AFFFF","#9AFFCC","#9AFF9A","#99FF66","#99FF33",
      "#99FE00","#9ACCFF","#99CCCC","#009865","#99CC66","#99CB32","#9ACD00",
      "#9A9AFF","#9999CC","#999999","#989865","#9A9A33","#989800","#9966FF",
      "#9966CC","#986598","#986565","#9A6633","#986500","#9933FF","#9832CB",
      "#9A339A","#9A3366","#9A3333","#983200","#9800FE","#9A00CD","#980098",
      "#980065","#980032","#980000","#66FFFF","#66FFCC","#66FF99","#66FF66",
      "#66FF33","#66FE00","#66CCFF","#66CCCC","#66CC99","#66CC66","#66CB32",
      "#66CD00","#6699FF","#6699CC","#659898","#659865","#669A33","#659800",
      "#6666FF","#6666CC","#656598","#666666","#656532","#666600","#6633FF",
      "#6532CB","#66339A","#653265","#653232","#663300","#6500FE","#6600CD",
      "#650098","#660066","#660033","#660000","#33FFFF","#33FFCC","#33FF99",
      "#33FF66","#33FF33","#33FE00","#33CCFF","#32CBCB","#32CB98","#32CB65",
      "#33CB32","#33CD00","#3399FF","#3299CB","#339A9A","#339A66","#339A33",
      "#329800","#3366FF","#3266CB","#33669A","#326565","#326532","#336600",
      "#3333FF","#3233CB","#33339A","#323265","#333333","#333300","#3200FE",
      "#3300CD","#320098","#330066","#330033","#330000","#00FEFE","#00FECB",
      "#00FE98","#00FE65","#00FE32","#00FE00","#00CBFE","#00CDCD","#00CD9A",
      "#00CD66","#00CD33","#00CD00","#0098FE","#009ACD","#009898","#009865",
      "#009832","#009800","#0066FE","#0066CD","#006598","#006666","#006633",
      "#006600","#0033FE","#0033CD","#003298","#003366","#003333","#003300",
      "#0000FE","#0000CD","#000098","#000066","#000033","#EF0000","#DC0000",
      "#BA0000","#AB0000","#890000","#770000","#550000","#440000","#220000",
      "#110000","#00EF00","#00DC00","#00BA00","#00AB00","#008900","#007700",
      "#005500","#004400","#002200","#001100","#0000EF","#0000DC","#0000BA",
      "#0000AB","#000089","#000077","#000055","#000044","#000022","#000011",
      "#EEEEEE","#DDDDDD","#BBBBBB","#AAAAAA","#888888","#777777","#555555",
      "#444444","#222222","#111111","#000000"
    };

    static const char *colors_4bit[16] =
    {
      "White","Yellow","LightSalmon","Red","DeepPink","BlueViolet","NavyBlue",
      "CornflowerBlue","ForestGreen","DarkGreen","SaddleBrown","Tan",
      "LightGrey","Grey","DimGrey","black"
    };

    static const char useable_chars[] =
    {
      ' ','!','#','$','%','&','\'','(',')','*','+',',','-','.','/','0','1','2',
      '3','4','5','6','7','8','9',':',';','<','=','>','?','@','A','B','C','D',
      'E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V',
      'W','X','Y','Z','[',']','^','_','`','a','b','c','d','e','f','g','h','i',
      'j','k','l','m','n','o','p','q','r','s','t','u','v','w','x','y','z','{',
      '|','}','~'
    };

    static const char *icontypes[] =
    { "ICN","ics","icl4","ics4","icl8","ics8","ICON","SICN","CURS" };

    static const int iconsizes[] =
    { 32, 16, 32, 16, 32, 16, 32, 16, 16 };

    unsigned int map[256];      /* XPM color number for this mac color */
    unsigned int map_cols[256]; /* mac color for this XPM color number */

    int x,y,ncols = 0,chars_per_pixel,has_mask = 0,width,height,frame = 0;

    width = height = iconsizes[type-1];

    for (x = 0; x < 256; ++x) map[x] = 257;
    for (y = 0; y < height; ++y)
      for (x = 0; x < width; ++x)
	if (map[image[y][x]] == 257) /* found new color */
	{
	  map_cols[ncols] = image[y][x];
	  map[image[y][x]] = ncols++;
	}
    
    chars_per_pixel = (ncols < ('~' - ' ' + 1 - 2) ? 1 : 2);
    
    if (type == ICN || type == ics || type == CURS)
    {
      for (y = 0; y < height; ++y)
	for (x = 0; x < width; ++x)
	  if (mask[y][x] == 0) has_mask = 1; /* found a hole! */
    }

    do
    {
      static char fname[1024];
      FILE *f;

      sprintf(fname,"%s%s%s_%s%d",shortnames ? "" : infile_name,
	      shortnames ? "" : "_",icontypes[type-1],
	      resnum < 0 ? "_" : "",abs(resnum));
      if (type == SICN && row/height > 1)
	sprintf(&fname[strlen(fname)],"_%d",frame);
      strcat(fname,".xpm");

      f = fopen(fname,"w");
      if (f)
      {
	fname[strlen(fname)-4] = '\0';  /* kill the .xpm extension */

	fprintf(f,"/* XPM */\nstatic char *%s[] = {\n"
		"\"%d %d %d %d",
		fname,width,height,ncols+has_mask,chars_per_pixel);
	if (type == CURS) fprintf(f," %d %d",hotx,hoty);
	fprintf(f,"\",\n");

	switch (type)
	{
	 case ICN: case ics: case ICON: case SICN: case CURS:
	  if (has_mask) fprintf(f,"\"%c m None\",\n",useable_chars[0]);
	  fprintf(f,"\"%c m white\",\n\"%c m black\",\n",
		  useable_chars[0+has_mask],useable_chars[1+has_mask]);
	  break;
	 default:
	  for (x = 0; x < ncols; ++x)
	  {
	    fprintf(f,"\"");
	    
	    if (chars_per_pixel == 1) fprintf(f,"%c",useable_chars[x]);
	    else fprintf(f,"%02x",x);
	    
	    fprintf(f," c ");
	    
	    if (type == icl4 || type == ics4) 
	      fprintf(f,colors_4bit[map_cols[x]]);
	    else fprintf(f,colors_8bit[map_cols[x]]);
	    
	    fprintf(f,"\",\n");
	  }
	  break;
	}
	
	for (y = frame*height; y < (frame+1)*height; ++y)
	{
	  fprintf(f,"\"");
	  for (x = 0; x < width; ++x)
	  {
	    if (has_mask)
	    {
	      if (mask[y][x])
		fprintf(f,"%c",useable_chars[map[image[y][x]]+1]);
	      else fprintf(f,"%c",useable_chars[0]);
	    }
	    else if (chars_per_pixel == 1)
	      fprintf(f,"%c",useable_chars[map[image[y][x]]]);
	    else fprintf(f,"%02x",map[image[y][x]]);
	  }
	  fprintf(f,"\",\n");
	}
	
	fprintf(f,"};\n");
	fclose(f);
      }
      else fprintf(stderr,"cannot create file `%s'\n",fname);

      ++frame;
    }
    while (type == SICN && frame < row/height);

    row = col = 0; /* get ready for next time */
  }
}
	;

%%

extern FILE *yyin;
extern char *yytext;
extern int lineno;

int main(int argc,char *argv[])
{
  if (argc < 2 || argc > 3)
  {
    fprintf(stderr,"usage: %s [-shortnames] mac_c_resource_file\n",argv[0]);
    return 1;
  }

  if (argc == 3 && strcmp(argv[1],"-shortnames") == 0)
  { shortnames = 1; infile_name = argv[2]; }
  else { shortnames = 0; infile_name = argv[1]; }

  fclose(yyin);
  yyin = fopen(infile_name,"r");
  if (!yyin)
  {
    fprintf(stderr,"cannot open `%s'\n",infile_name);
    return 1;
  }

  yyparse();

  return 0;
}

int yyerror(char *s)
{
  fprintf(stderr,"syntax error on line %d of %s\n",lineno,infile_name);
  exit(1);
}
