------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              E X P _ C H 9                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.90 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Exp_Ch3;  use Exp_Ch3;
with Exp_Ch6;  use Exp_Ch6;
with Exp_Util; use Exp_Util;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Ch11;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stand;    use Stand;
with Tbuild;   use Tbuild;
with Uintp;    use Uintp;

package body Exp_Ch9 is

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Add_Discriminal_Declarations
     (Decls : List_Id;
      Typ   : Node_Id;
      Name  : Name_Id;
      Loc   : Source_Ptr);
   --  This routine is used to add discriminal declarations to task bodies
   --  (and protected bodies ???). The discriminants are available by normal
   --  selection from the concurrent object (whose name is passed as the third
   --  parameter). Discriminant references inside the task body have already
   --  been replaced by references to the corresponding discriminals. The
   --  declarations constructed by this procedure hook the references up with
   --  the objects:
   --
   --    discriminal_name : discr_type renames name.discriminant_name;
   --
   --  Obviously we could have expanded the discriminant references in the
   --  first place to be the appropriate selection, but this turns out to
   --  be hard to do because it would introduce difference in handling of
   --  discriminant references depending on their location.

   function Build_Accept_Body
     (Stats : Node_Id;
      Loc   : Source_Ptr)
      return  Node_Id;

   --  Transform accept statement into a block with added exception handler.
   --  Cused both for simple accept statements and for accept alternatives in
   --  select statements.

   function Build_Task_Proc_Specification (T : Entity_Id) return Node_Id;
   --  This routine constructs a specification for the procedure that we will
   --  build for the task body. The specification has the form
   --
   --    procedure tnameB (_Task : access tnameV);
   --
   --  where name is the character name taken from the task type entity that
   --  is passed as the argument to the procedure, and tnameV is the task
   --  value type that is associated with the task type.

   procedure Outer_Handlers (N : Node_Id);
   --  The expansion of a task body must have exception handlers that take
   --  care of task termination in case an otherwise unhandled exception is
   --  raised. This catch-all handler must also handle the special Abort
   --  signal, and prevent its further propagation.

   function Find_Task_Pragma (T : Node_Id; P : Name_Id) return Node_Id;
   --  Searches the task definition T for the first occurrence of the pragma
   --  whose name is given by P. The caller has ensured that the pragma is
   --  present in the task definition.

   function Task_Ref (N : Node_Id) return Node_Id;
   --  Given the name of a task, or the name of an access to a task, this
   --  function returns an expression referencing the associated Task_Id.
   --  Note that a special case is when the name is a reference to a task
   --  type name. This can only happen within a task body, and the meaning
   --  is to get the Task_Id for the currently executing task.

   ----------------------------------
   -- Add_Discriminal_Declarations --
   ----------------------------------

   procedure Add_Discriminal_Declarations
     (Decls : List_Id;
      Typ   : Node_Id;
      Name  : Name_Id;
      Loc : Source_Ptr)
   is
      D : Entity_Id;

   begin
      if Has_Discriminants (Typ) then
         D := First_Discriminant (Typ);

         while Present (D) loop
            Prepend_To (Decls,
              Make_Object_Renaming_Declaration (Loc,
                Defining_Identifier => Discriminal (D),
                Subtype_Mark => New_Reference_To (Etype (D), Loc),
                Name =>
                  Make_Selected_Component (Loc,
                    Prefix        => Make_Identifier (Loc, Name),
                    Selector_Name => Make_Identifier (Loc, Chars (D)))));

            D := Next_Discriminant (D);
         end loop;
      end if;
   end Add_Discriminal_Declarations;

   -----------------------------------
   -- Build_Activation_Chain_Entity --
   -----------------------------------

   procedure Build_Activation_Chain_Entity (N : Node_Id) is
      P     : Node_Id;
      B     : Node_Id;
      Decls : List_Id;

   begin
      --  Loop to find enclosing construct containing activation chain variable

      P := Parent (N);

      while Nkind (P) /= N_Subprogram_Body
        and then Nkind (P) /= N_Package_Declaration
        and then Nkind (P) /= N_Package_Body
        and then Nkind (P) /= N_Block_Statement
      loop
         P := Parent (P);
      end loop;

      --  If we are in a package body, the activation chain variable is
      --  allocated in the corresponding spec
      --  First, we save the package body node because we enter the new
      --  entity in its Declarations list

      B := P;

      if Nkind (P) = N_Package_Body then
         P := Get_Declaration_Node (Corresponding_Spec (P));
         Decls := Declarations (B);

      elsif Nkind (P) = N_Package_Declaration then
         Decls := Visible_Declarations (Specification (B));

      else
         Decls := Declarations (B);
      end if;

      --  If activation chain entity not already declared, declare it

      if No (Activation_Chain_Entity (P)) then
         Set_Activation_Chain_Entity
           (P, Make_Defining_Identifier (Sloc (P), Name_uChain));

         Prepend (
           Make_Object_Declaration (Sloc (P),
             Defining_Identifier => Activation_Chain_Entity (P),
             Object_Definition   =>
               New_Reference_To (RTE (RE_Activation_Chain), Sloc (P))),
           Decls);

         Analyze (First (Decls));
      end if;

   end Build_Activation_Chain_Entity;

   --------------------------
   -- Build_Call_With_Task --
   --------------------------

   function Build_Call_With_Task
     (N : Node_Id;
      E : Entity_Id)
      return Node_Id
   is
      Loc : constant Source_Ptr := Sloc (N);

   begin
      return
        Make_Function_Call (Loc,
          Name => New_Reference_To (E, Loc),
          Parameter_Associations => New_List (Task_Ref (N)));
   end Build_Call_With_Task;

   -------------------------
   -- Build_Master_Entity --
   -------------------------

   procedure Build_Master_Entity (E : Entity_Id) is
      Loc  : constant Source_Ptr := Sloc (E);
      P    : Node_Id;
      Decl : Node_Id;

   begin
      --  Nothing to do if we already built a master entity for this scope

      if Has_Master_Entity (Scope (E)) then
         return;
      end if;

      --  Otherwise first build the master entity
      --    _Master : constant Master_Id := Current_Master;
      --  and insert it just before the current declaration

      Decl :=
        Make_Object_Declaration (Loc,
          Defining_Identifier =>
            Make_Defining_Identifier (Loc, Name_uMaster),
          Constant_Present => True,
          Object_Definition => New_Reference_To (RTE (RE_Master_Id), Loc),
          Expression => New_Reference_To (RTE (RE_Current_Master), Loc));

      P := Parent (E);
      Insert_Before (P, Decl);
      Analyze (Decl);
      Set_Has_Master_Entity (Scope (E));

      --  Now mark the containing scope as a task master

      while Nkind (P) /= N_Compilation_Unit loop
         P := Parent (P);

         --  If we fall off the top, we are at the outer level, and the
         --  environment task is our effective master, so nothing to mark.

         if Nkind (P) = N_Task_Body
           or else Nkind (P) = N_Block_Statement
           or else Nkind (P) = N_Subprogram_Body
         then
            Set_Is_Task_Master (P, True);
            return;
         end if;
      end loop;
   end Build_Master_Entity;

   --------------------------------
   -- Build_Task_Activation_Call --
   --------------------------------

   procedure Build_Task_Activation_Call (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Chain : Entity_Id;
      Call  : Node_Id;
      P     : Node_Id;

   begin
      --  Get the activation chain entity. Except in the case of a package
      --  body, this is in the node that was passed. For a package body,we
      --  have to find the corresponding package declaration node.

      --  Note: can remove generic case when we do generics properly???

      if Nkind (N) = N_Package_Body then
         P := Corresponding_Spec (N);

         loop
            P := Parent (P);
            exit when Nkind (P) = N_Package_Declaration
              or else Nkind (P) = N_Generic_Package_Declaration; -- ???
         end loop;

         Chain := Activation_Chain_Entity (P);

      else
         Chain := Activation_Chain_Entity (N);
      end if;

      if Present (Chain) then
         Call :=
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_Activate_Tasks), Loc),
             Parameter_Associations =>
               New_List (New_Reference_To (Chain, Loc)));

         if Present (Handled_Statement_Sequence (N)) then
            Prepend (Call, Statements (Handled_Statement_Sequence (N)));
         else
            Set_Handled_Statement_Sequence (N,
               Make_Handled_Sequence_Of_Statements (Loc,
                  Statements => New_List (Call)));
         end if;

         Analyze (Call);
      end if;

   end Build_Task_Activation_Call;

   -------------------------------
   -- Build_Task_Allocate_Block --
   -------------------------------

   procedure Build_Task_Allocate_Block
     (Actions : List_Id;
      N       : Node_Id;
      Args    : List_Id)
   is
      PtrT   : constant Entity_Id  := Etype (N);
      T      : constant Entity_Id  := Entity (Expression (N));
      Init   : constant Entity_Id  := Base_Init_Proc (T);
      Loc    : constant Source_Ptr := Sloc (N);
      Blkent : Entity_Id;
      Block  : Node_Id;

   begin
      Blkent := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));

      Block :=
        Make_Block_Statement (Loc,
          Identifier => New_Reference_To (Blkent, Loc),
          Declarations => New_List (

            --  _Chain  : Activation_Chain;

            Make_Object_Declaration (Loc,
              Defining_Identifier =>
                Make_Defining_Identifier (Loc, Name_uChain),
              Object_Definition   =>
                New_Reference_To (RTE (RE_Activation_Chain), Loc)),

            --  procedure _Expunge is
            --  begin
            --     Expunge_Unactivated_Tasks (_Chain);
            --  end;

            Make_Subprogram_Body (Loc,
              Specification =>
                Make_Procedure_Specification (Loc,
                  Defining_Unit_Name =>
                    Make_Defining_Identifier (Loc, Name_uExpunge)),

              Declarations => Empty_List,

              Handled_Statement_Sequence =>
                Make_Handled_Sequence_Of_Statements (Loc,
                  Statements => New_List (
                    Make_Procedure_Call_Statement (Loc,
                      Name =>
                        New_Reference_To (
                          RTE (RE_Expunge_Unactivated_Tasks), Loc),
                      Parameter_Associations => New_List (
                        Make_Identifier (Loc, Name_uChain))))))),

          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,

              Statements => New_List (

               --  Init (Args);

                Make_Procedure_Call_Statement (Loc,
                  Name => New_Reference_To (Init, Loc),
                  Parameter_Associations => Args),

               --  Activate_Tasks (_Chain);

                Make_Procedure_Call_Statement (Loc,
                  Name => New_Reference_To (RTE (RE_Activate_Tasks), Loc),
                  Parameter_Associations => New_List (
                    Make_Identifier (Loc, Name_uChain)))),

              Identifier => Make_Identifier (Loc, Name_uExpunge)),

          Has_Created_Identifier => True);

      Append_To (Actions,
        Make_Implicit_Label_Declaration (Loc,
          Defining_Identifier => Blkent,
          Label => Block));

      Append_To (Actions, Block);

   end Build_Task_Allocate_Block;

   -----------------------------------
   -- Build_Task_Proc_Specification --
   -----------------------------------

   function Build_Task_Proc_Specification (T : Entity_Id) return Node_Id is
      Loc : constant Source_Ptr := Sloc (T);
      Nam : constant Name_Id    := Chars (T);

   begin
      return
        Make_Procedure_Specification (Loc,
          Defining_Unit_Name =>
            Make_Defining_Identifier (Loc,
              Chars => New_External_Name (Nam, 'B')),

          Parameter_Specifications =>
            New_List (
              Make_Parameter_Specification (Loc,
                Defining_Identifier =>
                  Make_Defining_Identifier (Loc, Name_uTask),
                Parameter_Type =>
                  Make_Access_Definition (Loc,
                    Subtype_Mark =>
                      New_Reference_To
                        (Corresponding_Record_Type (T), Loc)))));

   end Build_Task_Proc_Specification;

   --------------------------
   -- Complete_Master_Call --
   --------------------------

   function Complete_Master_Call (N : Node_Id) return Node_Id is
   begin
      return
        Make_Procedure_Call_Statement (Sloc (N),
          Name => New_Reference_To (RTE (RE_Complete_Master), Sloc (N)));
   end Complete_Master_Call;

   ------------------
   -- Convert_Task --
   ------------------

   function Convert_Task (N : Node_Id; Typ : Entity_Id) return Node_Id is
      Loc : constant Source_Ptr := Sloc (N);

   begin
      if not Is_Task_Type (Typ) then
         return N;
      else
         return
           Make_Unchecked_Type_Conversion (Loc,
             Subtype_Mark =>
               New_Reference_To (Corresponding_Record_Type (Typ), Loc),
               Expression => New_Copy (N));
      end if;
   end Convert_Task;

   ----------------------------
   -- Entry_Index_Expression --
   ----------------------------

   function Entry_Index_Expression
     (Ename : Node_Id;
      Index : Node_Id;
      Taskv : Node_Id := Empty;
      Ttyp  : Entity_Id := Empty)
      return  Node_Id
   is
      Loc  : constant Source_Ptr := Sloc (Ename);
      Ent  : constant Entity_Id  := Entity (Ename);
      Expr : Node_Id;
      Prev : Entity_Id;
      S    : Node_Id;
      Trec : Node_Id;

   begin
      --  Simple entry case, create reference to entry index constant

      if Index = Empty then
         return
           New_Reference_To (Entry_Index_Constant (Ent), Loc);

      --  Entry family case, we construct an expression which is a series
      --  of addition operations. The first operand is the number of single
      --  entries in the task, the second operand is the index value relative
      --  to the start of the referenced family, and the remaining operands
      --  are the lengths of the entry families that precede this entry, i.e.
      --  the constructed expression is:

      --    number_simple_entries +
      --      (s'pos (index-value) - s'pos (family'first)) + 1 +
      --      family'length + ...

      --  where index-value is the given index value, and s is the index
      --  subtype (we have to use pos because the subtype might be an
      --  enumeration type preventing direct substraction). The family
      --  references are to the corresponding array components of the
      --  task record value. Note that the task entry array is one-indexed.

      else
         Trec := Corresponding_Record_Type (Ttyp);
         S := Etype (Discrete_Subtype_Definition (Declaration_Node (Ent)));

         Expr :=
           Make_Op_Add (Loc,
             Left_Opnd  =>
               Make_Integer_Literal (Loc, 1 + Number_Simple_Entries (Ttyp)),
             Right_Opnd =>
               Make_Op_Subtract (Loc,
                 Left_Opnd =>
                   Make_Attribute_Reference (Loc,
                     Attribute_Name => Name_Pos,
                     Prefix => New_Reference_To (S, Loc),
                     Expressions => New_List (New_Copy (Index))),
                 Right_Opnd =>
                   Make_Attribute_Reference (Loc,
                     Attribute_Name => Name_Pos,
                     Prefix => New_Reference_To (S, Loc),
                     Expressions => New_List (
                       Make_Attribute_Reference (Loc,
                         Prefix => New_Reference_To (S, Loc),
                         Attribute_Name => Name_First)))));

         --  Now add lengths of preceding entry families. Skip
         --  over simple entries, and over anonymous array types
         --  defined for each entry family type.

         Prev := First_Entity (Trec);

         while Chars (Prev) /= Chars (Ent) loop

            if Ekind (Prev) = E_Component
              and then Is_Array_Type (Etype (Prev))
            then
               Expr :=
                 Make_Op_Add (Loc,
                 Left_Opnd  => Expr,
                 Right_Opnd =>
                   Make_Attribute_Reference (Loc,
                     Attribute_Name => Name_Length,
                     Prefix => New_Reference_To (Etype (Prev), Loc)));
            end if;

            Prev := Next_Entity (Prev);
         end loop;

         return Expr;
      end if;
   end Entry_Index_Expression;

   ---------------------------
   -- Establish_Task_Master --
   ---------------------------

   procedure Establish_Task_Master (N : Node_Id) is
      Call : Node_Id;

   begin
      Call := Build_Runtime_Call (Sloc (N), RE_Enter_Master);
      Prepend_To (Declarations (N), Call);
      Analyze (Call);
      Protect_Statements (N, RTE (RE_Complete_Master));
   end Establish_Task_Master;

   --------------------------------
   -- Expand_Accept_Declarations --
   --------------------------------

   --  Part of the expansion of an accept statement involves the creation of
   --  a declaration that can be referenced from the statement sequence of
   --  the accept:

   --    Ann : Address;

   --  This declaration is inserted immediately before the accept statement
   --  and it is important that it be inserted before the statements of the
   --  statement sequence are analyzed. Thus it would be too late to create
   --  this declaration in the Expand_N_Accept_Statement routine, which is
   --  why there is a separate procedure to be called directly from Sem_Ch9.

   --  It is used to hold the address of the record containing the parameters
   --  (see Expand_N_Entry_Call for more details on how this record is built).
   --  References to the parameters do an unchecked conversion of this address
   --  to a pointer to the required record type, and then access the field that
   --  holds the value of the required parameter. The entity for the address
   --  variable is held in the Accept_Address field of the corresponding entry
   --  entity, and this field must be set before the statements are processed.

   --  The above description applies to the case of a stand alone accept
   --  statement, i.e. one not appearing as part of a select alternative.

   --  For the case of an accept that appears as part of a select alternative
   --  of a selective accept, we must still create the declaration right away,
   --  since Ann is needed immediately, but there is an important difference:

   --    The declaration is inserted before the selective accept, not before
   --    the accept statement (which is not part of a list anyway, and so would
   --    not accomodate inserted declarations)

   --    We only need one address variable for the entire selective accept. So
   --    the Ann declaration is created only for the first accept alternative,
   --    and subsequent accept alternatives reference the same Ann variable.

   --  We can distinguish the two cases by seeing whether the accept statement
   --  is part of a list. If not, then it must be in an accept alternative.

   procedure Expand_Accept_Declarations (N : Node_Id; Ent : Entity_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Etyp  : constant Entity_Id  := Entry_Parameters_Type (Ent);
      Ann   : Entity_Id;
      Enn   : Entity_Id;
      Adecl : Node_Id;
      Edecl : Node_Id;
      Fdecl : Node_Id;

   begin
      if Expander_Active then

         --  Case of stand alone accept statement

         if Is_List_Member (N) then
            Ann := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));

            Adecl :=
              Make_Object_Declaration (Loc,
                Defining_Identifier => Ann,
                Object_Definition => New_Reference_To (RTE (RE_Address), Loc));

            Insert_Before (N, Adecl);
            Analyze (Adecl);

         --  Case of accept statement which is in an accept alternative

         else
            declare
               Acc_Alt : constant Node_Id := Parent (N);
               Sel_Acc : constant Node_Id := Parent (Acc_Alt);
               Alt     : Node_Id;

            begin
               pragma Assert (Nkind (Acc_Alt) = N_Accept_Alternative);
               pragma Assert (Nkind (Sel_Acc) = N_Selective_Accept);

               --  Find first accept alternative of the selective accept. A
               --  valid selective accept must have at least one accept in it.

               Alt := First (Selective_Accept_Alternatives (Sel_Acc));

               while Nkind (Alt) /= N_Accept_Alternative loop
                  Alt := Next (Alt);
               end loop;

               --  If we are the first accept statement, then we have to
               --  create the Ann variable, as for the stand alone case,
               --  except that it is inserted before the selective accept.

               if N = Accept_Statement (Alt) then
                  Ann :=
                    Make_Defining_Identifier (Loc, New_Internal_Name ('A'));

                  Adecl :=
                    Make_Object_Declaration (Loc,
                      Defining_Identifier => Ann,
                      Object_Definition =>
                        New_Reference_To (RTE (RE_Address), Loc));

                  Insert_Before (Sel_Acc, Adecl);
                  Analyze (Adecl);

               --  If we are not the first accept statement, then find the
               --  Ann variable allocated by the first accept and use it

               else
                  Ann :=
                    Accept_Address
                      (Entity (Entry_Direct_Name (Accept_Statement (Alt))));
               end if;
            end;
         end if;

         --  Merge here with Ann either created or referenced, and Adecl
         --  pointing to the corresponding declaration. Remaining processing
         --  is the same for the two cases.

         Set_Accept_Address (Ent, Ann);
      end if;
   end Expand_Accept_Declarations;

   ------------------------------
   -- Expand_N_Abort_Statement --
   ------------------------------

   --  Expand abort T1, T2, .. Tn; into:
   --    Abort_Tasks (Task_List'(1 => T1.Task_Id, 2 => T2.Task_Id ...))

   procedure Expand_N_Abort_Statement (N : Node_Id) is
      Loc    : constant Source_Ptr := Sloc (N);
      Tlist  : constant List_Id    := Names (N);
      Count  : Nat;
      Aggr   : Node_Id;
      Tasknm : Node_Id;

   begin
      Aggr := Make_Aggregate (Loc, Component_Associations => New_List);
      Count := 0;

      Tasknm := First (Tlist);

      while Present (Tasknm) loop
         Count := Count + 1;
         Append_To (Component_Associations (Aggr),
           Make_Component_Association (Loc,
             Choices => New_List (
               Make_Integer_Literal (Loc, UI_From_Int (Count))),
             Expression => Task_Ref (Tasknm)));
         Tasknm := Next (Tasknm);
         Count := Count + 1;
      end loop;

      Replace_Substitute_Tree (N,
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RE_Abort_Tasks), Loc),
          Parameter_Associations => New_List (
            Make_Qualified_Expression (Loc,
              Subtype_Mark => New_Reference_To (RTE (RE_Task_List), Loc),
              Expression => Aggr))));

      Analyze (N);

   end Expand_N_Abort_Statement;

   -------------------------------
   -- Expand_N_Accept_Statement --
   -------------------------------

   --  This procedure handles expansion of accept statements that stand
   --  alone, i.e. they are not part of an accept alternative. The expansion
   --  of accept statement in accept alternatives is handled by the routines
   --  Expand_N_Accept_Alternative and Expand_N_Selective_Accept. The
   --  following description applies only to stand alone accept statements.

   --  If there is no handled statement sequence, then this is called a
   --  trivial accept, and the expansion is:

   --    Accept_Trivial (entry-index)

   --  If there is a handled statement sequence, then the expansion is:

   --    Ann : Address;
   --    Enn : entry-parameters-type [only if parameters present]
   --    for Enn'Address use Ann'Address;

   --    begin
   --       begin
   --          Accept_Call (entry-index, Ann);
   --          <statement sequence from N_Accept_Statement node>
   --          Complete_Rendezvous;
   --
   --       exception
   --          when ... =>
   --             <exception handler from N_Accept_Statement node>
   --             Complete_Rendezvous;
   --          when ... =>
   --             <exception handler from N_Accept_Statement node>
   --             Complete_Rendezvous;
   --          ...
   --       end;

   --    exception
   --       when others =>
   --          Exceptional_Complete_Rendezvous (Current_Exception);
   --    end;

   --  The first three declarations were already inserted ahead of the
   --  accept statement by the Expand_Accept_Declarations procedure, which
   --  was called directly from the semantics during analysis of the accept.
   --  statement, before analyzing its contained statements.

   --  The declarations from the N_Accept_Statement, as noted in Sinfo, come
   --  from possible expansion activity (the original source of course does
   --  not have any declarations associated with the accept statement, since
   --  an accept statement has no declarative part). In particular, if the
   --  expander is active, the first such declaration is the declaration of
   --  the Accept_Params_Ptr entity (see Sem_Ch9.Analyze_Accept_Statement).
   --
   --  The two blocks are merged into a single block if the inner block has
   --  no exception handlers, but otherwise two blocks are required, since
   --  exceptions might be raised in the exception handlers of the inner
   --  block, and Exceptional_Complete_Rendezvous must be called.

   procedure Expand_N_Accept_Statement (N : Node_Id) is
      Loc     : constant Source_Ptr := Sloc (N);
      Stats   : constant Node_Id    := Handled_Statement_Sequence (N);
      Ename   : constant Node_Id    := Entry_Direct_Name (N);
      Eindx   : constant Node_Id    := Entry_Index (N);
      Eent    : constant Entity_Id  := Entity (Ename);
      Ann     : constant Entity_Id  := Accept_Address (Eent);
      Etyp    : constant Entity_Id  := Entry_Parameters_Type (Eent);
      Ttyp    : constant Entity_Id  := Etype (Scope (Eent));
      Call    : Node_Id;
      Block   : Node_Id;

   begin
      --  If accept statement is not part of a list, then its parent must be
      --  an accept alternative, and, as described above, we do not do any
      --  expansion for such accept statements at this level.

      if not Is_List_Member (N) then
         pragma Assert (Nkind (Parent (N)) = N_Accept_Alternative);
         return;

      --  Trivial accept case (no statement sequence)

      elsif No (Stats) then
         Rewrite_Substitute_Tree (N,
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_Accept_Trivial), Loc),
             Parameter_Associations => New_List (
               Entry_Index_Expression (Ename, Eindx, Empty, Ttyp))));

         Analyze (N);
         return;

      --  Case of statement sequence present

      else
         --  Construct the block

         Block :=
           Make_Block_Statement (Loc,
             Handled_Statement_Sequence => Build_Accept_Body (Stats, Loc));

         --  Prepend call to Accept_Call to main statement sequence

         Call :=
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_Accept_Call), Loc),
             Parameter_Associations => New_List (
               Entry_Index_Expression (Ename, Eindx, Empty, Ttyp),
               New_Reference_To (Ann, Loc)));

         Prepend (Call, Statements (Stats));
         Analyze (Call);

         --  Finally we can replace the accept statement by the new block

         Rewrite_Substitute_Tree (N, Block);
         Analyze (N);
         return;
      end if;

   end Expand_N_Accept_Statement;

   -----------------------
   -- Build_Accept_Body --
   -----------------------

   function Build_Accept_Body
     (Stats : Node_Id;
      Loc   : Source_Ptr) return Node_Id
   is
      Block : Node_Id;
      New_S : Node_Id;
      Hand  : Node_Id;
      Call  : Node_Id;
   begin

      --  Add the end of the statement sequence, Complete_Rendezvous is called.

      Call := Build_Runtime_Call (Loc, RE_Complete_Rendezvous);
      Append (Call, Statements (Stats));
      Analyze (Call);

      --  If exception handlers are present, then append Complete_Rendezvous
      --  calls to the handlers, and construct the required outer block.

      if Present (Exception_Handlers (Stats)) then
         Hand := First (Exception_Handlers (Stats));

         while Present (Hand) loop
            Call := Build_Runtime_Call (Loc, RE_Complete_Rendezvous);
            Append (Call, Statements (Hand));
            Analyze (Call);
            Hand := Next (Hand);
         end loop;

         New_S :=
           Make_Handled_Sequence_Of_Statements (Loc,
              Statements => New_List (Block));
      else
         New_S := Stats;
      end if;

      --  At this stage we know that the new statement sequence does not
      --  have an exception handler part, so we supply one with the when
      --  others branch that calls Exceptional_Complete_Rendezvous.

      Set_Exception_Handlers (New_S,
        New_List (
          Make_Exception_Handler (Loc,
            Exception_Choices => New_List (Make_Others_Choice (Loc)),
            Statements => New_List (
              Make_Procedure_Call_Statement (Loc,
                Name => New_Reference_To (
                  RTE (RE_Exceptional_Complete_Rendezvous), Loc),
                Parameter_Associations => New_List (
                  Make_Function_Call (Loc,
                    Name => New_Reference_To (
                      RTE (RE_Current_Exception), Loc))))))));

      Sem_Ch11.Analyze_Exception_Handlers (Exception_Handlers (New_S));
      return New_S;

   end Build_Accept_Body;

   -------------------------------------
   -- Expand_N_Conditional_Entry_Call --
   -------------------------------------

   --  The conditional entry call is converted to a call to Task_Entry_Call:

   --    declare
   --       B : Boolean;
   --       P : parms := (parm, parm, parm);

   --    begin
   --       Task_Entry_Call
   --         (acceptor-task,
   --          entry-index,
   --          P'Address,
   --          Conditional_Call,
   --          B);
   --       parm := P.param;
   --       parm := P.param;
   --       ...
   --       if B then
   --          normal-statements
   --       else
   --          else-statements
   --       end if;
   --    end;

   --  For a description of the use of P and the assignments after the
   --  call, see Expand_N_Entry_Call_Statement. Note that the entry call
   --  of the conditional entry call has already been expanded (by the
   --  Expand_N_Entry_Call_Statement procedure) as follows:

   --    declare
   --       P : parms := (parm, parm, parm);
   --    begin
   --       Call_Simple (acceptor-task, entry-index, P'Address);
   --       parm := P.param;
   --       parm := P.param;
   --       ...
   --    end;

   --  so the task at hand is to convert the latter expansion into the former

   procedure Expand_N_Conditional_Entry_Call (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Alt   : constant Node_Id    := Entry_Call_Alternative (N);
      Blk   : constant Node_Id    := Entry_Call_Statement (Alt);
      Decls : List_Id;
      Parms : List_Id;
      Call  : Node_Id;
      Stmts : List_Id;
      B     : Entity_Id;

   begin
      B := Make_Defining_Identifier (Loc, Name_uB);

      --  Insert declaration of B in declarations of existing block

      if No (Declarations (Blk)) then
         Set_Declarations (Blk, New_List);
      end if;

      Prepend_To (Declarations (Blk),
        Make_Object_Declaration (Loc,
          Defining_Identifier => B,
          Object_Definition => New_Reference_To (Standard_Boolean, Loc)));

      --  Create new call statement

      Stmts := Statements (Handled_Statement_Sequence (Blk));
      Call := Remove_Head (Stmts);
      Parms := Parameter_Associations (Call);
      Append_To (Parms, New_Reference_To (RTE (RE_Conditional_Call), Loc));
      Append_To (Parms, New_Reference_To (B, Loc));

      Call :=
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RE_Task_Entry_Call), Loc),
          Parameter_Associations => Parms);

      --  Construct statement sequence for new block

      Append_To (Stmts,
        Make_If_Statement (Loc,
          Condition => New_Reference_To (B, Loc),
          Then_Statements => Statements (Alt),
          Else_Statements => Else_Statements (N)));

      Prepend (Call, Stmts);

      --  The result is the new block

      Rewrite_Substitute_Tree (N,
        Make_Block_Statement (Loc,
          Declarations => Declarations (Blk),
          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc, Stmts)));

      Analyze (N);

   end Expand_N_Conditional_Entry_Call;

   -----------------------------------
   -- Expand_N_Entry_Call_Statement --
   -----------------------------------

   --  The entry call is converted to a call to Call_Simple

   --    declare
   --       P : parms := (parm, parm, parm);
   --    begin
   --       Call_Simple (acceptor-task, entry-index, P'Address);
   --       parm := P.param;
   --       parm := P.param;
   --       ...
   --    end;

   --  Here Pnn is an aggregate of the type constructed for the entry to hold
   --  the parameters, and the constructed aggregate value contains either the
   --  parameters or, in the case of non-elementary types, references to these
   --  parameters. Then the address of this aggregate is passed to the runtime
   --  routine, along with the task id value and the task entry index value.
   --  Pnn is only required if parameters are present.

   --  The assignments after the call are present only in the case of in-out
   --  or out parameters for elementary types, and are used to assign back the
   --  resulting values of such parameters.

   --  Note: the reason that we insert a block here is that in the context
   --  of selects, conditional entry calls etc. the entry call statement
   --  appears on its own, not as an element of a list.

   procedure Expand_N_Entry_Call_Statement (N : Node_Id) is
   begin
      Expand_Call (N);

      --  Convert entry call to Call_Simple call

      declare
         Loc     : constant Source_Ptr := Sloc (N);
         Parms   : constant List_Id    := Parameter_Associations (N);
         Nam     : constant Node_Id    := Name (N);
         Pdecl   : Node_Id;
         Decls   : List_Id;
         Index   : Node_Id;
         Taskval : Node_Id;
         Tasktyp : Node_Id;
         Ename   : Node_Id;
         Ent     : Entity_Id;
         Ent_Acc : Entity_Id;
         P       : Entity_Id;
         Plist   : List_Id;
         Parm1   : Node_Id;
         Parm2   : Node_Id;
         Parm3   : Node_Id;
         Call    : Node_Id;
         Actual  : Node_Id;
         Formal  : Node_Id;
         Stats   : List_Id;

      begin
         --  For a simple entry, the name is a selected component, with the
         --  prefix being the task value, and the selector being the entry.

         if Nkind (Nam) = N_Selected_Component then
            Taskval := Prefix (Nam);
            Ename   := Selector_Name (Nam);
            Index   := Empty;

         --  For a member of an entry family, the name is an indexed component
         --  where the prefix is a selected component, whose prefix in turn
         --  is the task value, and whose selector is the entry family. The
         --  single expression in the expressions list of the indexed component
         --  is the subscript for the family.

         else
            pragma Assert (Nkind (Nam) = N_Indexed_Component);
            Taskval := Prefix (Prefix (Nam));
            Ename   := Selector_Name (Prefix (Nam));
            Index   := First (Expressions (Nam));
         end if;

         --  Simple entry and entry family cases merge here

         Ent     := Entity (Ename);
         Ent_Acc := Entry_Parameters_Type (Ent);
         Tasktyp := Etype (Taskval);

         --  If prefix is an access type, dereference to obtain the task type

         if Is_Access_Type (Tasktyp) then
            Tasktyp := Designated_Type (Tasktyp);
         end if;

         --  First parameter is the Task_Id value from the task value,
         --  obtained by selecting the _Task_Id from the result of
         --  doing an unchecked conversion to convert the task value
         --  to the corresponding record type.

         Parm1 := Task_Ref (Taskval);

         --  Second parameter is the task entry index, computed by
         --  the routine provided for this purpose.

         Parm2 := Entry_Index_Expression (Ename, Index, Taskval, Tasktyp);

         --  The third parameter is the packaged parameters. If there are
         --  none, then it is just the null address, since nothing is passed

         if No (Parms) then
            Parm3 := New_Reference_To (RTE (RE_Null_Address), Loc);
            Decls := No_List;

         --  Case of parameters present, where third argument is the address
         --  of a packaged record containing the required parameter values.

         else
            --  First build a list of parameter values, which are the actual
            --  parameters in the case of elementary types and pointers to
            --  the parameters (actually references), for composite types.

            Plist := New_List;

            Actual := First_Actual (N);
            Formal := First_Formal (Ent);

            while Present (Actual) loop
               if Is_Elementary_Type (Etype (Actual)) then
                  Append_To (Plist, New_Copy (Actual));
               else
                  Append_To (Plist,
                    Make_Reference (Loc, Prefix => New_Copy (Actual)));
               end if;

               Actual := Next_Actual (Actual);
               Formal := Next_Formal (Formal);
            end loop;

            --  Now build the declaration of parameters initialized with the
            --  aggregate containing this constructed parameter list.

            P := Make_Defining_Identifier (Loc, Name_uP);

            Pdecl :=
              Make_Object_Declaration (Loc,
                Defining_Identifier => P,
                Aliased_Present => True,
                Object_Definition =>
                  New_Reference_To (Designated_Type (Ent_Acc), Loc),
                Expression =>
                  Make_Aggregate (Loc, Expressions => Plist));

            Parm3 :=
               Make_Attribute_Reference (Loc,
                 Attribute_Name => Name_Address,
                 Prefix => New_Reference_To (P, Loc));

            Decls := New_List (Pdecl);
         end if;

         --  Now we can create the call

         Call :=
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_Call_Simple), Loc),
             Parameter_Associations => New_List (Parm1, Parm2, Parm3));

         Stats := New_List (Call);

         --  If there are elementary out or in/out parameters add
         --  assignment statements for the result values.

         if Present (Parms) then
            Actual := First_Actual (N);
            Formal := First_Formal (Ent);

            while Present (Actual) loop
               if Is_Elementary_Type (Etype (Actual))
                 and then Ekind (Formal) /= E_In_Parameter
               then
                  Insert_After (Call,
                    Make_Assignment_Statement (Loc,
                      Name => New_Copy (Actual),
                      Expression =>
                        Make_Selected_Component (Loc,
                          Prefix => New_Reference_To (P, Loc),
                          Selector_Name =>
                            Make_Identifier (Loc, Chars (Formal)))));
               end if;

               Actual := Next_Actual (Actual);
               Formal := Next_Formal (Formal);
            end loop;
         end if;

         --  Finally, create block and analyze it

         Rewrite_Substitute_Tree (N,
           Make_Block_Statement (Loc,
             Declarations => Decls,
             Handled_Statement_Sequence =>
               Make_Handled_Sequence_Of_Statements (Loc,
                 Statements => Stats)));

         Analyze (N);
      end;

   end Expand_N_Entry_Call_Statement;

   --------------------------------
   -- Expand_N_Entry_Declaration --
   --------------------------------

   --  If there are parameters, then first, each of the formals is marked
   --  by setting Is_Entry_Formal. Next a record type is built which is
   --  used to hold the parameter values. The name of this record type is
   --  entryP where entry is the name of the entry, with an additional
   --  corresponding access type called entryPA. The record type has matching
   --  components for each formal (the component names are the same as the
   --  formal names). For elementary types, the component type matches the
   --  formal type. For composite types, an access type is declared (with
   --  the name formalA) which designates the formal type, and the type of
   --  the component is this access type. Finally the Entry_Component of
   --  each formal is set to reference the corresponding record component.
   --  In addition, for the simple entry cases, the Entry_Index_Constant
   --  entity is created and declared.

   procedure Expand_N_Entry_Declaration (N : Node_Id) is
      Loc         : constant Source_Ptr := Sloc (N);
      Entry_Ent   : constant Entity_Id  := Defining_Identifier (N);
      Task_Ent    : constant Entity_Id  := Scope (Entry_Ent);
      Components  : List_Id;
      Formal      : Node_Id;
      Ftype       : Entity_Id;
      Last_Decl   : Node_Id;
      Component   : Entity_Id;
      Ctype       : Entity_Id;
      Decl        : Node_Id;
      Rec_Ent     : Entity_Id;
      Acc_Ent     : Entity_Id;
      Indx_Ent    : Entity_Id;
      Indx_Decl   : Node_Id;

   begin
      Formal := First_Formal (Entry_Ent);
      Last_Decl := N;

      --  Most processing is done only if parameters are present

      if Present (Formal) then
         Components := New_List;

         --  Loop through formals

         while Present (Formal) loop
            Set_Is_Entry_Formal (Formal);
            Component := Make_Defining_Identifier (Loc, Chars (Formal));
            Set_Entry_Component (Formal, Component);
            Ftype := Etype (Formal);

            --  Elementary type, just append to component list

            if Is_Elementary_Type (Ftype) then
               Ctype := Ftype;

            --  Composite type, declare new access type and then append

            else
               Ctype :=
                 Make_Defining_Identifier (Loc,
                   New_External_Name (Chars (Formal), 'A'));

               Decl :=
                 Make_Full_Type_Declaration (Loc,
                   Defining_Identifier => Ctype,
                   Type_Definition     =>
                     Make_Access_To_Object_Definition (Loc,
                       All_Present        => True,
                       Subtype_Indication => New_Reference_To (Ftype, Loc)));

               Insert_After (Last_Decl, Decl);
               Last_Decl := Decl;
            end if;

            Append_To (Components,
              Make_Component_Declaration (Loc,
                Defining_Identifier => Component,
                Subtype_Indication  => New_Reference_To (Ctype, Loc)));

            Formal := Next_Formal (Formal);
         end loop;

         --  Create the Entry_Parameter_Record declaration

         Rec_Ent :=
           Make_Defining_Identifier (Loc,
             Chars => New_External_Name (Chars (Entry_Ent), 'P'));

         Decl :=
           Make_Full_Type_Declaration (Loc,
             Defining_Identifier => Rec_Ent,
             Type_Definition     =>
               Make_Record_Definition (Loc,
                 Component_List =>
                   Make_Component_List (Loc,
                     Component_Declarations => Components)));

         Insert_After (Last_Decl, Decl);
         Last_Decl := Decl;

         --  Construct and link in the corresponding access type

         Acc_Ent :=
           Make_Defining_Identifier (Loc,
             Chars => New_External_Name (Chars (Rec_Ent), 'A'));

         Set_Entry_Parameters_Type (Entry_Ent, Acc_Ent);

         Decl :=
           Make_Full_Type_Declaration (Loc,
             Defining_Identifier => Acc_Ent,
             Type_Definition     =>
               Make_Access_To_Object_Definition (Loc,
                 All_Present        => True,
                 Subtype_Indication => New_Reference_To (Rec_Ent, Loc)));

         Insert_After (Last_Decl, Decl);
         Last_Decl := Decl;

      end if;

      --  Create entry index constant for simple entries. Note that we have to
      --  manually set the Is_Public flag, since this is a case of an entity
      --  not at the outer level that is accessed directly by the expanded code

      if Ekind (Entry_Ent) = E_Entry then
         Indx_Ent :=
           Make_Defining_Identifier (Loc,
             Chars => New_External_Name (Chars (Entry_Ent), 'X'));

         Set_Entry_Index_Constant (Entry_Ent, Indx_Ent);
         Set_Is_Public (Indx_Ent);

         Indx_Decl :=
           Make_Object_Declaration (Loc,
             Defining_Identifier => Indx_Ent,
             Constant_Present => True,
             Object_Definition =>
               New_Reference_To (RTE (RE_Task_Entry_Index), Loc),
             Expression =>
               Make_Integer_Literal (Loc, Number_Simple_Entries (Task_Ent)));

         Insert_After (Last_Decl, Indx_Decl);
         Last_Decl := Decl;
      end if;

   end Expand_N_Entry_Declaration;

   -------------------------------
   -- Expand_N_Selective_Accept --
   -------------------------------

   procedure Expand_N_Selective_Accept (N : Node_Id) is
      Block         : Node_Id;
      Decls         : List_Id := New_List;
      Stats         : List_Id := New_List;

      Accept_List   : List_Id := New_List;
      Body_List     : List_Id := New_List;
      Trailing_List : List_Id := New_List;

      Delay_Present     : Boolean := False;
      Else_Present      : Boolean := False;
      Terminate_Present : Boolean := False;
      Select_Mode       : Entity_Id;

      Alt     : Node_Id;
      End_Lab : Node_Id;
      Lab     : Node_Id;
      Lab_Id  : Node_Id;
      Loc     : constant Source_Ptr := Sloc (N);
      Ann     : Entity_Id := Empty;
      Num     : Int;
      Q       : Node_Id;
      Qnam    : Entity_Id := Make_Defining_Identifier (Loc,
                                    New_External_Name ('S', 0));
      X       : Node_Id;
      Xnam    : Entity_Id := Make_Defining_Identifier (Loc,
                                    New_External_Name ('X', 1));

      ----------------
      -- Add_Accept --
      ----------------

      procedure Add_Accept (Alt : Node_Id) is
         Ename     : Node_Id := Entry_Direct_Name (Accept_Statement (Alt));
         Null_Body : Node_Id;
         Proc_Body : Node_Id;
         Index     : Node_Id := Entry_Index (Accept_Statement (Alt));
         Expr      : Node_Id;

      begin
         if No (Ann) then
            Ann := Accept_Address (Entity (Ename));
         end if;

         if Present (Condition (Alt)) then
            Expr := Make_Conditional_Expression (Loc,
              New_List (Condition (Alt),
                        Entry_Index_Expression (Ename, Index),
                        New_Reference_To (RTE (RE_Null_Task_Entry), Loc)));
         else
            Expr := Entry_Index_Expression (Ename, Index);
         end if;

         if Present (Handled_Statement_Sequence (Accept_Statement (Alt))) then
            Null_Body := New_Reference_To (Standard_False, Loc);

            Proc_Body :=
              Make_Subprogram_Body (Loc,
                Specification =>
                  Make_Procedure_Specification (Loc,
                    Defining_Unit_Name =>
                      Make_Defining_Identifier (Loc,
                        New_External_Name (Chars (Ename), 'A'))),
               Declarations => New_List,
               Handled_Statement_Sequence =>
                 Build_Accept_Body (
                   Handled_Statement_Sequence (Accept_Statement (Alt)), Loc));

            Append (Proc_Body, Body_List);

         else
            Null_Body := New_Reference_To (Standard_True,  Loc);
         end if;

         Append (
            Make_Aggregate (Loc, Expressions => New_List (Null_Body, Expr)),
            Accept_List);

      end Add_Accept;


   begin
      --  First insert some declarations before the select. The first is:

      --    Ann : Address

      --  This variable holds the parameters passed to the accept body. This
      --  declaration has already been inserted by the time we get here by
      --  a call to Expand_Accept_Declarations made from the semantics when
      --  processing the first accept statement contained in the select. We
      --  can find this entity as Accept_Address (E), where E is any of the
      --  entries references by contained accept statements.

      --  The first step is to scan the list of Selective_Accept_Statements
      --  to find this entity, and also count the number of accepts, and
      --  determine if terminated, delay or else is present:

      Num := 0;
      Alt := First (Selective_Accept_Alternatives (N));

      while Present (Alt) loop

         if Nkind (Alt) = N_Accept_Alternative then
            Num := Num + 1;
            Add_Accept (Alt);

         elsif Nkind (Alt) = N_Delay_Alternative then
            Delay_Present := True;

         elsif Nkind (Alt) = N_Terminate_Alternative then
            Terminate_Present := True;
         end if;

         Alt := Next (Alt);
      end loop;

      Else_Present := Present (Else_Statements (N));

      --  At the same time (see procedure Add_Accept) we build the accept list:

      --    Qnn : Accept_List (1 .. num-select) := (
      --          (null-body, entry-index),
      --          (null-body, entry-index),
      --          ..
      --          (null_body, entry-index));

      --  In the above declaration, null-body is True if the corresponding
      --  accept has no body, and false otherwise. The entry is either the
      --  entry index expression if there is no guard, or if a guard is
      --  present, then a conditional expression of the form:

      --    (if guard then entry-index else Null_Task_Entry)

      --  If a guard is statically known to be false, the entry can simply
      --  be omitted from the accept list.

      Q :=
        Make_Object_Declaration (Loc,
          Defining_Identifier => Qnam,
          Object_Definition =>
            New_Reference_To (RTE (RE_Accept_List_Access), Loc),
          Expression =>
            Make_Allocator (Loc,
              Expression =>
                Make_Qualified_Expression (Loc,
                  Subtype_Mark =>
                    New_Reference_To (RTE (RE_Accept_List), Loc),
                  Expression =>
                    Make_Aggregate (Loc,  Expressions => Accept_List))));

      Append (Q, Decls);

      --  Then we declare the variable that holds the index for the accept
      --  that will be selected for service:

      --    Xnn : Select_Index;

      X :=
        Make_Object_Declaration (Loc,
          Defining_Identifier => Xnam,
          Object_Definition =>
            New_Reference_To (RTE (RE_Select_Index), Loc));

      Append (X,  Decls);

      --  After this follow  procedure declarations for each accept body.

      --    procedure Pnn is
      --    begin
      --       ...
      --    end;

      --  where the ... are statements from the corresponding procedure body.
      --  No parameters are involved, since the parameters are passed via Ann
      --  and the parameter references have already been expanded to be direct
      --  references to Ann (see Exp_Ch2.Expand_Entry_Parameter). Furthermore,
      --  any embedded tasking statements (which would normally be illegal in
      --  procedures, have been converted to calls to the tasking runtime so
      --  there is no problem in putting them into procedures.

      --  The original accept statement has been expanded into a block in
      --  the same fashion as for simple accepts (see Build_Accept_Body).

      --  Note: we don't really need to build these procedures for the case
      --  where no delay statement is present, but is is just as easy to
      --  build them unconditionally, and not significantly inefficient,
      --  since if they are short they will be inlined anyway.

      --  The procedure declarations have been assembled in Body_List.

      --  If delays are present, we must compute the required delay, so
      --  first generate the declarations:

      --    Dnn : Boolean := 0;
      --    Tnn : Some_Time_Type.Time;
      --    Vnn : Some_Time_Type.Time;

      --  Dnn will be set to the index of the minimum delay (i.e. the active
      --  delay that is actually chosen as the basis for the possible delay)

      if Delay_Present then

         --  Generate the required declarations

         null; -- TBD ???

         --  Now for each delay alternative generate:

         --    if guard-expression then
         --       Vnn := delay-expression;
         --       if Dnn > 0 or else Tnn < Vnn then
         --          Tnn := Vnn;
         --          Dnn := nn;
         --       end if;
         --    end if;

         null; -- TBD ???
      end if;

      --  Now we can issue the Selective_Wait call:

      --    Selective_Wait (Qnn, select-mode, Ann, Xnam);

      if Terminate_Present then
         Select_Mode := RTE (RE_Terminate_Mode);

      elsif Else_Present or Delay_Present then
         Select_Mode := RTE (RE_Else_Mode);

      else
         Select_Mode := RTE (RE_Simple_Mode);
      end if;

      declare
         Call   : Node_Id;
         Params : List_Id := New_List;

      begin
         Append (New_Reference_To (Qnam, Loc), Params);
         Append (New_Reference_To (Select_Mode,  Loc), Params);
         Append (New_Reference_To (Ann,  Loc), Params);
         Append (New_Reference_To (Xnam, Loc), Params);

         Call := Make_Procedure_Call_Statement (Loc,
            Name => New_Reference_To (RTE (RE_Selective_Wait), Loc),
            Parameter_Associations => Params);

         Append (Call, Stats);
      end;


      --  Now generate code to act on the result. There is an entry
      --  in this case for each accept statement with a non-null body.

      --    case X is
      --      when No_Rendezvous =>  --  omitted if simple mode
      --         Null;

      --      when 1 =>
      --         P1n;

      --      when 2 =>
      --         P2n;

      --      when others =>
      --         Null;
      --    end case;

      declare
         Case_Stat    : Node_Id;
         Choice       : Node_Id;
         Choices      : List_Id;
         Alt_List     : List_Id := New_List;
         Alt_Stats    : List_Id;
         Index        : Int := 1;
         Proc         : Node_Id;

         ----------------------------
         -- Make_And_Declare_Label --
         ----------------------------

         function Make_And_Declare_Label (Num : Int) return Node_Id is
            Lab_Id : Node_Id;

         begin
            Lab_Id := Make_Identifier (Loc, New_External_Name ('L', Num));
            Lab := Make_Label (Loc, Lab_Id);

            Append (
              Make_Implicit_Label_Declaration (Loc,
                Defining_Identifier  =>
                  Make_Defining_Identifier (Loc, Chars (Lab_Id)),
                Label => Lab),
               Decls);
            return Lab;
         end Make_And_Declare_Label;


      begin
         --  Generate label for common exit.

         End_Lab := Make_And_Declare_Label (Num + 1);

         --  First entry is the default case, when no rendezvous is possible.

         Choices := New_List (
           New_Reference_To (RTE (RE_No_Rendezvous), Loc));

         Alt_Stats := New_List (Make_Null_Statement (Loc));

         if Else_Present then

            --  If no rendezvous is possible,  the else part is executed.

            Lab := Make_And_Declare_Label (0);
            Append (
               Make_Goto_Statement (Loc,
                 Name => New_Copy (Identifier (Lab))),
            Alt_Stats);

            Append (Lab, Trailing_List);
            Append_List (Else_Statements (N), Trailing_List);
            Append_To (Trailing_List,
              Make_Goto_Statement (Loc,
                Name => New_Copy (Identifier (End_Lab))));
         end if;

         Append_To (Alt_List,
           Make_Case_Statement_Alternative (Loc,
             Discrete_Choices => Choices,
             Statements => Alt_Stats));

         --  We make use of the fact that Accept_Index is an integer type,
         --  and generate successive literals for entries for each accept.
         --  Only those for which there is a body or trailing statements are
         --  given a case entry.

         Alt := First (Selective_Accept_Alternatives (N));
         Proc := First (Body_List);

         while Present (Alt) loop

            if Nkind (Alt) = N_Accept_Alternative then
               Alt_Stats := No_List;

               if Present
                 (Handled_Statement_Sequence (Accept_Statement (Alt)))
               then

                  Choices := New_List (
                    Make_Integer_Literal (Loc,
                      Intval => UI_From_Int (Index)));

                  Alt_Stats := New_List (
                    Make_Procedure_Call_Statement (Loc,
                      Name => New_Reference_To (
                        Defining_Unit_Name (Specification (Proc)), Loc)));

                  Proc := Next (Proc);
               end if;

               if Statements (Alt) /= Empty_List then

                  if No (Alt_Stats) then

                     --  Accept with no body, followed by trailing statements.

                     Choices := New_List (
                       Make_Integer_Literal (Loc,
                         Intval => UI_From_Int (Index)));

                     Alt_Stats := New_List;
                  end if;

                  --  After the call, if any, branch to to traling statements.
                  --  We create a label for each, as well as the corresponding
                  --  label declaration.

                  Lab := Make_And_Declare_Label (Index);
                  Append_To (Alt_Stats,
                    Make_Goto_Statement (Loc,
                      Name => New_Copy (Identifier (Lab))));

                  Append (Lab, Trailing_List);
                  Append_List (Statements (Alt), Trailing_List);
                  Append_To (Trailing_List,
                    Make_Goto_Statement (Loc,
                      Name => New_Copy (Identifier (End_Lab))));
               end if;

               if Present (Alt_Stats) then

                  --  Procedure call. and/or trailing statements.

                  Append_To (Alt_List,
                    Make_Case_Statement_Alternative (Loc,
                       Discrete_Choices => Choices,
                       Statements => Alt_Stats));
               end if;

               Index := Index + 1;
            end if;

            Alt := Next (Alt);
         end loop;

         --  An others choice is always added.

         Append_To (Alt_List,
           Make_Case_Statement_Alternative (Loc,
              Discrete_Choices => New_List (Make_Others_Choice  (Loc)),
              Statements       => New_List (Make_Null_Statement (Loc))));

         Case_Stat :=
           Make_Case_Statement (Loc,
             Expression   => New_Reference_To (Xnam, Loc),
             Alternatives => Alt_List);

         Append (Case_Stat, Stats);
      end;

      Append_List (Body_List, Decls);

      --  Generate code for the no rendezvous case if no delay active
      --  This is just the code for the else part. If a delay is active
      --  then we generate:

      --    <<Nnn>>
      --       select
      --          delay [until] Tnn;

      --          case Dnn is
      --             when nn =>
      --                "code for nn'th delay alternative"
      --             ...
      --          end case;

      --          goto Znn;

      --       then abort
      --          Selective_Wait (Qnn, Simple_Mode, Ann, Xnn);
      --
      --          case X is
      --             when No_Rendezvous =>  --  omitted if simple mode
      --                null;

      --             when 1 =>
      --                Pnn;
      --                goto Exit1;

      --             when 2 =>
      --                Pnn;
      --                goto Exit2;

      --          end case;
      --       end select;

      null; -- ???

      --  Finally generate a separate case statement for the statements
      --  that follow accept statements. This case statement must be separate
      --  because there is no one-to-one correspondence between this one and
      --  the accept case: some accept statements have no bodies but trailing
      --  statements, and viceversa.

      --    <<Xnn>>
      --       "code after nn'th accept"
      --       goto Znn;

      --  And finally, the exit label for the whole operation:

      --    <<Znn>>

      Append_List (Trailing_List, Stats);
      Append (End_Lab, Stats);

      Block :=
        Make_Block_Statement (Loc,
          Declarations => Decls,
          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => Stats));

      Rewrite_Substitute_Tree (N,  Block);
      Analyze (N);

      --  Note: have to worry more about abort deferral in above code

   end Expand_N_Selective_Accept;


   --------------------------------------
   -- Expand_N_Single_Task_Declaration --
   --------------------------------------

   --  Single task declarations should never be present after semantic
   --  analysis, since we expect them to be replaced by a declaration of
   --  an anonymous task type, followed by a declaration of the task
   --  object. We include this routine to make sure that is happening!

   procedure Expand_N_Single_Task_Declaration (N : Node_Id) is
   begin
      pragma Assert (False); null;
   end Expand_N_Single_Task_Declaration;

   ------------------------
   -- Expand_N_Task_Body --
   ------------------------

   --  Given a task body

   --    task body tname is
   --       declarations
   --    begin
   --       statements
   --    end x;

   --  This expansion routine converts it into a procedure and sets the
   --  elaboration flag for the procedure to true, to represent the fact
   --  that the task body is now elaborated:

   --    procedure tnameB (_Task : access tnameV) is
   --       discrimal : dtype renames _Task.discriminant;

   --    begin
   --       System.Task_Stages.Complete_Activation;
   --       statements
   --    at end
   --       System.Task_Stages.Complete_Task;
   --    end tnameB;

   --    tnameE := True;

   --  In addition, if the task body is an activator, then a call to
   --  activate tasks is added at the start of the statements, before
   --  the call to Complete_Activation, and if in addition the task is
   --  a master then it must be established as a master.

   --  There is one discriminal declaration line generated for each
   --  discriminant that is present to provide an easy reference point
   --  for discriminant references inside the body (see Exp_Ch2.Expand_Name).

   --  Note on relationship to GNARLI definition. In the GNARLI definition,
   --  task body procedures have a profile (Arg : System.Address). That is
   --  needed because GNARLI has to use the same access-to-subprogram type
   --  for all task types. We depend here on knowing that in GNAT, passing
   --  an address argument by value is identical to passing a a record value
   --  by access (in either case a single pointer is passed), so even though
   --  this procedure has the wrong profile. In fact it's all OK, since the
   --  callings sequence is identical.

   procedure Expand_N_Task_Body (N : Node_Id) is
      Loc  : constant Source_Ptr := Sloc (N);
      Ttyp : constant Entity_Id  := Corresponding_Spec (N);
      Tval : constant Entity_Id  := Corresponding_Record_Type (Ttyp);
      Call : Node_Id;

   begin
      Add_Discriminal_Declarations (Declarations (N), Ttyp, Name_uTask, Loc);

      if Is_Task_Master (N) then
         Establish_Task_Master (N);
      end if;

      Build_Task_Activation_Call (N);
      Protect_Statements (N, RTE (RE_Complete_Task));
      Outer_Handlers (N);

      Call := Build_Runtime_Call (Loc, RE_Complete_Activation);
      Prepend (Call, Statements (Handled_Statement_Sequence (N)));
      Analyze (Call);

      Rewrite_Substitute_Tree (N,
        Make_Subprogram_Body (Loc,
          Specification => Build_Task_Proc_Specification (Ttyp),
          Declarations  => Declarations (N),
          Handled_Statement_Sequence => Handled_Statement_Sequence (N)));

      Analyze (N);

      Insert_After (N,
        Make_Assignment_Statement (Loc,
          Name =>
            Make_Identifier (Loc, New_External_Name (Chars (Ttyp), 'E')),
          Expression => New_Reference_To (Standard_True, Loc)));

   end Expand_N_Task_Body;

   ------------------------------------
   -- Expand_N_Task_Type_Declaration --
   ------------------------------------

   --  We have several things to do. First we must create a Boolean flag used
   --  to mark if the body is elaborated yet. This variable gets set to True
   --  when the body of the task is elaborated (we can't rely on the normal
   --  ABE mechanism for the task body, since we need to pass an access to
   --  this elaboration boolean to the runtime routines).

   --    taskE : aliased Boolean := False;

   --  Next a variable is declared to hold the task stack size (either
   --  the default, which is the initial value given here, or a value that
   --  is set by a pragma Storage_Size appearing later on.

   --    taskZ : Size_Type := Unspecified_Size;

   --  Next we create a corresponding record type declaration used to represent
   --  values of this task. The general form of this type declaration is

   --    type taskV (discriminants) is record
   --      _Task_Id     : Task_Id;
   --      entry_family : array (bounds) of Void;
   --      _Priority    : Integer   := priority_expression;
   --      _Size        : Size_Type := Size_Type (size_expression);
   --    end record;

   --  The discriminants are present only if the corresponding task type has
   --  discriminants, and they exactly mirror the task type discriminants.

   --  The Id field is always present. It contains the Task_Id value, as
   --  set by the call to Create_Task. Note that although the task is
   --  limited, the task value record type is not limited, so there is no
   --  problem in passing this field as an out parameter to Create_Task.

   --  One entry_family component is present for each entry family in the
   --  task definition. The bounds correspond to the bounds of the entry
   --  family (which may depend on discriminants). The element type is
   --  void, since we only need the bounds information for determining
   --  the entry index. Note that the use of an anonymous array would
   --  normally be illegal in this context, but this is a parser check,
   --  and the semantics is quite prepared to handle such a case.

   --  The Size field is present only if a Storage_Size pragma appears in
   --  the task definition. The expression captures the argument that was
   --  present in the pragma, and is used to override the task stack size
   --  otherwise associated with the task type.

   --  The Priority field is present only if a Priority or Interrupt_Priority
   --  pragma appears in the task definition. The expression captures the
   --  argument that was present in the pragma, and is used to provide
   --  the Size parameter to the call to Create_Task.

   --  When a task is declared, an instance of the task value record is
   --  created. The elaboration of this declaration creates the correct
   --  bounds for the entry families, and also evaluates the size and
   --  priority expressions if needed. The initialization routine for
   --  the task type itself then calls Create_Task with appropriate
   --  parameters to initialize the value of the Task_Id field.

   --  Note: the address of this record is passed as the "Discriminants"
   --  parameter for Create_Task. Since Create_Task merely passes this onto
   --  the body procedure, it does not matter that it does not quite match
   --  the GNARLI model of what is being passed (the record contains more
   --  than just the discriminants, but the discriminants can be found from
   --  the record value).

   --  The Entity_Id for this created record type is placed in the
   --  Corresponding_Record_Type field of the associated task type entity.

   --  Next we create a procedure specification for the task body procedure:

   --    procedure taskB (_Task : access taskV);

   --  Note that this must come after the record type declaration, since
   --  the spec refers to this type. It turns out that the initialization
   --  procedure for the value type references the task body spec, but that's
   --  fine, since it won't be generated till the freeze point for the type,
   --  which is certainly after the task body spec declaration.

   --  Finally we create constant declarations for the task entry index values
   --  for all simple entries. As noted in the Entry_Index_Constant definition
   --  in Einfo, we don't really need these, since we could simply use the
   --  proper integer literal values, but by declaring symbolic names, the
   --  generated source code is clarified with no resulting efficiency loss
   --  in the generated object code.

   procedure Expand_N_Task_Type_Declaration (N : Node_Id) is
      Loc       : constant Source_Ptr := Sloc (N);
      Tasktyp   : constant Entity_Id  := Etype (Defining_Identifier (N));
      Tasknm    : constant Name_Id    := Chars (Tasktyp);
      Taskdef   : constant Node_Id    := Task_Definition (N);
      Proc_Spec : Node_Id;
      Rec_Ent   : Entity_Id;
      Rec_Decl  : Node_Id;
      Cdecls    : List_Id;
      Dlist     : List_Id;
      Disc      : Node_Id;
      Efam      : Entity_Id;
      Elab_Decl : Node_Id;
      Size_Decl : Node_Id;
      Body_Decl : Node_Id;
      Task_Ent  : Entity_Id;
      Indx_Decl : Node_Id;
      Indx_Ent  : Entity_Id;

   begin
      --  First create the elaboration variable

      Elab_Decl :=
        Make_Object_Declaration (Loc,
          Defining_Identifier =>
            Make_Defining_Identifier (Loc, New_External_Name (Tasknm, 'E')),
          Aliased_Present      => True,
          Object_Definition    => New_Reference_To (Standard_Boolean, Loc),
          Expression           => New_Reference_To (Standard_False, Loc));
      Insert_After (N, Elab_Decl);

      --  Next create the declaration of the size variable (tasknmZ)

      Size_Decl :=
        Make_Object_Declaration (Loc,
          Defining_Identifier =>
            Make_Defining_Identifier (Loc, New_External_Name (Tasknm, 'Z')),
          Object_Definition => New_Reference_To (RTE (RE_Size_Type), Loc),
          Expression => New_Reference_To (RTE (RE_Unspecified_Size), Loc));

      Insert_After (Elab_Decl, Size_Decl);

      --  Next create the record declaration for the task value type.
      --  This is done last, since the corresponding record initialization
      --  procedure will reference the previously created entities.

      Rec_Ent :=
        Make_Defining_Identifier (Loc, New_External_Name (Tasknm, 'V'));
      Set_Corresponding_Record_Type (Tasktyp, Rec_Ent);
      Set_Ekind (Rec_Ent, E_Record_Type);
      Set_Is_Task_Record_Type (Rec_Ent, True);
      Set_Corresponding_Concurrent_Type (Rec_Ent, Tasktyp);

      --  Initialize component list, we will fill it in later

      Cdecls := New_List;

      --  Make a copy of the discriminant specifications

      if Present (Discriminant_Specifications (N)) then
         Dlist := New_List;
         Disc := First (Discriminant_Specifications (N));

         while Present (Disc) loop
            Append_To (Dlist,
              Make_Discriminant_Specification (Loc,
                Defining_Identifier =>
                  New_Copy (Defining_Identifier (Disc)),
                Discriminant_Type =>
                  New_Copy (Discriminant_Type (Disc)),
                Expression =>
                  New_Copy (Expression (Disc))));
            Disc := Next (Disc);
         end loop;

      else
         Dlist := No_List;
      end if;

      --  Now we can construct the record type declaration

      Rec_Decl :=
        Make_Full_Type_Declaration (Loc,
          Defining_Identifier => Rec_Ent,
          Discriminant_Specifications => Dlist,
          Type_Definition =>
            Make_Record_Definition (Loc,
              Component_List =>
                Make_Component_List (Loc,
                  Component_Declarations => Cdecls)));

      --  Fill in the component declarations. First the _Task_Id field:

      Append_To (Cdecls,
        Make_Component_Declaration (Loc,
          Defining_Identifier =>
            Make_Defining_Identifier (Loc, Name_uTask_Id),
          Subtype_Indication => New_Reference_To (RTE (RE_Task_ID), Loc)));

      --  Add components for entry families

      Efam := First_Entity (Tasktyp);

      while Present (Efam) loop
         if Ekind (Efam) = E_Entry_Family then
            Append_To (Cdecls,
              Make_Component_Declaration (Loc,
                Defining_Identifier =>
                  Make_Defining_Identifier (Loc, Chars (Efam)),
                Subtype_Indication =>
                  Make_Constrained_Array_Definition (Loc,
                    Discrete_Subtype_Definitions => (New_List (
                      New_Copy (Discrete_Subtype_Definition (Parent (Efam))))),
                    Subtype_Indication =>
                      New_Reference_To (Standard_Character, Loc))));
         end if;

         Efam := Next_Entity (Efam);
      end loop;

      --  Add the priority component if a priority pragma is present

      if Present (Taskdef) and then Has_Priority_Pragma (Taskdef) then
         Append_To (Cdecls,
           Make_Component_Declaration (Loc,
             Defining_Identifier =>
               Make_Defining_Identifier (Loc, Name_uPriority),
             Subtype_Indication => New_Reference_To (Standard_Integer, Loc),
             Expression => New_Copy (
               Expression (First (
                 Pragma_Argument_Associations (
                   Find_Task_Pragma (Taskdef, Name_Priority)))))));
      end if;

      --  Add the task_size component if a priority pragma is present

      if Present (Taskdef)
        and then Has_Storage_Size_Pragma (Taskdef)
      then
         Append_To (Cdecls,
           Make_Component_Declaration (Loc,
             Defining_Identifier =>
               Make_Defining_Identifier (Loc, Name_uSize),
             Subtype_Indication => New_Reference_To (RTE (RE_Size_Type), Loc),
             Expression =>
               Make_Type_Conversion (Loc,
                 Subtype_Mark => New_Reference_To (RTE (RE_Size_Type), Loc),
                 Expression => Relocate_Node (
                   Expression (First (
                     Pragma_Argument_Associations (
                       Find_Task_Pragma (Taskdef, Name_Storage_Size))))))));
      end if;

      Insert_After (Size_Decl, Rec_Decl);

      --  Analyze the record declaration immediately after construction,
      --  because the initialization procedure is needed for single task
      --  declarations before the next entity is analyzed.

      Analyze (Rec_Decl);

      --  Create the declaration of the task body procedure

      Proc_Spec := Build_Task_Proc_Specification (Tasktyp);
      Body_Decl :=
        Make_Subprogram_Declaration (Loc,
          Specification => Proc_Spec);
      Insert_After (Rec_Decl, Body_Decl);

      --  Complete the expansion of access types to the current task
      --  type, if any were declared.

      Expand_Previous_Access_Type (N, Tasktyp);

   end Expand_N_Task_Type_Declaration;

   ---------------------
   -- Outer_Handlers --
   ---------------------

   procedure Outer_Handlers (N : Node_Id) is
      Loc  : constant Source_Ptr := Sloc (N);
      Stmt : constant Node_Id := Handled_Statement_Sequence (N);
      Hdle : List_Id := Exception_Handlers (Stmt);
   begin
      Hdle :=  New_List (
        Make_Exception_Handler (Loc,
          Exception_Choices => New_List (
            New_Reference_To (Stand.Abort_Signal, Loc)),
          Statements => New_List (Make_Null_Statement (Loc))),

        Make_Exception_Handler (Loc,
          Exception_Choices =>
             New_List (Make_Others_Choice (Loc)),
          Statements =>  New_List (Make_Null_Statement (Loc))));

      Set_Handled_Statement_Sequence (N,
        Make_Handled_Sequence_Of_Statements (Loc,
          Statements => New_List (
            Make_Block_Statement (Loc,
              Identifier => Empty,
              Handled_Statement_Sequence => Stmt)),
          Exception_Handlers => Hdle));

   end Outer_Handlers;

   ----------------------
   -- Find_Task_Pragma --
   ----------------------

   function Find_Task_Pragma (T : Node_Id; P : Name_Id) return Node_Id is
      N : Node_Id;

   begin
      N := First (Visible_Declarations (T));

      while Present (N) loop
         if Nkind (N) = N_Pragma and then Chars (N) = P then
            return N;
         else
            N := Next (N);
         end if;
      end loop;

      N := First (Private_Declarations (T));

      while Present (N) loop
         if Nkind (N) = N_Pragma and then Chars (N) = P then
            return N;
         else
            N := Next (N);
         end if;
      end loop;

      pragma Assert (False);
   end Find_Task_Pragma;

   ---------------------------
   -- Make_Task_Create_Call --
   ---------------------------

   function Make_Task_Create_Call (Task_Rec : Entity_Id) return Node_Id is
      Loc    : constant Source_Ptr := Sloc (Task_Rec);
      Tdef   : Node_Id;
      Tdec   : Node_Id;
      Ttyp   : Node_Id;
      Tnam   : Name_Id;
      Args   : List_Id;
      Ent    : Entity_Id;
      Eindx  : Nat;
      Ecount : Node_Id;

   begin
      Ttyp := Corresponding_Concurrent_Type (Task_Rec);
      Tnam := Chars (Ttyp);

      --  Get task declaration. In the case of a task type declaration, this
      --  is simply the parent of the task type entity. In the single task
      --  declaration, this parent will be the implicit type, and we can find
      --  the corresponding single task declaration by searching forward in
      --  the declaration list in the tree.

      Tdec := Parent (Ttyp);

      while Nkind (Tdec) /= N_Task_Type_Declaration
        and then Nkind (Tdec) /= N_Single_Task_Declaration
      loop
         Tdec := Next (Tdec);
      end loop;

      --  Now we can find the task definition from this declaration

      Tdef := Task_Definition (Tdec);

      --  Build the parameter list for the call. Note that _Init is the name
      --  of the formal for the object to be initialized, which is the task
      --  value record itself.

      Args := New_List;

      --  Size parameter. If no Storage_Size pragma is present, then
      --  the size is taken from the taskZ variable for the type, which
      --  is either Unspecified_Size, or has been reset by the use of
      --  a Storage_Size attribute definition clause. If a pragma is
      --  present, then the size is taken from the _Size field of the
      --  task value record, which was set from the pragma value.

      if Present (Tdef)
        and then Has_Storage_Size_Pragma (Tdef)
      then
         Append_To (Args,
           Make_Selected_Component (Loc,
             Prefix => Make_Identifier (Loc, Name_uInit),
             Selector_Name => Make_Identifier (Loc, Name_uSize)));
      else
         Append_To (Args,
           Make_Identifier (Loc, New_External_Name (Tnam, 'Z')));
      end if;

      --  Priority parameter. Set to Unspecified_Priority unless there is a
      --  priority pragma, in which case we take the value from the pragma.

      if Present (Tdef)
        and then Has_Priority_Pragma (Tdef)
      then
         Append_To (Args,
           Make_Selected_Component (Loc,
             Prefix => Make_Identifier (Loc, Name_uInit),
             Selector_Name => Make_Identifier (Loc, Name_uPriority)));
      else
         Append_To (Args,
           New_Reference_To (RTE (RE_Unspecified_Priority), Loc));
      end if;

      --  Number of entries. This is an expression of the form:
      --
      --    n + _Init.a'Length + _Init.a'B'Length + ...
      --
      --  where a,b... are the entry family names for the task definition

      Ent := First_Entity (Ttyp);
      Eindx := 0;

      --  Count number of non-family entries

      while Present (Ent) loop
         if Ekind (Ent) = E_Entry then
            Eindx := Eindx + 1;
         end if;

         Ent := Next_Entity (Ent);
      end loop;

      Ecount := Make_Integer_Literal (Loc, UI_From_Int (Eindx));

      --  Loop through entry families building the addition nodes

      Ent := First_Entity (Ttyp);
      while Present (Ent) loop
         if Ekind (Ent) = E_Entry_Family then
            Ecount :=
              Make_Op_Add (Loc,
                Left_Opnd  => Ecount,
                Right_Opnd =>
                  Make_Attribute_Reference (Loc,
                    Prefix =>
                      Make_Selected_Component (Loc,
                        Prefix => Make_Identifier (Loc, Name_uInit),
                        Selector_Name => Make_Identifier (Loc, Chars (Ent))),
                    Attribute_Name => Name_Length));
         end if;

         Ent := Next_Entity (Ent);
      end loop;

      Append_To (Args, Ecount);

      --  Master parameter. This is a reference to the _Master parameter of
      --  the initialization procedure.

      Append_To (Args, Make_Identifier (Loc, Name_uMaster));

      --  State parameter. This is a pointer to the task body procedure. We get
      --  the required value by taking the address of the task body procedure,
      --  and then converting it (with an unchecked conversion) to the type
      --  required by the task kernel. See description of Expand_Task_Body
      --  for further details.

      Append_To (Args,
        Make_Unchecked_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (RTE (RE_Init_State), Loc),
          Expression =>
            Make_Attribute_Reference (Loc,
              Prefix =>
                Make_Identifier (Loc, New_External_Name (Tnam, 'B')),
              Attribute_Name => Name_Address)));

      --  Discriminants parameter. This is just the address of the task
      --  value record itself (which contains the discriminant values

      Append_To (Args,
        Make_Attribute_Reference (Loc,
          Prefix => Make_Identifier (Loc, Name_uInit),
          Attribute_Name => Name_Address));

      --  Elaborated parameter. This is an access to the elaboration Boolean

      Append_To (Args,
        Make_Attribute_Reference (Loc,
          Prefix => Make_Identifier (Loc, New_External_Name (Tnam, 'E')),
          Attribute_Name => Name_Access));

      --  Chain parameter. This is a reference to the _Chain parameter of
      --  the initialization procedure.

      Append_To (Args, Make_Identifier (Loc, Name_uChain));

      --  Created_Task parameter. This is the _Task_Id field of the task
      --  record value

      Append_To (Args,
        Make_Selected_Component (Loc,
          Prefix => Make_Identifier (Loc, Name_uInit),
          Selector_Name => Make_Identifier (Loc, Name_uTask_Id)));

      return
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RE_Create_Task), Loc),
          Parameter_Associations => Args);

   end Make_Task_Create_Call;

   --------------
   -- Task_Ref --
   --------------

   --  The expression returned for a reference to a task has the form:

   --    taskV!(name)._Task_Id

   --  For the case of an access to a task, there is an extra explicit
   --  dereference:

   --    taskV!(name.all)._Task_Id

   --  here taskV is the type for the associated record, which
   --  contains the required _Task_Id field.

   --  For the case of a task type name, the expression is

   --    Self;

   --  i.e. a call to the Self function which returns precisely this Task_Id

   function Task_Ref (N : Node_Id) return Node_Id is
      Loc  : constant Source_Ptr := Sloc (N);
      Ntyp : constant Entity_Id  := Etype (N);

   begin
      if Is_Access_Type (Ntyp) then
         return
           Make_Selected_Component (Loc,
             Prefix =>
               Make_Unchecked_Type_Conversion (Loc,
                 Subtype_Mark => New_Reference_To (
                   Corresponding_Record_Type (Designated_Type (Ntyp)), Loc),
                 Expression => Make_Explicit_Dereference (Loc, N)),
             Selector_Name => Make_Identifier (Loc, Name_uTask_Id));

      elsif Is_Entity_Name (N)
        and then Is_Task_Type (Entity (N))
      then
         return
           Make_Function_Call (Loc,
             Name => New_Reference_To (RTE (RE_Self), Loc));

      else
         pragma Assert (Is_Task_Type (Ntyp));
         return
           Make_Selected_Component (Loc,
             Prefix =>
               Make_Unchecked_Type_Conversion (Loc,
                 Subtype_Mark =>
                   New_Reference_To (Corresponding_Record_Type (Ntyp), Loc),
                 Expression => New_Copy_Tree (N)),
             Selector_Name => Make_Identifier (Loc, Name_uTask_Id));
      end if;
   end Task_Ref;

end Exp_Ch9;
