------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             L I B . L O A D                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.29 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;  use Atree;
with Debug;  use Debug;
with Excep;  use Excep;
with Par;
with Scn;    use Scn;

separate (Lib)
function Load
  (Uname    : Unit_Name_Type;
   Required : Boolean;
   Enode    : Node_Id)
   return     Unit_Number_Type
is
   Calling_Unit : Unit_Number_Type;
   Unum         : Unit_Number_Type;
   Fname        : File_Name_Type := Get_File_Name (Uname);
   Src_Ind      : Source_File_Index;
   Lptr         : Lines_Table_Ptr;

begin
   if Debug_Flag_L then
      Write_Str ("*** Load request for unit: ");
      Write_Unit_Name (Uname);

      if Required then
         Write_Str (" (Required = True)");
      else
         Write_Str (" (Required = False)");
      end if;

      Write_Eol;
   end if;

   --  Capture error location if it is for the main unit. The idea is to post
   --  errors on the main unit location, not on the most recent invocation

   if Present (Enode)
     and then Get_Sloc_Unit_Number (Sloc (Enode)) = Main_Unit
   then
      Load_Msg_Sloc := Sloc (Enode);
   end if;

   --  If we are generating error messages, then capture calling unit

   if Present (Enode) then
      Calling_Unit := Get_Sloc_Unit_Number (Sloc (Enode));
   end if;

   --  See if we already have an entry for this unit

   Unum := Main_Unit;

   while Unum <= Units.Last loop
      exit when Uname = Units.Table (Unum).Unit_Name;
      Unum := Unum + 1;
   end loop;

   --  Whether or not the entry was found, Unum is now the right value, since
   --  it is one more than Units.Last (i.e. the index of the new entry we will
   --  create) in the not found case.

   --  A special check is necessary in the unit not found case. If the unit
   --  is not found, but the file in which it lives has already been loaded,
   --  then we have the problem that the file does not contain the unit that
   --  is needed. We simply treat this as a file not found condition.

   if Unum > Units.Last then
      for J in Units.First .. Units.Last loop
         if Fname = Units.Table (J).Unit_File_Name then
            if Debug_Flag_L then
               Write_Str ("*** File does not contain unit, Unit_Number = ");
               Write_Int (Int (Unum));
               Write_Eol;
            end if;

            if Present (Enode) then
               Error_Msg_Name_1 := Fname;
               Error_Msg_Unit_1 := Uname;
               Error_Msg ("~File{ does not contain unit$", Load_Msg_Sloc);
               Write_Dependency_Chain;
               raise Unrecoverable_Error;
            else
               return No_Unit;
            end if;
         end if;
      end loop;
   end if;

   --  If we are proceeding with load, then make load stack entry

   Load_Stack.Increment_Last;
   Load_Stack.Table (Load_Stack.Last) := Unum;

   --  Case of entry already in table

   if Unum <= Units.Last then

      --  Here is where we check for a circular dependency, which is
      --  an attempt to load a unit which is currently in the process
      --  of being loaded. We do *not* care about a circular chain that
      --  leads back to a body, because this kind of circular dependence
      --  legitimately occurs (e.g. two package bodies that contain
      --  inlined subprogram referenced by the other).

      if Units.Table (Unum).Loading
        and then Is_Spec_Name (Units.Table (Unum).Unit_Name)
      then
         if Debug_Flag_L then
            Write_Str ("*** Circular dependency encountered");
            Write_Eol;
         end if;

         if Present (Enode) then
            Error_Msg ("~Circular unit dependency", Load_Msg_Sloc);
            Write_Dependency_Chain;
            raise Unrecoverable_Error;
         else
            Load_Stack.Decrement_Last;
            return No_Unit;
         end if;
      end if;

      if Debug_Flag_L then
         Write_Str ("*** Unit already in file table, Unit_Number = ");
         Write_Int (Int (Unum));
         Write_Eol;
      end if;

      Load_Stack.Decrement_Last;
      return Unum;

   --  File is not already in table, so try to open it

   else
      Src_Ind := Load_Source_File (Fname);

      --  Make a partial entry in the file table, used even in the file not
      --  found case to print the dependency chain including the last entry

      Units.Increment_Last;
      Units.Table (Unum).Unit_Name := Uname;

      --  File was found

      if Src_Ind /= No_Source_File then
         if Debug_Flag_L then
            Write_Str ("*** Building new unit table entry, Unit_Number = ");
            Write_Int (Int (Unum));
            Write_Eol;
         end if;

         Units.Table (Unum).Unit_File_Name := Fname;
         Units.Table (Unum).Source_Index   := Src_Ind;
         Units.Table (Unum).Cunit          := Empty;
         Units.Table (Unum).Cunit_Entity   := Empty;
         Units.Table (Unum).Fatal_Error    := False;
         Units.Table (Unum).Generate_Code  := False;
         Units.Table (Unum).Main_Priority  := Default_Main_Priority;

         --  Parse the new unit

         Units.Table (Unum).Loading := True;
         Initialize_Scanner (Unum);
         Par;
         Units.Table (Unum).Loading := False;

         if Debug_Flag_L then
            Write_Str ("*** Load completed successfully, Unit_Number = ");
            Write_Int (Int (Unum));
            Write_Eol;
         end if;

         --  If loaded unit had a fatal error, then caller inherits it!

         if Units.Table (Unum).Fatal_Error
           and then Present (Enode)
         then
            Units.Table (Calling_Unit).Fatal_Error := True;
         end if;

         --  Remove load stack entry and return the entry in the file table

         Load_Stack.Decrement_Last;
         return Unum;

      --  Case of file not found

      else
         if Debug_Flag_L then
            Write_Str ("*** File was not found, Unit_Number = ");
            Write_Int (Int (Unum));
            Write_Eol;
         end if;

         --  Generate message if unit required

         if Required and then Present (Enode) then
            Error_Msg_Name_1 := Fname;
            Error_Msg ("~file{ not found", Load_Msg_Sloc);
            Write_Dependency_Chain;
            raise Unrecoverable_Error;

         --  If unit not required, remove load stack entry and also the junk
         --  file table entry, and then return No_Unit to indicate not found,

         else
            Load_Stack.Decrement_Last;
            Units.Decrement_Last;
            return No_Unit;
         end if;
      end if;
   end if;
end Load;
