------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ A T T R                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.64 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Exp_Ch9;  use Exp_Ch9;
with Exp_Util; use Exp_Util;
with Itypes;   use Itypes;
with Nmake;    use Nmake;
with Nlists;   use Nlists;
with Opt;      use Opt;
with Output;   use Output;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Eval; use Sem_Eval;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stand;    use Stand;
with Tbuild;   use Tbuild;
with Ttypes;   use Ttypes;
with Uintp;    use Uintp;

package body Exp_Attr is

   ----------------------------------
   -- Expand_N_Attribute_Reference --
   ----------------------------------

   procedure Expand_N_Attribute_Reference (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Typ   : constant Entity_Id  := Etype (N);
      Pref  : constant Node_Id    := Prefix (N);
      Exprs : constant List_Id    := Expressions (N);

   begin
      case Get_Attribute_Id (Attribute_Name (N)) is

      --------------
      -- Callable --
      --------------

      --  Transforms 'Callable attribute into a call to the Callable function.

      when Attribute_Callable => Callable :
      begin
         Rewrite_Substitute_Tree (N,
           Build_Call_With_Task (Pref, RTE (RE_Callable)));
         Analyze (N);
         Resolve (N, Standard_Boolean);
      end Callable;

      -----------
      -- Count --
      -----------

      --  Transforms 'Count attribute into a call to the Count function

      when Attribute_Count => Count :
      declare
         Entnam : Node_Id;
         Index  : Node_Id;
         Call   : Node_Id;

      begin
         if Nkind (Pref) = N_Indexed_Component then
            Entnam := Prefix (Pref);
            Index := First (Expressions (Pref));
         else
            Entnam := Pref;
            Index := Empty;
         end if;

         Call :=
           Make_Function_Call (Loc,
             Name => New_Reference_To (RTE (RE_Task_Count), Loc),
             Parameter_Associations => New_List (
               Entry_Index_Expression
                 (Entnam, Index, Empty, Scope (Entity (Entnam)))));

         --  The call returns type Natural but the context is universal integer
         --  so any integer type is allowed. The attribute was already resolved
         --  so its Etype is the required result type. If the base type of the
         --  context type is other than Standard.Integer we put in a conversion
         --  to the required type. This can be a normal typed conversion since
         --  both input and output types of the conversion are integer types

         if Base_Type (Typ) /= Standard_Integer then
            Rewrite_Substitute_Tree (N,
              Make_Type_Conversion (Loc,
                Subtype_Mark => New_Reference_To (Typ, Loc),
                Expression => Call));
         else
            Rewrite_Substitute_Tree (N, Call);
         end if;

         Analyze (N);
         Resolve (N, Typ);

      end Count;

      --------------
      -- Enum_Rep --
      --------------

      --  If the argument is static then we replace the attribute reference by
      --  the corresponding integer literal being careful to mark this literal
      --  as not potentially static (impl-defined attributes are non-static).

      --  Otherwise, if the enumeration type has the standard, representation,
      --  then the result is simply obtained by doing an unchecked conversion
      --  from the enumeration value to the corresponding integer type, and
      --  then further converting this integer result to the target type.

      --  If there is a non-standard representation, then the Enum_Pos_To_Rep
      --  table is used to obtain the correct value.

      when Attribute_Enum_Rep => Enum_Rep :
      declare
         Etyp : constant Entity_Id := Base_Type (Entity (Pref));
         Expr : constant Node_Id   := First (Exprs);

      begin
         --  Case 1, argument is static expression

         if Is_Static_Expression (Expr) then
            Replace_Substitute_Tree (N,
              Make_Integer_Literal
                (Loc, Enumeration_Rep (Expr_Value (Expr))));

         --  Case 2, enumeration type has standard representation

         elsif No (Enum_Pos_To_Rep (Etyp)) then
            Replace_Substitute_Tree (N,
              Make_Type_Conversion (Loc,
                Subtype_Mark => New_Reference_To (Typ, Loc),
                Expression =>
                  Make_Unchecked_Type_Conversion (Loc,
                    Subtype_Mark => Corresponding_Integer_Type (Etyp),
                    Expression => Expr)));

         --  Case 3, enumeration type has non-standard representation

         else
            Replace_Substitute_Tree (N,
              Make_Type_Conversion (Loc,
                Subtype_Mark => New_Reference_To (Typ, Loc),
                Expression =>
                  Make_Indexed_Component (Loc,
                    Prefix => New_Reference_To (Enum_Pos_To_Rep (Etyp), Loc),
                    Expressions => New_List (
                      Make_Unchecked_Type_Conversion (Loc,
                        Subtype_Mark => Corresponding_Integer_Type (Etyp),
                        Expression => Expr)))));
         end if;

         --  Analyze and resolve the result. Note that we need to explicitly
         --  set Potentially_Static to False, because at least in case 1, the
         --  resolve call will set the literal as potentially static.

         Analyze (N);
         Resolve (N, Typ);
         Set_Potentially_Static (N, False);
      end Enum_Rep;

      ----------
      -- Fore --
      ----------

      --  For the fixed-point type Typ:

      --    Typ'Fore

      --  expands into

      --    Result_Type (System.Fore (Long_Long_Float (Type'First)),
      --                              Long_Long_Float (Type'Last))

      --  Note that we know that the type is a non-static subtype, or Fore
      --  would have itself been computed dynamically in Eval_Attribute.

      when Attribute_Fore => Fore :
      declare
         Ptyp : constant Entity_Id := Etype (Pref);

      begin
         Replace_Substitute_Tree (N,
           Make_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Typ, Loc),
             Expression =>
               Make_Function_Call (Loc,
                 Name => New_Reference_To (RTE (RE_Fore), Loc),
                 Parameter_Associations => New_List (
                   Make_Type_Conversion (Loc,
                     Subtype_Mark =>
                       New_Reference_To (Standard_Long_Long_Float, Loc),
                     Expression =>
                       Make_Attribute_Reference (Loc,
                         Prefix => New_Reference_To (Ptyp, Loc),
                         Attribute_Name => Name_First)),
                   Make_Type_Conversion (Loc,
                     Subtype_Mark =>
                       New_Reference_To (Standard_Long_Long_Float, Loc),
                     Expression =>
                       Make_Attribute_Reference (Loc,
                         Prefix => New_Reference_To (Ptyp, Loc),
                         Attribute_Name => Name_Last))))));

         Analyze (N);
         Resolve (N, Typ);
      end Fore;

      -----------
      -- Image --
      -----------

      --  For types other than user defined enumeration types,
      --  typ'Image (Val) expands into:

      --  The name xx and type tp depend on the root type of Val. The
      --  argument pm is an extra type dependent parameter only used in
      --  some cases as follows:

      --    For types whose root type is Character
      --      xx = Character
      --      tv = Character (Val)

      --    For types whose root type is Boolean
      --      xx = Boolean
      --      tv = Boolean (Val)

      --    For signed integer types with size <= Integer'Size
      --      xx = Integer
      --      tv = Integer (Val)

      --    For other signed integer types
      --      xx = Long_Long_Integer
      --      tv = Long_Long_Integer (Val)

      --    For modular types with modulus <= System.Unsigned_Types.Unsigned
      --      xx = Unsigned
      --      tv = System.Unsigned_Types.Unsigned (Val)

      --    For other modular integer types
      --      xx = Long_Long_Unsigned
      --      tv = System.Unsigned_Types.Long_Long_Unsigned (Val)

      --    For types whose root type is Wide_Character
      --      xx = Wide_Character
      --      tv = Wide_Character (Val)
      --      pm = Wide_Character_Encoding_Method

      --    For floating-point types
      --      xx = Floating_Point
      --      tv = Long_Long_Float (Val)
      --      pm = typ'Digits

      --    For ordinary fixed-point types
      --      xx = Ordinary_Fixed_Point
      --      tv = Long_Long_Float (Val)
      --      pm = typ'Aft

      --    For decimal fixed-point types whose corresponding integer type,
      --    ctype, has size <= Integer'Size
      --      xx = Decimal
      --      tv = Integer (Corresponding_Integer_Type!(Val))
      --      pm = typ'Scale


      --    For decimal fixed-point types whose corresponding integer type,
      --    ctype, has size > Integer'Size, typ'Image (X) expands into
      --      xx = Long_Long_Decimal
      --      tv = Integer (Corresponding_Integer_Type!(Val))
      --      pm = typ'Scale

      --  For enumeration types other than those derived from types Boolean,
      --  Character, and Wide_Character in Standard, typ'Image (X) expands to:

      --    Table (Enum'Pos (X)).all

      --  where table is the special table declared in the front end and
      --  constructed by special code in Gigi.

      when Attribute_Image => Image :
      declare
         Ptyp    : constant Entity_Id := Entity (Pref);
         Rtyp    : constant Entity_Id := Root_Type (Ptyp);
         Expr    : constant Node_Id   := First (Exprs);
         Imid    : RE_Id;
         Tent    : Entity_Id;
         Ctyp    : Entity_Id;
         Arglist : List_Id;
         Snn     : Entity_Id;

      begin
         if Rtyp = Standard_Boolean then
            Imid := RE_Image_Boolean;
            Tent := Rtyp;

         elsif Rtyp = Standard_Character then
            Imid := RE_Image_Character;
            Tent := Rtyp;

         elsif Rtyp = Standard_Wide_Character then
            Imid := RE_Image_Wide_Character;
            Tent := Rtyp;

         elsif Is_Signed_Integer_Type (Rtyp) then
            if UI_Le (Esize (Rtyp), Esize (Standard_Integer)) then
               Imid := RE_Image_Integer;
               Tent := Standard_Integer;
            else
               Imid := RE_Image_Long_Long_Integer;
               Tent := Standard_Long_Long_Integer;
            end if;

         elsif Is_Modular_Integer_Type (Rtyp) then
            if UI_Le (Modulus (Rtyp), Modulus (RTE (RE_Unsigned))) then
               Imid := RE_Image_Unsigned;
               Tent := RTE (RE_Unsigned);
            else
               Imid := RE_Image_Long_Long_Unsigned;
               Tent := RTE (RE_Long_Long_Unsigned);
            end if;

         elsif Is_Decimal_Fixed_Point_Type (Rtyp) then
            Ctyp := Corresponding_Integer_Type (Rtyp);

            Replace_Substitute_Tree (Expr,
              Make_Unchecked_Type_Conversion (Loc,
                Subtype_Mark => New_Reference_To (Ctyp, Loc),
                Expression => Relocate_Node (Expr)));

            if UI_To_Int (Esize (Ctyp)) <= Standard_Integer_Size then
               Imid := RE_Image_Decimal;
               Tent := Standard_Integer;
            else
               Imid := RE_Image_Long_Long_Decimal;
               Tent := Standard_Long_Long_Integer;
            end if;

         elsif Is_Ordinary_Fixed_Point_Type (Rtyp) then
            Imid := RE_Image_Ordinary_Fixed_Point;
            Tent := Standard_Long_Long_Float;

         elsif Is_Floating_Point_Type (Rtyp) then
            Imid := RE_Image_Floating_Point;
            Tent := Standard_Long_Long_Float;

         --  Only other possibility is user defined enumeration type

         else
            Replace_Substitute_Tree (N,
              Make_Explicit_Dereference (Loc,
                Prefix =>
                  Make_Indexed_Component (Loc,
                    Prefix =>
                      New_Reference_To (Lit_Name_Table (Entity (Pref)), Loc),

                    Expressions => New_List (
                      Make_Attribute_Reference (Loc,
                        Prefix         => Pref,
                        Attribute_Name => Name_Pos,
                        Expressions    => Exprs)))));
            Analyze (N);
            Resolve (N, Standard_String);
            return;

         end if;

         --  If we fall through, we have one of the cases that is handled by
         --  calling one of the System.Img_xx routines.

         Snn := Make_Defining_Identifier (Loc, New_Internal_Name ('S'));

         Arglist := New_List (
           Make_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Tent, Loc),
             Expression   => Relocate_Node (Expr)),

           Make_Attribute_Reference (Loc,
             Prefix         => New_Reference_To (Snn, Loc),
             Attribute_Name => Name_Access));

         --  For floating-point types, append Digits argument

         if Is_Floating_Point_Type (Rtyp) then
            Append_To (Arglist,
              Make_Attribute_Reference (Loc,
                Prefix         => New_Reference_To (Ptyp, Loc),
                Attribute_Name => Name_Digits));

         --  For ordinary fixed-point types, append Aft parameter

         elsif Is_Ordinary_Fixed_Point_Type (Rtyp) then
            Append_To (Arglist,
              Make_Attribute_Reference (Loc,
                Prefix         => New_Reference_To (Ptyp, Loc),
                Attribute_Name => Name_Aft));

         --  For wide character, append encoding method

         elsif Rtyp = Standard_Wide_Character then
            Append_To (Arglist,
              Make_Integer_Literal (Loc,
                Intval =>
                  UI_From_Int (Int (Wide_Character_Encoding_Method))));

         --  For decimal, append Scale

         elsif Is_Decimal_Fixed_Point_Type (Rtyp) then
            Append_To (Arglist,
              Make_Attribute_Reference (Loc,
                Prefix => New_Reference_To (Ptyp, Loc),
                Attribute_Name => Name_Scale));
         end if;

         Replace_Substitute_Tree (N,
           Make_Expression_Actions (Loc,

             Actions => New_List (
               Make_Object_Declaration (Loc,
                 Defining_Identifier => Snn,
                 Aliased_Present => True,
                 Object_Definition =>
                   Make_Subtype_Indication (Loc,
                     Subtype_Mark => New_Reference_To (Standard_String, Loc),
                     Constraint =>
                       Make_Index_Or_Discriminant_Constraint (Loc,
                         Constraints => New_List (
                           Make_Range (Loc,
                             Low_Bound  => Make_Integer_Literal (Loc, Uint_1),
                             High_Bound =>
                               Make_Attribute_Reference (Loc,
                                 Prefix => New_Reference_To (Rtyp, Loc),
                                 Attribute_Name => Name_Width))))))),

             Expression =>
               Make_Slice (Loc,
                 Prefix         => New_Reference_To (Snn, Loc),
                 Discrete_Range =>
                   Make_Range (Loc,
                     Low_Bound  => Make_Integer_Literal (Loc, Uint_1),
                     High_Bound =>
                       Make_Function_Call (Loc,
                         Name => New_Reference_To (RTE (Imid), Loc),
                         Parameter_Associations => Arglist)))));

         Analyze (N);
         Resolve (N, Standard_String);
      end Image;

      ---------
      -- Img --
      ---------

      --  X'Img is expanded to typ'Image (X), where typ is the type of X

      when Attribute_Img => Img :
      begin
         Replace_Substitute_Tree (N,
           Make_Attribute_Reference (Loc,
             Prefix => New_Reference_To (Etype (Pref), Loc),
             Attribute_Name => Name_Image,
             Expressions => New_List (Relocate_Node (Pref))));

         Analyze (N);
         Resolve (N, Standard_String);
      end Img;

      ----------
      -- Pred --
      ----------

      --  For fixed-point types, expand typ'Pred (x) into

      --    x - small-value

      --  For other types, nothing to do (floating point Pred not implemented
      --  yet, already flagged by semantics, and other types handled by Gigi)

      when Attribute_Pred => Pred :
      declare
         Ptyp : constant Entity_Id := Etype (Pref);

      begin
         if Is_Fixed_Point_Type (Ptyp) then
            Replace_Substitute_Tree (N,
              Make_Op_Subtract (Loc,
                Left_Opnd  => Relocate_Node (First (Exprs)),
                Right_Opnd => Make_Real_Literal (Loc, Small_Value (Ptyp))));

            Analyze (N);
            Resolve (N, Typ);
         end if;
      end Pred;

      ----------
      -- Size --
      ----------

      --  Transforms X'Size into a call to the primitive operation _Size.
      --  for class-wide types.

      --  For other types, nothing to do, to be handled by Gigi

      when Attribute_Size => Size :
      declare
         Ptyp     : constant Entity_Id := Etype (Pref);
         New_Node : Node_Id;

      begin
         if Is_Class_Wide_Type (Ptyp) then
            New_Node :=
              Make_Function_Call (Loc,
                Name => New_Reference_To
                  (Find_Prim_Op (Ptyp, Name_uSize), Loc),
                Parameter_Associations => New_List (Pref));

            if Typ /= Universal_Integer then
               New_Node :=
                  Make_Type_Conversion (Loc,
                    Subtype_Mark => New_Reference_To (Typ, Loc),
                    Expression => New_Node);
            end if;

            Replace_Substitute_Tree (N, New_Node);
            Analyze (N);
            Resolve (N, Typ);
         end if;
      end Size;

      ------------------
      -- Storage_Size --
      ------------------

      --  The case of access types is currently unimplemented (and should have
      --  been caught prior to this point, so we simply ignore this case here)

      --  The case of a task type results in the attribute reference being
      --  replaced by the literal zero, which indicates that it is not in
      --  general sensible to apply Storage_Size to a task type, since the
      --  storage size may depend on a dynamic expression, or on discriminants.

      --  For the case of a task object, if there is no pragma Storage_Size,
      --  then we also return the literal zero, otherwise if there is a
      --  Storage_Size pragma, then we replace the attribute reference by
      --  the expression:

      --    Universal_Integer (taskV!(name)._Size)

      --  to get the Size field of the record object associated with the task

      when Attribute_Storage_Size => Storage_Size :
      declare
         Ptyp : constant Entity_Id := Etype (Pref);

      begin
         if Is_Access_Type (Ptyp) then
            null;

         --  Task cases

         else
            pragma Assert (Is_Task_Type (Ptyp));

            --  Case of task type

            if Is_Entity_Name (Pref) and then Is_Task_Type (Entity (Pref)) then
               Replace_Substitute_Tree (N,
                 Make_Integer_Literal (Loc, Uint_0));

            --  Case of task object

            else
               declare
                  Rtyp : constant Entity_Id :=
                    Corresponding_Record_Type (Ptyp);

               begin
                  --  Task object which has Storage_Size pragma

                  if Chars (Last_Entity (Rtyp)) = Name_uSize then

                     Replace_Substitute_Tree (N,
                       Make_Type_Conversion (Loc,
                         Subtype_Mark =>
                           New_Reference_To (Universal_Integer, Loc),
                         Expression =>
                           Make_Selected_Component (Loc,
                             Prefix =>
                               Make_Unchecked_Type_Conversion (Loc,
                                 Subtype_Mark =>
                                   New_Reference_To
                                     (Corresponding_Record_Type (Ptyp), Loc),
                                 Expression => New_Copy_Tree (Pref)),
                             Selector_Name =>
                               Make_Identifier (Loc, Name_uSize))));

                  --  Task object not having Storage_Size pragma

                  else
                     Replace_Substitute_Tree (N,
                       Make_Integer_Literal (Loc, Uint_0));
                  end if;
               end;
            end if;

            Analyze (N);
            Resolve (N, Universal_Integer);
         end if;

      end Storage_Size;

      ----------
      -- Succ --
      ----------

      --  For fixed-point types, expand typ'Succ (x) into

      --    x + small-value

      --  For other types, nothing to do (floating point Pred not implemented
      --  yet, already flagged by semantics, and other types handled by Gigi)

      when Attribute_Succ => Succ :
      declare
         Ptyp : constant Entity_Id := Etype (Pref);

      begin
         if Is_Fixed_Point_Type (Ptyp) then
            Replace_Substitute_Tree (N,
              Make_Op_Add (Loc,
                Left_Opnd  => Relocate_Node (First (Exprs)),
                Right_Opnd => Make_Real_Literal (Loc, Small_Value (Ptyp))));

            Analyze (N);
            Resolve (N, Typ);
         end if;
      end Succ;

      ---------
      -- Tag --
      ---------

      --  Transforms X'Tag into a direct reference to the tag of X

      when Attribute_Tag => Tag :
      declare
         Ttyp : Entity_Id;

      begin
         if Is_Entity_Name (Pref) and then Is_Type (Entity (Pref)) then
            Ttyp := Entity (Pref);
         else
            Ttyp := Etype (Pref);
         end if;

         Replace_Substitute_Tree (N,
           Make_Unchecked_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (RTE (RE_Tag), Loc),
             Expression => New_Reference_To (Access_Disp_Table (Ttyp), Loc)));

         Analyze (N);
         Resolve (N, RTE (RE_Tag));
      end Tag;

      ----------------
      -- Terminated --
      ----------------

      --  Transforms 'Terminated attribute into a call to Terminated function.

      when Attribute_Terminated => Terminated :
      begin
         Replace_Substitute_Tree (N,
           Build_Call_With_Task (Pref, RTE (RE_Terminated)));
         Analyze (N);
         Resolve (N, Standard_Boolean);
      end Terminated;

      -----------
      -- Value --
      -----------

      --  For scalar types derived from Boolean, Character and integer types
      --  in package Standard, typ'Value (X) expands into:

      --    typ (Value_xx (X))

      --  where

      --    For types whose root type is Character
      --      xx = Character

      --    For types whose root type is Boolean
      --      xx = Boolean

      --    For signed integer types with size <= Integer'Size
      --      xx = Integer

      --    For other signed integer types
      --      xx = Long_Long_Integer

      --    For modular types with modulus <= System.Unsigned_Types.Unsigned
      --      xx = Unsigned

      --    For other modular integer types
      --      xx = Long_Long_Unsigned

      --    For floating-point types and ordinary fixed-point types
      --      xx = Real

      --  For types derived from Wide_Character, typ'Value (X) expands into

      --    Value_Wide_Character (X, Wide_Character_Encoding_Method)

      --  For decimal types, where the corresponding integer type, ctype, has
      --  size <= Integer'Size, typ'Value (X) expands into

      --    typ!(ctype (Value_Decimal (X, typ'Scale)));

      --  For all other decimal types, typ'Value (X) expands into

      --    typ!(ctype (Value_Long_Long_Decimal (X, typ'Scale)))

      --  For enumeration types other than those derived from types Boolean,
      --  Character, and Wide_Character in Standard, typ'Value (X) expands to:

      --    T'Val (Value_Enumeration (Table'Address, T'Pos (T'Last), X))

      --  where Table is the table of access to string built for each
      --  enumeration type by Gigi (see description under documentation
      --  in Einfo for Lit_Name_Table). The Value_Enum procedure will
      --  search the table looking for X and return the position number
      --  in the table if found and then we will use that with the 'Val
      --  to return the actual enumeration value.

      when Attribute_Value => Value :
      declare
         Rtyp : constant Entity_Id  := Root_Type (Typ);
         Vid  : RE_Id;
         Args : List_Id := Exprs;
         Ctyp : Entity_Id;

      begin
         if Rtyp = Standard_Character then
            Vid := RE_Value_Character;

         elsif Rtyp = Standard_Boolean then
            Vid := RE_Value_Boolean;

         elsif Rtyp = Standard_Wide_Character then
            Vid := RE_Value_Wide_Character;
            Append_To (Args,
              Make_Integer_Literal (Loc,
                Intval =>
                  UI_From_Int (Int (Wide_Character_Encoding_Method))));

         elsif Rtyp = Standard_Short_Short_Integer
           or else Rtyp = Standard_Short_Integer
           or else Rtyp = Standard_Integer
         then
            Vid := RE_Value_Integer;

         elsif Is_Signed_Integer_Type (Rtyp) then
            Vid := RE_Value_Long_Long_Integer;

         elsif Is_Modular_Integer_Type (Rtyp) then
            if UI_Le (Modulus (Rtyp), Modulus (RTE (RE_Unsigned))) then
               Vid := RE_Value_Unsigned;
            else
               Vid := RE_Value_Long_Long_Unsigned;
            end if;

         elsif Is_Decimal_Fixed_Point_Type (Rtyp) then
            Ctyp := Corresponding_Integer_Type (Rtyp);

            if UI_To_Int (Esize (Ctyp)) <= Standard_Integer_Size then
               Vid := RE_Value_Decimal;
            else
               Vid := RE_Value_Long_Long_Decimal;
            end if;

            Append_To (Args,
              Make_Attribute_Reference (Loc,
                Prefix => New_Reference_To (Typ, Loc),
                Attribute_Name => Name_Scale));

            Replace_Substitute_Tree (N,
              Make_Unchecked_Type_Conversion (Loc,
                Subtype_Mark => New_Reference_To (Typ, Loc),
                Expression =>
                  Make_Type_Conversion (Loc,
                    Subtype_Mark => New_Reference_To (Ctyp, Loc),
                    Expression =>
                      Make_Function_Call (Loc,
                        Name => New_Reference_To (RTE (Vid), Loc),
                        Parameter_Associations => Args))));

            Analyze (N);
            Resolve (N, Typ);

         elsif Is_Real_Type (Rtyp) then
            Vid := RE_Value_Real;

         --  Only other possibility is user defined enumeration type

         else
            pragma Assert (Is_Enumeration_Type (Rtyp));

            Prepend_To (Args,
              Make_Attribute_Reference (Loc,
                Prefix => New_Reference_To (Typ, Loc),
                Attribute_Name => Name_Pos,
                Expressions => New_List (
                  Make_Attribute_Reference (Loc,
                    Prefix => New_Reference_To (Typ, Loc),
                    Attribute_Name => Name_Last))));

            Prepend_To (Args,
              Make_Attribute_Reference (Loc,
                Prefix =>
                  New_Reference_To (Lit_Name_Table (Typ), Loc),
                Attribute_Name => Name_Address));

            Replace_Substitute_Tree (N,
              Make_Attribute_Reference (Loc,
                Prefix => New_Reference_To (Typ, Loc),
                Attribute_Name => Name_Val,
                Expressions => New_List (
                  Make_Function_Call (Loc,
                    Name => New_Reference_To (RTE (RE_Value_Enumeration), Loc),
                    Parameter_Associations => Args))));

            Analyze (N);
            Resolve (N, Typ);
            return;
         end if;

         --  Fall through for all cases except user defined enumeration type
         --  and decimal types, with Vid set to the Id of the entity for the
         --  Value routine and Args set to the list of parameters for the call.

         Replace_Substitute_Tree (N,
           Make_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Typ, Loc),
             Expression =>
               Make_Function_Call (Loc,
                 Name => New_Reference_To (RTE (Vid), Loc),
                 Parameter_Associations => Args)));

         Analyze (N);
         Resolve (N, Typ);

      end Value;

      ----------------
      -- Wide_Image --
      ----------------

      --  We expand typ'Wide_Image (X) into

      --    String_To_Wide_String
      --      (typ'Image (X), Wide_Character_Encoding_Method)

      --  This works in all cases because String_To_Wide_String converts any
      --  wide character escape sequences resulting from the Image call to the
      --  proper Wide_Character equivalent

      --  not quite right for typ = Wide_Character ???

      when Attribute_Wide_Image => Wide_Image :
      begin
         Replace_Substitute_Tree (N,
           Make_Function_Call (Loc,
             Name => New_Reference_To (RTE (RE_String_To_Wide_String), Loc),
             Parameter_Associations => New_List (
               Make_Attribute_Reference (Loc,
                 Prefix         => Pref,
                 Attribute_Name => Name_Image,
                 Expressions    => Exprs),

               Make_Integer_Literal (Loc,
                 Intval =>
                   UI_From_Int (Int (Wide_Character_Encoding_Method))))));

         Analyze (N);
         Resolve (N, Standard_Wide_String);
      end Wide_Image;

      ----------------
      -- Wide_Value --
      ----------------

      --  We expand typ'Wide_Value (X) into

      --    typ'Value
      --      (Wide_String_To_String (X, Wide_Character_Encoding_Method))

      --  Wide_String_To_String is a runtime function that converts its wide
      --  string argument to String, converting any non-translatable characters
      --  into appropriate escape sequences. This preserves the required
      --  semantics of Wide_Value in all cases, and results in a very simple
      --  implementation approach.

      --  It's not quite right where typ = Wide_Character, because the encoding
      --  method may not cover the whole character type ???

      when Attribute_Wide_Value => Wide_Value :
      begin
         Replace_Substitute_Tree (N,
           Make_Attribute_Reference (Loc,
             Prefix         => Pref,
             Attribute_Name => Name_Value,

             Expressions    => New_List (
               Make_Function_Call (Loc,
                 Name =>
                   New_Reference_To (RTE (RE_Wide_String_To_String), Loc),
                 Parameter_Associations => Exprs),

               Make_Integer_Literal (Loc,
                 Intval =>
                   UI_From_Int (Int (Wide_Character_Encoding_Method))))));

         Analyze (N);
         Resolve (N, Typ);
      end Wide_Value;

      -----------
      -- Width --
      -----------

      --  For scalar types derived from Boolean, Character and integer types
      --  in package Standard. Note that the Width attribute is computed at
      --  compile time for all cases except those involving non-static
      --  subtypes. For such subtypes, typ'Width expands into

      --    Result_Type (xx (yy (Ptyp'First), yy (Ptyp'Last)))

      --  where

      --    For types whose root type is Character
      --      xx = Width_Character
      --      yy = Character

      --    For types whose root type is Boolean
      --      xx = Width_Boolean
      --      yy = Boolean

      --    For signed integer types
      --      xx = Width_Long_Long_Integer
      --      yy = Long_Long_Integer

      --    For modular integer types
      --      xx = Width_Long_Long_Unsigned
      --      yy = Long_Long_Unsigned

      --  For types derived from Wide_Character, typ'Width expands into

      --    Result_Type (Width_Wide_Character (
      --      Wide_Character (typ'First),
      --      Wide_Character (typ'Last),
      --      Wide_Character_Encoding_Method

      --  For real types, typ'Width expands into

      --    if Ptyp'First > Ptyp'Last then 0 else btyp'Width end if

      --  where btyp is the base type. This looks recursive but it isn't
      --  because the base type is always static, and hence the expression
      --  in the else is reduced to an integer literal.

      --  For user defined enumeration types, typ'Width expands into

      --    Result_Type (Width_Enumeration (Table'Address,
      --                                    typ'Pos (typ'First),
      --                                    typ'Pos (Typ'Last)));

      when Attribute_Width => Width :
      declare
         Ptyp    : constant Entity_Id := Etype (Pref);
         Rtyp    : constant Entity_Id := Root_Type (Ptyp);
         XX      : Entity_Id;
         YY      : Entity_Id;
         Arglist : List_Id;

      begin
         --  Types derived from Standard.Boolean

         if Rtyp = Standard_Boolean then
            XX := RTE (RE_Width_Boolean);
            YY := Rtyp;

         --  Types derived from Standard.Character

         elsif Rtyp = Standard_Character then
            XX := RTE (RE_Width_Character);
            YY := Rtyp;

         --  Types derived from Standard.Wide_Character

         elsif Rtyp = Standard_Wide_Character then
            XX := RTE (RE_Width_Wide_Character);
            YY := Rtyp;

         --  Signed integer types

         elsif Is_Signed_Integer_Type (Rtyp) then
            XX := RTE (RE_Width_Long_Long_Integer);
            YY := Standard_Long_Long_Integer;

         --  Modular integer types

         elsif Is_Modular_Integer_Type (Rtyp) then
            XX := RTE (RE_Width_Long_Long_Unsigned);
            YY := RTE (RE_Long_Long_Unsigned);

         --  Real types

         elsif Is_Real_Type (Rtyp) then

            Replace_Substitute_Tree (N,
              Make_Conditional_Expression (Loc,
                Expressions => New_List (

                  Make_Op_Gt (Loc,
                    Left_Opnd =>
                      Make_Attribute_Reference (Loc,
                        Prefix => New_Reference_To (Ptyp, Loc),
                        Attribute_Name => Name_First),
                    Right_Opnd =>
                      Make_Attribute_Reference (Loc,
                        Prefix => New_Reference_To (Ptyp, Loc),
                        Attribute_Name => Name_Last)),

                  Make_Integer_Literal (Loc, Uint_0),

                  Make_Attribute_Reference (Loc,
                    Prefix => New_Reference_To (Base_Type (Ptyp), Loc),
                    Attribute_Name => Name_Width))));

            Analyze (N);
            Resolve (N, Typ);
            return;

         --  User defined enumeration types

         else
            pragma Assert (Is_Enumeration_Type (Rtyp));

            Replace_Substitute_Tree (N,
              Make_Type_Conversion (Loc,
                Subtype_Mark => New_Reference_To (Typ, Loc),
                Expression =>
                  Make_Function_Call (Loc,
                    Name =>
                      New_Reference_To (RTE (RE_Width_Enumeration), Loc),

                    Parameter_Associations => New_List (

                      Make_Attribute_Reference (Loc,
                        Prefix =>
                          New_Reference_To (Lit_Name_Table (Ptyp), Loc),
                        Attribute_Name => Name_Address),

                      Make_Attribute_Reference (Loc,
                        Prefix => New_Reference_To (Ptyp, Loc),
                        Attribute_Name => Name_Pos,

                        Expressions => New_List (
                          Make_Attribute_Reference (Loc,
                            Prefix => New_Reference_To (Ptyp, Loc),
                            Attribute_Name => Name_First))),

                      Make_Attribute_Reference (Loc,
                        Prefix => New_Reference_To (Ptyp, Loc),
                        Attribute_Name => Name_Pos,

                        Expressions => New_List (
                          Make_Attribute_Reference (Loc,
                            Prefix => New_Reference_To (Ptyp, Loc),
                            Attribute_Name => Name_Last)))))));

            Analyze (N);
            Resolve (N, Typ);
            return;
         end if;

         --  If we fall through XX and YY are set

         Arglist := New_List (
           Make_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (YY, Loc),
             Expression =>
               Make_Attribute_Reference (Loc,
                 Prefix => New_Reference_To (Ptyp, Loc),
                 Attribute_Name => Name_First)),

           Make_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (YY, Loc),
             Expression =>
               Make_Attribute_Reference (Loc,
                 Prefix => New_Reference_To (Ptyp, Loc),
                 Attribute_Name => Name_Last)));

         --  For Wide_Character, add encoding method parameter

         if Rtyp = Standard_Wide_Character then
            Append_To (Arglist,
              Make_Integer_Literal (Loc,
                Intval =>
                  UI_From_Int (Int (Wide_Character_Encoding_Method))));
         end if;

         Rewrite_Substitute_Tree (N,
           Make_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Typ, Loc),
             Expression =>
               Make_Function_Call (Loc,
                 Name => New_Reference_To (XX, Loc),
                 Parameter_Associations => Arglist)));

         Analyze (N);
         Resolve (N, Typ);
      end Width;

      --  The following attributes are handled by Gigi (in some cases only
      --  certain cases go to Gigi, e.g. the dynamic cases, where the static
      --  cases have been successfully handled already by the analyzer)

      when Attribute_Access                       |
           Attribute_Address                      |
           Attribute_Adjacent                     |
           Attribute_Aft                          |
           Attribute_Alignment                    |
           Attribute_Bit_Order                    |
           Attribute_Ceiling                      |
           Attribute_Component_Size               |
           Attribute_Compose                      |
           Attribute_Constrained                  |
           Attribute_Copy_Sign                    |
           Attribute_Definite                     |
           Attribute_Exponent                     |
           Attribute_First                        |
           Attribute_First_Bit                    |
           Attribute_Floor                        |
           Attribute_Fraction                     |
           Attribute_Last                         |
           Attribute_Last_Bit                     |
           Attribute_Leading_Part                 |
           Attribute_Length                       |
           Attribute_Machine                      |
           Attribute_Max                          |
           Attribute_Max_Size_In_Storage_Elements |
           Attribute_Min                          |
           Attribute_Model                        |
           Attribute_Passed_By_Reference          |
           Attribute_Pos                          |
           Attribute_Position                     |
           Attribute_Round                        |
           Attribute_Rounding                     |
           Attribute_Scale                        |
           Attribute_Scaling                      |
           Attribute_Truncation                   |
           Attribute_Unchecked_Access             |
           Attribute_Unrestricted_Access          |
           Attribute_Val                          =>

         null;

      --  The following attributes should not appear at this stage, since they
      --  have already been handled by the analyzer (and properly rewritten
      --  with corresponding values or entities to represent the right values)

      when Attribute_Abort_Signal                 |
           Attribute_Address_Size                 |
           Attribute_Base                         |
           Attribute_Body_Version                 |
           Attribute_Caller                       |
           Attribute_Class                        |
           Attribute_Delta                        |
           Attribute_Denorm                       |
           Attribute_Digits                       |
           Attribute_Emax                         |
           Attribute_Epsilon                      |
           Attribute_External_Tag                 |
           Attribute_Huge_Integer                 |
           Attribute_Identity                     |
           Attribute_Input                        |
           Attribute_Large                        |
           Attribute_Machine_Emax                 |
           Attribute_Machine_Emin                 |
           Attribute_Machine_Mantissa             |
           Attribute_Machine_Overflows            |
           Attribute_Machine_Radix                |
           Attribute_Machine_Rounds               |
           Attribute_Mantissa                     |
           Attribute_Max_Interrupt_Priority       |
           Attribute_Max_Priority                 |
           Attribute_Model_Emin                   |
           Attribute_Model_Epsilon                |
           Attribute_Model_Mantissa               |
           Attribute_Model_Small                  |
           Attribute_Modulus                      |
           Attribute_Output                       |
           Attribute_Partition_Id                 |
           Attribute_Range                        |
           Attribute_Read                         |
           Attribute_Remainder                    |
           Attribute_Safe_Emax                    |
           Attribute_Safe_First                   |
           Attribute_Safe_Large                   |
           Attribute_Safe_Last                    |
           Attribute_Safe_Small                   |
           Attribute_Signed_Zeros                 |
           Attribute_Small                        |
           Attribute_Storage_Pool                 |
           Attribute_Storage_Unit                 |
           Attribute_Tick                         |
           Attribute_Unbiased_Rounding            |
           Attribute_Universal_Literal_String     |
           Attribute_Valid                        |
           Attribute_Version                      |
           Attribute_Word_Size                    |
           Attribute_Write                        =>

         pragma Assert (False); null;

      end case;

   end Expand_N_Attribute_Reference;

end Exp_Attr;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.62
--  date: Tue Aug  9 07:29:30 1994;  author: dewar
--  (Img): Implement this new attribute
--  (Width): Fix bomb for user defined enumeration type case
--  ----------------------------
--  revision 1.63
--  date: Wed Aug 10 14:24:40 1994;  author: dewar
--  (Image): New more efficient calling sequence for image functions
--  (Wide_Value): Change name Wide_Value function to Wide_String_To_String
--  (Wide_Image): Change name Wide_Image function to String_To_WIde_String
--  ----------------------------
--  revision 1.64
--  date: Sun Aug 14 07:12:34 1994;  author: dewar
--  Minor reformatting and reorganization
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
