------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ D I S P                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.45 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Debug;    use Debug;
with Elists;   use Elists;
with Einfo;    use Einfo;
with Exp_Ch6;  use Exp_Ch6;
with Errout;   use Errout;
with Nlists;   use Nlists;
with Output;   use Output;
with Sem_Util; use Sem_Util;
with Snames;   use Snames;
with Sinfo;    use Sinfo;

package body Sem_Disp is

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Override_Dispatching_Operation
     (Tagged_Type : Entity_Id;
      Prev_Op     : Entity_Id;
      New_Op      : Entity_Id);
   --  Replace an implicit dispatching operation with an  explicit one.
   --  Prev_Op is an inherited primitive operation which is overriden by
   --  the explicit declaration of New_Op.

   procedure Add_Dispatching_Operation
     (Tagged_Type : Entity_Id;
      New_Op      : Entity_Id);
   --  Add New_Op in the list of primitive operations of Tagged_Type

   procedure Warning_On_Freezing_Point (T : Entity_Id);
   --  Search the declarative part after the definition of T in order to
   --  find its Freeze_Node. Print a warning using the Sloc of this node

   ---------------------------
   -- Find_Dispatching_Type --
   ---------------------------

   function Find_Dispatching_Type (Subp : Entity_Id) return Entity_Id is
      Formal      : Entity_Id := First_Formal (Subp);
      Tagged_Type : Entity_Id;
      Tagged_Seen : Entity_Id := Empty;

      procedure Check_Controlling_Type (T : Entity_Id) is
      begin
         if Is_Tagged_Type (T) then
            Tagged_Type := T;

         elsif Ekind (T) = E_Anonymous_Access_Type
           and then Is_Tagged_Type (Designated_Type (T))
           and then Ekind (Designated_Type (T)) /= E_Incomplete_Type
         then
            Tagged_Type := Designated_Type (T);
         else
            Tagged_Type := Empty;
         end if;

         if Present (Tagged_Type)
             and then not Is_Class_Wide_Type (Tagged_Type)
             and then Scope (Subp) = Scope (Tagged_Type)
         then
            if Present (Tagged_Seen) and then Tagged_Type /= Tagged_Seen then
               Error_Msg_N
                 ("operation can be dispatching in only one type", Subp);
            else
               Tagged_Seen := Tagged_Type;
            end if;
         end if;
      end Check_Controlling_Type;

   --  Start of processing Find_Dispatching_Type

   begin
      while Present (Formal) loop
         Check_Controlling_Type (Etype (Formal));
         Formal := Next_Formal (Formal);
      end loop;

      --  The subprogram may also be dispatching on result

      if Present (Etype (Subp)) then
         Check_Controlling_Type (Etype (Subp));
      end if;

      return Tagged_Seen;
   end Find_Dispatching_Type;

   ----------------------------------
   -- Check_Dispatching_Operation  --
   ----------------------------------

   procedure Check_Dispatching_Operation (Subp, Old_Subp : Entity_Id) is
      Tagged_Seen : constant Entity_Id := Find_Dispatching_Type (Subp);
      N           : Node_Id;

   begin
      Set_Is_Dispatching_Operation (Subp, False);

      if No (Tagged_Seen) then
         return;

      --  The subprograms build internally after the freezing point (such as
      --  the Init procedure) are not primitives

      elsif Is_Frozen (Tagged_Seen) and then not Comes_From_Source (Subp) then
         return;

      --  If the type is not defined in a package spec, that's a potential
      --  error in most cases we, at least, need a warning unless we are just
      --  overriding the current primitive operations

      elsif (Ekind (Scope (Subp)) /= E_Package
               and then Ekind (Scope (Subp)) /= E_Generic_Package)
        or else In_Package_Body (Scope (Subp))
      then
         if not Comes_From_Source (Subp)
            or else (Present (Old_Subp) and then not Is_Frozen (Tagged_Seen))
         then
            null;

         elsif Present (Old_Subp) then
            Error_Msg_N ("overriding is too late (subprogram spec"
              & " should appear immediately after the type)", Subp);

         else
            Error_Msg_N ("?not a dispatching operation"
              & "(must be defined in a package spec)", Subp);
            return;
         end if;

      --  Now, we are sure that the scope is a package spec. If the subprogram
      --  is declared after the freezing point ot the type that's an error

      elsif Is_Frozen (Tagged_Seen) then
         Error_Msg_N ("this primitive operation is declared too late", Subp);
         Warning_On_Freezing_Point (Tagged_Seen);
         return;
      end if;

      --  Now it should be a correct primitive operation, put it in the list

      if Present (Old_Subp) then
         Override_Dispatching_Operation (Tagged_Seen, Old_Subp, Subp);
      else
         Add_Dispatching_Operation (Tagged_Seen, Subp);
      end if;

      Set_Is_Dispatching_Operation (Subp, True);
   end Check_Dispatching_Operation;

   --------------------------------
   --  Add_Dispatching_Operation --
   --------------------------------

   procedure Add_Dispatching_Operation
     (Tagged_Type : Entity_Id;
      New_Op      : Entity_Id)
   is
      List : constant Elist_Id := Primitive_Operations (Tagged_Type);

   begin

      --  _Size if always first in the dispatch table

      if Chars (New_Op) = Name_uSize then
         Prepend_Elmt (New_Op, List);

      --  Equality if always second in the dispatch table

      elsif Chars (New_Op) = Name_Op_Eq
        and then not Is_Empty_Elmt_List (List)
      then
         Insert_Elmt_After (New_Op, First_Elmt (List));

      --  Otherwise primitives are in the order of appearence

      else
         Append_Elmt (New_Op, List);
      end if;
   end Add_Dispatching_Operation;

   ---------------------------------------
   -- Check_Operation_From_Private_View --
   ---------------------------------------

   procedure Check_Operation_From_Private_View (Subp, Old_Subp : Entity_Id) is
      Tagged_Type : Entity_Id;
   begin

      if Is_Dispatching_Operation (Alias (Subp))
        and then Is_Dispatching_Operation (Old_Subp)
      then
         Set_Scope (Subp, Current_Scope);
         Tagged_Type := Find_Dispatching_Type (Subp);
         Append_Elmt (Old_Subp, Primitive_Operations (Tagged_Type));
         if Present (Alias (Old_Subp)) then
            Set_Alias (Old_Subp, Alias (Subp));
         end if;
      end if;
   end Check_Operation_From_Private_View;

   ----------------------------
   -- Check_Dispatching_Call --
   ----------------------------

   procedure Check_Dispatching_Call (N : Node_Id) is
      Actual  : Node_Id;
      Control : Node_Id := Empty;

   begin
      --  Find a controlling argument, if any

      if Present (Parameter_Associations (N)) then
         Actual := First_Actual (N);

         while Present (Actual) loop
            Control := Find_Controlling_Arg (Actual);
            exit when Present (Control);
            Actual := Next_Actual (Actual);
         end loop;

         if Present (Control) then

            --  Verify that no controlling arguments are statically tagged

            if Debug_Flag_E then
               Write_Str ("Found Dispatching call");
               Write_Int (Int (N));
               Write_Eol;
            end if;

            Actual := First_Actual (N);

            while Present (Actual) loop
               if Actual /= Control then

                  if not Is_Tagged_Type (Etype (Actual)) then
                     null; -- can be anything

                  elsif (Is_Dynamically_Tagged (Actual)) then
                     null; --  valid parameter

                  elsif Is_Tag_Indeterminate (Actual) then

                     --  The tag is inherited from the enclosing call (the
                     --  node we are currently analyzing). Explicitly expand
                     --  the actual, since the previous call to Expand
                     --  (from Resolve_Call) had no way of knowing about
                     --  the required dispatching.

                     Propagate_Tag (Control, Actual);

                  elsif Is_Controlling_Actual (Actual) then
                     Error_Msg_N
                       ("controlling arguments is not dynamically tagged",
                        Actual);
                     return;
                  end if;
               end if;

               Actual := Next_Actual (Actual);
            end loop;

            --  Mark call as a dispatching call

            Set_Controlling_Argument (N, Control);
         end if;

      else
         --  If dispatching on result, the enclosing call, if any, will
         --  determine the controlling argument. Otherwise this is the
         --  primitive operation of the root type.

         null;
      end if;
   end Check_Dispatching_Call;

   ---------------------------
   -- Is_Dynamically_Tagged --
   ---------------------------

   function Is_Dynamically_Tagged (N : Node_Id) return Boolean is
   begin
      return Find_Controlling_Arg (N) /= Empty;
   end Is_Dynamically_Tagged;

   --------------------------
   -- Find_Controlling_Arg --
   --------------------------

   function Find_Controlling_Arg (N : Node_Id) return Node_Id is
      Orig_Node : constant Node_Id := Original_Node (N);
      Typ       : Entity_Id;

   begin
      if Is_Controlling_Actual (N) then
         return N;

      --  Dispatching on result

      elsif Nkind (Orig_Node) = N_Function_Call
        and then Present (Controlling_Argument (Orig_Node))
      then
         return Controlling_Argument (Orig_Node);

      else
         return Empty;
      end if;
   end Find_Controlling_Arg;

   --------------------------
   -- Is_Tag_Indeterminate --
   --------------------------

   function Is_Tag_Indeterminate (N : Node_Id) return Boolean is
      Nam       : Entity_Id;
      Actual    : Node_Id;
      Orig_Node : constant Node_Id := Original_Node (N);

   begin
      if Nkind (Orig_Node) = N_Function_Call then
         Nam := Entity (Name (Orig_Node));

         if Present (Parameter_Associations (Orig_Node)) then
            Actual := First_Actual (Orig_Node);

            while Present (Actual) loop
               if Is_Tagged_Type (Etype (Actual))
                 and then Is_Dynamically_Tagged (Actual)
               then
                  return False; -- one operand is dispatching
               end if;

               Actual := Next_Actual (Actual);
            end loop;

            return True;

         --  If there are no actuals, the call is tag-indeterminate

         else
            return True;
         end if;

      elsif Nkind (Orig_Node) = N_Qualified_Expression then
         return Is_Tag_Indeterminate (Expression (Orig_Node));

      else
         return False;
      end if;
   end Is_Tag_Indeterminate;

   ------------------------------------
   -- Override_Dispatching_Operation --
   ------------------------------------

   procedure Override_Dispatching_Operation
     (Tagged_Type : Entity_Id;
      Prev_Op     : Entity_Id;
      New_Op      : Entity_Id)
   is
      Op_Elmt : Elmt_Id := First_Elmt (Primitive_Operations (Tagged_Type));

   begin
      while Node (Op_Elmt) /= Prev_Op loop
         Op_Elmt := Next_Elmt (Op_Elmt);
      end loop;
      Replace_Elmt (Op_Elmt, New_Op);
   end Override_Dispatching_Operation;

   -------------------
   -- Propagate_Tag --
   -------------------

   procedure Propagate_Tag (Control : Node_Id; Actual : Node_Id) is
      Call_Node : Node_Id;
      Arg       : Node_Id;

   begin
      if Nkind (Actual) = N_Function_Call then
         Call_Node := Actual;

      --  Only other possibility is parenthesized or qualified expression

      else
         Call_Node := Expression (Actual);
      end if;

      Set_Controlling_Argument (Call_Node, Control);
      Arg := First_Actual (Call_Node);

      while Present (Arg) loop
         if Is_Tag_Indeterminate (Arg) then
            Propagate_Tag (Control,  Arg);
         end if;

         Arg := Next_Actual (Arg);
      end loop;

      Expand_Dispatch_Call (Call_Node);
   end Propagate_Tag;

   procedure Warning_On_Freezing_Point (T : Entity_Id) is
      N : Node_Id := Parent (T);

   begin
      --  Look for the Freeze Node for the tagged type

      while Present (N)
        and then (Nkind (N) /= N_Freeze_Entity or else Entity (N) /= T)
      loop
         N := Next (N);
      end loop;

      if Present (N) then
         Error_Msg_NE ("?no primitive operations for & after this line", N, T);
      end if;
   end Warning_On_Freezing_Point;

end Sem_Disp;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.43
--  date: Fri Jul 29 13:19:36 1994;  author: comar
--  (Check_Operation_From_Private_View): New Function.
--  (Check_Dispatching_Operation): force predefined primitives at the beginning
--   of the Dospatching operation lists
--   ??? new code commented out because of problems with new Elist functions
--  ----------------------------
--  revision 1.44
--  date: Sun Aug 28 21:32:30 1994;  author: comar
--  (Warning_On_Freezing_Point): new subprogram introduced for clarity
--  (Override_Dispatching_Operation) is now local and don't remove anymore
--   the last element of the primitive list.
--  (Add_Dispatching_Operation): new subprogram introduced for clarity
--  (Check_Dispatching_Operation): do not always add the dispatching op. at
--   the end. _size is always first "=" is always second and overridding is
--   done directly.
--  ----------------------------
--  revision 1.45
--  date: Mon Aug 29 23:43:08 1994;  author: dewar
--  Minor reformatting
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
