------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               S P R I N T                                --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.17 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package (source print) contains routines for printing the source
--  program corresponding to a specified syntax tree. These routines are
--  intended for debugging use in the compiler (not as a user level pretty
--  print tool). Only information present in the tree is output (e.g. no
--  comments are present in the output), and as far as possible we avoid
--  making any assumptions about the correctness of the tree, so a bad
--  tree may either blow up on a debugging check, or list incorrect source.

with Types; use Types;
package Sprint is

   -----------------------
   -- Syntax Extensions --
   -----------------------

   --  When the generated tree is printed, it contains constructs that are not
   --  pure Ada. For convenience, syntactic extensions to Ada have been defined
   --  purely for the purposes of this printout (they are not recognized by the
   --  parser)

   --    Allocator                    new xxx [storage_pool = xxx]
   --    Cleanup action               at end procedure name;
   --    Conditional expression       (if expr then expr else expr)
   --    Conversion wi Float_Truncate target^(source)
   --    Expression actions           [action; action; ...; value]
   --    Free statement               free expression
   --    Freeze action                freeze typename
   --    Implicit type                implicit type xxx is yyy
   --    Intrinsic calls              function-name!(arg, arg, arg)
   --    Label declaration            labelname : label
   --    Multiple concatenation       expr && expr && expr ... && expr
   --    Raise constraint error       [constraint_error]
   --    Rational literal             [numerator/denominator]
   --    Reference                    expression'reference
   --    Shift nodes                  shift_name!(expr, count)
   --    Unchecked conversion         target_type!(source_expression)

   --  Note: the storage_pool parameters for allocators and the free node
   --  are omitted if the Storage_Pool field is Empty, indicating use of
   --  the standard default pool.

   -----------------
   -- Subprograms --
   -----------------

   procedure Source_Dump;
   --  This routine is called from the GNAT main program to dump source as
   --  requested by debug options. The relevant debug options are:
   --    -ds  print source from tree, both original and generated code
   --    -dg  print source from tree, including only the generated code
   --    -do  print source from tree, including only the original code
   --    -df  modify the above to include all units, not just the main unit
   --    -sz  print source from tree for package Standard

   procedure Sprint_Comma_List (List : List_Id);
   --  Prints the nodes in a list, with separating commas. If the list
   --  is empty then no output is generated.

   procedure Sprint_Paren_Comma_List (List : List_Id);
   --  Prints the nodes in a list, surrounded by parentheses, and separated
   --  by comas. If the list is empty, then no output is generated. A blank
   --  is output before the initial left parenthesis.

   procedure Sprint_Opt_Paren_Comma_List (List : List_Id);
   --  Same as normal Sprint_Paren_Comma_List procedure, except that
   --  an extra blank is output if List is non-empty, and nothing at all is
   --  printed it the argument is No_List.

   procedure Sprint_Node_List (List : List_Id);
   --  Prints the nodes in a list with no separating characters. This is used
   --  in the case of lists of items which are printed on separate lines using
   --  the current indentation amount. Note that Sprint_Node_List itself
   --  does not generate any New_Line calls.

   procedure Sprint_Opt_Node_List (List : List_Id);
   --  Like Sprint_Node_List, but prints nothing if List = No_List.

   procedure Sprint_Indented_List (List : List_Id);
   --  Like Sprint_Line_List, except that the indentation level is
   --  increased before outputting the list of items, and then decremented
   --  (back to its original level) before returning to the caller.

   procedure Sprint_Node (Node : Node_Id);
   --  Prints a single node. No new lines are output, except as required for
   --  splitting lines that are too long to fit on a single physical line.
   --  No output is generated at all if Node is Empty. No trailing or leading
   --  blank characters are generated.

   procedure Sprint_Opt_Node (Node : Node_Id);
   --  Same as normal Sprint_Node procedure, except that one leading
   --  blank is output before the node if it is non-empty.

end Sprint;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.15
--  date: Wed May 11 14:52:31 1994;  author: dewar
--  Add comment for syntax used for shift node
--  ----------------------------
--  revision 1.16
--  date: Fri Jun 24 17:12:57 1994;  author: dewar
--  Add syntax note for N_Raise_Constraint_Error node
--  ----------------------------
--  revision 1.17
--  date: Sun Aug 14 07:15:46 1994;  author: dewar
--  Document sprint syntax for Float_Truncate conversions
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
