------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               U R E A L P                                --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                             $Revision: 1.12 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Alloc;  use Alloc;
with Nmake;  use Nmake;
with Output; use Output;
with Sinfo;  use Sinfo;

package body Urealp is

   type Ureal_Entry is record
      Num  : Uint;
      --  Numerator

      Den  : Uint;
      --  Denominator

      Rbase : Nat;
      --  Base value. If Rbase is zero, then the value is simply Num / Den.
      --  If Rbase is non-zero, then the value is Num / (Rbase ** Den)
   end record;

   package Ureals is new Table (
     Table_Component_Type => Ureal_Entry,
     Table_Index_Type     => Ureal,
     Table_Low_Bound      => Ureal_First_Entry,
--   Table_Initial        => Alloc_Ureals_Initial,
--   Table_Increment      => Alloc_Ureals_Increment,
     Table_Initial        => 50_000,
     Table_Increment      => 0,
--   For now, till problem of use in Gigi is solved ???
     Table_Name           => "Ureals");

   --  The following universal reals are the constant values returned by the
   --  function Ureal_0, Ureal_Half, Ureal_1, Ureal_2, Ureal_10. They are
   --  initialized by the initialization procedure.

   UR_0     : Ureal;
   UR_Tenth : Ureal;
   UR_Half  : Ureal;
   UR_1     : Ureal;
   UR_2     : Ureal;
   UR_10    : Ureal;

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Is_Integer (Num, Den : Uint) return boolean;
   --  Return true if the real quotient of Num / Den is an integer value

   function Normalize (Val : Ureal_Entry) return Ureal_Entry;
   --  Normalizes the Ureal_Entry by reducing it to lowest terms (with a
   --  base value of 0).

   function Store_Ureal (Val : Ureal_Entry) return Ureal;
   --  This store a new entry in the universal reals table and return
   --  its index in the table.

   -----------------
   -- Denominator --
   -----------------

   function Denominator (Real : Ureal) return Uint is
      Val : Ureal_Entry := Ureals.Table (Real);

   begin
      return Val.Den;
   end Denominator;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize is
   begin
      Ureals.Init;
      UR_0     := UR_From_Components (Uint_0, Uint_1);
      UR_Tenth := UR_From_Components (Uint_1, Uint_1, 10);
      UR_Half  := UR_From_Components (Uint_1, Uint_1, 2);
      UR_1     := UR_From_Components (Uint_1, Uint_1);
      UR_2     := UR_From_Components (Uint_1, Uint_Minus_1, 2);
      UR_10    := UR_From_Components (Uint_1, Uint_Minus_1, 10);
   end Initialize;

   ----------------
   -- Is_Integer --
   ----------------

   function Is_Integer (Num, Den : Uint) return boolean is
   begin
      return UI_Eq (UI_Product (UI_Quotient (Num, Den), Den), Num);
   end Is_Integer;

   --------------
   -- Norm_Den --
   --------------

   function Norm_Den (Real : Ureal) return Uint is
   begin
      return Normalize (Ureals.Table (Real)).Den;
   end Norm_Den;

   --------------
   -- Norm_Num --
   --------------

   function Norm_Num (Real : Ureal) return Uint is
   begin
      return Normalize (Ureals.Table (Real)).Num;
   end Norm_Num;

   ---------------
   -- Normalize --
   ---------------

   function Normalize (Val : Ureal_Entry) return Ureal_Entry is
      J, K, Tmp : Uint;
      Num, Den  : Uint;

   begin
      --  Start by setting J to the greatest of the absolute values of the
      --  numerator and the denominator (taking into account the base value),
      --  and K to the lesser of the two absolute values. The gcd of Num and
      --  Den is the gcd of J and K.

      J := UI_Abs (Val.Num);

      if Val.Rbase = 0 then
         K := UI_Abs (Val.Den);

      elsif UI_Is_Negative (Val.Den) then
         J := UI_Product (
                J, UI_Exponentiate (UI_From_Int (Val.Rbase),
                                    UI_Abs (Val.Den)));
         K := Uint_1;

      else
         K := UI_Exponentiate (UI_From_Int (Val.Rbase), Val.Den);
      end if;

      if UI_Is_Negative (Val.Num) then
         Num := UI_Negate (J);
      else
         Num := J;
      end if;

      Den := K;

      if UI_Gt (K, J) then
         Tmp := J;
         J := K;
         K := Tmp;
      end if;

      --  Now apply Euclid's algorithm to find the gcd, which is left in J.

      while not UI_Is_Zero (K) loop
         Tmp := UI_Mod (J, K);
         J := K;
         K := Tmp;
      end loop;

      --  Divide numerator and denominator by gcd and return result

      return (Num  => UI_Quotient (Num, J),
              Den  => UI_Quotient (Den, J),
              Rbase => 0);
   end Normalize;

   -----------------
   -- Numerator --
   -----------------

   function Numerator (Real : Ureal) return Uint is
      Val : Ureal_Entry := Ureals.Table (Real);

   begin
      return Val.Num;
   end Numerator;

   -----------
   -- Rbase --
   -----------

   function Rbase (Real : Ureal) return Nat is
      Val : Ureal_Entry := Ureals.Table (Real);

   begin
      return Val.Rbase;
   end Rbase;

   -----------------
   -- Store_Ureal --
   -----------------

   function Store_Ureal (Val : Ureal_Entry) return Ureal is
   begin
      Ureals.Increment_Last;
      Ureals.Table (Ureals.Last) := Val;
      return Ureals.Last;
   end Store_Ureal;

   ----------------
   -- Ureal_Half --
   ----------------

   function Ureal_Half return Ureal is
   begin
      return UR_Half;
   end Ureal_Half;

   -----------------
   -- Ureal_Tenth --
   -----------------

   function Ureal_Tenth return Ureal is
   begin
      return UR_Tenth;
   end Ureal_Tenth;

   -------------
   -- Ureal_0 --
   -------------

   function Ureal_0 return Ureal is
   begin
      return UR_0;
   end Ureal_0;

   -------------
   -- Ureal_1 --
   -------------

   function Ureal_1 return Ureal is
   begin
      return UR_1;
   end Ureal_1;

   -------------
   -- Ureal_2 --
   -------------

   function Ureal_2 return Ureal is
   begin
      return UR_2;
   end Ureal_2;

   --------------
   -- Ureal_10 --
   --------------

   function Ureal_10 return Ureal is
   begin
      return UR_10;
   end Ureal_10;

   --------------------
   -- Ureals_Address --
   --------------------

   function Ureals_Address return System.Address is
   begin
      return Ureals.Table (Ureal_First_Entry)'Address;
   end Ureals_Address;

   ------------
   -- UR_Abs --
   ------------

   function UR_Abs (Real : Ureal) return Ureal is
      Val : constant Ureal_Entry := Ureals.Table (Real);

   begin
      return Store_Ureal (
               (Num  => UI_Abs (Val.Num),
                Den  => Val.Den,
                Rbase => Val.Rbase));
   end UR_Abs;

   -------------------
   -- UR_Difference --
   -------------------

   function UR_Difference (Left, Right : Ureal) return Ureal is
   begin
      return UR_Sum (Left, UR_Negate (Right));
   end UR_Difference;

   -----------
   -- UR_Eq --
   -----------

   function UR_Eq (Left, Right : Ureal) return Boolean is
      Lval : constant Ureal_Entry := Normalize (Ureals.Table (Left));
      Rval : constant Ureal_Entry := Normalize (Ureals.Table (Right));

   begin
      return UI_Eq (Rval.Num, Lval.Num) and then UI_Eq (Rval.Den, Lval.Den);
   end UR_Eq;

   ---------------------
   -- UR_Exponentiate --
   ---------------------

   function UR_Exponentiate (Real : Ureal; N : Uint) return Ureal is
      Val : Ureal_Entry := Ureals.Table (Real);
      X   : Uint        := UI_Abs (N);

   begin
      --  if the exponent is negative then we raise the numerator and the
      --  denominator (after normalization) to the absolute value of the
      --  exponent and we return the reciprocal. An assert error will happen
      --  if the numerator is zero.

      if UI_Is_Negative (N) then
         pragma Assert (Val.Num /= 0);

         Val := Normalize (Val);

         return Store_Ureal (
                 (Num  => UI_Exponentiate (Val.Den, X),
                  Den  => UI_Exponentiate (Val.Num, X),
                  Rbase => 0));

      --  if positive, we distinguish the case when the base is not zero, in
      --  which case the new denominator is just the product of the old one
      --  with the exponent,

      else
         if Val.Rbase /= 0 then

            return Store_Ureal (
                    (Num  => UI_Exponentiate (Val.Num, X),
                     Den  => UI_Product (Val.Den, X),
                     Rbase => Val.Rbase));

      --  and when the base is zero, in which case we exponentiate the
      --  old denominator.

         else

            return Store_Ureal (
                    (Num  => UI_Exponentiate (Val.Num, X),
                     Den  => UI_Exponentiate (Val.Den, X),
                     Rbase => 0));
         end if;
      end if;
   end UR_Exponentiate;

   -------------------------
   --  UR_From_Components --
   -------------------------

   function UR_From_Components
     (Num   : Uint;
      Den   : Uint;
      Rbase : Nat := 0)
      return Ureal
   is
   begin
      return Store_Ureal (
               (Num  => Num,
                Den  => Den,
                Rbase => Rbase));
   end UR_From_Components;

   ------------------
   -- UR_From_Uint --
   ------------------

   function UR_From_Uint (UI : Uint) return Ureal is
   begin
      return UR_From_Components (UI, Uint_1);
   end UR_From_Uint;

   -----------
   -- UR_Ge --
   -----------

   function UR_Ge (Left, Right : Ureal) return Boolean is
      Lval : constant Ureal_Entry := Normalize (Ureals.Table (Left));
      Rval : constant Ureal_Entry := Normalize (Ureals.Table (Right));

   begin
      return UI_Ge (
               UI_Product (Lval.Num, Rval.Den),
               UI_Product (Rval.Num, Lval.Den));
   end UR_Ge;

   -----------
   -- UR_Gt --
   -----------

   function UR_Gt (Left, Right : Ureal) return Boolean is
      Lval : constant Ureal_Entry := Normalize (Ureals.Table (Left));
      Rval : constant Ureal_Entry := Normalize (Ureals.Table (Right));

   begin
      return UI_Gt (
               UI_Product (Lval.Num, Rval.Den),
               UI_Product (Rval.Num, Lval.Den));
   end UR_Gt;

   --------------------
   -- UR_Is_Negative --
   --------------------

   function UR_Is_Negative (Real : Ureal) return Boolean is
      Val : constant Ureal_Entry := Ureals.Table (Real);

   begin
      return UI_Is_Negative (Val.Num);
   end UR_Is_Negative;

   --------------------
   -- UR_Is_Positive --
   --------------------

   function UR_Is_Positive (Real : Ureal) return Boolean is
      Val : constant Ureal_Entry := Ureals.Table (Real);

   begin
      return UI_Is_Positive (Val.Num);
   end UR_Is_Positive;

   ----------------
   -- UR_Is_Zero --
   ----------------

   function UR_Is_Zero (Real : Ureal) return Boolean is
      Val : constant Ureal_Entry := Ureals.Table (Real);

   begin
      return UI_Is_Zero (Val.Num);
   end UR_Is_Zero;

   -----------
   -- UR_Le --
   -----------

   function UR_Le (Left, Right : Ureal) return Boolean is
      Lval : constant Ureal_Entry := Normalize (Ureals.Table (Left));
      Rval : constant Ureal_Entry := Normalize (Ureals.Table (Right));

   begin
      return UI_Le (
               UI_Product (Lval.Num, Rval.Den),
               UI_Product (Rval.Num, Lval.Den));
   end UR_Le;

   -----------
   -- UR_Lt --
   -----------

   function UR_Lt (Left, Right : Ureal) return Boolean is
      Lval : constant Ureal_Entry := Normalize (Ureals.Table (Left));
      Rval : constant Ureal_Entry := Normalize (Ureals.Table (Right));

   begin
      return UI_Lt (
               UI_Product (Lval.Num, Rval.Den),
               UI_Product (Rval.Num, Lval.Den));
   end UR_Lt;

   ------------
   -- UR_Max --
   ------------

   function UR_Max (Left, Right : Ureal) return Ureal is
      Lval : constant Ureal_Entry := Normalize (Ureals.Table (Left));
      Rval : constant Ureal_Entry := Normalize (Ureals.Table (Right));

   begin
      if UR_Ge (Left, Right) then
         return Left;
      else
         return Right;
      end if;
   end UR_Max;

   ------------
   -- UR_Min --
   ------------

   function UR_Min (Left, Right : Ureal) return Ureal is
      Lval : constant Ureal_Entry := Normalize (Ureals.Table (Left));
      Rval : constant Ureal_Entry := Normalize (Ureals.Table (Right));

   begin
      if UR_Ge (Left, Right) then
         return Right;
      else
         return Left;
      end if;
   end UR_Min;

   -----------
   -- UR_Ne --
   -----------

   function UR_Ne (Left, Right : Ureal) return Boolean is
      Lval : constant Ureal_Entry := Normalize (Ureals.Table (Left));
      Rval : constant Ureal_Entry := Normalize (Ureals.Table (Right));

   begin
      return UI_Ne (Rval.Num, Lval.Num) or else UI_Ne (Rval.Den, Lval.Den);
   end UR_Ne;

   ---------------
   -- UR_Negate --
   ---------------

   function UR_Negate (Real : Ureal) return Ureal is
      Val : constant Ureal_Entry := Ureals.Table (Real);

   begin
      return Store_Ureal (
               (Num   => UI_Negate (Val.Num),
                Den   => Val.Den,
                Rbase => Val.Rbase));
   end UR_Negate;

   ----------------
   -- UR_Product --
   ----------------

   function UR_Product (Left, Right : Ureal) return Ureal is
      Lval : constant Ureal_Entry := Ureals.Table (Left);
      Rval : constant Ureal_Entry := Ureals.Table (Right);
      Num  : Uint                 := UI_Product (Lval.Num, Rval.Num);
      Den  : Uint;

   begin

      if Lval.Rbase = 0 then
         if Rval.Rbase = 0 then
            return Store_Ureal (
                     Normalize (
                        (Num   => Num,
                         Den   => UI_Product (Lval.Den, Rval.Den),
                         Rbase => 0)));

         elsif Is_Integer (Num, Lval.Den) then
            return Store_Ureal (
                     (Num   => UI_Quotient (Num, Lval.Den),
                      Den   => Rval.Den,
                      Rbase => Rval.Rbase));

         elsif UI_Is_Negative (Rval.Den) then
            return Store_Ureal (
                     Normalize (
                       (Num   => UI_Product (Num,
                                           UI_Exponentiate (
                                             UI_From_Int (Rval.Rbase),
                                             UI_Negate (Rval.Den))),
                        Den   => Lval.Den,
                        Rbase => 0)));

         else
            return Store_Ureal (
                     Normalize (
                       (Num   => Num,
                        Den   => UI_Product (
                                   Lval.Den,
                                   UI_Exponentiate (
                                     UI_From_Int (Rval.Rbase),
                                     Rval.Den)),
                        Rbase => 0)));
         end if;

      elsif Lval.Rbase = Rval.Rbase then
         return Store_Ureal (
                  (Num   => Num,
                   Den   => UI_Sum (Lval.Den, Rval.Den),
                   Rbase => Lval.Rbase));

      elsif Rval.Rbase = 0 then
         if Is_Integer (Num, Rval.Den) then
            return Store_Ureal (
                     (Num   => UI_Quotient (Num, Rval.Den),
                      Den   => Lval.Den,
                      Rbase => Lval.Rbase));

         elsif UI_Is_Negative (Lval.Den) then
            return Store_Ureal (
                     Normalize (
                       (Num   => UI_Product (
                                   Num,
                                   UI_Exponentiate (
                                     UI_From_Int (Lval.Rbase),
                                     UI_Negate (Lval.Den))),
                        Den   => Rval.Den,
                        Rbase => 0)));

         else
            return Store_Ureal (
                     Normalize (
                       (Num   => Num,
                        Den   => UI_Product (
                                   UI_Exponentiate (
                                     UI_From_Int (Lval.Rbase),
                                     Lval.Den),
                                   Rval.Den),
                        Rbase => 0)));
         end if;

      else
         Den := Uint_1;

         if UI_Is_Negative (Lval.Den) then
            Num := UI_Product (
                     Num,
                     UI_Exponentiate (
                       UI_From_Int (Lval.Rbase),
                       UI_Negate (Lval.Den)));
         else
            Den := UI_Product (
                     Den,
                     UI_Exponentiate (
                       UI_From_Int (Lval.Rbase),
                       Lval.Den));
         end if;

         if UI_Is_Negative (Rval.Den) then
            Num := UI_Product (
                     Num,
                     UI_Exponentiate (
                       UI_From_Int (Rval.Rbase),
                       UI_Negate (Rval.Den)));
         else
            Den := UI_Product (
                     Den,
                     UI_Exponentiate (
                       UI_From_Int (Rval.Rbase),
                       Rval.Den));
         end if;

         return Store_Ureal (
                  Normalize (
                    (Num   => Num,
                     Den   => Den,
                     Rbase => 0)));
      end if;

   end UR_Product;

   -----------------
   -- UR_Quotient --
   -----------------

   function UR_Quotient (Left, Right : Ureal) return Ureal is

      function Sign (UI : Uint) return Uint;
      --  Return Uint_1 if UI >= 0 and Uint_Minus_1 otherwize

      Lval   : constant Ureal_Entry := Ureals.Table (Left);
      Rval   : constant Ureal_Entry := Ureals.Table (Right);
      R_Sign : constant Uint        := Sign (Rval.Num);

      function Sign (UI : Uint) return Uint is
      begin
         if UI_Is_Negative (UI) then
            return Uint_Minus_1;
         else
            return Uint_1;
         end if;
      end Sign;

   --  begining of UR_Quotient

   begin
      pragma Assert (Rval.Num /= 0);

      if Lval.Rbase = 0 then

         if Rval.Rbase = 0 then
            return Store_Ureal (
                     Normalize (
                       (Num   =>
                          UI_Product (R_Sign, UI_Product (Lval.Num, Rval.Den)),
                        Den   =>
                          UI_Product (R_Sign, UI_Product (Lval.Den, Rval.Num)),
                        Rbase => 0)));

         elsif Is_Integer (Lval.Num, UI_Product (Rval.Num, Lval.Den)) then
            return Store_Ureal (
                     (Num  => UI_Quotient (
                                Lval.Num,
                                UI_Product (Rval.Num, Lval.Den)),
                      Den   => UI_Negate (Rval.Den),
                      Rbase => Rval.Rbase));

         elsif UI_Is_Negative (Rval.Den) then
            return Store_Ureal (
                     Normalize (
                       (Num   => UI_Product (R_Sign, Lval.Num),
                        Den   => UI_Product (
                                   UI_Product (
                                     R_Sign,
                                     UI_Exponentiate (
                                       UI_From_Int (Rval.Rbase),
                                       UI_Negate (Rval.Den))),
                                   UI_Product (Rval.Num, Lval.Den)),
                        Rbase => 0)));

         else
            return Store_Ureal (
                     Normalize (
                       (Num   => UI_Product (
                                   UI_Product (R_Sign, Lval.Num),
                                   UI_Exponentiate (
                                     UI_From_Int (Rval.Rbase),
                                     Rval.Den)),
                        Den   => UI_Product (
                                   R_Sign,
                                   UI_Product (Rval.Num, Lval.Den)),
                        Rbase => 0)));
         end if;

      elsif Is_Integer (Lval.Num, Rval.Num) then

         if Rval.Rbase = Lval.Rbase then
            return Store_Ureal (
                     (Num   => UI_Quotient (Lval.Num, Rval.Num),
                      Den   => UI_Difference (Lval.Den, Rval.Den),
                      Rbase => Lval.Rbase));

         elsif Rval.Rbase = 0 then
            return Store_Ureal (
                     (Num   => UI_Product (
                                 UI_Quotient (Lval.Num, Rval.Num),
                                 Rval.Den),
                      Den   => Lval.Den,
                      Rbase => Lval. Rbase));

         elsif UI_Is_Negative (Rval.Den) then
            declare
               Num, Den : Uint;
            begin

               if UI_Is_Negative (Lval.Den) then
                  Num :=
                    UI_Product (
                      UI_Quotient (Lval.Num, Rval.Num),
                      UI_Exponentiate (
                        UI_From_Int (Lval.Rbase),
                        UI_Negate (Lval.Den)));
                  Den :=
                    UI_Exponentiate (
                      UI_From_Int (Rval.Rbase),
                      UI_Negate (Rval.Den));
               else
                  Num := UI_Quotient (Lval.Num, Rval.Num);
                  Den :=
                    UI_Product (
                      UI_Exponentiate (
                        UI_From_Int (Lval.Rbase),
                        Lval.Den),
                      UI_Exponentiate (
                        UI_From_Int (Rval.Rbase),
                        UI_Negate (Rval.Den)));
               end if;

               return Store_Ureal (
                        (Num   => Num,
                         Den   => Den,
                         Rbase => 0));
            end;

         else
            return Store_Ureal (
                     (Num   => UI_Product (
                                 UI_Quotient (Lval.Num, Rval.Num),
                                 UI_Exponentiate (
                                   UI_From_Int (Rval.Rbase),
                                   Rval.Den)),
                      Den   => Lval.Den,
                      Rbase => Lval. Rbase));
         end if;

      else
         declare
            Num, Den : Uint;
         begin

            if UI_Is_Negative (Lval.Den) then
               Num :=
                 UI_Product (
                   UI_Product (R_Sign, Lval.Num),
                   UI_Exponentiate (
                     UI_From_Int (Lval.Rbase),
                     UI_Negate (Lval.Den)));
               Den := UI_Product (R_Sign, Rval.Num);

            else
               Num := UI_Product (R_Sign, Lval.Num);
               Den :=
                 UI_Product (
                   UI_Product (R_Sign, Rval.Num),
                   UI_Exponentiate (UI_From_Int (Lval.Rbase), Lval.Den));
            end if;

            if Rval.Rbase /= 0 then
               if UI_Is_Negative (Rval.Den) then
                  Den :=
                    UI_Product (
                      Den,
                      UI_Exponentiate (
                        UI_From_Int (Rval.Rbase),
                        UI_Negate (Rval.Den)));
               else
                  Num :=
                    UI_Product (
                      Num,
                      UI_Exponentiate (
                        UI_From_Int (Rval.Rbase),
                        Rval.Den));
               end if;

            else
               Num := UI_Product (Num, Rval.Den);
            end if;

            return Store_Ureal (
                     Normalize (
                       (Num   => Num,
                        Den   => Den,
                        Rbase => 0)));
         end;
      end if;
   end UR_Quotient;

   ------------
   -- UR_Sum --
   ------------

   function UR_Sum (Left, Right : Ureal) return Ureal is
      Lval : constant Ureal_Entry := Ureals.Table (Left);
      Rval : constant Ureal_Entry := Ureals.Table (Right);

   begin
      if Lval.Rbase /= 0 and then Lval.Rbase = Rval.Rbase then

         declare
            Opd_Min, Opd_Max   : Ureal_Entry;
            Exp_Min, Exp_Max   : Uint;

         begin

            if UI_Lt (Lval.Den, Rval.Den) then
               Exp_Min := Lval.Den;
               Exp_Max := Rval.Den;
               Opd_Min := Lval;
               Opd_Max := Rval;
            else
               Exp_Min := Rval.Den;
               Exp_Max := Lval.Den;
               Opd_Min := Rval;
               Opd_Max := Lval;
            end if;

            return Store_Ureal (
                     (Num   => UI_Sum (
                                 UI_Product (
                                   Opd_Min.Num,
                                   UI_Exponentiate (
                                     UI_From_Int (Lval.Rbase),
                                     UI_Difference (Exp_Max, Exp_Min))),
                                 Opd_Max.Num),
                      Den   => Exp_Max,
                      Rbase => Lval.Rbase));
         end;

      else
         declare
            Ln : constant Ureal_Entry := Normalize (Lval);
            Rn : constant Ureal_Entry := Normalize (Rval);

         begin
            return Store_Ureal (
                     Normalize (
                       (Num   => UI_Sum (
                                   UI_Product (Ln.Num, Rn.Den),
                                   UI_Product (Rn.Num, Ln.Den)),
                        Den   => UI_Product (Ln.Den, Rn.Den),
                        Rbase => 0)));
         end;
      end if;
   end UR_Sum;

   ----------------
   -- UR_To_Uint --
   ----------------

   function UR_To_Uint (Real : Ureal) return Uint is
      Val : Ureal_Entry := Normalize (Ureals.Table (Real));

   begin

      if UI_Is_Negative (Val.Num) then
         Val.Num := UI_Difference (Val.Num, UI_Quotient (Val.Den, Uint_2));
      else
         Val.Num := UI_Sum (Val.Num, UI_Quotient (Val.Den, Uint_2));
      end if;

      return UI_Quotient (Val.Num, Val.Den);
   end UR_To_Uint;

   --------------
   -- UR_Trunc --
   --------------

   function UR_Trunc (Real : Ureal) return Uint is
      Val : constant Ureal_Entry := Normalize (Ureals.Table (Real));

   begin
      return UI_Quotient (Val.Num, Val.Den);
   end UR_Trunc;

   --------------
   -- UR_Write --
   --------------

   procedure UR_Write (Real : Ureal) is
      Val : constant Ureal_Entry := Ureals.Table (Real);

   begin
      Write_Str ("[");
      UI_Write (Val.Num);
      Write_Str (" / ");

      if Val.Rbase = 0 then
         UI_Write (Val.Den);

      else
         Write_Int (Val.Rbase);
         Write_Str ("**");
         UI_Write (Val.Den);
      end if;

      Write_Str ("]");
   end UR_Write;

end Urealp;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.10
--  date: Sat Jun 18 15:38:13 1994;  author: dewar
--  (Ureal_Tenth): New function (additional constant 0.1)
--  ----------------------------
--  revision 1.11
--  date: Thu Jul 28 18:58:42 1994;  author: dewar
--  Change name Ureal.Base to Ureal.Rbase
--  Minor reformatting
--  ----------------------------
--  revision 1.12
--  date: Tue Aug 16 11:44:31 1994;  author: dewar
--  Allocate Ureals table big and static (temp fix for Gigi use problem)
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
