/* Find file command for the Midnight Commander
   Copyright (C) 1995 Miguel de Icaza

   Complete rewrote.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <ncurses.h>
#include <string.h>
#include <stdio.h>
#include <malloc.h>	/* For free() */
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
    #include <unistd.h>
#endif

/* unistd.h defines _POSIX_VERSION on POSIX.1 systems. */
#if defined(HAVE_DIRENT_H) || defined(_POSIX_VERSION)
    #include <dirent.h>
    #define NLENGTH(dirent) (strlen ((dirent)->d_name))
#else
    #define dirent direct
    #define NLENGTH(dirent) ((dirent)->d_namlen)

    #ifdef HAVE_SYS_NDIR_H
        #include <sys/ndir.h>
    #endif /* HAVE_SYS_NDIR_H */

    #ifdef HAVE_SYS_DIR_H
        #include <sys/dir.h>
    #endif /* HAVE_SYS_DIR_H */

    #ifdef HAVE_NDIR_H
        #include <ndir.h>
    #endif /* HAVE_NDIR_H */
#endif /* not (HAVE_DIRENT_H or _POSIX_VERSION) */

#include <sys/stat.h>
#include <sys/param.h>
#include "global.h"
#include "mad.h"
#include "win.h"
#include "input.h"
#include "color.h"
#include "util.h"
#include "global.h"

extern int verbose;		/* Should be in a more sensible header file */

/* Dialog manager and widgets */
#include "dlg.h"
#include "widget.h"

#include "dialog.h"     /* For do_refresh() */
#define  DIR_H_INCLUDE_HANDLE_DIRENT
#include "dir.h"
#include "panel.h"		/* current_panel */
#include "main.h"		/* do_cd, try_to_select */
#include "wtools.h"
#include "tree.h"

/* Size of the find parameters window */
#define FIND_Y 10
#define FIND_X 50

/* Size of the find window */
#define FIND2_Y 20
#define FIND2_X 50

/* A couple of extra messages we need */
#define B_STOP   B_USER+1
#define B_AGAIN  B_USER+2
#define B_PANELIZE	B_USER+3

#define B_TREE	 B_USER+1

static WINDOW *find_win;	/* The window */
static Dlg_head *find_dlg;	/* The dialog */
static WInput *in_start;	/* Start path */
static WInput *in_name;		/* Pattern to search */
static WListbox *find_list;	/* Listbox with the file list */
static chtype colors [4];	/* colors for the dialogs */
static int running = 0;		/* nice flag */
static WButton *stop_button;	/* pointer to the stop button */
static char *find_pattern;	/* Pattern to search */
static int count;		/* Number of files displayed */
static int matches;		/* Number of matches */
static int is_start;		/* Status of the start/stop toggle button */


static int find_par_callback (struct Dlg_head *h, int id, int Msg)
{
    switch (Msg){
    case DLG_DRAW:
	wattrset (h->window, REVERSE_COLOR);
	wclr (h->window);
	draw_box (h->window, 1, 1, FIND_Y-2, FIND_X-2);
	mvwprintw (h->window, 3, 3, "Start at:");
	mvwprintw (h->window, 5, 3, "Filename:");
	wattrset (h->window, COLOR_HOT_NORMAL);
	mvwprintw (h->window, 1, 2, " Find file ");
	break;
    }
    return 0;
}

static int find_parameters (char **start_dir, char **pattern)
{
    int return_value;
    char *temp_dir;
    char *in_start_dir = strdup (".");

 find_par_start:
    init_box_colors (colors);
    find_win = centerwin (FIND_Y, FIND_X);
    find_dlg = dlg_new (find_win, colors, find_par_callback,
			winpos (FIND_Y, FIND_X), "[Find File]");

    in_start = input_new (find_win, 3, 14, NORMAL_COLOR, 30, in_start_dir);
    in_name  = input_new (find_win, 5, 14, NORMAL_COLOR, 30,
			  easy_patterns ? "*" : ".");

    add_widget (find_dlg, in_start);

    add_widget (find_dlg, button_new (7, 36, B_CANCEL,"[ Cancel ]",'c',2,0,0));
    
#if HAVE_AVOID_DIRS
    add_widget (find_dlg, button_new (7, 22, B_USER, "[ Avoid... ]",
				      'a', 2,
				      avoid_box, 0));
#endif

    add_widget (find_dlg, button_new (7, 12, B_TREE, "[ Tree ]", 't', 2,0,0));
    add_widget (find_dlg, button_new (7, 4, B_ENTER,"[ Ok ]",'o', 2, 0, 0));
    add_widget (find_dlg, in_name);

    run_dlg (find_dlg);
    if (find_dlg->ret_value == B_CANCEL)
	return_value = 0;
    else if (find_dlg->ret_value == B_TREE){
	temp_dir = strdup (in_start->in->buffer);
	destroy_dlg (find_dlg);
	delwin (find_win);
	free (in_start_dir);
	in_start_dir = tree (temp_dir);
	if (in_start_dir)
	    free (temp_dir);
	else
	    in_start_dir = temp_dir;
	/* Warning: Dreadful goto */
	goto find_par_start;
    } else {
	return_value = 1;
	*start_dir = strdup (in_start->in->buffer);
	*pattern   = strdup (in_name->in->buffer);
    }

    destroy_dlg (find_dlg);
    delwin (find_win);
    free (in_start_dir);
    do_refresh ();
			 
    return return_value;
}

typedef struct dir_stack {
    char *name;
    struct dir_stack *prev;
} dir_stack ;

dir_stack *dir_stack_base = 0;

static void push_directory (char *dir)
{
    dir_stack *new;

    new = xmalloc (sizeof (dir_stack), "find: push_directory");
    new->name = strdup (dir);
    new->prev = dir_stack_base;
    dir_stack_base = new;
}

static char *pop_directory (void)
{
    char *name; 
    dir_stack *next;

    if (dir_stack_base){
	name = dir_stack_base->name;
	next = dir_stack_base->prev;
	free (dir_stack_base);
	dir_stack_base = next;
	return name;
    } else
	return 0;
}

int max_loops_in_idle = 10;
static char *old_dir;
    
static void insert_file (char *dir, char *file)
{
    char *tmp_name;
    static char *dirname;
    
    if (old_dir){
	if (strcmp (old_dir, dir)){
	    free (old_dir);
	    old_dir = strdup (dir);
	    dirname = listbox_add_item (find_list, 0, 0, dir, 0);
	}
    } else {
	old_dir = strdup (dir);
	dirname = listbox_add_item (find_list, 0, 0, dir, 0);
    }
    
    tmp_name = copy_strings ("    ", file, 0);
    listbox_add_item (find_list, 0, 0, tmp_name, dirname);
    free (tmp_name);
}

static char *rotating_dash = "|/-\\";

static void do_search (struct Dlg_head *h)
{
    static DIR           *dirp = 0;
    static struct dirent *dp   = 0;
    static char          directory [MAXPATHLEN+2];
    static int           pos;

    struct stat tmp_stat;
    char *tmp_name;		/* For bulding file names */
    int p;

    while (!dp){
	
	if (dirp){
	    closedir (dirp);
	    dirp = 0;
	}
	
	while (!dirp){
	    char *tmp;
	    
	    tmp = pop_directory ();
	    
	    if (!tmp){
		running = 0;
		mvwprintw (h->window, FIND2_Y-6, 4, "Finished     ");
		wrefresh (h->window);
		set_idle_proc (h, 0);
		return;
	    } else {
		strcpy (directory, tmp);
		free (tmp);
	    }
	    
	    dirp = opendir (directory);
	}
	dp = readdir (dirp);
    }

    if (strcmp (dp->d_name, ".") == 0 ||
	strcmp (dp->d_name, "..") == 0){
	dp = readdir (dirp);
	return;
    }
    
    tmp_name = copy_strings (directory, "/", dp->d_name, 0);
    lstat (tmp_name, &tmp_stat);
    
    if (S_ISDIR (tmp_stat.st_mode))
	push_directory (tmp_name);
    
    if (regexp_match (find_pattern, dp->d_name, match_file)){
	insert_file (directory, dp->d_name);
	matches++;
    }

    p = matches % (FIND2_Y-8);
    
    if (!p)
	listbox_select_last (find_list, 1);
    else
	listbox_select_last (find_list, 0);
    
    /* Updates the current listing */
    listbox_draw (h->window, find_list, h, 1);

    /* Displays the nice dot */
    if (verbose){
	count++;
	if (!(count % 20)){
	    pos = (++pos) % 4;
	
	    wattron (h->window, NORMALC);
	    mvwaddch (h->window, FIND2_Y-6, 16, rotating_dash [pos]);
	}
    }
    
    wrefresh (h->window);
    free (tmp_name);
    dp = readdir (dirp);
}

static int find_callback (struct Dlg_head *h, int id, int Msg)
{
    switch (Msg){
    case DLG_DRAW:
	wattrset (h->window, REVERSE_COLOR);
	wclr (h->window);
	draw_box (h->window, 1, 1, FIND2_Y-2, FIND2_X-2);
	mvwprintw (h->window, FIND2_Y-6, 4, "Searching...");
	wattrset (h->window, COLOR_HOT_NORMAL);
	mvwprintw (h->window, 1, 2, " Find file ");
	break;
	
    case DLG_IDLE:
	do_search (h);
	break;

    }
    return 0;
}

#define BUTTON_NEW(i,msg,str) \
   button_new (FIND2_Y-4, 3+i*12, msg, str, str [2], 2, 0, 0)

#define BUTTON_NEW_CBACK(i,msg,str,a,b) \
   button_new (FIND2_Y-4, 3+i*12, msg, str, str [2], 2, a, b)

/* Handles the Stop/Start button in the find window */
static int start_stop (int button, void *extra)
{
    char *button_labels [2] = { "[ Stop ] ", "[ Start ]" };
    
    running = is_start;
    set_idle_proc (find_dlg, running);
    is_start = !is_start;

    wattron (((Dlg_head *) extra)->window, REVERSE_COLOR);
    mvwprintw (((Dlg_head *) extra)->window, FIND2_Y-6, 4,
	       is_start ? "Stopped       " : "Searching...");
    button_set_text (stop_button, button_labels [is_start]);
    
    return 0;
}

static void init_find_vars (void)
{
    char *dir;
    
    if (old_dir){
	free (old_dir);
	old_dir = 0;
    }
    count = 0;
    matches = 0;

    /* Remove all the items in the stack */
    while ((dir = pop_directory ()) != NULL)
	free (dir);
}

static int find_file (char *start_dir, char *pattern,
		      char **dirname,  char **filename)
{
    int return_value = 0;
    char *dir;
    char *dir_tmp, *file_tmp;
    
    find_win = centerwin (FIND2_Y, FIND2_X);
    find_dlg = dlg_new (find_win, colors, find_callback,
			winpos (FIND2_Y, FIND2_X), "[Find File]");

    find_list = listbox_new (2, 2, FIND2_X-6, FIND2_Y-8, listbox_finish, 0);

    add_widget (find_dlg, find_list);

    add_widget (find_dlg,
		button_new (FIND2_Y-3, 3, B_PANELIZE, "[ Panelize ]", 'p', 2, 0, 0));
    add_widget (find_dlg, BUTTON_NEW (3, B_CANCEL, "[ Quit ]"));
    stop_button = BUTTON_NEW_CBACK (2, B_STOP,   "[ Stop  ]", start_stop,
				    find_dlg);
    add_widget (find_dlg, stop_button);
    add_widget (find_dlg, BUTTON_NEW (1, B_AGAIN,  "[ Again ]"));
    add_widget (find_dlg, BUTTON_NEW (0, B_ENTER,  "[ Chdir ]"));

    /* Need to cleanup this */
    find_pattern = pattern;
    
    set_idle_proc (find_dlg, 1);
    init_find_vars ();
    push_directory (start_dir);
    
    run_dlg (find_dlg);

    return_value = find_dlg->ret_value;

    /* Remove all the items in the stack */
    while ((dir = pop_directory ()) != NULL)
	free (dir);
    
    listbox_get_current (find_list, &file_tmp, &dir_tmp);

    *dirname = *filename = 0;
    if (dir_tmp)
	*dirname  = strdup (dir_tmp);
    if (file_tmp)
	*filename = strdup (file_tmp);
    if (return_value == B_PANELIZE && dirname && filename){
	struct dirent dp;
	int status, link_to_dir;
	int next_free = 0;
	int i;
	struct stat buf;
	WLEntry *entry = find_list->list;
	dir_list *list = &cpanel->dir;
	char *dir, *name;

	clean_dir (list, cpanel->count);
	for (i = 0; entry && i < find_list->count; entry = entry->next, i++){
	    if (!entry->text || !entry->data)
		continue;
	    dir = entry->data;
	    if (dir [0] == '.' && dir [1] == 0)
		name = strdup (entry->text + 4);
	    else if (dir [0] == '.' && dir [1] == '/')
		name = get_full_name (dir + 2, entry->text + 4);
	    else
		name = get_full_name (dir, entry->text + 4);
	    strcpy (dp.d_name, name);
	    free (name);
#if !defined(HAVE_DIRENT_H) && !defined(_POSIX_VERSION)
	    NLENGTH (&dp) = strlen (dp.d_name);
#endif
	    status = handle_dirent (list, NULL, &dp, &buf, next_free, &link_to_dir);
	    if (status == 0)
		continue;
	    if (status == -1)
		break;
	    list->list [next_free].fnamelen = strlen (dp.d_name);
	    list->list [next_free].fname = strdup (dp.d_name);
	    list->list [next_free].f.marked = 0;
	    list->list [next_free].f.link_to_dir = link_to_dir;
	    list->list [next_free].buf = buf;
	    next_free++;
	}
	if (next_free){
	    cpanel->count = next_free;
	    cpanel->dont_reload = 1;
	    if (start_dir [0] == '/'){
		strcpy (cpanel->cwd, "/");
		chdir ("/");
	    }
	    if (!is_view_file_listing (cpanel->view_type))
		cpanel->view_type = view_full;
	}
    }

    destroy_dlg (find_dlg);
    do_refresh ();
    if (old_dir){
	free (old_dir);
	old_dir = 0;
    }
    return return_value;
}

void do_find (void)
{
    char *start_dir, *pattern;
    char *filename, *dirname;
    int  v;
    int done = 0;
    
    while (!done){
	if (!find_parameters (&start_dir, &pattern))
	    break;

	v = find_file (start_dir, pattern, &dirname, &filename);
	free (start_dir);
	free (pattern);

	if (v == B_ENTER){
	    if (dirname){
		do_cd (dirname);
		if (filename)
		    try_to_select (current_panel, filename + 4);
		paint_panel (current_panel);
		select_item (cpanel);
	    }
	}
	if (dirname)  free (dirname);
	if (filename) free (filename);
	if (v == B_CANCEL)
	    break;
	if (v == B_PANELIZE){
	    try_to_select (cpanel, NULL);
	    paint_panel (cpanel);
	    break;
	}
    }
}

