------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               U R E A L P                                --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                             $Revision: 1.9 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  Support for universal real arithmetic

with System; use System;
with Types;  use Types;
with Uintp;  use Uintp;

package Urealp is

   ---------------------------------------
   -- Representation of Universal Reals --
   ---------------------------------------

   --  A universal real value is represented by a single value (which is
   --  an index into an internal table). These values are not hashed, so
   --  the equality operator should not be used on Ureal values (instead
   --  use the UR_Eq function).

   --  A Ureal value represents an arbitrary precision universal real value,
   --  stored internally using three components, the numerator (a Uint value),
   --  the denominator (a Uint value), and a real base (Nat). If the base
   --  is zero, then the value of the Ureal is simply numerator/denominator.
   --  If the base is non-zero, the value is num / (rbase ** den).

   --  Negative numbers are represented by the sign of the numerator being
   --  negative. The denominator is always positive.

   --  A normalized Ureal value has base = 0, and numerator/denominator
   --  reduced to lowest terms, with zero itself being represented as 0/1.
   --  This is a canonical format, so that for normalized Ureal values it
   --  is the case that two equal values always have the same denominator
   --  and numerator values.

   ---------------------
   -- Ureal Constants --
   ---------------------

   function Ureal_0 return Ureal;
   --  Returns value 0.0

   function Ureal_Tenth return Ureal;
   --  Returns value 0.1

   function Ureal_Half return Ureal;
   --  Returns value 0.5

   function Ureal_1 return Ureal;
   --  Returns value 1.0

   function Ureal_2 return Ureal;
   --  Returns value 2.0

   function Ureal_10 return Ureal;
   --  Returns value 10.0

   -----------------
   -- Subprograms --
   -----------------

   procedure Initialize;
   --  Initialize Ureal tables

   function Rbase (Real : Ureal) return Nat;
   --  Return the base of the universal real.

   function Denominator (Real : Ureal) return Uint;
   --  Return the denominator of the universal real.

   function Numerator (Real : Ureal) return Uint;
   --  Return the numerator of the universal real.

   function Norm_Den (Real : Ureal) return Uint;
   --  Return the denominator of the universal real after a normalization.

   function Norm_Num (Real : Ureal) return Uint;
   --  Return the numerator of the universal real after a normalization.

   function Ureals_Address return Address;
   --  Return address of Ureals table (used for Back_End call to Gigi)

   function UR_From_Uint (UI : Uint) return Ureal;
   --  Returns real corresponding to universal integer value

   function UR_To_Uint (Real : Ureal) return Uint;
   --  Return integer value obtained by accurate rounding of real value.
   --  The rounding of values half way between two integers is away from
   --  zero, as required by normal Ada 9X rounding semantics.

   function UR_From_Components
     (Num   : Uint;
      Den   : Uint;
      Rbase : Nat := 0)
      return Ureal;
   --  Builds real value from given numerator, denominator and base.

   function UR_Trunc (Real : Ureal) return Uint;
   --  Return integer value obtained by a truncation of real value.

   function UR_Sum (Left, Right : Ureal) return Ureal;
   --  Returns sum of two reals.

   function UR_Difference (Left, Right : Ureal) return Ureal;
   --  Returns difference of two reals.

   function UR_Quotient (Left, Right : Ureal) return Ureal;
   --  Returns quotient of two reals. Fatal error if Right is zero

   function UR_Product (Left, Right : Ureal) return Ureal;
   --  Returns product of two reals

   function UR_Exponentiate (Real  : Ureal; N : Uint) return  Ureal;
   --  Returns result of raising Ureal to Uint power.
   --  Fatal error if Left is 0 and Right is negative.

   function UR_Abs (Real : Ureal) return Ureal;
   --  Returns abs function of real

   function UR_Negate (Real : Ureal) return Ureal;
   --  Returns negative of real

   function UR_Eq (Left, Right : Ureal) return Boolean;
   --  Compares reals for equality.

   function UR_Max (Left, Right : Ureal) return Ureal;
   --  Returns the maximum of two reals

   function UR_Min (Left, Right : Ureal) return Ureal;
   --  Returns the minimum of two reals

   function UR_Ne (Left, Right : Ureal) return Boolean;
   --  Compares reals for inequality.

   function UR_Lt (Left, Right : Ureal) return Boolean;
   --  Compares reals for less than.

   function UR_Le (Left, Right : Ureal) return Boolean;
   --  Compares reals for less than or equal.

   function UR_Gt (Left, Right : Ureal) return Boolean;
   --  Compares reals for greater than.

   function UR_Ge (Left, Right : Ureal) return Boolean;
   --  Compares reals for greater than or equal.

   function UR_Is_Zero (Real : Ureal) return Boolean;
   --  Tests if real value is zero

   function UR_Is_Negative (Real : Ureal) return Boolean;
   --  Tests if real value is less than zero

   function UR_Is_Positive (Real : Ureal) return Boolean;
   --  Test if real value is greater than zero

   procedure UR_Write (Real : Ureal);
   --  Writes value of real to standard output. Used only for debugging and
   --  tree/source output. If the result is easily representable as a standard
   --  Ada literal, it will be given that way, but as a result of evaluation
   --  of static expressions, it is possible to generate constants (e.g. 1/13)
   --  which have no such representation. In such cases (and in cases where it
   --  is too much work to figure out the Ada literal), the string that is
   --  output is of the form [numerator/denominator].

private
   pragma Inline (Rbase);
   pragma Inline (Denominator);
   pragma Inline (Norm_Num);
   pragma Inline (Norm_Den);
   pragma Inline (Numerator);
   pragma Inline (Ureal_0);
   pragma Inline (Ureal_Tenth);
   pragma Inline (Ureal_Half);
   pragma Inline (Ureal_1);
   pragma Inline (Ureal_2);
   pragma Inline (Ureal_10);
   pragma Inline (Ureals_Address);
   pragma Inline (UR_From_Components);

end Urealp;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.7
--  date: Thu Jun  2 16:07:34 1994;  author: crozes
--  (Make_Real_Literal, Realval) : remove procedure and function.
--  ----------------------------
--  revision 1.8
--  date: Sat Jun 18 15:38:19 1994;  author: dewar
--  (Ureal_Tenth): New function (additional constant 0.1)
--  ----------------------------
--  revision 1.9
--  date: Thu Jul 28 18:58:50 1994;  author: dewar
--  Change name from Base to Rbase to avoid clash with Types.Base
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
