/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#undef USE_GNOME
#include "style.h"
#include "xchat.h"
#include <sys/stat.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>
#include "dcc.h"
#include "gtkutil.h"

/* update the window every 4k */
#define DCC_UPDATE_WINDOW 4096


static char *dcctypes[]= { "SEND", "RECV", "CHAT", "CHAT", "DRAW", "DRAW"};
static char *dccstat[]= { "Queued", "Active", "Failed", "Done", "Connect", "Aborted" };

struct dccwindow dccrwin; /* recv */
struct dccwindow dccswin; /* send */
struct dccwindow dcccwin; /* chat */

extern GSList *sess_list;
extern GSList *serv_list;
extern GSList *dcc_list;

extern struct xchatprefs prefs;

void dcc_close(struct DCC *dcc, int stat, int destroy);
struct DCC *new_dcc(void);
void update_dcc_recv(struct DCC *dcc);
void open_dcc_recv_window(void);
void update_dcc_recv_window(void);
void open_dcc_send_window(void);
void update_dcc_send_window(void);
void open_dcc_chat_window(void);
void update_dcc_chat_window(void);
void dcc_send(struct session *sess, char *tbuf, char *to, char *file);
struct DCC *find_dcc(char *nick, char *file, int type);
void update_dcc_recv(struct DCC *dcc);
void update_dcc_send(struct DCC *dcc);

extern int is_data_avail(int sok);
extern char *errorstring(int err);
extern char *get_home_dir(void);
extern void private_msg(struct server *serv, char *tbuf, char *from, char *ip, char *text);
extern void PrintText(struct session *sess, char *text);
extern int waitline(int sok, char *buf, int bufsize);
extern void dcc_open_draw_window(struct DCC *dcc);
extern void dcc_close_draw_window(struct DCC *dcc);
extern void dcc_read_draw(struct DCC *dcc, gint sok);
extern char *file_part(char *file);
extern void check_homedir(char *file);
extern void path_part(char *file, char *path);


void dcc_check_timeouts(void)
{
   time_t tim = time(0);
   struct DCC *dcc;
   GSList *list = dcc_list;

   while(list)
   {
      dcc = (struct DCC *) list -> data;
      switch(dcc->stat)
      {
       case STAT_ACTIVE:
	 if(dcc->type == TYPE_SEND || dcc->type == TYPE_RECV)
	 {
	    if(prefs.dccstalltimeout > 0)
	    {
	       if(tim - dcc->lasttime > prefs.dccstalltimeout)
	       {
		  char tbuf[1024];
		  snprintf(tbuf, sizeof tbuf,
			   STARTON" DCC %s \00311%s \003 to \00311%s \003 stalled - aborting.\n",
			   dcctypes[(int)dcc->type], file_part(dcc->file), dcc->nick);
		  PrintText(dcc->serv->front_session, tbuf);
		  dcc_close(dcc, 0, TRUE);
		  dcc_check_timeouts();
		  return;
	       }
	    }
	 }
	 break;
       case STAT_QUEUED:
	 if(dcc->type == TYPE_SEND || dcc->type == TYPE_CHATSEND)
	 {
	    if(tim - dcc->offertime > prefs.dcctimeout)
	    {
	       if(prefs.dcctimeout > 0)
	       {
		  char tbuf[1024];
		  snprintf(tbuf, sizeof tbuf,
			   STARTON" DCC %s \00311%s \003 to \00311%s \003 timed out - aborted.\n",
			   dcctypes[(int)dcc->type], file_part(dcc->file), dcc->nick);
		  PrintText(dcc->serv->front_session, tbuf);
		  dcc_close(dcc, 0, TRUE);
		  dcc_check_timeouts();
		  return;
	       }
	    }
	 }
	 break;
      }
      list = list -> next;
   }
}

void dcc_send_filereq_done(struct session *sess, char *nick, char *file)
{
   char tbuf[1024];

   if(file)
   {
      dcc_send(sess, tbuf, nick, file);
      free(file);
   }
   free(nick);
}

void dcc_send_filereq(struct session *sess, char *nick)
{
   char tbuf[128];

   nick = strdup(nick);
   snprintf(tbuf, sizeof tbuf, "Send file to %s", nick);
   gtkutil_file_req(tbuf, dcc_send_filereq_done, sess, nick, FALSE);
}

long net_connect(struct DCC *dcc, long port)
{
   long sok;

   sok = socket(AF_INET, SOCK_STREAM, 0);
   if(sok == -1) return -1;

   dcc->SAddr.sin_port = htons(port);
   dcc->SAddr.sin_family = AF_INET;
   dcc->SAddr.sin_addr.s_addr = dcc->addr;

   fcntl(sok, F_SETFL, O_NONBLOCK);
   connect(sok, (struct sockaddr *)&dcc->SAddr, sizeof(struct sockaddr_in));
   fcntl(sok, F_SETFL, 0);

   return sok;
}

void update_dcc_window(int type)
{
   switch(type)
   {
    case TYPE_SEND: update_dcc_send_window(); break;
    case TYPE_RECV: update_dcc_recv_window(); break;
    case TYPE_CHATRECV:
    case TYPE_CHATSEND: update_dcc_chat_window(); break;
   }
}

void dcc_close(struct DCC *dcc, int stat, int destroy)
{
   char type = dcc->type;
   if(dcc->sok != -1)
   {
      gdk_input_remove(dcc->iotag);
      close(dcc->sok);
      dcc->sok = -1;
   }
   if(dcc->fp != -1 )
   {
      close(dcc->fp);
      dcc->fp = -1;
   }
   dcc->stat = stat;
   if(type == TYPE_DRAWSEND || type == TYPE_DRAWRECV)
   {
      if(dcc->dccdraw) dcc_close_draw_window(dcc);
   }
   if(dcc->dccchat)
   {
      free(dcc->dccchat);
      dcc->dccchat = 0;
   }
   if(destroy)
   {
      dcc_list = g_slist_remove(dcc_list, dcc);
      free(dcc);
      update_dcc_window(type);
      return;
   }
   switch(type)
   {
    case TYPE_SEND:
      update_dcc_send(dcc);
      break;
    case TYPE_RECV:
      update_dcc_recv(dcc);
      break;
    default:
      update_dcc_window(type);
   }
}

void dcc_notify_kill(struct server *serv)
{
   struct server *replaceserv = 0;
   struct DCC *dcc;
   GSList *list = dcc_list;
   if(serv_list) replaceserv = (struct server *)serv_list->data;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if(dcc->serv == serv) dcc->serv = replaceserv;
      list = list->next;
   }
}

struct sockaddr_in *dcc_write_chat(char *nick, char *text)
{
   struct DCC *dcc;
   dcc = find_dcc(nick, "", TYPE_CHATRECV);
   if(!dcc) dcc = find_dcc(nick, "", TYPE_CHATSEND);
   if(dcc && dcc->stat == STAT_ACTIVE)
   {
      int len = strlen(text);
      dcc->size += len;
      send(dcc->sok, text, len, 0);
      send(dcc->sok, "\n", 1, 0);
      update_dcc_chat_window();
      return &dcc->SAddr;
   }  
   return 0;
}

void dcc_read_chat(struct DCC *dcc, gint sok)
{
   int i;
   long len;
   char tbuf[1226];
   char lbuf[1026];

   do
   {
      len = recv(sok, lbuf, sizeof lbuf-2, 0);
      if(len < 1)
      {
	 snprintf(tbuf, sizeof(tbuf), STARTON" DCC CHAT failed. Connection to %s lost.\n", dcc->nick);
	 PrintText(dcc->serv->front_session, tbuf);
	 dcc_close(dcc, STAT_FAILED, FALSE);
	 return;
      }

      i = 0;
      lbuf[len] = 0;
      while(i < len)
      {
	 switch(lbuf[i])
	 {
	  case '\r':
	    break;
	  case '\n':
	    dcc->dccchat->linebuf[dcc->dccchat->pos] = 0;
	    private_msg(dcc->serv, tbuf, dcc->nick, "", dcc->dccchat->linebuf);
	    update_dcc_chat_window();
	    dcc->dccchat->pos = 0;
	    break;
	  default:
	    dcc->dccchat->linebuf[dcc->dccchat->pos] = lbuf[i];
	    dcc->dccchat->pos++;
	    if(dcc->dccchat->pos == 1023) dcc->dccchat->pos = 1022;
	 }
	 i++;
      }
   } while(is_data_avail(sok));
}

void dcc_read(struct DCC *dcc, gint sok)
{
   long n;
   char buf[4098];

   if(dcc->fp == -1)
   {
      if(dcc->resumable)
      {
	 dcc->fp = open(dcc->destfile, O_WRONLY | O_APPEND);
	 dcc->pos = dcc->resumable;
	 dcc->ack = dcc->resumable;
      } else
	 dcc->fp = open(dcc->destfile, O_TRUNC | O_WRONLY | O_CREAT, 0600);
   }

   if(dcc->fp == -1)
   {
      snprintf(buf, sizeof(buf), STARTON" DCC RECV: Cannot open %s for writing - aborting.\n", dcc->destfile);
      PrintText(dcc->serv->front_session, buf);
      dcc_close(dcc, STAT_FAILED, FALSE);
      return;
   }

   do
   {
      n = recv(dcc->sok, buf, sizeof buf-2, 0);
      if(n < 1)
      {
	 snprintf(buf, sizeof(buf), STARTON" DCC RECV %s (%s) failed. Connection to %s lost.\n", dcc->file, dcc->destfile, dcc->nick);
	 PrintText(dcc->serv->front_session, buf);
	 dcc_close(dcc, STAT_FAILED, FALSE);
	 return;
      }
      if(n > 0)
      {
	 long pos, sec;
	 write(dcc->fp, buf, n);
	 dcc->pos += n;
	 pos = htonl(dcc->pos);
	 send(dcc->sok, (char *)&pos, 4, 0);

	 sec = time(0) - dcc->starttime;
	 if(sec < 1) sec = 1;
	 dcc->cps = dcc->pos / sec;
	 dcc->lasttime = time(0);

	 if(dcc->pos >= dcc->size)
         {
	    dcc_close(dcc, STAT_DONE, FALSE);
	    snprintf(buf, sizeof buf, STARTON" DCC RECV %s (%s) from %s complete (%ld cps).\n", dcc->file, dcc->destfile, dcc->nick, dcc->cps);
	    PrintText(dcc->serv->front_session, buf);
	    return;
	 } else {
	    if(dcc->pos >= (dcc->oldpos + DCC_UPDATE_WINDOW))
	      {
		 update_dcc_recv(dcc);
		 dcc->oldpos = dcc->pos;
	      }
	 }
      }
   } while(is_data_avail(sok));
}

void dcc_connect_finished(struct DCC *dcc, gint sok)
{
   int er;
   char outbuf[300];

   gdk_input_remove(dcc->iotag);

   if(connect(sok, (struct sockaddr *)&dcc->SAddr, sizeof(struct sockaddr_in)) < 0)
   {
      er = errno;
      if(er != EISCONN)
      {
	 snprintf(outbuf, sizeof(outbuf), STARTON" DCC %s connect attempt to %s failed (err=%s).\n",
	      dcctypes[(int)dcc->type], dcc->nick, errorstring(er));
	 PrintText(dcc->serv->front_session, outbuf);
	 dcc->stat = STAT_FAILED;
	 update_dcc_window(dcc->type);
	 return;
      }
   }

   dcc->stat = STAT_ACTIVE;
   switch(dcc->type)
   {
    case TYPE_RECV: 
      dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ,
				 (GdkInputFunction)dcc_read, dcc);
      break;

    case TYPE_CHATRECV: 
      dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ,
				 (GdkInputFunction)dcc_read_chat, dcc);
      dcc->dccchat = malloc(sizeof(struct dcc_chat));
      dcc->dccchat->pos = 0;
      break;

    case TYPE_DRAWRECV: 
      dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ,
				 (GdkInputFunction)dcc_read_draw, dcc);
      dcc_open_draw_window(dcc);
      break;
   }

   update_dcc_window(dcc->type);
   time(&dcc->starttime);

   snprintf(outbuf, sizeof(outbuf), STARTON" DCC %s connection established to %s (%s)\n",
	   dcctypes[(int)dcc->type], dcc->nick, inet_ntoa(dcc->SAddr.sin_addr));
   PrintText(dcc->serv->front_session, outbuf);
}

void dcc_connect(struct session *sess, struct DCC *dcc)
{
   if(dcc->stat == STAT_CONNECTING) return;
   dcc->stat = STAT_CONNECTING;
   dcc->sok = net_connect(dcc, dcc->port);
   if(dcc->sok == -1)
   {
      dcc->stat = STAT_FAILED;
      if(sess) PrintText(sess, STARTON" Connection failed.\n");
      update_dcc_window(dcc->type);
      return;
   }
   update_dcc_window(dcc->type);
   dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_WRITE,
			 (GdkInputFunction)dcc_connect_finished, dcc);
}

void update_dcc_recv(struct DCC *dcc)
{
   char pos[14], cps[14];
   gint row;

   if(!dccrwin.window) return;

   row = gtk_clist_find_row_from_data(GTK_CLIST(dccrwin.list), (gpointer)dcc);

   sprintf(pos, "%lu", dcc->pos);
   sprintf(cps, "%lu", dcc->cps);

   gtk_clist_freeze(GTK_CLIST(dccrwin.list));
   gtk_clist_set_text(GTK_CLIST(dccrwin.list), row, 0, dccstat[(int)dcc->stat]);
   gtk_clist_set_text(GTK_CLIST(dccrwin.list), row, 3, pos);
   gtk_clist_set_text(GTK_CLIST(dccrwin.list), row, 4, cps);
   gtk_clist_thaw(GTK_CLIST(dccrwin.list));
}

void update_dcc_send(struct DCC *dcc)
{
   char pos[14], cps[14], ack[14];
   gint row;

   if(!dccswin.window) return;

   row = gtk_clist_find_row_from_data(GTK_CLIST(dccswin.list), (gpointer)dcc);

   sprintf(pos, "%lu", dcc->pos);
   sprintf(cps, "%lu", dcc->cps);
   sprintf(ack, "%lu", dcc->ack);

   gtk_clist_freeze(GTK_CLIST(dccswin.list));
   gtk_clist_set_text(GTK_CLIST(dccswin.list), row, 0, dccstat[(int)dcc->stat]);
   gtk_clist_set_text(GTK_CLIST(dccswin.list), row, 3, pos);
   gtk_clist_set_text(GTK_CLIST(dccswin.list), row, 4, ack);
   gtk_clist_set_text(GTK_CLIST(dccswin.list), row, 5, cps);
   gtk_clist_thaw(GTK_CLIST(dccswin.list));
}

void dcc_send_data(struct DCC *dcc, gint sok)
{
   char buf[2048];
   long len;

   if(sok != -1)
   {
      unsigned long ack;
      len = recv(dcc->sok, (char *)&ack, 4, MSG_PEEK);
      if(len < 1)
      {
	 if(errno == EWOULDBLOCK) return;
       	 snprintf(buf, sizeof(buf), STARTON" DCC SEND %s failed. Connection to %s lost.\n", file_part(dcc->file), dcc->nick);
	 PrintText(dcc->serv->front_session, buf);
	 dcc_close(dcc, STAT_FAILED, FALSE);
	 return;
      }
      if(len < 4) return;
      recv(dcc->sok, (char *)&ack, 4, 0);
      dcc->ack = ntohl(ack);
   }

   if(dcc->pos < dcc->size)
   {
      long sec;

      if(!prefs.fastdccsend)
      {
	 if((dcc->ack + 4096) < dcc->pos) return;
      }

      len = read(dcc->fp, buf, sizeof buf);

      if(send(dcc->sok, buf, len, 0) < 0)
      {
	 switch(errno)
	 {
	  case EWOULDBLOCK:
	  case ENOBUFS:
	    return;
	 }
	 snprintf(buf, sizeof(buf), STARTON" DCC SEND %s failed. Connection to %s lost.\n", file_part(dcc->file), dcc->nick);
	 PrintText(dcc->serv->front_session, buf);
	 dcc_close(dcc, STAT_FAILED, FALSE);
	 return;
      }

      sec = time(0) - dcc->starttime;
      if(sec < 1) sec = 1;
      dcc->cps = dcc->pos / sec;
      dcc->pos += len;
      dcc->lasttime = time(0);
   }

   if(dcc->pos >= dcc->size  &&  dcc->ack >= dcc->size)
   {
      dcc_close(dcc, STAT_DONE, FALSE);
      snprintf(buf, sizeof(buf), STARTON" DCC SEND %s to %s complete (%ld cps).\n", file_part(dcc->file), dcc->nick, dcc->cps);
      PrintText(dcc->serv->front_session, buf);
   } else {
      if(dcc->pos >= (dcc->oldpos + DCC_UPDATE_WINDOW) ||
	 dcc->ack >= (dcc->oldack + DCC_UPDATE_WINDOW))
      {
	 dcc->oldack = dcc->ack;
	 dcc->oldpos = dcc->pos;
	 update_dcc_send(dcc);
      }
   }
}

void dcc_accept(struct DCC *dcc, gint sokk)
{
   struct sockaddr_in CAddr;
   int len, sok;
   char tbuf[256];
   
   len = sizeof(CAddr);
   sok = accept(dcc->sok, (struct sockaddr *)&CAddr, &len);
   gtk_input_remove(dcc->iotag);
   close(dcc->sok);
   if(sok < 0)
   {
      dcc_close(dcc, STAT_FAILED, FALSE);
      return;
   }
   fcntl(sok, F_SETFL, O_NONBLOCK);
   dcc->sok = sok;
   dcc->addr = ntohl(CAddr.sin_addr.s_addr);
   memcpy((char *)&dcc->SAddr.sin_addr, (char *)&CAddr.sin_addr, sizeof(struct in_addr));
   dcc->stat = STAT_ACTIVE;
   time(&dcc->starttime);
   switch(dcc->type)
   {
    case TYPE_SEND:
      dcc->iotag = gdk_input_add(sok, GDK_INPUT_READ,
			      (GdkInputFunction)dcc_send_data, dcc);
      dcc_send_data(dcc, -1);
      break;

    case TYPE_CHATSEND:
      dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ,
			 (GdkInputFunction)dcc_read_chat, dcc);
      dcc->dccchat = malloc(sizeof(struct dcc_chat));
      dcc->dccchat->pos = 0;
      break;

    case TYPE_DRAWSEND:
      dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ,
			 (GdkInputFunction)dcc_read_draw, dcc);
      dcc_open_draw_window(dcc);
      break;
   }

   snprintf(tbuf, sizeof(tbuf), STARTON" DCC %s connection established from %s (%s)\n",
	   dcctypes[(int)dcc->type], dcc->nick, inet_ntoa(CAddr.sin_addr));
   PrintText(dcc->serv->front_session, tbuf);
}

int dcc_listen_init(struct DCC *dcc)
{
   int len;
   struct sockaddr_in SAddr;

   dcc->sok = socket(AF_INET, SOCK_STREAM, 0);
   if(dcc->sok == -1) return 0;

   memset(&SAddr, 0, sizeof(struct sockaddr_in));

   len = sizeof(SAddr);
   getsockname(dcc->serv->sok, (struct sockaddr *)&SAddr, &len); 

   SAddr.sin_family = AF_INET;
   SAddr.sin_port = 0;

   bind(dcc->sok, (struct sockaddr *)&SAddr, sizeof(SAddr));

   len = sizeof(SAddr);
   getsockname(dcc->sok, (struct sockaddr *) &SAddr, &len); 

   memcpy((char *)&dcc->SAddr, (char *)&SAddr, sizeof(struct sockaddr_in));
   dcc->port = ntohs(SAddr.sin_port);
   dcc->addr = ntohl(SAddr.sin_addr.s_addr);

   fcntl(dcc->sok, F_SETFL, O_NONBLOCK);
   listen(dcc->sok, 1);
   fcntl(dcc->sok, F_SETFL, 0);

   dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ,
			      (GdkInputFunction)dcc_accept, dcc);

   return TRUE;
}

void dcc_send(struct session *sess, char *tbuf, char *to, char *file)
{
   struct stat st;
   struct DCC *dcc = new_dcc();
   if(!dcc) return;

   strcpy(dcc->file, file);
   check_homedir(dcc->file);

   if(stat(dcc->file, &st) != -1)
   {
      if(*file_part(dcc->file))
      {
	 if(st.st_size > 0)
	 {
	    dcc->offertime = time(0);
	    dcc->serv = sess->server;
	    dcc->stat = STAT_QUEUED;
	    dcc->size = st.st_size;
	    dcc->type = TYPE_SEND;
	    dcc->fp = open(dcc->file, O_RDONLY);
	    if(dcc->fp != -1)
	    {
	       if(dcc_listen_init(dcc))
	       {
		  file = dcc->file;
		  while(*file)
		  {
		     if(*file == ' ') *file = '_';
		     file++;
		  }
		  strcpy(dcc->nick, to);
		  if(!prefs.noautoopendccsendwindow)
		    open_dcc_send_window();
		  else
		    update_dcc_send_window();
		  snprintf(tbuf, 255, "PRIVMSG %s :\001DCC SEND %s %lu %ld %ld\001\r\n",
			  to, file_part(dcc->file), dcc->addr, dcc->port, dcc->size);
		  send(sess->server->sok, tbuf, strlen(tbuf), 0);
		  snprintf(tbuf, 255, STARTON" Offering \00311%s \003 to \00311%s\003 \n", file_part(dcc->file), to);
		  PrintText(sess, tbuf);
	       } else {
		  dcc_close(dcc, 0, TRUE);
	       }
	       return;
	    }
	 }
      }
   }
   snprintf(tbuf, 255, STARTON" Cannot access %s\n", dcc->file);
   PrintText(sess, tbuf);
   dcc_close(dcc, 0, TRUE);
}

struct DCC *find_dcc_from_port(int port, int type)
{
   struct DCC *dcc;
   GSList *list = dcc_list;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if(dcc->port == port &&
	 dcc->stat == STAT_QUEUED &&
	 dcc->type == type)
	return dcc;
      list = list->next;
   }
   return 0;  
}

struct DCC *find_dcc(char *nick, char *file, int type)
{
   GSList *list = dcc_list;
   struct DCC *dcc;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if(!strcasecmp(nick, dcc->nick))
      {
	 if(type == -1 || dcc->type == type)
	 {
	    if(!file[0]) return dcc;
	    if(!strcasecmp(file, file_part(dcc->file))) return dcc;
	    if(!strcasecmp(file, dcc->file)) return dcc;
	 }
      }
      list = list->next;
   }
   return 0;
}

void dcc_abort(struct DCC *dcc)
{
   char tbuf[256];
   if(dcc)
   {
      switch(dcc->stat)
      {
       case STAT_QUEUED:
       case STAT_CONNECTING:
       case STAT_ACTIVE:
	  dcc_close(dcc, STAT_ABORTED, FALSE);
	  snprintf(tbuf, sizeof tbuf, STARTON" DCC %s \00311%s \003 to \00311%s \003 aborted.\n",
		  dcctypes[(int)dcc->type], file_part(dcc->file), dcc->nick);
	  PrintText(dcc->serv->front_session, tbuf);
       	  break;
       default:
	  dcc_close(dcc, 0, TRUE);
      }
   }
}

void dcc_resume(struct DCC *dcc)
{
   if(dcc)
   {
     if(dcc->stat == STAT_QUEUED && dcc->resumable)
     {
	char tbuf[256];
	snprintf(tbuf, sizeof tbuf, "PRIVMSG %s :\001DCC RESUME %s %ld %lu\001\r\n",
	      dcc->nick, dcc->file, dcc->port, dcc->resumable);
	send(dcc->serv->sok, tbuf, strlen(tbuf), 0);
     }
   }
}

void dcc_get(struct DCC *dcc)
{
   if(dcc)
   {
      switch(dcc->stat)
      {
	 case STAT_QUEUED:
	   dcc->resumable = 0;
	   dcc->pos = 0;
	   dcc_connect(0, dcc);
	   break;
	 case STAT_DONE:
	 case STAT_FAILED:
	 case STAT_ABORTED:
	   dcc_close(dcc, 0, TRUE);
	   break;
      }
   }
}

void dcc_get_nick(struct session *sess, char *nick)
{
   struct DCC *dcc;
   GSList *list = dcc_list;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if(!strcasecmp(nick, dcc->nick))
      {
	   if(dcc->stat == STAT_QUEUED && dcc->type == TYPE_RECV)
	   {
	      dcc->resumable = 0;
	      dcc->pos = 0;
	      dcc_connect(sess, dcc);
	      return;
	   }
      }
      list = list->next;
   }
   if(sess) PrintText(sess, STARTON" No such DCC offer.\n");
}

struct DCC *new_dcc(void)
{
   struct DCC *dcc = malloc(sizeof(struct DCC));
   if(!dcc) return 0;
   memset(dcc, 0, sizeof(struct DCC));
   dcc->sok = -1;
   dcc->fp = -1;
   dcc_list = g_slist_append(dcc_list, dcc);
   return(dcc);
}

void dcc_chat(struct session *sess, char *nick)
{
   char outbuf[256];
   struct DCC *dcc;

   dcc = find_dcc(nick, "", TYPE_CHATSEND);
   if(dcc)
   {
      switch(dcc->stat)
      {
       case STAT_ACTIVE:
       case STAT_QUEUED:
       case STAT_CONNECTING:
	 snprintf(outbuf, sizeof(outbuf), STARTON" Already offering CHAT to %s\n", nick);
	 PrintText(sess, outbuf);
	 return;
       case STAT_ABORTED:
       case STAT_FAILED:
	 dcc_close(dcc, 0, TRUE);
      }
   }

   dcc = find_dcc(nick, "", TYPE_CHATRECV);
   if(dcc)
   {
      switch(dcc->stat)
      {
	 case STAT_QUEUED:
	   dcc_connect(0, dcc);
	   break;
	 case STAT_FAILED:
	 case STAT_ABORTED:
	   dcc_close(dcc, 0, TRUE);
      }
      return;
   }

   /* offer DCC CHAT */

   dcc = new_dcc();
   if(!dcc) return;
   dcc->offertime = time(0);
   dcc->serv = sess->server;
   dcc->stat = STAT_QUEUED;
   dcc->type = TYPE_CHATSEND;
   time(&dcc->starttime);
   strcpy(dcc->nick, nick);
   if(dcc_listen_init(dcc))
   {
      snprintf(outbuf, sizeof(outbuf), "PRIVMSG %s :\001DCC CHAT chat %lu %ld\001\r\n",
		 nick, dcc->addr, dcc->port);
      send(dcc->serv->sok, outbuf, strlen(outbuf), 0);
      sprintf(outbuf, STARTON" Offering DCC CHAT to %s\n", nick);
      PrintText(sess, outbuf);
   } else
      dcc_close(dcc, 0, TRUE);
}

void dcc_malformed(struct session *sess, char *nick, char *data)
{
   char tbuf[1024];
   snprintf(tbuf, sizeof tbuf,
	    STARTON" Received a malformed DCC request from %s\n"
	    STARTON" Exact contents: \"%s\"\n",
	    nick, data);
   PrintText(sess, tbuf);
}

void handle_dcc(struct session *sess, char *outbuf, char *nick, char *word[], char *word_eol[])
{
   struct DCC *dcc;
   char *type = word[5];
   int port;

   if(!strcasecmp(type, "DRAW"))
   {
      unsigned long addr;
      long port = atol(word[9]);

      sscanf(word[8], "%lu", &addr);
      addr = ntohl(addr);
      if(!addr || port < 1024)
      {
	 dcc_malformed(sess, nick, word_eol[4]+2);
	 return;
      }

      dcc = new_dcc();
      if(dcc)
      {
	 dcc->pos = atoi(word[6]); /* x */
	 dcc->size = atoi(word[7]); /* y */
	 dcc->serv = sess->server;
	 dcc->type = TYPE_DRAWRECV;
	 dcc->stat = STAT_QUEUED;
	 dcc->addr = addr;
	 dcc->port = port;
	 strcpy(dcc->nick, nick);
	 snprintf(outbuf, 255,
		 STARTON" Received a DCC DRAW offer from %s\n"
		 STARTON" Type \00311/DCC DRAW %s \003 to accept\n",
		 nick, nick);
	 PrintText(sess->server->front_session, outbuf);
	 return;
      }  
   }

   if(!strcasecmp(type, "CHAT"))
   {
      unsigned long addr;
      long port = atol(word[8]);

      sscanf(word[7], "%lu", &addr);
      addr = ntohl(addr);

      if(!addr || port < 1024)
      {
	 dcc_malformed(sess, nick, word_eol[4]+2);
	 return;
      }

      dcc = find_dcc(nick, "", TYPE_CHATSEND);
      if(dcc) dcc_close(dcc, 0, TRUE);

      dcc = new_dcc();
      if(dcc)
      {
	 dcc->serv = sess->server;
	 dcc->type = TYPE_CHATRECV;
	 dcc->stat = STAT_QUEUED;
	 dcc->addr = addr;
	 dcc->port = port;
	 strcpy(dcc->nick, nick);
	 snprintf(outbuf, 255, STARTON" Received a DCC CHAT offer from %s\n", nick);
	 PrintText(sess->server->front_session, outbuf);
	 if(prefs.autodccchat) dcc_connect(0, dcc);
	 return;
      }
   }

   if(!strcasecmp(type, "RESUME"))
   {
      port = atol(word[7]);
      dcc = find_dcc_from_port(port, TYPE_SEND);
      if(dcc && dcc->stat == STAT_QUEUED)
      {
	 sscanf(word[8], "%lu", &dcc->resumable);
	 dcc->pos = dcc->resumable;
	 dcc->ack = dcc->resumable;
	 lseek(dcc->fp, dcc->pos, SEEK_SET);
	 snprintf(outbuf, 255, "PRIVMSG %s :\001DCC ACCEPT %s %ld %lu\001\r\n",
		 dcc->nick, file_part(dcc->file), dcc->port, dcc->resumable);
	 send(dcc->serv->sok, outbuf, strlen(outbuf), 0);
	 sprintf(outbuf,
		 STARTON" \00311%s \003 has requested to resume \00311%s \003 from \00311%lu\003 .\n",
		 nick, file_part(dcc->file), dcc->pos);
	 PrintText(sess, outbuf);
	 return;
      }
   }

   if(!strcasecmp(type, "ACCEPT"))
   {
      port = atol(word[7]);
      dcc = find_dcc_from_port(port, TYPE_RECV);
      if(dcc && dcc->stat == STAT_QUEUED)
      {
	 dcc_connect(0, dcc);
	 return;
      }
   }

   if(!strcasecmp(type, "SEND"))
   {
      unsigned long addr;
      char *file= file_part(word[6]);
      long port = atol(word[8]);
      long size = atol(word[9]);

      sscanf(word[7], "%lu", &addr);
      addr = ntohl(addr);

      if(!addr || !size || port < 1024)
      {
	 dcc_malformed(sess, nick, word_eol[4]+2);
	 return;
      }

      dcc = new_dcc();
      if(dcc)
      {
	 struct stat st;

         strcpy(dcc->destfile, prefs.dccdir);	 
	 if(prefs.dccdir[strlen(prefs.dccdir)-1] != '/')
	   strcat(dcc->destfile, "/");
	 if (prefs.dccwithnick) {
	    strcat(dcc->destfile, nick);
	    strcat(dcc->destfile, ".");
	 }
	 strcat(dcc->destfile, file);
	 strcpy(dcc->file, file);

	 if(stat(dcc->destfile, &st) != -1)
	 {
	   if(st.st_size < 1)
	   {
	      snprintf(outbuf, 255, STARTON" Cannot accept %s from %s, filesize is zero.\n", file, nick);
	      PrintText(sess->server->front_session, outbuf);
	      dcc_close(dcc, 0, TRUE);
	      return;
	   }
	   dcc->resumable = st.st_size;
	 } else
	   dcc->resumable = 0;
	 if(st.st_size == size) dcc->resumable = 0;
	 dcc->serv = sess->server;
	 dcc->type = TYPE_RECV;
	 dcc->stat = STAT_QUEUED;
	 dcc->addr = addr;
	 dcc->port = port;
	 dcc->size = size;
	 strcpy(dcc->nick, nick);
	 if(!prefs.noautoopendccrecvwindow)
	   open_dcc_recv_window();
	 else
	   update_dcc_recv_window();
	 if(prefs.autodccsend) dcc_connect(sess, dcc);
      }
      snprintf(outbuf, 255, STARTON" \00311%s \003 has offered \00311%s \003 (\00311%ld \003 bytes)\n",
	      nick, file, size);
   } else
     sprintf(outbuf, STARTON" Received '%s' from %s\n", 
	    word_eol[4]+2, nick);
   PrintText(sess->server->front_session, outbuf);
}

void dcc_show_list(struct session *sess, char *outbuf)
{
   int i = 0;
   struct DCC *dcc;
   GSList *list = dcc_list;

   PrintText(sess,
	     STARTON" \00308,02 Type  To/From    Status  Size    Pos     File   \003 \n"
	     STARTON"\002 \00303-------------------------------------------------\002\003 \n"
	     );
   while(list)
   {
      dcc = (struct DCC *)list->data;
      i++;
      snprintf(outbuf, 255, STARTON"  %-5.5s %-10.10s %-7.7s %-7ld %-7ld %s\n",
	      dcctypes[(int)dcc->type], dcc->nick, dccstat[(int)dcc->stat],
	      dcc->size, dcc->pos, dcc->file);
      PrintText(sess, outbuf);
      list = list->next;
   }
   if(!i) PrintText(sess, STARTON" No active DCCs\n");
}

void close_dcc_recv_window(void)
{
   if(dccrwin.window)
   {
      gtk_widget_destroy(dccrwin.window);
      dccrwin.window = 0;
   }
}

void update_dcc_recv_window(void)
{
   struct DCC *dcc;
   GSList *list = dcc_list;
   gchar *new[1][6];
   char size[16];
   char pos[16];
   char cps[16];
   gint row;
   int selrow;

   if(!dccrwin.window) return;

   selrow = gtkutil_clist_selection(dccrwin.list);

   gtk_clist_clear((GtkCList*)dccrwin.list);
   new[0][2] = size;
   new[0][3] = pos;
   new[0][4] = cps;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if(dcc->type == TYPE_RECV)
      {
	 new[0][0] = dccstat[(int)dcc->stat];
	 new[0][1] = dcc->file;
	 new[0][5] = dcc->nick;
	 sprintf(size, "%lu", dcc->size);
	 if(dcc->stat == STAT_QUEUED)
	   sprintf(pos, "%lu", dcc->resumable);
	 else
	   sprintf(pos, "%lu", dcc->pos);
	 sprintf(cps, "%lu", dcc->cps);
	 row = gtk_clist_append(GTK_CLIST(dccrwin.list), new[0]);
	 gtk_clist_set_row_data(GTK_CLIST(dccrwin.list), row, (gpointer)dcc);
      }
      list = list->next;
   }
   if(selrow != -1)
     gtk_clist_select_row((GtkCList*)dccrwin.list, selrow, 0);
}

void dcc_info(struct DCC *dcc)
{
   char tbuf[256];
   snprintf(tbuf, 255, "      File: %s\n"
	         "      From: %s\n"
	         "      Size: %ld\n"
	         "      Port: %ld\n"
	         " IP Number: %s\n"
	         "Start Time: %s",
	   dcc->file,
	   dcc->nick,
	   dcc->size,
	   dcc->port,
	   inet_ntoa(dcc->SAddr.sin_addr),
	   ctime(&dcc->starttime));
   gtkutil_simpledialog(tbuf);
}

void resume_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;

   row = gtkutil_clist_selection(dccrwin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dccrwin.list), row);
      dcc_resume(dcc);
   }
}

void abort_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;

   row = gtkutil_clist_selection(dccrwin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dccrwin.list), row);
      dcc_abort(dcc);
   }
}

void accept_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;

   row = gtkutil_clist_selection(dccrwin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dccrwin.list), row);
      dcc_get(dcc);
   }
}

void info_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;

   row = gtkutil_clist_selection(dccrwin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dccrwin.list), row);
      if(dcc) dcc_info(dcc);
   }
}

void recv_row_selected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even)
{
     if(even && even->type == GDK_2BUTTON_PRESS) accept_clicked(0, 0);
}

void open_dcc_recv_window(void)
{  
   GtkWidget *vbox, *bbox;
   static gchar *titles[] = { "Status","File","Size","Position","CPS","From" };

   if(dccrwin.window)
   {
      update_dcc_recv_window();
      return;
   }

   dccrwin.window = gtkutil_window_new("X-Chat: DCC Receive List", 500, 0,
				       close_dcc_recv_window, 0);

   vbox = gtk_vbox_new(FALSE, 0);
   gtk_container_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(dccrwin.window), vbox);
   gtk_widget_show(vbox);

   dccrwin.list = gtkutil_clist_new(6, titles, vbox, GTK_POLICY_ALWAYS,
				    recv_row_selected, 0,
				    0, 0);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 0, 65);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 1, 100);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 2, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 3, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 4, 50);

   bbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);
   gtk_widget_show(bbox);

   gtkutil_button_new("Accept", 0, 0, accept_clicked, 0, bbox);

   gtkutil_button_new("Resume", 0, 0, resume_clicked, 0, bbox);

   gtkutil_button_new("Abort", 0, 0, abort_clicked, 0, bbox);

   gtkutil_button_new("Info", 0, 0, info_clicked, 0, bbox);

   gtk_widget_show(dccrwin.window);
   update_dcc_recv_window();
}

void close_dcc_send_window(void)
{
   if(dccswin.window)
   {
      gtk_widget_destroy(dccswin.window);
      dccswin.window = 0;
   }
}

void update_dcc_send_window(void)
{
   struct DCC *dcc;
   GSList *list = dcc_list;
   gchar *new[1][7];
   char size[14];
   char pos[14];
   char cps[14];
   char ack[14];
   gint row;
   int selrow;

   if(!dccswin.window) return;

   selrow = gtkutil_clist_selection(dccswin.list);

   gtk_clist_clear((GtkCList*)dccswin.list);
   new[0][2] = size;
   new[0][3] = pos;
   new[0][4] = ack;
   new[0][5] = cps;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if(dcc->type == TYPE_SEND)
      {
	 new[0][0] = dccstat[(int)dcc->stat];
	 new[0][1] = file_part(dcc->file);
	 new[0][6] = dcc->nick;
	 sprintf(size, "%lu", dcc->size);
	 sprintf(pos, "%lu", dcc->pos);
	 sprintf(cps, "%lu", dcc->cps);
	 sprintf(ack, "%lu", dcc->ack);
	 row = gtk_clist_append(GTK_CLIST(dccswin.list), new[0]);
	 gtk_clist_set_row_data(GTK_CLIST(dccswin.list), row, (gpointer)dcc);
      }
      list = list->next;
   }
   if(selrow != -1)
     gtk_clist_select_row((GtkCList*)dccswin.list, selrow, 0);
}

void send_row_selected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even)
{
   if(even && even->type == GDK_2BUTTON_PRESS)
   {
      struct DCC *dcc;

      dcc = gtk_clist_get_row_data(GTK_CLIST(clist), row);
      if(dcc)
      {
	 switch(dcc->stat)
	 {
	  case STAT_FAILED:
	  case STAT_ABORTED:
	  case STAT_DONE:
	    dcc_abort(dcc);
	 }
      }
   }
}

void info_send_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;

   row = gtkutil_clist_selection(dccswin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dccswin.list), row);
      if(dcc) dcc_info(dcc);
   }
}

void abort_send_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;
   
   row = gtkutil_clist_selection(dccswin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dccswin.list), row);
      dcc_abort(dcc);
   }
}

void open_dcc_send_window(void)
{  
   GtkWidget *vbox, *bbox;
   static gchar *titles[] = { "Status","File","Size","Position","Ack","CPS","To" };

   if(dccswin.window)
   {
      update_dcc_send_window();
      return;
   }

   dccswin.window = gtkutil_window_new("X-Chat: DCC Send List", 550, 0,
				       close_dcc_send_window, 0);

   vbox = gtk_vbox_new(FALSE, 0);
   gtk_container_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(dccswin.window), vbox);
   gtk_widget_show(vbox);

   dccswin.list = gtkutil_clist_new(7, titles, vbox, GTK_POLICY_ALWAYS,
				    send_row_selected, 0,
				    0, 0);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 0, 65);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 1, 100);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 2, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 3, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 4, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 5, 50);

   bbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);
   gtk_widget_show(bbox);

   gtkutil_button_new("Abort", 0, 0, abort_send_clicked, 0, bbox);

   gtkutil_button_new("Info", 0, 0, info_send_clicked, 0, bbox);

   gtk_widget_show(dccswin.window);
   update_dcc_send_window();
}


/* DCC CHAT GUIs BELOW */


void abort_chat_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;

   row = gtkutil_clist_selection(dcccwin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dcccwin.list), row);
      dcc_abort(dcc);
   }
}

void update_dcc_chat_window(void)
{
   struct DCC *dcc;
   GSList *list = dcc_list;
   gchar *new[1][5];
   char pos[14];
   char siz[14];
   gint row;
   int selrow;

   if(!dcccwin.window) return;

   selrow = gtkutil_clist_selection(dcccwin.list);

   gtk_clist_clear((GtkCList*)dcccwin.list);
   new[0][2] = pos;
   new[0][3] = siz;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if((dcc->type == TYPE_CHATSEND || dcc->type == TYPE_CHATRECV))
      {
	 new[0][0] = dccstat[(int)dcc->stat];
	 new[0][1] = dcc->nick;
	 sprintf(pos, "%lu", dcc->pos);
	 sprintf(siz, "%lu", dcc->size);
	 new[0][4] = ctime(&dcc->starttime);
	 row = gtk_clist_append(GTK_CLIST(dcccwin.list), new[0]);
	 gtk_clist_set_row_data(GTK_CLIST(dcccwin.list), row, (gpointer)dcc);
      }
      list = list->next;
   }
   if(selrow != -1)
     gtk_clist_select_row((GtkCList*)dcccwin.list, selrow, 0);
}

void close_dcc_chat_window(void)
{
   if(dcccwin.window)
   {
      gtk_widget_destroy(dcccwin.window);
      dcccwin.window = 0;
   }
}

void open_dcc_chat_window(void)
{  
   GtkWidget *vbox, *bbox;
   static gchar *titles[] = { "Status","To/From","Recv","Sent","StartTime" };

   if(dcccwin.window)
   {
      update_dcc_chat_window();
      return;
   }

   dcccwin.window = gtkutil_window_new("X-Chat: DCC Chat List", 550, 0,
				       close_dcc_chat_window, 0);

   vbox = gtk_vbox_new(FALSE, 0);
   gtk_container_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(dcccwin.window), vbox);
   gtk_widget_show(vbox);

   dcccwin.list = gtkutil_clist_new(5, titles, vbox, GTK_POLICY_ALWAYS,
				    0, 0,
				    0, 0);
   gtk_clist_set_column_width(GTK_CLIST(dcccwin.list), 0, 65);
   gtk_clist_set_column_width(GTK_CLIST(dcccwin.list), 1, 100);
   gtk_clist_set_column_width(GTK_CLIST(dcccwin.list), 2, 65);
   gtk_clist_set_column_width(GTK_CLIST(dcccwin.list), 3, 65);

   bbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);
   gtk_widget_show(bbox);

   gtkutil_button_new("Abort", 0, 0, abort_chat_clicked, 0, vbox);

   gtk_widget_show(dcccwin.window);
   update_dcc_chat_window();
}
/*

Here is a description of the mIRC DCC Resume Protocol.

User1 is sending the file.
User2 is receiving the file.

To initiate a DCC Send, User1 sends:

PRIVMSG User2 :DCC SEND filename ipaddress port filesize

Normally, if User2 accepts the DCC Send request, User2 connects to the
address 
and port number given by User1 and the file transfer begins.

If User2 chooses to resume a file transfer of an existing file, the
following 
negotiation takes place:

User2 sends:

PRIVMSG User1 :DCC RESUME filename port position

filename = the filename sent by User1.
port = the port number sent by User1.
position = the current size of the file that User2 has.

User1 then responds:

PRIVMSG User2 :DCC ACCEPT filename port position

This is simply replying with the same information that User2 sent as 
acknowledgement.

At this point User2 connects to User1 address and port and the transfer
begins 
from the specified position.

NOTE: the newer versions of mIRC actually ignore the filename as it is 
redundant since the port uniquely identifies the connection. However, to
remain 
compatible mIRC still sends a filename as "file.ext" in both RESUME and
ACCEPT.

----

*/
