/*
 * Copyright (c) 1980, 1990 Regents of the University of California. All
 * rights reserved.
 * 
 * This code is derived from software contributed to Berkeley by Robert Elz at
 * The University of Melbourne.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution. 3. All advertising
 * materials mentioning features or use of this software must display the
 * following acknowledgement: This product includes software developed by the
 * University of California, Berkeley and its contributors. 4. Neither the
 * name of the University nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific
 * prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
char copyright[] =
"@(#) Copyright (c) 1980, 1990 Regents of the University of California.\n\
 All rights reserved.\n";
#endif /* not lint */

#ifndef lint
static char rcsid[] = "$Id: setquota.c,v 1.1 1998/06/11 09:58:06 mvw Exp mvw $";
#endif /* not lint */

/*
 * set disk quota from command line 
 */
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/quota.h>
#include <errno.h>
#include <mntent.h>
#include <pwd.h>
#include <grp.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>

extern char *qfextension[];
char *quotagroup = QUOTAGROUP;

struct quotause {
   struct quotause *next;
   long flags;
   struct dqblk dqblk;
   char fsname[MAXPATHLEN + 1];
   char qfname[1];   /* actually longer */
} *getprivs();

#define   FOUND   0x01

main(int  argc, char **argv)
{
   struct quotause *qup, *protoprivs, *curprivs;
   extern char *optarg;
   extern int optind;
   long id, protoid;
   int quotatype, tmpfd;
   char *protoname, ch;
   int tflag = 0, pflag = 0;
   char *fsys;
   char gotfs = 0;
   __u32 bh, bs, ih, is;
   char *cnum;

   if (getuid()) {
      fprintf(stderr, "edquota: permission denied\n");
      exit(1);
   }

   quotatype = USRQUOTA;
   argv++;

   if (strcmp(*argv,"-g")==0) {
     quotatype = GRPQUOTA;
     argv++;
     argc--;
   }
   if (strcmp(*argv,"-u")==0) {
     quotatype = USRQUOTA;
     argv++;
     argc--;
   }

   if (argc != 7)
      usage();

   if ((id = getentry(*argv, quotatype)) == -1)
     exit(-1);

   argv++;
   fsys = *argv;

   argv++;
   if (alldigits(*argv))
     bs=atoi(*argv);
   else {
     fprintf(stderr,"invalid number: block-soft\n");
     exit(-1);
   }
     
   argv++;
   if (alldigits(*argv))
     bh=atoi(*argv);
   else {
     fprintf(stderr,"invalid number: block-hard\n");
     exit(-1);
   }
     
   argv++;
   if (alldigits(*argv))
     is=atoi(*argv);
   else {
     fprintf(stderr,"invalid number: inode-soft\n");
     exit(-1);
   }
     
   argv++;
   if (alldigits(*argv))
     ih=atoi(*argv);
   else {
     fprintf(stderr,"invalid number: inode-hard\n");
     exit(-1);
   }
     
   curprivs = getprivs(id, quotatype);

   for (qup = curprivs; qup; qup = qup->next) {
     if (strcmp(qup->fsname,fsys)==0) {
       gotfs++;
       /* 
       printf ("FS: %s  BS:%d, BH:%d, IS:%d, IH:%d\n",
               qup->fsname,
               qup->dqblk.dqb_bsoftlimit,
	       qup->dqblk.dqb_bhardlimit,
	       qup->dqblk.dqb_isoftlimit,
	       qup->dqblk.dqb_ihardlimit);
       printf ("NEW:           BS:%d, BH:%d, IS:%d, IH:%d\n",bs,bh,is,ih);
       */ 
       qup->dqblk.dqb_bsoftlimit=bs;
       qup->dqblk.dqb_bhardlimit=bh;
       qup->dqblk.dqb_isoftlimit=is;
       qup->dqblk.dqb_ihardlimit=ih;
     }
   }
   if (! gotfs) {
     fprintf (stderr,"file system not found\n");
     exit(-1);
   }

   putprivs(id, quotatype, curprivs);
   freeprivs(curprivs);

   exit(0);
}

usage(void)
{
   fprintf(stderr, "%s", 
          "Usage: setquota [-u | -g] name filesystem block-soft block-hard inode-soft inode-hard\n");
   exit(1);
}

/*
 * This routine converts a name for a particular quota type to an identifier.
 */
getentry(char *name, int quotatype)
{
   struct passwd  *pw;
   struct group   *gr;

   switch (quotatype) {
      case USRQUOTA:
         if (pw = getpwnam(name))
            return (pw->pw_uid);
         fprintf(stderr, "%s: no such user\n", name);
         break;
      case GRPQUOTA:
         if (gr = getgrnam(name))
            return (gr->gr_gid);
         fprintf(stderr, "%s: no such group\n", name);
         break;
      default:
         fprintf(stderr, "%d: unknown quota type\n", quotatype);
         break;
   }
   return (-1);
}

/*
 * Collect the requested quota information.
 */
struct quotause *getprivs(long id, int quotatype)
{
   struct mntent *mnt;
   struct quotause *qup, *quptail;
   FILE *fp;
   struct quotause *quphead;
   int qcmd, qupsize, fd;
   char *qfpathname;
   static int warned = 0;
   extern int errno;

   fp = setmntent(MNTTAB, "r");
   quphead = (struct quotause *) 0;
   qcmd = QCMD(Q_GETQUOTA, quotatype);
   while ((mnt = getmntent(fp)) != (struct mntent *) 0) {
      if (!hasquota(mnt, quotatype, &qfpathname))
         continue;
      qupsize = sizeof(*qup) + strlen(qfpathname);
      if ((qup = (struct quotause *) malloc(qupsize)) == (struct quotause *)NULL) {
         fprintf(stderr, "edquota: out of memory\n");
         exit(2);
      }
      if (quotactl(qcmd, mnt->mnt_fsname, id, (caddr_t) &qup->dqblk) != 0) {
         if ((errno == EOPNOTSUPP || errno == ENOSYS) && !warned) {
            warned++;
            fprintf(stderr, "Warning: %s\n",
            "Quotas are not compiled into this kernel");
            sleep(3);
         }
         if ((fd = open(qfpathname, O_RDONLY)) < 0) {
            fd = open(qfpathname, O_RDWR | O_CREAT, 0640);
            if (fd < 0 && errno != ENOENT) {
               perror(qfpathname);
               free(qup);
               continue;
            }
            fprintf(stderr, "Creating quota file %s\n", qfpathname);
            sleep(3);
            (void) fchown(fd, getuid(),
                   getentry(quotagroup, GRPQUOTA));
            (void) fchmod(fd, 0640);
         }
         lseek(fd, (long) (id * sizeof(struct dqblk)), L_SET);
         switch (read(fd, &qup->dqblk, sizeof(struct dqblk))) {
            case 0:/* EOF */
               /*
                * Convert implicit 0 quota (EOF) into an
                * explicit one (zero'ed dqblk)
                */
               bzero((caddr_t) & qup->dqblk,
                     sizeof(struct dqblk));
               break;

            case sizeof(struct dqblk):   /* OK */
               break;

            default:   /* ERROR */
               fprintf(stderr, "edquota: read error in ");
               perror(qfpathname);
               close(fd);
               free(qup);
               continue;
         }
         close(fd);
      }
      strcpy(qup->qfname, qfpathname);
      strcpy(qup->fsname, mnt->mnt_fsname);
      if (quphead == NULL)
         quphead = qup;
      else
         quptail->next = qup;
      quptail = qup;
      qup->next = 0;
   }
   endmntent(fp);
   return (quphead);
}

/*
 * Store the requested quota information.
 */
putprivs(long id, int quotatype, struct quotause *quplist)
{
   struct quotause *qup;
   int qcmd, fd;

   qcmd = QCMD(Q_SETQUOTA, quotatype);
   for (qup = quplist; qup; qup = qup->next) {
      if (quotactl(qcmd, qup->fsname, id, (caddr_t) &qup->dqblk) == 0)
         continue;
      if ((fd = open(qup->qfname, O_WRONLY)) < 0) {
         perror(qup->qfname);
      } else {
         lseek(fd, (long) id * (long) sizeof(struct dqblk), 0);
         if (write(fd, &qup->dqblk, sizeof(struct dqblk)) !=
             sizeof(struct dqblk)) {
            fprintf(stderr, "edquota: ");
            perror(qup->qfname);
         }
         close(fd);
      }
   }
}

/*
 * Convert seconds to ASCII times.
 */
char *cvtstoa(time_t time)
{
   static char buf[20];

   if (time % (24 * 60 * 60) == 0) {
      time /= 24 * 60 * 60;
      sprintf(buf, "%d day%s", time, time == 1 ? "" : "s");
   } else if (time % (60 * 60) == 0) {
      time /= 60 * 60;
      sprintf(buf, "%d hour%s", time, time == 1 ? "" : "s");
   } else if (time % 60 == 0) {
      time /= 60;
      sprintf(buf, "%d minute%s", time, time == 1 ? "" : "s");
   } else
      sprintf(buf, "%d second%s", time, time == 1 ? "" : "s");
   return (buf);
}

/*
 * Convert ASCII input times to seconds.
 */
cvtatos(time_t time, char *units, time_t *seconds)
{
   if (bcmp(units, "second", 6) == 0)
      *seconds = time;
   else if (bcmp(units, "minute", 6) == 0)
      *seconds = time * 60;
   else if (bcmp(units, "hour", 4) == 0)
      *seconds = time * 60 * 60;
   else if (bcmp(units, "day", 3) == 0)
      *seconds = time * 24 * 60 * 60;
   else {
      printf("%s: bad units, specify %s\n", units,
             "days, hours, minutes, or seconds");
      return (0);
   }
   return (1);
}

/*
 * Free a list of quotause structures.
 */
freeprivs(struct quotause *quplist)
{
   struct quotause *qup, *nextqup;

   for (qup = quplist; qup; qup = nextqup) {
      nextqup = qup->next;
      free(qup);
   }
}

/*
 * Check whether a string is completely composed of digits.
 */
alldigits(char *s)
{
   int c;

   c = *s++;
   do {
      if (!isdigit(c))
         return (0);
   } while (c = *s++);
   return (1);
}

