/*****************************************************************************/
/*  gftp.c - the main user interface for the ftp program                     */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"
#include "pix/connect.xpm"
#include "pix/left.xpm"
#include "pix/right.xpm"
#include "pix/stop.xpm"
#include "pix/up.xpm"
#include "pix/down.xpm"
#include "options.h"

char version[] = "gFTP " VERSION;

static void menu_exit (GtkWidget *widget, gpointer data);
static gint delete_event (GtkWidget *widget, GdkEvent *event, gpointer data);
static void doexit (GtkWidget *widget, gpointer data);
static void destroy (GtkWidget *widget, gpointer data);
static void sig_child (int signo);
static void init_gftp (int argc, char *argv[], GtkWidget *parent);
static void usage (void);
static GtkWidget *CreateFTPWindows (GtkWidget *ui);
static GtkWidget *CreateFTPWindow (struct ftp_window_data *wdata);
static void setup_column (GtkWidget *listbox, int column, int width);
static void listbox_drag (GtkWidget *widget, GdkDragContext *context, GtkSelectionData *selection_data, guint info, guint32 clk_time, gpointer data);
static void listbox_get_drag_data (GtkWidget *widget, GdkDragContext *context, gint x, 
	gint y, GtkSelectionData *selection_data, guint info, guint32 clk_time, 
	struct ftp_window_data *wdata);
static void setanonymous (gpointer data, int menuitem, GtkWidget *widget);
static void change_setting (struct ftp_window_data *wdata, int menuitem, GtkWidget *checkmenu);
static void selectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data);
static void unselectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data);
static void selectall (struct ftp_window_data *wdata);
static void selectallfiles (struct ftp_window_data *wdata);
static void deselectall (struct ftp_window_data *wdata);
static gint log_click (GtkWidget *widget, GdkEventButton *event, gpointer data);
static gint list_dblclick (GtkWidget *widget, GdkEventButton *event, gpointer data);
static gint list_enter (GtkWidget *widget, GdkEventKey *event, gpointer data);
static void chfunc (struct ftp_window_data *wdata);
static int chdirfunc (struct ftp_window_data *wdata);
static int chdiredit (GtkWidget *widget, struct ftp_window_data *wdata);
static void disconn (struct ftp_window_data *wdata);
static void selectdl (GtkCList *clist, gint row, gint column, GdkEventButton *event,
	gpointer data);
static void unselectdl (GtkCList *clist, gint row, gint column, GdkEventButton *event,
	gpointer data);
static void stop_transfer (GtkWidget *widget, gpointer data);
static void compare_windows (gpointer data);
static gint update_downloads (gpointer data);
static void check_done_process (void);
static void do_upload (GtkWidget *widget, struct dialog_data *data);
static void free_edit_data (GtkWidget *widget, struct dialog_data *data);

static volatile sig_atomic_t viewedit_process_done;
static GtkItemFactory *log_factory;

int main (int argc, char *argv[]) {
   GtkWidget *window, *ui;

   gtk_set_locale ();
#ifdef HAVE_GETTEXT
   setlocale (LC_ALL, "");
   bindtextdomain ("gftp", LOCALE_DIR);
   textdomain ("gftp");
#endif
   
   gtk_init (&argc, &argv);
   signal (SIGCHLD, sig_child);
   signal (SIGPIPE, SIG_IGN);
   if (argc > 1) {
      if (strcmp (argv[1], "--help") == 0 || strcmp (argv[1], "-h") == 0) {
         usage ();
      }
      else if (strcmp (argv[1], "--version") == 0 || strcmp (argv[1], "-v") == 0) {
         printf ("%s\n", version);
         exit (0);
      }
   }

   read_config_file ();
   window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   gtk_signal_connect (GTK_OBJECT (window), "delete_event", 
   	GTK_SIGNAL_FUNC (delete_event), NULL);
   gtk_signal_connect (GTK_OBJECT (window), "destroy", 
   	GTK_SIGNAL_FUNC (destroy), NULL);
   gtk_window_set_title (GTK_WINDOW (window), version);
   gtk_window_set_policy (GTK_WINDOW (window), TRUE, TRUE, FALSE);
   gtk_widget_realize (window);
   
   ui = CreateFTPWindows (window);
   gtk_container_add (GTK_CONTAINER (window), ui);
   gtk_widget_show (window);

   
   ftp_log (gftp_logging_misc, NULL, "%s, Copyright (C) 1998-1999 Brian Masney <", version);
   ftp_log (gftp_logging_recv, NULL, "masneyb@seul.org");
   ftp_log (gftp_logging_misc, NULL, _(">. If you have any questions, comments, or suggestions about this program, please feel free to email them to me. You can always find out the latest news about gFTP from my website at http://gftp.seul.org/\n"));
   ftp_log (gftp_logging_misc, NULL, _("gFTP comes with ABSOLUTELY NO WARRANTY; for details, see the COPYING file. This is free software, and you are welcome to redistribute it under certain conditions; for details, see the COPYING file\n"));
   gtk_timeout_add (1000, update_downloads, NULL);
   init_gftp (argc, argv, window);
   add_local_files (&window1); 
   gtk_main ();
   return (0);
}  
/*****************************************************************************/
static void menu_exit (GtkWidget *widget, gpointer data) {
   if (!delete_event (widget, NULL, data)) {
      doexit (widget, data);
   }
}
/*****************************************************************************/
static gint delete_event (GtkWidget *widget, GdkEvent *event, gpointer data) {
   if (file_transfers == NULL) doexit (widget, data);
   else {
      MakeYesNoDialog (_("Exit"), _("There are file transfers in progress.\nAre you sure you want to exit?"), 1, 2,
      	_("Exit"), doexit, NULL, _("Don't Exit"), NULL, NULL);
      return (TRUE);
   }
   return (FALSE);
}
/*****************************************************************************/
static void doexit (GtkWidget *widget, gpointer data) {
   if (save_geometry) {
      listbox_local_width = GTK_WIDGET (local_frame)->allocation.width;
      listbox_remote_width = GTK_WIDGET (remote_frame)->allocation.width;
      listbox_file_height = GTK_WIDGET (remote_frame)->allocation.height;
      log_height = GTK_WIDGET (log_table)->allocation.height;
      transfer_height = GTK_WIDGET (transfer_scroll)->allocation.height;
   }
   write_config_file ();
   clear_cache_files ();
   exit (0);
}
/*****************************************************************************/
static void destroy (GtkWidget *widget, gpointer data) {
   exit (0);
}
/*****************************************************************************/
static void sig_child (int signo) {
   viewedit_process_done = 1;
   signal (SIGCHLD, sig_child);
}
/*****************************************************************************/
void init_gftp (int argc, char *argv[], GtkWidget *parent) {
   struct pix_ext *tempext;
   GtkWidget *sort_wid;
   struct utsname unme;
   struct passwd *pw;
   struct hostent *hent;

   window1.local = window2.local = -1;
   if (emailaddr == NULL || *emailaddr == '\0') {
      /* If there is no email address specified, then we'll just use the
         currentuser@currenthost */
      pw = getpwuid (geteuid ());
      uname (&unme);
      hent = gethostbyname (unme.nodename);
      if (strchr (unme.nodename, '.') == NULL && hent != NULL) {
         emailaddr = g_strconcat (pw->pw_name, "@", hent->h_name, NULL);
      }
      else {
         emailaddr = g_strconcat (pw->pw_name, "@", unme.nodename, NULL);
      }
      write_config_file ();
   }
                        
   open_xpm ("dotdot.xpm", parent, &dotdot_pixmap, &dotdot_mask, 1);
   open_xpm ("dir.xpm", parent, &dir_pixmap, &dir_mask, 1);
   open_xpm ("open_dir.xpm", parent, &open_dir_pixmap, &open_dir_mask, 1);
   open_xpm ("linkdir.xpm", parent, &linkdir_pixmap, &linkdir_mask, 1);
   open_xpm ("linkfile.xpm", parent, &linkfile_pixmap, &linkfile_mask, 1);
   open_xpm ("exe.xpm", parent, &exe_pixmap, &exe_mask, 1);
   open_xpm ("doc.xpm", parent, &doc_pixmap, &doc_mask, 1);
   
   if (window1.flags & WINDOW_SORTASDS) sort_wid = toolbar_pixmap (window1.listbox, down_xpm);
   else sort_wid = toolbar_pixmap (window1.listbox, up_xpm);
   gtk_clist_set_column_widget (GTK_CLIST (window1.listbox), 0, sort_wid);

   if (window2.flags & WINDOW_SORTASDS) sort_wid = toolbar_pixmap (window2.listbox, down_xpm);
   else sort_wid = toolbar_pixmap (window2.listbox, up_xpm);
   gtk_clist_set_column_widget (GTK_CLIST (window2.listbox), 0, sort_wid);

   tempext = registered_exts;
   while (tempext != NULL) {
      tempext->pixmap = NULL;
      open_xpm (tempext->filename, parent, &tempext->pixmap, &tempext->mask, 1);
      tempext = tempext->next;
   }
   if (argc > 2) usage ();
   else if (argc == 2) {
      if (gftp_parse_url (window2.hdata->ftpdata, argv[1]) == 0) {
         ftp_connect (window2.hdata, 1);
      }
      else usage ();
   }
}
/*****************************************************************************/
static void usage (void) {
   printf (_("usage: gftp [[ftp://][user:pass@]ftp-site[:port][/directory]]\n"));
   exit (0);
}
/*****************************************************************************/
static GtkWidget *CreateFTPWindows (GtkWidget *ui) {
   GtkWidget *box, *dlbox, *winpane, *dlpane, *logpane, *vscrollbar, *mainvbox, 
   	*tempwid, *button, *pix;
   GtkAccelGroup *accel_group;
   char *dltitles[3] = {_("Filename"), _("Progress"), _("Hostname")};
   const GtkTargetEntry possible_types[] = {
      {"STRING",			0,	0},
      {"text/plain", 			0, 	0},
      {"application/x-rootwin-drop", 	0, 	1}};
   int i, remote_start, remote_len, local_start, local_len, log_start, log_len;
   static GtkItemFactoryEntry dummy_item, menu_items[] = {
      {"/_FTP",			NULL,	0,		0,	"<Branch>"},
      {"/FTP/tearoff",		NULL,	0,		0,	"<Tearoff>"},
      {"/FTP/Ascii", 		NULL, 	change_setting,	1,	"<RadioItem>"},
      {"/FTP/Binary", 		NULL, 	change_setting,	2,	"/FTP/Ascii"},
      {"/FTP/sep", 		NULL, 	0, 		0,	"<Separator>"},
      {"/FTP/Use proxy",	NULL,	0,		0,	"<CheckItem>"},
      {"/FTP/Login as anonymous", NULL,	setanonymous,	0},
      {"/FTP/sep", 		NULL, 	0, 		0,	"<Separator>"},
      {"/FTP/_Options...", 	"<control>O",	options_dialog, 0},
      {"/FTP/sep", 		NULL, 	0, 		0,	"<Separator>"},
      {"/FTP/_Quit", 		"<control>Q", 	menu_exit,	0},
      {"/_Local",		NULL,	0,		0,	"<Branch>"},
      {"/Local/tearoff",	NULL,	0,		0,	"<Tearoff>"},
      {"/Local/Change Filespec...", NULL, change_filespec, 0},
      {"/Local/Select All", 	NULL, 	selectall, 	0},
      {"/Local/Select All Files", 	NULL, 	selectallfiles, 	0},
      {"/Local/Deselect All",	NULL, 	deselectall, 	0},
      {"/Local/sep",		NULL,	0,		0,	"<Separator>"},
      {"/Local/Change Directory", NULL,	chfunc, 	0},
      {"/Local/Chmod...",	NULL,	chmod_dialog,	0},
      {"/Local/Make Directory...", NULL, mkdir_dialog, 	0},
      {"/Local/Rename...", 	NULL, 	rename_dialog, 	0},
      {"/Local/Delete...", 	NULL, 	delete_dialog, 	0},
      {"/Local/Edit...",	NULL,	edit_dialog,	0},
      {"/Local/View...",	NULL,	view_dialog,	0},
      {"/Local/Refresh", 	NULL, 	refresh, 	0},
      {"/_Remote",		NULL,	0,		0,	"<Branch>"},
      {"/Remote/tearoff",	NULL,	0,		0,	"<Tearoff>"},
      {"/Remote/Open _URL...",	"<control>U",	openurl_dialog,	0},
      {"/Remote/_Disconnect",	"<control>D", 	disconn, 	0},
      {"/Remote/sep",		NULL,	0,		0,	"<Separator>"},
      {"/Remote/Change Filespec...", NULL, change_filespec, 0},
      {"/Remote/Select All", 	NULL, 	selectall, 	0},
      {"/Remote/Select All Files", 	NULL, 	selectallfiles, 	0},
      {"/Remote/Deselect All", 	NULL, 	deselectall, 	0},
      {"/Remote/sep",		NULL,	0,		0,	"<Separator>"},
      {"/Remote/Send SITE Command...", NULL, site_dialog, 0},
      {"/Remote/sep",		NULL,	0,		0,	"<Separator>"},
      {"/Remote/Change Directory", NULL, chfunc, 	0},
      {"/Remote/Chmod...",	NULL,	chmod_dialog,	0},
      {"/Remote/Make Directory...", NULL, mkdir_dialog,	0},
      {"/Remote/Rename...", 	NULL, 	rename_dialog, 	0},
      {"/Remote/Delete...", 	NULL, 	delete_dialog, 	0},
      {"/Remote/Edit...",	NULL,	edit_dialog,	0},
      {"/Remote/View...",	NULL,	view_dialog,	0},
      {"/Remote/Refresh", 	NULL, 	refresh, 	0},
      {"/_Bookmarks",		NULL,	0,		0,	"<Branch>"},
      {"/Bookmarks/tearoff",	NULL,	0,		0,	"<Tearoff>"},
      {"/Bookmarks/Add bookmark", "<control>A", add_bookmark, 0},
      {"/Bookmarks/Edit bookmarks", NULL, edit_bookmarks,		0},
      {"/Bookmarks/sep",	NULL,	0,		0,	"<Separator>"},
      {"/_Transfers",		NULL,	0,		0,	"<Branch>"},
      {"/Transfers/tearoff",	NULL,	0,		0,	"<Tearoff>"},
      {"/Transfers/Stop Transfer", NULL, stop_transfer,	0},
      {"/Transfers/sep",	NULL,	0,		0,	"<Separator>"},
      {"/Transfers/Retrieve Files", "<control>R",  retrCB,	0},
      {"/Transfers/Put Files",	"<control>P",	putCB,	0},
      {"/L_ogging",		NULL,	0,		0,	"<Branch>"},
      {"/Logging/tearoff",	NULL,	0,		0,	"<Tearoff>"},
      {"/Logging/Clear", 	NULL, 	clearlog, 	0},
      {"/Logging/View log...", 	NULL, 	viewlog, 	0},
      {"/Logging/Save log...", 	NULL, 	savelog, 	0},
      {"/Tool_s",		NULL,	0,		0,	"<Branch>"},
      {"/Tools/tearoff",	NULL,	0,		0,	"<Tearoff>"},
      {"/Tools/Compare Windows", NULL, 	compare_windows,  0},
      {"/_Help", 		NULL, 	0,		0,	"<LastBranch>"},
      {"/Help/tearoff",		NULL,	0,		0,	"<Tearoff>"},
      {"/Help/About...", 	NULL, 	about_dialog,	0}};
      
   mainvbox = gtk_vbox_new (FALSE, 0);
   gtk_widget_show (mainvbox);

   accel_group = gtk_accel_group_new ();
   factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", accel_group);

   /* The create_item_factory() function will call gettext on the strings above */
   i = 0;
   /* FTP Menu */
   create_item_factory (factory, 11, menu_items, &window2);
   i += 11;
   /* Local Menu */
   local_start = i;
   local_len = 15;
   create_item_factory (factory, local_len, menu_items + i, &window1);
   i += 15;
   /* Remote Menu */
   remote_start = i;
   remote_len = 20;
   create_item_factory (factory, remote_len, menu_items + i, &window2);
   i += 20;
   /* Bookmarks Menu */
   create_item_factory (factory, 5, menu_items + i, &window2);
   i += 5;
   /* Transfers Menu */
   create_item_factory (factory, 6, menu_items + i, NULL);
   i += 6;
   /* Logging Menu */
   log_start = i;
   log_len = 5;
   create_item_factory (factory, log_len, menu_items + i, NULL);
   i += 5;
   /* Tools Menu */
   create_item_factory (factory, 3, menu_items + i, NULL);
   i += 3;
   /* Help Menu */
   create_item_factory (factory, 3, menu_items + i, NULL);
   i += 3;

   gtk_accel_group_attach (accel_group, GTK_OBJECT (ui));
   gtk_box_pack_start (GTK_BOX (mainvbox), factory->widget, FALSE, FALSE, FALSE);
   gtk_widget_show (factory->widget);

   tempwid = gtk_item_factory_get_widget (factory, _(menu_items[3].path));
   gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM (tempwid), TRUE);

   firewall_btn = gtk_item_factory_get_widget (factory, _(menu_items[5].path));
   if (*proxy_config  == '\0') {
      gtk_widget_set_sensitive (firewall_btn, 0);
   }
   else {
      gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM (firewall_btn), TRUE);
   }

   build_bookmarks_menu ();

   tempwid = gtk_handle_box_new ();
   gtk_box_pack_start (GTK_BOX (mainvbox), tempwid, FALSE, FALSE, FALSE);
   gtk_widget_show (tempwid);

   box = gtk_hbox_new (FALSE, 5);
   gtk_container_add (GTK_CONTAINER (tempwid), box);
   gtk_container_border_width (GTK_CONTAINER (tempwid), 5);
   gtk_widget_show (box);

   pix = toolbar_pixmap (ui, connect_xpm);
   openurl_btn = gtk_button_new ();
   gtk_container_add (GTK_CONTAINER (openurl_btn), pix);
   gtk_signal_connect_object (GTK_OBJECT (openurl_btn), "clicked", 
   	GTK_SIGNAL_FUNC (openurl_dialog), NULL);
   gtk_signal_connect (GTK_OBJECT (openurl_btn), "drag_data_received", 
   	GTK_SIGNAL_FUNC (openurl_get_drag_data), NULL);
   gtk_drag_dest_set (openurl_btn, GTK_DEST_DEFAULT_ALL, possible_types, 2, 
   	GDK_ACTION_COPY | GDK_ACTION_MOVE);
   gtk_box_pack_start (GTK_BOX (box), openurl_btn, FALSE, FALSE, FALSE);
   gtk_widget_show (openurl_btn);
   
   tempwid = gtk_label_new (_("Host: "));
   gtk_box_pack_start (GTK_BOX (box), tempwid, FALSE, FALSE, FALSE);
   gtk_widget_show (tempwid);
   
   hostedit = gtk_entry_new ();
   gtk_widget_set_usize (hostedit, 150, -1);
   gtk_signal_connect (GTK_OBJECT (hostedit), "activate", 
   	GTK_SIGNAL_FUNC (toolbar_hostedit), (gpointer) NULL);
   gtk_box_pack_start (GTK_BOX (box), hostedit, TRUE, TRUE, TRUE);
   gtk_widget_show (hostedit);

   tempwid = gtk_label_new (_("Port: "));
   gtk_box_pack_start (GTK_BOX (box), tempwid, FALSE, FALSE, FALSE);
   gtk_widget_show (tempwid);
   
   portedit = gtk_entry_new ();
   gtk_widget_set_usize (portedit, 25, -1);
   gtk_signal_connect (GTK_OBJECT (portedit), "activate", 
   	GTK_SIGNAL_FUNC (toolbar_hostedit), (gpointer) NULL);
   gtk_box_pack_start (GTK_BOX (box), portedit, TRUE, TRUE, TRUE);
   gtk_widget_show (portedit);

   tempwid = gtk_label_new (_("User: "));
   gtk_box_pack_start (GTK_BOX (box), tempwid, FALSE, FALSE, FALSE);
   gtk_widget_show (tempwid);
   
   useredit = gtk_entry_new ();
   gtk_widget_set_usize (useredit, 75, -1);
   gtk_signal_connect (GTK_OBJECT (useredit), "activate", 
   	GTK_SIGNAL_FUNC (toolbar_hostedit), (gpointer) NULL);
   gtk_box_pack_start (GTK_BOX (box), useredit, TRUE, TRUE, TRUE);
   gtk_widget_show (useredit);

   tempwid = gtk_label_new (_("Pass: "));
   gtk_box_pack_start (GTK_BOX (box), tempwid, FALSE, FALSE, FALSE);
   gtk_widget_show (tempwid);

   passedit = gtk_entry_new ();
   gtk_widget_set_usize (passedit, 75, -1);
   gtk_entry_set_visibility (GTK_ENTRY (passedit), FALSE);
   gtk_signal_connect (GTK_OBJECT (passedit), "activate", 
   	GTK_SIGNAL_FUNC (toolbar_hostedit), (gpointer) NULL);
   gtk_box_pack_start (GTK_BOX (box), passedit, TRUE, TRUE, TRUE);
   gtk_widget_show (passedit);
   
   pix = toolbar_pixmap (ui, stop_xpm);   
   stop_btn = gtk_button_new ();
   gtk_container_add (GTK_CONTAINER (stop_btn), pix);
   gtk_widget_set_sensitive (stop_btn, 0);
   gtk_signal_connect_object (GTK_OBJECT (stop_btn), "clicked", 
   	GTK_SIGNAL_FUNC (stop_button), (gpointer) &window2);
   gtk_box_pack_start (GTK_BOX (box), stop_btn, TRUE, TRUE, TRUE);
   gtk_widget_show (stop_btn);

   winpane = gtk_hpaned_new ();

   box = gtk_hbox_new (FALSE, 0);

   window1.ifactory = gtk_item_factory_new (GTK_TYPE_MENU, "<local>", NULL);
   for (i=local_start + 2; i<local_start + local_len; i++) {
      memcpy (&dummy_item, menu_items + i, sizeof (GtkItemFactoryEntry));
      dummy_item.path = strrchr (_(dummy_item.path), '/');
      gtk_item_factory_create_item (window1.ifactory, &dummy_item, &window1, 1);
   }
   window1.local = 1;
   local_frame = CreateFTPWindow (&window1);
   window1.local = -1;
   gtk_box_pack_start (GTK_BOX (box), local_frame, TRUE, TRUE, FALSE);
   gtk_widget_show (local_frame);

   dlbox = gtk_vbox_new (FALSE, 0);
   gtk_container_border_width (GTK_CONTAINER (dlbox), 5);
   gtk_box_pack_start (GTK_BOX (box), dlbox, FALSE, FALSE, FALSE);
   gtk_widget_show (dlbox);

   tempwid = toolbar_pixmap (ui, right_xpm);
   button = gtk_button_new ();
   gtk_box_pack_start (GTK_BOX (dlbox), button, TRUE, FALSE, FALSE);
   gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (putCB), NULL);
   gtk_container_add (GTK_CONTAINER (button), tempwid);
   gtk_widget_show (button);

   tempwid = toolbar_pixmap (ui, left_xpm);
   button = gtk_button_new ();
   gtk_box_pack_start (GTK_BOX (dlbox), button, TRUE, FALSE, FALSE);
   gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (retrCB), NULL);
   gtk_container_add (GTK_CONTAINER (button), tempwid);
   gtk_widget_show (button);

   gtk_widget_show (box);
   gtk_paned_pack1 (GTK_PANED (winpane), box, 1, 1);
   
   window2.ifactory = gtk_item_factory_new (GTK_TYPE_MENU, "<remote>", NULL);
   for (i=remote_start + 2; i<remote_start + remote_len; i++) {
      memcpy (&dummy_item, menu_items + i, sizeof (GtkItemFactoryEntry));
      dummy_item.path = strrchr (_(dummy_item.path), '/');
      gtk_item_factory_create_item (window2.ifactory, &dummy_item, &window2, 1);
   }
   remote_frame = CreateFTPWindow (&window2);
   gtk_paned_pack2 (GTK_PANED (winpane), remote_frame, 1, 1);
   gtk_widget_show (remote_frame);
   gtk_widget_show (winpane);

   dlpane = gtk_vpaned_new ();
   gtk_paned_pack1 (GTK_PANED (dlpane), winpane, 1, 1);

   transfer_scroll = gtk_scrolled_window_new (NULL, NULL);
   gtk_widget_set_usize (transfer_scroll, 1, transfer_height);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (transfer_scroll),
   	GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   dlwdw = gtk_clist_new_with_titles (3, dltitles);
   gtk_container_add (GTK_CONTAINER (transfer_scroll), dlwdw);
   gtk_clist_set_column_width (GTK_CLIST (dlwdw), 0, 100);
   gtk_clist_set_column_width (GTK_CLIST (dlwdw), 1, 250);
   gtk_clist_set_column_width (GTK_CLIST (dlwdw), 2, 100);
   gtk_container_border_width (GTK_CONTAINER (dlwdw), 5);
   gtk_signal_connect (GTK_OBJECT (dlwdw), "select_row", GTK_SIGNAL_FUNC (selectdl), NULL);
   gtk_signal_connect (GTK_OBJECT (dlwdw), "unselect_row", GTK_SIGNAL_FUNC (unselectdl), NULL);
   gtk_paned_pack2 (GTK_PANED (dlpane), transfer_scroll, 1, 1);
   gtk_widget_show (dlwdw);
   gtk_widget_show (transfer_scroll);
   gtk_widget_show (dlpane);

   log_factory = gtk_item_factory_new (GTK_TYPE_MENU, "<log>", NULL);
   for (i=log_start + 2; i<log_start + log_len; i++) {
      memcpy (&dummy_item, menu_items + i, sizeof (GtkItemFactoryEntry));
      dummy_item.path = strrchr (_(dummy_item.path), '/');
      gtk_item_factory_create_item (log_factory, &dummy_item, NULL, 1);
   }
   
   logpane = gtk_vpaned_new ();
   gtk_paned_pack1 (GTK_PANED (logpane), dlpane, 1, 1);

   log_table = gtk_table_new (1, 2, FALSE);
   gtk_widget_set_usize (log_table, 1, log_height);
   logwdw = gtk_text_new (NULL, NULL);
   gtk_text_set_editable (GTK_TEXT (logwdw), FALSE);
   gtk_text_set_word_wrap (GTK_TEXT (logwdw), TRUE);
   gtk_table_attach (GTK_TABLE (log_table), logwdw, 0, 1, 0, 1,
   	GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 0);
   gtk_signal_connect (GTK_OBJECT (logwdw), "button_press_event", GTK_SIGNAL_FUNC (log_click), NULL);
   gtk_widget_show (logwdw);

   vscrollbar = gtk_vscrollbar_new (GTK_TEXT (logwdw)->vadj);
   gtk_table_attach (GTK_TABLE (log_table), vscrollbar, 1, 2, 0, 1,
   	GTK_FILL, GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0);
   gtk_widget_show (vscrollbar);
   gtk_widget_show (log_table);
   
   gtk_paned_pack2 (GTK_PANED (logpane), log_table, 1, 1);
   gtk_widget_show (logpane);      
   gtk_box_pack_start (GTK_BOX (mainvbox), logpane, TRUE, TRUE, TRUE);

   return (mainvbox);
}   
/*****************************************************************************/
static GtkWidget *CreateFTPWindow (struct ftp_window_data *wdata) {
   char *titles[7] = {"", _("Filename"), _("Size"), _("User"), _("Group"), _("Date"), _("Attribs")};
   const GtkTargetEntry possible_types[] = {
      {"STRING",			0,	0},
      {"text/plain", 			0, 	0},
      {"application/x-rootwin-drop", 	0, 	1}};
   GtkWidget *box, *scroll_list, *parent;

   wdata->filespec = g_malloc (2);
   strcpy (wdata->filespec, "*");
   wdata->hdata = new_hdata ();
   wdata->hdata->wdata = wdata;
   wdata->sortcol = 1;
   wdata->flags |= WINDOW_SORTASDS;
   wdata->totalitems = wdata->numselected = 0;

   parent = gtk_frame_new (NULL);
   gtk_widget_set_usize (parent, wdata->local == 1 ? listbox_local_width : listbox_remote_width,
   	listbox_file_height);
   gtk_container_border_width (GTK_CONTAINER (parent), 5);
   gtk_widget_show (parent);

   box = gtk_vbox_new (FALSE, 0);
   gtk_container_border_width (GTK_CONTAINER (box), 5);
   gtk_container_add (GTK_CONTAINER (parent), box);
   
   wdata->combo = gtk_combo_new ();
   gtk_box_pack_start (GTK_BOX (box), wdata->combo, FALSE, FALSE, FALSE);
   gtk_signal_connect (GTK_OBJECT (GTK_COMBO (wdata->combo)->entry), "activate", GTK_SIGNAL_FUNC (chdiredit), (gpointer) wdata);
   if (wdata->history) gtk_combo_set_popdown_strings (GTK_COMBO (wdata->combo), wdata->history);
   gtk_combo_disable_activate (GTK_COMBO (wdata->combo));
   gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry), "");
   gtk_widget_show (wdata->combo);
   
   wdata->hoststxt = gtk_label_new (_("Not connected"));
   gtk_misc_set_alignment (GTK_MISC (wdata->hoststxt), 0, 0);
   gtk_box_pack_start (GTK_BOX (box), wdata->hoststxt, FALSE, FALSE, FALSE);
   gtk_widget_show (wdata->hoststxt);

   scroll_list = gtk_scrolled_window_new (NULL, NULL);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll_list),
   	GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   wdata->listbox = gtk_clist_new_with_titles (7, titles);
   gtk_container_add (GTK_CONTAINER (scroll_list), wdata->listbox);
   gtk_drag_source_set (wdata->listbox, GDK_BUTTON2_MASK, possible_types, 3,
   	GDK_ACTION_COPY | GDK_ACTION_MOVE);
   gtk_drag_dest_set (wdata->listbox, GTK_DEST_DEFAULT_ALL, possible_types, 2,
   	GDK_ACTION_COPY | GDK_ACTION_MOVE);
   gtk_clist_set_selection_mode (GTK_CLIST (wdata->listbox), GTK_SELECTION_EXTENDED);
   gtk_clist_set_column_width (GTK_CLIST (wdata->listbox), 0, 16);
   gtk_clist_set_column_justification (GTK_CLIST (wdata->listbox), 0, GTK_JUSTIFY_CENTER);
   setup_column (wdata->listbox, 1, listbox_filename_width);
   gtk_clist_set_column_justification (GTK_CLIST (wdata->listbox), 2, GTK_JUSTIFY_RIGHT);
   setup_column (wdata->listbox, 2, listbox_size_width);
   setup_column (wdata->listbox, 3, listbox_user_width);
   setup_column (wdata->listbox, 4, listbox_group_width);
   setup_column (wdata->listbox, 5, listbox_date_width);
   setup_column (wdata->listbox, 6, listbox_attribs_width);
   gtk_box_pack_start (GTK_BOX (box), scroll_list, TRUE, TRUE, TRUE);
   gtk_signal_connect (GTK_OBJECT (wdata->listbox), "click_column", GTK_SIGNAL_FUNC (sortrows), (gpointer) wdata);
   gtk_signal_connect_after (GTK_OBJECT (wdata->listbox), "select_row", GTK_SIGNAL_FUNC (selectrow), (gpointer) wdata);
   gtk_signal_connect_after (GTK_OBJECT (wdata->listbox), "unselect_row", GTK_SIGNAL_FUNC (unselectrow), (gpointer) wdata);
   gtk_signal_connect_after (GTK_OBJECT (wdata->listbox), "button_press_event", GTK_SIGNAL_FUNC (list_dblclick), (gpointer) wdata);
   gtk_signal_connect (GTK_OBJECT (wdata->listbox), "key_press_event", GTK_SIGNAL_FUNC (list_enter), (gpointer) wdata);
   gtk_signal_connect (GTK_OBJECT (wdata->listbox), "drag_data_get", GTK_SIGNAL_FUNC (listbox_drag), (gpointer) wdata);
   gtk_signal_connect (GTK_OBJECT (wdata->listbox), "drag_data_received", GTK_SIGNAL_FUNC (listbox_get_drag_data), (gpointer) wdata);
   gtk_widget_show (wdata->listbox);
   gtk_widget_show (scroll_list);

   gtk_widget_show (box);
   return (parent);
}
/*****************************************************************************/
static void setup_column (GtkWidget *listbox, int column, int width) {
   if (width == 0) {
      gtk_clist_set_column_auto_resize (GTK_CLIST (listbox), column, TRUE);
   }
   else if (width == -1) {
      gtk_clist_set_column_visibility (GTK_CLIST (listbox), column, FALSE);
   }
   else {
      gtk_clist_set_column_width (GTK_CLIST (listbox), column, width);
   }
}
/*****************************************************************************/
static void listbox_drag (GtkWidget *widget, GdkDragContext *context, GtkSelectionData *selection_data, guint info, guint32 clk_time, gpointer data) {
   struct ftp_window_data *wdata;
   struct ftp_file_data *tempfle;
   size_t totlen, oldlen;
   char tempstr[MAXSTR];
   char *str = NULL;
   
   totlen = 0;
   wdata = (struct ftp_window_data *) data;
   if (!check_status (_("Drag-N-Drop"), wdata, 0, 1, 0)) return;

   if ((tempfle = get_next_selected_filename (wdata->hdata->files)) == NULL) {
      ftp_log (gftp_logging_misc, NULL, _("Internal gFTP Error: Could not find a selected file. This is probably a bug. Please email masneyb@seul.org about it\n"));
      return;
   }

   while (tempfle != NULL) {
      if (totlen != 0) totlen--;
      oldlen = totlen;
      if (wdata->local == 1) {
          g_snprintf (tempstr, sizeof(tempstr), "file:%s/%s\n", GFTP_GET_DIRECTORY (wdata->hdata->ftpdata), tempfle->file);
      }
      else {
         g_snprintf (tempstr, sizeof (tempstr), "%s://%s:%s@%s:%d%s/%s\n", 
            wdata->hdata->protocol == ftp ? "ftp" : "http",
            GFTP_GET_USERNAME (wdata->hdata->ftpdata), 
            GFTP_GET_PASSWORD (wdata->hdata->ftpdata), 
            GFTP_GET_HOSTNAME (wdata->hdata->ftpdata),
            GFTP_GET_PORT (wdata->hdata->ftpdata),
            GFTP_GET_DIRECTORY (wdata->hdata->ftpdata), tempfle->file);
      }
      tempstr[sizeof (tempstr) - 1] = '\0';
      remove_double_slashes (tempstr + 6);
      totlen += strlen (tempstr) + 1;
      str = g_realloc (str, totlen);
      strcpy (str + oldlen, tempstr);
      tempfle = get_next_selected_filename (tempfle->next);
   }

   if(totlen > 1) str[totlen - 2] = '\0';
   gtk_selection_data_set (selection_data, selection_data->target, 8,
   	str, strlen (str));
   g_free (str);
}
/*****************************************************************************/
static void listbox_get_drag_data (GtkWidget *widget, GdkDragContext *context, gint x, 
	gint y, GtkSelectionData *selection_data, guint info, guint32 clk_time, 
	struct ftp_window_data *wdata) {
	
   struct ftp_transfer_data *tdata, *temptdata, *new_file_transfers;
   struct ftp_file_data *newfle;
   char *newpos, *oldpos, *pos;
   char *tempstr, tempchar, *str;
   gftp_request *current_ftpdata;
   int finish_drag;
   struct stat st;
   
   if (!check_status (_("Drag-N-Drop"), &window2, 0, 0, 0)) return;

   new_file_transfers = NULL;
   finish_drag = 0;
   if ((selection_data->length >= 0) && (selection_data->format == 8)) {
      if (wdata->local != 1) {
         tdata = new_tdata ();
         tdata->hdata->files = tdata->hdata->last = NULL;
         tdata->next = NULL;
         tdata->curfle = NULL;
         new_file_transfers = tdata;
         copy_hdata_struct (window2.hdata, tdata->hdata);
      }
      else tdata = NULL;

      oldpos = (char *) selection_data->data;
      while ((newpos = strchr (oldpos, '\n')) || (*oldpos != '\0' && (newpos = strchr (oldpos, '\0')))) {
         if (*(newpos-1) == '\r') newpos--;
         tempstr = g_malloc (newpos - oldpos + 1);
         tempchar = *newpos;
         *newpos = '\0';
         strcpy (tempstr, oldpos);
         *newpos = tempchar;
         if (tempchar == '\r') newpos++;

         if (wdata->local != 1 && strncmp (oldpos, "ftp://", 6) == 0) {
            ftp_log (gftp_logging_misc, NULL, _("Drag-N-Drop: Ignoring file %s: Cannot drag a ftp site to a ftp site\n"), tempstr);
            oldpos = newpos + 1;
            continue;
         }
         else if (wdata->local == 1 && strncmp (oldpos, "file:", 5) == 0) {
            ftp_log (gftp_logging_misc, NULL, _("Drag-N-Drop: Ignoring file %s: Cannot drag a local file to the local window\n"), tempstr);
            oldpos = newpos + 1;
            continue;
         }
         else {
            newfle = g_malloc0 (sizeof (struct ftp_file_data));
            newfle->next = NULL;
            if (wdata->local == 1) {
               current_ftpdata = gftp_request_new ();
               if (gftp_parse_url (current_ftpdata, tempstr) != 0) {
                  ftp_log (gftp_logging_misc, NULL, _("Drag-N-Drop: Ignoring url %s: Not a valid url\n"), tempstr);
                  oldpos = newpos + 1;
                  gftp_request_destroy (current_ftpdata);
                  free_fdata (newfle);
                  continue;
               }
               
               if ((str = GFTP_GET_DIRECTORY (current_ftpdata)) != NULL) {
                  if ((pos = strchr (str, '/')) == NULL) pos = str;
                  else pos++;
               }
               else pos = str = "";

               newfle->remote_file = g_malloc (strlen (str) + 1);
               strcpy (newfle->remote_file, str);
               newfle->file = g_strconcat (GFTP_GET_DIRECTORY (window1.hdata->ftpdata), "/", pos, NULL);
               newfle->flags = get_file_transfer_mode (newfle->file);
               newfle->size = 0;
               gftp_set_directory (current_ftpdata, "");

               temptdata = new_file_transfers;
               while (temptdata != NULL) {
                  if (compare_hdata_structs (temptdata->hdata->ftpdata, current_ftpdata, 1)) {
                     tdata = temptdata;
                     break;
                  }
                  temptdata = temptdata->next;
               }

               if (temptdata == NULL) {
                  tdata = new_tdata ();
                  tdata->hdata->files = tdata->hdata->last = NULL;
                  tdata->curfle = NULL;
                  tdata->flags |= TRANSFER_DIRECTION;
                  gftp_request_destroy (tdata->hdata->ftpdata);
                  tdata->hdata->ftpdata = current_ftpdata;

                  tdata->next = new_file_transfers;
                  new_file_transfers = tdata;
               }
               else {
                  gftp_request_destroy (current_ftpdata);
               }
            }
            else {
               if (strncmp (tempstr, "file:", 5) == 0) {
                  str = tempstr + 5;
               }
               else str = tempstr;
               
               stat (str, &st);
               newfle->file = g_malloc (strlen (str) + 1);
               strcpy (newfle->file, str);
               if ((pos = strrchr (str, '/')) == NULL) pos = str;
               else pos++;
               newfle->remote_file = g_strconcat (GFTP_GET_DIRECTORY (window2.hdata->ftpdata), "/", pos, NULL);
               newfle->size = st.st_size;
            }
            if (tdata->hdata->last == NULL) tdata->hdata->files = newfle;
            else tdata->hdata->last->next = newfle;
            tdata->hdata->last = newfle;
            tdata->hdata->totalfiles++;
            if (tdata->curfle == NULL) tdata->curfle = newfle;
            finish_drag = 1;
         }
         oldpos = newpos + 1;
      }
   }
   if (finish_drag) {
      temptdata = new_file_transfers;
      while (temptdata != NULL) {
         dotrans (temptdata);
         temptdata = temptdata->next;
      }
      add_file_transfer (new_file_transfers);
   }
   gtk_drag_finish (context, finish_drag, FALSE, clk_time);
}
/*****************************************************************************/
static void setanonymous (gpointer data, int menuitem, GtkWidget *widget) {
   gtk_entry_set_text (GTK_ENTRY (useredit), ANON_LOGIN);
   gtk_entry_set_text (GTK_ENTRY (passedit), emailaddr);
}
/*****************************************************************************/
static void change_setting (struct ftp_window_data *wdata, int menuitem, GtkWidget *checkmenu) {
   switch (menuitem) {
      case 1 : if (wdata->hdata) gftp_set_data_type (wdata->hdata->ftpdata, gftp_type_ascii);
               break;
      case 2 : if (wdata->hdata) gftp_set_data_type (wdata->hdata->ftpdata, gftp_type_binary);
               break;
   }
}
/*****************************************************************************/
static void selectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data) {
   struct ftp_window_data *wdata;
   struct ftp_file_data *tempfle;
   int i;
   
   wdata = (struct ftp_window_data *) data;
   i = 0;
   tempfle = wdata->hdata->files;
   while (tempfle != NULL) {
      if (tempfle->flags & FILE_SHOWN) {
         if (i == row) {
            if (!(tempfle->flags & FILE_SELECTED)) wdata->numselected++;
            tempfle->flags |= FILE_SELECTED;
            break;
         }
         i++;
      }
      tempfle = tempfle->next;
   }
}
/*****************************************************************************/
static void unselectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data) {
   struct ftp_window_data *wdata;
   struct ftp_file_data *tempfle;
   int i;
   
   wdata = (struct ftp_window_data *) data;
   i = 0;
   tempfle = wdata->hdata->files;
   while (tempfle != NULL) {
      if (tempfle->flags & FILE_SHOWN) {
         if (i == row) {
            if (tempfle->flags & FILE_SELECTED) wdata->numselected--;
            tempfle->flags &= ~(FILE_SELECTED);
            break;
         }
         i++;
      }
      tempfle = tempfle->next;
   }
}
/*****************************************************************************/
static void selectall (struct ftp_window_data *wdata) {
   struct ftp_file_data *tempfle;
   int i = 0;
   
   if (!check_status (_("Select all"), wdata, 0, 0, 0)) return;
   tempfle = wdata->hdata->files;
   while (tempfle != NULL) {
      if (tempfle->flags & FILE_SHOWN) {
         if (strcmp (tempfle->file, "..") != 0) {
            gtk_clist_select_row (GTK_CLIST (wdata->listbox), i++, 0);
         }
         else {
            gtk_clist_unselect_row (GTK_CLIST (wdata->listbox), i++, 0);
         }
      }
      tempfle = tempfle->next;
   }
}
/*****************************************************************************/
static void selectallfiles (struct ftp_window_data *wdata) {
   struct ftp_file_data *tempfle;
   int i = 0;
   
   if (!check_status (_("Select all files"), wdata, 0, 0, 0)) return;
   tempfle = wdata->hdata->files;
   while (tempfle != NULL) {
      if (tempfle->flags & FILE_SHOWN) {
         if (tempfle->flags & FILE_ISDIR)  {
            gtk_clist_unselect_row (GTK_CLIST (wdata->listbox), i++, 0);
         }
         else {
            gtk_clist_select_row (GTK_CLIST (wdata->listbox), i++, 0);
         }
      }
      tempfle = tempfle->next;
   }
}
/*****************************************************************************/
static void deselectall (struct ftp_window_data *wdata) {
   struct ftp_file_data *tempfle;
   int i = 0;
   
   if (!check_status (_("Deselect all"), wdata, 0, 0, 0)) return;
   tempfle = wdata->hdata->files;
   while (tempfle != NULL) {
      if (tempfle->flags & FILE_SHOWN) {
         gtk_clist_unselect_row (GTK_CLIST (wdata->listbox), i++, 0);
      }
      tempfle = tempfle->next;
   }
}
/*****************************************************************************/
static gint log_click (GtkWidget *widget, GdkEventButton *event, gpointer data) {
   if (event->button == 3) {
      gtk_item_factory_popup (log_factory, (guint) event->x_root, (guint) event->y_root, 1, 0);
   }
   return (FALSE);
}
/*****************************************************************************/
static gint list_dblclick (GtkWidget *widget, GdkEventButton *event, gpointer data) {
   struct ftp_window_data *wdata;
   struct ftp_file_data *tempfle;
   long flagcopy;
   int success;
   
   wdata = (struct ftp_window_data *) data;
   if (event->button == 3) {
      gtk_item_factory_popup (wdata->ifactory, (guint) event->x_root, (guint) event->y_root, 1, 0);
   }
   else if (wdata->local == -1 || wdata->numselected != 1) return (FALSE);

   if (event->type == GDK_2BUTTON_PRESS && event->button == 1) {
      if ((tempfle = get_next_selected_filename (wdata->hdata->files)) == NULL) {
         ftp_log (gftp_logging_misc, NULL, "Internal gFTP Error: Could not find a selected file. This is probably a bug. Please email masneyb@seul.org about it\n");
         return (TRUE);
      }
      flagcopy = tempfle->flags;
      success = 0;
      if ((wdata->local != 1 && ((tempfle->flags & FILE_ISDIR) || (tempfle->flags & FILE_ISLINK))) ||
      		(wdata->local == 1 && tempfle->flags & FILE_ISDIR)) {
            success = chdirfunc (data);
      }
      if (!(flagcopy & FILE_ISDIR) && !success) view_dialog (data);
      return (FALSE);
   }
   return (TRUE);
}
/*****************************************************************************/
static gint list_enter (GtkWidget *widget, GdkEventKey *event, gpointer data) {
   struct ftp_window_data *wdata;
   struct ftp_file_data *tempfle;
   int success;
   
   wdata = (struct ftp_window_data *) data;
   if (wdata->local == -1 || wdata->numselected == 0) return (TRUE);
   if (wdata->numselected == 1 && event->type == GDK_KEY_PRESS && event->keyval == GDK_Return) {
      if ((tempfle = get_next_selected_filename (wdata->hdata->files)) == NULL) {
         ftp_log (gftp_logging_misc, NULL, "Internal gFTP Error: Could not find a selected file. This is probably a bug. Please email masneyb@seul.org about it\n");
         return (FALSE);
      }
      success = 0;
      if ((wdata->local != 1 && ((tempfle->flags & FILE_ISDIR) || (tempfle->flags & FILE_ISLINK))) ||
      		(wdata->local == 1 && tempfle->flags & FILE_ISDIR)) {
            success = chdirfunc (data);
      }
      if (!(tempfle->flags & FILE_ISDIR) && !success) view_dialog (data);
   }
   else if (event->type == GDK_KEY_PRESS && (event->keyval == GDK_KP_Delete || event->keyval == GDK_Delete)) {
      delete_dialog (data);
   }
   else return (TRUE);
   return (FALSE);
}
/*****************************************************************************/
static void chfunc (struct ftp_window_data *wdata) {
   chdirfunc (wdata);
}
/*****************************************************************************/
static int chdirfunc (struct ftp_window_data *wdata) {
   char *newdir, tempstr[MAXSTR], *resp;
   struct ftp_file_data *tempfle;
   
   if (!check_status (_("Chdir"), wdata, 1, 0, 0)) return (0);
   if ((tempfle = get_next_selected_filename (wdata->hdata->files)) == NULL) {
      ftp_log (gftp_logging_misc, NULL, "Internal gFTP Error: Could not find a selected file. This is probably a bug. Please email masneyb@seul.org about it\n");
      return (0);
   }
   if (wdata->local == 1) {
      if (chdir(tempfle->file) == 0) {
         ftp_log (gftp_logging_misc, NULL, _("Successfully changed local directory to %s\n"), tempfle->file);
         update_ftp_info (wdata);
         gtk_clist_freeze (GTK_CLIST (wdata->listbox));
         delete_ftp_file_info (wdata);
         add_local_files (wdata);
         gtk_clist_thaw (GTK_CLIST (wdata->listbox));
         return (1);
      }
      else {
         if (wdata->local == 1) {
            ftp_log (gftp_logging_misc, NULL, _("Could not change local directory to %s: %s\n"),
            	tempfle->file, g_strerror (errno));
         }
         else {
            ftp_log (gftp_logging_misc, NULL, _("Could not change remote directory to %s: %s\n"), 
            	tempfle->file, g_strerror (errno));
         }
      }
   }
   else {
      newdir = g_strconcat (GFTP_GET_DIRECTORY (wdata->hdata->ftpdata), "/", tempfle->file, NULL);
      remove_double_slashes (newdir);
      expand_path (newdir, tempstr, sizeof (tempstr));
      g_free (newdir);
      if (gftp_set_directory (wdata->hdata->ftpdata, tempstr) == 0) {
         if (wdata->local == 2 && !ftp_connect (wdata->hdata, 0)) return (FALSE);
         wdata->hdata->getdir = 1;
         update_ftp_info (wdata);
         gtk_clist_freeze (GTK_CLIST (wdata->listbox));
         delete_ftp_file_info (wdata);
         ftp_list_files (wdata, 1);
         gtk_clist_thaw (GTK_CLIST (wdata->listbox));
         return (1);
      }
      else {
         resp = GFTP_GET_LAST_RESPONSE (wdata->hdata->ftpdata);
         if (resp && *resp == '4') disconnect (wdata);
         return (0);
      }
   }
   return (0);
}
/*****************************************************************************/
static int chdiredit (GtkWidget *widget, struct ftp_window_data *wdata) {
   char tempstr[MAXSTR], *str;
   int success, i;
   GList *node;
   
   if (!check_status (_("Chdir"), wdata, 0, 0, 0)) return (FALSE);
   if (!expand_path (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry)), tempstr, sizeof (tempstr))) return (FALSE);
   success = 0;
   if (wdata->local == 1) {
      if (chdir(tempstr) == 0) {
         success = 1;
         ftp_log (gftp_logging_misc, NULL, _("Successfully changed local directory to %s\n"), tempstr);
         update_ftp_info (wdata);
         gtk_clist_freeze (GTK_CLIST (wdata->listbox));
         delete_ftp_file_info (wdata);
         add_local_files (wdata);
         gtk_clist_thaw (GTK_CLIST (wdata->listbox));
      }
      else {
         if (wdata->local == 1) {
            ftp_log (gftp_logging_misc, NULL, _("Could not change local directory to %s: %s\n"),
            	tempstr, g_strerror (errno));
         }
         else {
            ftp_log (gftp_logging_misc, NULL, _("Could not change remote directory to %s: %s\n"), 
            	tempstr, g_strerror (errno));
         }
      }
   }
   else {
      if (gftp_set_directory (wdata->hdata->ftpdata, tempstr) == 0) {
         if (wdata->local == 2 && !ftp_connect (wdata->hdata, 0)) return (FALSE);
         wdata->hdata->getdir = 1;
         success = 1;
         update_ftp_info (wdata);
         gtk_clist_freeze (GTK_CLIST (wdata->listbox));
         delete_ftp_file_info (wdata);
         ftp_list_files (wdata, 1);
         gtk_clist_thaw (GTK_CLIST (wdata->listbox));
      }
   }
   if (success) {
      for (node = wdata->history; node != NULL; node = node->next) {
         if (strcmp ((char *) node->data, tempstr) == 0) break;
      }

      if (node == NULL) {
         if (wdata->histlen >= MAX_HIST_LEN) {
            node = wdata->history;
            for (i = 1; i < MAX_HIST_LEN; i++) node = node->next;
            node->prev->next = NULL;
            node->prev = NULL;
            g_list_free (node);
            wdata->histlen = 9;
         }
         str = g_malloc (strlen (tempstr) + 1);
         strcpy (str, tempstr);
         wdata->history = g_list_prepend (wdata->history, str);
         wdata->histlen++;
      }
      else if (node->prev != NULL) {
         node->prev->next = node->next;
         if (node->next != NULL) node->next->prev = node->prev;
         node->prev = NULL;
         node->next = wdata->history;
         if (node->next != NULL) node->next->prev = node;
         wdata->history = node;
      }
      gtk_combo_set_popdown_strings (GTK_COMBO (wdata->combo), wdata->history);
   }
   return (FALSE);
}
/*****************************************************************************/
void toolbar_hostedit (GtkWidget *widget, gpointer data) {
   char *hosttxt;

   hosttxt = gtk_entry_get_text (GTK_ENTRY (hostedit));
   if (*hosttxt == '\0') {
      ftp_log (gftp_logging_error, NULL, _("Error: You must type in a host to connect to\n"));
      return;
   }
   
   if (window2.local != -1) disconnect (&window2);
   
   if (*gtk_entry_get_text (GTK_ENTRY (useredit)) == '\0') setanonymous (NULL, 0, NULL);
   gftp_set_hostname (window2.hdata->ftpdata, hosttxt);
   gftp_set_port (window2.hdata->ftpdata, strtol (gtk_entry_get_text (GTK_ENTRY (portedit)), NULL, 10));
   gftp_set_username (window2.hdata->ftpdata, gtk_entry_get_text (GTK_ENTRY (useredit)));
   gftp_set_password (window2.hdata->ftpdata, gtk_entry_get_text (GTK_ENTRY (passedit)));
   gftp_set_directory (window2.hdata->ftpdata, gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (window2.combo)->entry)));
   if (*proxy_config != '\0' && GTK_CHECK_MENU_ITEM (firewall_btn)->active) {
      if (firewall_host == NULL || *firewall_host == '\0') {
         ftp_log (gftp_logging_error, NULL, "Connect Error: You must specify a proxy hostname in the options dialog\n");
         return;
      }
      
      if (strcmp (proxy_config, "http") == 0) {
         gftp_set_proxy_config (window2.hdata->ftpdata, "");
         window2.hdata->protocol = http;
      }
      else {
         gftp_set_proxy_config (window2.hdata->ftpdata, proxy_config);
      }
   }
   else {
      gftp_set_proxy_config (window2.hdata->ftpdata, "");
      if (strcmp (proxy_config, "http") == 0) window2.hdata->protocol = ftp;
   }
   ftp_connect (window2.hdata, 1);
}
/*****************************************************************************/
static void disconn (struct ftp_window_data *wdata) {
   if (wdata->local == -1) {
      ftp_log (gftp_logging_misc, NULL, _("Disconnect: Not connected to a remote site\n"));
      return;
   }
   disconnect (wdata);
}
/*****************************************************************************/
static void selectdl (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data) {
   struct ftp_transfer_data *tdata;
   int i;
   
   if ((tdata = file_transfers) == NULL) return;
   for (i=0; i<row; i++) {
      tdata = tdata->next;
      if (tdata == NULL) return;
   }
   pthread_mutex_lock (&tdata->mutex);
   tdata->flags |= TRANSFER_SELECTED;
   pthread_mutex_unlock (&tdata->mutex);
}
/*****************************************************************************/
static void unselectdl (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data) {
   struct ftp_transfer_data *tdata;
   int i;
   
   if ((tdata = file_transfers) == NULL) return;
   for (i=0; i<row; i++) {
      tdata = tdata->next;
      if (tdata == NULL) return;
   }
   pthread_mutex_lock (&tdata->mutex);
   tdata->flags &= ~(TRANSFER_SELECTED);
   pthread_mutex_unlock (&tdata->mutex);
}
/*****************************************************************************/
static void stop_transfer (GtkWidget *widget, gpointer data) {
   struct ftp_transfer_data *tdata;

   if (file_transfers == NULL) {
      ftp_log (gftp_logging_misc, NULL, _("There are currently no file transfers in progress to stop\n"));
      return;
   }
   tdata = file_transfers;
   while (tdata != NULL) {
      if (tdata->flags & TRANSFER_SELECTED) {
         pthread_mutex_lock (&tdata->mutex);
         tdata->flags &= ~(TRANSFER_NEW | TRANSFER_SHOW | TRANSFER_UPDATE | TRANSFER_DONE);
         tdata->flags |= (tdata->flags & TRANSFER_STARTED) ? TRANSFER_CANCEL : TRANSFER_DONE;
         pthread_mutex_unlock (&tdata->mutex);
         ftp_log (gftp_logging_misc, NULL, _("Stopping the transfer of %s\n"), tdata->curfle->file);
      }
      tdata = tdata->next;
   }
}
/*****************************************************************************/
static void compare_windows (gpointer data) {
   struct ftp_file_data *curfle, *tempfle;
   gint num;

   if (!check_status (_("Compare Windows"), &window2, 0, 0, 0)) return;
   
   deselectall (&window1);
   deselectall (&window2);
   
   /* Select the items in Window1 that aren't in Window2 */
   curfle = window1.hdata->files;
   num = 0;
   while (curfle != NULL) {
      tempfle = window2.hdata->files;
      if (!(curfle->flags & FILE_ISDIR)) {
         while (tempfle != NULL) {
            if (strcmp (tempfle->file, curfle->file) == 0 && tempfle->size == curfle->size) {
               break;
            }
            tempfle = tempfle->next;
         }
      }
      if (tempfle == NULL) gtk_clist_select_row (GTK_CLIST (window1.listbox), num, 0);
      num++;
      curfle = curfle->next;
   }

   /* Select the items in Window2 that aren't in Window1 */
   curfle = window2.hdata->files;
   num = 0;
   while (curfle != NULL) {
      tempfle = window1.hdata->files;
      if (!(curfle->flags & FILE_ISDIR)) {
         while (tempfle != NULL) {
            if (strcmp (tempfle->file, curfle->file) == 0 && tempfle->size == curfle->size) {
               break;
            }
            tempfle = tempfle->next;
         }
      }
      if (tempfle == NULL) gtk_clist_select_row (GTK_CLIST (window2.listbox), num, 0);
      num++;
      curfle = curfle->next;
   }
}
/*****************************************************************************/
void sortrows (GtkCList *clist, gint column, gpointer data) {
   struct ftp_file_data *tempfle, *prevfle, *curfle, *dirs, *files, *dotdot;
   struct ftp_window_data *wdata;
   GtkWidget *sort_wid;
   int sortdir;

   wdata = (struct ftp_window_data *) data;
   if (!wdata->hdata->files) return;
   if (!check_status (_("Sort"), wdata, 0, 0, 0)) return;
   
   gtk_label_set (GTK_LABEL (wdata->hoststxt), _("Sorting..."));
   fix_display ();
   if (column == 0 || (column == wdata->sortcol && (wdata->flags & WINDOW_SORTED))) {
      if (wdata->flags & WINDOW_SORTASDS) wdata->flags &= ~(WINDOW_SORTASDS);
      else wdata->flags |= WINDOW_SORTASDS;
      column = wdata->sortcol;
      sort_wid = gtk_clist_get_column_widget (clist, 0);
      gtk_widget_destroy (sort_wid);
      if (wdata->flags & WINDOW_SORTASDS) sort_wid = toolbar_pixmap (wdata->listbox, down_xpm);
      else sort_wid = toolbar_pixmap (wdata->listbox, up_xpm);
      gtk_clist_set_column_widget (clist, 0, sort_wid);
   }
   else wdata->sortcol = column;
   sortdir = wdata->flags & WINDOW_SORTASDS;
   dotdot = NULL;
   dirs = files = NULL;
   while (wdata->hdata->files != NULL) {
      curfle = wdata->hdata->files;
      if (strcmp (curfle->file, "..") == 0) {
         dotdot = curfle;
         wdata->hdata->files = wdata->hdata->files->next;
         continue;
      }
      wdata->hdata->files = wdata->hdata->files->next;
      if (sort_dirs_first && curfle->flags & FILE_ISDIR) tempfle = prevfle = dirs;
      else tempfle = prevfle = files;
      if (column == 1) {
         while (tempfle != NULL && (sortdir ?
         	strcmp (tempfle->file, curfle->file) <= 0 :
         	strcmp (tempfle->file, curfle->file) >= 0)) {
            prevfle = tempfle;
            tempfle = tempfle->next;
         }
      }
      else if (column == 2) {
         while (tempfle != NULL && (sortdir ? 
         	tempfle->size <= curfle->size : 
         	tempfle->size >= curfle->size)) {
            prevfle = tempfle;
            tempfle = tempfle->next;
         }
      }
      else if (column == 3) {
         while (tempfle != NULL && (sortdir ?
         	strcmp (tempfle->user, curfle->user) <= 0 :
         	strcmp (tempfle->user, curfle->user) >= 0)) {
            prevfle = tempfle;
            tempfle = tempfle->next;
         }
      }
      else if (column == 4) {
         while (tempfle != NULL && (sortdir ?
         	strcmp (tempfle->group, curfle->group) <= 0 :
         	strcmp (tempfle->group, curfle->group) >= 0)) {
            prevfle = tempfle;
            tempfle = tempfle->next;
         }
      }
      else if (column == 5) { 
         while (tempfle != NULL && (sortdir ? 
         	tempfle->datetime <= curfle->datetime : 
         	tempfle->datetime >= curfle->datetime)) {
            prevfle = tempfle;
            tempfle = tempfle->next;
         }
      }
      else if (column == 6) {
         while (tempfle != NULL && (sortdir ?
         	strcmp (tempfle->attribs, curfle->attribs) <= 0 :
         	strcmp (tempfle->attribs, curfle->attribs) >= 0)) {
            prevfle = tempfle;
            tempfle = tempfle->next;
         }
      }

      if (tempfle == NULL || tempfle == prevfle) {
         if (prevfle == tempfle) {
            if (sort_dirs_first && curfle->flags & FILE_ISDIR) {
               curfle->next = dirs;
               dirs = curfle;
            }
            else {
               curfle->next = files;
               files = curfle;
            }
         }
         else {
            curfle->next = NULL;
            prevfle->next = curfle;
         }
      }
      else {            
         curfle->next = prevfle->next;
         prevfle->next = curfle;
      }
   }
   if (dirs == NULL) wdata->hdata->files = files;
   else {
      tempfle = dirs;
      wdata->hdata->files = dirs;
      while (tempfle->next != NULL) tempfle = tempfle->next;
      tempfle->next = files;
   }
   if (dotdot != NULL) {
      dotdot->next = wdata->hdata->files;
      wdata->hdata->files = dotdot;
   }
   wdata->flags |= WINDOW_SORTED;
      
   gtk_clist_freeze (clist);
   wdata->numselected = 0;
   gtk_clist_clear (clist);
   tempfle = wdata->hdata->files;
   while (tempfle != NULL) {
      add_file_listbox (wdata, tempfle);
      tempfle = tempfle->next;
   }
   gtk_clist_thaw (clist);
   update_ftp_info (wdata);
}
/*****************************************************************************/
void delete_ftp_file_info (struct ftp_window_data *wdata) {
   gtk_clist_clear (GTK_CLIST (wdata->listbox));
   free_file_list (wdata->hdata->files);
   wdata->hdata->last = NULL;
   wdata->hdata->files = NULL;
}
/*****************************************************************************/
void queue_log (gftp_logging_type level, void *ptr, const char *string, ...) {
   struct ftp_log_queue *newlog;
   va_list argp;
   
   newlog = g_malloc (sizeof (struct ftp_log_queue));
   newlog->type = level;
   va_start (argp, string);
   newlog->msg = g_strdup_vprintf (string, argp);
   pthread_mutex_lock (&log_mutex);
   file_transfer_logs = g_list_append (file_transfer_logs, newlog);
   pthread_mutex_unlock (&log_mutex);
}
/*****************************************************************************/
void ftp_log (gftp_logging_type level, void *ptr, const char *string, ...) {
   char tempstr[512];
   GdkColor fore;
   va_list argp;
   guint pos;
   int upd;

   upd = GTK_TEXT (logwdw)->vadj->upper - GTK_TEXT (logwdw)->vadj->page_size == GTK_TEXT (logwdw)->vadj->value;
   fore.red = fore.green = fore.blue = 0;
   if (level == gftp_logging_send) fore.green = 0x8600;
   else if (level == gftp_logging_recv) fore.blue = 0xffff;
   else fore.red = 0xffff;
   gtk_text_freeze (GTK_TEXT (logwdw));

   va_start (argp, string);
   g_vsnprintf (tempstr, sizeof (tempstr), string, argp);
   if (strncmp (tempstr, "PASS", 4) == 0) {
      strncpy (tempstr, "PASS xxxx\r\n", sizeof (tempstr));
   }
   tempstr[sizeof (tempstr) - 1] = '\0';
   gtk_text_insert (GTK_TEXT (logwdw), NULL, &fore, NULL, tempstr, -1);
   
   pos = gtk_text_get_length (GTK_TEXT(logwdw));
   gtk_text_set_point (GTK_TEXT (logwdw), pos);
   gtk_text_thaw (GTK_TEXT (logwdw));
   if (upd) {
      gtk_adjustment_set_value (GTK_TEXT (logwdw)->vadj, 
      		GTK_TEXT (logwdw)->vadj->upper - GTK_TEXT (logwdw)->vadj->page_size);
   }
   fix_display ();
}
/*****************************************************************************/
void update_ftp_info (struct ftp_window_data *wdata) {
   char *tempstr, *dir, empty[] = "";
   
   dir = GFTP_GET_DIRECTORY (wdata->hdata->ftpdata);
   if (wdata->local == 1) {
      tempstr = g_strconcat (_("Local ["), strcmp (wdata->filespec, "*") == 0 ? _("All Files") : wdata->filespec, "]", NULL);
      gtk_label_set (GTK_LABEL (wdata->hoststxt), tempstr);
      if (dir != NULL) gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry), dir);
      else gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry), "");
      g_free (tempstr);
   }
   else if (wdata->local == 0 || wdata->local == 2) {
      tempstr = g_strconcat (GFTP_GET_HOSTNAME (wdata->hdata->ftpdata), wdata->flags & WINDOW_CACHED ? _(" (Cached) [") : " [", strcmp(wdata->filespec, "*") == 0 ? _("All Files") : wdata->filespec, "]", NULL);
      gtk_label_set (GTK_LABEL(wdata->hoststxt), tempstr);
      if (dir != NULL) gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry), dir);
      else gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry), "");
      g_free (tempstr);
      
      if ((tempstr = GFTP_GET_HOSTNAME (wdata->hdata->ftpdata)) == NULL) {
         tempstr = empty;
      }
      gtk_entry_set_text (GTK_ENTRY (hostedit), tempstr);
      
      if ((tempstr = GFTP_GET_USERNAME (wdata->hdata->ftpdata)) == NULL) {
         tempstr = empty;
      }
      gtk_entry_set_text (GTK_ENTRY (useredit), tempstr);
      
      if ((tempstr = GFTP_GET_PASSWORD (wdata->hdata->ftpdata)) == NULL) {
         tempstr = empty;
      }
      gtk_entry_set_text (GTK_ENTRY (passedit), tempstr);

      tempstr = g_strdup_printf ("%d", GFTP_GET_PORT (wdata->hdata->ftpdata));
      gtk_entry_set_text (GTK_ENTRY (portedit), tempstr);
      g_free (tempstr);
   }
   else {
      gtk_label_set (GTK_LABEL (wdata->hoststxt), _("Not connected"));
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry), "");
   }
   fix_display ();
}
/*****************************************************************************/
void refresh (struct ftp_window_data *wdata) {
   if (!check_status (_("Refresh"), wdata, 0, 0, 0)) return;
   gtk_clist_freeze (GTK_CLIST (wdata->listbox));
   delete_ftp_file_info (wdata);
   if (wdata->local == 0) ftp_list_files (wdata, 0);
   else add_local_files (wdata);
   gtk_clist_thaw (GTK_CLIST (wdata->listbox));
   update_ftp_info (wdata);
}
/*****************************************************************************/
void fix_display (void) {
   while (gtk_events_pending ()) gtk_main_iteration ();
}
/*****************************************************************************/
static gint update_downloads (gpointer data) {
   char *add_data[3] = {NULL, NULL, NULL}, *pos;
   struct ftp_transfer_data *tdata, *prevtdata;
   struct ftp_window_data *tempwdata, *ckwdata;
   struct ftp_log_queue *templog;
   gint num, flag, updated;
   char *tempstr;
   
   if (file_transfer_logs != NULL) {
      pthread_mutex_lock (&log_mutex);
      while (file_transfer_logs != NULL) {
         templog = file_transfer_logs->data;
         ftp_log (templog->type, NULL, templog->msg);
         g_free (templog->msg);
         file_transfer_logs = g_list_remove (file_transfer_logs, templog);
      }
      pthread_mutex_unlock (&log_mutex);
   }

   if (viewedit_process_done) check_done_process ();

   if (file_transfers == NULL) {
      gtk_timeout_add (1000, update_downloads, data);
      return (0);
   }
   else {
      tdata = prevtdata = file_transfers;
      num = 0;
      while (tdata != NULL) {
         pthread_mutex_lock (&tdata->mutex);
         if (tdata->flags & TRANSFER_ON_NEXT_FILE) {
            tdata->flags &= ~(TRANSFER_ON_NEXT_FILE);
            while (tdata->updfle != tdata->curfle) {
               if (tdata->updfle->flags & FILE_TRANS_DONE_VIEW) {
                  view_file (tdata->updfle->file, 1, 1, 1, tdata->updfle->remote_file);
               }
               else if (tdata->updfle->flags & FILE_TRANS_DONE_EDIT) {
                  view_file (tdata->updfle->file, 0, 1, 1, tdata->updfle->remote_file);
               }
               else if (tdata->updfle->flags & FILE_TRANS_DONE_RM) {
                  unlink (tdata->updfle->file);
               }
               tdata->updfle = tdata->updfle->next;
            }
            tempwdata = (tdata->flags & TRANSFER_DIRECTION) ? &window1 : &window2;
            if (refresh_files && tdata->curfle && tdata->curfle->next &&
            	((tdata->flags & TRANSFER_DIRECTION) || 
            	compare_hdata_structs (tdata->hdata->ftpdata, tempwdata->hdata->ftpdata, 1))) {
               refresh (tempwdata);
            }

            if (tdata->curfle != NULL) {
               if ((pos = strrchr (tdata->curfle->file, '/')) == NULL) {
                  pos = tdata->curfle->file;
               }
               else pos++;
               gtk_clist_set_text (GTK_CLIST(dlwdw), num, 0, pos);
            }
         }
      
         if (tdata->flags & TRANSFER_SHOW) {
            g_snprintf (tdata->progressstr, sizeof (tdata->progressstr), _("Waiting... (%d files)"), 
            	tdata->hdata->totalfiles);
            tdata->progressstr[sizeof (tdata->progressstr) - 1] = '\0';
            gtk_clist_insert (GTK_CLIST (dlwdw), num, add_data);
            tdata->flags &= ~(TRANSFER_NEW | TRANSFER_SHOW | TRANSFER_UPDATE | TRANSFER_DONE);
            tdata->flags |= TRANSFER_UPDATE | TRANSFER_CREATED | TRANSFER_NEED_UPDATED;
            tdata->curfle = tdata->hdata->files;
            tdata->updfle = tdata->hdata->files;

            if ((pos = strrchr (tdata->curfle->file, '/')) == NULL) {
               pos = tdata->curfle->file;
            }
            else pos++;
            gtk_clist_set_text (GTK_CLIST(dlwdw), num, 0, pos);

            tempstr = g_strconcat (GFTP_GET_HOSTNAME (tdata->hdata->ftpdata), GFTP_GET_DIRECTORY (tdata->hdata->ftpdata), NULL);
            gtk_clist_set_text (GTK_CLIST (dlwdw), num, 2, tempstr);
            g_free (tempstr);
         }
         else if (tdata->flags & TRANSFER_DONE) {
            if (tdata->flags & TRANSFER_DIRECTION) {
               tempwdata = &window1;
               ckwdata = &window2;
            }
            else {
               tempwdata = &window2;
               ckwdata = &window1;
            }
            updated = 0;
            if (!window2.hdata->stopable && window2.local == 2 && GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata) >= 0 &&
            	compare_hdata_structs (tdata->hdata->ftpdata, window2.hdata->ftpdata, 0)) {

               window2.local = 0;
               updated = 1;
               GFTP_GET_CONTROL_FD (window2.hdata->ftpdata) = GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata);
               GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata) = -1;
               refresh (&window2);
            }
            else gftp_disconnect (tdata->hdata->ftpdata);
            
            if (compare_hdata_structs (tdata->hdata->ftpdata, ckwdata->hdata->ftpdata, 1)) {
               refresh (tempwdata);
            }

            if (tdata->flags & TRANSFER_CREATED) {
               /* We have to release the mutex because when the list item gets
                  deleted, it will call my deselect function. This func will
                  also lock the mutex */
               pthread_mutex_unlock (&tdata->mutex);
               gtk_clist_remove (GTK_CLIST (dlwdw), num);
            }

            if (tdata == prevtdata) {
               flag = 1;
               file_transfers = file_transfers->next;
            }
            else {
               flag = 0;
               prevtdata->next = tdata->next;
            }
            if ((tdata->flags & TRANSFER_STARTED) && (do_one_transfer_at_a_time || file_transfers == NULL)) transfer_in_progress = 0;
            free_tdata (tdata);
            pthread_mutex_lock (&transfer_mutex);
            if (flag) tdata = prevtdata = file_transfers;
            else tdata = prevtdata->next;
            pthread_mutex_unlock (&transfer_mutex);
            continue;
         }

         if ((tdata->curfle != NULL) && (tdata->flags & (TRANSFER_UPDATE | TRANSFER_NEED_UPDATED))) {
            if (!(tdata->flags & TRANSFER_STARTED) && start_file_transfers && (!do_one_transfer_at_a_time || (do_one_transfer_at_a_time && !transfer_in_progress))) {
               if (window2.local == 0 && !window2.hdata->stopable &&
               		compare_hdata_structs (tdata->hdata->ftpdata, window2.hdata->ftpdata, 0)) {

                  GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata) = GFTP_GET_CONTROL_FD (window2.hdata->ftpdata);
                  GFTP_GET_CONTROL_FD (window2.hdata->ftpdata) = -1;
                  window2.local = 2;
                  window2.flags |= WINDOW_CACHED;
                  update_ftp_info (&window2);
               }
               transfer_in_progress = 1;
               tdata->flags |= TRANSFER_STARTED;
               if (tdata->flags & TRANSFER_DIRECTION) pthread_create (&tdata->tid, NULL, ftp_get_files, tdata);
               else pthread_create (&tdata->tid, NULL, ftp_put_files, tdata);
            }

            gtk_clist_set_text (GTK_CLIST(dlwdw), num, 1, tdata->progressstr);

            tdata->flags &= ~(TRANSFER_NEED_UPDATED);
         }
         num++;
         pthread_mutex_unlock (&tdata->mutex);
         prevtdata = tdata;
         tdata = tdata->next;
      }
   }
   gtk_timeout_add (1000, update_downloads, data);
   return (0);
}
/*****************************************************************************/
static void check_done_process (void) {
   struct viewedit_data *ve_proc;
   GList *curdata, *deldata;
   struct stat st;
   int ret, i;
   char *str;
   pid_t pid;

   viewedit_process_done = 0;
   while ((pid = waitpid (-1, &ret, WNOHANG)) > 0) {
      curdata = viewedit_processes;
      while (curdata != NULL) {
         ve_proc = (struct viewedit_data *) curdata->data;
         if (ve_proc->pid == pid) {
            for (i=0; ve_proc->argv[i] != NULL; i++) g_free (ve_proc->argv[i]);
            if (ret != 0) {
               ftp_log (gftp_logging_error, NULL, _("Error: Child %d returned %d\n"), pid, ret);
            }
            else {
               ftp_log (gftp_logging_misc, NULL, _("Child %d returned successfully\n"), pid);
            }
            if (!(ve_proc->flags & VIEWEDIT_VIEW_FILE)) {
               /* We was editing the file. Upload it */
               if (stat (ve_proc->filename, &st) == -1) {
                  ftp_log (gftp_logging_error, NULL, _("Error: Cannot get information about file %s: %s\n"), ve_proc->filename, g_strerror (errno));
               }
               else if (st.st_mtime == ve_proc->st.st_mtime) {
                  ftp_log (gftp_logging_misc, NULL, _("File %s was not changed\n"), ve_proc->remote_filename);
               }
               else {
                  memcpy (&ve_proc->st, &st, sizeof (ve_proc->st));
                  str = g_strdup_printf (_("File %s has changed.\nWhat would you like to do?"), ve_proc->remote_filename);
                  MakeYesNoDialog (_("Edit File"), str, 1, 2, _("Upload"), do_upload, curdata, _("  Cancel  "), free_edit_data, curdata);
                  g_free (str);
                  curdata = g_list_next (curdata);
                  continue;
               }
            }
            else if (ve_proc->flags & VIEWEDIT_RM_FILE) unlink (ve_proc->filename);
         
            deldata = curdata;
            curdata = curdata->next;
            if (ve_proc->filename) g_free (ve_proc->filename);
            if (ve_proc->remote_filename) g_free (ve_proc->remote_filename);
            g_free (ve_proc);
            viewedit_processes = g_list_remove_link (viewedit_processes, deldata);
            continue;
         }
         curdata = curdata->next;
      }
   }
}
/*****************************************************************************/
static void do_upload (GtkWidget *widget, struct dialog_data *data) {
   struct viewedit_data *ve_proc;
   struct ftp_transfer_data *tdata;
   GList *curdata;

   curdata = (GList *) data->data;
   ve_proc = (struct viewedit_data *) curdata->data;
   tdata = transfer_one_file (ve_proc->filename, ve_proc->remote_filename, 0);
   tdata->hdata->files->size = ve_proc->st.st_size;
   tdata->hdata->files->flags |= FILE_TRANS_DONE_RM;
   add_file_transfer (tdata);
   free_edit_data (NULL, data);
}
/*****************************************************************************/
static void free_edit_data (GtkWidget *widget, struct dialog_data *data) {
   struct viewedit_data *ve_proc;
   GList *curdata;

   curdata = (GList *) data->data;
   ve_proc = (struct viewedit_data *) curdata->data;
   if (ve_proc->filename) g_free (ve_proc->filename);
   if (ve_proc->remote_filename) g_free (ve_proc->remote_filename);
   g_free (ve_proc);
   viewedit_processes = g_list_remove_link (viewedit_processes, curdata);
}
/*****************************************************************************/
