/*****************************************************************************/
/*  misc.c - general purpose routines                                        */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"

char *insert_commas (unsigned long number, char *dest, int size) {
   char *frompos, *topos, *src;
   int num, rem, i;
   
   src = g_malloc (size);
   g_snprintf (src, size, "%ld", number);
   src[size-1] = '\0';

   num = strlen (src) / 3 - 1;
   rem = strlen (src) % 3;
   if (strlen (src) + num + (rem == 0 ? 0 : 1) + 1 > size) {
      if (strlen (src) < size) strncpy (dest, src, size);
      else memset (dest, 'X', size-1);
      dest[size-1] = '\0';
      g_free (src);
      return (dest);
   }
   frompos = src;
   topos = dest;
   for (i=0; i<rem; i++) *topos++ = *frompos++;
   if (*frompos != '\0') {
      if (rem != 0) *topos++ = ',';
      while (num > 0) {
         for (i=0; i<3; i++) *topos++ = *frompos++;
         *topos++ = ',';
         num--;
      }
      for (i=0; i<3; i++) *topos++ = *frompos++;
   }
   *topos = '\0';
   g_free (src);
   return (dest);
}
/*****************************************************************************/
void add_local_files (struct ftp_window_data *wdata) {
   char tempstr[MAXSTR];

   wdata->local = 1;
   wdata->flags &= ~(WINDOW_CACHED);
   wdata->totalitems = wdata->numselected = 0;
   wdata->flags &= ~(WINDOW_SORTED);
   gftp_set_directory (wdata->hdata->ftpdata, getcwd (tempstr, sizeof (tempstr)));
   if ((wdata->hdata->files = get_local_files (NULL, &wdata->totalitems)) == NULL) {
      wdata->local = -1;
      wdata->totalitems = wdata->numselected = 0;
   }
   else sortrows (GTK_CLIST(wdata->listbox), wdata->sortcol, (gpointer) wdata);
   gtk_clist_select_row (GTK_CLIST (wdata->listbox), 0, 0);
   update_ftp_info (wdata);
}
/*****************************************************************************/
struct ftp_file_data *get_local_files (char *path, int *total) {
   struct ftp_file_data *files;
   char curdir[MAXSTR];
   FILE *dir;
   
   *total = 0;
   if (path != NULL) {
      if (getcwd (curdir, sizeof (curdir)) == NULL) return (NULL);
      if (chdir(path) == -1) return (NULL);
   }
   dir = popen ("/bin/ls -al", "r");
   files = parse_local_file (dir, 1, total, ftp);
   pclose (dir);
   if (path != NULL) chdir (curdir);
   return (files);
}
/*****************************************************************************/
struct ftp_file_data *parse_local_file (FILE *fd, int local, int *total, protocol_type proto) {
   struct ftp_file_data *files, *newfle, *lastfle;
   char tempstr[MAXSTR];
   int isdotdot = 0;
   struct stat st;
   gftp_file fle;

   lastfle = files = NULL;
   while (fgets (tempstr, sizeof (tempstr)-1, fd)) {
      if (tempstr[strlen(tempstr)-1] == '\n') tempstr[strlen(tempstr)-1] = '\0';
      if (tempstr[strlen(tempstr)-1] == '\r') tempstr[strlen(tempstr)-1] = '\0';

      if (proto == ftp ? gftp_parse_ls (tempstr, &fle) == 0 : parse_html_line (tempstr, &fle)) {
         if (strcmp (fle.file, ".") == 0) continue;
         if (strcmp(fle.file, "..") == 0) isdotdot = 1;

         newfle = g_malloc0 (sizeof (struct ftp_file_data));
         memcpy (newfle, &fle, sizeof (gftp_file));
         newfle->flags = 0;
         if (local) {
            stat (newfle->file, &st);
            if (S_ISDIR (st.st_mode)) newfle->flags |= FILE_ISDIR;
            if (newfle->attribs[0] == 'l') newfle->flags |= FILE_ISLINK;
            if ((strchr (newfle->attribs, 'x') != NULL) && !(newfle->flags & FILE_ISDIR)
            	&& !(newfle->flags & FILE_ISLINK)) newfle->flags |= FILE_ISEXE;
         }
         else {
            if (strchr (newfle->attribs, 'd') != NULL) newfle->flags |= FILE_ISDIR;
            if (strchr (newfle->attribs, 'l') != NULL) newfle->flags |= FILE_ISLINK;
            if ((strchr (newfle->attribs, 'x') != NULL) && !(newfle->flags & FILE_ISDIR)
            	&& !(newfle->flags & FILE_ISLINK)) newfle->flags |= FILE_ISEXE;
         }
         newfle->next = NULL;
         if (lastfle == NULL) files = newfle;
         else lastfle->next = newfle;
         lastfle = newfle;  
         (*total)++;
      }
   }
   if (!isdotdot) {
      newfle = g_malloc0 (sizeof (struct ftp_file_data));
      newfle->file = g_malloc (3);
      strcpy (newfle->file, "..");
      newfle->user = g_malloc (1);
      *newfle->user = '\0';
      newfle->group = g_malloc (1);
      *newfle->group = '\0';
      newfle->attribs = g_malloc (1);
      *newfle->attribs = '\0';
      
      newfle->next = NULL;
      newfle->flags = FILE_ISDIR;
      if (lastfle == NULL) files = newfle;
      else lastfle->next = newfle;
      lastfle = newfle;  
      (*total)++;
   }
   return (files);
}
/*****************************************************************************/
void add_file_listbox (struct ftp_window_data *wdata, struct ftp_file_data *fle) {
   char *add_data[7] = {NULL, NULL, NULL, NULL, NULL, NULL, NULL};
   struct pix_ext *tempext;
   char tempstr[50], *str;
   size_t stlen;

   if (!(!show_hidden_files && *fle->file == '.' && strcmp (fle->file, "..") != 0) &&
   	gtk_pattern_match_simple (wdata->filespec, fle->file)) {
      fle->flags |= FILE_SHOWN;
   }
   else {
      fle->flags &= ~(FILE_SHOWN | FILE_SELECTED);
      return;
   }
   stlen = strlen (fle->file);
   fle->clist_num = gtk_clist_append (GTK_CLIST (wdata->listbox), add_data);
   if (fle->flags & FILE_SELECTED) {
      gtk_clist_select_row (GTK_CLIST (wdata->listbox), fle->clist_num, 0);
      wdata->numselected++;
   }
   if (strcmp(fle->file, "..") == 0) {
      gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), fle->clist_num, 0, dotdot_pixmap, dotdot_mask);
   }
   else if ((fle->flags & FILE_ISLINK) && (fle->flags & FILE_ISDIR)) {
      gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), fle->clist_num, 0, linkdir_pixmap, linkdir_mask);
   }
   else if (fle->flags & FILE_ISLINK) {
      gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), fle->clist_num, 0, linkfile_pixmap, linkfile_mask);
   }
   else if (fle->flags & FILE_ISDIR) {
      gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), fle->clist_num, 0, dir_pixmap, dir_mask);
   }
   else if (fle->flags & FILE_ISEXE) {
      gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), fle->clist_num, 0, exe_pixmap, exe_mask);
   }
   else {
      tempext = registered_exts;
      while (tempext != NULL) {
         if (stlen >= tempext->stlen &&
            strcmp (&fle->file[stlen-tempext->stlen], tempext->ext) == 0) {

            if (toupper(*tempext->ascii_binary) == 'A') {
               fle->flags |= FILE_ASCII;
            }
            
            if (tempext->pixmap != NULL && *tempext->filename != '\0') {
               gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), fle->clist_num, 0, tempext->pixmap, tempext->mask);
            }
            else tempext = NULL;
            break;
         }
         tempext = tempext->next;
      }
      if (tempext == NULL) {
         gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), fle->clist_num, 0, doc_pixmap, doc_mask);
      }
   }
   gtk_clist_set_text (GTK_CLIST (wdata->listbox), fle->clist_num, 1, fle->file);

   if (fle->attribs && (*fle->attribs == 'b' || *fle->attribs == 'c')) {
      g_snprintf (tempstr, sizeof (tempstr), "%d, %d", (int) fle->size >> 16, (int) fle->size & 0xFF);
   }
   else {
      insert_commas (fle->size, tempstr, sizeof (tempstr));
   }
   tempstr[sizeof (tempstr) - 1] = '\0';
   gtk_clist_set_text (GTK_CLIST (wdata->listbox), fle->clist_num, 2, tempstr);

   if (fle->user) {
      gtk_clist_set_text (GTK_CLIST (wdata->listbox), fle->clist_num, 3, fle->user);
   }
   if (fle->group) {
      gtk_clist_set_text (GTK_CLIST (wdata->listbox), fle->clist_num, 4, fle->group);
   }
   if ((str = ctime (&fle->datetime))) {
      gtk_clist_set_text (GTK_CLIST (wdata->listbox), fle->clist_num, 5, str);
   }
   if (fle->attribs) {
      gtk_clist_set_text (GTK_CLIST (wdata->listbox), fle->clist_num, 6, fle->attribs);
   }
}
/*****************************************************************************/
long file_countlf (int filefd, long endpos) {
   char tempstr[MAXSTR];
   long num = 0, mypos;
   ssize_t n;
   int i;
            
   lseek (filefd, 0, SEEK_SET);
   mypos = 0;
   while ((n = read (filefd, tempstr, sizeof (tempstr))) > 0) {
      for (i=0; i<n; i++) {
         if ((tempstr[i] == '\n') && (i > 0) && (tempstr[i-1] != '\r')
            && (endpos == 0 || mypos+i <= endpos)) ++num;
      }
      mypos += n;
   }
   lseek (filefd, 0, SEEK_SET);
   return (num);
}
/*****************************************************************************/
char *alltrim (char *str) {
   char *pos, *newpos;
   int diff;

   pos = str + strlen (str);
   while (*pos == ' ') *pos-- = '\0';
   
   pos = str;
   diff = 0;
   while (*pos++ == ' ') diff++;

   pos = str + diff;
   newpos = str;
   while (*pos != '\0') *newpos++ = *pos++;
   *newpos = '\0';
   return (str);
}
/*****************************************************************************/
int expand_path (char *src, char *dest, size_t size) {
   char *prevpos, *pos, *endpos, tempchar, *newstr, *str;
   struct passwd *pw;
   size_t len;
   
   pw = NULL;
   if (strcmp(src, "~") == 0 || strncmp (src, "~/", 2) == 0) pw = getpwuid (geteuid ());
   else if (strncmp(src, "~", 1) == 0) {
      if ((pos = strchr (src, '/')) == NULL) {
         *dest = '\0';
         return (0);
      }
      tempchar = *pos;
      *pos = '\0';
      pw = getpwnam (src+1);
      *pos = tempchar;
   }
   
   newstr = g_malloc (size);
   memset (newstr, 0, size);
   len = size-1;
   str = g_malloc (strlen (src) + 1);
   strcpy (str, src);
   endpos = str;
   while ((pos = strchr (endpos, '/')) != NULL) {
      pos++;
      if ((endpos = strchr (pos, '/')) == NULL) endpos = strchr (pos, '\0');
      tempchar = *endpos;
      *endpos = '\0';
      if (strcmp (pos, "..") == 0) {
         *(pos-1) = '\0';
         if ((prevpos = strrchr (newstr, '/')) != NULL) *prevpos = '\0';
      }
      else if (strcmp (pos, ".") != 0) {
         len -= strlen (pos);
         if (len < 1) break;
         strncat (newstr, pos-1, len);
      }
      *endpos = tempchar;
      if (*endpos == '\0') break;
      endpos = pos+1;
   }
   g_free (str);
   if (*newstr == '\0') {
      *newstr = '/';
      *(newstr+1) = '\0';
   }
   
   if (pw != NULL) {
      if (pos == NULL) {
         strncpy (dest, pw->pw_dir, size);
      }
      else {
         g_snprintf (dest, size, "%s%s", pw->pw_dir, newstr);
      }
      dest[size-1] = '\0';
   }
   else {
      strncpy (dest, newstr, size);
      dest[size-1] = '\0';
   }

   g_free (newstr);   
   return (1);
}
/*****************************************************************************/
void remove_double_slashes (char *string) {
   char *newpos, *oldpos;
   
   oldpos = newpos = string;
   while (*oldpos != '\0') {
      *newpos++ = *oldpos++;
      if (*oldpos == '\0') break;
      while (*(newpos-1) == '/' && *(oldpos) == '/') oldpos++;
   }
   *newpos = '\0';
   if (string[strlen (string) - 1] == '/') string[strlen (string) - 1] = '\0';
}
/*****************************************************************************/
char *make_temp_filename (char *destbuf, char *ext, int size) {
   char tempstr[MAXSTR];
   
   expand_path (BASE_CONF_DIR "/temp", tempstr, sizeof(tempstr));
   if (access (tempstr, F_OK) == -1) mkdir (tempstr, 0x1C0);

   srand (time (NULL));
   g_snprintf (destbuf, size, "%s/temp%ld%s", tempstr, 1+(long) (99999999.0*rand()/(RAND_MAX+1.0)), ext == NULL ? "" : ext);
   destbuf[size-1] = '\0';
   while (access (destbuf, F_OK) != -1) {
      g_snprintf (destbuf, size, "%s/temp%ld%s", tempstr, 1+(long) (99999999.0*rand()/(RAND_MAX+1.0)), ext == NULL ? "" : ext);
      destbuf[size-1] = '\0';
   }
   return (destbuf);
}
/*****************************************************************************/
void free_file_list (struct ftp_file_data *filelist) {
   struct ftp_file_data *tempfle;
   
   while (filelist != NULL) {
      tempfle = filelist;
      filelist = filelist->next;
      free_fdata (tempfle);
   }
}
/*****************************************************************************/
void free_fdata (struct ftp_file_data *fle) {
   if (fle->file) g_free (fle->file);
   if (fle->user) g_free (fle->user);
   if (fle->group) g_free (fle->group);
   if (fle->attribs) g_free (fle->attribs);
   g_free (fle);
}
/*****************************************************************************/
void free_hdata (struct ftp_host_data *hdata) {
   free_file_list (hdata->files);
   gftp_request_destroy (hdata->ftpdata);
   g_free (hdata);
}
/*****************************************************************************/
struct ftp_host_data *new_hdata (void) {
   struct ftp_host_data *hdata;
   
   hdata = g_malloc0 (sizeof (struct ftp_host_data));
   hdata->files = hdata->last = NULL;
   hdata->ftpdata = gftp_request_new ();
   gftp_set_username (hdata->ftpdata, ANON_LOGIN);
   gftp_set_password (hdata->ftpdata, emailaddr);
   gftp_set_proxy_hostname (hdata->ftpdata, firewall_host);
   gftp_set_proxy_username (hdata->ftpdata, firewall_username);
   gftp_set_proxy_password (hdata->ftpdata, firewall_password);
   gftp_set_proxy_account (hdata->ftpdata, firewall_account);
   gftp_set_proxy_port (hdata->ftpdata, firewall_port);
   gftp_set_data_type (hdata->ftpdata, gftp_type_binary);
   gftp_logging (hdata->ftpdata, 1, ftp_log, NULL);
   hdata->protocol = strcmp (proxy_config, "http") == 0 ? http : ftp;
   gftp_set_proxy_config (hdata->ftpdata, proxy_config);
   GFTP_SET_TRANSFER_TYPE (hdata->ftpdata, 
   	passive_transfer ? gftp_transfer_passive : gftp_transfer_active);
   return (hdata);
}
/*****************************************************************************/
void free_tdata (struct ftp_transfer_data *tdata) {
   free_hdata (tdata->hdata);
   pthread_mutex_destroy (&tdata->mutex);
   g_free (tdata);
}
/*****************************************************************************/
struct ftp_transfer_data *new_tdata (void) {
   struct ftp_transfer_data *tdata;

   tdata = g_malloc0 (sizeof (struct ftp_transfer_data));
   tdata->hdata = new_hdata ();
   gftp_logging (tdata->hdata->ftpdata, 1, queue_log, NULL);
   tdata->flags = TRANSFER_NEW; /* Signal to make this listbox entry */
   tdata->curtrans = 0;
   tdata->dirs_to_be_made = g_malloc (sizeof (char *));
   tdata->dirs_to_be_made = NULL;
   tdata->num_dirs_to_be_made = 0;
   pthread_mutex_init (&tdata->mutex, NULL);
   tdata->hdata->wdata = &window2;
   return (tdata);
}
/*****************************************************************************/
void copy_hdata_struct (struct ftp_host_data *hdata, struct ftp_host_data *newhdata) {
   char *str;

   if ((str = GFTP_GET_HOSTNAME (hdata->ftpdata))) {
      gftp_set_hostname (newhdata->ftpdata, str);
   }
   if ((str = GFTP_GET_USERNAME (hdata->ftpdata))) {
      gftp_set_username (newhdata->ftpdata, str);
   }
   if ((str = GFTP_GET_PASSWORD (hdata->ftpdata))) {
      gftp_set_password (newhdata->ftpdata, str);
   }
   if ((str = GFTP_GET_DIRECTORY (hdata->ftpdata))) {
      gftp_set_directory (newhdata->ftpdata, str);
   }
   gftp_set_port (newhdata->ftpdata, GFTP_GET_PORT (hdata->ftpdata));
   if ((str = GFTP_GET_PROXY_HOSTNAME (hdata->ftpdata))) {
      gftp_set_proxy_hostname (newhdata->ftpdata, str);
   }
   if ((str = GFTP_GET_PROXY_USERNAME (hdata->ftpdata))) {
      gftp_set_proxy_username (newhdata->ftpdata, str);
   }
   if ((str = GFTP_GET_PROXY_PASSWORD (hdata->ftpdata))) {
      gftp_set_proxy_password (newhdata->ftpdata, str);
   }
   gftp_set_proxy_port (newhdata->ftpdata, GFTP_GET_PROXY_PORT (hdata->ftpdata));
   GFTP_SET_LOGGING (newhdata->ftpdata, GFTP_GET_LOGGING (hdata->ftpdata));
   gftp_set_data_type (newhdata->ftpdata, GFTP_GET_DATA_TYPE (hdata->ftpdata));
   newhdata->ftpdata->transfer_type = hdata->ftpdata->transfer_type;
   newhdata->ftpdata->logging_function = hdata->ftpdata->logging_function;
   newhdata->ftpdata->user_data = hdata->ftpdata->user_data;
   newhdata->protocol = hdata->protocol;
}
/*****************************************************************************/
void copy_fdata_struct (struct ftp_file_data *fle, struct ftp_file_data *newfle) {
   if (fle->file) {
      newfle->file = g_malloc (strlen (fle->file) + 1);
      strcpy (newfle->file, fle->file);
   }
   if (fle->remote_file) {
      newfle->remote_file = g_malloc (strlen (fle->remote_file) + 1);
      strcpy (newfle->remote_file, fle->remote_file);
   }
   if (fle->user) {
      newfle->user = g_malloc (strlen (fle->user) + 1);
      strcpy (newfle->user, fle->user);
   }
   if (fle->group) {
      newfle->group = g_malloc (strlen (fle->group) + 1);
      strcpy (newfle->group, fle->group);
   }
   if (fle->attribs) {
      newfle->attribs = g_malloc (strlen (fle->attribs) + 1);
      strcpy (newfle->attribs, fle->attribs);
   }
   newfle->datetime = fle->datetime;
   newfle->size = fle->size;
   newfle->remote_size = fle->remote_size;
   newfle->flags = fle->flags;
}
/*****************************************************************************/
int compare_hdata_structs (gftp_request *request1, gftp_request *request2, int compare_dirs) {
   char *strarr[4][2];
   int i, ret;

   ret = 1;
   if(GFTP_GET_PORT (request1) == GFTP_GET_PORT (request2)) {
      strarr[0][0] = GFTP_GET_HOSTNAME (request1);
      strarr[0][1] = GFTP_GET_HOSTNAME (request2);
      strarr[1][0] = GFTP_GET_USERNAME (request1);
      strarr[1][1] = GFTP_GET_USERNAME (request2);
      strarr[2][0] = GFTP_GET_PASSWORD (request1);
      strarr[2][1] = GFTP_GET_PASSWORD (request2);
      if (compare_dirs) {
         strarr[3][0] = GFTP_GET_DIRECTORY (request1);
         strarr[3][1] = GFTP_GET_DIRECTORY (request2);
      }
      else {
         strarr[3][0] = strarr[3][1] = "";
      }
      for (i=0; i<4; i++) {
         if ((strarr[i][0] && !strarr[i][1]) || (!strarr[i][0] && strarr[i][1])) {
            ret = 0;
            break;
         }

         if (strarr[i][0] && strarr[i][1] && strcmp (strarr[i][0], strarr[i][1]) != 0) {
            ret = 0;
            break;
         }
      }
   }
   else ret = 0;
   return (ret);
}
/*****************************************************************************/
struct ftp_file_data *get_next_selected_filename (struct ftp_file_data *filelist) {
   struct ftp_file_data *tempfle;
   
   tempfle = filelist;
   while (tempfle != NULL) {
      if ((tempfle->flags & FILE_SHOWN) && (tempfle->flags & FILE_SELECTED)) {
         return (tempfle);
      }
      tempfle = tempfle->next;
   }
   return (NULL);
}
/*****************************************************************************/
void open_xpm (char *filename, GtkWidget *parent, GdkPixmap **pixmap, GdkBitmap **mask, int quit_on_err) {
   char tempstr[MAXSTR], *temp1str;
   GtkStyle *style;

   if (*filename == '\0') return;
   style = gtk_widget_get_style (parent);
   temp1str = g_strconcat (BASE_CONF_DIR, "/", filename, NULL);
   expand_path (temp1str, tempstr, sizeof (tempstr));
   if (access (tempstr, F_OK) != 0) {
      g_free (temp1str);
      temp1str = g_strconcat (SHARE_DIR, "/", filename, NULL);
      expand_path (temp1str, tempstr, sizeof (tempstr));
      if (access (tempstr, F_OK) != 0) {
         g_free (temp1str);
         temp1str = g_strconcat ("/usr/share/icons/", filename, NULL);
         expand_path (temp1str, tempstr, sizeof (tempstr));
         if (access (tempstr, F_OK) != 0) {
            g_free (temp1str);
            if (!quit_on_err) return;
            printf (_("gFTP Error: Cannot find file %s in %s or %s\n"), filename, SHARE_DIR, BASE_CONF_DIR);
            exit (-1);
         }
      }
   }
   *pixmap = gdk_pixmap_create_from_xpm (parent->window, mask, &style->bg[GTK_STATE_NORMAL], tempstr);
   if (*pixmap == NULL) {
      g_free (temp1str);
      if (!quit_on_err) return;
      printf(_("gFTP Error: Error opening file %s\n"), tempstr);
      exit (-1);
   }
   g_free (temp1str);
}
/*****************************************************************************/
GtkWidget *toolbar_pixmap (GtkWidget *widget, char **xpmdata) {
   GtkWidget *pix;
   GdkPixmap *pixmap;
   GdkBitmap *bitmap;
   
   pixmap = gdk_pixmap_create_from_xpm_d (widget->window, &bitmap, 
  	&widget->style->bg[GTK_STATE_NORMAL], xpmdata);

   pix = gtk_pixmap_new (pixmap, bitmap);
   gtk_widget_show (pix);

   gdk_pixmap_unref (pixmap);
   gdk_pixmap_unref (bitmap);
   return (pix);
}
/*****************************************************************************/
int check_status(char *name, struct ftp_window_data *wdata, int only_one, 
	int at_least_one, int use_ftp_protocol) {

   if (wdata->hdata->stopable) {
      ftp_log (gftp_logging_misc, NULL, _("%s: Please hit the stop button first to do anything else\n"), name);
      return (0);
   }

   if ((wdata)->local == -1) {
      ftp_log (gftp_logging_misc, NULL, _("%s: Not connected to a remote site\n"), name);
      return (0);
   }

   if (use_ftp_protocol && wdata->hdata->protocol != ftp && wdata->local != 1) {
      ftp_log (gftp_logging_misc, NULL, _("%s: You must be connected through the FTP protocol\n"), name);
      return (0);
   }

   if (only_one && (wdata)->numselected != 1) {
      ftp_log (gftp_logging_misc, NULL, _("%s: You must only have one item selected\n"), name);
      return (0);
   }

   if (at_least_one && !only_one && (wdata)->numselected == 0) {
      ftp_log (gftp_logging_misc, NULL, _("%s: You must have at least one item selected\n"), name);
      return (0);
   }
   return (1);
}
/*****************************************************************************/
void make_nonnull (char **str) {
   if (*str == NULL) {
      *str = g_malloc (1);
      **str = '\0';
   }
}
/*****************************************************************************/
void create_item_factory (GtkItemFactory *ifactory, guint n_entries, GtkItemFactoryEntry *entries, gpointer callback_data) {
   GtkItemFactoryEntry dummy_item;
   int i;
   
   for (i=0; i<n_entries; i++) {
      memcpy (&dummy_item, entries + i, sizeof (GtkItemFactoryEntry));
      dummy_item.path = _(entries[i].path);
      gtk_item_factory_create_item (ifactory, &dummy_item, callback_data, 1);
   }
}
/*****************************************************************************/
