/* db.c - Interface to database.

   Copyright (C) 1998 Tom Tromey

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */


/* In our case, the database maps the local file name onto the remote
   file name and the checksum.  This is done by storing the value as a
   buffer holding two strings separated by `\0': `NAME\0CHECKSUM\0'.
   The trailing `\0' is also kept in the key.  */

#include <config.h>

#include <string.h>
#include <stdlib.h>
#include <fcntl.h>

#include "sync.h"


DB *
database_open (const char *filename)
{
  return dbopen (filename, O_CREAT | O_RDWR, 0700, DB_HASH, NULL);
}

void
database_fill_list (struct database *db, GtkCList *list)
{
  DBT key, value;
  int flag;
  DB *database = db->database;

  gtk_clist_freeze (list);
  gtk_clist_clear (list);

  flag = R_FIRST;
  while (1)
    {
      gchar *text[3];
      int ret = database->seq (database, &key, &value, flag);
      flag = R_NEXT;

      if (ret == -1)
	{
	  db_posix_error (_("Error while reading database file"),
			  db->filename);
	  break;
	}
      else if (ret)
	break;

      text[0] = key.data;
      text[1] = value.data;
      text[2] = "";
      gtk_clist_append (list, text);
    }

  gtk_clist_thaw (list);
}

struct entry *
lookup (const DB *database, const char *local)
{
  DBT key, value;
  struct entry *info;
  char *buf;

  key.data = (void *) local;
  key.size = strlen (local) + 1;
  if (database->get (database, &key, &value, 0))
    return NULL;

  info = (struct entry *) malloc (sizeof (struct entry));
  buf = value.data;
  info->name = strdup (buf);
  info->checksum = strdup (&buf[strlen (info->name) + 1]);

  return info;
}

void
free_entry (struct entry *entry)
{
  free (entry->name);
  free (entry->checksum);
  free (entry);
}

int
update_entry (const DB *database, const char *local_name,
	      const char *remote_name, const char *checksum)
{
  char *buf;
  int rlen = strlen (remote_name), ret, total;
  DBT key, value;

  key.data = (void *) local_name;
  key.size = strlen (local_name) + 1;

  total = rlen + strlen (checksum) + 2;
  buf = malloc (total);
  strcpy (buf, remote_name);
  strcpy (&buf[rlen + 1], checksum);
  value.data = buf;
  value.size = total;

  ret = database->put (database, &key, &value, 0);

  free (buf);

  return ret;
}

int
remove_entry (const DB *database, const char *name)
{
  DBT key;

  key.data = (void *) name;
  key.size = strlen (name) + 1;

  return database->del (database, &key, 0);
}
