/*
 *  LinKT - the Linux Kde pr-Terminal
 *  Copyright (C) 1997-1999 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "bin.h"
#include "channel.h"
#include "main.h"
#include "toolbox.h"
#include "global.h"
#include "dostime.h"
#include "output.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <kwmmapp.h>

extern TopLevel *toplevel;


/************************************************************************/
/**                      7plus-Autosave                                **/
/************************************************************************/

auto7plus::auto7plus( QWidget *rxchan )
{
   chan = rxchan;
   rxfile = false;              // Noch wird kein File empfangen
   win = NULL;
   transInfo = NULL;
   textfile = false;
}


auto7plus::~auto7plus()
{
   if (win != NULL)
   {
      delete win;
      win = NULL;
   }
   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
   }
   toplevel->chanListTable->setMode( chan, MODE_INFO );
   ((Channel *)chan)->updateStatusBar("");
}


void auto7plus::proceed( const char *data, int len )
{
   char *str;

   // Lokale Kopie der Daten zum Bearbeiten anlegen
   str = (char *) malloc(len+1);
   memcpy(str,data,len);
   str[len] = '\0';

   // Wenn noch ein CR dahinter ist, wird es geloescht
   if (str[len-1] == '\r')
   {
      str[len-1] = '\0';
      len--;
   }

   // Jede 7plus-Zeile ist laenger als 10 Zeichen
   if (len < 10 && !textfile)
   {
      free(str);
      return;
   }

   if (rxfile)
      saveLine(str, len);       // Diese Zeile speichern und gucken, ob das
                                // File zuende ist.
   else
      lookForLine(str, len);    // Faengt ein 7plus-File an? Wenn ja: Infos
                                // speichern etc

   free(str);
}


bool auto7plus::isRxfile()
{
   return rxfile;
}


void auto7plus::saveLine( char *str, int len )
{
   char tmp[500], *timeptr;
   time_t timediff;


   // String abspeichern
   write(fd,str,len);
   write(fd,"\n",1);
   bytes_got += len+1;

   if (!extraTransferWin)
      transInfo->setReceivedBytes( bytes_got );
   else
      win->setReceivedBytes( bytes_got );
   if (textfile)
   {
      filelen += len;
      if (!extraTransferWin)
         transInfo->setFilesizeBytes( filelen );
      else
         win->setFilesizeBytes( filelen );
   }

   if (textfile)
   {
      if (strncmp(str," stop_text.", 11))
         return;
   }
   else
   {
      COPY(tmp,str,0,10);
      if (strcmp(tmp," stop_7+. "))
         return;
   }

   // Das wars, das 7plus-File ist zu Ende
   ::close(fd);
    
   // Informationen an die Gegenstation verschicken */
   timediff = time(NULL)-starttime;

   if (timediff == 0) timediff = 1;
   timeptr = (char *)spec_time(timediff);
   if (textfile)
      sprintf(tmp,"\r<LinKT>: 7plus-Textsave: '%s' saved.\r"
                  "         %i bytes, %s, %li baud.\r\r"
                 ,shortname, bytes_got
                 ,timeptr,(bytes_got*8)/timediff);
   else
      sprintf(tmp,"\r<LinKT>: 7plus-Autosave: '%s' saved.\r"
                  "         %i/%i bytes, %s, %li baud.\r\r"
                 ,shortname, bytes_got, filelen
                 ,timeptr,(bytes_got*8)/timediff);
   free(timeptr);

   if (win != NULL)
   {
      delete win;
      win = NULL;
   }
   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
   }

   if ((((Channel *)chan)->userinfo->getType() & TYPE_TERMINAL) == TYPE_TERMINAL)
      ((Channel *)chan)->sendString( tmp );
   else
      ((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );


   rxfile = false;
   textfile = false;
   toplevel->chanListTable->setMode( chan, MODE_INFO );
   ((Channel *)chan)->updateStatusBar("");
}


void auto7plus::lookForLine( char *str, int len )
{
   char tmp[500];
   int binbytes, of, part, i, blocklines;

   // Ist dies ein 7plus-Textfile?
   if (!strncmp(str, " go_text. ", 10))
   {
      textfile = true;

      // Ein 7plus-Textfile. Dateiname?
      memcpy(tmp, str+10, strlen(str)-10);
      tmp[strlen(str)-10] = '\0';
      Klein(tmp);

      strcpy(name, tmp);
      sprintf(tmp,"%s/%s", config->dir7plus, name);
      strcpy(shortname, name);


      // Dateien nicht ueberschreiben!?!
      if (!config->overwriteExisting)
      {
         i = 0;
         while (file_exist(tmp))
         {
            sprintf(tmp,"%s/%s--%i",config->dir7plus, name, i);
            i++;
         }
      }

      if ((fd = open(tmp,O_WRONLY|O_CREAT)) == -1)
         return;

      // Zugriffsrechte einstellen
      fchmod(fd,S_IRUSR|S_IWUSR);

      strcpy(name, tmp);

      // Das ganze scheint ja wohl ein 7plus-File zu sein...
      bytes_got = len+1;
      starttime = time(NULL);

      // Den String schreiben
      write(fd,str,len);
      write(fd,"\n",1);

      // Merken: Das hier ist jetzt ein 7plus-File
      rxfile = true;
      toplevel->chanListTable->setMode( chan, MODE_7AUTOSAVE );
      ((Channel *)chan)->updateStatusBar(klocale->translate("7plus-Autosave"));

      // Entsprechendes Fenster oeffnen
      extraTransferWin = config->extraTransferWin;
      if (!extraTransferWin)
      {
         transInfo = new TransferInfo( ((Channel *)chan)->pannerHome, TRANSART_SIEBEN, shortname, bytes_got );
         ((Channel *)chan)->withTransInfo( transInfo );
         transInfo->setReceivedBytes( len+1 );
      }
      else
      {
         win = new TransferWin( this, TRANSART_SIEBEN, shortname, bytes_got );
         win->show();
         win->setReceivedBytes( len+1 );
         if (toplevel->currentChannel != NULL)
            if (toplevel->currentChannel->isActiveWindow())
            {
               KWM::activate(toplevel->currentChannel->winId());
               toplevel->currentChannel->setFocus();
            }
      }

      return;
   }


   if (strncmp(str," go_7+. ",8))
      return;

   // Koennte ein 7plus-File sein

   // Steht " of " an der richtigen Stelle?
   if (strncmp(str+11," of ",4))
      return;   // Nein

   /* Teile-Nummern extrahieren */
   COPY(tmp,str,8,3);
   part = atoi(tmp);
   COPY(tmp,str,16,3);
   of = atoi(tmp);
   // Dateinamen
   COPY(tmp,str,19,12);
   if ((i = POS('.',tmp)) != -1) tmp[i] = '\0';
   KillSpacesLeft(tmp);
   KillSpacesRight(tmp);
   Klein(tmp);
   strcpy(name,tmp);
   // Laenge des dekodierten Files
   COPY(tmp,str,32,7);
   binbytes = atoi(tmp);
   // Anzahl Zeilen in einem Block
   COPY(tmp,str,45,3);
   blocklines = get_hex (tmp);

   // Laenge errechen. Hochgradig abhaengig von der Teil-Nummer
   if (part == 1)
      if (str[65] == '*')
         filelen = (blocklines+3)*70;
      else
         filelen = (blocklines+2)*70;
   else if (part == of)
           filelen = (int) ((((binbytes + 61) / 62) % blocklines)+2)*70;
        else
           filelen = (blocklines+2)*70;

   if ((of == 0) || (part == 0) || (part > of))
      return;

   if (of == 1)
   {
      sprintf(tmp,"%s/%s.7pl",config->dir7plus, name);
      sprintf(shortname, "%s.7pl", name);
   }
   else
   {
      sprintf(tmp,"%s/%s.p%02x",config->dir7plus, name, part);
      sprintf(shortname,"%s.p%02x",name,part);
   }

   // Dateien nicht ueberschreiben!?!
   if (!config->overwriteExisting)
   {
      i = 0;
      while (file_exist(tmp))
      {
         if (of == 1)
         {
            sprintf(tmp,"%s/%s.7pl--%i",config->dir7plus, name, i);
            sprintf(shortname,"%s.7pl--%i",name, i);
         }
         else
         {
            sprintf(tmp,"%s/%s.p%02x--%i",config->dir7plus, name, part, i);
            sprintf(shortname,"%s.p%02x--%i",name, part, i);
         }
         i++;
      }
   }

   if ((fd = open(tmp,O_WRONLY|O_CREAT)) == -1)
      return;

   // Zugriffsrechte einstellen
   fchmod(fd,S_IRUSR|S_IWUSR);

   strcpy(name, tmp);

   // Das ganze scheint ja wohl ein 7plus-File zu sein...
   bytes_got = len+1;
   starttime = time(NULL);

   // Den String schreiben
   write(fd,str,len);
   write(fd,"\n",1);

   // Merken: Das hier ist jetzt ein 7plus-File
   rxfile = true;
   toplevel->chanListTable->setMode( chan, MODE_7AUTOSAVE );
   ((Channel *)chan)->updateStatusBar(klocale->translate("7plus-Autosave"));

   // Entsprechendes Fenster oeffnen
   extraTransferWin = config->extraTransferWin;
   if (!extraTransferWin)
   {
      transInfo = new TransferInfo( ((Channel *)chan)->pannerHome, TRANSART_SIEBEN, shortname, filelen );
      ((Channel *)chan)->withTransInfo( transInfo );
      transInfo->setReceivedBytes( len+1 );
   }
   else
   {
      win = new TransferWin( this, TRANSART_SIEBEN, shortname, filelen );
      win->show();
      win->setReceivedBytes( len+1 );
      if (toplevel->currentChannel != NULL)
         if (toplevel->currentChannel->isActiveWindow())
         {
            KWM::activate(toplevel->currentChannel->winId());
            toplevel->currentChannel->setFocus();
         }
   }
}


/*
 *  void auto7plus::abortRX()
 * Auf die Abort-Taste wurde gedrueckt. Transfer abbrechen.
 */
void auto7plus::abortRX()
{
   char tmp[500];


   // Das wars, das 7plus-File ist zu Ende
   ::close(fd);

   // Informationen an die Gegenstation verschicken */
   strcpy(tmp,"\r<LinKT>: 7plus-Autosave aborted.\r\r");

   if ((((Channel *)chan)->userinfo->getType() & TYPE_TERMINAL) == TYPE_TERMINAL)
      ((Channel *)chan)->sendString( tmp );
   else
      ((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );

   rxfile = false;

   if (win != NULL)
   {
      delete win;
      win = NULL;
   }
   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
   }
   toplevel->chanListTable->setMode( chan, MODE_INFO );
   ((Channel *)chan)->updateStatusBar("");
}


void auto7plus::closeTransferwin()
{
   if (win != NULL)
   {
      delete win;
      win = NULL;
      toplevel->chanListTable->setMode( chan, MODE_INFO );
      ((Channel *)chan)->updateStatusBar("");
   }
   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
      toplevel->chanListTable->setMode( chan, MODE_INFO );
      ((Channel *)chan)->updateStatusBar("");
   }
}



///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////


BinRX::BinRX( QWidget *rxchan )
{
   chan = rxchan;
   rxfile = false;
   win = NULL;
   transInfo = NULL;
   extraTransferWinSet = false;

   filename = NULL;
   shortname = NULL;
}


BinRX::~BinRX()
{
   if (filename != NULL) free(filename);
   if (shortname != NULL) free(shortname);

   if (win != NULL)
   {
      delete win;
      win = NULL;
   }

   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
   }

   extraTransferWinSet = false;
   toplevel->chanListTable->setMode( chan, MODE_INFO );
   ((Channel *)chan)->updateStatusBar("");
}


void BinRX::deleteDataFile()
{
   unlink(filename);
}


bool BinRX::isRxfile()
{
   return rxfile;
}


/*
 *  void BinRX::proceed( char *data, int len )
 * Verarbeitet Zeile fuer Zeile.
 *
 * Rueckgabe:    0, wenn das Frame komplett abgespeichert wurde,
 *            != 0, naemlich die Anzahl Bytes, die abgespeichert wurden,
 *                  wenn in dem Frame noch etwas anderes drin war.
 */
int BinRX::proceed( const char *data, int len )
{
   char *line, str[100];
   int n;


   line = (char *) malloc(len+1);
   memcpy(line, data, len);

   if (!strncmp(line, "SP\\-", 4))
   {
      memmove(line, line+4, len-4);
      line[len-4] = '\0';
      ((Channel *)chan)->outText( line, strlen(line), OUTCOLOR_RXTEXT );
      free(line);
      return 0;
   }

   // #ABORT#\r empfangen?
   if (!strcmp(line, "#ABORT#\r"))
   {
      ::close(fd);
      fd = -1;

      sprintf(str,"<LinKT>: AutoBIN-RX aborted by peer.\r");

      if ((((Channel *)chan)->userinfo->getType() & TYPE_TERMINAL) == TYPE_TERMINAL)
          ((Channel *)chan)->sendString( str );
      else
          ((Channel *)chan)->outText( str, strlen(str), OUTCOLOR_TXTEXT );

      if (filename != NULL)
      {
         free(filename);
         filename = NULL;
      }
      if (shortname != NULL)
      {
         free(shortname);
         shortname = NULL;
      }

      rxfile = false;
      if (win != NULL)
      {
         delete win;
         win = NULL;
      }
      if (transInfo != NULL)
      {
         delete transInfo;
         transInfo = NULL;
         ((Channel *)chan)->withoutTransInfo();
      }
      extraTransferWinSet = false;
      toplevel->chanListTable->setMode( chan, MODE_INFO );
      ((Channel *)chan)->updateStatusBar("");

      // Linemode wieder einschalten
      ((Channel *)chan)->flags |= CH_LINEMODE;
      return 0;
   }

   if (rxfile)
   {
      n = saveFrame(line, len);
      free(line);
      return n;
   }
   else
   {
      lookForLine(line, len);
      free(line);
      return 0;
   }
}


void BinRX::lookForLine( char *line, int len )
{
   char tmp[500],str[500];
   int i;


   if (strncmp(line,"#BIN#",5) != 0) return;

   if (line[len-1] == '\xD')
   {
      line[len-1] = '\0';
      len--;
   }

   memcpy(tmp,line+5,len-5);
   tmp[len-5] = '\0';

   size = -1;
   filename = NULL;
   crc = 0;
   crc_rec = false;
   rxtime = 0;

   while (tmp[0] != '\0')
   {
      if ((i = POS('#',tmp)) == -1)
      {
         strcpy(str,tmp);
         tmp[0] = '\0';
      }
      else
      {
         COPY(str,tmp,0,i);
         COPY(tmp,tmp,i+1,strlen(tmp)-i-1);
      }

      switch (str[0])
      {
         case '|': // CRC
                   crc = atoi(str+1);
                   crc_rec = true;
                   break;
         case '$': // DOS-Time
                   rxtime = get_hex (str+1);
                   break;
        default: if (size == -1)
                    size = atoi(str);
                 else
                    filename = (char *) strdup(str);
      }
   }

   if (size == -1) return;

   if (filename == NULL)
   {
      i = 0;
      do
      {
         sprintf(tmp,"%s/%lX.%.3i",config->dirABin, time(NULL), i);
         i++;
      }
      while (file_exist(tmp));
      filename = (char *) strdup(tmp);
   }
   else
   {
      // Eventuell vorhandenen Pfad (mit '/' oder '\') enfernen
      if ((i = lPOS('/',filename)) != -1)
         COPY(filename,filename,i+1,strlen(filename)-i-1);
      if ((i = lPOS('\\',filename)) != -1)
         COPY(filename,filename,i+1,strlen(filename)-i-1);

      sprintf(tmp,"%s/%s",config->dirABin, filename);

      // Gucken, ob der Filename schon existiert. Wenn ja wird eine
      // Zahl angehaengt
      i = 0;
      while (file_exist(tmp))
      {
         sprintf(tmp,"%s/%s--%i",config->dirABin,filename,i);
         i++;
      }
      free(filename);
      filename = (char *) strdup(tmp);
   }

   if ((fd = open(filename,O_WRONLY|O_CREAT|O_TRUNC)) == -1)
   {
      ((Channel *)chan)->sendString( "#NO#\xD" );
         return;
   }

   fchmod(fd,S_IRUSR|S_IWUSR);

   ((Channel *)chan)->sendString( "#OK#\xD" );

   our_crc = 0;
   starttime = time(NULL);
   size_written = 0;

   toplevel->chanListTable->setMode( chan, MODE_AUTOBINRX );

   // Linemode ausschalten
   ((Channel *)chan)->flags &= ~CH_LINEMODE;

   // Shortname
   if ((i = lPOS('/',filename)) == -1)
      shortname = (char *) strdup(filename);
   else
   {
      shortname = (char *) malloc(strlen(filename));
      COPY(shortname, filename,i+1,strlen(filename)-i-1);
   }

   rxfile = true;

   ((Channel *)chan)->updateStatusBar(klocale->translate("AutoBIN-save"));
}


int BinRX::saveFrame(char *line, int len)
{
   int i,timediff;
   char str[500],*tmp;

   i = size - size_written;

   // i Bytes schreiben.
   if (len < i) i = len;
   write(fd,line,i);
   our_crc = calc_crcthp_str(line,i,our_crc);


   if (!extraTransferWinSet)
   {
      extraTransferWin = config->extraTransferWin;
      extraTransferWinSet = true;
   }

   if (!extraTransferWin)
   {
      if (transInfo == NULL)
      {
         transInfo = new TransferInfo( ((Channel *)chan)->pannerHome, TRANSART_ABINRX, shortname, size );
         ((Channel *)chan)->withTransInfo( transInfo );
      }
   }
   else
   {
      if (win == NULL && transInfo == NULL)
      {
         // Entsprechendes Fenster oeffnen
         win = new TransferWin( this, TRANSART_ABINRX, shortname, size );
         win->show();
         if (toplevel->currentChannel != NULL)
            if (toplevel->currentChannel->isActiveWindow())
            {
               KWM::activate(toplevel->currentChannel->winId());
               toplevel->currentChannel->setFocus();
            }
      }
   }



   size_written += i;
   if (!extraTransferWin)
      transInfo->setReceivedBytes( size_written );
   else
      win->setReceivedBytes( size_written );

   // Wurden schon alle Daten empfangen?
   if (size_written == size)
   {
      ::close(fd);
      fd = -1;

      timediff = time(NULL)-starttime;
      if(timediff == 0) timediff = 1;
      tmp = spec_time(timediff);

      if (!crc_rec)
         sprintf(str,"<LinKT>: AutoBIN-RX OK. No CRC-check (time: %s, %li baud)\xD",
                      tmp,(size*8)/timediff);
      else
         if (our_crc == crc)
            sprintf(str,"<LinKT>: AutoBIN-RX OK. CRC: %i (time: %s, %li baud)\xD",
                        crc,tmp,(size*8)/timediff);
         else
            sprintf(str,"<LinKT>: AutoBIN-RX failed (CRC-ERROR %i != %i)\xD",
                        crc,our_crc);
      free(tmp);

      if ((((Channel *)chan)->userinfo->getType() & TYPE_TERMINAL) == TYPE_TERMINAL)
          ((Channel *)chan)->sendString( str );
      else
          ((Channel *)chan)->outText( str, strlen(str), OUTCOLOR_TXTEXT );

      if (filename != NULL)
      {
         free(filename);
         filename = NULL;
      }
      if (shortname != NULL)
      {
         free(shortname);
         shortname = NULL;
      }

      rxfile = false;
      if (win != NULL)
      {
         delete win;
         win = NULL;
      }
      if (transInfo != NULL)
      {
         delete transInfo;
         transInfo = NULL;
         ((Channel *)chan)->withoutTransInfo();
      }
      extraTransferWinSet = false;
      toplevel->chanListTable->setMode( chan, MODE_INFO );
      ((Channel *)chan)->updateStatusBar("");

      // Linemode wieder einschalten
      ((Channel *)chan)->flags |= CH_LINEMODE;
   }

   return i;
}


void BinRX::abortRX()
{
   char tmp[500];


   // Das wars, das File ist zu Ende
   ::close(fd);

   // Informationen an die Gegenstation verschicken */
   strcpy(tmp,"\r<LinKT>: AutoBIN-save aborted.\r\r");

   if ((((Channel *)chan)->userinfo->getType() & TYPE_TERMINAL) == TYPE_TERMINAL)
      ((Channel *)chan)->sendString( tmp );
   else
      ((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );

   rxfile = false;
   toplevel->chanListTable->setMode( chan, MODE_INFO );
   ((Channel *)chan)->updateStatusBar("");

   if (win != NULL)
   {
      delete win;
      win = NULL;
   }
   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
   }
   extraTransferWinSet = false;

   if (filename != NULL)
   {
      free(filename);
      filename = NULL;
   }
   if (shortname != NULL)
   {
      free(shortname);
      shortname = NULL;
   }

   // Linemode wieder einschalten
   ((Channel *)chan)->flags |= CH_LINEMODE;
}


void BinRX::closeTransferwin()
{
   if (win != NULL)
   {
      delete win;
      win = NULL;
      toplevel->chanListTable->setMode( chan, MODE_INFO );
      ((Channel *)chan)->updateStatusBar("");
   }

   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
      toplevel->chanListTable->setMode( chan, MODE_INFO );
      ((Channel *)chan)->updateStatusBar("");
   }
   extraTransferWinSet = false;
}



///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////


BinTX::BinTX( QWidget *txchan, char *fname )
{
   char tmp[500];
   int i,fd;
   struct stat stat_data;
   unsigned short time,date;


   chan = txchan;
   wascomp = 0;
   longname = (char *) strdup(fname);
   crc = calc_crctph_wholefile(longname);
   size = filesize(longname);

   if ((fd = open(longname,O_RDONLY)) == -1)
   {
      bin_state = 1;
      return;
   }
   ::close(fd);

   i = lPOS('/',longname);
   if (i == -1)
      strcpy(tmp,longname);
   else
   {
      memcpy(tmp,longname+i+1,strlen(longname)-i-1);
      tmp[strlen(longname)-i-1] = '\0';
   }
   shortname = (char *) strdup(tmp);

   // File-Datum
   stat(longname, &stat_data);
   date_unix2dos(stat_data.st_mtime,&time,&date);
   fulldostime = (date << 16) + time;

/*   // Wenn online-kompression noch an ist, wird sie abgeschaltet.
   if ((((Channel *)chan)->flags & CH_HUF_RX) ||
       (((Channel *)chan)->flags & CH_HUF_TX))
	{
		((Channel *)chan)->sendStringComp(9, "#HUF:NO#\r", true, COMP_HUF);
		((Channel *)chan)->flags &= ~(CH_HUF_TX|CH_HUF_RX);
		wascomp = 2;
   }

   if (((Channel *)chan)->flags & CH_COMP)
	{
	   ((Channel *)chan)->sendString( "//COMP OFF\r" );
		wascomp = 1;
   }*/

   // Normaler #BIN# - Header
   sprintf(tmp,"#BIN#%li#|%i#$%lX#%s\xD",size,crc,fulldostime,shortname);
   ((Channel *)chan)->sendString( tmp );
//   ((Channel *)chan)->sendStringComp( strlen(tmp), tmp, 1, COMP_NO);

   // Status: Warten auf #OK# / #NO# / #ABORT#
   bin_state = 10;

   win = NULL;
   transInfo = NULL;
   size_tx = 0;
}


BinTX::~BinTX()
{
	// Kompression wieder einschalten
/*	switch (wascomp)
   {
      case 1:
			((Channel *)chan)->sendString( "//COMP ON\r" );
      	break;
      case 2:
			((Channel *)chan)->sendString( "#HUF:ON#\r" );
			((Channel *)chan)->flags |= CH_HUF_TX;
      	break;
   }*/

   if (longname != NULL) free(longname);
   if (shortname != NULL) free(shortname);

   if (win != NULL)
   {
      delete win;
      win = NULL;
      toplevel->chanListTable->setMode( chan, MODE_INFO );
      ((Channel *)chan)->updateStatusBar("");
   }
   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
      toplevel->chanListTable->setMode( chan, MODE_INFO );
      ((Channel *)chan)->updateStatusBar("");
   }
}


//   int BinTX::check_okno( const char *data, int len )
//
// Kontrolliert die empfangene Zeile, ob eine Antwort auf die #BIN#-Zeile
// empfangen wurde.
//
// Rueckgabe: 0, wenn nichts erkannt wurde
//            1, wenn eine BIN-Uebertragung gestartet werden soll
//            2, wenn die Ubertragung abgebrochen wurde
int BinTX::check_okno( const char *data, int len )
{
   if (len < 4) return 0;

   if (!strncmp(data,"#OK#",4))
      return 1;

   if (!strncmp(data,"#NO#",4))
      return 2;

   if (!strncmp(data,"#ABORT#",7))
      return 2;

   return 0;
}


void BinTX::createTransferWin()
{
   extraTransferWin = config->extraTransferWin;

   if (!extraTransferWin)
   {
      transInfo = new TransferInfo( ((Channel *)chan)->pannerHome, TRANSART_ABINTX, shortname, size );
      ((Channel *)chan)->withTransInfo( transInfo );
   }
   else
   {
      win = new TransferWin( chan, TRANSART_ABINTX, shortname, size );
      win->show();
      if (toplevel->currentChannel != NULL)
         if (toplevel->currentChannel->isActiveWindow())
         {
            KWM::activate(toplevel->currentChannel->winId());
            toplevel->currentChannel->setFocus();
         }
   }
   starttime = time(NULL);

   toplevel->chanListTable->setMode( chan, MODE_AUTOBINTX );
   ((Channel *)chan)->updateStatusBar(klocale->translate("AutoBIN-Transmission"));
}


void BinTX::updateTransferWin( int received_bytes )
{
   size_tx += received_bytes;
   if (!extraTransferWin)
      transInfo->setReceivedBytes( size_tx );
   else
      win->setReceivedBytes( size_tx );
}


void BinTX::showEndText()
{
   time_t timediff;
   char *timeptr;
   char tmp2[100];

   timediff = time(NULL) - starttime;
   if (timediff == 0) timediff = 1;
   timeptr = (char *)spec_time(timediff);

   sprintf(tmp2,"<LinKT>: AutoBIN-TX OK. CRC: %i (time: %s, %li baud)\xD",
                crc,timeptr,(size*8)/timediff);

   if ((((Channel *)chan)->userinfo->getType() & TYPE_TERMINAL) == TYPE_TERMINAL)
      ((Channel *)chan)->sendString( tmp2 );
   else
      ((Channel *)chan)->outText( tmp2, strlen(tmp2), OUTCOLOR_TXTEXT );

   free(timeptr);
}


char * BinTX::getFilename()
{
   return longname;
}


void BinTX::closeTransferwin()
{
   if (win != NULL)
   {
      delete win;
      win = NULL;
   }
   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
   }
}


