#ifndef __GNOME_FONT_H__
#define __GNOME_FONT_H__

#include <gtk/gtkobject.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* GnomeFont - A gnome font object.
 *
 * The following arguments are available:
 *
 * name		       type		read/write	description
 * --------------------------------------------------------------------------------
 * ascender            double           R               maximum ascender for the font
 * descender           double           R               maximum descender for the font
 * underline_position  double           R               underline position for the font
 * underline_thickness double           R               underline thickness for the font
 */

#define GNOME_TYPE_FONT		 (gnome_font_get_type ())
#define GNOME_FONT(obj)		 (GTK_CHECK_CAST ((obj), GNOME_TYPE_FONT, GnomeFont))
#define GNOME_FONT_CLASS(klass)	 (GTK_CHECK_CLASS_CAST ((klass), GNOME_TYPE_FONT, GnomeFontClass))
#define GNOME_IS_FONT(obj)	 (GTK_CHECK_TYPE ((obj), GNOME_TYPE_FONT))
#define GNOME_IS_FONT_CLASS(klass) (GTK_CHECK_CLASS_TYPE ((klass), GNOME_TYPE_FONT))


/* We are going to do a renaming of GnomeFontMap to GnomeFontUnsized,
   and export a bunch of operations on GnomeFontUnsized to the outside
   world. */

typedef struct _GnomeFont            GnomeFont;
typedef struct _GnomeFontClass       GnomeFontClass;
typedef struct _GnomeFontMap         GnomeFontUnsized;
typedef struct _GnomeFontMap         GnomeFontMap;
typedef struct _GnomeFontKernPair    GnomeFontKernPair;
typedef struct _GnomeFontLigList     GnomeFontLigList;

struct _GnomeFont
{
  GtkObject object;

  int num;
  double size;
  GnomeFontMap *fontmap_entry;
  double scale;
};

struct _GnomeFontClass
{
  GtkObjectClass parent_class;

  int n_fonts;
  GnomeFontMap *fontmap;
};

/* These numbers are very loosely adapted from the Univers numbering
   convention. I've had to insert some to accomodate all the
   distinctions in the Adobe font catalog. Also, I've offset the
   numbering so that the default is 0. */
typedef enum {
  GNOME_FONT_LIGHTEST = -3,
  GNOME_FONT_EXTRA_LIGHT = -3,
  GNOME_FONT_THIN = -2,
  GNOME_FONT_LIGHT = -1,
  GNOME_FONT_BOOK = 0, /* also known as "regular" or "roman" */
  /* gap here so that if book is missing, light wins over medium */
  GNOME_FONT_MEDIUM = 2,
  GNOME_FONT_SEMI = 3, /* also known as "demi" */
  GNOME_FONT_BOLD = 4,
  /* gap here so that if bold is missing, semi wins over heavy */
  GNOME_FONT_HEAVY = 6,
  GNOME_FONT_EXTRABOLD = 7,
  GNOME_FONT_BLACK = 8,
  GNOME_FONT_EXTRABLACK = 9, /* also known as "ultra" */
  GNOME_FONT_HEAVIEST = 9
} GnomeFontWeight;

#define GNOME_FONT_NUM_WEIGHTS (GNOME_FONT_HEAVIEST + 1 - GNOME_FONT_LIGHTEST)

struct _GnomeFontMap
{
  char *font_name;
  char *afm_fn;
  char *pfb_fn;
  char *fullname;
  char *familyname;
  char *weight;
  char *alias;

  /* Font metric info follows */
  GnomeFontWeight weight_code;
  gboolean italic;
  /* above is font metric info stored in the fontmap, below is font metric
     info parsed from the afm file. */
  int   ascender;
  int   descender;
  int   underline_position;
  int   underline_thickness;
  int  *widths;
  GnomeFontKernPair *kerns;
  int num_kerns;
  GnomeFontLigList **ligs; /* one liglist for each glyph */

  int first_cov_page;
  int num_cov_pages;
  int **cov_pages; /* each page is an array of 256 ints */
};

struct _GnomeFontKernPair
{
  int glyph1;
  int glyph2;
  int x_amt;
};

struct _GnomeFontLigList {
  GnomeFontLigList *next;
  int succ, lig;
};

GtkType gnome_font_get_type (void);

GnomeFont *gnome_font_new (const char *name, double size);

GnomeFont *gnome_font_new_closest (const char *family_name,
				   GnomeFontWeight weight,
				   gboolean italic,
				   double size);

GnomeFont *gnome_font_new_from_full_name (const char *string);

/* Find the closest weight matching the family name, weight, and italic
   specs. Return the unsized font. */
GnomeFontUnsized *
gnome_font_unsized_closest (const char *family_name,
			    GnomeFontWeight weight,
			    gboolean italic);

/* return a pointer to the (PostScript) name of the font */
const char *
gnome_font_unsized_get_name (const GnomeFontUnsized *font);

/* return a pointer to the (PostScript) name of the font */
const char *
gnome_font_unsized_get_glyph_name (const GnomeFontUnsized *font);

const char *
gnome_font_get_name (const GnomeFont *font);

const char *
gnome_font_get_glyph_name (const GnomeFont *font);

char *
gnome_font_unsized_get_pfa (const GnomeFontUnsized *font);

char *
gnome_font_get_pfa (const GnomeFont *font);

char *
gnome_font_get_full_name (const GnomeFont *font);

/* this routine is fast; go ahead and use it */
double
gnome_font_get_width (const GnomeFont *font, int ch);

/* Returns the glyph width in 0.001 unit */
int
gnome_font_unsized_get_width (const GnomeFontUnsized *font, int ch);

/* these routines are fast; go ahead and use them */
double
gnome_font_get_width_string (const GnomeFont *font, const char *s);
double
gnome_font_get_width_string_n (const GnomeFont *font, const char *s, int n);

/* these routines return the ascender, descender, and underline
   information as specified by postscript.  Descender is negated since
   most people expect a positive descender for descenders below the
   baseline */
double
gnome_font_get_ascender (const GnomeFont *font);
double
gnome_font_get_descender (const GnomeFont *font);
double
gnome_font_get_underline_position (const GnomeFont *font);
double
gnome_font_get_underline_thickness (const GnomeFont *font);

/* Return the amount of kerning for the two glyphs. */
double
gnome_font_kern (const GnomeFont *font, int glyph1, int glyph2);

/* Return the amount of kerning for the two glyphs. */
int
gnome_font_unsized_kern (const GnomeFontUnsized *font, int glyph1, int glyph2);

/* Get the glyph number corresponding to a given unicode, or -1 if it
   is not mapped. */
int
gnome_font_get_glyph (const GnomeFont *font, int unicode);

/* Get the glyph number corresponding to a given unicode, or -1 if it
   is not mapped. */
int
gnome_font_unsized_get_glyph (const GnomeFontUnsized *font, int unicode);

/* Return a list of fonts, as a g_list of strings */
GList *
gnome_font_list (GnomeFontClass *klass);

void
gnome_font_list_free (GList *fontlist);

/* Return a list of font families, as a g_list of newly allocated strings */
GList *
gnome_font_family_list (GnomeFontClass *klass);

void
gnome_font_family_list_free (GList *fontlist);


typedef struct
{
  GnomeFontUnsized *unsized_font; /* font before scaling */
  GnomeFont *gnome_font; /* scaled font */
  double scale; /* scaling factor requested by a view */

  char *x_font_name; /* x name that got us gdk_font */
  GdkFont *gdk_font; /* used for drawing */
} GnomeDisplayFont;

GnomeDisplayFont *gnome_font_get_display_font (GnomeFont *font);
const char       *gnome_font_weight_to_string (GnomeFontWeight gfw);
GnomeFontWeight   string_to_gnome_font_weight (const char *weight);
GnomeDisplayFont *gnome_get_display_font      (const char *family,
					       GnomeFontWeight weight,
					       gboolean italic,
					       double points,
					       double scale);
int               gnome_display_font_height   (GnomeDisplayFont *gdf);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __GNOME_FONT_H__ */
