/* loadspk.c
   configuration utility for the speakup screen review package
   written by: Andy Berdan.

    Copyright (C) 1998  Andy Berdan.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Kirk Reiser <kirk@braille.uwo.ca>
    261 Trott dr. London, Ontario, Canada. N6G 1B6

    Andy Berdan <ed@facade.dhs.org>
    3-337 Wharncliffe Rd. N  London, Ontario, Canada N6G 1E4
    */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <linux/kd.h>
#include <sys/ioctl.h>
#include <ctype.h>
#include <stdarg.h>

#include "console_fd.h"
#include "symbols.h"

#define VER "loadspk v" VERSION
/* VERSION passed in from makefile */

#define MAXLEN 162	/* 80 chars + \n + \0 */ 
#define VARIABLE_RESET "\xff"

#define eq(x,y) (strcmp((x),(y))==0)

/* all of the synths' details... */
static struct spk_variable spkup_vars[] = SPKUP_VARS;

static struct spk_variable synth_static[] = ALL_STATIC;
static struct spk_variable synth_xtend[] = ALL_XTEND;
static struct spk_variable synth_alias[] = ALL_ALIAS;
static int num_static[] = ALL_NUM_STATIC;
static int num_xtend[] = ALL_NUM_XTEND;
static int num_alias[] = ALL_NUM_ALIAS;

static int synth = 0xff;	/* which synth are we talkin' to? */ 
int verbose = 0;		/* should we be noisy? */

#define perr(x) printf("loadspk:%d: " x "\n",linenum)

void doerr(int errval, int linenum) {
	switch(errval) {
	case -ESYNTAX:
		perr("syntax error");
		break;
	case -ESPKRANGE:
		perr("value out of range");
		break;
	case -ENOVAR:
		perr("invalid keyword");
		break;
	case -EIOCTL:
		perr("couldn't access kernel");
		break;
	case -EALIAS:
		perr("invalid alias");
		break;
	default:
		perr("Unknown error");
		break;
	}
}

char *translate(char *string)
{
      char *here=string;
      size_t len=strlen(string);
      int num,test;
      int numlen;

      while (NULL!=(here=strchr(here,'\\')))
      {
            numlen=1;
            switch (here[1])
            {
            case '\\':
                  break;

            case 'r':
                  *here = '\r';
                  break;

            case 'n':
                  *here = '\n';
                  break;

            case 't':
                  *here = '\t';
                  break;

            case 'v':
                  *here = '\v';
                  break;

            case 'a':
                  *here = '\a';
                  break;

            case '0':
            case '1':
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
                  if (sscanf(&here[1],"%3o%n",&num,&numlen)) 
                  	*here = (char)num;
                  break;

            case 'x':
                  if ((test = sscanf(&here[2],"%2x%n",&num,&numlen)) != 0) {
                  	numlen++;		/* to account for the 'x' */
                  	*here = (char)num;
                  } 
                  break;
            default:
            	  *here = *(here + 1);
            }
            num = here - string + numlen;
            here++;
            memmove(here,here+numlen,len-num );
      }
      return string;
}

static int inline synth_offset(int *num_cnt) {
	int i=0,sum=0;
	for (; i<synth; ++i) sum += num_cnt[i];
	return sum;
}

struct spk_variable *rangeok(struct spk_variable *ident, char *parm) {

	if (!(ident->flags & (HARD_DIRECT | SOFT_DIRECT))) return ident;
	if (ident->flags & USE_RANGE) { 
		int tval=0;
		if ((strlen(parm) > 0) && (ident->flags & NUMERIC)) {
			/* ascii-represented number, like "25" == 25.  evil.  */
			sscanf(parm,"%d",&tval);
		} else
			tval = parm[0];

		noisy("Range checking [%d]\n",tval);

		if ((tval >= atoi((ident->valid))) && (tval <= atoi(strchr(ident->valid,',')+1))) {
			return ident;
		}
	} else { /* matching set */
		int check, i=0;

		if (((ident->valid[0] == '*') && (ident->valid[1] == -1))
			|| (ident->valid[0] == -1))
			/* global match */ return ident;
		while ((check = *(ident->valid+(i++))) != -1)
			if (check == parm[0]) return ident;
	}
	return NULL;
}

int send_command(int index, struct spk_variable *cmd, char *newparam) {
/*  old === int send_command(unsigned int cmd, char *data) { */
	int fd;
	unsigned char tmpstr[MAXLEN+2];
	int check;

	fd = getfd();

	tmpstr[0] = index;		/* synth variable indicator */
	newparam[MAXLEN] = '\0';	/* yay, paranoia */

	if (*newparam != '\0') {
		if (cmd->flags & (HARD_DIRECT | SOFT_DIRECT)) {
			/* direct command, like "pitch = 98" or "tone = 2"*/

			/* notify the kernel of hardware directs */
			if (cmd->flags & HARD_DIRECT)
				tmpstr[0] |= 0x20;

#ifdef OBSOLETE
			/* no extra stuff allowed! */
			check = 0;
			sscanf(newparam,"%d%n",&check,&count);
			if (count != strlen(newparam)) return -1;
#endif

			strcpy(&(tmpstr[1]),cmd->build);
			strcpy(strchr(&(tmpstr[1]),'_'),newparam);
			strcat(&(tmpstr[1]),strchr(cmd->build,'_')+1);
		} else 
			strcpy(&tmpstr[1],newparam);
	} else {
		/* reset to default */
		strcpy(&(tmpstr[1]),cmd->build);
		strcpy(strchr(&(tmpstr[1]),'_'),cmd->param);
		strcat(&(tmpstr[1]),strchr(cmd->build,'_')+1);
		noisy("Resetting to default\n");
		noisy("--->%d:(0x%x)\n",tmpstr[0],(unsigned char)tmpstr[1]);
	}

	check = strlen(tmpstr) - 1;
	/* chuck trailing whitespace */
	while ((tmpstr[check] == ' ') || (tmpstr[check] == '\t'))
		tmpstr[check--] = '\0';

		noisy("Translated and sent [%d:\\x%x%s]\n",(int)tmpstr[0],tmpstr[1],&tmpstr[2]);

	#ifndef NOSEND
	if (ioctl(fd,SPKLOADCFG,&tmpstr)) 	/* send to kernel */ 
		return -EIOCTL;
	#endif
	return 0;
}

struct spk_variable *ident_var(int *index, char *keyword) {

	int i=0, count = num_static[synth];
	struct spk_variable *tester = &(synth_static[ synth_offset(num_static) ]);

	/* search synth_static */
	while (--count > -1) {
		noisy("<static>%s\n",(tester+i)->id);
		if (eq((tester+i)->id,keyword)) {
			*index = i;
			return tester+i;
		}
		i++;
	}

	/* search synth_xtend */
	i = 0;
	count = num_xtend[synth];
	tester = &(synth_xtend[ synth_offset(num_xtend) ]);
	while (--count > -1) {
		noisy("<xtend>%s\n",(tester+i)->id);
		if (eq((tester+i)->id,keyword)) {
			*index = i+0x40;
			return tester+i;
		}
		i++;
	}

	/* search spkup_vars */
	i = 0;
	count = NUM_SPKUP_VARS;
	tester = spkup_vars;
	while (--count > -1) {
		noisy("<spkup_vars>%s\n",(tester+i)->id);
		if (eq((tester+i)->id,keyword)) {
			*index = i+0x80;
			return tester+i;
		}
		i++;
	}
	return NULL; /* invalid keyword */
}

struct spk_variable *ident_alias(char *keyword) {
	struct spk_variable *tester = &(synth_alias[ synth_offset(num_alias) ]);
	int i=0, count=num_alias[synth];

	if (keyword[0] == '\0') return NULL; /* for strings that start with
						'(' */

	while ((count--) > 0) {
		if (eq((tester+i)->id,keyword)) return (tester+i);
		++i;
	}
	return NULL; /* invalid keyword */
}

int process (char *keyword, char *value)
{
	struct spk_variable *cmd;
	int index;	/* ok, so it's a kludge to get the index of the
			   spk_variable to send to the kernel.  Bleah.  */

	if ((cmd = ident_var(&index, keyword)) == NULL) {
		noisy("ident_var failed\n");
		return -ENOVAR;
	}

	if (value[0]=='\0') {
		strcpy(value,cmd->param);
		noisy("Blank value converted to [%s]\n",value);
	}

	if ((cmd = rangeok(cmd,value)) == NULL) {
		noisy("rangeok failed\n");
		return -ESPKRANGE;
	}
	if (send_command(index,cmd,value)) {
		noisy("send_command failed\n");
		return -EIOCTL;
	}
	return 0;
}

int processval(char *input) {	/* converts aliases to hardware literals */
	/* return 0 on success */
	/* FIXME: at the moment, this uses the input buffer itself for
		workspace, so the aliases must convert to something
		smaller than they themselves are.  */

	char sub[80], fn[10], check[2] = "";
	int parm;
	char *value = input;
	int scanlen;
	struct spk_variable *ident;

	while (1) {
		sub[0] = '\0';
		scanlen = 0;
		sscanf(value,"%s %n",sub,&scanlen);
		value += scanlen;

		/* done? */
		if (sub[0] == '\0') {
			*input = '\0';
			break;
		}
		
		fn[0] = '\0'; 
		parm = 0; 

		sscanf(sub,"%[^(^ ](%d)%1s", fn, &parm, check);

		if ((ident = ident_alias(fn)) == NULL) {
			noisy("Alias search [%s] failed\n",fn);
			/* not known alias, check if error */
			if (parm != 0) return -1;

			noisy("... no error\n");

			/* no error -- rescan, and straight copy */
			sscanf(sub,"%s",fn);
			translate(fn);
			sprintf(input,"%s",fn);	/* add extra space to
							punctuate words */
			input += strlen(fn);
			*(input++) = ' ';	/* put the space back, 
						  overwriting the null*/
		} else { /* known alias */
			char strparm[6]; /* 5 digits is *probably* enough */
			char buildspace[20];
			/* check & set */
			noisy("Found alias [%s]\n",fn);

			sprintf(strparm,"%d",parm);
			if (rangeok(ident,strparm) != NULL) {
				strcpy(buildspace,ident->build);
				strcpy(strchr(buildspace,'_'),strparm);
				strcat(buildspace,strchr(ident->build,'_')+1);

				sprintf(input,"%s%n",buildspace,&scanlen);
				input += scanlen;
			} else return -1;
		} /* end known alias */
	} /* while (1) */

	scanlen = strlen(value)-1;
	while ((value[scanlen] == ' ') || (value[scanlen] == '\t'))
		value[scanlen--] = '\0';
	return 0;
}

int getfields (char *keyword, char *value, char *instr)
{
	char check[2];
	int result;

	value[0] = '\0';
	result = sscanf(instr,"%[a-zA-Z0-9_] %1[=] %[^\n]",
		keyword,	/* alphanumeric string */ 
		check,		
		value		/* any string; rest of line...  */
	); 

	noisy("Parsed [%s][%s][%s]\n",keyword,check,value);

	if ((result == 2) && (value[0] == '\0')) {
		return 0;		/* 'keyword =' blank out command */
	} else return (result - 3);		/* zero on success */
}

/* prints usage and bails with errcode */
void bail_out(int errcode) {
printf("\
Usage: loadspk [OPTIONS] [file]\n\
Loads configuration from [file] into speakup.  If [file] is not specified,\n\
input is read from stdin\n\
OPTIONS:\n\
\t-h	this help listing\n\
\t-v	verbose output\n");
exit(errcode);
}

int main(int argc, char *argv[])
{
	int linenum=1;
	char keyword[MAXLEN],value[MAXLEN*2];
	char instr[MAXLEN];
	unsigned long fd;
	char errval=0;
	int i;

	while ((i=getopt(argc,argv, "hv")) != EOF) {
		switch (i) {
		case 'h':	/* help */
			bail_out(0);
			break;
		case 'v':	/* (really) verbose */
			verbose=1;
			break;
		}
	}
	 /*  command line leftovers argv[optind++] .. argv[argc] */
	argc -= optind;
	argv += optind;

	if (argc == 1) {
		if (NULL == freopen(argv[0],"r",stdin)) {
			perror("Error loading file");
			exit(1);
		}
	} else if ( argc ) { /* more than one filename */
		/* bzzzt... no soup for you. */
		bail_out(1);
	}

	printf(VER "\n");
	fd = getfd();
	if (fd) {
		unsigned char tmp[2] = "\xff";
		ioctl(fd,SPKHARDWARE,tmp);
		if (tmp[0] == 0xff) {
			fprintf(stderr,"\nError: Synth not identified!\n");
			return -1;
		} else synth = tmp[0];
	} else {
		fprintf(stderr,"\nError: Couldn't get console file descriptor!\n");
		return -1;
	}

	fgets(instr,MAXLEN,stdin);
	while ((instr[strlen(instr)-1] != '\n') && !feof(stdin)) {
		printf("loadspk:%d: line too long.  Max %d characters\n",linenum,MAXLEN-2);
		fgets(instr,MAXLEN,stdin);
	}

	while (!feof(stdin)) {
		strncpy(value,"",MAXLEN*2);
		if ((*instr != '#') && (*instr != '\0')) {
		    if (!getfields(keyword,value,instr)) { 
			if (processval(value))
				doerr(-EALIAS,linenum);
			if (eq(keyword,DIRECT_CMD)) {
				if (ioctl(fd, SPKDIRECTSEND, translate(value)))
					doerr(-EIOCTL,linenum);
			} else {
				if ((errval = process(keyword,value)))
					doerr(errval,linenum);
			}
		    } else {
			doerr(-ESYNTAX,linenum);
		    }
		}

		linenum++;
		fgets(instr,MAXLEN,stdin);

		while ((instr[strlen(instr)-1] != '\n') && !feof(stdin)) {
			printf("loadspk:%d: line too long.  Max %d characters\n",linenum,MAXLEN-2);
			fgets(instr,MAXLEN,stdin);
		}
	}
	return 0;
}

