/*
 *  LinKT - the Linux Kde pr-Terminal
 *  Copyright (C) 1997-2001 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
//---------------------------------------------------------------------------
#include "gpg.h"
#include "gpg.moc"
#include "global.h"
#include "version.h"
#include "toolbox.h"
#include "output.h"
//---------------------------------------------------------------------------
#include <qaccel.h>
#include <qfiledialog.h>
#include <qlayout.h>
//---------------------------------------------------------------------------
#include <kmessagebox.h>
#include <kapp.h>
//---------------------------------------------------------------------------
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
//---------------------------------------------------------------------------
#define BTNHEIGHT 30
#define BTNWIDTH 80
//---------------------------------------------------------------------------
extern KApplication *mykapp;
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
DlgGetPW::DlgGetPW( QWidget *parent ) : QDialog( parent, "", true )
{
   setCaption( i18n("Passphrase") );

   setFixedSize( 220, 100 );

	l = new QLabel( this );
   l->setGeometry( 10, 10, 200, 20 );
   l->setText( i18n("Please enter your passphrase:") );

   pw = new QLineEdit( this );
   pw->setGeometry( 10, 35, 200, 20 );
   pw->setEchoMode( QLineEdit::Password );

  	ok = new QPushButton(i18n("&Ok"), this);
	ok->setGeometry((width()-(2*80))/3, height() - 30, 80, 25);
  	connect(ok, SIGNAL(clicked()), this, SLOT(accept()));
	ok->setDefault(TRUE);


	abort = new QPushButton(i18n("&Abort"), this);
	abort->setGeometry(((width()-(2*80))/3)*2+80, height() - 30, 80, 25);
	connect(abort, SIGNAL(clicked()), this, SLOT(reject()));

	a = new QAccel( this );
	a->connectItem( a->insertItem(Key_Return), this, SLOT(accept()));

   pw->setFocus();
}
//---------------------------------------------------------------------------
DlgGetPW::~DlgGetPW()
{
}
//---------------------------------------------------------------------------
QString DlgGetPW::getPw()
{
	return pw->text().stripWhiteSpace();
}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
GPGDialog::GPGDialog( QWidget *parent, Channel *chan ) : QDialog( parent, "", false ), SendData( chan )
{
	QLabel *l;
   QFontMetrics fm( conf->gpgFont() );
   int screenwidth = 84 * fm.width("m");
   QVBoxLayout *vboxMain = new QVBoxLayout( this );
   QHBoxLayout *hbox;


   infile = NULL;
   outfile = NULL;
   pw = NULL;


   resize( screenwidth, screenwidth*0.6 );

   setCaption( i18n("LinKT: GnuPG upload...") );

   vboxMain->setMargin( 5 );
   vboxMain->setSpacing( 5 );


   lReplyComment = new QLabel( this );
   lReplyComment->setFixedSize( 0, 0 );
   QFont f(lReplyComment->font().family(), lReplyComment->font().pointSize()+2);
   f.setBold( true );
   lReplyComment->setFont( f );
   vboxMain->addWidget( lReplyComment );


   hbox = new QHBoxLayout( vboxMain );

	cmbType = new QComboBox( this );
   cmbType->insertItem( "S" );
   cmbType->insertItem( "SB" );
   cmbType->insertItem( "SP" );
   cmbType->insertItem( "none" );
   cmbType->setCurrentItem( 0 );
   connect( cmbType, SIGNAL(activated(const QString &)), SLOT(slotTypeSelected(const QString &)) );

   hbox->addWidget( cmbType );

   edBoard = new QLineEdit( this );
   edBoard->resize( 80, 20 );

   hbox->addWidget( edBoard );

   l = new QLabel( this );
   l->setText( "@" );

   hbox->addWidget( l );

   edMbx = new QLineEdit( this );
   edMbx->resize( 80, 20 );

	hbox->addWidget( edMbx );

   l = new QLabel( this );
   l->setText( "#" );

   hbox->addWidget( l );

   edLt = new QLineEdit( this );
   edLt->resize( 40, 20 );

	hbox->addWidget( edLt );
   hbox->addStretch( 1 );


   hbox = new QHBoxLayout( vboxMain );

   lSubject = new QLabel( this );
   lSubject->setText( i18n("Subject:") );

   hbox->addWidget( lSubject );
   hbox->addSpacing( 5 );

   edSubject = new QLineEdit( this );

	hbox->addWidget( edSubject );

   medMessage = new QMultiLineEdit( this );
   medMessage->setGeometry( 10, 55, width()-20, height()-55-BTNHEIGHT-10 );
   medMessage->setWordWrap( QMultiLineEdit::FixedColumnWidth );
   medMessage->setWrapColumnOrWidth( 75 );
   medMessage->setFont( conf->gpgFont() );

   vboxMain->addWidget( medMessage, 1 );
   vboxMain->addSpacing( 5 );

   hbox = new QHBoxLayout( vboxMain );

   btnImport = new QPushButton( this );
   btnImport->setText( i18n("&Import") );
   connect( btnImport, SIGNAL(clicked()), SLOT(slotImportClicked()) );

   hbox->addWidget( btnImport );
   hbox->addStretch( 1 );

   btnSend = new QPushButton( this );
   btnSend->setText( i18n("&Send") );
   connect( btnSend, SIGNAL(clicked()), SLOT(slotSendClicked()) );

   hbox->addWidget( btnSend );
   hbox->addSpacing( 10 );

   btnCancel = new QPushButton( this );
   btnCancel->setText( i18n("&Cancel") );
   connect( btnCancel, SIGNAL(clicked()), SLOT(close()) );

   hbox->addWidget( btnCancel );


   addSignature();

   edBoard->setFocus();


	strReplyComment = "";
}
//---------------------------------------------------------------------------
GPGDialog::~GPGDialog()
{
	unsigned int i;


	if (infile != NULL)
   {
   	unlink( infile );
      free( infile );
   }

	if (outfile != NULL)
   {
   	unlink( outfile );
      free( outfile );
   }

   for (i=0; i<strlen(pw); i++)
   	pw[i] = 'X';
   free(pw);
}
//---------------------------------------------------------------------------
void GPGDialog::addSignature()
{
	const char *filename = (const char *)conf->getSig();
   FILE *f;
   char tmp[500];
   int i;


   if (conf->getFlag(CFG_GPGADDSIG) && file_exist(filename))
   {
		if ((f = fopen(filename, "r")) == NULL) return;

      medMessage->insertLine("");
      medMessage->insertLine("-- ");
      while (fgets(tmp, 499, f) != NULL)
      {
      	if ((i = POS('\n', tmp)) != -1) tmp[i] = '\0';
      	if ((i = POS('\r', tmp)) != -1) tmp[i] = '\0';

         medMessage->insertLine( tmp );
      }

      fclose( f );
   }
}
//---------------------------------------------------------------------------
void GPGDialog::resizeEvent(QResizeEvent *event)
{
	QDialog::resizeEvent( event );

   edSubject->setGeometry( 10+lSubject->width()+5, 30, width()-lSubject->width()-25, 20 );
   medMessage->setGeometry( 10, 55, width()-20, height()-55-BTNHEIGHT-10 );
   btnImport->setGeometry( 10, height()-BTNHEIGHT-5, BTNWIDTH, BTNHEIGHT );
   btnCancel->setGeometry( width()-1*(BTNWIDTH+5), height()-BTNHEIGHT-5, BTNWIDTH, BTNHEIGHT );
   btnSend->setGeometry( width()-2*(BTNWIDTH+5), height()-BTNHEIGHT-5, BTNWIDTH, BTNHEIGHT );
}
//---------------------------------------------------------------------------
void GPGDialog::slotSendClicked()
{
	char tmp[1000];
   int fd, i, count;
   QString text;


   if (strReplyComment.isEmpty() && cmbType->currentText() != "none")
   {
		edBoard->setText( edBoard->text().stripWhiteSpace() );
		if (edBoard->text().isEmpty())
	   {
			KMessageBox::error( this, i18n("You have to specify a boardname.") );
	   	return;
	   }

	   edMbx->setText( edMbx->text().stripWhiteSpace() );

	   edLt->setText( edLt->text().stripWhiteSpace() );

	   edSubject->setText( edSubject->text().stripWhiteSpace() );
	   if (edSubject->text().isEmpty())
	   {
			KMessageBox::error( this, i18n("You have to specify a subject.") );
			return;
	   }
	}

   // Passphrase abfragen
	if (!getPW())
   	return;

   // Dateinamen erzeugen
	sprintf( tmp, "/tmp/lkti%li.%i", time(NULL), getpid() );
   infile = (char *) strdup(tmp);
	sprintf( tmp, "/tmp/lkto%li.%i", time(NULL), getpid() );
   outfile = (char *) strdup(tmp);

   // Text ins Infile packen
   if ((fd = open(infile, O_WRONLY|O_CREAT|O_TRUNC)) == -1)
   {
   	free(infile);
      infile = NULL;
      free(outfile);
      outfile = NULL;
      sprintf(tmp, i18n("Cannot create GPG input file.\n%s"), strerror(errno) );
		KMessageBox::error( this, tmp );
   	return;
   }

   // Zugriffsrechte fuer das File erstellen
	fchmod( fd, S_IRUSR|S_IWUSR );

   for (i=0; i<medMessage->numLines(); i++)
   {
   	sprintf(tmp, "%s\n", (const char *)medMessage->textLine(i) );
      write( fd, tmp, strlen(tmp) );
   }

   ::close(fd);

   i = callGPG();

printf("i: %i\n", i);
	switch (i)
   {
   	case 0: break;
   	case 512:
			KMessageBox::error( this, i18n("You have entered the wrong passphrase.") );
         for (i=0; i<(int)strlen(pw); i++)
         	pw[i] = 'X';
         free(pw);
         pw = NULL;
         return;
   }


   // Send-Befehl erzeugen, wenn kein Reply oder Comment gewaehlt wurde
   if (strReplyComment.isEmpty() && cmbType->currentText() != "none")
   {
	   text = cmbType->currentText() + " " + edBoard->text();
	   if (!edMbx->text().isEmpty())
	   	text += " @ " + edMbx->text();
		if (!edLt->text().isEmpty())
	   	text += " #" + edLt->text();
		text += " " + edSubject->text();
	}
   else
		text = strReplyComment;

	if ((fd = open(outfile, O_RDONLY)) == -1)
   {
		KMessageBox::error( this, i18n("Cannot open GPG output file.") );
   	return;
   }

   if (!text.isEmpty())
   {
	   sprintf( tmp, "%s\r", (const char *)text );
	   sendString( tmp );
	}

   do
   {
   	count = read( fd, tmp, 999 );
      for (i=0; i<count; i++)
      	if (tmp[i] == '\n') tmp[i] = '\r';
		sendData( count, tmp );
   } while (count == 999);

   ::close(fd);

   if (!text.isEmpty())
	   sendString( "***END\r" );

   close();
}
//---------------------------------------------------------------------------
void GPGDialog::slotImportClicked()
{
   char tmp[1000];
   int i;
   FILE *f;


   QString str = QFileDialog::getOpenFileName( getenv("HOME"));

   // Cancel
   if (str.isEmpty()) return;

   strcpy(tmp, (const char *)str);

   if (!file_exist(tmp))
   {
		KMessageBox::error( this,
								i18n("Cannot find specified file."),
								i18n("File not found"));
      return;
   }


   if ((f = fopen(tmp, "r")) != NULL)
   {
      while (fgets(tmp, 4999, f) != NULL)
      {
         if ((i = POS('\r', tmp)) > -1)
            tmp[i] = '\0';
         if ((i = POS('\n', tmp)) > -1)
            tmp[i] = '\0';
         medMessage->insertLine( tmp );
      }
      fclose(f);
   }
   else
		KMessageBox::error( this,
                        i18n("Cannot open specified file."),
								i18n("Cannot open file"));
}
//---------------------------------------------------------------------------
// Fragt die Passphrase vom User ab
bool GPGDialog::getPW()
{
	bool ok=false;
   QString p;


   DlgGetPW *dlg = new DlgGetPW( this );

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   if (dlg->exec() == 1)
   {
   	p = dlg->getPw();
     	if (!p.isEmpty())
		{
			ok = true;
			pw = (char *) strdup((const char *)p);
		}
   }

   delete dlg;

   return ok;
}
//---------------------------------------------------------------------------
int GPGDialog::callGPG()
{
	int fds[2];
   int status;
	char pwfd[500], out[500], comment[500], user[500];


	if (pipe(fds) == -1)
	{
printf("error: %s\n", strerror(errno));
		return -1;
	}

	if (fork() == 0)
	{
		// Kindprozess. gpg aufrufen
		::close(fds[1]);
		sprintf(pwfd, "--passphrase-fd=%i", fds[0]);
		sprintf(out, "--output=%s", outfile);
		sprintf(comment, "--comment=LinKT V%s", LINKT_VERSION);
		if (conf->getGpgID()[0] == '\0')
		{
			if (execlp("gpg", "--no-greeting", "--batch", "--clearsign", pwfd, out, comment, infile, 0) == -1)
			_exit(25);
		}
		else
		{
			sprintf(user, "--local-user=%s", conf->getGpgID());
			if (execlp("gpg", "--no-greeting", "--batch", "--clearsign", user, pwfd, out, comment, infile, 0) == -1)
	      	_exit(25);
		}
	}
	else
	{
		// Elternprozess. Passwort ans Kind schicken und warten
		::close(fds[0]);
		write(fds[1], pw, strlen(pw));
		::close(fds[1]);

		wait(&status);
		return status;
	}

printf("fork-error\n");
	return -1;
}
//---------------------------------------------------------------------------
void GPGDialog::closeEvent( QCloseEvent * )
{
printf("close event\n");
   delete this;
}
//---------------------------------------------------------------------------
void GPGDialog::setData( const QString & board, const QString & mbx, const QString & subject )
{
   edBoard->setText( board );
   edMbx->setText( mbx );
   edSubject->setText( subject );
}
//---------------------------------------------------------------------------
void GPGDialog::slotTypeSelected( const QString & str )
{
	bool enabled = (str != "none");

	edBoard->setEnabled( enabled );
	edMbx->setEnabled( enabled );
	edLt->setEnabled( enabled );
	edSubject->setEnabled( enabled );
}
//---------------------------------------------------------------------------
void GPGDialog::setComment( const QString & sendString, int nr, const QString & board, const QString & sender )
{
	QString tmp;


   strReplyComment = sendString;

   tmp.sprintf( i18n("Comment to %s %i"), board.latin1(), nr );
	if (!sender.isEmpty())
   	tmp += " "+i18n("by")+" "+sender;
 	tmp += ":";

   lReplyComment->setText( tmp );
   lReplyComment->setFixedSize( lReplyComment->sizeHint() );

   // Aenderung des Types der Nachricht abschalten
   cmbType->clear();
   cmbType->insertItem( "none" );
	slotTypeSelected( "none" );
}
//---------------------------------------------------------------------------
void GPGDialog::setReply( const QString & sendString, int nr, const QString & board, const QString & sender )
{
	QString tmp;


   strReplyComment = sendString;

   tmp.sprintf( i18n("Reply to %s %i"), board.latin1(), nr );
	if (!sender.isEmpty())
   	tmp += " "+i18n("by")+" "+sender;
 	tmp += ":";

   lReplyComment->setText( tmp );
   lReplyComment->setFixedSize( lReplyComment->sizeHint() );

   // Aenderung des Types der Nachricht abschalten
   cmbType->clear();
   cmbType->insertItem( "none" );
	slotTypeSelected( "none" );
}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
CheckGPG::CheckGPG( Channel *chan) : SendData( chan )
{
	fd = -1;
	lines = 0;
	filename = NULL;
	save[0] = '\0';
	type = TYPE_UNKNOWN;
}
//---------------------------------------------------------------------------
CheckGPG::~CheckGPG()
{
	if (fd != -1)
   {
   	::close(fd);
      unlink(filename);
	}
   if (filename != NULL)
   	free(filename);
}
//---------------------------------------------------------------------------
void CheckGPG::proceed( const char *data, int len )
{
	char *tmp;


   // Wenn der komplette GnuPG-Support deaktiviert ist, wird garnicht erst
   // geguckt
   if (!conf->getFlag(CFG_GPGSUPPORT))
   	return;

   tmp = (char *) malloc( len );
   len--;
   memcpy( tmp, data, len );
   tmp[len] = '\0';

	if (!strcmp(tmp, "-----BEGIN PGP SIGNED MESSAGE-----"))
   {
   	if (fd != -1)
      {
         lines = 0;
         // Bisher empfangene Daten entfernen
         lseek( fd, 0, SEEK_SET );
         ftruncate( fd, 0 );
      }
      else
      {
         if (filename != NULL)
         	free(filename);
      	filename = (char *) strdup( getUniqueFilename() );
         fd = open( filename, O_WRONLY|O_CREAT|O_TRUNC );
			fchmod( fd, S_IRUSR|S_IWUSR );
      }
      type = TYPE_SIGNATURE;
   }


	if ((!strcmp(tmp, "-----BEGIN PGP PUBLIC KEY BLOCK-----")) && (conf->getFlag(CFG_GPGIMPORTKEYS)))
   {
   	if (fd != -1)
      {
         lines = 0;
         // Bisher empfangene Daten entfernen
         lseek( fd, 0, SEEK_SET );
         ftruncate( fd, 0 );
      }
      else
      {
         if (filename != NULL)
         	free(filename);
      	filename = (char *) strdup( getUniqueFilename() );
         fd = open( filename, O_WRONLY|O_CREAT|O_TRUNC );
			fchmod( fd, S_IRUSR|S_IWUSR );
      }
      type = TYPE_PUBKEY;
   }


   if (fd != -1)
   {
		write( fd, tmp, len );
		write( fd, "\n", 1 );
	}
   else
   {
   	free(tmp);
   	return;
	}

   if (!strcmp(tmp, "-----END PGP SIGNATURE-----") && (type == TYPE_SIGNATURE))
   {
   	::close(fd);
      fd = -1;
   	checkSignature();
      type = TYPE_UNKNOWN;
   }

   if (!strcmp(tmp, "-----END PGP PUBLIC KEY BLOCK-----") && (type == TYPE_PUBKEY))
   {
   	::close(fd);
      fd = -1;

      setReceiving( false );
      if (conf->getFlag(CFG_GPGASKIMPORTKEYS))
      {
	      if (KMessageBox::questionYesNo( (QWidget *)getChannel(),
	      				i18n("PGP-Key received. Do you want to import it?"),
	      				i18n("GnuPG-Import")) == KMessageBox::No)
			{
				setReceiving( true );
	         free(tmp);
	         return;
			}
		}

   	checkPubKey();
      type = TYPE_UNKNOWN;
		setReceiving( true );
   }

   free(tmp);
}
//---------------------------------------------------------------------------
void CheckGPG::abort()
{
}
//---------------------------------------------------------------------------
char * CheckGPG::getUniqueFilename()
{
	char fname[200];
   static char tmp[200];
   int i=0;


   sprintf( fname, "/tmp/%li.%i", time(NULL), getpid() );
   strcpy( tmp, fname );
   for (;;)
   {
      if (!file_exist(tmp))
      	return tmp;

		sprintf( tmp, "%s.%i", fname, i );
      i++;
	}
}
//---------------------------------------------------------------------------
bool CheckGPG::rxSignatureData( int fd )
{
	int count;
   char tmp[500], id[50], tmp2[500];
   int i, len;
   struct tm *t;


   count = read( fd, tmp, 499 );
   tmp[count] = '\0';

   if (count == 0)
   {
      unlink(filename);
      ::close(fd);

      // Info-Text fuer den User ausgeben
		switch (sig)
      {
      	case SIG_BAD:
		      sprintf( tmp, "\r<LinKT>: %s %s\r",
		      		(const char *)i18n("Bad signature from"),
	               username );
				statusText( tmp, OUTCOLOR_GPG );
				break;
      	case SIG_GOOD:
		      sprintf( tmp, "\r<LinKT>: %s %s\r",
		      		(const char *)i18n("Good signature from"),
	               username );
				statusText( tmp, OUTCOLOR_GPG );
				break;
 			case SIG_ERR_UNKNOWN:
		      sprintf( tmp, "\r<LinKT>: %s\r",
		      		(const char *)i18n("Cannot check signature -- unknown algorithm."));
				statusText( tmp, OUTCOLOR_GPG );
         	break;
 			case SIG_ERR_MISSING:
		      sprintf( tmp, "\r<LinKT>: %s%s\r",
		      		(const char *)i18n("Cannot check signature -- missing key ID 0x"),
                  keyid+8 );
				statusText( tmp, OUTCOLOR_GPG );
         	break;
		}

      if (sig != SIG_BAD)
      {
			t = localtime( &sigdate );
	      sprintf( tmp, "         %s %.2i.%.2i.%i %.2i:%.2i\r",
	      		(const char *)i18n("Signature created"),
	      		t->tm_mday, t->tm_mon+1, t->tm_year+1900, t->tm_hour, t->tm_min );
			statusText( tmp, OUTCOLOR_GPG );
		}

      if (sig == SIG_GOOD)
      {
	      switch (trust)
	      {
				case TRUST_UNDEFINED:
	         	sprintf( tmp, "         %s\r         %s\r",
	                  (const char *)i18n("This key is not certified with a trusted signature!"),
                     (const char *)i18n("I cannot verify that this signature belongs to the person named!") );
					statusText( tmp, OUTCOLOR_GPG );
	         	break;
				case TRUST_NEVER:
	         	sprintf( tmp, "         %s\r         %s\r",
	                  (const char *)i18n("This signature is not trusted!"),
                     (const char *)i18n("There is no indication that the signature belongs to the owner!") );
					statusText( tmp, OUTCOLOR_GPG );
            	break;
				case TRUST_MARGINAL:
	         	sprintf( tmp, "         %s\r",
	                  (const char *)i18n("This signature is trusted marginally.") );
					statusText( tmp, OUTCOLOR_GPG );
            	break;
				case TRUST_FULLY:
				case TRUST_ULTIMATE:
	         	sprintf( tmp, "         %s\r",
	                  (const char *)i18n("This signature is trusted completely!") );
					statusText( tmp, OUTCOLOR_GPG );
            	break;
	      }
		}

      return true;
   }

	strcat( save, tmp );
	len = strlen(save);

   while ((i = POS('\n', save)) != -1)
   {
   	if (i > 9)
      {
	      memcpy( tmp, save+9, i-9 );
	      tmp[i-9] = '\0';
		}
      len -= i+1;
      memmove( save, save+i+1, len );
      save[len] = '\0';


      getarg( tmp, id );

      if (!strcmp(id, "GOODSIG"))
      {
         sig = SIG_GOOD;
      	getarg( tmp, tmp2 );
         strcpy( username, tmp );
      	continue;
      }
      if (!strcmp(id, "BADSIG"))
      {
      	sig = SIG_BAD;
      	getarg( tmp, tmp2 );
         strcpy( username, tmp );
      	continue;
      }
      if (!strcmp(id, "ERRSIG"))
      {
      	getarg( tmp, tmp2 );
      	getarg( tmp, tmp2 );
      	getarg( tmp, tmp2 );
      	getarg( tmp, tmp2 );
      	getarg( tmp, tmp2 );
         sigdate = atol( tmp2 );
         switch (atoi(tmp))
         {
         	case 4: sig = SIG_ERR_UNKNOWN; break;
				case 9: sig = SIG_ERR_MISSING; break;
         }
      	continue;
      }
      if (!strcmp(id, "VALIDSIG"))
      {
      	getarg( tmp, tmp2 );
      	getarg( tmp, tmp2 );
      	getarg( tmp, tmp2 );
         sigdate = atol( tmp2 );
      	continue;
      }
      if (!strcmp(id, "TRUST_UNDEFINED"))
      {
      	trust = TRUST_UNDEFINED;
      	continue;
      }
      if (!strcmp(id, "TRUST_NEVER"))
      {
      	trust = TRUST_NEVER;
      	continue;
      }
      if (!strcmp(id, "TRUST_MARGINAL"))
      {
      	trust = TRUST_MARGINAL;
      	continue;
      }
      if (!strcmp(id, "TRUST_FULLY"))
      {
      	trust = TRUST_FULLY;
      	continue;
      }
      if (!strcmp(id, "TRUST_ULTIMATE"))
      {
      	trust = TRUST_ULTIMATE;
      	continue;
      }
      if (!strcmp(id, "NO_PUBKEY"))
      {
      	getarg( tmp, keyid );
         continue;
      }
   }

   return false;
}
//---------------------------------------------------------------------------
void CheckGPG::checkSignature()
{
	int fds[2];
   int status;
	char statusfd[500];
   fd_set fd_mask;
   bool ready;


   username[0] = '\0';
   trust = TRUST_UNDEFINED;
   sig = SIG_BAD;
   sigdate = 0;
   keyid[0] = '\0';
   ready = false;


	// Kindprozess - Arbeit machen
	if (pipe(fds) == -1)
   {
printf("error: %s\n", strerror(errno));
   	return;
   }


   if (fork() == 0)
   {
   	// Kindprozess. gpg aufrufen
      ::close(fds[0]);
      sprintf( statusfd, "--status-fd=%i", fds[1] );
      if (execlp("gpg", "--no-greeting", statusfd, "--batch", "--verify", filename, 0) == -1)
      	_exit(25);
   }
   else
   {
		// Elternprozess. Auf Daten warten und fertig.
      ::close(fds[1]);
      wait(&status);
printf("status: %i\n", status);

      if (status == 25)
         return;
		ready = false;
		while (!ready)
      {
      	FD_ZERO( &fd_mask );
	      FD_SET( fds[0], &fd_mask);

      	select( fds[0]+1, &fd_mask, NULL, NULL, NULL );

	      if (FD_ISSET(fds[0], &fd_mask))
            ready = rxSignatureData( fds[0] );
      }
   }
}
//---------------------------------------------------------------------------
bool CheckGPG::rxPubkeyData( int fd )
{
	int count;
   char tmp[500], id[50], tmp2[500];
   int i, len;


   count = read( fd, tmp, 499 );
   tmp[count] = '\0';

   if (count == 0)
   {
      unlink(filename);
      ::close(fd);

      // Info-Text fuer den User ausgeben
      sprintf( tmp2, (const char *)i18n("Total number of keys processed: %i"), k_count );
      sprintf( tmp, "\r<LinKT>: %s\r", tmp2 );
      statusText( tmp, OUTCOLOR_GPG );

      if (k_imported > 0)
      {
      	if (k_imported == 1)
	         sprintf( tmp2, (const char *)i18n("Key imported: %s (0x%s)"), username, keyid+8 );
			else
         	sprintf( tmp2, (const char *)i18n("%i keys imported"), k_imported );
	      sprintf( tmp, "         %s\r", tmp2 );
	      statusText( tmp, OUTCOLOR_GPG );
      }

      if (k_unchanged > 0)
      {
			sprintf( tmp2, (const char *)i18n("%i keys unchanged (key already in keyring?)"), k_unchanged );
	      sprintf( tmp, "         %s\r", tmp2 );
	      statusText( tmp, OUTCOLOR_GPG );
      }

      if (k_nouid > 0)
      {
      	if (k_nouid == 1)
				sprintf( tmp2, (const char *)i18n("1 key has no valid user-id") );
			else
				sprintf( tmp2, (const char *)i18n("%i keys have no valid user-id"), k_nouid );
	      sprintf( tmp, "         %s\r", tmp2 );
	      statusText( tmp, OUTCOLOR_GPG );
      }

      if (k_subk > 0)
      {
			sprintf( tmp2, (const char *)i18n("%i subkeys found"), k_subk );
	      sprintf( tmp, "         %s\r", tmp2 );
	      statusText( tmp, OUTCOLOR_GPG );
      }

      if (k_sigs > 0)
      {
			sprintf( tmp2, (const char *)i18n("%i new signatures"), k_sigs );
	      sprintf( tmp, "         %s\r", tmp2 );
	      statusText( tmp, OUTCOLOR_GPG );
      }

      if (k_revoc > 0)
      {
			sprintf( tmp2, (const char *)i18n("%i revocation certificates read"), k_revoc );
	      sprintf( tmp, "         %s\r", tmp2 );
	      statusText( tmp, OUTCOLOR_GPG );
      }

      return true;
   }

	strcat( save, tmp );
	len = strlen(save);

   while ((i = POS('\n', save)) != -1)
   {
   	if (i > 9)
      {
	      memcpy( tmp, save+9, i-9 );
	      tmp[i-9] = '\0';
		}
      len -= i+1;
      memmove( save, save+i+1, len );
      save[len] = '\0';


      getarg( tmp, id );

      if (!strcmp(id, "IMPORTED"))
      {
      	import = IMPORT_OK;
      	getarg( tmp, keyid );
         strcpy( username, tmp );
      	continue;
      }
      if (!strcmp(id, "IMPORT_RES"))
      {
      	getarg( tmp, tmp2 );
         k_count = atoi( tmp2 );
      	getarg( tmp, tmp2 );
         k_nouid = atoi( tmp2 );
      	getarg( tmp, tmp2 );
         k_imported = atoi( tmp2 );
      	getarg( tmp, tmp2 );
      	getarg( tmp, tmp2 );
         k_unchanged = atoi( tmp2 );
      	getarg( tmp, tmp2 );
			k_uid = atoi( tmp2 );
      	getarg( tmp, tmp2 );
			k_subk = atoi( tmp2 );
      	getarg( tmp, tmp2 );
			k_sigs = atoi( tmp2 );
      	getarg( tmp, tmp2 );
   		k_revoc = atoi( tmp2 );
      	getarg( tmp, tmp2 );
   		k_secread = atoi( tmp2 );
      	getarg( tmp, tmp2 );
   		k_secimp = atoi( tmp2 );
   		k_secups = atoi( tmp );
      	continue;
   	}
   }

   return false;
}
//---------------------------------------------------------------------------
void CheckGPG::checkPubKey()
{
	int fds[2];
   int status;
	char statusfd[500];
   fd_set fd_mask;
   bool ready;


   username[0] = '\0';
   keyid[0] = '\0';
   ready = false;
   import = IMPORT_UNKNOWN;
	k_count = 0;
	k_nouid = 0;
	k_imported = 0;
	k_unchanged = 0;
	k_uid = 0;
	k_subk = 0;
	k_sigs = 0;
   k_revoc = 0;
   k_secread = 0;
   k_secimp = 0;
   k_secups = 0;


	// Kindprozess - Arbeit machen
	if (pipe(fds) == -1)
   {
printf("error: %s\n", strerror(errno));
   	return;
   }


   if (fork() == 0)
   {
   	// Kindprozess. gpg aufrufen
      ::close(fds[0]);
      sprintf( statusfd, "--status-fd=%i", fds[1] );
      if (execlp("gpg", "--no-greeting", statusfd, "--batch", "--import", filename, 0) == -1)
      	_exit(25);
   }
   else
   {
		// Elternprozess. Auf Daten warten und fertig.
      ::close(fds[1]);

		ready = false;
		while (!ready)
      {
      	FD_ZERO( &fd_mask );
	      FD_SET( fds[0], &fd_mask);

      	select( fds[0]+1, &fd_mask, NULL, NULL, NULL );

	      if (FD_ISSET(fds[0], &fd_mask))
            ready = rxPubkeyData( fds[0] );
      }
      wait(&status);
   }
}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
GPGSendPKey::GPGSendPKey(  QWidget *parent, Channel *chan )
		: QDialog( parent, "", true ), SendData( chan )
{
	int fds[2], status;
   bool ready;
   fd_set fd_mask;
   QLabel *l;
   QVBoxLayout *vbox, *vboxFragm;
   QHBoxLayout *hbox;


   setCaption( i18n("Send Public-Keys...") );

   save[0] = '\0';
   len = 0;
   data = NULL;
	keyList = new QList<QListViewItem> ();

   vbox = new QVBoxLayout( this );
   vbox->setMargin( 5 );
   vbox->setSpacing( 5 );

   l = new QLabel( this );
   l->setText( i18n("Please select the user-ids you want to send.") );

	// Bildschirm aufbauen
 	IDs = new QListView( this );
   IDs->addColumn( i18n("ID") );
   IDs->addColumn( i18n("Typ") );
   IDs->addColumn( i18n("Length") );
   IDs->addColumn( i18n("User-ID") );
   IDs->setSorting( 3 );
   IDs->setAllColumnsShowFocus( true );
   IDs->setVScrollBarMode( QScrollView::Auto );
   IDs->setHScrollBarMode( QScrollView::Auto );
   IDs->setMultiSelection( true );

   vbox->addWidget( l );
   vbox->addWidget( IDs );

   fragmWidget = new QWidget( this );
   vbox->addWidget( fragmWidget );
   vboxFragm = new QVBoxLayout( fragmWidget );

   l = new QLabel( i18n("Search string:"), fragmWidget );
   edIDFrag = new QLineEdit( fragmWidget );
   connect( edIDFrag, SIGNAL(textChanged(const QString &)), SLOT(slotIDFragChanged(const QString &)) );

   vboxFragm->addSpacing( 5 );
   vboxFragm->addWidget( l );
   vboxFragm->addWidget( edIDFrag );



   hbox = new QHBoxLayout( vbox );

   btnSend = new QPushButton( this );
   btnSend->setText( i18n("&Send") );
   btnSend->setDefault( true );
	connect( btnSend, SIGNAL(clicked()), SLOT(slotSendClicked()) );

   btnAbort = new QPushButton( this );
   btnAbort->setText( i18n("&Abort") );
   connect( btnAbort, SIGNAL(clicked()), SLOT(reject()) );

   hbox->addStretch( 1 );
   hbox->addWidget( btnSend );
   hbox->addWidget( btnAbort );


   // Aktuelle Daten in die Liste packen

	// Kindprozess - Arbeit machen
	if (pipe(fds) == -1)
   {
printf("pipe-error: %s\n", strerror(errno));
   	return;
   }

   if (fork() == 0)
   {
   	// Kindprozess. gpg aufrufen
	   if (dup2(fds[1], 1) != 1)
	   {
printf("dup2-error: %s\n", strerror(errno));
			_exit(25);
	   }

      ::close(fds[0]);
      if (execlp("gpg", "--no-greeting", "--batch", "--list-keys", "--with-colons", 0) == -1)
      	_exit(25);
   }
   else
   {
		// Elternprozess. Auf Daten warten und fertig.
      ::close(fds[1]);

		ready = false;
		while (!ready)
      {
      	FD_ZERO( &fd_mask );
	      FD_SET( fds[0], &fd_mask);

      	select( fds[0]+1, &fd_mask, NULL, NULL, NULL );

	      if (FD_ISSET(fds[0], &fd_mask))
            ready = rxData( fds[0] );
      }
      wait(&status);
   }
}
//---------------------------------------------------------------------------
GPGSendPKey::~GPGSendPKey()
{
	if (data != NULL)
   	free( data );

	delete keyList;
}
//---------------------------------------------------------------------------
bool GPGSendPKey::rxData( int fd )
{
	int count;
   char tmp[500], tmp2[500];
   QString uid, bits, type, id;
   int i, len, iType;


   count = read( fd, tmp, 200 );
   tmp[count] = '\0';

   if (count == 0)
   {
      ::close(fd);

      return true;
   }

	strcat( save, tmp );
	len = strlen(save);

   while ((i = POS('\n', save)) != -1)
   {
      memcpy( tmp, save, i );
      tmp[i] = '\0';
      len -= i+1;
      memmove( save, save+i+1, len );
      save[len] = '\0';

      if ((i = POS('\r', tmp)) != -1) tmp[i] = '\0';

      if (tmp[3] != ':') continue;

		getcharg( tmp, tmp2, ':' );

      if (strcmp(tmp2, "pub")) continue;

		getcharg( tmp, tmp2, ':' );
		getcharg( tmp, tmp2, ':' );
      bits = tmp2;
		getcharg( tmp, tmp2, ':' );
      iType = atoi(tmp2);
		getcharg( tmp, tmp2, ':' );
      id = (char *)(tmp2+8);
		getcharg( tmp, tmp2, ':' );
		getcharg( tmp, tmp2, ':' );
		getcharg( tmp, tmp2, ':' );
		getcharg( tmp, tmp2, ':' );
		getcharg( tmp, tmp2, ':' );
      uid = tmp2;

	   switch (iType)
	   {
	   	case 1: type = "RSA"; break;
	      case 16: type = "ElGamal (e)"; break;
	      case 17: type = "DSA"; break;
	      case 20: type = "ElGamal (se)"; break;
	      default: type = "?????"; break;
	   }

      keyList->append( new QListViewItem(IDs, "0x"+id, type, bits, uid) );
   }

   return false;
}
//---------------------------------------------------------------------------
bool GPGSendPKey::rxDataB( int fd )
{
   char tmp[500], tmp2[500], *str;
   int i, k, wpos, count;
   bool ready=false;


   while (!ready)
   {
	   count = read( fd, tmp, 200 );
	   tmp[count] = '\0';
      if (count != 200) ready = true;

	   if (count == 0)
	   {
	      ::close(fd);

	      // Ok, komplett empfangen - aussenden
         if (data != NULL)
         {
				data[len++] = '\r';
		      data[len] = '\0';

		      sendData( len, data );

				free(data);
		      data = NULL;
		      len = 0;
			}

	      return true;
	   }

	   str = data;
	   data = (char *) malloc( len+(2*count) );
	   if (str != NULL)
		{
			memcpy( data, str, len );
	      free( str );
		}
	   wpos = len;


	   while ((i = POS('\n', tmp)) != -1)
	   {
	      memcpy( tmp2, tmp, i );
	      tmp2[i] = '\0';
	      count -= i+1;
	      memmove( tmp, tmp+i+1, count );
	      tmp[count] = '\0';

	      if ((k = POS('\r', tmp2)) != -1) tmp2[k] = '\0';

			strcat(tmp2, "\r");
			k = strlen(tmp2);
	      memcpy( data+wpos, tmp2, k );
	      wpos += k;
	   }

	   if (count > 0)
	   {
	      memcpy( data+wpos, tmp, count );
	      wpos += count;
	   }

	   len = wpos;
   }

   return false;
}
//---------------------------------------------------------------------------
void GPGSendPKey::slotIDFragChanged( const QString & string )
{
	QListViewItem *item;
   QString str = string.lower();


	if (string.length() == 0)
   {
   	// Kein Suchstring - alles sichtbar machen
		for (item=keyList->first(); item != 0; item=keyList->next())
      {
         if (item->parent() == NULL)
         	IDs->insertItem( item );
      }
      return;
   }

   for (item=keyList->first(); item != 0; item=keyList->next())
   {
   	if (item->text(3).lower().find(str) > -1)
      {
	   	if (item->listView() == 0)
         	IDs->insertItem( item );
		}
		else
      {
	   	if (item->listView() != 0)
	        	IDs->takeItem( item );
		}
   }
}
//---------------------------------------------------------------------------
void GPGSendPKey::slotSendClicked()
{
	int fds[2], status;
   bool ready;
   fd_set fd_mask;
   char **args, comment[100];
   int x=6, i=0;
   QListViewItem *item;


   data = NULL;
   len = 0;

   // Aktuelle Daten in die Liste packen

	// Kindprozess - Arbeit machen
	if (pipe(fds) == -1)
   {
printf("pipe-error: %s\n", strerror(errno));
   	return;
   }

   if (fork() == 0)
   {
   	// Kindprozess. gpg aufrufen
	   if (dup2(fds[1], 1) != 1)
	   {
printf("dup2-error: %s\n", strerror(errno));
			_exit(25);
	   }

      ::close(fds[0]);


	   for (item = IDs->firstChild(); item; item=item->nextSibling())
	   {
	   	if (item->isSelected())
	      	x++;
	   }

		args = (char **)malloc(sizeof(char *)*x);

	   args[0] = (char *) strdup("gpg");
	   args[1] = (char *) strdup("--no-greeting");
	   args[2] = (char *) strdup("--batch");
	   args[3] = (char *) strdup("-a");
      sprintf(comment, "--comment=LinKT V%s", LINKT_VERSION);
      args[4] = (char *) strdup(comment);
	   args[5] = (char *) strdup("--export");
	   for (item = IDs->firstChild(); item; item=item->nextSibling())
	   	if (item->isSelected())
	      {
				args[6+i] = (char *) strdup(item->text(0).latin1());
	      	i++;
	      }
	   args[6+i] = NULL;

      if (execvp("gpg", args) == -1)
     	_exit(25);
   }
   else
   {
		// Elternprozess. Auf Daten warten und fertig.
      ::close(fds[1]);
/*printf("status: %i\n", status);

      if (status == 25)
         return;*/
		ready = false;
		while (!ready)
      {
      	FD_ZERO( &fd_mask );
	      FD_SET( fds[0], &fd_mask);

      	select( fds[0]+1, &fd_mask, NULL, NULL, NULL );

	      if (FD_ISSET(fds[0], &fd_mask))
            ready = rxDataB( fds[0] );
      }

      wait(&status);
	   accept();
		return;
   }
}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
GPGKeys::GPGKeys()
{
	save[0] = '\0';
}
//---------------------------------------------------------------------------
GPGKeys::~GPGKeys()
{
}
//---------------------------------------------------------------------------
void GPGKeys::listPub()
{
	int fds[2];
   int status;
   fd_set fd_mask;
   bool ready;


	// Kindprozess - Arbeit machen
	if (pipe(fds) == -1)
   {
printf("pipe-error: %s\n", strerror(errno));
   	return;
   }

   if (fork() == 0)
   {
   	// Kindprozess. gpg aufrufen
	   if (dup2(fds[1], 1) != 1)
	   {
printf("dup2-error: %s\n", strerror(errno));
			_exit(25);
	   }

      ::close(fds[0]);
      if (execlp("gpg", "--no-greeting", "--batch", "--list-keys", "--with-colons", 0) == -1)
      	_exit(25);
   }
   else
   {
		// Elternprozess. Auf Daten warten und fertig.
      ::close(fds[1]);

		ready = false;
		while (!ready)
      {
      	FD_ZERO( &fd_mask );
	      FD_SET( fds[0], &fd_mask);

      	select( fds[0]+1, &fd_mask, NULL, NULL, NULL );

	      if (FD_ISSET(fds[0], &fd_mask))
            ready = rxData( fds[0] );
      }
      emit gpgKey( NULL );
      wait(&status);
   }
}
//---------------------------------------------------------------------------
void GPGKeys::listSec()
{
	int fds[2];
   int status;
   fd_set fd_mask;
   bool ready;


	// Kindprozess - Arbeit machen
	if (pipe(fds) == -1)
   {
printf("pipe-error: %s\n", strerror(errno));
   	return;
   }

   if (fork() == 0)
   {
   	// Kindprozess. gpg aufrufen
	   if (dup2(fds[1], 1) != 1)
	   {
printf("dup2-error: %s\n", strerror(errno));
			_exit(25);
	   }

      ::close(fds[0]);
      if (execlp("gpg", "--no-greeting", "--batch", "--list-secret-keys", "--with-colons", 0) == -1)
      	_exit(25);
   }
   else
   {
		// Elternprozess. Auf Daten warten und fertig.
      ::close(fds[1]);

		ready = false;
		while (!ready)
      {
      	FD_ZERO( &fd_mask );
	      FD_SET( fds[0], &fd_mask);

      	select( fds[0]+1, &fd_mask, NULL, NULL, NULL );

	      if (FD_ISSET(fds[0], &fd_mask))
            ready = rxData( fds[0] );
      }
      emit gpgKey( NULL );
      wait(&status);
   }
}
//---------------------------------------------------------------------------
bool GPGKeys::rxData( int fd )
{
	int count;
   char tmp[500], tmp2[500];
   int i, len;
   s_GPGKey *key;


   count = read( fd, tmp, 200 );
   tmp[count] = '\0';

   if (count == 0)
   {
      ::close(fd);
      return true;
   }

	strcat( save, tmp );
	len = strlen(save);

   while ((i = POS('\n', save)) != -1)
   {
      memcpy( tmp, save, i );
      tmp[i] = '\0';
      len -= i+1;
      memmove( save, save+i+1, len );
      save[len] = '\0';

      if ((i = POS('\r', tmp)) != -1) tmp[i] = '\0';

      if (tmp[3] != ':') continue;

	   key = (s_GPGKey *) malloc(sizeof(s_GPGKey));
      key->type = NULL;
   	key->bits = NULL;
   	key->iType = NULL;
   	key->id = NULL;
   	key->uid = NULL;

		getcharg( tmp, tmp2, ':' );
      key->type = (char *) strdup(tmp2);
		getcharg( tmp, tmp2, ':' );
		getcharg( tmp, tmp2, ':' );
      key->bits = (char *) strdup(tmp2);
		getcharg( tmp, tmp2, ':' );
      key->iType = (char *) strdup(tmp2);
		getcharg( tmp, tmp2, ':' );
      key->id = (char *) strdup(tmp2+8);
		getcharg( tmp, tmp2, ':' );
		getcharg( tmp, tmp2, ':' );
		getcharg( tmp, tmp2, ':' );
		getcharg( tmp, tmp2, ':' );
		getcharg( tmp, tmp2, ':' );
      key->uid = (char *) strdup(tmp2);

      emit gpgKey( key );
   }

   return false;
}
//---------------------------------------------------------------------------
void GPGKeys::freeKey( s_GPGKey *key )
{
	free( key->type );
	free( key->bits );
	free( key->iType );
	free( key->id );
	free( key->uid );
   free( key );
}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------

