// locator.cpp
// Written by Stefan Pfeiffer, DL1ELY
// ely@1409.org, dl1ely@qsl.net, Stefan@Pfeiffer.net
// (C) 1999 by DL1ELY


#include "locator.h"


// Default Constructor
Locator::Locator()
{
	pv_locator = "\0";
	pv_state   = Locator::LOC_INVALID;
	pv_north   = 0.0;
	pv_east    = 0.0;
};


Locator::Locator(const Locator& a_loc)
{
    pv_locator = a_loc.m_getLocator();
    pv_state   = a_loc.m_getState();

    m_stringLocator2Coordinates();
};


// DG6VJ: Direktes setzen der Position (ohne Locator)
Locator::Locator(const double north, const double east)
{
	pv_locator = "\0";
	pv_state   = Locator::LOC_COMPLETE;

   pv_north = north;
   pv_east = east;
};


Locator::Locator(const char* a_loc)
{
    string s1(a_loc);
    m_set(s1);
};


Locator::~Locator()
{
    pv_state   = Locator::LOC_INVALID;
};


int Locator::m_getState() const
{
    return pv_state;
};


void Locator::m_set(const string& a_loc)
{
    pv_locator = a_loc.substr(0,6);
	m_stringLocator2Coordinates();
};


string Locator::m_getLocator() const
{
    return pv_locator;
};

double Locator::m_getNorth() const
{
     return pv_north;
};
	
double Locator::m_getEast() const
{
     return pv_east;
};

bool Locator::m_isCompleteLocator() const
{
     return(Locator::LOC_COMPLETE==pv_state);
};

bool Locator::m_isExpandableLocator() const
{
     return(Locator::LOC_SUFFIXMISSING==pv_state);
};


// Locator  Returns
// JO31     JO31LL
// JO31H    JO31HL
// JO31JK   JO31JK
string Locator::m_getCompletedLocator() const
{
    if(pv_state == Locator::LOC_SUFFIXMISSING)
    {
	   if(pv_locator.length()<5) return(pv_locator+"LL");
	   if(pv_locator.length()<6) return(pv_locator+"L");
	};
	
	return(pv_locator);
};

// Calculates the geographical coordinates from the locator
void Locator::m_stringLocator2Coordinates()
{
	 string ascii_correction("AA00AA");
	 string tmp_locator;
	 unsigned char mask[6] = {223,223,255,255,223,223};
	 unsigned char locinfo[6];
	 unsigned char i;
	
	 for(i = 0; i < pv_locator.length(); i++)
	    pv_locator[i] = toupper(pv_locator[i]);
	
	 pv_state = Locator::LOC_INVALID;
	 i = pv_locator.length();
	 switch(i)
	 {
	    case 6:
	        if((pv_locator[0]>='A' && pv_locator[0]<='Z') &&
	           (pv_locator[1]>='A' && pv_locator[1]<='Z') &&
	           (pv_locator[2]>='0' && pv_locator[2]<='9') &&
	           (pv_locator[3]>='0' && pv_locator[3]<='9') &&
	           (pv_locator[4]>='A' && pv_locator[5]<='X') &&
	           (pv_locator[5]>='A' && pv_locator[5]<='X')) pv_state = Locator::LOC_COMPLETE;
	        break;
	    case 4:
	        if((pv_locator[0]>='A' && pv_locator[0]<='Z') &&
	           (pv_locator[1]>='A' && pv_locator[1]<='Z') &&
	           (pv_locator[2]>='0' && pv_locator[2]<='9') &&
	           (pv_locator[3]>='0' && pv_locator[3]<='9')) pv_state = Locator::LOC_SUFFIXMISSING;
	        if((pv_locator[0]>='0' && pv_locator[0]<='9') &&
	           (pv_locator[1]>='0' && pv_locator[1]<='9') &&
	           (pv_locator[2]>='A' && pv_locator[2]<='X') &&
	           (pv_locator[3]>='A' && pv_locator[3]<='X')) pv_state = Locator::LOC_PREFIXMISSING;
	        break;
     };

	 if(pv_state == Locator::LOC_COMPLETE || pv_state == Locator::LOC_SUFFIXMISSING)
	 {
	    tmp_locator = m_getCompletedLocator();
	    for(i=0;i<6;i++)
	 	  locinfo[i] = (tmp_locator[i] & mask[i]) - ascii_correction[i];
		  
	    pv_east  = -180.0+ locinfo[0]*20.0 + locinfo[2]*2.0 + locinfo[4]/12.0 + 1.0/24.0;
	    pv_north = - 90.0+ locinfo[1]*10.0 + locinfo[3]*1.0   + locinfo[5]/24.0 + 1.0/48.0;
	 };
};

// Calculates the sinus of a given angle (in degrees)
double Locator::m_degreeSin(const double angle)

{
   return(sin(angle*M_PI/180));
};

// Calculates the cosinus of a given angle (in degrees)
double Locator::m_degreeCos(const double angle)
{
   return(cos(angle*M_PI/180));
};

/* Berechnung des Gradwinkels zum gegebenen Cosinuswert    */
double Locator::m_degreeArcCos(const double cosinus)
{
   double arcbog; /* Hilfsvariable vor Gradumrechnung */

   if(cosinus>=1) return(0);  /* Sonderfall   0 Grad */
   else
      if(cosinus<=-1) return(180);  /* Sonderfall 180 Grad */
      else
      {
         arcbog = M_PI/2-atan(cosinus/(sqrt(1-(cosinus*cosinus))));
         /* Umrechnung vom Bogenmass in Grad */
         return(arcbog*180/M_PI);
      };
};


Locator::Locator(const string& a_loc)
{
   pv_locator = a_loc.substr(0,6);
   m_stringLocator2Coordinates();
};


double Locator::m_getDistanceTo(const Locator& locator2) const
{
	  double result;

	  if((Locator::LOC_INVALID == pv_state) || (Locator::LOC_PREFIXMISSING == pv_state)) return(0.0);
	
	  if((pv_north == locator2.m_getNorth()) && (pv_east == locator2.m_getEast())) return(0.0);
	  
	  result = m_degreeArcCos(m_degreeSin(pv_north)*m_degreeSin(locator2.m_getNorth())
	           +m_degreeCos(pv_north)*m_degreeCos(locator2.m_getNorth())
			   *m_degreeCos(locator2.m_getEast()-pv_east));

	  result = 111.2 * result;
	  
	  return(result);
};

double Locator::m_getDirectionTo(const Locator& locator2) const
{
    double result;
	double angle;
	
    if((Locator::LOC_INVALID == pv_state) || (Locator::LOC_PREFIXMISSING == pv_state)) return(0.0);

	angle = m_degreeArcCos(m_degreeSin(pv_north)*
	        m_degreeSin(locator2.m_getNorth())+m_degreeCos(pv_north)
	  		*m_degreeCos(locator2.m_getNorth())*
	  		m_degreeCos(locator2.m_getEast()-pv_east));

    result = m_degreeArcCos((m_degreeSin(locator2.m_getNorth())-
             m_degreeSin(pv_north)*m_degreeCos(angle))/
	         (m_degreeCos(pv_north)*m_degreeSin(angle)));

    if(m_degreeSin(locator2.m_getEast()-pv_east)>=0) return(result);

    return(360-result);
};
	  




