/*  XMPS - X MPEG Player System
 *  Copyright (C) 1999 Damien Chavarria
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * vob_subtitles.c : subtitles reading/display from vob files
 *
 */

/*********************************************************************
 *                             INCLUDES                              *
 *********************************************************************/

#include <vob_subtitles.h>

/*********************************************************************
 *                            VARIABLES                              *
 *********************************************************************/

typedef struct {

  int dummy;
  xmps_system_plugin_t *system;
  unsigned int          stream_id;
  char                  size1;
  char                  size2;
  unsigned char        *buffer;
  unsigned int          offset[2];
  unsigned char         color[5];
  unsigned char         trans[4];
  unsigned int          width;
  unsigned int          height;
  int                   id; 
  int                   aligned;
  int                   x;
  int                   y;
  int                   x_decal;
  int                   y_decal;
  int                   global_decal;
  int                   data_size;
  int                   packet_size;
  int                   next;
  int                   start_date;
  int                   end_date;
  int                   displaying;
  int                   display_time;
  int                   enabled;

} vob_subtitles_t;

unsigned char colors[17] = {255, 255, 100, 70, 40, 30, 200, 190, 180, 170, 160, 150, 140, 130, 120, 110, 255};

/*********************************************************************
 *                             FUNCTIONS                             *
 *********************************************************************/

/*
 * get_video_addon_info : MANDATORY
 *
 * Used by the plugin_center to get a pointer to the video_addon object.
 *
 */

xmps_video_addon_plugin_t *get_video_addon_info(void)
{
  xmps_video_addon_plugin_t *video_addon;
  
  video_addon = (xmps_video_addon_plugin_t *) malloc(sizeof(xmps_video_addon_plugin_t));

  video_addon->id   = 0;
  video_addon->name = "VOB/DVD Subtitles";
  video_addon->data = (void *) malloc(sizeof(vob_subtitles_t));

  video_addon->open  = vob_open;
  video_addon->get   = vob_get;
  video_addon->set   = vob_set;
  video_addon->apply = vob_apply;
  video_addon->close = vob_close;

  ((vob_subtitles_t *) video_addon->data)->width   = 0;
  ((vob_subtitles_t *) video_addon->data)->height  = 0;
  ((vob_subtitles_t *) video_addon->data)->x       = 0;
  ((vob_subtitles_t *) video_addon->data)->y       = 0;
  ((vob_subtitles_t *) video_addon->data)->next    = 0;
  ((vob_subtitles_t *) video_addon->data)->enabled = 0;

  return video_addon;
}


/*
 * vob_open
 *
 */

unsigned int vob_open(xmps_video_addon_plugin_t *video_addon)
{
  if(video_addon == NULL) {
   
    XMPS_DEBUG("wrong parameters!");
    return 0;
  }
  
  return 1;
}

unsigned int get_byte (vob_subtitles_t *data)
{
  unsigned char byte;
  
  byte = data->buffer[data->offset[data->id]];
  data->offset[data->id]++;
  
  return byte;
}

int get_nibble (vob_subtitles_t *data)
{
  if (data->aligned)
    {
      data->next = get_byte (data);
      data->aligned = 0;
      return data->next >> 4;
    }
  else
    {
      data->aligned = 1;
      return data->next & 0xf;
    }
}

void send_rle (vob_subtitles_t *data, void *image, int length, int colorid)
{
  int l;

  l = length;

  while (l--)
    if (data->x++ < data->width)
      {
	if(colorid != 0 && colorid != 4) {
	  *( ((char *)((char **) image)[0]) + data->global_decal + data->y*720 + data->x) = colors[colorid];
	}
      }
}

void next_line (vob_subtitles_t *data, void *image)
{
  if (data->y >= data->height)
    return;

  send_rle (data, image, data->width, 4);

  data->x = 0;
  data->y++;
  data->id = 1 - data->id;

}

/*
 * vob_get_info
 *
 * - generic info routine
 *
 */

void* vob_get(xmps_video_addon_plugin_t *video_addon, unsigned int flag, void *data)
{
  return NULL;
}

/*
 * vob_set_info
 *
 * - generic info routine
 *
 */

void parse_current_title(vob_subtitles_t *data)
{
  int i;
  unsigned char dummy;
  
  i =  data->data_size + 2;
  
  while (i < data->packet_size)
    {
      dummy = data->buffer [i];
      
      switch (dummy)
	{
	case 0x01: 

	  data->start_date = data->buffer[i - 4];
	  data->start_date = data->start_date << 8;
	  data->start_date = data->start_date + data->buffer[i - 3];

	  //XMPS_DEBUG("start date : %d", data->start_date);

	  i++;
	  break;
	  
	case 0x02: 

	  data->end_date = data->buffer[i - 4];
	  data->end_date = data->end_date << 8;
	  data->end_date = data->end_date + data->buffer[i - 3];

	  //XMPS_DEBUG("end date : %d", data->end_date);
	  

	  i = data->packet_size;
	  break;
	  
	case 0x03: 
	  
	  /*
	   * palette 
	   */
	  
	  //XMPS_DEBUG("found palette");
	  
	  data->color[0] = data->buffer [i+1] >> 4;
	  data->color[1] = data->buffer [i+1] & 0xf;
	  data->color[2] = data->buffer [i+2] >> 4;
	  data->color[3] = data->buffer [i+2] & 0xf;
	  data->color[4] = 0xf;
	  i += 3;
	  break;
	  
	case 0x04: 
	  
	  /* 
	   * transparency palette
	   */
	  
	  //XMPS_DEBUG("found transparency palette");
	  
	  data->trans[0] = data->buffer [i+1] >> 4;
	  data->trans[1] = data->buffer [i+1] & 0xf;
	  data->trans[2] = data->buffer [i+2] >> 4;
	  data->trans[3] = data->buffer [i+2] & 0xf;
	  i += 3;
	  break;
	  
	case 0x05: 
	  
	  /* 
	   * image coordinates 
	   */
	  
	  //XMPS_DEBUG("found image coordinates");
	  
	  data->width = 1 + ( (((unsigned int)data->buffer[i+2] & 0x0f) << 8) + data->buffer[i+3] ) - ( (((unsigned int)data->buffer[i+1]) << 4) + (data->buffer[i+2] >> 4) );
	  data->height = 1 + ( (((unsigned int)data->buffer[i+5] & 0x0f) << 8) + data->buffer[i+6] ) - ( (((unsigned int)data->buffer[i+4]) << 4) + (data->buffer[i+5] >> 4) );
		
	  data->x_decal =  ( (((unsigned int)data->buffer[i+1]) << 4) + (data->buffer[i+2] >> 4) );
	  data->y_decal =  ( (((unsigned int)data->buffer[i+4]) << 4) + (data->buffer[i+5] >> 4) );
	  
	  i += 7;
	  break;
	  
	case 0x06:
	  
	  /* 
	   * image 1 / image 2 offsets 
	   */
	  
	  //XMPS_DEBUG("found offsets");
	  
	  data->offset[0] = data->buffer[i + 1];
	  
	  data->offset[0] = (data->offset[0] << 8) + data->buffer[i + 2] - 2;
	  
	  data->offset[1] = data->buffer[i + 3];
	  
	  data->offset[1] = (data->offset[1] << 8);
	  
	  data->offset[1] =  data->offset[1] + data->buffer[i + 4] - 2;
	  
	  i += 5;
	  
	  break;
	  
	case 0xff: 
	  
	  /* "ff xx yy zz uu" with 'zz uu' == start of control packet
	   * I don't know yet what xx and yy stand for
	   */

	  if ( (data->buffer[i+3] != data->buffer[data->data_size])
	       || (data->buffer[i+4] != data->buffer[data->data_size + 1]) )
	    {
	      XMPS_DEBUG("invalid control header (%.2x%.2x != %.2x%.2x) !\n",
			 data->buffer[i + 3], data->buffer[i + 4], data->buffer[data->data_size], data->buffer[data->data_size + 1] );
	    }
	  i += 5;
	  break;
	  
	default:
	  XMPS_DEBUG("invalid sequence in control header (%.2x) !\n", dummy);
	}
    }
  
  //XMPS_DEBUG("subtitle is %dx%d at %d and %d, located at %dx%d", data->width, data->height, data->offset[0], data->offset[1], data->x_decal, data->y_decal);
}

void compute_next_title(vob_subtitles_t *data)
{
  unsigned char dummy;

  data->system->read(data->system, data->stream_id, &dummy, 1);
  data->packet_size = dummy;
  
  data->system->read(data->system, data->stream_id, &dummy, 1);
  data->packet_size = (data->packet_size << 8) + dummy;
  
  //XMPS_DEBUG("packet size : %d", data->packet_size);
  
  data->buffer = (char *) malloc(data->packet_size - 2);
  data->system->read(data->system, data->stream_id, data->buffer, data->packet_size - 2);
  
  data->data_size = (((unsigned int) data->buffer[0]) << 8) + data->buffer[1];
  
  //XMPS_DEBUG("data size : %d", data->data_size);
  
  /*
   * parsing the control header
   */

  parse_current_title(data);
}


unsigned int vob_set(xmps_video_addon_plugin_t *video_addon, unsigned int flag, void *arg)
{
  vob_subtitles_t *data;

  if(video_addon == NULL) {
    return 0;
  }
  
  data = (vob_subtitles_t *) video_addon->data;
  
  if(data == NULL) {
    return 0;
  }

  switch(flag) {

  case XMPS_FLAG_INPUT:
    {
      xmps_data_t *data_stream;

      data_stream = (xmps_data_t *) arg;

      if(data_stream == NULL) {
	return 0;
      }

      XMPS_DEBUG("checking subtitles type");

      if(data_stream->type == XMPS_DATA_SUBTITLES && data_stream->plugin_type == XMPS_PLUGIN_SYSTEM) {

	XMPS_DEBUG("we accept subtitles");
	
	data->system    = (xmps_system_plugin_t *) data_stream->plugin;
	data->stream_id = data_stream->id;

	compute_next_title(data);
	data->displaying = 0;

	return 1;
      }

    }
    break;
  default:
    break;

  }

  return 0;
}

unsigned int vob_apply(xmps_video_addon_plugin_t *video_addon, void *image, xmps_video_info_t *info, unsigned int time)
{
  vob_subtitles_t *data;
  int             *sub_pts, *vid_pts;
  unsigned int     video_stream = 1;
  
  if(video_addon == NULL) {
    return 0;
  }
  
  data = (vob_subtitles_t *) video_addon->data;
  
  if(data == NULL || data->system == NULL) {
    return 0;
  }

  if(data->enabled) {

    data->x       = 0;
    data->y       = 0;
    data->id      = 0;
    data->aligned = 1;
  
    data->global_decal = data->y_decal * info->width + data->x_decal;
    
    sub_pts = (int *) data->system->get(data->system, XMPS_FLAG_TIME, &data->stream_id);   
    vid_pts = (int *) data->system->get(data->system, XMPS_FLAG_TIME, &video_stream);   
    
    if(*(vid_pts) < *(sub_pts) + data->start_date*10) {
      return 1;
    }
    
    if(!data->displaying) {
      
      data->display_time = time;
    }
    
    data->displaying = 1;
    
    if(*(vid_pts) > *(sub_pts) && + (time - data->display_time) > data->end_date*10) {
      
      compute_next_title(data);
      data->displaying = 0;
      
      return 1;
    }
    
    parse_current_title(data);  
    
    while ((data->offset[1] < data->data_size) && (data->y < data->height))
      {
	int code;
	
	code = get_nibble (data);
	

	if (code >= 0x4)	
	  {
	  found_code:
	    send_rle (data, image, code >> 2, code & 3);
	    if (data->x >= data->width)
	      {
		if (!data->aligned)
		  get_nibble (data);
		next_line (data, image);
	      }
	    continue;
	  }
	
	code = (code << 4) + get_nibble (data);
	if (code >= 0x10)	
	  goto found_code;
	
	code = (code << 4) + get_nibble (data);
	if (code >= 0x40)	
	  goto found_code;
	
	code = (code << 4) + get_nibble (data);
	if (code >= 0x100)
	  goto found_code;
	
	if (!data->aligned)
	  code = (code << 4) + get_nibble (data); 
	
	if (code)
	  {
	    XMPS_DEBUG ("got unknown code 00%x (offset %x, x=%d, y=%d)\n", code, data->offset[data->id], data->x, data->y);
	    next_line (data, image);
	    continue;
	  }
	
	next_line (data, image);
      }
  }

  return 1;
}

unsigned int vob_close(xmps_video_addon_plugin_t *video_addon)
{
  if(video_addon == NULL) {
    return 0;
  }
  
  return 1;
}





