/*  XMPS - X MPEG Player System
 *  Copyright (C) 1999 Damien Chavarria
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 *
 * wav_codec.c
 *
 * WAV Video Codec.
 * @Author Chavarria Damien.
 * 	   Copyright 1999-2000
 */

/*********************************************************************
 *                             INCLUDES                              *
 *********************************************************************/

#include "wav_codec.h"

typedef struct {

  int                   our_stream_id;
  xmps_system_plugin_t *our_system;
  xmps_audio_info_t    *info;
  int                   channels;
  int                   format;
  int                   bit_count;
  int                   sample_size;
  int                   frequency;
  int                   block_align;
  int                   avg_bytes_per_sec;
  int                   data_offset;
  int                   data_length;

} wav_t;

/*
 * get_audio_codec_info : MANDATORY
 *
 * Used by scanplugins to get a pointer to the codec.
 *
 */

xmps_audio_decoder_plugin_t *get_audio_decoder_info(void)
{
  xmps_audio_decoder_plugin_t *audio_decoder;
  
  audio_decoder = (xmps_audio_decoder_plugin_t *) malloc(sizeof(xmps_audio_decoder_plugin_t));

  audio_decoder->name = "WAV";
  audio_decoder->data = malloc(sizeof(wav_t));

  audio_decoder->open       = wav_open;
  audio_decoder->get        = wav_get;
  audio_decoder->set        = wav_set;
  audio_decoder->decompress = wav_decompress;
  audio_decoder->close      = wav_close;

  ((wav_t *) audio_decoder->data)->our_system    = NULL;
  ((wav_t *) audio_decoder->data)->our_stream_id = 0;

  ((wav_t *) audio_decoder->data)->info = (xmps_audio_info_t *) malloc(sizeof(xmps_audio_info_t));

  return audio_decoder;
}

/*********************************************************************
 *                            FUNCTIONS                              *
 *********************************************************************/


unsigned int wav_open(xmps_audio_decoder_plugin_t *audio_decoder)
{

  return 1;
}

void* wav_get(xmps_audio_decoder_plugin_t *audio_decoder, unsigned int flag, void *arg)
{
  wav_t *data;

  if(audio_decoder == NULL) {
    return NULL;
  }

  data = (wav_t *) audio_decoder->data;

  if(data == NULL) {
    return NULL;
  }

  switch(flag)
    {
    case XMPS_FLAG_AUDIO_FORMAT_LIST:
      {
	GList *list = NULL;

	switch(data->bit_count)
	  {
	  case 8:
	    data->info->format.type = XMPS_AUDIO_FORMAT_U8;
	    break;
	  case 16:
	  default:
	    data->info->format.type = XMPS_AUDIO_FORMAT_S16;
	    break;
	  }

	data->info->format.bit_count = data->bit_count;
	
	list = g_list_prepend(list, (void *) &data->info->format);

	return (void *) list;
      }
      break;

    case XMPS_FLAG_AUDIO_INFO:
      {
	data->info->channels          = data->channels;
	data->info->frequency         = data->frequency;
	
	switch(data->bit_count)
	  {
	  case 8:
	    data->info->format.type    = XMPS_AUDIO_FORMAT_U8;
	    break;
	  case 16:
	  default:
	    data->info->format.type    = XMPS_AUDIO_FORMAT_S16;
	    break;
	  }

	data->info->format.bit_count =   data->bit_count;
	data->info->sample_size      = ( data->bit_count + 7 / 8)*data->channels;
	
	return data->info;
      }
      break;

    default:
      break;
      
    }

  return NULL;
}

unsigned int wav_set(xmps_audio_decoder_plugin_t *audio_decoder, unsigned int flag, void *arg)
{
  wav_t *data;

  if(audio_decoder == NULL) {
    return 0;
  }

  data = (wav_t *) audio_decoder->data;

  if(data == NULL) {
    return 0;
  }

  switch(flag)
    {
    case XMPS_FLAG_INPUT:
      {
	xmps_data_t *data_stream;
	char         buffer[12];
	long         len;	
	
	/*
	 * we need to check for data
	 * consistency
	 */
	
	data_stream = (xmps_data_t *) arg;

	if(data_stream == NULL) {
	  return 0;
	}

	if(data_stream->type != XMPS_DATA_AUDIO_COMP && data_stream->type != XMPS_DATA_RAW) {
	  return 0;
	}

	/*
	 * if stream has been setup 
	 * we do not need one
	 *
	 */

	if(data_stream->audio_compression != NULL) {
	
	  if(strstr(data_stream->audio_compression, "WAVE") != NULL) {
	    return 1;
	  }
	}

	/*
	 * try to see from the data 
	 * if we really can play it
	 *
	 */
	
	if(data_stream->plugin_type != XMPS_PLUGIN_SYSTEM) {
	  return 0;
	}
	
	data->our_stream_id = data_stream->id;
	
	data->our_system = (xmps_system_plugin_t *) data_stream->plugin;

	if(data->our_system == NULL) {
	  return 0;
	}
	
	data->our_system->seek(data->our_system, data->our_stream_id, 0, XMPS_SEEK_SET);

	if(data->our_system->read(data->our_system, data->our_stream_id, buffer, 4) != 4) {

	  XMPS_DEBUG("could not read from system:");
	  
	  return 0;
	}

	/*
	 * check for RIFF header
	 *
	 */

	if(!strncasecmp(buffer, "RIFF", 4)) {

	  XMPS_DEBUG("found RIFF header");
    
	  data->our_system->seek(data->our_system, data->our_stream_id, 4, XMPS_SEEK_CUR);

	  if(data->our_system->read(data->our_system, data->our_stream_id, buffer, 4) != 4) {
	    
	    XMPS_DEBUG("could not read from system:");
	  }

	  /*
	   * check for WAVE
	   *
	   */
    
	  if(!strncasecmp(buffer, "WAVE", 4)) {
      
	    XMPS_DEBUG("found WAVE header");
      
	    /*
	     * now we need to find the fmt chunk 
	     * and the start ot the data chunk
	     *
	     */

	    for(;;)
	      {
		if(data->our_system->read(data->our_system, data->our_stream_id, buffer, 4) != 4) {
		  
		  XMPS_DEBUG("could not read from system:");
	    
		  return 0; 
		}
		
		if(data->our_system->read(data->our_system, data->our_stream_id, &len, 4) != 4) {
		  
		  XMPS_DEBUG("could not read from system:");
	    
		  return 0; 
		}
	  
		if (!strncmp("fmt ", buffer, 4))
		  break;
		
		data->our_system->seek(data->our_system, data->our_stream_id, len, XMPS_SEEK_CUR);
		
	      }

	    /*
	     * here we found it so we gather
	     * all wanted information.
	     *
	     */
      
	    if( len < 16) {
	      
	      XMPS_DEBUG("wrong fmt size!");
	      
	      return 0;
	    }

	    data->our_system->read(data->our_system, data->our_stream_id, (void *) &data->format, 2);
	    data->our_system->read(data->our_system, data->our_stream_id, (void *) &data->channels, 2);
	    data->our_system->read(data->our_system, data->our_stream_id, (void *) &data->frequency, 4);
	    data->our_system->read(data->our_system, data->our_stream_id, (void *) &data->avg_bytes_per_sec, 4);
	    data->our_system->read(data->our_system, data->our_stream_id, (void *) &data->block_align, 2);
	    data->our_system->read(data->our_system, data->our_stream_id, (void *) &data->bit_count, 2);
	    
	    XMPS_DEBUG("found fmt : WAVE is %d Hz, %d channels, %d bits",
		       data->frequency, data->channels, data->bit_count);
	    
	    /*
	     * now find the data chunk
	     *
	     */
	    
	    len -= 16;
	    
	    if(len)
	      data->our_system->seek(data->our_system, data->our_stream_id, len, XMPS_SEEK_CUR);
	    
	    for(;;)
	      {
		if(data->our_system->read(data->our_system, data->our_stream_id, buffer, 4) != 4) {
		  
		  XMPS_DEBUG("could not read from system:");
		  
		  return 0; 
		}
		
		if(data->our_system->read(data->our_system, data->our_stream_id, &len, 4) != 4) {
		  
		  XMPS_DEBUG("could not read from system:");
		  
		  return 0; 
		}
		
		if (!strncmp("data", buffer, 4))
		  break;
		
		data->our_system->seek(data->our_system, data->our_stream_id, len, XMPS_SEEK_CUR);
		
	      }
	    
	    /*
	     * here we found it so
	     * memo it.
	     *
	     */
	    
	    data->data_offset = data->our_system->seek(data->our_system, data->our_stream_id, 0, XMPS_SEEK_CUR);
	    data->data_length = len;
	    
	    XMPS_DEBUG("data offset is %d, length %d",
		       data->data_offset, data->data_length);
	    
	    /*
	     * we are at the right offset!
	     *
	     */
	    
	    return 1;
	  }    
	  
	  return 0;
	}
	else {
	  return 0;
	}	
	
      }
      break;

    default:
      break;
    }

  return 1;
}

unsigned int wav_decompress(xmps_audio_decoder_plugin_t *audio_decoder, void *in_buffer, void *out_buffer, unsigned int size)
{
  wav_t *data;

  if(audio_decoder == NULL) {
    return 0;
  }

  data = (wav_t *) audio_decoder->data;

  if(data == NULL) {
    return 0;
  }  

  if(data->our_system == NULL) {
    return 0;
  }

  /*
   * simply read the asked size intoout_buffer
   *
   */

  return (data->our_system->read(data->our_system, data->our_stream_id, out_buffer, size) == size);
}

unsigned int wav_close(xmps_audio_decoder_plugin_t *audio_decoder)
{
  wav_t *data;

  if(audio_decoder == NULL) {
    return 0;
  }

  data = (wav_t *) audio_decoder->data;

  if(data == NULL) {
    return 0;
  }

  data->our_system    = NULL;
  data->our_stream_id = 0;

  return 0;
}



