/*  XMPS - X MPEG Player System
 *  Copyright (C) 1999 Damien Chavarria
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*********************************************************************
 *                             INCLUDES                              *
 *********************************************************************/

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

/*
 * includes : local
 *
 */

unsigned long read_time;
unsigned long avg_read_time  = 900;
unsigned long read_size      = 16384;
unsigned long last_read      = 0;

#include "avi_read.h"

/*********************************************************************
 *                              DEFINES                              *
 *********************************************************************/

#define AVI_MAX_LEN 2000000000
#define HEADERBYTES 2048

#define PAD_EVEN(x) ( ((x)+1) & ~1 )

#define ERR_EXIT(x) \
{ \
   AVI_close(AVI); \
   AVI_errno = x; \
   return 0; \
}

/*********************************************************************
 *                            VARIABLES                              *
 *********************************************************************/

long                 AVI_errno = 0;
xmps_media_plugin_t *our_media;

/*********************************************************************
 *                            FUNCTIONS                              *
 *********************************************************************/

/*
 * copy n into dst as a 4 byte, 
 * little endian number.
 * should also work on 
 * big endian machines 
 *
 */

static void long2str(unsigned char *dst, int n)
{
  dst[0] = (n      ) & 0xff;
  dst[1] = (n >>  8) & 0xff;
  dst[2] = (n >> 16) & 0xff;
  dst[3] = (n >> 24) & 0xff;
}

/* convert a string of 4 
 * or 2 bytes to a number,
 * also working on big 
 * endian machines 
 *
 */

static unsigned long str2ulong(unsigned char *str)
{
  return ( str[0] | (str[1]<<8) | (str[2]<<16) | (str[3]<<24) );
}

static unsigned long str2ushort(unsigned char *str)
{
  return ( str[0] | (str[1]<<8) );
}

/* 
 * Calculate audio sample size 
 * from number of bits and number 
 * of channels. 
 *
 */

int AVI_sampsize(avi_t *AVI)
{
   int s;
   
   s = ((AVI->a_bits + 7) / 8) * AVI->a_chans;
   
   if(s == 0)
     s = 1;
 
   return s;
}

/*
 * adds an index entry
 * video / audio
 *
 */

static int avi_add_index_entry(avi_t *AVI, 
			       unsigned char *tag, 
			       long flags, 
			       long pos, 
			       long len)
{
  void *ptr;

  if(AVI->n_idx >= AVI->max_idx)
   {
     ptr = realloc((void *)AVI->idx,
		   (AVI->max_idx + 4096) * 16);
     
     if(ptr == 0)
       {
         AVI_errno = AVI_ERR_NO_MEM;
         return -1;
       }
     
     AVI->max_idx += 4096;
     AVI->idx = (unsigned char((*)[16]) ) ptr;
   }

   /* 
    * add the index entry 
    *
    */

   memcpy(AVI->idx[AVI->n_idx], tag, 4);
   
   long2str(AVI->idx[AVI->n_idx]+ 4, flags);
   
   long2str(AVI->idx[AVI->n_idx]+ 8, pos);
   
   long2str(AVI->idx[AVI->n_idx]+12, len);

   /* 
    * update counter
    *
    */

   AVI->n_idx++;

   return 0;
}

/*******************************************************************
 *                                                                 *
 *    Utilities for reading video and audio from an AVI File       *
 *                                                                 *
 *******************************************************************/

int AVI_close(avi_t *AVI)
{
  /* 
   * even if there happened a 
   * error, we first clean up 
   *
   */

  if(AVI->idx) 
    free(AVI->idx);
  
  if(AVI->video_index) 
    free(AVI->video_index);
  
  if(AVI->audio_index) 
    free(AVI->audio_index);
  
  free(AVI);

  return 0;
}

/*
 * returns 1 if media is
 * an AVI, 0 if not.
 *
 */

unsigned int AVI_is_avi(xmps_media_plugin_t *media)
{
  char data[24];

  if(media == NULL)
     return 0;

  if( media->read(media, data, 12) != 12 )
    return 0;

  media->seek(media, 0, XMPS_SEEK_SET);

  if( strncasecmp(data, "RIFF", 4) !=0 || 
      strncasecmp(data + 8, "AVI ", 4) !=0 ) {
    return 0;
  }

  return 1;
}

avi_t *AVI_fill_header(avi_t *AVI, int getIndex)
{
  long           i, n, rate; 
  long           scale, idx_type;
  unsigned char *hdrl_data;
  long           hdrl_len = 0;
  long           nvi, nai, ioff;
  long           tot;
  int            lasttag = 0;
  int            vids_strh_seen = 0;
  int            vids_strf_seen = 0;
  int            auds_strh_seen = 0;
  int            auds_strf_seen = 0;
  int            num_stream = 0;
  char           data[256];

   /* go through the AVI file and 
    * extract the header list,
    * the start position of the 'movi' 
    * list and an optionally
    * present idx1 tag 
    *
    */

  hdrl_data = 0;
   
  while(1)
    {
      if( our_media->read(our_media, data, 8) != 8 ) 
	break; 
      
      /*
       * we assume it's EOF 
       *
       */

      n = str2ulong(data+4);
      n = PAD_EVEN(n);

      if(strncasecmp(data, "LIST", 4) == 0)
	{
	  if( our_media->read(our_media, data,4) != 4 ) 
	    ERR_EXIT(AVI_ERR_READ);
         
	  n -= 4;
	  
	  if(strncasecmp(data, "hdrl", 4) == 0)
	    {
	      hdrl_len = n;
	      
	      hdrl_data = (unsigned char *) malloc(n);
            
	      if(hdrl_data==0) 
		ERR_EXIT(AVI_ERR_NO_MEM);
	      
	      if( our_media->read(our_media, hdrl_data,n) != n ) 
		ERR_EXIT(AVI_ERR_READ);
	    }
	  else if(strncasecmp(data, "movi", 4) == 0)
	    {
	      AVI->movi_start = our_media->seek(our_media, 0, XMPS_SEEK_CUR);
	      our_media->seek(our_media, n, XMPS_SEEK_CUR);
	    }
	  else {
            our_media->seek(our_media, n, XMPS_SEEK_CUR);
	  
	  }
	}
      else if(strncasecmp(data,"idx1",4) == 0)
      {
         /*
	  * n must be a multiple of 16, 
	  * but the reading does not
          * break if this is not the case 
	  *
	  */

	AVI->n_idx = AVI->max_idx = n/16;
	AVI->idx = (unsigned  char((*)[16]) ) malloc(n);
        
	if(AVI->idx == 0) 
	  ERR_EXIT(AVI_ERR_NO_MEM);
         
	if( our_media->read(our_media, AVI->idx,n) != n ) 
	  ERR_EXIT(AVI_ERR_READ);
      }
      else
	our_media->seek(our_media, n, XMPS_SEEK_CUR);
   }
  
  if(!hdrl_data) 
    ERR_EXIT(AVI_ERR_NO_HDRL);
	
  if(!AVI->movi_start) 
    ERR_EXIT(AVI_ERR_NO_MOVI);

   /* 
    * interpret the header list 
    *
    */

   for(i=0; i < hdrl_len; )
   {
      /* 
       * list tags are completly ignored 
       *
       */

     if(strncasecmp(hdrl_data + i, "LIST", 4)==0) 
       { 
	 i+= 12; 
	 continue; 
       }

     n = str2ulong(hdrl_data+i+4);
     n = PAD_EVEN(n);

     /* 
      * interpret the tag and its args 
      *
      */

     if(strncasecmp(hdrl_data + i, "strh", 4)==0)
       {
         i += 8;
         if(strncasecmp(hdrl_data + i, "vids", 4) == 0 && !vids_strh_seen)
	   {
	     memcpy(AVI->compressor,hdrl_data+i+4, 4);
            
	     AVI->compressor[4] = 0;
            
	     scale = str2ulong(hdrl_data+i+20);
	     rate  = str2ulong(hdrl_data+i+24);
	     
	     if(scale!=0) 
	       AVI->fps = (double)rate/(double)scale;
            
	     AVI->video_frames = str2ulong(hdrl_data+i+32);
	     AVI->video_strn = num_stream;
            
	     vids_strh_seen = 1;
	     lasttag = 1; 
	   }
         else if (strncasecmp (hdrl_data + i, "auds", 4) == 0 && ! auds_strh_seen)
	   {
	     AVI->audio_bytes = str2ulong(hdrl_data+i+32)*AVI_sampsize(AVI);
	     AVI->audio_strn = num_stream;
            
	     auds_strh_seen = 1;
	     lasttag = 2; 
	   }
         else
            lasttag = 0;
         num_stream++;
       }
     else if(strncasecmp(hdrl_data + i, "strf", 4)==0)
       {
         i += 8;
         if(lasttag == 1)
	   {

	     /*
	      * keep a copy of 
	      * the bitmapinfoheader
	      *
	      */

	     memcpy(&AVI->bitmapinfoheader, 
		    hdrl_data + i,
		    sizeof(BITMAPINFOHEADER));  

	     if(AVI->bitmapinfoheader.biCompression == 4) {
	       
	       AVI->bitmapinfoheader.biCompression = str2ulong("DIVX");
	     }

	     AVI->width  = str2ulong(hdrl_data+i+4);
	     AVI->height = str2ulong(hdrl_data+i+8);
	     vids_strf_seen = 1;
	   }
         else if(lasttag == 2)
         {
            AVI->a_fmt   = str2ushort(hdrl_data + i  );
            AVI->a_chans = str2ushort(hdrl_data + i + 2);
            AVI->a_rate  = str2ulong (hdrl_data + i + 4);
            AVI->a_bits  = str2ushort(hdrl_data + i + 14) == 0 ? 16 : str2ushort(hdrl_data + i + 14);
            
	    auds_strf_seen = 1;
         }
         lasttag = 0;
      }
      else
      {
         i += 8;
         lasttag = 0;
      }

      i += n;
   }

   free(hdrl_data);

   if(!vids_strh_seen || !vids_strf_seen || AVI->video_frames==0) 
     ERR_EXIT(AVI_ERR_NO_VIDS);

   AVI->video_tag[0] = AVI->video_strn/10 + '0';
   AVI->video_tag[1] = AVI->video_strn%10 + '0';
   AVI->video_tag[2] = 'd';
   AVI->video_tag[3] = 'b';

   /* 
    * audio tag is set to "99wb" 
    * if no audio present 
    *
    */
   
   if(!AVI->a_chans) 
     AVI->audio_strn = 99;

   AVI->audio_tag[0] = AVI->audio_strn/10 + '0';
   AVI->audio_tag[1] = AVI->audio_strn%10 + '0';
   AVI->audio_tag[2] = 'w';
   AVI->audio_tag[3] = 'b';

   our_media->seek(our_media, AVI->movi_start, 
		   XMPS_SEEK_SET);

   /* 
    * get index if wanted 
    *
    */

   if(!getIndex) 
     return AVI;

   /* 
    * if the file has an idx1, 
    * check if this is relative
    * to the start of the file or 
    * to the start of the movi list 
    *
    */

   idx_type = 0;

   if(AVI->idx)
     {
       long pos, len;

       /*
	* search the first videoframe 
	* in the idx1 and look where
        * it is in the file 
	*
	*/

       for(i=0;i<AVI->n_idx;i++) {
	 if( strncasecmp(AVI->idx[i], AVI->video_tag, 3) == 0 ) 
	   break;
       }

       if(i >= AVI->n_idx) 
	 ERR_EXIT(AVI_ERR_NO_VIDS);

       pos = str2ulong(AVI->idx[i]+ 8);
       len = str2ulong(AVI->idx[i]+12);

       our_media->seek(our_media, pos, XMPS_SEEK_SET);

       if(our_media->read(our_media, data,8)!=8) 
	 ERR_EXIT(AVI_ERR_READ);
	 
	 if( strncasecmp(data, AVI->idx[i], 4) == 0 && str2ulong(data + 4) == len )
	   {
	     /* 
	      * index from start of file
	      *
	      */

	     idx_type = 1; 
	   }
	 else
	   {
	     our_media->seek(our_media, pos+AVI->movi_start-4,XMPS_SEEK_SET);
         
	     if(our_media->read(our_media, data, 8) != 8) 
	       ERR_EXIT(AVI_ERR_READ);
	     
	     if( strncasecmp(data, AVI->idx[i], 4) == 0 && str2ulong(data + 4) == len )
	       {
		 /* 
		  * index from start of movi list 
		  *
		  */

		 idx_type = 2; 
	       }
	   }
       /* 
	* idx_type remains 0 if neither 
	* of the two tests above succeeds 
	*
	*/
     }

   if(idx_type == 0)
     {
       /* 
	* we must search through 
	* the file to get the index 
	*
	*/

       our_media->seek(our_media, AVI->movi_start, XMPS_SEEK_SET);

       AVI->n_idx = 0;

       while(1)
	 {
	   
	   if( our_media->read(our_media, data, 8) != 8 ) 
	     break;
	   
	   n = str2ulong(data + 4);

	   /* 
	    * the movi list may contain sub-lists, 
	    * ignore them 
	    *
	    */

	   if(strncasecmp(data,"LIST", 4) == 0)
	     {
	       our_media->seek(our_media, 4, XMPS_SEEK_CUR);
	       continue;
	     }

	   /* 
	    * check if we got a tag ##db, 
	    * ##dc or ##wb 
	    * 
	    */

	   if( ( (data[2]=='d' || data[2]=='D') &&
		 (data[3]=='b' || data[3]=='B' || data[3]=='c' || data[3]=='C') ) || 
	       ( (data[2]=='w' || data[2]=='W') &&
		 (data[3]=='b' || data[3]=='B') ) )
	     {
	       avi_add_index_entry(AVI,
				   data,
				   0, 
				   our_media->seek(our_media, 0, XMPS_SEEK_CUR) - 8, 
				   n);
	     }

         our_media->seek(our_media, PAD_EVEN(n),
			 XMPS_SEEK_CUR);
      }
       idx_type = 1;
     }
   
   /* 
    * now generate the video index 
    * and audio index arrays 
    *
    */
   
   nvi = 0;
   nai = 0;
   
   for(i=0; i < AVI->n_idx; i++)
     {
       if(strncasecmp(AVI->idx[i], AVI->video_tag, 3) == 0) 
	 nvi++;
       
       if(strncasecmp(AVI->idx[i], AVI->audio_tag, 4) == 0) 
	 nai++;
     }
   
   AVI->video_frames = nvi;
   AVI->audio_chunks = nai;

   if(AVI->video_frames == 0) 
     ERR_EXIT(AVI_ERR_NO_VIDS);
   
   AVI->video_index = (video_index_entry *) malloc(nvi*sizeof(video_index_entry));
   
   if(AVI->video_index == 0) 
     ERR_EXIT(AVI_ERR_NO_MEM);

   if(AVI->audio_chunks)
     {
       AVI->audio_index = (audio_index_entry *) malloc(nai*sizeof(audio_index_entry));
       
       if(AVI->audio_index==0) 
	 ERR_EXIT(AVI_ERR_NO_MEM);
     }

   nvi = 0;
   nai = 0;
   tot = 0;
   
   ioff = idx_type == 1 ? 8 : AVI->movi_start+4;

   for(i=0; i < AVI->n_idx; i++)
   {
      if(strncasecmp(AVI->idx[i], AVI->video_tag, 3) == 0)
      {
	AVI->video_index[nvi].flags = str2ulong(AVI->idx[i]+ 4);
	AVI->video_index[nvi].pos   = str2ulong(AVI->idx[i]+ 8)+ioff;
	AVI->video_index[nvi].len   = str2ulong(AVI->idx[i]+12);
	
	nvi++;
      }
      if(strncasecmp(AVI->idx[i], AVI->audio_tag, 4) == 0)
      {
         AVI->audio_index[nai].pos = str2ulong(AVI->idx[i]+ 8)+ioff;
         AVI->audio_index[nai].len = str2ulong(AVI->idx[i]+12);
         AVI->audio_index[nai].tot = tot;

         tot += AVI->audio_index[nai].len;

         nai++;
      }
   }

   AVI->audio_bytes = tot;

   /* 
    * reposition the file 
    *
    */

   our_media->seek(our_media, AVI->movi_start, 
		   XMPS_SEEK_SET);
   
   AVI->video_pos = 0;

   return AVI;
}

/*
 * opens an avi file for reading
 *
 */

avi_t *AVI_open_input(xmps_media_plugin_t *media, 
		      int getIndex)
{
  avi_t *AVI;
  char   data[24];

  if(media == NULL)
    return NULL;

  our_media = media;

  /* 
   * create avi_t structure 
   *
   */

  AVI = (avi_t *) malloc(sizeof(avi_t));
  
  if(AVI == NULL)
    {
      AVI_errno = AVI_ERR_NO_MEM;
      return 0;
    }
  
  memset((void *) AVI, 0, sizeof(avi_t));

  /* 
   * read first 12 bytes and 
   * check that this is an AVI file 
   *
   */

  if( our_media->read(our_media, data, 12) != 12 ) 
    ERR_EXIT(AVI_ERR_READ);

  if( strncasecmp(data  ,"RIFF", 4) !=0 ||
      strncasecmp(data + 8,"AVI ", 4) !=0 ) 
    ERR_EXIT(AVI_ERR_NO_AVI);

  return AVI_fill_header(AVI, getIndex);
}

unsigned int AVI_has_audio(avi_t *AVI)
{
  if(AVI == NULL)
    return 0;

  return (AVI->audio_index != NULL);
}

unsigned int AVI_has_video(avi_t *AVI)
{
  if(AVI == NULL)
    return 0;

  return (AVI->video_index != NULL);
}

/*
 * gives the number of video frames
 *
 */

long AVI_video_frames(avi_t *AVI)
{
   return AVI->video_frames;
}

int  AVI_video_width(avi_t *AVI)
{
   return AVI->width;
}

int  AVI_video_height(avi_t *AVI)
{
   return AVI->height;
}

double AVI_frame_rate(avi_t *AVI)
{
   return AVI->fps;
}

char* AVI_video_compressor(avi_t *AVI)
{
   return AVI->compressor;
}

int AVI_audio_channels(avi_t *AVI)
{
   return AVI->a_chans;
}

int AVI_audio_bits(avi_t *AVI)
{
   return AVI->a_bits;
}

int AVI_audio_format(avi_t *AVI)
{
   return AVI->a_fmt;
}

long AVI_audio_rate(avi_t *AVI)
{
   return AVI->a_rate;
}

long AVI_audio_bytes(avi_t *AVI)
{
   return AVI->audio_bytes;
}

long AVI_frame_size(avi_t *AVI, long frame)
{
  if(!AVI->video_index)         
    { 
      AVI_errno = AVI_ERR_NO_IDX;   
      return -1; 
    }

  if(frame < 0 || frame >= AVI->video_frames) {

    XMPS_DEBUG("frame out of bounds");
    return 0;
  }

  //XMPS_DEBUG("this is the frame size : %d", AVI->video_index[frame].len);

  return(AVI->video_index[frame].len);
}

long AVI_video_time(avi_t *AVI)
{
  if(AVI == NULL)
    return 0;

  /*
   * times are in ms
   *
   */
  
  return (long) (AVI->video_pos*1000/AVI_frame_rate(AVI));
}

long AVI_audio_time(avi_t *AVI)
{
  /*
   * times are in ms
   *
   */
  
  return 0;
}

int AVI_seek_start(avi_t *AVI)
{
  if(AVI == NULL)
    return 0;

  our_media->seek(our_media, AVI->movi_start,
		  XMPS_SEEK_SET);
  
  AVI->video_pos = 0;
  
  return 0;
}

int AVI_set_video_position(avi_t *AVI, long frame)
{
   if(!AVI->video_index)         
     { 
       AVI_errno = AVI_ERR_NO_IDX;   
       return -1; 
     }

   if (frame < 0 ) frame = 0;
   AVI->video_pos = frame;
   return 0;
}
      

/*
 * read next frame's data
 *
 */

long AVI_read_frame(avi_t *AVI, char *vidbuf)
{
   long n;

   if(!AVI->video_index)         
     { 
       AVI_errno = AVI_ERR_NO_IDX;   
       return -1; 
     }

   if(AVI->video_pos < 0 || AVI->video_pos >= AVI->video_frames) 
     return 0;
   
   n = AVI->video_index[AVI->video_pos].len;

   our_media->seek(our_media, AVI->video_index[AVI->video_pos].pos, XMPS_SEEK_SET);
   
   if (our_media->read(our_media, vidbuf, n) != n)
     {
       AVI_errno = AVI_ERR_READ;
       return -1;
     }

   AVI->video_pos++;

   return n;
}

unsigned int AVI_is_keyframe(avi_t *AVI, long frame)
{
  if(AVI == NULL)
    return 0;
  
  if(frame < 0)
    frame = 0;

  if(!AVI->video_index)         
    { 
      AVI_errno = AVI_ERR_NO_IDX;   
      
      /*
       * we still return 1 to avoid looping
       * on waiting for a keyframe.
       *
       */

      return 1; 
    }

  return AVI->video_index[frame].flags & AVIIF_KEYFRAME;

}


int AVI_set_audio_position(avi_t *AVI, long byte)
{
   long n0, n1, n;

   if(!AVI->audio_index)         
     { 
       AVI_errno = AVI_ERR_NO_IDX;   
       return -1; 
     }

   if(byte < 0) byte = 0;

   n0 = 0;
   n1 = AVI->audio_chunks;

   while(n0 < n1 - 1)
   {
      n = (n0 + n1) / 2;
      if(AVI->audio_index[n].tot > byte)
         n1 = n;
      else
         n0 = n;
   }

   AVI->audio_posc = n0;

   XMPS_DEBUG("avi audio chunck size : %d", (int) AVI->audio_index[n0].len);

   if(AVI->audio_index[n0].len > 2000) {
     AVI->audio_posb = byte - AVI->audio_index[n0].tot;
   }
   else {
     AVI->audio_posb = 0;
   }

   return 0;
}

long AVI_read_audio(avi_t *AVI, char *audbuf, long bytes)
{
   long nr, pos, left, todo;

   if(!AVI->audio_index)         
     { 
       AVI_errno = AVI_ERR_NO_IDX;   
       return -1; 
     }

   nr = 0; 

   while(bytes>0)
   {
      left = AVI->audio_index[AVI->audio_posc].len - AVI->audio_posb;
      if(left==0)
      {
         if(AVI->audio_posc>=AVI->audio_chunks-1) return nr;
         AVI->audio_posc++;
         AVI->audio_posb = 0;
         continue;
      }
      if(bytes<left)
         todo = bytes;
      else
         todo = left;
      pos = AVI->audio_index[AVI->audio_posc].pos + AVI->audio_posb;
      our_media->seek(our_media, pos, XMPS_SEEK_SET);
      if (our_media->read(our_media, audbuf+nr,todo) != todo)
      {
         AVI_errno = AVI_ERR_READ;
         return -1;
      }
      bytes -= todo;
      nr    += todo;
      AVI->audio_posb += todo;
   }

   return nr;
}

/* AVI_read_data: Special routine for reading the next audio or video chunk
                  without having an index of the file. */

int AVI_read_data(avi_t *AVI, char *vidbuf, long max_vidbuf,
                              char *audbuf, long max_audbuf,
                              long *len)
{

/*
 * Return codes:
 *
 *    1 = video data read
 *    2 = audio data read
 *    0 = reached EOF
 *   -1 = video buffer too small
 *   -2 = audio buffer too small
 */

   int n;
   char data[8];

   while(1)
   {
      /* Read tag and length */

      if( our_media->read(our_media, data,8) != 8 ) return 0;

      /* if we got a list tag, ignore it */

      if(strncasecmp(data,"LIST",4) == 0)
      {
         our_media->seek(our_media, 4,XMPS_SEEK_CUR);
         continue;
      }

      n = PAD_EVEN(str2ulong(data+4));

      if(strncasecmp(data,AVI->video_tag,3) == 0)
      {
         *len = n;
         AVI->video_pos++;
         if(n>max_vidbuf)
         {
            our_media->seek(our_media, n,XMPS_SEEK_CUR);
            return -1;
         }
         if(our_media->read(our_media, vidbuf,n) != n ) return 0;
         return 1;
      }
      else if(strncasecmp(data,AVI->audio_tag,4) == 0)
      {
         *len = n;
         if(n>max_audbuf)
         {
            our_media->seek(our_media, n,XMPS_SEEK_CUR);
            return -2;
         }
         if(our_media->read(our_media, audbuf,n) != n ) return 0;
         return 2;
         break;
      }
      else
         if(our_media->seek(our_media, n,XMPS_SEEK_CUR)<0)  return 0;
   }
}

/* AVI_print_error: Print most recent error (similar to perror) */

char *(avi_errors[]) =
{
  /*  0 */ "avilib - No Error",
  /*  1 */ "avilib - AVI file size limit reached",
  /*  2 */ "avilib - Error opening AVI file",
  /*  3 */ "avilib - Error reading from AVI file",
  /*  4 */ "avilib - Error writing to AVI file",
  /*  5 */ "avilib - Error writing index (file may still be useable)",
  /*  6 */ "avilib - Error closing AVI file",
  /*  7 */ "avilib - Operation (read/write) not permitted",
  /*  8 */ "avilib - Out of memory (malloc failed)",
  /*  9 */ "avilib - Not an AVI file",
  /* 10 */ "avilib - AVI file has no header list (corrupted?)",
  /* 11 */ "avilib - AVI file has no MOVI list (corrupted?)",
  /* 12 */ "avilib - AVI file has no video data",
  /* 13 */ "avilib - operation needs an index",
  /* 14 */ "avilib - Unkown Error"
};
static int num_avi_errors = sizeof(avi_errors)/sizeof(char*);

static char error_string[4096];

void AVI_print_error(char *str)
{
   int aerrno;

   aerrno = (AVI_errno>=0 && AVI_errno<num_avi_errors) ? AVI_errno : num_avi_errors-1;

   fprintf(stderr,"%s: %s\n",str,avi_errors[aerrno]);

   /* for the following errors, perror should report a more detailed reason: */

   if(AVI_errno == AVI_ERR_OPEN ||
      AVI_errno == AVI_ERR_READ ||
      AVI_errno == AVI_ERR_WRITE ||
      AVI_errno == AVI_ERR_WRITE_INDEX ||
      AVI_errno == AVI_ERR_CLOSE )
   {
      perror("REASON");
   }
}

char *AVI_strerror()
{
   int aerrno;

   aerrno = (AVI_errno>=0 && AVI_errno<num_avi_errors) ? AVI_errno : num_avi_errors-1;

   if(AVI_errno == AVI_ERR_OPEN ||
      AVI_errno == AVI_ERR_READ ||
      AVI_errno == AVI_ERR_WRITE ||
      AVI_errno == AVI_ERR_WRITE_INDEX ||
      AVI_errno == AVI_ERR_CLOSE )
   {
      sprintf(error_string,"%s - %s",avi_errors[aerrno],strerror(errno));
      return error_string;
   }
   else
   {
      return avi_errors[aerrno];
   }
}
