/*
Donut Bump Mapping Demo
This demo shows how to use a bump mapping technique using Glide(tm)
Copyright (C) 1999  3Dfx Interactive, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "basics.h"
#include "mathutil.h"
#include "lighting.h"
#include "crystal.h"
#include "tlib.h"
#include "torus_xforms.h"
#include "torus.h"

// vertex data
typedef struct
{
	GrVertex xformed_vert;
	Vector vert;
	Vector norm;
	Matrix xy_to_st_mat;
} TorusVertInfo;

// local variables
static TorusVertInfo gTorusVerts[(MAX_TORUS_DETAIL+1)*(MAX_TORUS_DETAIL+1)];

// local function prototypes
static void WorldToTextureSpaceMat(Matrix m, Vector n, Vector u);

// extern variables
extern int gNumTMUsUsed, gWireframe, gDrawNormals;
extern Vector gXformedLightDir;
extern Matrix gProjMat;
extern CrystalBall gObjectCrystalBall;
extern float gLightAmbientFactor, gLightDiffuseFactor, gLightSpecularFactor;

// n1 and n2 must both be less than MAX_TORUS_DETAIL
// I calculate one extra vertex data to avoid having to wrap
void PreCalcTorusVerts(int n1, int n2, float R1, float R2)
{
	int i, j;
	float angle1, angle2, d1, d2, R;
	float r1, z1;
	float cos2, sin2, sin1, cos1;
	Vector tangent;
	TorusVertInfo *vert_data;

	d1 = (2.0f*PI)/(float)n1;
	d2 = (2.0f*PI)/(float)n2;
	R = 0.5f*(R2 - R1);

	vert_data = &gTorusVerts[0];

	for (i=0, angle1=0.0f; i<=n1; i++, angle1+=d1)
	{
		fsincos(angle1, &sin1, &cos1);

		z1 = R*sin1;
		r1 = R1 + R*(1.0f + cos1);

		fsincos(d2, &sin2, &cos2);
		for (j=0, angle2=0.0f; j<=n2; j++, angle2+=d2)
		{
			fsincos(angle2, &sin2, &cos2);

			vert_data->vert[X] = r1*cos2;
			vert_data->vert[Y] = r1*sin2;
			vert_data->vert[Z] = z1;
			vert_data->vert[W] = 1.0f;
			vert_data->norm[X] = cos1*cos2;
			vert_data->norm[Y] = cos1*sin2;
			vert_data->norm[Z] = sin1;
			vert_data->norm[W] = 0.0f;
			tangent[X] = -sin1*cos2;
			tangent[Y] = -sin1*sin2;
			tangent[Z] = cos1;
			tangent[W] = 0.0f;
			WorldToTextureSpaceMat(vert_data->xy_to_st_mat, vert_data->norm, tangent);

			vert_data++;
		}
	}
}

// transform and calculate the lighting for each vertex of the torus
// this is used to take advantage of the shared vertices
void XformAndLightTorusVerts(int n1, int n2, FxBool calc_specular)
{
	int i;
	float diffuse, specular;
	TorusVertInfo *vert_data;

	SetCurrMatrix(gProjMat);
	BeginXforms();

	vert_data = &gTorusVerts[0];
	for (i=(n1+1)*(n2+1); i>0; i--)
	{
		// lighting
		if (calc_specular)
		{
			DirLightDiffuseAndSpecular(vert_data->norm, &diffuse, &specular);
			specular *= gLightSpecularFactor;
		}
		else
		{
			diffuse = DirLightDiffuse(vert_data->norm);
			specular = 0.0f;
		}
		diffuse = gLightAmbientFactor + gLightDiffuseFactor*diffuse;

		vert_data->xformed_vert.r = diffuse;
		vert_data->xformed_vert.g = diffuse;
		vert_data->xformed_vert.b = diffuse;
		vert_data->xformed_vert.a = specular;

		// transform
		XformVertex(&vert_data->xformed_vert, &vert_data->vert);
#ifndef USE_GLIDE3
		vert_data->xformed_vert.x = SNAP(vert_data->xformed_vert.x);
		vert_data->xformed_vert.y = SNAP(vert_data->xformed_vert.y);
#endif // !USE_GLIDE3

		vert_data++;
	}

	EndXforms();
}

// uses the pre-calculated transformed and lit vertices to draw the torus
// the function returns the number of polygons (triangles) rendered
int DrawTorus(int n1, int n2, float R1, float R2, float alpha, float depth, FxU8 mode)
{
	int i, j;
	float s, t, ds, dt;
	Vector norm1, norm2, offset;
	TorusVertInfo *vert_data0, *vert_data1;
	GrVertex verts[4], *vert_ptrs[4], *tmp_vert;

	vert_ptrs[0] = &verts[0];
	vert_ptrs[1] = &verts[1];
	vert_ptrs[2] = &verts[2];
	vert_ptrs[3] = &verts[3];

	// the texture will wrap 4 times (4*256 = 1024)
	ds = 1024.0f/(float)n1;
	dt = 1024.0f/(float)n2;

	vert_data0 = &gTorusVerts[0];
	vert_data1 = &gTorusVerts[n2+1];

	s = 0.0f;
	for (i=n1-1; i>=0; i--, s+=ds)
	{
		t = 0.0f;
		for (j=n2; j>=0; j--, t+=dt)
		{
			vert_ptrs[0]->x = vert_data0->xformed_vert.x;
			vert_ptrs[0]->y = vert_data0->xformed_vert.y;
			vert_ptrs[0]->oow = vert_data0->xformed_vert.oow;
			vert_ptrs[0]->r = vert_data0->xformed_vert.r;
			vert_ptrs[0]->g = vert_data0->xformed_vert.g;
			vert_ptrs[0]->b = vert_data0->xformed_vert.b;
			vert_ptrs[0]->a = vert_data0->xformed_vert.a + alpha;

			if (gNumTMUsUsed == 1 && (mode & TORUS_FIRST_PASS))
			{
				vert_ptrs[0]->tmuvtx[0].sow =      s*vert_ptrs[0]->oow;
				vert_ptrs[0]->tmuvtx[0].tow = (t+dt)*vert_ptrs[0]->oow;
			}
			else
			{
				if (gNumTMUsUsed >= 2)
				{
					if (mode & (TORUS_BUMPMAP | TORUS_ENVMAP))
					{
						vert_ptrs[0]->tmuvtx[1].sow =      s*vert_ptrs[0]->oow;
						vert_ptrs[0]->tmuvtx[1].tow = (t+dt)*vert_ptrs[0]->oow;
					}
					else
					{
						vert_ptrs[0]->tmuvtx[0].sow =      s*vert_ptrs[0]->oow;
						vert_ptrs[0]->tmuvtx[0].tow = (t+dt)*vert_ptrs[0]->oow;
					}
				}
				if (mode & TORUS_BUMPMAP)
				{
					// shifted bumpmap
					MatMultVec3x3_2(offset, vert_data0->xy_to_st_mat, gXformedLightDir);
					vert_ptrs[0]->tmuvtx[0].sow = (   s + offset[0]*depth)*vert_ptrs[0]->oow;
					vert_ptrs[0]->tmuvtx[0].tow = (t+dt - offset[1]*depth)*vert_ptrs[0]->oow;
				}
				else if (mode & TORUS_ENVMAP)
				{
					// envmap pass
					MatMultVec3x3_2(norm1, gObjectCrystalBall.rot_mat, vert_data0->norm);
					vert_ptrs[0]->tmuvtx[0].sow = 256.0f*(0.5f + 0.5f*norm1[X])*vert_ptrs[0]->oow;
					vert_ptrs[0]->tmuvtx[0].tow = 256.0f*(0.5f + 0.5f*norm1[Y])*vert_ptrs[0]->oow;
				}
			}

			vert_ptrs[1]->x = vert_data1->xformed_vert.x;
			vert_ptrs[1]->y = vert_data1->xformed_vert.y;
			vert_ptrs[1]->oow = vert_data1->xformed_vert.oow;
			vert_ptrs[1]->r = vert_data1->xformed_vert.r;
			vert_ptrs[1]->g = vert_data1->xformed_vert.g;
			vert_ptrs[1]->b = vert_data1->xformed_vert.b;
			vert_ptrs[1]->a = vert_data1->xformed_vert.a + alpha;

			if (gNumTMUsUsed == 1 && (mode & TORUS_FIRST_PASS))
			{
				vert_ptrs[1]->tmuvtx[0].sow = (s+ds)*vert_ptrs[1]->oow;
				vert_ptrs[1]->tmuvtx[0].tow = (t+dt)*vert_ptrs[1]->oow;
			}
			else
			{
				if (gNumTMUsUsed >= 2)
				{
					if (mode & (TORUS_BUMPMAP | TORUS_ENVMAP))
					{
						vert_ptrs[1]->tmuvtx[1].sow = (s+ds)*vert_ptrs[1]->oow;
						vert_ptrs[1]->tmuvtx[1].tow = (t+dt)*vert_ptrs[1]->oow;
					}
					else
					{
						vert_ptrs[1]->tmuvtx[0].sow = (s+ds)*vert_ptrs[1]->oow;
						vert_ptrs[1]->tmuvtx[0].tow = (t+dt)*vert_ptrs[1]->oow;
					}
				}
				if (mode & TORUS_BUMPMAP)
				{
					// shifted bumpmap
					MatMultVec3x3_2(offset, vert_data1->xy_to_st_mat, gXformedLightDir);
					vert_ptrs[1]->tmuvtx[0].sow = (s+ds + offset[0]*depth)*vert_ptrs[1]->oow;
					vert_ptrs[1]->tmuvtx[0].tow = (t+dt - offset[1]*depth)*vert_ptrs[1]->oow;
				}
				else if (mode & TORUS_ENVMAP)
				{
					// envmap pass
					MatMultVec3x3_2(norm2, gObjectCrystalBall.rot_mat, vert_data1->norm);
					vert_ptrs[1]->tmuvtx[0].sow = 256.0f*(0.5f + 0.5f*norm2[X])*vert_ptrs[1]->oow;
					vert_ptrs[1]->tmuvtx[0].tow = 256.0f*(0.5f + 0.5f*norm2[Y])*vert_ptrs[1]->oow;
				}
			}

			// swap verts
			tmp_vert = vert_ptrs[0];
			vert_ptrs[0] = vert_ptrs[2];
			vert_ptrs[2] = tmp_vert;
			tmp_vert = vert_ptrs[1];
			vert_ptrs[1] = vert_ptrs[3];
			vert_ptrs[3] = tmp_vert;

			// if we've already setup the previous pair of vertices then draw the pair of triangles
			if (j != n2)
			{
				if (gWireframe)
				{
					grDrawLine(vert_ptrs[0], vert_ptrs[1]);
					grDrawLine(vert_ptrs[0], vert_ptrs[2]);
					grDrawLine(vert_ptrs[1], vert_ptrs[2]);
				}
				// backface cull
				else if (NormZ((float *)vert_ptrs[3], (float *)vert_ptrs[2], (float *)vert_ptrs[0]) >= 0.0f)
				{
#ifdef USE_GLIDE3
					grDrawVertexArray(GR_TRIANGLE_STRIP, 4, vert_ptrs);
#else
					grDrawTriangle(vert_ptrs[0], vert_ptrs[1], vert_ptrs[2]);
					grDrawTriangle(vert_ptrs[1], vert_ptrs[3], vert_ptrs[2]);
#endif // USE_GLIDE3
				}

				// draw normals
				if (gDrawNormals)
				{
					GrVertex v_norm;
					Vector pt;

					pt[X] = vert_data0->vert[X] + 0.125f*vert_data0->norm[X];
					pt[Y] = vert_data0->vert[Y] + 0.125f*vert_data0->norm[Y];
					pt[Z] = vert_data0->vert[Z] + 0.125f*vert_data0->norm[Z];
					pt[W] = 1.0f;
					XformVertex(&v_norm, &pt);
#ifndef USE_GLIDE3
					v_norm.x = SNAP(v_norm.x);
					v_norm.y = SNAP(v_norm.y);
#endif // USE_GLIDE3
					v_norm.r = 255.0f; v_norm.g = 0.0f; v_norm.b = 0.0f; v_norm.a = 255.0f;
					v_norm.tmuvtx[0].sow = 0.0f;
					v_norm.tmuvtx[0].tow = 0.0f;
					grDrawLine(vert_ptrs[2], &v_norm);
				}
			}

			vert_data0++;
			vert_data1++;
		}
	}

	return ((n1*n2)<<1);
}

// n and u must be normalized vectors
// m will be filled with a rotation matrix that
// rotates n to align it with the z-axis
// and u will be aligned with the x-axis
static void WorldToTextureSpaceMat(Matrix m, Vector n, Vector u)
{
	Vector v;

	CrossProduct(v, n, u);

	// create the rotation matrix that maps uvn to xyz
	m[0][0] = u[X]; m[0][1] = u[Y]; m[0][2] = u[Z]; m[0][3] = 0.0f;
	m[1][0] = v[X]; m[1][1] = v[Y]; m[1][2] = v[Z]; m[1][3] = 0.0f;
	m[2][0] = n[X]; m[2][1] = n[Y]; m[2][2] = n[Z]; m[2][3] = 0.0f;
	m[3][0] = 0.0f; m[3][1] = 0.0f; m[3][2] = 0.0f; m[3][3] = 1.0f;
}
