/*****************************************************************************/
/*  misc.c - general purpose routines                                        */
/*  Copyright (C) 1998-2001 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "gftp.h"
#include "options.h"

char *
insert_commas (unsigned long number, char *dest_str, size_t dest_len)
{
  char *frompos, *topos, src[50], *dest;
  unsigned long tempnum;
  int len, num, rem, i;

  len = 1;
  tempnum = number;
  while (tempnum > 0)
    {
      tempnum /= 10;
      len++;
    }

  if (len % 3 > 0)
    len++;

  len += len / 3;
  if (len <= 0) 
    {
      if (dest_str != NULL)
        {
          strncpy (dest_str, "0", dest_len);
          return (dest_str);
        }
      return (NULL);
    }

  if (dest_str != NULL && len > dest_len)
    {
      *dest_str = '\0';
      return (dest_str);
    }

  if (dest_str == NULL)
    dest = g_malloc (len);
  else
    dest = dest_str;

  g_snprintf (src, sizeof (src), "%ld", number);

  num = strlen (src) / 3 - 1;
  rem = strlen (src) % 3;
  frompos = src;
  topos = dest;
  for (i = 0; i < rem; i++)
    *topos++ = *frompos++;

  if (*frompos != '\0')
    {
      if (rem != 0)
	*topos++ = ',';
      while (num > 0)
	{
	  for (i = 0; i < 3; i++)
	    *topos++ = *frompos++;
	  *topos++ = ',';
	  num--;
	}
      for (i = 0; i < 3; i++)
	*topos++ = *frompos++;
    }
  *topos = '\0';
  return (dest);
}


long
file_countlf (int filefd, long endpos)
{
  char tempstr[255];
  long num, mypos;
  ssize_t n;
  int i;

  mypos = num = 0;
  lseek (filefd, 0, SEEK_SET);
  while ((n = read (filefd, tempstr, sizeof (tempstr))) > 0)
    {
      for (i = 0; i < n; i++)
	{
	  if ((tempstr[i] == '\n') && (i > 0) && (tempstr[i - 1] != '\r')
	      && (endpos == 0 || mypos + i <= endpos))
	    ++num;
	}
      mypos += n;
    }
  lseek (filefd, 0, SEEK_SET);
  return (num);
}


char *
alltrim (char *str)
{
  char *pos, *newpos;
  int diff;

  pos = str + strlen (str) - 1;
  while (pos >= str && *pos == ' ')
    *pos-- = '\0';

  pos = str;
  diff = 0;
  while (*pos++ == ' ')
    diff++;

  pos = str + diff;
  newpos = str;
  while (*pos != '\0')
    *newpos++ = *pos++;
  *newpos = '\0';
  return (str);
}


char *
expand_path (char *src)
{
  char *str, *pos, *endpos, *prevpos, *newstr, *tempstr, tempchar;
  struct passwd *pw;

  pw = NULL;
  str = g_malloc (strlen (src) + 1);
  strcpy (str, src);

  if (*str == '~')
    {
      if (*(str + 1) == '/' || (pos = strchr (str, '/')) == NULL)
	pw = getpwuid (geteuid ());
      else
	{
	  *pos = '\0';
	  pw = getpwnam (str + 1);
	  *pos = '/';
	}
    }

  endpos = str;
  newstr = NULL;
  while ((pos = strchr (endpos, '/')) != NULL)
    {
      pos++;
      while (*pos == '/')
        pos++;
      if ((endpos = strchr (pos, '/')) == NULL)
	endpos = strchr (pos, '\0');
      tempchar = *endpos;
      *endpos = '\0';
      if (strcmp (pos, "..") == 0)
	{
	  *(pos - 1) = '\0';
	  if (newstr != NULL && (prevpos = strrchr (newstr, '/')) != NULL)
	    *prevpos = '\0';
	}
      else if (strcmp (pos, ".") != 0)
	{
	  if (newstr == NULL)
	    {
	      newstr = g_malloc (strlen (pos - 1) + 1);
	      strcpy (newstr, pos - 1);
	    }
	  else
	    {
	      tempstr = g_strconcat (newstr, pos - 1, NULL);
	      g_free (newstr);
	      newstr = tempstr;
	    }
	}
      *endpos = tempchar;
      if (*endpos == '\0')
	break;
      endpos = pos + 1;
    }

  if (newstr == NULL || *newstr == '\0')
    {
      if (newstr != NULL)
	g_free (newstr);
      newstr = g_malloc (2);
      *newstr = '/';
      *(newstr + 1) = '\0';
    }

  if (newstr != NULL)
    {
      g_free (str);
      str = newstr;
    }

  if (pw != NULL)
    {
      if ((pos = strchr (str, '/')) == NULL)
	{
	  newstr = g_malloc (strlen (pw->pw_dir) + 1);
	  strcpy (newstr, pw->pw_dir);
	}
      else
	newstr = g_strconcat (pw->pw_dir, pos, NULL);
      if (str)
	g_free (str);
      return (newstr);
    }
  else
    {
      newstr = g_malloc (strlen (str) + 1);
      strcpy (newstr, str);
    }
  g_free (str);
  return (newstr);
}


void
remove_double_slashes (char *string)
{
  char *newpos, *oldpos;

  oldpos = newpos = string;
  while (*oldpos != '\0')
    {
      *newpos++ = *oldpos++;
      if (*oldpos == '\0')
	break;
      while (*(newpos - 1) == '/' && *(oldpos) == '/')
	oldpos++;
    }
  *newpos = '\0';
  if (string[strlen (string) - 1] == '/')
    string[strlen (string) - 1] = '\0';
}


void
make_nonnull (char **str)
{
  if (*str == NULL)
    {
      *str = g_malloc (1);
      **str = '\0';
    }
}


int
copyfile (char *source, char *dest)
{
  FILE *srcfd, *destfd;
  char buf[8192];
  size_t n;

  if ((srcfd = fopen (source, "rb")) == NULL)
    return (0);

  if ((destfd = fopen (dest, "wb")) == NULL)
    {
      fclose (srcfd);
      return (0);
    }

  while ((n = fread (buf, 1, sizeof (buf), srcfd)) > 0)
    fwrite (buf, 1, n, destfd);

  fclose (srcfd);
  fclose (destfd);
  return (1);
}


int
gftp_match_filespec (char *filename, char *filespec)
{
  char *filepos, *wcpos, *pos, *newpos, search_str[20];
  size_t len, curlen;
  
  if (filename == NULL || *filename == '\0' || 
      filespec == NULL || *filespec == '\0') 
    return(1);

  filepos = filename;
  wcpos = filespec;
  while(1) 
    {
      if (*wcpos == '\0') 
        return (1);
      else if (*filepos == '\0') 
        return(0);
      else if(*wcpos == '?') 
        {
          wcpos++;
          filepos++;
        }
      else if(*wcpos == '*' && *(wcpos+1) == '\0') 
        return(1);
      else if(*wcpos == '*') 
        {
          len = sizeof (search_str);
          for (pos = wcpos + 1, newpos = search_str, curlen = 0;
               *pos != '*' && *pos != '?' && *pos != '\0' && curlen < len;
               curlen++, *newpos++ = *pos++);
          *newpos = '\0';

          if ((filepos = strstr (filepos, search_str)) == NULL)
            return(0);
          wcpos += curlen + 1;
          filepos += curlen;
        }
      else if(*wcpos++ != *filepos++) 
        return(0);
    }
  return (1);
}


int
gftp_parse_command_line (int *argc, char ***argv)
{
  if (*argc > 1)
    {
      if (strcmp (**argv, "--help") == 0 || strcmp (**argv, "-h") == 0)
	gftp_usage ();
      else if (strcmp (**argv, "--version") == 0 || strcmp (**argv, "-v") == 0)
	{
	  printf ("%s\n", version);
	  exit (0);
	}
    }
  return (0);
}


void
gftp_usage (void)
{
  printf (_("usage: gftp [[ftp://][user:[pass]@]ftp-site[:port][/directory]]\n"));
  exit (0);
}


char *
get_xpm_path (char *filename, int quit_on_err)
{
  char *tempstr, *exfile;

  tempstr = g_strconcat (BASE_CONF_DIR, "/", filename, NULL);
  exfile = expand_path (tempstr);
  g_free (tempstr);
  if (access (exfile, F_OK) != 0)
    {
      g_free (exfile);
      tempstr = g_strconcat (SHARE_DIR, "/", filename, NULL);
      exfile = expand_path (tempstr);
      g_free (tempstr);
      if (access (exfile, F_OK) != 0)
	{
	  g_free (exfile);
	  exfile = g_strconcat ("/usr/share/icons/", filename, NULL);
	  if (access (exfile, F_OK) != 0)
	    {
	      g_free (exfile);
	      if (!quit_on_err)
		return (NULL);
	      printf (_("gFTP Error: Cannot find file %s in %s or %s\n"),
		      filename, SHARE_DIR, BASE_CONF_DIR);
	      exit (-1);
	    }
	}
    }
  return (exfile);
}


gint
string_hash_compare (gconstpointer path1, gconstpointer path2)
{
  return (strcmp ((char *) path1, (char *) path2) == 0);
}


guint
string_hash_function (gconstpointer key)
{
  return (((char *) key)[0] + ((char *) key)[1] + ((char *) key)[2]);
}


void
free_file_list (GList * filelist)
{
  gftp_file * tempfle;
  GList * templist;

  templist = filelist;
  while (templist != NULL)
    {
      tempfle = templist->data;
      free_fdata (tempfle);
      templist = templist->next;
    }
  g_list_free (filelist);
}


void
free_fdata (gftp_file * fle)
{
  if (fle->file)
    g_free (fle->file);
  if (fle->user)
    g_free (fle->user);
  if (fle->group)
    g_free (fle->group);
  if (fle->attribs)
    g_free (fle->attribs);
  if (fle->destfile)
    g_free (fle->destfile);
  g_free (fle);
}


gftp_file *
copy_fdata (gftp_file * fle)
{
  gftp_file * newfle;

  newfle = g_malloc0 (sizeof (*newfle));
  memcpy (newfle, fle, sizeof (*newfle));

  if (fle->file)
    {
      newfle->file = g_malloc (strlen (fle->file) + 1);
      strcpy (newfle->file, fle->file);
    }

  if (fle->user)
    {
      newfle->user = g_malloc (strlen (fle->user) + 1);
      strcpy (newfle->user, fle->user);
    }

  if (fle->group)
    {
      newfle->group = g_malloc (strlen (fle->group) + 1);
      strcpy (newfle->group, fle->group);
    }

  if (fle->attribs)
    {
      newfle->attribs = g_malloc (strlen (fle->attribs) + 1);
      strcpy (newfle->attribs, fle->attribs);
    }

  if (fle->destfile)
    {
      newfle->destfile = g_malloc (strlen (fle->destfile) + 1);
      strcpy (newfle->destfile, fle->destfile);
    }
  return (newfle);
}


void
swap_socks (gftp_request * dest, gftp_request * source)
{
  dest->sockfd = source->sockfd;
  dest->datafd = source->datafd;
  dest->sockfd_write = source->sockfd_write;
  dest->cached = 0;
  if (!source->always_connected)
    {
      source->sockfd = NULL;
      source->datafd = NULL;
      source->sockfd_write = NULL;
      source->cached = 1;
    }
}


int
compare_request (gftp_request * request1, gftp_request * request2,
                 int compare_dirs)
{
  char *strarr[3][2];
  int i, ret;

  ret = 1;
  if (strcmp (request1->protocol_name, request2->protocol_name) == 0 &&
      request1->port == request2->port)
    {
      strarr[0][0] = request1->hostname;
      strarr[0][1] = request2->hostname;
      strarr[1][0] = request1->username;
      strarr[1][1] = request2->username;
      if (compare_dirs)
        {
          strarr[2][0] = request1->directory;
          strarr[2][1] = request2->directory;
        }
      else
        strarr[2][0] = strarr[2][1] = "";

      for (i = 0; i < 3; i++)
        {
          if ((strarr[i][0] && !strarr[i][1])
              || (!strarr[i][0] && strarr[i][1]))
            {
              ret = 0;
              break;
            }

          if (strarr[i][0] && strarr[i][1]
              && strcmp (strarr[i][0], strarr[i][1]) != 0)
            {
              ret = 0;
              break;
            }
        }
    }
  else
    ret = 0;
  return (ret);
}


void
free_tdata (gftp_transfer * tdata)
{
  if (tdata->mutex)
    g_free (tdata->mutex);
  if (tdata->fromreq != NULL)
    gftp_request_destroy (tdata->fromreq);
  if (tdata->toreq != NULL)
    gftp_request_destroy (tdata->toreq);
  free_file_list (tdata->files);
  g_free (tdata);
}


gftp_request * 
copy_request (gftp_request * req)
{
  gftp_request * newreq;

  newreq = g_malloc0 (sizeof (*newreq));
  memcpy (newreq, req, sizeof (*newreq));

  if (req->hostname)
    newreq->hostname = g_strconcat (req->hostname, NULL);
  if (req->username)
    newreq->username = g_strconcat (req->username, NULL);
  if (req->password)
    newreq->password = g_strconcat (req->password, NULL);
  if (req->account)
    newreq->account = g_strconcat (req->account, NULL);
  if (req->directory)
    newreq->directory = g_strconcat (req->directory, NULL);
  if (req->proxy_config)
    newreq->proxy_config = g_strconcat (req->proxy_config, NULL);
  if (req->proxy_hostname)
    newreq->proxy_hostname = g_strconcat (req->proxy_hostname, NULL);
  if (req->proxy_username)
    newreq->proxy_username = g_strconcat (req->proxy_username, NULL);
  if (req->proxy_password)
    newreq->proxy_password = g_strconcat (req->proxy_password, NULL);
  if (req->proxy_account)
    newreq->proxy_account = g_strconcat (req->proxy_account, NULL);
  if (req->url_prefix)
    newreq->url_prefix = g_strconcat (req->url_prefix, NULL);
  if (req->protocol_name)
    newreq->protocol_name = g_strconcat (req->protocol_name, NULL);

  newreq->last_ftp_response = NULL;
  newreq->last_dir_entry = NULL;
  newreq->sockfd = NULL;
  newreq->sockfd_write = NULL;
  newreq->datafd = NULL;
  newreq->cachefd = NULL;
  newreq->hostp = NULL;
  newreq->protocol_data = NULL;
  req->init (newreq);
  return (newreq);
}


void
sig_quit (int signo)
{
  signal (signo, sig_quit);
  siglongjmp (envir, signo == SIGINT ? 1 : 2);
}


char *
make_temp_filename (char *ext)
{
  char *destbuf;
 
  srand (time (NULL));
  destbuf = NULL;
  do
    {
      if (destbuf)
        g_free (destbuf);
      destbuf = g_strdup_printf ("%s/temp%ld%s", tmp_directory,
                         1 + (long) (99999999.0 * rand () / (RAND_MAX + 1.0)),
                         ext == NULL ? "" : ext);
    }
  while (access (destbuf, F_OK) != -1);

  return (destbuf);
}

