/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "embed.h"
#include "misc.h"
#include "bookmarks.h"
#include "history.h"
#include "window.h"
#include "mozilla.h"
#include "context.h"

#include <gtkmozembed.h>
#include <gtk/gtkmain.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-app-helper.h>

/* defines */
#define CONTEXT_MENU_ITEM(label, stock_id, action) \
 { GNOME_APP_UI_ITEM, label, NULL, NULL, GINT_TO_POINTER(action), NULL, \
 GNOME_APP_PIXMAP_STOCK, stock_id, 0, 0, NULL}
#define CONTEXT_MENU_TOGGLEITEM(label, stock_id, action) \
 { GNOME_APP_UI_TOGGLEITEM, label, NULL, NULL, GINT_TO_POINTER(action), NULL, \
 GNOME_APP_PIXMAP_STOCK, stock_id, 0, 0, NULL}

/* common menus */

/* are these still needed? do we need to make them all like this?*/
static char BACK_STR[]          = N_("Back");
static char FORWARD_STR[]       = N_("Forward");
static char UP_STR[]            = N_("Up");
static char RELOAD_STR[]        = N_("Reload");
static char OPEN_NEW_WIN_STR[]  = N_("Open in a new window");
static char OPEN_NEW_TAB_STR[]  = N_("Open in a new tab");
static char DOWNLOAD_FILE_EXTERNAL_STR[] = N_("Download link with External Handler");
static char DOWNLOAD_FILE_BUILT_IN_STR[] = N_("Download link with Built-in Handler");
static char COPY_LINK_LOC_STR[] = N_("Copy link location");
static char ADD_BM_STR[]        = N_("Add bookmark");
static char OPEN_IMG_STR[]      = N_("Open image");
static char OPEN_IMG_WIN_STR[]  = N_("Open image in a new window");
static char OPEN_IMG_TAB_STR[]  = N_("Open image in a new tab");
static char SAVE_IMG_STR[]      = N_("Save image");
static char SAVE_IMG_AS_STR[]   = N_("Save image as...");
static char USE_IMG_AS_BG_STR[] = N_("Use image as background");
static char COPY_IMG_LOC_STR[]  = N_("Copy image location");
static char BLOCK_IMG_SRC_STR[] = N_("Block image source");
static char OPEN_FRAME_WIN_STR[]     = N_("Open frame in window");
static char OPEN_FRAME_NEW_WIN_STR[] = N_("Open frame in new window");
static char OPEN_FRAME_NEW_TAB_STR[] = N_("Open frame in new tab");
static char RELOAD_FRAME_STR[]       = N_("Reload frame");
static char COPY_FRAME_LOC_STR[]     = N_("Copy frame location");
static char ADD_BM_FRAME_STR[]       = N_("Add bookmark for frame");
static char VIEW_SOURCE_STR[]        = N_("View source");
static char SAVE_AS_STR[]            = N_("Save As...");
static char FULLSCREEN_VIEW_STR[]    = N_("Full Screen View");

#define NAVIGATION_CONTEXT_MENU \
  CONTEXT_MENU_ITEM(BACK_STR,    GNOME_STOCK_MENU_BACK,    GALEON_CONTEXT_BACK),    \
  CONTEXT_MENU_ITEM(FORWARD_STR, GNOME_STOCK_MENU_FORWARD, GALEON_CONTEXT_FORWARD), \
  CONTEXT_MENU_ITEM(UP_STR,      GNOME_STOCK_MENU_UP,      GALEON_CONTEXT_UP),       \
  CONTEXT_MENU_ITEM(RELOAD_STR,  GNOME_STOCK_MENU_REFRESH, GALEON_CONTEXT_RELOAD),
#define LINK_CONTEXT_MENU \
  CONTEXT_MENU_ITEM(OPEN_NEW_WIN_STR,  GNOME_STOCK_MENU_NEW,   GALEON_CONTEXT_OPENWINDOW), \
  CONTEXT_MENU_ITEM(OPEN_NEW_TAB_STR,  GNOME_STOCK_MENU_NEW,   GALEON_CONTEXT_OPENTAB),    \
  CONTEXT_MENU_ITEM(DOWNLOAD_FILE_BUILT_IN_STR, GNOME_STOCK_MENU_SAVE,  GALEON_CONTEXT_DL_INT), \
  CONTEXT_MENU_ITEM(DOWNLOAD_FILE_EXTERNAL_STR, GNOME_STOCK_MENU_SAVE,  GALEON_CONTEXT_DL_EXT), \
  CONTEXT_MENU_ITEM(COPY_LINK_LOC_STR, GNOME_STOCK_MENU_COPY,  GALEON_CONTEXT_COPYLOC),    \
  CONTEXT_MENU_ITEM(ADD_BM_STR,        GNOME_STOCK_MENU_BLANK, GALEON_CONTEXT_ADDTEMPBMLINK),
#define IMAGE_CONTEXT_MENU \
  CONTEXT_MENU_ITEM(OPEN_IMG_STR,      GNOME_STOCK_MENU_OPEN,    GALEON_CONTEXT_OPENIMAGE),       \
  CONTEXT_MENU_ITEM(OPEN_IMG_WIN_STR,  GNOME_STOCK_MENU_NEW,     GALEON_CONTEXT_OPENIMAGEWINDOW), \
  CONTEXT_MENU_ITEM(OPEN_IMG_TAB_STR,  GNOME_STOCK_MENU_NEW,     GALEON_CONTEXT_OPENIMAGETAB),    \
  CONTEXT_MENU_ITEM(SAVE_IMG_STR,      GNOME_STOCK_MENU_SAVE,    GALEON_CONTEXT_SAVEIMAGE),       \
  CONTEXT_MENU_ITEM(SAVE_IMG_AS_STR,   GNOME_STOCK_MENU_SAVE_AS, GALEON_CONTEXT_SAVEIMAGEAS),     \
  CONTEXT_MENU_ITEM(USE_IMG_AS_BG_STR, GNOME_STOCK_MENU_BLANK,   GALEON_CONTEXT_USEIMAGEASBG),    \
  CONTEXT_MENU_ITEM(COPY_IMG_LOC_STR,  GNOME_STOCK_MENU_COPY,    GALEON_CONTEXT_COPYIMAGELOC),    \
  CONTEXT_MENU_ITEM(BLOCK_IMG_SRC_STR, GNOME_STOCK_MENU_BLANK,   GALEON_CONTEXT_BLOCKIMAGESOURCE),
#define FRAME_CONTEXT_MENU \
  CONTEXT_MENU_ITEM(OPEN_FRAME_WIN_STR,     GNOME_STOCK_MENU_OPEN,    GALEON_CONTEXT_OPENFRAME),       \
  CONTEXT_MENU_ITEM(OPEN_FRAME_NEW_WIN_STR, GNOME_STOCK_MENU_OPEN,    GALEON_CONTEXT_OPENFRAMEWINDOW), \
  CONTEXT_MENU_ITEM(OPEN_FRAME_NEW_TAB_STR, GNOME_STOCK_MENU_OPEN,    GALEON_CONTEXT_OPENFRAMETAB),    \
  CONTEXT_MENU_ITEM(RELOAD_FRAME_STR,       GNOME_STOCK_MENU_REFRESH, GALEON_CONTEXT_RELOADFRAME),     \
  CONTEXT_MENU_ITEM(COPY_FRAME_LOC_STR,     GNOME_STOCK_MENU_COPY,    GALEON_CONTEXT_COPYFRAMELOC),    \
  CONTEXT_MENU_ITEM(ADD_BM_FRAME_STR,       GNOME_STOCK_MENU_BLANK,   GALEON_CONTEXT_ADDTEMPBMFRAME),
#define SHORT_CONTEXT_MENU \
  CONTEXT_MENU_ITEM(SAVE_AS_STR,               GNOME_STOCK_MENU_SAVE_AS, GALEON_CONTEXT_SAVEAS),     \
  CONTEXT_MENU_ITEM(VIEW_SOURCE_STR,           GNOME_STOCK_MENU_BLANK,   GALEON_CONTEXT_VIEWSOURCE),     \
  CONTEXT_MENU_TOGGLEITEM(FULLSCREEN_VIEW_STR, GNOME_STOCK_MENU_BLANK,   GALEON_CONTEXT_FULLSCREEN), \
  CONTEXT_MENU_ITEM(ADD_BM_STR,                GNOME_STOCK_MENU_BLANK,   GALEON_CONTEXT_ADDTEMPBM),

/* local data structures */
typedef enum
{
	GALEON_CONTEXT_NONE,
	GALEON_CONTEXT_BACK,
	GALEON_CONTEXT_FORWARD,
	GALEON_CONTEXT_UP,
	GALEON_CONTEXT_RELOAD,
	GALEON_CONTEXT_SAVEAS,
	GALEON_CONTEXT_FULLSCREEN,
	GALEON_CONTEXT_ADDTEMPBM,
	GALEON_CONTEXT_OPENWINDOW,
	GALEON_CONTEXT_OPENTAB,
	GALEON_CONTEXT_DL_EXT,
	GALEON_CONTEXT_DL_INT,
	GALEON_CONTEXT_COPYLOC,
	GALEON_CONTEXT_ADDTEMPBMLINK,
	GALEON_CONTEXT_OPENIMAGE,
	GALEON_CONTEXT_OPENIMAGEWINDOW,
	GALEON_CONTEXT_OPENIMAGETAB,
	GALEON_CONTEXT_SAVEIMAGE,
	GALEON_CONTEXT_SAVEIMAGEAS,
	GALEON_CONTEXT_USEIMAGEASBG,
	GALEON_CONTEXT_COPYIMAGELOC,
	GALEON_CONTEXT_BLOCKIMAGESOURCE,
	GALEON_CONTEXT_OPENFRAME,
	GALEON_CONTEXT_OPENFRAMEWINDOW,
	GALEON_CONTEXT_OPENFRAMETAB,
	GALEON_CONTEXT_RELOADFRAME,
	GALEON_CONTEXT_COPYFRAMELOC,
	GALEON_CONTEXT_ADDTEMPBMFRAME,
	GALEON_CONTEXT_VIEWSOURCE	
} GaleonContextAction;

typedef enum
{
	GALEON_CONTEXT_LINK = CONTEXT_LINK | CONTEXT_OTHER,
	GALEON_CONTEXT_IMAGELINK = CONTEXT_LINK | CONTEXT_IMAGE | CONTEXT_OTHER,
	GALEON_CONTEXT_DOC = CONTEXT_DOCUMENT | CONTEXT_OTHER,
	GALEON_CONTEXT_IMAGE = CONTEXT_IMAGE | CONTEXT_DOCUMENT | CONTEXT_OTHER,
	GALEON_CONTEXT_INPUT = CONTEXT_DOCUMENT | CONTEXT_INPUT | CONTEXT_OTHER
} GaleonContextEvent;

/* local functions */
static GnomeUIInfo *context_make_menu 	     (WrapperContextInfo *info);
static guint context_show_context_menu 	     (GaleonEmbed *embed, 
					      GnomeUIInfo *menu);
static void context_do_action 		     (GaleonContextAction action, 
					      WrapperContextInfo *info,
					      GaleonEmbed *embed);
static void context_set_navigation_buttons   (GaleonEmbed *embed, 
					      GnomeUIInfo *uiinfo,
					      GtkWidget *popup);
static void context_set_full_screen_check    (GaleonWindow *window,
					      GnomeUIInfo *menu,
					      GtkWidget *popup);
static void context_menu_add_bookmarks (GtkMenu *menu, GaleonEmbed *embed);
static BookmarkItem *context_menu_create_fake_bookmarks_root (void);
static void context_menu_create_fake_bookmarks_root_recursively 
						(BookmarkItem *b, GList **l);
static void context_menu_bookmarks_cleanup (GtkObject *o, GaleonWindow *window);

/**
 * context_set_full_screen_check: searches for a check button and sets it
 **/
static void context_set_full_screen_check (GaleonWindow *window,
					   GnomeUIInfo *menu,
					   GtkWidget *popup)
{
	gint i;

	return_if_not_window (window);
	if (!window->menuBarOn)
	{
		return;
	}

	for (i = 0; menu[i].type != GNOME_APP_UI_ENDOFINFO; i++)
	{
		if (GPOINTER_TO_INT (menu[i].user_data) == 
		    GALEON_CONTEXT_FULLSCREEN)
		{
			GtkCheckMenuItem *menu_view_fullscreen = 
				GTK_CHECK_MENU_ITEM (menu[i].widget);
			g_return_if_fail (menu_view_fullscreen != NULL); 
			gtk_check_menu_item_set_active (menu_view_fullscreen,
				(GTK_CHECK_MENU_ITEM 
					(window->view_fullscreen))->active);
		}
	}
}

/**
 * context_set_navigation_buttons: sets up back/forward/up buttons
 */
static void context_set_navigation_buttons (GaleonEmbed *embed, 
					    GnomeUIInfo *uiinfo,
					    GtkWidget *popup)
{
	gboolean can_go_back, can_go_forward, can_go_up;
	gboolean show_history_popup, show_up_button;
	GtkMenu *menu_back, *menu_forward, *menu_up;
	gint context_action, i;
	gint go_back_pos = 0, go_forward_pos = 1, go_up_pos = 2;
	
	show_up_button = 
	      gnome_config_get_bool ("/galeon/Appearance/show_up_button=FALSE");
	show_history_popup = 
	      gnome_config_get_bool ("/galeon/Mouse/history_popup=FALSE");
	can_go_back = gtk_moz_embed_can_go_back (GTK_MOZ_EMBED(embed->mozEmbed));
	can_go_forward = gtk_moz_embed_can_go_forward (GTK_MOZ_EMBED(embed->mozEmbed));
	can_go_up = embed_can_go_up (embed);
	
	for (i = 0; uiinfo[i].type != GNOME_APP_UI_ENDOFINFO; i++)
	{
		context_action = GPOINTER_TO_INT (uiinfo[i].user_data);

		if (context_action == GALEON_CONTEXT_BACK)
		{
			go_back_pos    = i;
			go_forward_pos = i + 1;
			go_up_pos      = i + 2;

			break;
		}
	}

	gtk_widget_set_sensitive(uiinfo[go_back_pos].widget,
				 can_go_back);
	gtk_widget_set_sensitive(uiinfo[go_forward_pos].widget,
		 		 can_go_forward);
	if (show_history_popup)
	{
		if (can_go_back)
		{
			menu_back = embed_create_back_menu(embed);
			gtk_signal_connect(
				GTK_OBJECT(menu_back),
				"selection-done",
				(GtkSignalFunc)popup_selection_done_cb,
				popup);
			gnome_popup_menu_attach(
				GTK_WIDGET(menu_back),
				uiinfo[go_back_pos].widget,
				NULL);
		}      
		if (can_go_forward) {
			menu_forward = embed_create_forward_menu(embed);
			gtk_signal_connect(
				GTK_OBJECT(menu_forward),
				"selection-done",
				(GtkSignalFunc)popup_selection_done_cb,
				popup);
			gnome_popup_menu_attach(
				GTK_WIDGET(menu_forward),
				uiinfo[go_forward_pos].widget,
				NULL);
		}
		if (can_go_up && show_up_button) {
			menu_up = embed_create_up_menu (embed);
			gtk_signal_connect(
				GTK_OBJECT(menu_up),
				"selection-done",
				(GtkSignalFunc)popup_selection_done_cb,
				popup);
			gnome_popup_menu_attach (GTK_WIDGET(menu_up),
					 uiinfo[go_up_pos].widget, NULL);
		}
	}

	if (show_up_button)
		gtk_widget_set_sensitive (uiinfo[go_up_pos].widget,
						 can_go_up);
	else
		gtk_widget_hide (uiinfo[go_up_pos].widget);
}

/**
 * context_make_menu: helper func to create the correct menu
 **/
static GnomeUIInfo *context_make_menu (WrapperContextInfo *info)
{
	switch (info->context) /* there has to be a better way too do this, */
	{                      /* nonetheless it evaded me */
	case GALEON_CONTEXT_LINK:
	{
		GnomeUIInfo link_uiinfo[] =
		{
			LINK_CONTEXT_MENU
			GNOMEUIINFO_SEPARATOR,
			NAVIGATION_CONTEXT_MENU
			GNOMEUIINFO_END
		};
		return g_memdup (&link_uiinfo, sizeof(link_uiinfo));
	}
	case GALEON_CONTEXT_IMAGELINK:
	{
		GnomeUIInfo imagelink_uiinfo[] =
		{
			LINK_CONTEXT_MENU
			GNOMEUIINFO_SEPARATOR,
			IMAGE_CONTEXT_MENU
			GNOMEUIINFO_SEPARATOR,
			NAVIGATION_CONTEXT_MENU
			GNOMEUIINFO_END
		};
		return g_memdup (&imagelink_uiinfo, sizeof(imagelink_uiinfo));
	}
	case GALEON_CONTEXT_DOC:
	{
		if (info->framed_page)
		{
			GnomeUIInfo framed_uiinfo[] =
			{
				NAVIGATION_CONTEXT_MENU
				GNOMEUIINFO_SEPARATOR,
				SHORT_CONTEXT_MENU
				GNOMEUIINFO_SEPARATOR,
				FRAME_CONTEXT_MENU
				GNOMEUIINFO_END
			};
			
			return g_memdup (&framed_uiinfo, sizeof(framed_uiinfo));
		}
		else
		{
			GnomeUIInfo doc_uiinfo[] =
			{
				NAVIGATION_CONTEXT_MENU
				GNOMEUIINFO_SEPARATOR,
				SHORT_CONTEXT_MENU
				GNOMEUIINFO_END
			};
			return g_memdup (&doc_uiinfo, sizeof(doc_uiinfo));
		}
		break;
	}
	case GALEON_CONTEXT_IMAGE:
	{
		GnomeUIInfo image_uiinfo[] =
		{
			NAVIGATION_CONTEXT_MENU
			GNOMEUIINFO_SEPARATOR,
			IMAGE_CONTEXT_MENU
			GNOMEUIINFO_END
		};
		return g_memdup (&image_uiinfo, sizeof(image_uiinfo));
	}
	case GALEON_CONTEXT_INPUT:
	{ /* FIXME: maybe make this context have cut/copy/paste etc. */
		GnomeUIInfo input_uiinfo[] =
		{
			NAVIGATION_CONTEXT_MENU
			GNOMEUIINFO_SEPARATOR,
			SHORT_CONTEXT_MENU
			GNOMEUIINFO_END
		};
		return g_memdup (&input_uiinfo, sizeof(input_uiinfo));
	}
	default:
		g_warning ("No menu data for that context\n");
		return NULL;
	}
}

/**
 * context_show_menu: show appropriate context menu based on the target type
 **/
static guint context_show_context_menu (GaleonEmbed *embed, 
					GnomeUIInfo *menu)
{
	guint ret;
	GtkWidget *popup;

	popup = gnome_popup_menu_new (menu);
	context_set_navigation_buttons (embed, menu, popup);
	context_set_full_screen_check (embed->parent_window, menu, popup);
	context_menu_add_bookmarks (GTK_MENU (popup), embed);
	ret = gnome_popup_menu_do_popup_modal (popup, NULL, NULL,
					       NULL , NULL);
	/* Wait for the menu to be destroyed if a menu item is selected */
	if (ret != -1) 
		while (gtk_events_pending()) 
			gtk_main_iteration();

	gtk_widget_destroy (popup);
	return ret;
}

/**
 * context_do_action: performs the acual action
 **/
static void context_do_action (GaleonContextAction action, 
			       WrapperContextInfo *info,
			       GaleonEmbed *embed)
{
	gchar *url;

	gboolean tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);

	switch (action)
	{
	case GALEON_CONTEXT_NONE:
		break;
	case GALEON_CONTEXT_BACK:
		gtk_moz_embed_go_back (GTK_MOZ_EMBED(embed->mozEmbed));
		break;
	case GALEON_CONTEXT_FORWARD:
		gtk_moz_embed_go_forward (GTK_MOZ_EMBED(embed->mozEmbed));
		break;
	case GALEON_CONTEXT_UP:
		embed_go_up (embed, 0);
		break;
	case GALEON_CONTEXT_RELOAD:
		embed_reload (embed);
		break;
	case GALEON_CONTEXT_SAVEAS:
		embed_save_document (embed, FALSE);
		break;
	case GALEON_CONTEXT_FULLSCREEN:
		window_toggle_fullscreen_mode (embed->parent_window);
		break;
	case GALEON_CONTEXT_ADDTEMPBM:
		add_bookmark_default (BM_SITE, embed->site_title, 
				      embed->site_location, NULL);
		break;
	case GALEON_CONTEXT_OPENWINDOW:
		embed_create_from_url (embed, info->link, FALSE, TRUE);
		break;
	case GALEON_CONTEXT_OPENTAB:
		embed_create_from_url (embed, info->link, FALSE, FALSE);
		break;
	case GALEON_CONTEXT_DL_INT:
		mozilla_save_link (embed, info->link);
		break;
	case GALEON_CONTEXT_DL_EXT:
		save_url (info->link);
		break;
	case GALEON_CONTEXT_COPYLOC:
		copy_to_clipboard (info->link, embed);
		break;
	case GALEON_CONTEXT_ADDTEMPBMLINK:
		add_temp_link_bookmark (info->linktext, info->link, NULL);
		break;
	case GALEON_CONTEXT_OPENIMAGE:
		embed_load_url(embed, info->img);
		break;
	case GALEON_CONTEXT_OPENIMAGEWINDOW:
		embed_create_from_url (embed, info->img, FALSE, TRUE);
		break;
	case GALEON_CONTEXT_OPENIMAGETAB:
		embed_create_from_url (embed, info->img, FALSE, FALSE);
		break;
	case GALEON_CONTEXT_SAVEIMAGE:
		embed_save_url_default (embed, info->img);
		break;
	case GALEON_CONTEXT_SAVEIMAGEAS:
		embed_save_url (embed, info->img);
		break;
	case GALEON_CONTEXT_USEIMAGEASBG:
		embed_set_image_as_background (embed, info->img);
		break;
	case GALEON_CONTEXT_COPYIMAGELOC:
		copy_to_clipboard (g_strdup (info->img), embed);
		break;
	case GALEON_CONTEXT_BLOCKIMAGESOURCE:
		mozilla_block_url (info->img);
		break;
	case GALEON_CONTEXT_OPENFRAME:
		embed_open_frame (embed, TRUE, FALSE);
		break;
	case GALEON_CONTEXT_OPENFRAMEWINDOW:
		embed_open_frame (embed, FALSE, TRUE);
		break;
	case GALEON_CONTEXT_OPENFRAMETAB:
		embed_open_frame (embed, FALSE, FALSE);
		break;
	case GALEON_CONTEXT_RELOADFRAME:
		mozilla_reload (embed);
		break;
	case GALEON_CONTEXT_COPYFRAMELOC:
		copy_to_clipboard (mozilla_get_document_url (embed), embed);
		break;
	case GALEON_CONTEXT_ADDTEMPBMFRAME:
		url = mozilla_get_document_url (embed);
		if (url)
		{
			add_bookmark_default (BM_SITE, embed->site_title, 
					      url, NULL);
			g_free (url);
		}
		break;
	case GALEON_CONTEXT_VIEWSOURCE:
		embed_view_source (embed, FALSE, !tabbed_mode);
		break;
	default:
		/* nothing happened, most likely the user did not end up
		   selecting an item from the popup */
		break;
	}
}

/**
 * context_show_menu: show appropriate context menu based on the target type
 **/
void context_show_menu (GaleonEmbed *embed, WrapperContextInfo *info)
{
	GnomeUIInfo *menu;

	return_if_not_embed (embed);
	menu = context_make_menu (info);
	
	if (menu != NULL)
	{
		guint result = context_show_context_menu (embed, menu);
		if (result != -1)
		{
			GaleonContextAction action = 
				GPOINTER_TO_INT(menu[result].user_data);
			context_do_action (action, info, embed);
		}
		g_free (menu);
	}
}

/**
 * context_show_bookmark_menu: show a bookmarks popup menu 
 **/
void context_show_bookmark_menu (GaleonEmbed *embed)
{
	GtkWidget *menu;
	GtkTooltips *tips;
	GaleonWindow *window;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);
	
	menu = gtk_menu_new ();
	tips = gtk_tooltips_new ();
	bookmarks_menu_create_recursively (bookmarks_root, GTK_MENU (menu),
					   NULL, tips, TRUE, TRUE);
	
	/* warning: ugly code */
	gtk_object_set_data (GTK_OBJECT (menu), "WMain", (window->WMain)); 
	gtk_object_set_data (GTK_OBJECT (window->WMain), "bm_context_menu",
			     menu);
	gnome_popup_menu_do_popup_modal (menu, NULL, NULL,
					 NULL , NULL);

	gtk_object_remove_data (GTK_OBJECT (window->WMain), "bm_context_menu");
	gtk_widget_destroy (menu);
	gtk_object_destroy (GTK_OBJECT (tips));
}

/**
 * context_menu_add_bookmarks: add the bookmarks that have the 
 * create_context_menu flag to a menu
 */
static void
context_menu_add_bookmarks (GtkMenu *menu, GaleonEmbed *embed)
{
	GtkTooltips *tips = gtk_tooltips_new ();
	GaleonWindow *window;
	BookmarkItem *fake_bookmarks_root = 
		context_menu_create_fake_bookmarks_root ();

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	gtk_menu_append (menu, gtk_menu_item_new ());

	bookmarks_menu_create_recursively (fake_bookmarks_root, 
					   menu, NULL, tips, FALSE, TRUE);
	
	bookmarks_free_bookmark (fake_bookmarks_root);
	
	/* warning: ugly code */
	gtk_object_set_data (GTK_OBJECT (menu), "WMain", (window->WMain)); 
	gtk_object_set_data (GTK_OBJECT (window->WMain), "bm_context_menu",
			     menu);

	gtk_signal_connect_object (GTK_OBJECT (menu), "destroy", 
				   gtk_object_destroy, GTK_OBJECT (tips));
	gtk_signal_connect (GTK_OBJECT (menu), "destroy", 
			    context_menu_bookmarks_cleanup, window);
}

/**
 * Do some cleanups before destroying the context menu
 */
static void
context_menu_bookmarks_cleanup (GtkObject *o, GaleonWindow *window)
{
	gtk_object_remove_data (GTK_OBJECT (window->WMain), "bm_context_menu");
}

/**
 * Create a fake bookmark category that haves all the bookmarks that need to be
 * addeed to context menus
 */
static BookmarkItem *
context_menu_create_fake_bookmarks_root (void)
{
	BookmarkItem *r = bookmarks_new_bookmark 
		(BM_CATEGORY, TRUE, "fake bookmark", NULL, NULL, NULL, NULL);
	context_menu_create_fake_bookmarks_root_recursively 
		(bookmarks_root, &r->list);
	return r;
}

static void
context_menu_create_fake_bookmarks_root_recursively (BookmarkItem *b, 
						     GList **l)
{
	GList *li;
	if (b->create_context_menu) 
		*l = g_list_append (*l, b);
	if (!b->alias_of)
		for (li = b->list; li; li = li->next)
			context_menu_create_fake_bookmarks_root_recursively
				(li->data, l);
}
