/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "embed.h"
#include "window.h"
#include "misc.h"
#include "history.h"
#include "prefs.h"
#include "mozilla_prefs.h"
#include "mozilla.h"
#include "mozcallbacks.h"
#include "bookmarks.h"
#include "mozilla_i18n.h"

#include <stdlib.h>
#include <string.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-exec.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>

extern const GtkTargetEntry drop_types[];

/* local function prototypes */
static GtkWidget *embed_notebook_tab_new (GaleonEmbed *embed,
					  const gchar *title);
static gchar *embed_get_save_file_name (gchar *url);
static void embed_notebook_tab_update_closebutton (GaleonEmbed *embed,
						   GtkWidget *tab);
static void embed_add_as_tab (GaleonEmbed *embed);
static void embed_switch_to_page (GaleonEmbed *embed);
static void embed_connect_signals (GaleonEmbed *embed);
static void embed_single_init (void);
static void embed_view_source_external (GaleonEmbed *embed, const gchar *url);
static void add_bf_menu (GtkWidget *menu, char* label, int index, 
			 GaleonEmbed *embed, int level);
static GtkWidget * new_num_accel_menu_item (gint num, gchar *origtext, 
					    gboolean lettersok, 
					    GtkWidget *menu);
static gint strcasestr_cb (gconstpointer a, gconstpointer b);

/** The global list of all GaleonEmbed structures */
GList *all_embeds = NULL;

/* global character set hash and sorted title list*/
GHashTable *charsets = NULL;
GList *sorted_charset_titles = NULL;

/* global last save dir */
gchar *save_dir = NULL;

/* tab close button global pixmap data */
PixmapData *close_pix = NULL;

/* global var to indicate whether xpcom has been initialised */
gboolean pushed_startup = FALSE;

/**
 * embed_create_from_url: create a browser from a given url string
 */
GaleonEmbed *
embed_create_from_url (GaleonEmbed *previous, const gchar *url, 
		       gboolean force_jump, gboolean new_window)
{
	GaleonEmbed *embed;

	/* check argument */
	g_assert(url != NULL);

	/* create a window */
	embed = embed_create (previous, new_window, force_jump, 
			      GTK_MOZ_EMBED_FLAG_ALLCHROME);

	/* show the window */
	embed_set_visibility (embed, TRUE);

	/* load the url */
	embed_load_url (embed, url);

	/* return completed embed */
	return embed;
}

/**
 * embed_create_default: create a new browser pointing at the page
 * specified by configuration
 */
GaleonEmbed *
embed_create_default (GaleonEmbed *previous_embed, gboolean new_window)
{
	GaleonEmbed *embed;
	gchar *url;

	/* get default URL */
	url = embed_get_default_url (previous_embed);

	/* load into a new window */
	embed = embed_create_from_url (previous_embed, url, TRUE, new_window);

	/* free and return */
	g_free (url);
	return embed;
}

/**
 * embed_create_from_url_view_source: create a browser from a given 
 * url string  in view source mode
 */
GaleonEmbed *
embed_create_from_url_view_source (GaleonEmbed *previous, const gchar *url,
				   gboolean new_window)
{
	GaleonEmbed *embed;
	gchar *vs_url;

	/* prepend view-source: protocol */
	vs_url = g_strconcat ("view-source:", url, NULL);

	/* get completed browser */
	embed = embed_create_from_url (previous, vs_url, TRUE, new_window);

	/* free allocated string */
	g_free (vs_url);
	
	/* return completed browser */
	return embed;
}

/**
 * embed_create: create a GaleonEmbed structure based on a given Embed
 * @previous can be NULL (orphan popups)
 */
GaleonEmbed *
embed_create (GaleonEmbed *previous, gboolean new_window, 
	      gboolean force_jump, guint32 chrome_mask)
{
	GaleonWindow *window = NULL;
	GaleonEmbed *embed;

	/* get the parent window if we're using it */
	if (!new_window && previous)
	{
		window = previous->parent_window;
	}

	/* if we haven't been passed a valid window, create one */
	if (window == NULL)
	{
		window = window_create (chrome_mask);
	}

	/* create the embed inside the window and return it */
	embed = embed_create_in_window (window, force_jump);

	/* return completed embed */
	return embed;
}

/**
 * embed_create_in_window: create a GaleonEmbed to go in a given GaleonWindow
 */
GaleonEmbed *
embed_create_in_window (GaleonWindow *window, gboolean force_jump)
{
	GaleonEmbed *embed;
	gboolean autojump;
	gboolean tabs;

	/* check window */
	return_val_if_not_window (window, NULL);

	/* build an embed structure */
	embed = g_new0 (GaleonEmbed, 1);

	/* set parent and store in parents list */
	embed->parent_window = window;
	window->embed_list = g_list_prepend (window->embed_list, embed);

	/* add to global list of all embeds */
	all_embeds = g_list_prepend (all_embeds, embed);

	/* not closing */
	embed->is_closing = FALSE;

	/* no content for the location or title */
	embed->site_location = NULL;
	embed->site_title = g_strdup (_("Untitled"));

	/* initial zoom level */
	embed->zoom = 100;
	
	/* set magic */
	embed->magic = GALEON_EMBED_MAGIC;

	/* make an embedding widget */
	embed->mozEmbed = gtk_moz_embed_new ();
	g_assert (GTK_IS_MOZ_EMBED (embed->mozEmbed));

	/* ref/unref widget -- is this really needed? */
	/* NB: apparently it is to get rid of some warnings, but I'm
	 * still not clear why -- MattA 13/4/2001 */
	/* as of 1/6/2001, it still doens't work... ! */
	gtk_widget_ref (GTK_WIDGET (embed->mozEmbed));
	gtk_object_set_data_full (GTK_OBJECT (embed->mozEmbed), "mozEmbed",
				  GTK_WIDGET (embed->mozEmbed),
				  (GtkDestroyNotify) gtk_widget_unref);

	/* setup data so we can always find GaleonEmbed */
	gtk_object_set_data (GTK_OBJECT (embed->mozEmbed), 
			     "GaleonEmbed", embed);

	/* connect appropriate signals */
	embed_connect_signals (embed);

	/* show tabs if more than one embed or if the user has asked for it */
	tabs = gnome_config_get_bool (CONF_APPEARANCE_TABBED_ALWAYS_SHOW);
	tabs = (tabs || (g_list_length (window->embed_list) > 1));
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (window->notebook), tabs);
	embed_add_as_tab (embed);

	/* switch to page if set in config */
	autojump = gnome_config_get_bool (CONF_APPEARANCE_TABBED_AUTOJUMP);
	if (autojump || force_jump)
	{
		embed_switch_to_page (embed);
		if (GTK_WIDGET (window->WMain)->window != NULL)
		{
			gdk_window_raise (GTK_WIDGET (window->WMain)->window);
		}
	}
	
	/* return completed structure */
	return embed;
}

/**
 * embed_get_default_url: get the default URL on new page creation
 */
gchar *
embed_get_default_url (GaleonEmbed *previous_embed)
{
	const gchar *last_page_url;
	gchar *home_page_url;
	gint page_type;

	/* find out where we're supposed to start */
	if (previous_embed == NULL)
	{
		page_type = gnome_config_get_int (CONF_GENERAL_HOMEPAGE_TYPE);
	}
	else
	{
		page_type = gnome_config_get_int (CONF_GENERAL_NEWPAGE_TYPE);
	}

       	/* return the appropriate page */
	if (page_type == STARTPAGE_HOME)
	{
		/* get location of home page */
		home_page_url = gnome_config_get_string(CONF_GENERAL_HOMEPAGE);

		/* load home page, if set */
		if (home_page_url != NULL)
		{
			return home_page_url;
		}
	}

	/* get location of last page: use previous browser if one,
	 * otherwise resort to fetching it from global history */
	if (previous_embed != NULL && previous_embed->site_location != NULL)
	{
		last_page_url = previous_embed->site_location;
	}
	else
	{
		last_page_url = history_get_last_url ();
	}
	
	if (page_type == STARTPAGE_LAST && last_page_url != NULL)
	{
		/* return last page pointed at */
		return g_strdup (last_page_url);
	}

	/* even in case of error, it's a good default */
	return g_strdup ("about:blank");
}

/**
 * embed_set_visibility: 
 */
void
embed_set_visibility (GaleonEmbed *embed, gboolean visibility)
{
	GaleonWindow *window;
	gint width, height;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* test for hiding */
	if (!visibility && embed->is_visible)
	{
		/* hiding */
		window->visible_embeds--;
		embed->is_visible = FALSE;
		if (window->visible_embeds == 0)
		{
			gtk_widget_hide (GTK_WIDGET (window->WMain));
		}
	}

	/* test for showing */
	if (visibility && !embed->is_visible)
	{
		/* showing */
		window->visible_embeds++;
		embed->is_visible = TRUE;
		gtk_widget_show (GTK_WIDGET (embed->mozEmbed));
		if (window->visible_embeds == 1)
		{
			/* find the set width and height if any */
			if (!window->set_size)
			{
				/* use system defaults */
				width = gnome_config_get_int 
					(CONF_APPEARANCE_WINWIDTH);
				height = gnome_config_get_int 
					(CONF_APPEARANCE_WINHEIGHT);
				gtk_window_set_default_size 
					(GTK_WINDOW (window->WMain), 
					 width, height);
			}
			gtk_widget_show (GTK_WIDGET (window->WMain));
		}
	}
}

/**
 * embed_load_url: interpret and load a URL into a given GaleonEmbed
 */
void
embed_load_url (GaleonEmbed *embed, const gchar *url)
{
	/* check arguments */
	return_if_not_embed (embed);
	g_assert(url != NULL);

	/* error if we can't handle this URL */
	if (handle_foreign_protocols(url))
	return;

	/* update location bar */
	if (embed->site_location != NULL)
	{
		g_free (embed->site_location);
	}
	embed->site_location = g_strdup (url);
	embed_update_page_location (embed);

	/* reset title */
	g_free (embed->site_title);
	embed->site_title = g_strdup (_("Untitled"));
	embed_update_page_title (embed);

	/* load the URL */
	if (strcmp (url, MYPORTAL_URL) == 0)
	{
		gtk_moz_embed_stop_load (GTK_MOZ_EMBED(embed->mozEmbed));

		/* display my portal */
		portal_render_into_embed (embed);
	}
	else
	{
		/* load the url */
		gtk_moz_embed_load_url (GTK_MOZ_EMBED(embed->mozEmbed), 
					url);
	}

	/* initialise embed whenever a document is first loaded */
	if (embed->wrapper == NULL)
	{
		embed_wrapper_init (embed);
	}
}

/**
 * embed_add_as_tab: add the embed to the window as a tab
 */
static void 
embed_add_as_tab (GaleonEmbed *embed)
{
	GtkWidget *tab;
	GaleonWindow *window = embed->parent_window;

	/* add as a tab into the notebook */
	tab = embed_notebook_tab_new (embed, embed->site_title);
	if ((g_list_length (window->embed_list) > 1) &&
	     gnome_config_get_bool (CONF_APPEARANCE_TABBED_INSERT_NEW_TABS))
	{
		gtk_notebook_insert_page (GTK_NOTEBOOK (window->notebook),
				  	  GTK_WIDGET (embed->mozEmbed), tab,
					  gtk_notebook_current_page (
						GTK_NOTEBOOK (window->notebook))					  + 1);
	}
	else
	{
		gtk_notebook_append_page (GTK_NOTEBOOK (window->notebook),
				  	  GTK_WIDGET (embed->mozEmbed), tab);
	}
	embed_set_notebook_label_status (embed, TAB_NEW);
}

/**
 * embed_switch_to_page: switch to the embed notebook page
 */
static void 
embed_switch_to_page (GaleonEmbed *embed)
{
	gint page;
	GaleonWindow *window = embed->parent_window;

	gtk_widget_show (GTK_WIDGET (embed->mozEmbed));

	/* argh, this is laborious! */
	page = gtk_notebook_page_num (GTK_NOTEBOOK (window->notebook),
				      GTK_WIDGET (embed->mozEmbed));
	gtk_notebook_set_page (GTK_NOTEBOOK (window->notebook), page);
}

/**
 * embed_startup_init: initialitation that need to occur only one time
 * before any gtkmozembed object is created
 */
void
embed_startup_init (void)
{
	embed_single_init ();
	mozilla_init_promptService ();
	mozilla_init_ContentHandler ();
	mozilla_register_stored_contentOverrideHandlers ();
	mozilla_init_FilePicker ();
	mozilla_load_prefs ();
	/* get the charset config from mozilla */
	mozilla_get_charsets (&charsets, &sorted_charset_titles);
	mozilla_prefs_set ();
}

/* signals to connect on each embed widget */
static const struct
{ 
	char *event; 
	void *func; /* should be a GtkSignalFunc or similar */
}
signal_connections[] =
{
	{ "location",        mozembed_location_changed_cb  },
	{ "title",           mozembed_title_changed_cb     },
	{ "net_start",       mozembed_load_started_cb      },
	{ "net_stop",        mozembed_load_finished_cb     },
	{ "net_state",       mozembed_net_status_change_cb },
	{ "progress",        mozembed_progress_change_cb   },
	{ "link_message",    mozembed_link_message_cb      },
	{ "js_status",       mozembed_js_status_cb         },
	{ "open_uri",        mozembed_open_uri_cb          },
	{ "visibility",      mozembed_visibility_cb        },
	{ "destroy_browser", mozembed_destroy_brsr_cb      },
	{ "dom_mouse_down",  mozembed_dom_mouse_down_cb    },	
	{ "dom_mouse_click", mozembed_dom_mouse_click_cb   },
	{ "dom_key_press",   mozembed_dom_key_press_cb     },
	{ "size_to",         mozembed_size_to_cb           },
	{ "new_window",      mozembed_new_window_cb        },
	/* terminator -- must be last in the list! */
	{ NULL, NULL } 
};

static void 
embed_connect_signals (GaleonEmbed *embed)
{
	gint i;

	/* connect signals */
	for (i = 0; signal_connections[i].event != NULL; i++)
	{
		gtk_signal_connect_while_alive (GTK_OBJECT(embed->mozEmbed),
						signal_connections[i].event,
						signal_connections[i].func, 
						embed,
						GTK_OBJECT(embed->mozEmbed));
	}
	gtk_signal_connect (GTK_OBJECT (embed->mozEmbed), "destroy",
			    GTK_SIGNAL_FUNC (mozembed_destroy_cb), embed);

	/* set gtkmozembed drag and drop destination */
	gtk_drag_dest_set (GTK_WIDGET(embed->mozEmbed), GTK_DEST_DEFAULT_ALL,
			   drop_types, drop_types_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK | GDK_ACTION_ASK );

	/* set gtkmozembed drag and drop signals */
	gtk_signal_connect (GTK_OBJECT (embed->mozEmbed), "drag_data_received",
			    GTK_SIGNAL_FUNC (embed_drag_data_received), embed);

	/* set links drag signal */
	gtk_signal_connect (GTK_OBJECT (embed->mozEmbed), "drag_data_get",
			    GTK_SIGNAL_FUNC (window_drag_data_get_cb),
			    embed);
}

/**
 * embed_single_init: init the gtkmozembed Single object
 */
static void
embed_single_init (void)
{
#if MOZILLA_VERSION > VERSION2(0,9)
	GtkMozEmbedSingle *single;
#endif
	gtk_moz_embed_push_startup ();
	pushed_startup = TRUE;

#if MOZILLA_VERSION > VERSION2(0,9)
	single = gtk_moz_embed_single_get();
	if (!single) {
		g_warning("Failed to get singleton embed object!\n");
	}

	gtk_signal_connect(GTK_OBJECT(single), "new_window_orphan",
			   GTK_SIGNAL_FUNC(new_window_orphan_cb), 
			   NULL);
#endif
}

/**
 * embed_wrapper_init: call it after the first page is loaded
 */
void embed_wrapper_init (GaleonEmbed *embed)
{
	gboolean event_listener = gnome_config_get_bool(CONF_MOUSE_LINKS_DRAG);
	embed->wrapper = mozilla_wrapper_init (embed, event_listener);
}

/**
 * embed_progress_clear: clear all infos about download progress
 */
void
embed_progress_clear(GaleonEmbed *embed)
{
	/* set all progress values to 0 */ 
	embed->loadPercent = 0;
	embed->bytesLoaded = 0;
	embed->maxBytesLoaded = 0;
}

/**
 * embed_add_bookmark_default: add a bookmark
 */
void embed_add_bookmark_default (GaleonEmbed *embed)
{
	add_bookmark_default (BM_SITE, embed->site_title, 
			      embed->site_location, NULL);
}

/**
 * embed_save_url
 */
void embed_save_url (GaleonEmbed *embed, gchar *url)
{
	GtkWidget *fs = NULL;
	GladeXML *gxml;
	gchar *filename, *path;

	gxml = glade_widget_new ("galeon.glade", "fssave", &fs, 
				 embed->parent_window);

	if (save_dir == NULL)
	{
		save_dir = g_strdup (g_get_home_dir());
	}
	filename = embed_get_save_file_name (url);
	path = g_strconcat (save_dir, "/", filename, NULL);
	gtk_file_selection_set_filename (GTK_FILE_SELECTION (fs), path);
	g_free (filename);
	g_free (path);

	gtk_object_set_data (GTK_OBJECT(fs), "url", (gpointer)g_strdup (url));

	gtk_window_set_modal(GTK_WINDOW(fs), TRUE);
	gtk_widget_show(fs);
	window_set_layer(fs);
}

/**
 * embed_save_url_default
 */
void embed_save_url_default (GaleonEmbed *embed, gchar *url)
{
	gchar *filename, *path, *temp_message;
	gboolean result;
	
	if (save_dir == NULL)
	{
		save_dir = g_strdup (g_get_home_dir());
	}
	filename = embed_get_save_file_name (url);
	path = g_strconcat (save_dir, "/", filename, NULL);
	
	result = mozilla_save_url (embed, url, path);
	if (!result)
	{
		 gnome_error_dialog(_("Cannot save the file"));
	}
	else
	{
		temp_message = g_strconcat(N_("Image saved as "), path, NULL);
		window_update_temp_message (embed->parent_window,
					    temp_message);
		g_free (temp_message);
	}

	g_free (filename);
	g_free (path);
}

/**
 * embed_set_image_as_background
 */
void embed_set_image_as_background (GaleonEmbed *embed, gchar *url)
{
	gchar *file, *path, *command;
	gboolean result;

	/* get the filename and check */
	file = embed_get_save_file_name (url);
	g_return_if_fail (file != NULL);
	g_return_if_fail (strlen (file) != 0);

	/* build a path to save it in */
	path = g_strconcat (g_get_home_dir (), "/.galeon/", file, NULL);

	/* save the image */
	result = mozilla_save_url (embed, url, path);

	/* if successful, use gnome to set the background */
	if (result)
	{
		/* build command */
		command = g_strconcat ("background-properties-capplet "
				       "--init-session-settings --ignore "
				       "--background-image=",
				       path, NULL);

		/* execute it synchronously */
		gnome_execute_shell (g_get_home_dir (), command);

		/* free */
		g_free (command);
	}

	/* free */
	g_free (file);
	g_free (path);
}

/**
 * embed_sav_document
 */
void embed_save_document (GaleonEmbed *embed, gboolean main)
{
	GtkWidget *fs = NULL;
	GladeXML *gxml;
	gchar *target = NULL, *filename;

	gxml = glade_widget_new ("galeon.glade", "fssave", &fs, 
				 embed->parent_window);
	if (main) {
		target = mozilla_get_main_document_url (embed);
	} else {
		target = mozilla_get_document_url (embed);
	}

	filename = embed_get_save_file_name (target);
	gtk_file_selection_set_filename (GTK_FILE_SELECTION (fs), filename);

	gtk_object_set_data (GTK_OBJECT(fs), "url", (gpointer *) target);

	g_free (filename);

	gtk_window_set_modal(GTK_WINDOW(fs), TRUE);
	gtk_widget_show(fs);
	window_set_layer(fs);
}

/**
 * embed_view_source: view web page source 
 */
void embed_view_source (GaleonEmbed *embed, gboolean main, 
			gboolean new_window)
{
	gchar *url;
	
	if (main) {
		url = mozilla_get_main_document_url (embed);
	} else {
		url = mozilla_get_document_url (embed);
	}
	g_return_if_fail (url);

	if (gnome_config_get_bool (CONF_HANDLERS_USE_EXTERNAL_SOURCE_VIEWER))
	{
		embed_view_source_external (embed, url);
	}
	else
	{
		embed_create_from_url_view_source (embed, url, new_window);
	}
	g_free (url);
}

/**
 * embed_view_source_external: view web page source with an external viewer 
 */
static void embed_view_source_external (GaleonEmbed *embed, const gchar *url)
{
	/* FIXME is use /tmp ok ? */
	char *filename = g_strdup("/tmp/galeon-viewsource-XXXXXX");
	gboolean save_result = FALSE;
	int result;

	/* get a name for the temporary file */
	result = mkstemp (filename);

	if (result == -1)
	{
		gnome_error_dialog ( _("Could not create a "
				       "temporary file"));
		return;
	}
	close (result);

	save_result = mozilla_save_url (embed, url, filename);
	
	if (!save_result)
	{
		gnome_error_dialog (_("Something went wrong while "
				      "saving the file to a temporary "
				      "location"));
		g_free(filename);
		return;
	}
	
	launch_external_viewer (filename);
	g_free (filename);
	return;
}

/**
 * embed_close: close a GaleonEmbed
 */
void
embed_close (GaleonEmbed *embed)
{
	GaleonWindow *window;

	/* already closing */
	if (embed->is_closing)
	{
		return;
	}
	embed->is_closing = TRUE;

	/* select the next page */
	if (embed->is_active)
	{
		window = embed->parent_window;
		return_if_not_window (window);
		gtk_notebook_next_page (GTK_NOTEBOOK (window->notebook));
	}

	/* destroy the embedding widget -- this will also destroy 
	 * the notebook tab and it's label, and remove it from 
	 * the relevant lists */
	gtk_widget_destroy (embed->mozEmbed);
}

/**
 * embed_open_frame: open the frame pointed by the event target 
 */
void 
embed_open_frame (GaleonEmbed *embed, gboolean same_embed, gboolean new_window)
{
	gchar *url = NULL;
	
	url = mozilla_get_document_url (embed);

	if (url != NULL)
	{
		if (same_embed)
		{
			embed_load_url (embed, url);
		} 
		else 
		{
			embed_create_from_url (embed, url, TRUE, new_window);
		}
	}
}

/*
 * embed_reload: call gtk_moz_embed_reload but first check if 
 * we are not in MyPortal
 */
void
embed_reload (GaleonEmbed *embed)
{
	/* check if it's the portal */
	if (strcmp (embed->site_location, MYPORTAL_URL) == 0)
	{
		/* reload by regenerating */
		gtk_moz_embed_stop_load (GTK_MOZ_EMBED(embed->mozEmbed));
		portal_render_into_embed (embed);
	}
	else
	{
		/* reload as usual */
		gtk_moz_embed_reload (GTK_MOZ_EMBED(embed->mozEmbed), 
				      GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
	}
}

/**
 * embed_update_page_location: called if the page location changes, or to
 * bring the GaleonEmbed.site_location field in sync with the currently 
 * viewed page
 */
void
embed_update_page_location (GaleonEmbed *embed)
{
	const PixmapData *drag_pixmap;
	GaleonWindow *window;
	gchar *new_location;
	GtkWidget *entry;

	/* check we're currently being viewed */
	if (!embed->is_active)
		return;

	/* get the parent window */
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the location string */
	entry = window->toolbar_entry;
	new_location = window->active_embed->site_location;

	/* clear text */
	window_clear_url_entry (window);
	if (new_location && strcmp(new_location, "about:blank") != 0)
	{
		/* change the url entry text */
		window_set_url_entry (window, new_location);
		
		/* update the drag location pixmap */
		if (gnome_config_get_bool (CONF_GENERAL_FAVICONS_ENABLED)
			&& embed->parent_window->toolBarOn)
		{
			drag_pixmap = bookmarks_get_siteicon (new_location);
			gtk_pixmap_set (GTK_PIXMAP (window->drag_pixmap), 
					drag_pixmap->pixmap, 
					drag_pixmap->mask);
		}
	}

	window_update_temp_message (window, NULL);
	window_update_nav_controls (window);
}

/**
 * embed_update_page_title: called if the page title changes, or to bring
 * the main window title in sync with the currently viewed page
 */
void
embed_update_page_title (GaleonEmbed *embed)
{
	gchar *full_title;
	gchar *title_string;
	GaleonWindow *window;

	/* set notebook label (although this might not be visible) */
	embed_set_notebook_label (embed);

	/* if this page isn't being viewed, get out now */
	if (!embed->is_active)
		return;

	/* get the window */
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the format string */
	title_string = gnome_config_get_string (CONF_APPEARANCE_WINDOWS_TITLE);

	/* format the full title */
	full_title = g_strdup_printf (title_string, embed->site_title);

	/* set the toplevel window title to the document title */
	gtk_window_set_title (GTK_WINDOW (window->WMain), full_title);

	/* free allocated strings */
	if (full_title) g_free (full_title);
	if (title_string) g_free (title_string);  
}

/**
 * embed_set_notebook_label: sets the notebook tab label belonging to embed
 * to the string contained in text.
 */
void
embed_set_notebook_label (GaleonEmbed *embed)
{
	GtkWidget *tab, *label;
	gchar *shortened;
	GaleonWindow *window;
	gint length;

	/* get the parent window */
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the tab widget */
	tab = gtk_notebook_get_tab_label (GTK_NOTEBOOK (window->notebook),
					  GTK_WIDGET (embed->mozEmbed));

	/* get out if theres a problem */
	g_return_if_fail (tab != NULL);

	/* take care of stupid web pages that have a single non-breaking
	 * space in their titles (yes, i've seen this done) */
        if (strlen (embed->site_title) == 0)
        {
                g_free (embed->site_title);
        	embed->site_title = g_strdup (_("Untitled"));
        }

        /* shorten notebook's label */
        switch (gnome_config_get_int (CONF_APPEARANCE_TABBED_SHORTEN_STYLE))
        {
        /* abbreviate the text label */
        case 0:
                gtk_widget_set_usize (GTK_WIDGET (tab), -1, -1);
                length = gnome_config_get_int (
			CONF_APPEARANCE_TABBED_SHORTEN_CHARS);   
                shortened = shorten_name (embed->site_title, length);
                break;
        /* set fixed size of the label widget */
        case 1: 
                gtk_widget_set_usize (GTK_WIDGET (tab), 
                        gnome_config_get_int (
                                CONF_APPEARANCE_TABBED_SHORTEN_POINTS), -2);
               	shortened = g_strdup (embed->site_title);
                break;
        default:
                g_assert_not_reached ();
                return;
        }
	g_return_if_fail (shortened != NULL);

	/* get the text label */
	label = gtk_object_get_data (GTK_OBJECT (tab), "label");
	if (!GTK_IS_LABEL (label))
		return;
	else
		gtk_misc_set_alignment (GTK_MISC (label), 0.00, 0.5);

	/* if it's different than the new shortened text, change it */
	if (strcmp (GTK_LABEL (label)->label, shortened))
	{
		gtk_label_set_text (GTK_LABEL (label), shortened);
	}

	/* the menu text is the full text */
	gtk_notebook_set_menu_label_text (GTK_NOTEBOOK (window->notebook),
					  GTK_WIDGET (embed->mozEmbed),
					  embed->site_title);

	/* free allocated strings */
	g_free (shortened);
}

/**
 * embed_set_notebook_label_status: sets the status for the notebook tab
 * label belonging to embed.
 */
void
embed_set_notebook_label_status (GaleonEmbed *embed, TabbedStatus status)
{
	GaleonWindow *window;
	GtkWidget *tab, *label;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the tab widget */
	tab = gtk_notebook_get_tab_label (GTK_NOTEBOOK (window->notebook),
					  GTK_WIDGET (embed->mozEmbed));
	g_return_if_fail (tab != NULL);

	label = gtk_object_get_data (GTK_OBJECT (tab), "label");
	if (!GTK_IS_LABEL (label))
		return;

	switch (status)
	{
		case TAB_NORMAL:
			gtk_widget_set_rc_style (label);
			break;
			
		case TAB_NEW:
			gtk_widget_set_style (label, blue_text_style);
			break;
			
		case TAB_LOADING:
			gtk_widget_set_style (label, red_text_style);
			break;
	}
}

/**
 * embed_notebook_tab_update_closebutton (internal function): tab, the
 * notebook tab belonging to embed, is updated (if necessary) to reflect
 * the user's preference in the display of a close button.  if tab is
 * null, it will be found using embed.
 */
static void
embed_notebook_tab_update_closebutton (GaleonEmbed *embed, GtkWidget *tab)
{
	gint use_button;
	GaleonWindow *window;
	GtkWidget *pixmap;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	if (tab == NULL)
	{
		/* get the tab widget */
		tab = gtk_notebook_get_tab_label 
			(GTK_NOTEBOOK (window->notebook),
			 GTK_WIDGET (embed->mozEmbed));
	}

	use_button = 
		gnome_config_get_bool (CONF_APPEARANCE_TABBED_CLOSEBUTTON);

	if (!use_button && embed->notebook_close_button != NULL)
	{
		gtk_widget_destroy (embed->notebook_close_button);
		gtk_widget_destroy (embed->notebook_close_pixmap);
	}
	else if (use_button && embed->notebook_close_button == NULL)
	{
		embed->notebook_close_button = gtk_event_box_new ();
		
		/* load the close pixmap if it's not already loaded */
		if (close_pix == NULL)
		{
			close_pix = pixmap_data_new_from_file
				(SHARE_DIR "/small-close.xpm");
			if (close_pix == NULL)
			{
				close_pix = g_new0 (PixmapData, 1);
			}
		}
		pixmap = gtk_pixmap_new (close_pix->pixmap, close_pix->mask);
		gtk_widget_show (pixmap);
		gtk_container_add 
			(GTK_CONTAINER (embed->notebook_close_button),
			 GTK_WIDGET (pixmap));

		/* Create another pixmap widget so we can use it when a tab
		   has been made inactive.  We do this because the button has
		   its own GdkWindow and cannot be made transparent, leading
		   to much ugliness.  --Josh */
		embed->notebook_close_pixmap = 
			gtk_pixmap_new (close_pix->pixmap, close_pix->mask);
		gtk_widget_set_sensitive (embed->notebook_close_pixmap, FALSE);

		/* pack both forms of the widget into the tab */
		gtk_box_pack_start (GTK_BOX (tab), 
				    GTK_WIDGET (embed->notebook_close_button),
				    FALSE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (tab), 
				    GTK_WIDGET (embed->notebook_close_pixmap),
				    FALSE, FALSE, 0);

		/* when the close button is clicked, close the embed */
		gtk_signal_connect 
			(GTK_OBJECT (embed->notebook_close_button),
			 "button_press_event", 
			 GTK_SIGNAL_FUNC (embed_notebook_close_clicked_cb),
			 embed);

		/* start with insensitive pixmap showing */
		gtk_widget_show (GTK_WIDGET (embed->notebook_close_pixmap));
	}
}

/**
 * embed_update_notebook_closebutton: adds or removes embed's notebook
 * tab's closebutton, as appropriate
 */
void
embed_update_notebook_closebutton (GaleonEmbed *embed)
{
	embed_notebook_tab_update_closebutton (embed, NULL);
}

GtkWidget *
embed_notebook_tab_new (GaleonEmbed *embed, const gchar *title)
{
	GtkWidget *box = gtk_hbox_new (FALSE, 4);
	GtkWidget *label = gtk_label_new (title);

	gtk_misc_set_alignment (GTK_MISC (label), 0.00, 0.5);
	
	gtk_object_set_data (GTK_OBJECT (box), "label", label);
	gtk_drag_dest_set (box, GTK_DEST_DEFAULT_ALL,
			   drop_types, drop_types_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK | GDK_ACTION_ASK );
	gtk_signal_connect (GTK_OBJECT (box), "drag_data_received",
			    GTK_SIGNAL_FUNC(embed_drag_data_received), embed);

	gtk_box_pack_start (GTK_BOX (box), label, TRUE, TRUE, 0);
	gtk_widget_show (label);

	embed_notebook_tab_update_closebutton (embed, box);

        if (gnome_config_get_int (CONF_APPEARANCE_TABBED_SHORTEN_STYLE) == 1){
                gtk_widget_set_usize (GTK_WIDGET (box),
                        gnome_config_get_int (
                                CONF_APPEARANCE_TABBED_SHORTEN_POINTS), -2);
        }

	return box;
}

/**
 * embed_get_save_file_name: Parse a URL to find a filename for saving
 *
 * The string passed in will be modified. The returned string should be freed.
 */
static gchar *
embed_get_save_file_name (gchar *url)
{
	GnomeVFSURI *uri;
	char *filename;

	uri = gnome_vfs_uri_new (url);
	g_return_val_if_fail (uri, NULL);

	filename = gnome_vfs_uri_extract_short_name (uri);

	gnome_vfs_uri_unref (uri);

	return filename;
}

/**
 * embed_go_up: go to the nth parent directory
 */
void
embed_go_up (GaleonEmbed *embed, gint levels)
{
	GnomeVFSURI *uri, *up_uri;
	gchar *location;

	/* use gnome-vfs to find parent */
	uri = gnome_vfs_uri_new (embed->site_location);
	if (uri == NULL)
	{
		return;
	}

	/* go upwards to find the nth level up */
	do
	{
		up_uri = gnome_vfs_uri_get_parent (uri);
		gnome_vfs_uri_unref (uri);
		uri = up_uri;

		/* this can happen if Up is selected from the menu */
		if (uri == NULL)
		{
			return;
		}
	}
	while (levels--);

	/* get the location */
	location = gnome_vfs_uri_to_string (uri, 0);
	gnome_vfs_uri_unref (uri);

	/* visit it if apparently valid */
	if (location == NULL) return;

	if (strlen (location) != 0)
	{
		embed_load_url (embed, location);
	}
	g_free (location);
}

/**
 * embed_can_go_up: test to see if we can go to a parent directory
 */
gboolean
embed_can_go_up (GaleonEmbed *embed)
{
	GnomeVFSURI *uri;
	gboolean result;

	/* check embed location is valid */
	if (embed->site_location == NULL || strlen (embed->site_location) == 0)
	{
		return FALSE;
	}

	/* use gnome-vfs to find parent */
	uri = gnome_vfs_uri_new (embed->site_location);
	if (uri == NULL)
	{
		return FALSE;
	}

	result = gnome_vfs_uri_has_parent (uri);

	gnome_vfs_uri_unref (uri);

	return result;
}

/**
 * embed_set_zoom: set the zoom level for a given embed
 */
void
embed_set_zoom (GaleonEmbed *embed, gint zoom)
{
	/* sanity check */
	if (zoom == 0)
	{
		g_warning ("ignoring request to set zoom to 0");
		return;
	}

	/* check we're not already set */
	if (zoom == embed->zoom || embed->wrapper == NULL)
	{
		return;
	}

	/* set in mozilla */
	mozilla_set_zoom (embed, (float)zoom / 100.0);
	embed->zoom_auto_set = FALSE;
	embed->zoom = zoom;

	/* set in window */
	window_update_zoom (embed->parent_window);
}

/** 
 * Creates the back history menu 
 */
GtkMenu *
embed_create_back_menu (GaleonEmbed *embed)
{
	int index, count, i, level;
	char **titles;
	GtkWidget *menu = gtk_menu_new ();

	if (!mozilla_session_history (embed, &titles, &count, &index))
	{
		return NULL;
	}

	for (i = index - 1, level = 0; i >= 0; i--, level++) 
	{
		add_bf_menu (menu, titles[i], i, embed, level);
	}
	
	free_string_array (titles, count);
	return GTK_MENU(menu);
}

/**
 * Creates the forward history menu
 */
GtkMenu *
embed_create_forward_menu (GaleonEmbed *embed)
{
	int index, count, i, level;
	char **titles;
	GtkWidget *menu = gtk_menu_new ();

	if (!mozilla_session_history (embed, &titles, &count, &index))
	{
		return NULL;
	}	

	for (i = index + 1, level = 0; i < count; i++, level++)
	{
		add_bf_menu (menu, titles[i], i, embed, level);
	}
	
	free_string_array (titles, count);
	return GTK_MENU(menu);
}

/** 
 * Creates the multi-level up menu 
 */
GtkMenu *
embed_create_up_menu (GaleonEmbed *embed)
{
	GnomeVFSURI *uri, *up_uri;
	GtkWidget *menu, *item;
	gint level;

	/* check embed location is valid */
	if (embed->site_location == NULL || strlen (embed->site_location) == 0)
	{
		return NULL;
	}

	/* create a vfs entry for this level */
	uri = gnome_vfs_uri_new (embed->site_location);
	if (uri == NULL)
	{
		return NULL;
	}

	/* create the menu */
	menu = gtk_menu_new ();

	/* create each possible up entry */
	for (level = 0;; level++)
	{
		up_uri = gnome_vfs_uri_get_parent (uri);
		gnome_vfs_uri_unref (uri);
		uri = up_uri;
		
		/* get out of loop if no parent */
		if (uri == NULL)
		{
			break;
		}

		/* create the menu entry */
		item = new_num_accel_menu_item (level, uri->text, TRUE, menu);
		gtk_widget_show (GTK_WIDGET (item));
		gtk_object_set_user_data (GTK_OBJECT (item), embed);
		gtk_menu_append (GTK_MENU (menu), GTK_WIDGET (item));
		gtk_signal_connect (GTK_OBJECT (item), "activate",
				    up_menu_menuitem_activate_cb, 
				    GINT_TO_POINTER (level));
	}

	/* the menu is completed */
	return GTK_MENU (menu);
}

/**
 * Adds a menuitem to a back/forward history menu
 */
static void
add_bf_menu (GtkWidget *menu, char* label, int index, GaleonEmbed *embed, int level)
{
	GtkWidget *item = new_num_accel_menu_item (level, label, TRUE, menu);

	gtk_widget_show (item);
	gtk_object_set_user_data (GTK_OBJECT (item), embed);
	gtk_menu_append (GTK_MENU (menu), item);
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    history_menu_menuitem_activate_cb, 
			    GINT_TO_POINTER (index));
}

/**
 * Creates a menu item with a numbered/lettered accel
 */
static GtkWidget *
new_num_accel_menu_item (gint num, gchar *origtext, gboolean lettersok, 
			 GtkWidget *menu)
{
	gchar *text = new_num_accel_str(num, origtext, lettersok);
	if (text == NULL)
		return gtk_menu_item_new_with_label (text);
	else
	{
		GtkWidget *item = gtk_menu_item_new_with_label ("");
		label_set_accel_text (text, GTK_BIN (item)->child, menu, item);
		g_free(text);
		return item;
	}
}

/*
 * create the charset titles submenu structure 
 */
void
embed_create_charset_submenus (GtkMenuItem *encoding_menu_item, 
			       GList *charset_titles, 
			       GaleonWindow *window)
{
	GtkWidget *encoding_menu, *tw, *tw2 = NULL, *tw3;
	GList *tl, *tl2 = NULL;
	gint i, j;
	gint lgroups_count = get_lgroups_count ();
	gint translated_cscount = get_translated_cscount ();
	
 	encoding_menu = gtk_menu_new();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (encoding_menu_item), 
				   encoding_menu);

	tl = g_list_copy(charset_titles);
	
	for (i = 0;i < lgroups_count;i++) {
		tw = gtk_menu_item_new_with_label (_(lgroups[i]));
		gtk_menu_append (GTK_MENU(encoding_menu), tw);
		gtk_widget_show (tw);
		
		tw2 = gtk_menu_new();
		gtk_menu_item_set_submenu (GTK_MENU_ITEM (tw), tw2);
		
		for (j = 0; j < translated_cscount; j++)
			if (charset_trans_array[j].lgroup == i) {
				tl2 = g_list_find_custom (
					tl, 
					_(charset_trans_array[j].charset_title),
					strcasestr_cb);
				if (tl2 != NULL) {
					tl = g_list_remove_link (tl, tl2);
					g_list_free_1 (tl2);
				} 
				else /* we dont want to add menuitems for
				      * charset titles not in mozilla */
					continue;
				
				tw3 = gtk_menu_item_new_with_label (
				       _(charset_trans_array[j].charset_title));
				gtk_menu_append (GTK_MENU (tw2), tw3);
				gtk_widget_show (tw3);
				gtk_signal_connect (
					GTK_OBJECT (tw3), 
					"activate",
					window_menu_encoding_activate_cb, 
					window);
			}
	}
	/* add the leftovers /if any/ to the Other submenu */
	while (tl != NULL) { 
		tw = gtk_menu_item_new_with_label (tl->data);
		gtk_menu_append (GTK_MENU(tw2), tw);
		gtk_widget_show (tw);
		gtk_signal_connect (GTK_OBJECT (tw), "activate",
				    window_menu_encoding_activate_cb, window);
		tl2 = tl->next;
		g_list_free_1 (tl);
		tl = tl2;
	}
}

static gint 
strcasestr_cb (gconstpointer a, gconstpointer b)
{
	return g_strcasestr (a, b) == NULL ? 1 : 0;
}

