/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "window.h"
#include "prefs.h"
#include "history.h"
#include "embed.h"
#include "menubar.h"
#include "misc.h"
#include "bookmarks.h"
#include "print.h"
#include "find.h"
#include "mozilla.h"
#include "xlink.h"

#include <string.h>
#include <gtkmozembed.h>
#include <gtk/gtkspinbutton.h>
#include <libgnomeui/gnome-about.h>
#include <libgnome/gnome-help.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

void 
window_menu_recent_activate_cb (GtkMenuItem *menuitem, gpointer user_data);

/**
 * window_menu_prefs_activate_cb: show preferences dialog
 */
void
window_menu_prefs_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	preferences_open ();
}

/**
 * window_menu_bookmarks_edit_activate_cb: show edit bookmarks dialog
 */
void
window_menu_bookmarks_edit_activate_cb (GtkMenuItem *menuitem,
				        GaleonWindow *window) 
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	bookmarks_editor_show_dialog (window, STANDARD_BOOKMARKS_EDITOR);
}

/**
 * window_menu_find_activate_cb: show find dialog
 */
void
window_menu_find_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	find_show_dialog (window);
}

/*
 * Find which widget is currently in focus
 */
static GtkWidget *
window_menu_edit_find_focus(GaleonWindow *window)
{
	GtkWidget *widget;
	GList *tb_entry;

	if (GTK_WIDGET_HAS_FOCUS ( window -> toolbar_entry ) )
		return(window -> toolbar_entry);

	/* Search through the toolbars, then down each toolbar
	** looking for a focused widget
	*/

	for (widget=NULL, tb_entry = g_list_first(window->bookmarks_toolbars);
		widget==NULL && tb_entry!=NULL;
		tb_entry = g_list_next(tb_entry)) {

		GtkToolbar *toolbar=(GtkToolbar *) tb_entry->data;
		GList *tbc_entry;

		/* Step through the toolbar children */
		for (tbc_entry = g_list_first(toolbar->children);
			widget==NULL && tbc_entry!=NULL;
			tbc_entry = g_list_next(tbc_entry)) {

			GtkToolbarChild *toolbarchild =
				(GtkToolbarChild *) tbc_entry->data;

			if (toolbarchild->type == GTK_TOOLBAR_CHILD_WIDGET &&
				GTK_WIDGET_HAS_FOCUS (toolbarchild->widget)) {

				widget=toolbarchild->widget;
			}
		}
	}

	if (widget)
		return(widget);

	/* Add getting the embed Mozilla widget here */

	return(NULL);
}

/*
 * Called before the Cut menu item is exposed. We allow it to be sensitive
 * if we are currently in an updateable field and there is something selected
 */
static gboolean
window_menu_edit_cut_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	GaleonWindow *window=data;

	if (event->type == GDK_EXPOSE) {
		gboolean sensitive=FALSE;
		GtkWidget *focus_widget;

		focus_widget=window_menu_edit_find_focus(window);
		if (GTK_IS_EDITABLE(focus_widget)) {
			GtkEditable *editable = GTK_EDITABLE(focus_widget);

			if (editable->has_selection && editable->editable) {
				sensitive=TRUE;
			}
		}
		gtk_widget_set_sensitive(widget, sensitive);
	}

	return(FALSE);
}

/*
 * Called before the Copy menu item is exposed. We allow it to be sensitive
 * if there is something selected
 */
static gboolean
window_menu_edit_copy_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	GaleonWindow *window=data;

	if (event->type == GDK_EXPOSE) {
		gboolean sensitive=FALSE;
		GtkWidget *focus_widget;

		focus_widget=window_menu_edit_find_focus(window);
		if (GTK_IS_EDITABLE(focus_widget)) {
			GtkEditable *editable = GTK_EDITABLE(focus_widget);

			if (editable->has_selection) {
				sensitive=TRUE;
			}
		}
		gtk_widget_set_sensitive(widget, sensitive);
	}

	return(FALSE);
}

/*
 * Called before the Paste menu item is exposed. We allow it to be sensitive
 * if the current field is updateable and there is something in the clipboard.
 * Note the call to galeon_x_clipboard_present().
 */
static gboolean
window_menu_edit_paste_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	GaleonWindow *window=data;

	if (event->type == GDK_EXPOSE) {
		gboolean sensitive=FALSE;
		GtkWidget *focus_widget;

		focus_widget=window_menu_edit_find_focus(window);
		if (GTK_IS_EDITABLE(focus_widget)) {
			GtkEditable *editable = GTK_EDITABLE(focus_widget);
			if (editable->editable && galeon_x_clipboard_present()) {
				sensitive=TRUE;
			}
		}
		gtk_widget_set_sensitive(widget, sensitive);
	}

	return(FALSE);
}

/*
 * Add event handlers for the expose event onto the cut/copy/paste items
 * on the edit menu
 */
void
window_menu_edit_add_events(GaleonWindow *window)
{
	GnomeUIInfo *editmenu = edit_menu_uiinfo;
	gint count;

	for (count=0; editmenu[count].type != GNOME_APP_UI_ENDOFINFO; count++) {
		gpointer cb = editmenu[count].moreinfo;

		if (cb == window_menu_edit_cut_cb) {
			gtk_signal_connect ( GTK_OBJECT (editmenu[count].widget),
				"expose_event",
				GTK_SIGNAL_FUNC (window_menu_edit_cut_event),
				window);
		} else if (cb == window_menu_edit_copy_cb) {
			gtk_signal_connect ( GTK_OBJECT (editmenu[count].widget),
				"expose_event",
				GTK_SIGNAL_FUNC (window_menu_edit_copy_event),
				window);
		} else if (cb == window_menu_edit_paste_cb) {	
			gtk_signal_connect ( GTK_OBJECT (editmenu[count].widget),
				"expose_event",
				GTK_SIGNAL_FUNC (window_menu_edit_paste_event),
				window);
		}
	}

}

/**
 * window_menu_edit_cut_cb: Cut selected text into clipboard
 */
void
window_menu_edit_cut_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GtkEditable *editable;
	GtkWidget *widget;

	if ((widget=window_menu_edit_find_focus(window)) == NULL)
		return;

	editable = GTK_EDITABLE (widget);

	gtk_editable_cut_clipboard (editable);
}

/**
 * window_menu_edit_copy_cb: Copy selected text into clipboard
 */
void
window_menu_edit_copy_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GtkEditable *editable;
	GtkWidget *widget;

	if ((widget=window_menu_edit_find_focus(window)) == NULL)
		return;

	editable = GTK_EDITABLE (widget);

	gtk_editable_copy_clipboard (editable);
}

/**
 * window_menu_edit_paste_cb: Paste selected text into clipboard
 */
void
window_menu_edit_paste_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GtkEditable *editable;
	GtkWidget *widget;

	if ((widget=window_menu_edit_find_focus(window)) == NULL)
		return;

	editable = GTK_EDITABLE (widget);

	gtk_editable_paste_clipboard (editable);
}

/**
 * Goes to the url of the item
 */
void 
window_menu_recent_activate_cb (GtkMenuItem *menuitem, gpointer user_data)
{
        GtkWidget *WMain = window_lookup_widget (GTK_WIDGET(menuitem), "WMain");
        GaleonWindow *window = (GaleonWindow *) gtk_object_get_data
		(GTK_OBJECT (WMain), "GaleonWindow");
        HistoryItem *hi = gtk_object_get_user_data (GTK_OBJECT (menuitem));

	return_if_not_window (window);
	window_load_url(window, hi->url);
}

/**
 * window_spinner_button_clicked_cb: spinner clicked
 */
void
window_spinner_button_clicked_cb(GtkButton *button, GaleonWindow *window)
{
	return_if_not_window (window);
	/* FIXME: don't hardcode this */
	window_load_url (window, "http://galeon.sourceforge.net");
}

/**
 * window_menu_view_source_activate_cb: view source
 */
void
window_menu_view_source_activate_cb (GtkMenuItem *menuitem,
				     GaleonWindow *window)
{
	gboolean tabbed_mode;

	return_if_not_window (window);

	tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);
	embed_view_source (window->active_embed, TRUE, !tabbed_mode);
}

/**
 * window_menu_history_activate_cb: show history dialog
 */
void window_menu_history_activate_cb (GtkMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	history_show_dialog (window);
}

/**
 * window_menu_about_activate_cb: show about dialog 
 */
void
window_menu_about_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	static gchar *authors[] = {
		"Marco Pesenti Gritti <mpeseng@tin.it>",
		"Remco de Jong <rdj@rdj.cg.nu>",
		/* if your encoding allows it, use aacute (U00E1)
		 * for the 'a' of 'Fernandez' */
		N_("Ricardo Fernandez Pascual <ric@users.sourceforge.net>"),
		"Nate Case <nd@kracked.com>",
		"Bob Phan <ex-freek@yahoo.com>",
		"Matt Aubury <matt@ookypooky.com>",
		"Joshua Horvath <jmhorvat@home.com>",
		/* this one is supposed to be in cyrillic */
		N_("Yanko Kaneti <yaneti@declera.com>"),
		"Philip Langdale <philipl@mail.utexas.edu>",
		"Matthew Mueller <donut@azstarnet.com>",
		" ", /* dont delete the space */
		N_("Graphic arts: "),
	       /* if your encoding allows it, use eacute (U00E9)
		* for the two 'e's of Frederic */
		N_("Toussaint Frederic (fred@linuxgraphic.org)"),
		NULL
	};

	static GtkWidget * about;

	return_if_not_window (window);

	if (!about)
	{
		gint i, num_authors = sizeof(authors) / sizeof(gchar *);
		gchar *translated_names[num_authors];
		gchar *bottom_line;

		if (strcmp (_("Translation:"), "Translation:") == 0) {
			bottom_line = g_strdup("A GNOME browser based on mozilla");
		} else {
			bottom_line = g_strconcat (
					_("A GNOME browser based on mozilla"),
					"\n \n",
			/* Translators should localize the following string
			 * which will be displayed at the bottom of the about
			 * box to give credit to the translator(s).
			 * Translate the "Translation:", add your name and an
			 * email address. You can span several lines with a
			 * newline character if needed, but it shouldn't be
			 * too long; vertical space is limited in the about
			 * dialog. 
			 * If you dont translate it nothing will be added */
					_("Translation:"), NULL);
		}
		
		for (i = 0; i < num_authors ; i++)
			translated_names[i] = _(authors[i]);
		
		about = gnome_about_new (_(PACKAGE), VERSION,
				 _("Copyright (C) 2000-2001 Marco Pesenti Gritti"),
				 (const gchar **)translated_names, 
				 bottom_line,
				 SHARE_DIR"/logo.png");
		g_free(bottom_line);

		gtk_signal_connect (GTK_OBJECT (about), "destroy",
			    GTK_SIGNAL_FUNC(gtk_widget_destroyed), &about);
		gtk_widget_show (about);
	}

	window_set_layer (about);
}

/**
 * window_menu_manual_activate: show the main Galeon manual page
 */
void
window_menu_manual_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	gnome_help_goto (NULL, "ghelp:galeon");
}

/** 
 * window_menu_new_window_activate_cb: open a new window
 */
void
window_menu_new_window_activate_cb (GtkMenuItem *menuitem,
				    GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	embed_create_default (window->active_embed, TRUE);
}

/** 
 * window_menu_new_tab_activate_cb: open a new tab
 */
void
window_menu_new_tab_activate_cb (GtkMenuItem *menuitem,
				 GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	embed_create_default (window->active_embed, FALSE);
}

/** 
 * window_menu_open_activate_cb: show open file dialog
 */
void
window_menu_open_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	window_show_open_dialog (window);
}

/** 
 * on_new_window_activate: show open url dialog
*/
void
window_menu_open_url_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	window_show_openurl_dialog (window);
}

/** 
 * on_save_as1_actuvate: show save dialog
 */
void
window_menu_save_as_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	embed_save_document (window->active_embed, TRUE);
}

/** 
 * window_menu_exit_activate_cb: exit galeon
 */
void
window_menu_exit_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);

	/* We call the quit function indirectly, to make sure control returns
	   to the main loop first.  This is because we might have been
	   activated from within the mozembed_key_press callback, and if the
	   widget is destroyed before returning from the callback, very bad
	   things happen.  --Josh */
	gtk_timeout_add (50, (GtkFunction) galeon_quit, window);
}

/** 
 * window_menu_exit_with_session_activate_cb: exit galeon
 */
void
window_menu_exit_with_session_activate_cb (GtkMenuItem *menuitem, 
					   GaleonWindow *window)
{
	return_if_not_window (window);
	galeon_quit_with_session ();
}

/** 
 * window_menu_print_activate_cb: print document
 */
void
window_menu_print_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	print_show_dialog (window);
}


/** 
 * window_menu_close_activate_cb: close current active embed
 */
void
window_menu_close_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window) 
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	embed_close (window->active_embed);
}

/** 
 * window_menu_back_activate_cb: back menu activated
 */
void
window_menu_back_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	gtk_moz_embed_go_back (GTK_MOZ_EMBED(window->active_embed->mozEmbed));
}

/** 
 * window_menu_up_activate_cb: up menu activated
 */
void
window_menu_up_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	embed_go_up (window->active_embed, 0);
}

/** 
 * window_menu_forward_activate_cb: forward menu activated
 */
void
window_menu_forward_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	gtk_moz_embed_go_forward(GTK_MOZ_EMBED(window->active_embed->mozEmbed));
}

/** 
 * window_menu_prev_tab_activate_cb: previous tab activated
 */
void
window_menu_prev_tab_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	gtk_notebook_prev_page (GTK_NOTEBOOK (window->notebook));
}

/** 
 * window_menu_next_tab_activate_cb: next tab activated
 */
void
window_menu_next_tab_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	gtk_notebook_next_page (GTK_NOTEBOOK (window->notebook));
}

/** 
 * window_menu_stop_activate_cb: stop menu activated
 */
void
window_menu_stop_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	gtk_moz_embed_stop_load(GTK_MOZ_EMBED(window->active_embed->mozEmbed));
}

/** 
 * window_menu_reload_activate_cb: reload menu activated
 */
void
window_menu_reload_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	embed_reload (window->active_embed);
}

/** 
 * window_menu_home_page_activate_cb: home page menu item activated
 */
void
window_menu_home_page_activate_cb (GtkMenuItem *menuitem,
				    GaleonWindow *window)
{
	return_if_not_window (window);
	window_go_home (window, FALSE, FALSE);
}

void
window_menu_add_bookmark_activate_cb (GtkMenuItem *menuitem,
					   GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	embed_add_bookmark_default (window->active_embed);
}

/** 
 * window_menu_toolbar_activate_cb: hide/show the toolbar
 */
void window_menu_toolbar_activate_cb (GtkCheckMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{		
		window_toolbar_show (window);
	}
	else
	{
		window_toolbar_hide (window);
	}
}

/** 
 * window_menu_menubar_activate_cb: hide/show the menubar
 */
void window_menu_menubar_activate_cb (GtkCheckMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		window_menubar_show (window);
	}
	else
	{
		window_menubar_hide (window);
	}
}

/** 
 * window_menu_statusbar_activate_cb: hide/show the menubar
 */
void window_menu_statusbar_activate_cb (GtkCheckMenuItem *menuitem,
					GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		window_statusbar_show (window);
	}	
	else
	{
		window_statusbar_hide (window);
	}
}

/** 
 * window_menu_bookmarks_dock_activate_cb: hide/show the bookmarks dock
 */
void window_menu_bookmarks_dock_activate_cb (GtkCheckMenuItem *menuitem,
					     GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active)
	{
		bookmarks_editor_show_dock (window);
	}
	else
	{
		window_undock (window);
	}
}

/** 
 * window_menu_history_dock_activate_cb: hide/show the history dock
 */
void window_menu_history_dock_activate_cb (GtkCheckMenuItem *menuitem,
					   GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active)
	{
		history_show_dock (window);
	}
	else
	{
		window_undock (window);
	}
}

/** 
 * window_menu_fullscreen_activate_cb: set fullscreen mode
 *
 */
void window_menu_fullscreen_activate_cb (GtkCheckMenuItem *menuitem,
					 GaleonWindow *window)
{
	return_if_not_window (window);
	window_set_fullscreen_mode (window, menuitem->active);
}

void
window_menu_own_colors_activate_cb (GtkCheckMenuItem *menuitem,
				    GaleonWindow *window)
{
	return_if_not_window (window);
	preferences_set_use_own_colors (menuitem->active);
}

void
window_menu_enable_javascript_activate_cb (GtkCheckMenuItem *menuitem,
					   GaleonWindow *window)
{
	return_if_not_window (window);
	preferences_set_enable_javascript (menuitem->active);
}

void
window_menu_enable_proxy_activate_cb (GtkCheckMenuItem *menuitem,
                                           GaleonWindow *window)
{
        return_if_not_window (window);
        preferences_set_enable_proxy (menuitem->active);
}


void
window_menu_own_fonts_activate_cb (GtkCheckMenuItem *menuitem,
				   GaleonWindow *window)
{
	return_if_not_window (window);
	preferences_set_use_own_fonts (menuitem->active);
}

void
window_menu_enable_java_activate_cb (GtkCheckMenuItem *menuitem,
				     GaleonWindow *window)
{
	return_if_not_window (window);
	preferences_set_enable_java (menuitem->active);
}

void
window_menu_images_never_activate_cb (GtkCheckMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		preferences_set_image_loading (IMAGE_LOADING_NEVER);
	}
}

void
window_menu_images_current_server_activate_cb (GtkCheckMenuItem *menuitem,
					       GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		preferences_set_image_loading
			(IMAGE_LOADING_FROM_CURRENT_SERVER_ONLY);
	}
}

void
window_menu_images_always_activate_cb (GtkCheckMenuItem *menuitem,
				       GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		preferences_set_image_loading (IMAGE_LOADING_ALWAYS);
	}
}

void
window_block_images_from_activate_cb (GtkMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	mozilla_set_permission (window->active_embed, FALSE, IMAGEPERMISSION);
}

void
window_allow_images_from_activate_cb (GtkMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	mozilla_set_permission (window->active_embed, TRUE, IMAGEPERMISSION);
}

void
window_view_stored_sites_activate_cb (GtkMenuItem *menuItem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	cookies_show_page (IMAGESITES);
}

void
window_menu_animate_always_activate_cb (GtkCheckMenuItem *menuitem,
					GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		preferences_set_animate (ANIMATE_ALWAYS);
	}
}

void
window_menu_animate_once_through_activate_cb (GtkCheckMenuItem *menuitem,
					      GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		preferences_set_animate (ANIMATE_ONCE_THROUGH);
	}
}

void
window_menu_animate_never_activate_cb (GtkCheckMenuItem *menuitem,
				       GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		preferences_set_animate (ANIMATE_NEVER);
	}
}

void
window_cookie_manager_launch_cb (GtkMenuItem *menuitem,
				 GaleonWindow *window)
{
	return_if_not_window (window);
	cookies_show_manager ();
}

void
window_block_cookies_from_activate_cb (GtkMenuItem *menuitem,
				       GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	mozilla_set_permission (window->active_embed, FALSE, COOKIEPERMISSION);
}

void
window_allow_cookies_from_activate_cb (GtkMenuItem *menuitem,
				       GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	mozilla_set_permission (window->active_embed, TRUE, COOKIEPERMISSION);	
}

void
window_menu_zoom_in_activate_cb (GtkCheckMenuItem *menuitem,
				 GaleonWindow *window)
{
	return_if_not_window (window);
	gtk_spin_button_spin (GTK_SPIN_BUTTON (window->zoom_spin),
			      GTK_SPIN_STEP_FORWARD, 10); 
}

void
window_menu_zoom_100_activate_cb (GtkCheckMenuItem *menuitem,
				 GaleonWindow *window)
{
	return_if_not_window (window);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (window->zoom_spin), 100);
}

void
window_menu_zoom_out_activate_cb (GtkCheckMenuItem *menuitem,
				  GaleonWindow *window)
{
	return_if_not_window (window);
	gtk_spin_button_spin (GTK_SPIN_BUTTON (window->zoom_spin),
			      GTK_SPIN_STEP_BACKWARD, 10); 
}

void 
window_menu_encoding_activate_cb (GtkWidget *menuitem,
				  GaleonWindow *window)
{
	char *menulabel;

	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	gtk_label_get (GTK_LABEL(GTK_BIN(menuitem)->child), &menulabel);
	mozilla_force_character_set (window->active_embed,
				     g_hash_table_lookup (charsets,
							  menulabel));
	embed_reload (window->active_embed);
}

void
window_view_passwords_activate_cb (GtkMenuItem *menuItem,
			           GaleonWindow *window)
{
	return_if_not_window (window);
	cookies_show_page (PASSWORDS);
}
void
window_view_password_sites_activate_cb (GtkMenuItem *menuItem,
				        GaleonWindow *window)
{
	return_if_not_window (window);
	cookies_show_page (PASSWORDSITES);
}

void window_menu_offline_mode_activate_cb (GtkCheckMenuItem *menuitem,
					   GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	mozilla_set_offline(window->active_embed, menuitem->active);
}
