/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "misc.h"
#include "embed.h"
#include "bookmarks.h"

#include <string.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtktable.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

/* local function prototypes */
void new_user_druid_finish_cb (GtkWidget *widget, gpointer dummy);
void new_user_druid_cancel_cb (GtkWidget *widget, gpointer dummy);
gboolean new_user_druid_delete_event_cb (GtkWidget *widget, GdkEventAny *event,
					 gpointer dummy);
void new_user_druid_create_smart_bookmarks_check_button_toggled_cb 
     (GtkWidget *toggle);
static void add_smart (BookmarkItem *parent, gchar *name, gchar *nick,
		       gchar *url, gchar *pixmap_file);

/**
 * new_user_druid_finish_cb: called when the user gets to the end of
 * the introductory druid and clicks on finish.
 */
void
new_user_druid_finish_cb (GtkWidget *widget, gpointer dummy)
{
	static const gchar *galeon_show = "galeon \"%s\"";
	GtkWidget *dialog;
	gint i;

	/* values fetched from widgets */
	gboolean import_netscape_bookmarks_checked;
	gboolean import_mozilla_bookmarks_checked;
	gboolean import_netscape_preferences_checked;
	gboolean create_smart_bookmarks_checked;
	gboolean google_web_checked;
	gboolean google_groups_checked;
	gboolean dogpile_checked;
	gboolean yahoo_checked;
	gboolean merriam_websters_checked;
	gboolean freshmeat_checked;
	gboolean http_checked;
	gboolean https_checked;
	gboolean ftp_checked;
	gboolean file_checked;
	gchar *http_proxy_site;
	gchar *ssl_proxy_site;
	gchar *ftp_proxy_site;
	gint http_proxy_port;
	gint ssl_proxy_port;
	gint ftp_proxy_port;

	/* lookup table for check buttons */
	const struct { gchar *name; gboolean *state; } check_lookups[] = {
        { "new_user_druid_import_netscape_bookmarks_radio_button",
	  &import_netscape_bookmarks_checked, },
        { "new_user_druid_import_mozilla_bookmarks_radio_button",
	  &import_mozilla_bookmarks_checked, },
        { "new_user_druid_import_netscape_preferences_check_button",
	  &import_netscape_preferences_checked, },
	{ "new_user_druid_create_smart_bookmarks_check_button",
	  &create_smart_bookmarks_checked, },
	{ "new_user_druid_google_web_check_button",
	  &google_web_checked, },
	{ "new_user_druid_google_groups_check_button",
	  &google_groups_checked, },
	{ "new_user_druid_dogpile_check_button",
	  &dogpile_checked, },
	{ "new_user_druid_yahoo_check_button",
	  &yahoo_checked, },
	{ "new_user_druid_merriam_websters_check_button",
	  &merriam_websters_checked, },
	{ "new_user_druid_freshmeat_check_button",
	  &freshmeat_checked, },
	{ "new_user_druid_http_check_button",
	  &http_checked, },
	{ "new_user_druid_https_check_button",
	  &https_checked, },
	{ "new_user_druid_ftp_check_button",
	  &ftp_checked, },
	{ "new_user_druid_file_check_button",
	  &file_checked, },
	{ NULL, NULL }/* terminator, must be last */
        };

	/* lookup table for string entry fields */
	const struct { gchar *name; gchar **value; } edit_lookups[] = {
		{ "new_user_http_proxy_site", &http_proxy_site },
		{ "new_user_ssl_proxy_site",  &ssl_proxy_site },
		{ "new_user_ftp_proxy_site",  &ftp_proxy_site },
		{ NULL, NULL }/* terminator, must be last */
	};

	/* lookup table for spin entrys */
	const struct { gchar *name; gint *value; } spin_lookups[] = {
		{ "new_user_http_proxy_port", &http_proxy_port },
		{ "new_user_ssl_proxy_port",  &ssl_proxy_port },
		{ "new_user_ftp_proxy_port",  &ftp_proxy_port },
		{ NULL, NULL }/* terminator, must be last */
	};

	/* lookup all the widgets in the list */
	for (i = 0; check_lookups[i].name != NULL; i++)
	{
		GtkWidget *check;

		/* find the checkbutton widget */
		check = glade_lookup_widget (widget, check_lookups[i].name);

		/* get it's state and store it */
		*(check_lookups[i].state) = gtk_toggle_button_get_active
			(GTK_TOGGLE_BUTTON (check));
	}

	/* lookup all the widgets in the list */
	for (i = 0; edit_lookups[i].name != NULL; i++)
	{
		GtkWidget *edit;

		/* find the entry widget */
		edit = glade_lookup_widget (widget, edit_lookups[i].name);

		/* get all the text into a new string */
		*(edit_lookups[i].value) = 
			gtk_editable_get_chars (GTK_EDITABLE (edit), 0, -1);
	}

	/* lookup all the widgets in the list */
	for (i = 0; spin_lookups[i].name != NULL; i++)
	{
		GtkWidget *spin;

		/* find the spinbutton widget */
		spin = glade_lookup_widget (widget, spin_lookups[i].name);

		/* get the value as an int */
		*(spin_lookups[i].value) = 
			gtk_spin_button_get_value_as_int 
			(GTK_SPIN_BUTTON (spin));
	}

	/* get preferences from netscape */
	if (import_netscape_preferences_checked)
	{
		netscape_import_prefs ();
	}

	/* create smart bookmarks toolbar */
	if (create_smart_bookmarks_checked)
	{
		BookmarkItem *toolbar;

		/* create the toolbar */
		toolbar = bookmarks_new_bookmark 
			(BM_CATEGORY, TRUE, _("Smart Bookmarks Toolbar"),
			 NULL, NULL, NULL, NULL);
		toolbar->create_toolbar = TRUE;

		/* add it into the root folder */
		bookmarks_root->list = 
			g_list_append (bookmarks_root->list, toolbar);
		toolbar->parent = bookmarks_root;

		/* add each smart bookmark checked */
		if (google_web_checked)
		{
			add_smart (toolbar, "Google", "google",
				   "http://www.google.com/search?q=%s",
				   SHARE_DIR "/google.png");
		}
		if (google_groups_checked)
		{
			add_smart (toolbar, "Usenet", "news",
				   "http://groups.google.com/groups?q=%s"
				   "&meta=site%3Dgroups", NULL);
		}
		if (dogpile_checked)
		{
			add_smart (toolbar, "Dogpile", "dogpile",
				   "http://search.dogpile.com/texis/search"
				   "?q=%s&geo=no&fs=web&av=custom&"
				   "engines=goto&engines=looksmart&"
				   "engines=thunderstone&engines=findwhat&"
				   "engines=sprinks&engines=directhit&"
				   "engines=google&engines=infoseek&"
				   "engines=lycos&engines=kanoodle&"
				   "engines=opendir&engines=realnames&"
				   "engines=altavista&engines=yahoo",
				   NULL);
		}
		if (yahoo_checked)
		{
			add_smart (toolbar, "Yahoo!", "yahoo",
				   "http://search.yahoo.com/bin/search?p=%s",
				   NULL);
		}
		if (merriam_websters_checked)
		{
			/* URL marked as translatable so that non-English
			 * dictionaries can be used -- don't forget to
			 * make the equivalent change in the glade file */
			add_smart (toolbar, _("Dictionary"), "dict",
				   /* xgettext: no-c-format */
				   _("http://www.m-w.com/cgi-bin/"
				     "dictionary?va=%s"), NULL);
		}
		if (freshmeat_checked)
		{
			add_smart (toolbar, "fm ][", "fm",
				   "http://freshmeat.net/search?q=%s", NULL);
		}
	}

	/* import bookmarks */
	if (import_netscape_bookmarks_checked)
	{
  	        /* FIXME: dont use this callback directly! */
	        bookmarks_import_netscape_activate_cb (NULL, NULL);
	}
	if (import_mozilla_bookmarks_checked)
	{
  	        /* FIXME: dont use this callback directly! */
	        bookmarks_import_mozilla_activate_cb (NULL, NULL);

#ifdef ENABLE_NLS
		/* FIXME: right thing in the right place? */
		mozilla_prefs_import_language ();
#endif
	}

	/* set URL handlers as requested */
	gnome_config_push_prefix ("/Gnome/URL Handlers/");
	if (http_checked)
	{
		gnome_config_set_string ("http-show", galeon_show);
	}
	if (https_checked)
	{
		gnome_config_set_string ("https-show", galeon_show);
	}
	if (ftp_checked)
	{
		gnome_config_set_string ("ftp-show", galeon_show);
	}
	if (file_checked)
	{
		gnome_config_set_string ("file-show", galeon_show);
	}
	gnome_config_pop_prefix ();

	/* setup proxies */
	gnome_config_push_prefix("/galeon/Advanced/");
	if (http_proxy_site != NULL && strlen (http_proxy_site) > 0)
	{
		gnome_config_set_bool ("proxy_enabled", TRUE);
		gnome_config_set_string ("http_proxy", http_proxy_site);
		gnome_config_set_string ("ssl_proxy",  ssl_proxy_site);
		gnome_config_set_string ("ftp_proxy",  ftp_proxy_site);
		gnome_config_set_int ("http_proxy_port", http_proxy_port);
		gnome_config_set_int ("ssl_proxy_port",  ssl_proxy_port);
		gnome_config_set_int ("ftp_proxy_port",  ftp_proxy_port);
	}
	gnome_config_pop_prefix ();
	
	/* kill the dialog */
	dialog = glade_lookup_widget (widget, "new_user_druid");
	gtk_widget_destroy (GTK_WIDGET (dialog));

	/* save all the initial settings */
	gnome_config_sync ();
	bookmarks_save ();

	/* create an initial window */
	embed_create_default (NULL, TRUE);
}

/**
 * new_user_druid_cancel_cb: called when the user cancels the druid
 * using the buttons provided.
 */
void
new_user_druid_cancel_cb (GtkWidget *widget, gpointer dummy)
{
	GtkWidget *dialog;

	/* kill the dialog */
	dialog = glade_lookup_widget (widget, "new_user_druid");
	gtk_widget_destroy (GTK_WIDGET (dialog));

	/* create an initial window */
	embed_create_default (NULL, TRUE);
}

/**
 * new_user_druid_delete_event_cb: called when the user closes the 
 * druid with a window manager delete event.
 */
gboolean
new_user_druid_delete_event_cb (GtkWidget *widget, GdkEventAny *event,
				gpointer dummy)
{
	/* create an initial window */
	embed_create_default (NULL, TRUE);

	/* still destroy the druid window */
	return FALSE;
}

/**
 * new_user_druid_create_smart_bookmarks_check_button_toggled_cb: turn a list
 * of options on and off when the user toggles a master switch.
 */
void
new_user_druid_create_smart_bookmarks_check_button_toggled_cb
(GtkWidget *toggle)
{
	GtkWidget *options_table;
	gboolean sensitive;
	GList *l;
	
	/* find the new state */
	sensitive = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggle));

	/* find the table containing the list of options */
	options_table = 
		glade_lookup_widget (GTK_WIDGET (toggle), 
				     "new_user_druid_smart_bookmarks_list");

	/* iterate over it's children (each is an check button) */
	l = GTK_TABLE (options_table)->children;
	for (; l != NULL; l = g_list_next (l))
	{
		GtkTableChild *child = (GtkTableChild *)(l->data);
		gtk_widget_set_sensitive (GTK_WIDGET (child->widget),
					  sensitive);
	}
}

/**
 * add_smart: add a bookmark to the smart bookmarks toolbar
 */
static
void add_smart (BookmarkItem *parent, gchar *name, gchar *nick, 
		gchar *url, gchar *pixmap_file)
{
	BookmarkItem *site;

	/* create the bookmarks */
	site = bookmarks_new_bookmark (BM_SITE, TRUE, name, url, nick, NULL, 
				       pixmap_file);

	/* add it to the toolbar */
	parent->list = g_list_append (parent->list, site);
	site->parent = parent;
}
