/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "window.h"
#include "embed.h"
#include "dialog.h"
#include "misc.h"
#include "menubar.h"
#include "toolbar.h"
#include "history.h"
#include "bookmarks.h"
#include "prefs.h"
#include "find.h"

#include <string.h>
#include <time.h>

#include <gtk/gtkdnd.h>
#include <gtk/gtkpaned.h>
#include <gtkmozembed.h>

/* GNOME includes */
#include <libgnomeui/gnome-animator.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-winhints.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-window-icon.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

/* indicates a window is in fullscreen mode */
gboolean fullscreen_active = FALSE;

/* local function prototypes */
static void window_init_data (GaleonWindow *window);
static void window_set_menu_data (GaleonWindow *window);
static void window_set_chrome (GaleonWindow *window, guint32 chrome_mask);
static void window_menubar_create (GaleonWindow *window);
static void window_toolbars_create (GaleonWindow *window);
static void window_set_selection_signals (GaleonWindow *window);
static void window_create_embed_container (GaleonWindow *window);
static void window_set_chrome (GaleonWindow *window, guint32 chrome_mask);
static void window_init_menu_data (GaleonWindow *window);
static void window_statusbar_create (GaleonWindow *window);

/* clallbacks used here */
void file_open_ok_button_clicked_cb (const gchar *file, gpointer data);

/** The global list of all GaleonWindow structures */
GList *all_windows = NULL;

/* the types which can be dropped either into the GtkMozEmbed or onto
 * the location entry field */
const GtkTargetEntry drop_types[] =
{
	{ "GALEON_URL",    0, DND_TARGET_GALEON_URL    },
	{ "text/uri-list", 0, DND_TARGET_TEXT_URI_LIST },
	{ "_NETSCAPE_URL", 0, DND_TARGET_NETSCAPE_URL  },
	{ "STRING",        0, DND_TARGET_STRING        }
};
const gint drop_types_num_items = (sizeof (drop_types) / 
				   sizeof (GtkTargetEntry));

/* the types which can be dragged from various places in Galeon, such as
 * the bookmarks editor, the toolbar bookmark icon, and the history */
const GtkTargetEntry url_drag_types[] = 
{
	{ "GALEON_BOOKMARK", 0, DND_TARGET_GALEON_BOOKMARK },
	{ "GALEON_URL",      0, DND_TARGET_GALEON_URL      },
	{ "_NETSCAPE_URL",   0, DND_TARGET_NETSCAPE_URL    },
	{ "STRING",          0, DND_TARGET_STRING          }
};
const gint url_drag_types_num_items = (sizeof (url_drag_types) /
				       sizeof (GtkTargetEntry));

typedef enum
{
	STRING,
	COMPOUND_TEXT
} SelType;

/* selection types */
const GtkTargetEntry selection_types[] =
{
	{ "STRING",        0, STRING        },
	{ "COMPOUND_TEXT", 0, COMPOUND_TEXT }
};
const gint selection_types_num_targets = (sizeof (selection_types) /
					  sizeof (GtkTargetEntry));


/**
 * window_load_url: load a URL into the active embed of a window
 */
void
window_load_url (GaleonWindow *window, const gchar *url)
{
	/* check arguments */
	return_if_not_window (window);

	/* load into the active GaleonEmbed */
	embed_load_url (window->active_embed, url);
}

void window_clear_url_entry (GaleonWindow *window)
{
	if (window->toolBarOn)
	{	
		gtk_editable_delete_text (GTK_EDITABLE (window->toolbar_entry),
					  0, -1);
	}
}

void window_set_url_entry (GaleonWindow *window, char *location)
{
	if (window->toolBarOn)
	{
		gtk_entry_set_text(GTK_ENTRY (window->toolbar_entry), 
				   location);
	}
}

/**
 * window_create: create a browser structure and main window, 
 * but without an embedding widget. Should only be called from
 * embed_create
 */
GaleonWindow *
window_create (guint32 chrome_mask)
{
        GaleonWindow *window;
	gchar *file;

	/* allocate and initialise the GaleonWindow structure */
	window = g_new0 (GaleonWindow, 1);

	/* set magic */
	window->magic = GALEON_WINDOW_MAGIC;

	window_set_chrome (window, chrome_mask);

	/* create the browser window */
	window->WMain = gnome_app_new ("galeon", _("Galeon"));
	gtk_window_set_wmclass (GTK_WINDOW (window->WMain), "",
				"galeon_browser");
	gtk_object_set_data (GTK_OBJECT (window->WMain), "WMain",
			     window->WMain);

	/* init dialogs data */
	window->print_dialog = NULL;

	/* initialize GaleonWindow data */
	window_init_data (window);

	/* create the statusbar */
	if (window->statusBarOn)
	{
		window_statusbar_create (window);
	}

	window->dock = GNOME_APP (window->WMain)->dock;

	/* save the toolbar and menubar layout */
	window_save_layout (window);

	/* create the toolbar */
	if (window->toolBarOn)
	{
		window_toolbars_create (window);
	}

	/* create the menubar */
	if (window->menuBarOn)
	{
		window_menubar_create (window);
	}

	/* restore the toolbar and menubar layout */
	window_restore_layout (window);

	/* set mini icon */
	file = gnome_pixmap_file ("galeon.png");
	gnome_window_icon_set_from_file (GTK_WINDOW (window->WMain), file);
	g_free (file);

	/* so widgets can look the data up */
	gtk_object_set_data (GTK_OBJECT (window->WMain), "GaleonWindow",
			     window);

	/* connect the delete signal handler*/
	gtk_signal_connect(GTK_OBJECT(window->WMain), "delete-event",
			   GTK_SIGNAL_FUNC(window_delete_cb),
			   window);

	/* add it to the list of windows */
	all_windows = g_list_prepend (all_windows, window);

	window_set_selection_signals (window);

	window_create_embed_container (window);

	/* NB: window isn't shown until we add an embed */
	/* don't know the size at which to show it yet -- if we discover
	 * a size (e.g. for a popup) then we'll use that, otherwise we'll
	 * find a default, but not until the window is made visible */
	window->set_size = FALSE;

	/* set window policies regarding resizing */
	gtk_window_set_policy (GTK_WINDOW (window->WMain), TRUE, TRUE, TRUE);

	/* return the completed GaleonWindow */
	return window;
}

/**
 * window_init_data: init GaleonWindow data structure
 */
static void
window_init_data (GaleonWindow *window)
{
	/* these are all nullified */
	window->spinner = NULL;
	window->tempMessage = NULL;
	window->bookmarks_toolbars = NULL;
	window->bookmarks_tooltips = NULL;
}

static void
window_init_menu_data (GaleonWindow *window)
{
	int pos; //A temp hack for backward compatibility
	pos = 5;

	window->view_menubar = view_menu_uiinfo[2].widget;
	window->view_toolbar = view_menu_uiinfo[3].widget;
	window->view_statusbar = view_menu_uiinfo[4].widget;
	window->view_bookmarks_dock = view_menu_uiinfo[5].widget;
	window->view_history_dock = view_menu_uiinfo[6].widget;
	window->view_fullscreen = view_menu_uiinfo[7].widget;
	window->load_images_always = load_images_always_uiinfo[0].widget;
	window->load_images_from_current_server_only = 
					load_images_always_uiinfo[1].widget;
	window->load_images_never = load_images_always_uiinfo[2].widget;
	window->animate_always = animate_always_uiinfo[0].widget;
	window->animate_once_through = animate_always_uiinfo[1].widget;
	window->animate_never = animate_always_uiinfo[2].widget;
	window->use_own_fonts =  settings_menu_uiinfo[pos++].widget;
	window->use_own_colors = settings_menu_uiinfo[pos++].widget;
	window->enable_java = settings_menu_uiinfo[pos++].widget;
	window->enable_javascript = settings_menu_uiinfo[pos++].widget;
	window->enable_proxy = settings_menu_uiinfo[pos++].widget;

	gtk_widget_ref(menubar_uiinfo[4].widget);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain), "bookmarks",
					    menubar_uiinfo[4].widget,
					   (GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(GTK_MENU_ITEM(menubar_uiinfo[4].widget)->submenu);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain), "bookmarks_menu",
			GTK_MENU_ITEM(menubar_uiinfo[4].widget)->submenu,
			(GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(GTK_MENU_ITEM(menubar_uiinfo[2].widget)->submenu);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain), "view_menu",
			GTK_MENU_ITEM(menubar_uiinfo[2].widget)->submenu,
			(GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(GTK_MENU_ITEM(menubar_uiinfo[5].widget)->submenu);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain), "go_menu",
			GTK_MENU_ITEM(menubar_uiinfo[5].widget)->submenu,
			(GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(bookmarks_menu_uiinfo[0].widget);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain), "file_bookmark",
				 bookmarks_menu_uiinfo[0].widget,
				 (GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(bookmarks_menu_uiinfo[3].widget);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain),
				 "bookmarks_separator",
				 bookmarks_menu_uiinfo[3].widget,
				 (GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(view_menu_uiinfo[18].widget);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain),
				 "encoding_menu_item",
				 view_menu_uiinfo[18].widget,
				 (GtkDestroyNotify) gtk_widget_unref);
}

static void
window_menubar_create (GaleonWindow *window)
{
	GtkWidget *bookmarks_item;

	g_assert (window->WMain);

	/* need to set the user_data field in all the GnomeUIInfo structs */
	window_set_menu_data (window);

	gnome_app_create_menus (GNOME_APP(window->WMain), 
				menubar_uiinfo);

	window->menubar = GNOME_APP(window->WMain)->menubar;

	window_init_menu_data (window);

	gnome_app_install_menu_hints (GNOME_APP(window->WMain), 
				      menubar_uiinfo);
	bookmarks_menu_create (window);

	/* set bookmark menuitem drop destination */
	bookmarks_item = gtk_object_get_data(GTK_OBJECT(window->WMain),
					     "bookmarks");
	gtk_drag_dest_set (bookmarks_item, GTK_DEST_DEFAULT_ALL,
			   url_drag_types, url_drag_types_num_items,
			   GDK_ACTION_COPY);
	gtk_signal_connect (GTK_OBJECT(bookmarks_item), "drag_data_received",
			    GTK_SIGNAL_FUNC
			    (bookmarks_menuitem_drag_data_received_cb),
			    window);

	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_menubar), TRUE);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_toolbar), TRUE);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_statusbar), TRUE);

	/* populate the encoding submenu */
	if (!(window->encodings_menu_set))
	{
		window_set_encoding_menu (window, sorted_charset_titles);
		window->encodings_menu_set = TRUE;
	}

	window_menu_edit_add_events(window);

	/* setup view menu */
	set_settings_menu_window (window);
}

static void
window_toolbars_create (GaleonWindow *window)
{
	/* create the toolbars */
	toolbar_create (window);
	bookmarks_toolbars_create (window);
}

void
window_save_layout (GaleonWindow *window)
{
	g_return_if_fail (window->layout == NULL);

	/* get the saved layout string and store it in the window structure */
	window->layout = gnome_config_get_string ("/galeon/Placement/Dock");

	/* make sure there's a layout to build into */
	if (GNOME_APP (window->WMain)->layout == NULL)
	{
		GNOME_APP (window->WMain)->layout = gnome_dock_layout_new ();
	}
}

void
window_restore_layout (GaleonWindow *window)
{
	GnomeApp *app;
	GList *lp;
	
	/* get the app */
	app = GNOME_APP (window->WMain);

	/* restore placements of recreated items */
	/* have to do this manually to get the ordering right, since
	 * we don't want to disturb things which we haven't recreated,
	 * such as the menu bar */
	gnome_dock_layout_parse_string (app->layout, window->layout);
	for (lp = app->layout->items; lp != NULL; lp = g_list_next (lp))
	{
		GnomeDockLayoutItem *item = (GnomeDockLayoutItem *)(lp->data);
		
		if (item->placement == GNOME_DOCK_FLOATING)
		{
			gnome_dock_add_floating_item
				(GNOME_DOCK (app->dock),
				 item->item,
				 item->position.floating.x,
				 item->position.floating.y,
				 item->position.floating.orientation);
		}
		else
		{
			gnome_dock_add_item
				(GNOME_DOCK (app->dock),
				 item->item,
				 item->placement,
				 item->position.docked.band_num,
				 0,
				 item->position.docked.offset,
				 TRUE);
		}
	}

	/* get rid of the layout object, it's no longer needed */
	gtk_object_unref (GTK_OBJECT (app->layout));
	app->layout = NULL;

	/* free the layout string */
	g_free (window->layout);
	window->layout = NULL;
}

void
window_toolbars_recreate (GaleonWindow *window)
{
	/* skip this window if no toolbars */
	if (!(window->toolBarOn))
	{
		return;
	}

	/* save layout of window */
	window_save_layout (window);
	
	/* recreate the main toolbar */
	toolbar_recreate (window);

	/* recreate the bookmarks toolbars for this window */
	bookmarks_toolbars_recreate (window);

	/* restore the layout */
	window_restore_layout (window);
}

static void
window_statusbar_create (GaleonWindow *window)
{
	gnome_app_set_statusbar (
		GNOME_APP(window->WMain),
		gnome_appbar_new (TRUE, TRUE,
				  GNOME_PREFERENCES_NEVER));
	window->appbar = GNOME_APP(window->WMain)->statusbar;
}

static void
window_set_selection_signals (GaleonWindow *window)
{
        /* set selection signal */
	gtk_selection_add_targets (GTK_WIDGET(window->WMain),
				   GDK_SELECTION_PRIMARY, 
				   selection_types, 
				   selection_types_num_targets);
	gtk_selection_add_targets (GTK_WIDGET(window->WMain),
				   gdk_atom_intern("CLIPBOARD", FALSE),
				   selection_types, 
				   selection_types_num_targets);
	gtk_signal_connect (GTK_OBJECT(window->WMain), "selection_received",
			    GTK_SIGNAL_FUNC (window_selection_received_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT(window->WMain), "selection_get",
			    GTK_SIGNAL_FUNC (window_selection_get_cb),
			    NULL);
}

static void
window_create_embed_container (GaleonWindow *window)
{
	/* make the toplevel notebobok */
	window->notebook = gtk_notebook_new ();

	/* set some notebook properties */
	gtk_notebook_popup_enable (GTK_NOTEBOOK (window->notebook));
	gtk_notebook_set_scrollable (GTK_NOTEBOOK (window->notebook), TRUE);
	gtk_signal_connect_after (GTK_OBJECT (window->notebook), "switch_page",
				  embed_notebook_switch_page_cb, window);
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK (window->notebook),
		gnome_config_get_int (CONF_APPEARANCE_TABBED_POSITION));
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (window->notebook),
		gnome_config_get_int (CONF_APPEARANCE_TABBED_ALWAYS_SHOW));
	gtk_notebook_set_show_border (GTK_NOTEBOOK (window->notebook), FALSE);
	gtk_widget_show (GTK_WIDGET (window->notebook));

	/* make the toplevel hpaned, for docks */
	window->hpaned = gtk_hpaned_new ();
	gtk_paned_set_gutter_size (GTK_PANED (window->hpaned), 12);
	gtk_paned_set_handle_size (GTK_PANED (window->hpaned), 10);

	/* make a topleve hbox for inserting the paned into */
	window->hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (window->hbox),
			    GTK_WIDGET (window->hpaned),
			    TRUE, TRUE, 0);
	gtk_box_pack_end (GTK_BOX (window->hbox),
			  GTK_WIDGET (window->notebook),
			  TRUE, TRUE, 0);

	/* insert hbox into toplevel window */
	gnome_app_set_contents (GNOME_APP (window->WMain), 
				GTK_WIDGET (window->hbox));
}

static void
window_set_chrome (GaleonWindow *window, guint32 chrome_mask)
{
	/* set the chrome info */
	window->menuBarOn = FALSE;
	window->toolBarOn = FALSE;
	window->locationBarOn = FALSE;
	window->statusBarOn = FALSE;

	if (chrome_mask & GTK_MOZ_EMBED_FLAG_MENUBARON)
	{
		window->menuBarOn = TRUE;
	}
	if (chrome_mask & GTK_MOZ_EMBED_FLAG_TOOLBARON)
	{
		window->toolBarOn = TRUE;
	}
	if (chrome_mask & GTK_MOZ_EMBED_FLAG_LOCATIONBARON)
	{
		window->locationBarOn = TRUE;
	}
	if (chrome_mask & GTK_MOZ_EMBED_FLAG_STATUSBARON)
	{
		window->statusBarOn = TRUE;
	}
}

/**
 * window_update_status_bar: update the status bar of the toplevel window
 */
void
window_update_status_bar (GaleonWindow *window)
{
	GaleonEmbed *embed;
	GtkWidget *progress;
	const gchar *status;
	gchar message[256];

	return_if_not_window (window);
	embed = window->active_embed;

	/* this may be called before any embed is activate */
	if (embed == NULL)
	{
		return;
	}

	/* still check the magic number */
	return_if_not_embed (embed);

	/* check if the statusbar exist */
	if (!window->statusBarOn) return;

	/* get progress bar widget */
	progress = GTK_WIDGET (gnome_appbar_get_progress 
			       (GNOME_APPBAR(window->appbar)));

	/* have we started loading? */
	if (embed->load_started > 0 && embed->bytesLoaded > 0)
	{
		if (embed->loadPercent == 0) 
		{
			/* not really, go into activity mode */
			gtk_progress_set_activity_mode 
				(GTK_PROGRESS (progress), TRUE);
			if (!(window->progress_timeout))
			{
				window->progress_timeout = TRUE;
				g_timeout_add (100, (GSourceFunc)
					       window_progress_action, 
					       window);
			}
		} 
		else
		{
			/* yes, show the progress in progress bar */
			window->progress_timeout = FALSE;
			gtk_progress_set_activity_mode 
				(GTK_PROGRESS (progress), FALSE);
			gnome_appbar_set_progress 
				(GNOME_APPBAR (window->appbar), 
				 embed->loadPercent / 100.0);
		}
	}
	else
	{
		/* go to sleep */
		window->progress_timeout = FALSE;
		gtk_progress_set_activity_mode (GTK_PROGRESS (progress),
						FALSE);
		gnome_appbar_set_progress (GNOME_APPBAR (window->appbar), 0);
	}

	/* get rid of last message */
	gnome_appbar_pop (GNOME_APPBAR (window->appbar));

	/* temporary message takes priority */
	if (window->tempMessage != NULL && (strlen (window->tempMessage) != 0))
	{
		gnome_appbar_push (GNOME_APPBAR (window->appbar),
				   window->tempMessage);
		return;
	}

	/* get a status message */
	status = (embed->statusMessage == NULL ? "" : embed->statusMessage);

	/* make a progress message */
	if (embed->bytesLoaded == 0)
	{
		g_snprintf (message, 255, "%s", status);
	}
	else if (embed->bytesLoaded <= embed->maxBytesLoaded)
	{
		gchar time_remaining[64] = "";
		gint hours, minutes, seconds;
		GTime time_delta;

		/* try to compute roughly the time remaining for the load */
		time_delta = (time (NULL) - embed->when_started);
		if (embed->loadPercent > 5 && 
		    embed->loadPercent < 100 && time_delta > 0)
		{
			time_delta *= ((double)embed->maxBytesLoaded / 
				       (double)embed->bytesLoaded) - 1.0;
			hours = time_delta / (60 * 60);
			time_delta %= (60 * 60);
			minutes = time_delta / 60;
			seconds = time_delta % 60;
			snprintf (time_remaining, 64,
				  ", %02d:%02d:%02d remaining",
				  hours, minutes, seconds);
		}
		
		/* fill the status bar text with progress info */
		g_snprintf (message, 255, 
			    _("%s (%d%% complete, %d kB of %d kB loaded%s)"),
			    status, 
			    embed->loadPercent, 
			    embed->bytesLoaded / 1024,
			    embed->maxBytesLoaded / 1024,
			    time_remaining);
	}
	else
	{
		/* fill the status bar text with progress info: only kb */
		g_snprintf (message, 255, _("%s (%d kB loaded)"), 
			    status, 
			    embed->bytesLoaded / 1024);
	}

	/* show the new message */
	gnome_appbar_push (GNOME_APPBAR (window->appbar), message);
}

/**
 * window_update_temp_message: update the temporary message
 */
void
window_update_temp_message(GaleonWindow *window, const char *message)
{
	/* check if the statusbar exist */
	if (!window->statusBarOn) return;

	/* free the previous message */
	if (window->tempMessage) 
	{
		g_free(window->tempMessage);
	}
	
	/* fill the new temporary message */
	if (message) 
	{
		window->tempMessage = g_strdup(message);
	}	
	else
	{	
		window->tempMessage = NULL;
	}
	window_update_status_bar (window);
}

/**
 * window_update_nav_controls: update back and forward toolbar buttons status,
 *                            and now go_menu menutiems
 */
void
window_update_nav_controls (GaleonWindow *window)
{
	gboolean can_go_back;
	gboolean can_go_up;
	gboolean can_go_forward;
	gboolean next_tab;
	gboolean prev_tab;
	gboolean can_stop;
	gboolean can_refresh;
	gint index, size, tearoff = 0;
	GaleonEmbed *embed;
	GtkWidget *menu;
	GList *children;

	if (!(window->toolBarOn || window->menuBarOn))
	{
		return;
	}
			
	/* get active embed */
	embed = window->active_embed;
		
	/* query mozilla / embed */
	can_go_back = gtk_moz_embed_can_go_back (GTK_MOZ_EMBED(embed->mozEmbed));
	can_go_up = embed_can_go_up (embed);
	can_go_forward = gtk_moz_embed_can_go_forward (GTK_MOZ_EMBED(embed->mozEmbed));
	can_stop = (embed->load_started > 0);
	can_refresh = (embed->load_started == 0);

	/* query next/prev tabs -- NOTE: embeds are in reverse order */
	index = gtk_notebook_page_num (GTK_NOTEBOOK (window->notebook),
				       GTK_WIDGET (embed->mozEmbed));
	size = g_list_length (GTK_NOTEBOOK (window->notebook)->children);
	prev_tab = (index > 0);
	next_tab = (index < (size - 1));
	
	if (window->menuBarOn)
	{
		/* gets gnome detachabilty of menus so we set sensitivity on
		   the right item */
		if (gnome_preferences_get_menus_have_tearoff ())
		{
			tearoff++;
		}
		
		/* get the go_menu widgets */
		menu = gtk_object_get_data 
			(GTK_OBJECT (window->WMain), "go_menu");
		children = gtk_container_children (GTK_CONTAINER(menu));
		
		/* set go_menu buttons */
		gtk_widget_set_sensitive (
			GTK_WIDGET(g_list_nth_data(children, 0 + tearoff)),
			can_go_back);
		gtk_widget_set_sensitive (
			GTK_WIDGET(g_list_nth_data(children, 1 + tearoff)),
			can_go_forward);
		gtk_widget_set_sensitive (
			GTK_WIDGET(g_list_nth_data(children, 2 + tearoff)),
			can_go_up);
		gtk_widget_set_sensitive (
			GTK_WIDGET(g_list_nth_data(children, 3 + tearoff)),
			can_refresh);
		gtk_widget_set_sensitive (
			GTK_WIDGET(g_list_nth_data(children, 5 + tearoff)),
			can_stop);
		/* free the list of children */
		g_list_free (children);

		/* get the view_menu widgets */
		menu = gtk_object_get_data 
			(GTK_OBJECT (window->WMain), "view_menu");
		children = gtk_container_children (GTK_CONTAINER(menu));
		
		gtk_widget_set_sensitive (
			GTK_WIDGET(g_list_nth_data(children, 13 + tearoff)),
			prev_tab);
		gtk_widget_set_sensitive (
			GTK_WIDGET(g_list_nth_data(children, 14 + tearoff)),
			next_tab);
		
		/* free the list of children */
		g_list_free (children);
	}

        /* check if the toolbar exist and then set buttons */
	if (window->toolBarOn) 
	{
		/* update back and forward buttons */
		gtk_widget_set_sensitive (GTK_WIDGET (window->BBack), 
					  can_go_back);
		gtk_widget_set_sensitive (GTK_WIDGET (window->BUp), 
				  can_go_up);
		gtk_widget_set_sensitive (GTK_WIDGET (window->BForward), 
					  can_go_forward);
		
		/* update stop and refresh toolbar buttons */
		gtk_widget_set_sensitive (GTK_WIDGET (window->BStop),
					  can_stop);
		gtk_widget_set_sensitive (GTK_WIDGET (window->BRefresh),
					  can_refresh);
	}
}

/**
 * window_update_zoom: update the zoom spin control
 */
void
window_update_zoom (GaleonWindow *window)
{
	if (window->zoom_spin != NULL && window->active_embed != NULL)
	{
		gtk_spin_button_set_value 
			(GTK_SPIN_BUTTON (window->zoom_spin),
			 (gfloat)(window->active_embed->zoom));
	}
}

/**
 * window_show_open_dialog: show the open dialog
 */
void window_show_open_dialog (GaleonWindow *window)
{
	gchar *dir;

	dir = gnome_config_get_string (CONF_DIR_OPEN);
	dialog_ask_file_name (_("Open a file"), _("Select the file to open:"), 
			      dir, file_open_ok_button_clicked_cb, NULL,
			      TRUE, FALSE, window);
	g_free (dir);	
}

/**
 * window_show_openurl_dialog: show the open url dialog
 */
void window_show_openurl_dialog (GaleonWindow *window)
{
	GladeXML *gxml;
	GtkWidget *dialog = NULL;
	GtkWidget *entry;
	extern gchar *open_url_str;

	gxml = glade_widget_new ("galeon.glade", "open_url_dialog", 
				 &dialog, window);
	entry = glade_xml_get_widget (gxml, "open_url_entry");

	g_return_if_fail (dialog != NULL);
	g_return_if_fail (entry != NULL);

	/* unref the xml object - we don't need it any more */
	gtk_object_unref (GTK_OBJECT (gxml));

	/* this is done so that the enter key is not captured by the entry */
	gnome_dialog_editable_enters (GNOME_DIALOG (dialog),
				      GTK_EDITABLE (entry));

	/* FIXME: is this really desirable? */
	if (open_url_str)
	{
		gtk_entry_set_text (GTK_ENTRY (entry), open_url_str);
	}

	gtk_object_set_data (GTK_OBJECT (window->WMain), "open_url_entry",
			     entry);

	gtk_window_set_transient_for (GTK_WINDOW (dialog),
				      GTK_WINDOW (window->WMain));
	window_set_layer (dialog);
	gtk_widget_show (dialog);
}

/**
 * window_toolbar_show: show the toolbar
 */
void window_toolbar_show (GaleonWindow *window)
{
	if (!(window->toolBarOn))
	{
		return;
	}

	gtk_widget_show (window->main_dockitem);	
	if (gnome_config_get_int ("/galeon/Appearance/url_location=1") == 0)
	{
		gtk_widget_show (window->location_dockitem);
	}
	bookmarks_toolbars_set_visibility (window, TRUE);
}

/**
 * window_toolbar_hide: hide the toolbar
 */
void window_toolbar_hide (GaleonWindow *window)
{
	if (window->toolBarOn) 
	{
		gtk_widget_hide (GTK_WIDGET (window->main_dockitem));
		if (window->location_dockitem)
		{
			gtk_widget_hide (GTK_WIDGET (window->location_dockitem));
		}
		bookmarks_toolbars_set_visibility (window, FALSE);
		gtk_widget_queue_resize (GTK_WIDGET (window->hbox));
	}
}

/**
 * window_menubar_show: show the menubar
 */
void window_menubar_show (GaleonWindow *window)
{
	if (window->menuBarOn) 
	{
		gtk_widget_show (GTK_WIDGET (window->menubar->parent));	
	}
}

/**
 * window_menubar_hide: hide the menubar
 */
void window_menubar_hide (GaleonWindow *window)
{
	if (window->menuBarOn) 
	{
		gtk_widget_hide (GTK_WIDGET (window->menubar->parent));
		gtk_widget_queue_resize (GTK_WIDGET (window->hbox));
	}
}

/**
 * window_statusbar_show: show the statusbar
 */
void window_statusbar_show (GaleonWindow *window)
{
	if (window->statusBarOn) 
	{
		gtk_widget_show (GTK_WIDGET (window->appbar->parent));
	}
}

/**
 * window_statusbar_hide: hide the statusbar
 */
void window_statusbar_hide (GaleonWindow *window)
{
	if (window->statusBarOn) {
		gtk_widget_hide (window->appbar->parent);
		gtk_widget_queue_resize (GTK_WIDGET (window->hbox));
	}
}

/**
 * window_close: close a GaleonWindow
 */
void 
window_close (GaleonWindow *window)
{
	GList *e, *next;

	return_if_not_window (window);

	/* hide the window, this make the app seem more responsive */
	gtk_widget_hide (window->WMain);

	/* stop the spinner if it exists; I'm not sure if this is essential
	 * but it's an attempt to workaround GnomeAnimator crashes -- MattA */
	if (window->spinner != NULL)
	{
		gnome_animator_stop (GNOME_ANIMATOR (window->spinner));
	}

	/* kill off find dialog if active */
	if (window->find_dialog != NULL)
	{
		find_destroy_dialog (window);
	}

	/* any children? */
	if (window->embed_list != NULL)
	{
		/* close any embed children */
		for (e = window->embed_list; e != NULL;)
		{
			GaleonEmbed *embed = (GaleonEmbed *)(e->data);
			return_if_not_embed (embed);
			next = g_list_next (e);
			embed_close (embed);
			e = next;
		}

		/* window_close will now be called again automatically */
	}
	else
	{
		/* remove window from list */
		all_windows = g_list_remove (all_windows, window);

		/* stop the activity bar */
		window->progress_timeout = FALSE;

		/* destroy the window, and all it contains */
		gtk_widget_destroy (window->WMain);

		/* quit if this was the last window to go and
		 * we're not in server mode */
		if (g_list_length (all_windows) == 0 && !galeon_server_mode)
		{
			galeon_exit ();
		}
	}
}

/**
 * window_go_home:
 */
void
window_go_home (GaleonWindow *window, gboolean new_embed, gboolean new_window)
{
	gchar *startpage;

	startpage = gnome_config_get_string 
		("/galeon/General/startpage=www.gnome.org");

        if (startpage != NULL && strlen(startpage) != 0)
        {
		if (new_embed)
		{
			embed_create_from_url (window->active_embed,
					       startpage, TRUE, new_window);
		}
		else
		{
			window_load_url (window, startpage);
		}
	}
        else
	{
		gnome_error_dialog (_("You must specify a Start Page "
				      "in the Preferences dialog!"));
	}

	g_free(startpage);
}

/**
 *  Request that the widget be placed above or below the Gnome panels
 */
void
window_set_layer (GtkWidget *widget)
{
	/* fullscreen mode flag */
	extern gboolean fullscreen_active;
	gboolean raise_windows;

	raise_windows = gnome_config_get_bool(
				"/galeon/Appearance/fullscreen_stacking=TRUE");

	if (fullscreen_active && raise_windows)
		gnome_win_hints_set_layer(widget, WIN_LAYER_ABOVE_DOCK);
	else
		gnome_win_hints_set_layer(widget, WIN_LAYER_NORMAL);

	gdk_window_raise(widget->window);
}

/**
 * populate the View/Encodings menu with the available charsets 
 */
void
window_set_encoding_menu (GaleonWindow *window, GList *charset_titles)
{
	GtkWidget *encoding_menu_item;

	encoding_menu_item = gtk_object_get_data (GTK_OBJECT(window->WMain),
						  "encoding_menu_item");

	if (GTK_MENU_ITEM(encoding_menu_item)->submenu == NULL)
	{
		embed_create_charset_submenus (GTK_MENU_ITEM(encoding_menu_item),
					       charset_titles, window);
	}
}

void
window_toggle_fullscreen_mode (GaleonWindow *window)
{
	window_set_fullscreen_mode (window, !fullscreen_active);
}

void
window_set_fullscreen_mode (GaleonWindow *window, gboolean active)
{
	GdkWindow *gdk_window = window->WMain->window;
	gint client_x, client_y, root_x, root_y;
	gint width, height;

	return_if_not_window (window);

	if (window->menuBarOn)
	{
		GTK_CHECK_MENU_ITEM (window->view_fullscreen)->active = active;

		if (!gnome_config_get_bool
		    ("/galeon/Appearance/show_menubar_in_fullscreen")) 
		{
			gtk_check_menu_item_set_active
				(GTK_CHECK_MENU_ITEM (window->view_menubar),
				 !active);
		}
		if (!gnome_config_get_bool
		    ("/galeon/Appearance/show_toolbars_in_fullscreen")) 
		{
			gtk_check_menu_item_set_active
				(GTK_CHECK_MENU_ITEM (window->view_toolbar),
				 !active);
		}
		if (!gnome_config_get_bool
		    ("/galeon/Appearance/show_statusbar_in_fullscreen"))
		{
			gtk_check_menu_item_set_active
				(GTK_CHECK_MENU_ITEM (window->view_statusbar),
				 !active);
		}
	}

	if (active)
	{
		fullscreen_active = TRUE;

		window_set_layer (window->WMain);

		gdk_window_get_origin (gdk_window, &root_x, &root_y);
		gdk_window_get_geometry (gdk_window, &client_x, &client_y,
					 &width, &height, NULL);

		gdk_window_move_resize (gdk_window, -client_x, -client_y,
					gdk_screen_width () + 1,
					gdk_screen_height () + 1);
		
		window->x = root_x - client_x;
		window->y = root_y - client_y;
		window->width = width;
		window->height = height;
	}
	else
	{
		/* Send an X state change request to place the window
		   back in the normal layer */
		gnome_win_hints_set_layer (window->WMain,
					   WIN_LAYER_NORMAL);

		gdk_window_move_resize (gdk_window, 
					window->x, window->y,
					window->width, 
					window->height);
		fullscreen_active = FALSE;
	}
}

static void 
window_set_menu_data (GaleonWindow *window)
{
	gint i, j;

	for (i = 0; i < num_menus; i++)
	{
		for (j = 0; j < menus_num_items[i]; j++)
		{
			all_menus[i][j].user_data = window;
		}
	}
}

/* Use this function instead of glade_lookup_widget() to find a widget
   you have attached to an object via gtk_object_set_data() from one
   of its child widgets. This is necessary because libglade is no longer
   being used to create the main window */
GtkWidget *
window_lookup_widget (GtkWidget *widget, const char *name)
{
	GtkWidget *found_widget = NULL;
	GtkWidget *parent = NULL;

	g_return_val_if_fail(widget != NULL, NULL);
	while (!found_widget)
	{
		found_widget = gtk_object_get_data (GTK_OBJECT(widget), name);

		if (GTK_IS_MENU(widget))
			parent = gtk_menu_get_attach_widget(GTK_MENU(widget));
		else
			parent = widget->parent;

		if (parent == NULL)
			break;

		widget = parent;
	}

	if (!found_widget)
		g_warning("%s not found.", name);

	return found_widget;
}

/**
 * window_reload_all: reload all tabs in a window
 */
void
window_reload_all (GaleonWindow *window)
{
	GList *l;

	/* check arguments */
	return_if_not_window (window);

	/* reload all tabs in the window */
	for (l = window->embed_list; l != NULL; l = g_list_next (l))
	{
		embed_reload ((GaleonEmbed *)(l->data));
	}
}

/**
 * window_undock: remove the current dock
 */
void
window_undock (GaleonWindow *window)
{
	/* check something is docked */
	if (window->docked == NULL)
	{
		return;
	}

	/* store width in pref */
	if (window->docked_width_pref != NULL && 
	    GTK_PANED (window->hpaned)->position_set)
	{
		gnome_config_set_int (window->docked_width_pref,
				      GTK_PANED (window->hpaned)->child1_size);
	}

	/* hide the hpaned and notebook */
	gtk_widget_hide (GTK_WIDGET (window->hpaned));
	gtk_widget_hide (GTK_WIDGET (window->notebook));
 
	/* remove the notebook from the hpaned and insert it into the box */
	gtk_widget_reparent (GTK_WIDGET (window->notebook),
			     GTK_WIDGET (window->hbox));

        /* destroy the docked widget */
	gtk_widget_destroy (GTK_WIDGET (window->docked));
	window->docked = NULL;

	/* show the notebook again */
	gtk_widget_show (GTK_WIDGET (window->notebook));
}

/**
 * window_dock: dock a widget into a window, destroying anything
 * that was there previously.
 */
void
window_dock (GaleonWindow *window, GtkWidget *widget, const gchar *width_pref)
{
	gint width;

	/* hide the notebook */
	gtk_widget_hide (GTK_WIDGET (window->notebook));

	/* destroy current dock */
	if (window->docked != NULL)
	{
		window_undock (window);
	}

	/* dock this */
	gtk_paned_add1 (GTK_PANED (window->hpaned), GTK_WIDGET (widget));
	window->docked = widget;
	window->docked_width_pref = width_pref;

	/* move pane to accomodate width */
	if (width_pref != NULL)
	{
		width = gnome_config_get_int (width_pref);
		if (width != -1)
		{
			gtk_paned_set_position (GTK_PANED (window->hpaned),
						width);
		}
	}

	/* show the paned view */
	gtk_widget_show (GTK_WIDGET (window->hpaned));

	/* move notebook into the other side of the paned */
	gtk_widget_reparent (GTK_WIDGET (window->notebook),
			     GTK_WIDGET (window->hpaned));

	/* show the notebook again */
	gtk_widget_show (GTK_WIDGET (window->notebook));
}
