/* vi:set ts=8 sts=0 sw=8:
 * $Id: shellcmd.c,v 1.15 2000/01/19 17:31:50 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "main.h"
#ifdef USE_SHELL_INSERT
#ifdef APP_GNP
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <gtk/gtk.h>
#include "dialog.h"
#include "win.h"
#include "doc.h"
#include "undo.h"
#include "prefs.h"
#include "misc.h"
#include "shellcmd.h"

#include "gnpintl.h"

/*** local types ***/
typedef struct {
	/* shell exec read dialog information */
	GtkWidget *toplev;
	GtkWidget *entry;
	win_t *w;
} serd_t;


/*** local function prototypes */
static void shell_exec_read_dstr(GtkWidget *wgt, gpointer cbdata);
static void shell_exec_read_cncl(GtkWidget *wgt, gpointer cbdata);
static void shell_exec_read_ok(GtkWidget *wgt, gpointer cbdata);
static void shell_exec_read_do(const char *cmd, doc_t *d);


/*** global function defintions ***/
/*
 * PUBLIC: shell_exec_read_dlg
 *
 * creates a dialog box to ask the user what command they want to execute in
 * the shell.  this command's output is then inserted directly into the current
 * document.
 */
void
shell_exec_read_dlg(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	serd_t *serdp;

	serdp = g_new(serd_t, 1);
	serdp->w = w;

	get_dialog_entry(_("Enter shell command"),
			 _("Enter shell command whose output will be inserted"),
			 _("Shell Command:"), 128, NULL,
			 GTK_SIGNAL_FUNC(shell_exec_read_ok), (gpointer)serdp,
			 GTK_SIGNAL_FUNC(shell_exec_read_cncl), (gpointer)serdp,
			 GTK_SIGNAL_FUNC(shell_exec_read_dstr), (gpointer)serdp,
			 &(serdp->toplev), &(serdp->entry));
} /* shell_exec_read_dlg */


/*** local function defintions ***/
/*
 * PRIVATE: shell_exec_read_dstr
 *
 * destroy callback for 'enter shell command' dialog
 */
static void
shell_exec_read_dstr(GtkWidget *wgt, gpointer cbdata)
{
	serd_t *serdp = (serd_t *)cbdata;

	g_free(serdp);
} /* shell_exec_read_dstr */


/*
 * PRIVATE: shell_exec_read_dstr
 *
 * cancel callback for 'enter shell command' dialog
 */
static void
shell_exec_read_cncl(GtkWidget *wgt, gpointer cbdata)
{
	serd_t *serdp = (serd_t *)cbdata;

	gtk_widget_destroy(serdp->toplev);
} /* shell_exec_read_cncl */


/*
 * PRIVATE: shell_exec_read_dstr
 *
 * "Ok" callback for 'enter shell command' dialog.  gets the text from the
 * entry widget, and invokes shell_exec_read_do() to actually execute the shell
 * command.
 */
static void
shell_exec_read_ok(GtkWidget *wgt, gpointer cbdata)
{
	serd_t *serdp = (serd_t *)cbdata;
	char *cmd;

	if ((cmd = gtk_entry_get_text(GTK_ENTRY(serdp->entry))) == NULL)
		return;

	shell_exec_read_do((const char *)cmd, (doc_t *)(serdp->w->curdoc));

	gtk_widget_destroy(serdp->toplev);
} /* shell_exec_read_ok */


/*
 * PRIVATE: shell_exec_read_do
 *
 * forks and execs 'cmd' in a separate shell, then inserts the output directly
 * into the text document.
 */
static void
shell_exec_read_do(const char *cmd, doc_t *d)
{
	int to_sh[2], fr_sh[2];
	pid_t pid;
	char buf[1024];
	int pstat, start;
	size_t count, total = 0;

	if (!GTK_IS_TEXT(d->data))
		return;

	to_sh[0] = to_sh[1] = fr_sh[0] = fr_sh[1] = -1;
	if (pipe(to_sh) < 0 || pipe(fr_sh) < 0)
		goto err;

	switch (pid = fork()) {
	case -1:
		perror("shell_exec_read_do: fork error");
		(void)do_dialog_error("Fork error!", " Fork error! ");
		goto err;
	case 0:
		if (dup2(to_sh[0], STDIN_FILENO) == -1)
			perror("shell_exec_read_do: dup2 err 1");
		if (dup2(fr_sh[1], STDOUT_FILENO) == -1)
			perror("shell_exec_read_do: dup2 err 2");
		if (dup2(fr_sh[1], STDERR_FILENO) == -1)
			perror("shell_exec_read_do: dup2 err 3");

		(void)close(to_sh[0]);
		(void)close(to_sh[1]);
		(void)close(fr_sh[0]);

		if (execl(prefs.shell, my_basename(prefs.shell), "-c", cmd,
			  NULL) == -1) {
			perror("shell_exec_read_do: execl error");
			(void)do_dialog_error("Exec error!", " Exec error! ");
			return;
		}
		exit(127);
	default:
		(void)close(to_sh[0]);
		(void)close(to_sh[1]);
		(void)close(fr_sh[1]);

		start = gtk_editable_get_position(GTK_EDITABLE(d->data));
		gtk_text_set_point(GTK_TEXT(d->data), start);
		total = 0;
		while ((count = read(fr_sh[0], buf, sizeof(buf)))) {
			if ((int)count == -1) {
				perror("shell_exec_read_do: pipe read error");
				(void)do_dialog_error("pipe read error!",
						      " pipe read error! ");
				break;

			}

			g_assert(count > 0);
			gtk_text_insert(GTK_TEXT(d->data), NULL, NULL, NULL,
					buf, (int)count);
			total += count;
		}

		(void)close(fr_sh[0]);
		(void)kill(pid, SIGTERM);
		(void)waitpid(pid, &pstat, 0);

		if (total > 0) {
			gtk_signal_emit_by_name(GTK_OBJECT(d->data), "changed");
#if 0
			int pt = gtk_text_get_point(GTK_TEXT(d->data));
			int end = pt;
			char *text;
			gtk_editable_set_position(GTK_EDITABLE(d->data), pt);
			text = gtk_editable_get_chars(GTK_EDITABLE(d->data),
						      start, total);
			/* XXX - this causes problems with the text widget
			 * when redoing an undo. */
			undo_list_add(d, text, start, end, UndoInsert);
#endif
		}
	} /* switch pid */

	return;

err:
	if (to_sh[0] != -1)
		(void)close(to_sh[0]);
	if (to_sh[1] != -1)
		(void)close(to_sh[1]);
	if (fr_sh[0] != -1)
		(void)close(fr_sh[0]);
	if (fr_sh[1] != -1)
		(void)close(fr_sh[1]);
} /* shell_exec_read_do */


#endif	/* APP_GNP */
#endif	/* USE_SHELL_INSERT */


/* the end */
