/*****************************************************************

Copyright (c) 2000 Matthias Elter <elter@kde.org>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <qhbox.h>
#include <qlayout.h>
#include <qxembed.h>
#include <qfile.h>

#include <kglobal.h>
#include <kconfig.h>
#include <kwin.h>
#include <kapp.h>
#include <kprocess.h>
#include <kdebug.h>
#include <kstddirs.h>
#include <dcopclient.h>

#include "applethandle.h"
#include "extensionop_mnu.h"
#include "global.h"
#include "pluginmgr.h"

#include "container_extension.h"
#include "container_extension.moc"

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

ExtensionContainer::ExtensionContainer(const AppletInfo& info, QWidget *parent)
  : QFrame(parent, "ExtensionContainer", WStyle_Customize | WStyle_NoBorderEx)
  , _pos(Bottom)
  , _orient(Horizontal)
  , _id(QString::null)
  , _layout(0)
  , _opMnu(0)
  , _info(info)
  , _type(KPanelExtension::Normal)
  , _actions(0)
  , _activeWidget(0)
{
    hide();
    // extensions live in the dock
    KWin::setType( winId(), NET::Dock );
    KWin::setState( winId(), NET::StaysOnTop | NET::Sticky );
    KWin::setOnAllDesktops( winId(), TRUE );

    // setup handle
    _handle = new AppletHandle(this);
    _handle->setOrientation(orientation());
    _handle->installEventFilter(this);

    //setup frame
    _frame = new QHBox(this);
    _frame->setFrameStyle(QFrame::NoFrame);
    _frame->installEventFilter(this);

    setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
    setLineWidth(2);

    if (orientation() == Horizontal)
	{
	    _handle->setFixedWidth(7);
	    _handle->setMaximumHeight(128);
	    _layout = new QBoxLayout(this, QBoxLayout::LeftToRight, lineWidth(), 0);
	}
    else
	{
	    _handle->setFixedHeight(7);
	    _handle->setMaximumWidth(128);
	    _layout = new QBoxLayout(this, QBoxLayout::TopToBottom, lineWidth(), 0);
	}

    _layout->setResizeMode( QLayout::FreeResize );

    _layout->addWidget(_handle);
    _layout->addWidget(_frame);
    _layout->activate();
}

ExtensionContainer::~ExtensionContainer()
{
    delete _opMnu;
}

void ExtensionContainer::resetLayout()
{
    _handle->setOrientation(orientation());

    if (orientation() == Horizontal)
	{
	    _layout->setDirection(QBoxLayout::LeftToRight);
	    _handle->setFixedWidth(7);
	    _handle->setMaximumHeight(128);
	}
    else
	{
	    _layout->setDirection(QBoxLayout::TopToBottom);
	    _handle->setFixedHeight(7);
	    _handle->setMaximumWidth(128);
	}
    _layout->activate();
}

void ExtensionContainer::configure()
{
    KConfig *config = KGlobal::config();
    config->setGroup("General");
    _handle->setFadeOutHandle(config->readBoolEntry("FadeOutAppletHandles", false));
}

bool ExtensionContainer::eventFilter (QObject *o, QEvent *e)
{
    switch (e->type())
    {
	case QEvent::MouseButtonPress:
        {
            QMouseEvent* ev = (QMouseEvent*) e;
            if ( ev->button() == RightButton )
            {
                if (!_opMnu)
                    _opMnu = new PanelExtensionOpMenu(_actions);

                switch(_opMnu->exec(mapToGlobal(ev->pos())))
                {
                    case PanelExtensionOpMenu::Move:
                        emit moveme(this);
                        break;
                    case PanelExtensionOpMenu::Remove:
                        emit removeme(this);
                        break;
                    case PanelExtensionOpMenu::About:
                        about();
                        break;
                    case PanelExtensionOpMenu::Help:
                        help();
                        break;
                    case PanelExtensionOpMenu::Preferences:
                        preferences();
                        break;
                    case PanelExtensionOpMenu::ReportBug:
                        reportBug();
                        break;
                    default:
                        break;
                }
                return true;
            }
            else if ( ev->button() == MidButton
                      || ev->button() == LeftButton )
            {
                emit moveme(this);
            }
            return false;
        }
	default:
	    return QWidget::eventFilter(o, e); // standard event processing
    }
    return false;
}

void ExtensionContainer::slotSetPosition(Position p)
{
    if(p == Bottom || p == Top)
	_orient = Horizontal;
    else
	_orient = Vertical;
    _pos = p;
}

void ExtensionContainer::removeSessionConfigFile()
{
    if (_info.configFile().isEmpty()) return;
    if (_info.isUniqueApplet()) return;
    QString path = locate("config", _info.configFile());

    QFile f (path);
    if (f.exists()) {
	kdDebug(1210) << "Removing session config file: " << path << endl;
	f.remove();
    }
}

bool ExtensionContainer::x11Event( XEvent* e )
{
    // "Catch" XEvents which occur on the containers's edges, and redirect them
    // to the apropriate widgets in the container.
    switch ( e->type ) {

    case ButtonPress:
	{
	    // Only ButtonPress events which occur on the widget's frame need to
	    // be handled.
	    if (contentsRect().contains( QPoint(e->xbutton.x, e->xbutton.y) ) ||
		!rect().contains( QPoint(e->xbutton.x, e->xbutton.y) ))
		return false;

	    // Determine the difference between the catched event's position and
	    // the position of the new event that we will contruct. The new
	    // position may not be on the frame, but has to be in
	    // contentsRect(). Using a difference is easier because it can be
	    // applied it to both the local and global positions.
	    int dx, dy;
	    dx = QMAX( 0, contentsRect().left() - e->xbutton.x ) ;
	    dy = QMAX( 0, contentsRect().top() - e->xbutton.y );
	    if (dx == 0)
		dx = QMIN( 0, contentsRect().right() - e->xbutton.x );
	    if (dy == 0)
		dy = QMIN( 0, contentsRect().bottom() - e->xbutton.y );

	    // The widget which will be the destination of the new event.
	    QWidget* destWidget = QApplication::widgetAt(e->xbutton.x_root + dx,
							 e->xbutton.y_root + dy, true);

	    // If there is no such widget, we leave the event to Qt's event
	    // handler. If destWidget is equal to this widget pass it to Qt's
	    // event handler too to avoid nasty loops.
	    if (!destWidget || destWidget == this)
		return false;

	    // Now construct the new event.
	    XEvent ne;
	    memset(&ne, 0, sizeof(ne));
	    ne = *e;
	    ne.xbutton.window = destWidget->winId();
	    Window child; // Not used
	    XTranslateCoordinates(qt_xdisplay(), winId(), destWidget->winId(),
				  e->xbutton.x + dx, e->xbutton.y + dy,
				  &ne.xbutton.x, &ne.xbutton.y, &child);
	    ne.xbutton.x_root = e->xbutton.x_root + dx;
	    ne.xbutton.y_root = e->xbutton.y_root + dy;

	    // Pretty obvious... Send the event.
	    XSendEvent(qt_xdisplay(), destWidget->winId(), false, NoEventMask, &ne);

	    // Make the receiver our active widget. It will receive all events
	    // until the mouse button is released.
	    _activeWidget = destWidget;

	    // We're done with this event.
	    return true;
	}

	// The rest of the cases are more or less a duplication of the first
	// one with off course some minor differences. ButtonRelease is almost
	// the same as MotionNotify, but there's xbutton and xmotion.
    case ButtonRelease:
	{
	    // Handle events outside the widget's rectangle too, since the mouse
	    // can be grabbed.
	    if (contentsRect().contains( QPoint(e->xbutton.x, e->xbutton.y) ))
		return false;

	    int dx, dy;
	    dx = QMAX( 0, contentsRect().left() - e->xbutton.x ) ;
	    dy = QMAX( 0, contentsRect().top() - e->xbutton.y );
	    if (dx == 0)
		dx = QMIN( 0, contentsRect().right() - e->xbutton.x );
	    if (dy == 0)
		dy = QMIN( 0, contentsRect().bottom() - e->xbutton.y );

	    // If there is a widget active it should receive the new event.
	    QWidget* destWidget;
	    if (_activeWidget)
		destWidget = _activeWidget;
	    else
		destWidget = QApplication::widgetAt(e->xbutton.x_root + dx,
						    e->xbutton.y_root + dy, true);

	    if (!destWidget || destWidget == this)
		return false;

	    // The event's position can be outside the widget as well, so
	    // there's no need to adjust the position.
	    if (!rect().contains( QPoint(e->xbutton.x, e->xbutton.y)) ) {
		dx = 0;
		dy = 0;
	    }

	    XEvent ne;
	    memset(&ne, 0, sizeof(ne));
	    ne = *e;
	    ne.xbutton.window = destWidget->winId();
	    Window child;
	    XTranslateCoordinates(qt_xdisplay(), winId(), destWidget->winId(),
				  e->xbutton.x + dx, e->xbutton.y + dy,
				  &ne.xbutton.x, &ne.xbutton.y, &child);
	    ne.xbutton.x_root = e->xbutton.x_root + dx;
	    ne.xbutton.y_root = e->xbutton.y_root + dy;

	    XSendEvent(qt_xdisplay(), destWidget->winId(), false, NoEventMask, &ne);

	    // Turn off the active widget.
	    _activeWidget = 0;

	    return true;
	}

    case MotionNotify:
	{
	    if (contentsRect().contains( QPoint(e->xmotion.x, e->xmotion.y) ))
		return false;

	    int dx, dy;
	    dx = QMAX( 0, contentsRect().left() - e->xmotion.x ) ;
	    dy = QMAX( 0, contentsRect().top() - e->xmotion.y );
	    if (dx == 0)
		dx = QMIN( 0, contentsRect().right() - e->xmotion.x );
	    if (dy == 0)
		dy = QMIN( 0, contentsRect().bottom() - e->xmotion.y );

	    QWidget* destWidget;
	    if (_activeWidget)
		destWidget = _activeWidget;
	    else
		destWidget = QApplication::widgetAt(e->xmotion.x_root + dx,
						    e->xmotion.y_root + dy, true);

	    if (!destWidget || destWidget == this)
		return false;

	    if (!rect().contains( QPoint(e->xmotion.x, e->xmotion.y)) ) {
		dx = 0;
		dy = 0;
	    }

	    XEvent ne;
	    memset(&ne, 0, sizeof(ne));
	    ne = *e;
	    ne.xmotion.window = destWidget->winId();
	    Window child;
	    XTranslateCoordinates(qt_xdisplay(), winId(), destWidget->winId(),
				  e->xmotion.x + dx, e->xmotion.y + dy,
				  &ne.xmotion.x, &ne.xmotion.y, &child);
	    ne.xmotion.x_root = e->xmotion.x_root + dx;
	    ne.xmotion.y_root = e->xmotion.y_root + dy;

	    XSendEvent(qt_xdisplay(), destWidget->winId(), false, NoEventMask, &ne);

	    return true;
	}

    case EnterNotify:
    case LeaveNotify:
	{
	    if (contentsRect().contains(
					QPoint(e->xcrossing.x, e->xcrossing.y) ) ||
		!rect().contains( QPoint(e->xcrossing.x, e->xcrossing.y) ))
		return false;

	    int dx, dy;
	    dx = QMAX( 0, contentsRect().left() - e->xcrossing.x ) ;
	    dy = QMAX( 0, contentsRect().top() - e->xcrossing.y );
	    if (dx == 0)
		dx = QMIN( 0, contentsRect().right() - e->xcrossing.x );
	    if (dy == 0)
		dy = QMIN( 0, contentsRect().bottom() - e->xcrossing.y );

	    QWidget* destWidget;
	    if (_activeWidget)
		destWidget = _activeWidget;
	    else
		destWidget = QApplication::widgetAt(e->xcrossing.x_root + dx,
						    e->xcrossing.y_root + dy, true);

	    if (!destWidget || destWidget == this)
		return false;

	    if (!rect().contains( QPoint(e->xcrossing.x, e->xcrossing.y)) ) {
		dx = 0;
		dy = 0;
	    }

	    XEvent ne;
	    memset(&ne, 0, sizeof(ne));
	    ne = *e;
	    ne.xcrossing.window = destWidget->winId();
	    Window child;
	    XTranslateCoordinates(qt_xdisplay(), winId(), destWidget->winId(),
				  e->xcrossing.x + dx, e->xcrossing.y + dy,
				  &ne.xcrossing.x, &ne.xcrossing.y, &child);
	    ne.xcrossing.x_root = e->xcrossing.x_root + dx;
	    ne.xcrossing.y_root = e->xcrossing.y_root + dy;

	    XSendEvent(qt_xdisplay(), destWidget->winId(), false, NoEventMask, &ne);

	    return true;
	}
    default:
	break;
    }

    return false;
}

InternalExtensionContainer::InternalExtensionContainer(const AppletInfo& info, QWidget *parent)
    : ExtensionContainer(info, parent)
{
    _deskFile = info.desktopFile();
    _configFile = info.configFile();
    _extension = PGlobal::pluginmgr->loadExtension(_deskFile, _configFile, _frame);

    if (!_extension) return;

    _pos = static_cast<Position>(static_cast<int>(_extension->preferedPosition()));
    _extension->slotSetPosition((KPanelExtension::Position)_pos);
    _type = _extension->type();
    _actions = _extension->actions();
    ExtensionContainer::slotSetPosition(_pos);
    resetLayout();

    connect(_extension, SIGNAL(updateLayout()), SIGNAL(updateLayout()));
}


InternalExtensionContainer::~InternalExtensionContainer()
{
}

QSize InternalExtensionContainer::sizeHint(Position p, QSize maxSize)
{
    if (!_extension)
	return maxSize;

    QSize offset;
    QSize ms = maxSize;

    if (p == Bottom || p == Top) { // horizontal
	offset = QSize(_handle->width() + 2*lineWidth(), 0);
	ms -= offset;
    }
    else { // vertical
	offset = QSize(0, _handle->height() + 2*lineWidth());
	ms -= offset;
    }


    return (_extension->sizeHint((KPanelExtension::Position)p, ms) + offset);
}

void InternalExtensionContainer::saveConfiguration(const QString& g)
{
    KConfig *config = KGlobal::config();

    QString group = g;
    if (group.isNull()) group = extensionId();

    config->setGroup(group);
    config->writeEntry("ConfigFile", _configFile);
    config->writeEntry("DesktopFile", _deskFile);
    config->sync();
}

void InternalExtensionContainer::slotSetPosition(Position p)
{
    if (_pos == p) return;

    ExtensionContainer::slotSetPosition(p);

    if (!_extension) return;
    _extension->slotSetPosition((KPanelExtension::Position)(p));

    resetLayout();
}

void InternalExtensionContainer::about()
{
    if (!_extension) return;
    _extension->action(KPanelExtension::About);
}

void InternalExtensionContainer::help()
{
    if (!_extension) return;
    _extension->action(KPanelExtension::Help);
}

void InternalExtensionContainer::preferences()
{
    if (!_extension) return;
    _extension->action(KPanelExtension::Preferences);
}

void InternalExtensionContainer::reportBug()
{
    if (!_extension) return;
    _extension->action(KPanelExtension::ReportBug);
}

ExternalExtensionContainer::ExternalExtensionContainer(const AppletInfo& info, QWidget *parent)
    : ExtensionContainer(info, parent)
    , DCOPObject(QCString("ExternalExtensionContainer_") + kapp->randomString(20).lower().local8Bit())
    , _isdocked(false)
{
    _deskFile = info.desktopFile();
    _configFile = info.configFile();

    // init QXEmbed
    _embed = new QXEmbed( _frame );
    connect (_embed, SIGNAL(embeddedWindowDestroyed()),
	     this, SIGNAL(embeddedWindowDestroyed()));

    KProcess process;
    process << "extensionproxy"
	    << QCString("--configfile")
	    << info.configFile()
	    << QCString("--callbackid")
	    << objId()
    	    << info.desktopFile();
    process.start(KProcess::DontCare);
}

ExternalExtensionContainer::~ExternalExtensionContainer()
{
    QByteArray data;
    kapp->dcopClient()->send( _app, "ExtensionProxy", "removedFromPanel()", data);
}

void ExternalExtensionContainer::saveConfiguration(const QString& g)
{
    KConfig *config = KGlobal::config();

    QString group = g;
    if (group.isNull()) group = extensionId();

    config->setGroup(group);
    config->writeEntry("ConfigFile", _configFile);
    config->writeEntry("DesktopFile", _deskFile);
    config->sync();
}

void ExternalExtensionContainer::slotSetPosition(Position p)
{
    if (_pos == p) return;

    ExtensionContainer::slotSetPosition(p);

    if (!_isdocked) return;

    QByteArray data;
    QDataStream dataStream( data, IO_WriteOnly );
    dataStream << static_cast<int>(p);

    kapp->dcopClient()->send( _app, "ExtensionProxy", "setPosition(int)", data );

    resetLayout();
}

void ExternalExtensionContainer::about()
{
    if (!_isdocked) return;

    QByteArray data;
    kapp->dcopClient()->send( _app, "ExtensionProxy", "about()", data );
}

void ExternalExtensionContainer::help()
{
    if (!_isdocked) return;

    QByteArray data;
    kapp->dcopClient()->send( _app, "ExtensionProxy", "help()", data );
}

void ExternalExtensionContainer::preferences()
{
    if (!_isdocked) return;

    QByteArray data;
    kapp->dcopClient()->send( _app, "ExtensionProxy", "preferences()", data );
}

void ExternalExtensionContainer::reportBug()
{
    if (!_isdocked) return;

    QByteArray data;
    kapp->dcopClient()->send( _app, "ExtensionProxy", "reportBug()", data );
}

QSize ExternalExtensionContainer::sizeHint(Position p, QSize maxSize)
{
    if (!_isdocked)
	return maxSize;

    QSize offset;
    QSize ms = maxSize;

    if (p == Bottom || p == Top) { // horizontal
	offset = QSize(_handle->width() + 2*lineWidth(), 0);
	ms -= offset;
    }
    else { // vertical
	offset = QSize(0, _handle->height() + 2*lineWidth());
	ms -= offset;
    }

    DCOPClient* dcop = kapp->dcopClient();

    QByteArray data;
    QCString replyType;
    QByteArray replyData;
    QDataStream dataStream( data, IO_WriteOnly );
    dataStream << static_cast<int>(p);
    dataStream << ms;

    if (dcop->call( _app, "ExtensionProxy", "sizeHint(int,QSize)", data, replyType, replyData ) )
	{
	    QDataStream reply( replyData, IO_ReadOnly );
	    reply >> ms;
	}
    return (ms + offset);
}

bool ExternalExtensionContainer::process(const QCString &fun, const QByteArray &data,
				      QCString& replyType, QByteArray & replyData)
{
    if ( fun == "dockRequest(int,int)" )
	{
	    QDataStream reply( replyData, IO_WriteOnly );
	    replyType = "WId";
	    reply << _embed->winId();

	    QDataStream sdata( data, IO_ReadOnly );
	    int actions, type;
	    sdata >> actions;
	    sdata >> type;

	    dockRequest(kapp->dcopClient()->senderId(), actions, type);
	    return true;
	}
    else if(fun == "updateLayout()")
	{
	    emit updateLayout();
	    return(true);
	}
    return true;
}

void ExternalExtensionContainer::dockRequest(QCString app, int actions, int type)
{
    _app = app;
    _type = static_cast<KPanelExtension::Type>(type);
    _actions = actions;

    kdDebug(1210) << "ExternalExtensionContainer::dockRequest: " << app << endl;

    // get prefered position
    {
        QByteArray data;
        QCString replyType;
        QByteArray replyData;
        int pos;

        if (kapp->dcopClient()->call(_app, "ExtensionProxy", "preferedPosition()",
                                     data, replyType, replyData))
	{
	    QDataStream reply( replyData, IO_ReadOnly );
	    reply >> pos;
            ExtensionContainer::slotSetPosition(static_cast<Position>(pos));
	}
    }

    // set position
    {
	QByteArray data;
	QDataStream dataStream( data, IO_WriteOnly );
	dataStream << static_cast<int>(_pos);

	kapp->dcopClient()->send( _app, "ExtensionProxy", "setPosition(int)", data );
    }

    _isdocked = true;
    resetLayout();
    emit docked(this);
}
