/* Yo Emacs, this -*- C++ -*-

  Copyright (C) 1999,2000 Jens Hoefkens
  jens@hoefkens.com

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
*/

#include "kbgconnection.h"
#include "kbgconnection.moc"

#include <errno.h>
#include <fcntl.h>


/**
 *
 * This class handles the network connection for the KBgEngineFIBS class. 
 *
 * We have to filter the incoming data, since all messages from FIBS are
 * line-oriented - but the raw date can be broken in the middle of lines.
 * This class collects data until a whole line has been received and then 
 * emits the line w/o the terminating newline/carriage return. The class 
 * also sends data - newline/carriage return is appended before the data
 * are sent.
 */

/*
 * Establish a connection to port at host
 */
KBgConnection::KBgConnection(QObject *parent, QString *name, QString *host, unsigned short int port)
	: QObject(parent, name->latin1())
{
	/*
	 * set up the buffers
	 */
	txBuffer.clear();
	rxBuffer = "";
	
	/* 
	 * create the socket and set it non-blocking - ignore errors
	 */
	socket = new KSocket(host->latin1(), port, 1000);
	if (socket->socket() >= 0) {
		fcntl(socket->socket(), F_SETFL, O_NONBLOCK);
		socket->enableRead(true);
		socket->enableWrite(false);
	} 
	
	/*
	 * connect all the socket handlers
	 */
	connect(socket, SIGNAL(readEvent (KSocket *)), this,   SLOT(receiveData  (KSocket *)));
	connect(socket, SIGNAL(writeEvent(KSocket *)), this,   SLOT(flushData    (KSocket *)));
	connect(socket, SIGNAL(closeEvent(KSocket *)), this,   SLOT(transmitClose(KSocket *)));
}

/*
 * Read characters from the socket and whenever a full line has been
 * received, it is emitted.
 */
void KBgConnection::receiveData(KSocket *s)
{
	char buffer;
	
	do {
		if (recv(s->socket(), &buffer, 1, 0) < 1) {
			if (errno != EWOULDBLOCK)
				emit connectionDown();
			return;
		} else
			rxBuffer += buffer;
		
	} while (buffer != '\n');

	/*
	 * remove \r\n from the string and don't transmit empty lines 
	 * it is safe to expose the data member rxBuffer!
	 */
	if (rxBuffer.length() > 2) {
		rxBuffer.truncate(rxBuffer.length()-2);
		emit newData(&rxBuffer);
		rxBuffer = "";

	} else
		rxBuffer = "";
}

/*
 * Socket can be writte, write a single line
 */
void KBgConnection::flushData(KSocket *s)
{
	send(s->socket(), txBuffer.getFirst(), strlen(txBuffer.getFirst()), 0); 
	txBuffer.removeFirst();
	s->enableWrite(!txBuffer.isEmpty());
}

/*
 * Emit a simple connectionDown signal if the connection dies
 */
void KBgConnection::transmitClose(KSocket *s)
{
	if (s->socket() != -12345) // always true, avoids compiler warning
		emit connectionDown();
}

/*
 * Delete the socket
 */
KBgConnection::~KBgConnection()
{
	delete socket;
}

/*
 * Return the socket file descriptor
 */
int KBgConnection::status()
{
	return socket->socket();
}

/*
 * Queue a line for transmission, append \r\n
 */
void KBgConnection::sendData(const QString &line)
{
	txBuffer.append((line+"\r\n").latin1());
	socket->enableWrite(true);
}

// EOF
