/*
 * noatun.cpp
 *
 * Copyright (C) 1999 Charles Samuels <charles@kde.org>
 */

#include "userinterface.h"
#include <noatunapp.h>
#include "player.h"
#include "controls.h"
#include <effects.h>

#include <qpushbutton.h>
#include <qdragobject.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qobjcoll.h>

#include <kpopupmenu.h>
#include <kstatusbar.h>
#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <kconfig.h>
#include <noatunplaylist.h>

Noatun::Noatun() : QWidget(0,"NoatunUI"), UserInterface()
{
	NOATUNPLUGINC(Noatun);
	setAcceptDrops(true);
	static const int buttonSize=32;

	{ // The buttons
		mBack=new QPushButton(this);
		mBack->setFixedSize(buttonSize,buttonSize);
		mBack->setPixmap(BarIcon("noatunback"));
		connect(mBack, SIGNAL(clicked()), napp->player(), SLOT(back()));
		QToolTip::add(mBack,i18n("Back"));

		mStop=new QPushButton(this);
		mStop->setFixedSize(buttonSize,buttonSize);
		mStop->setPixmap(BarIcon("noatunstop"));
		connect(mStop, SIGNAL(clicked()), napp->player(), SLOT(stop()));
		QToolTip::add(mStop, i18n("Stop"));
	
		mPlay=new QPushButton(this);
		mPlay->setToggleButton(true);
		mPlay->setFixedSize(buttonSize,buttonSize);
		mPlay->setPixmap(BarIcon("noatunplay"));
		connect(mPlay, SIGNAL(clicked()), napp->player(), SLOT(playpause()));
		QToolTip::add(mStop, i18n("Play"));
	
		mForward=new QPushButton(this);
		mForward->setFixedSize(buttonSize,buttonSize);
		mForward->setPixmap(BarIcon("noatunforward"));
		connect(mForward, SIGNAL(clicked()), napp->player(), SLOT(fastForward()));
		QToolTip::add(mStop, i18n("Forward"));
	
		mPlaylist=new QPushButton(this);
		mPlaylist->setToggleButton(true);
		mPlaylist->setFixedSize(buttonSize,buttonSize);
		mPlaylist->setPixmap(BarIcon("noatunplaylist"));
		connect(mPlaylist, SIGNAL(clicked()), napp->player(), SLOT(toggleListView()));
		QToolTip::add(mStop, i18n("Playlist"));
		
		mRemoveCurrent=new QPushButton(this);
		mRemoveCurrent->setFixedSize(buttonSize,buttonSize);
		mRemoveCurrent->setPixmap(BarIcon("editdelete"));
		connect(mRemoveCurrent, SIGNAL(clicked()), napp->player(), SLOT(removeCurrent()));
		QToolTip::add(mRemoveCurrent, i18n("Remove Current File from Playlist"));
		
		mLoop=new QPushButton(this);
		mLoop->setFixedSize(buttonSize,buttonSize);
		mLoop->setPixmap(BarIcon("noatunloopnone"));
		connect(mLoop, SIGNAL(clicked()), napp->player(), SLOT(loop()));
		QToolTip::add(mLoop, i18n("Change Loop Style"));
	}
	
	mVolume=new L33tSlider(0,100,10,0, Horizontal, this);
	mVolume->setValue(napp->player()->volume());
	mSeeker=new L33tSlider(0,1000,100,0, Horizontal, this);
	
	mStatusBar=new KStatusBar(this);
	
	
	QGridLayout *l=new QGridLayout(this);
	l->addWidget(mBack,0,0);
	l->addWidget(mStop,0,1);
	l->addWidget(mPlay,0,2);
	l->addWidget(mForward,0,3);
	l->addWidget(mPlaylist,0,4, Qt::AlignLeft);
	l->addWidget(mRemoveCurrent,0,5);
	l->addWidget(mLoop,0,6);
	l->addColSpacing(4, buttonSize+8);
	
	l->addMultiCellWidget(mVolume,1,1,0,6);
	l->addMultiCellWidget(mSeeker,2,2,0,6);
	l->addMultiCellWidget(mStatusBar,3,3,0,6);
	
	statusBar()->message(i18n("No File Loaded"));
	statusBar()->insertItem("00:00/00:00", 1, 0, true);

	connect( napp->player(), SIGNAL(hideYourself()), this, SLOT(hide()) );
	connect( napp->player(), SIGNAL(showYourself()), this, SLOT(show()) );

	connect(napp->player(), SIGNAL(playing()), this, SLOT(slotPlaying()));
	connect(napp->player(), SIGNAL(stopped()), this, SLOT(slotStopped()));
	connect(napp->player(), SIGNAL(paused()), this, SLOT(slotPaused()));
	napp->player()->handleButtons();
	
	connect(napp->player(), SIGNAL(timeout()), this, SLOT(slotTimeout()));
	connect(napp->player(), SIGNAL(loopTypeChange(int)), this, SLOT(changeLoopType(int)));

//	if(seeker())
	{
		connect(seeker(), SIGNAL(userChanged(int)), napp->player(), SLOT(skipTo(int)));
		connect(seeker(), SIGNAL(sliderMoved(int)), SLOT(sliderMoved(int)));
	}
	connect(mVolume, SIGNAL(sliderMoved(int)), napp->player(), SLOT(setVolume(int)));
	connect(mVolume, SIGNAL(userChanged(int)), napp->player(), SLOT(setVolume(int)));
	
	
	connect(napp->player(), SIGNAL(playlistShown()), SLOT(playlistShown()));
	connect(napp->player(), SIGNAL(playlistHidden()), SLOT(playlistHidden()));
	
	// Event Filter for the RMB
	for (QListIterator<QObject> i(*children()); i.current(); ++i)
		(*i)->installEventFilter(this);
	
	setCaption("Noatun");
	setIcon(BarIcon("noatun"));
	show();

	// What it is now, stay, stay.. roll over, good boy!
	setFixedSize(minimumSize());
}

Noatun::~Noatun()
{
	// If cfg dialog is still open, delete it so it saves it's position
//	if(prefDlgExist)
//		delete prefDlg;
}

void Noatun::closeEvent(QCloseEvent*e)
{
	unload();
}

void Noatun::showEvent(QShowEvent*e)
{
	QWidget::showEvent(e);
}

void Noatun::dragEnterEvent(QDragEnterEvent *event)
{
    // accept uri drops only
    event->accept(QUriDrag::canDecode(event));
}

void Noatun::dropEvent(QDropEvent *event)
{
	QStrList uri;
	if (QUriDrag::decode(event, uri))
	{
		for (char *file=uri.first(); file != 0; file=uri.next() )
			napp->player()->openFile(KURL(file), false);
	}
}

void Noatun::mouseReleaseEvent(QMouseEvent *e)
{
	QWidget::mouseReleaseEvent(e);
	if (e->button()!=RightButton) return;
	napp->showContextMenu();
}

void Noatun::changeStatusbar(const QString& text, const QString &text2)
{
	if (!text2.isNull())
		statusBar()->changeItem(text2, 1);

	statusBar()->message(!text.isNull() ? text : napp->player()->current()->title());
}

void Noatun::changeCaption(const QString& text)
{
    setCaption(text);
}

void Noatun::slotPlaying()
{
//	connect(kwinmodule, SIGNAL(windowAdded(WId)), view, SLOT(attemptReparent(WId)));
	changeStatusbar(napp->player()->current()->title(), napp->player()->lengthString());
	mPlay->setOn(true);
	mStop->setEnabled(true);
	mPlay->setPixmap(BarIcon("noatunpause"));
}

void Noatun::slotStopped()
{
	if (!napp->player()->current()) return;
	changeStatusbar(i18n("No File Loaded"), napp->player()->lengthString());
	mStop->setEnabled(false);
	mPlay->setOn(false);
	seeker()->setValue(0);
	mPlay->setPixmap(BarIcon("noatunplay"));
}

void Noatun::slotPaused()
{
	mStop->setEnabled(true);
	mPlay->setOn(false);
	mPlay->setPixmap(BarIcon("noatunplay"));
}

void Noatun::slotTimeout()
{
	mVolume->setValue(napp->player()->volume());
	
	if (!napp->player()->current()) return;
	if (static_cast<L33tSlider*>(seeker())->currentlyPressed()) return;
	if (seeker())
		seeker()->setValue(napp->player()->getTime());
	changeStatusbar(0, napp->player()->lengthString());
}

void Noatun::sliderMoved(int mille)
{
	if (napp->player()->current())
		changeStatusbar(0, napp->player()->lengthString(napp->player()->current()->fromMille(mille)));
}

void Noatun::changeLoopType(int t)
{
	static const int time=2000;
	switch (t)
	{
	case(Player::None):
		statusBar()->message(i18n("No looping"), time);
		mLoop->setPixmap(BarIcon("noatunloopnone"));
		break;
	case(Player::Song):
		statusBar()->message(i18n("Song looping"), time);
		mLoop->setPixmap(BarIcon("noatunloopsong"));
		break;
	case(Player::Playlist):
		statusBar()->message(i18n("Playlist looping"), time);
		mLoop->setPixmap(BarIcon("noatunloopplaylist"));
	}

}

bool Noatun::eventFilter(QObject *o, QEvent *e)
{
	if ((e->type() == QEvent::MouseButtonRelease)
		&& ((static_cast<QMouseEvent*>(e))->button()==RightButton))
	{
		mouseReleaseEvent(static_cast<QMouseEvent*>(e));
		return true;
	}

	if (e->type() == QEvent::Wheel)
	{
		wheelEvent(static_cast<QWheelEvent*>(e));
		return true;
	}
	return QWidget::eventFilter(o, e);
}

void Noatun::playlistShown()
{
	mPlaylist->setOn(true);
}

void Noatun::playlistHidden()
{
	mPlaylist->setOn(false);
}

void Noatun::wheelEvent(QWheelEvent *e)
{
	int delta=e->delta();
	mVolume->setValue(mVolume->value()+(delta/120));
	napp->player()->setVolume(mVolume->value()+(delta/120));
}

#include "userinterface.moc"
