#include "conf.h"

typedef struct test_directory {
  char *path;
  int nents;
  int curent;
  char *ent[25];

  struct dirent dirent;
} testdir_t;


static char local[MAXPATHLEN]     = {'\0'};
static char local_cwd[MAXPATHLEN] = "/";

static int get_local(fsdir_t *f, const char *path, char *buf, int maxlen)
{
  char tmp[MAXPATHLEN] = {'\0'};
  register int l = strlen(f->name);

  if(*path == '/' && !strncmp(f->name,path,l))
    sstrncpy(tmp, path + l, sizeof(tmp) > maxlen ? maxlen : sizeof(tmp));
  else {
    *tmp = '\0';
    fs_dircat(tmp,sizeof(tmp),local_cwd,path);
  }
  
  fs_clean_path(tmp, buf, maxlen);
  return 0;
}

static int my_chdir(fsdir_t *f, const char *path)
{
  get_local(f,path,local,sizeof(local));
  
  if(!local[0] || !strcmp(local,"./") || !strcmp(local,"/")) {
    sstrncpy(local_cwd,"/",sizeof(local_cwd));
    return 0;
  } else if(!strcmp(local,"/buttmunch")) {
    sstrncpy(local_cwd,"/buttmunch",sizeof(local_cwd));
    return 0;
  } else if(!strcmp(local,"/")) {
    sstrncpy(local_cwd,"/",sizeof(local_cwd));
    return 0;
  }

  errno = ENOENT;
  return -1;
}

static int my_readlink(fsdir_t *f, const char *path, char *buf, size_t bufsiz)
{
  get_local(f,path,local,sizeof(local));

  if(!strcmp(local,"/foobar")) {
    sstrncpy(buf,"/u2/staff/flood",bufsiz);
    return strlen(buf);
  }

  errno = ENOENT;
  return -1;
}

static int my_lstat(fsdir_t *f, const char *path, struct stat *sbuf)
{
  get_local(f,path,local,sizeof(local));

  if(!strcmp(local,"/")) {
    bzero(sbuf,sizeof(struct stat));
    sbuf->st_ino = 666;
    sbuf->st_mode = 0755 | S_IFDIR;
    sbuf->st_nlink = 3;
    sbuf->st_uid = 0;
    sbuf->st_gid = 0;
    sbuf->st_size = 0;
    sbuf->st_blksize = 0;
    sbuf->st_blocks = 0;
    time(&sbuf->st_atime);
    time(&sbuf->st_mtime);
    time(&sbuf->st_ctime);

    return 0;
  } else if(!strcmp(local,"/foobar")) {
    bzero(sbuf,sizeof(struct stat));
    sbuf->st_ino = 4523;
    sbuf->st_mode = 0755 | S_IFLNK;
    sbuf->st_nlink = 1;
    sbuf->st_uid = 5;
    sbuf->st_gid = 5;
    sbuf->st_size = 4321;
    sbuf->st_blksize = 1024;
    sbuf->st_blocks = 5;
    time(&sbuf->st_atime);
    time(&sbuf->st_mtime);
    time(&sbuf->st_ctime);

    return 0;
  } else if(!strcmp(local,"/buttmunch")) {
    bzero(sbuf,sizeof(struct stat));
    sbuf->st_ino = 6543;
    sbuf->st_mode = 0755 | S_IFDIR;
    sbuf->st_nlink = 1;
    sbuf->st_uid = 6;
    sbuf->st_gid = 6;
    sbuf->st_size = 0;
    sbuf->st_blksize = 1024;
    sbuf->st_blocks = 0;

    time(&sbuf->st_atime);
    time(&sbuf->st_mtime);
    time(&sbuf->st_ctime);
    return 0;
  } else if(!strncmp(local,"/buttmunch/",11)) {
    char *cp = local+11;
    sbuf->st_ino = atoi(cp);
    sbuf->st_mode = 4755 | S_IFREG;
    sbuf->st_uid = 0;
    sbuf->st_gid = 0;
    sbuf->st_size = atoi(cp);
    sbuf->st_blksize = 1024;
    sbuf->st_blocks = sbuf->st_size / 1024;

    if(!sbuf->st_blocks)
      sbuf->st_blocks++;

    return 0;
  }

  errno = ENOENT;
  return -1;
}

static int my_stat(fsdir_t *f, const char *path, struct stat *sbuf)
{
  if(my_lstat(f,path,sbuf) == -1)
    return -1;

  if(S_ISLNK(sbuf->st_mode)) {
    char linkbuf[MAXPATHLEN] = {'\0'};

    if(my_readlink(f,path,linkbuf,sizeof(linkbuf)) == -1)
      return -1;

    return fs_stat(linkbuf,sbuf);
  }

  return 0;
}

static void *my_opendir(fsdir_t *f, const char *path)
{
  char *ptr,*cp,*tmp;
  int i;

  get_local(f,path,local,sizeof(local));

  if(!strcmp(local,"/buttmunch")) {
    testdir_t *dir;

    dir = (testdir_t*)malloc(sizeof(testdir_t));
    dir->path = strdup(local);

    dir->nents = 5;
    dir->curent = 0;
    dir->ent[0] = "5678";
    dir->ent[1] = "43215";
    dir->ent[2] = "12578";
    dir->ent[3] = "14";
    dir->ent[4] = "56";

    return dir;
  }

  errno = ENOENT;
  return NULL;
}

static struct dirent *my_readdir(fsdir_t *f, void *_dir)
{
  testdir_t *dir = (testdir_t*)_dir;
  int i;
  int		name_max;

  i = dir->curent;
  if(i + 1 == dir->nents) {
    errno = 0;
    return NULL;
  }

  /*
  ** XXX NB: POSIX.1 only defines d_name for the dirent structure,
  ** XXX while SUS2 also defines d_ino.
  ** XXX All others should be autoconfiscated.
  ** XXX For example, Solaris (2.5-2.8) doesn't have dirent.d_type.
  */
  dir->dirent.d_ino = atoi(dir->ent[i]);
  dir->dirent.d_type = DT_REG;

  if ( (name_max = get_name_max(t->t_path, 0)) < 1 ) {
	log_debug(DEBUG1, "get_name_max(%s, %d) = %d, using %d",
		dir->t_dir ? dir->t_dir : "(NULL)", 0, name_max, NAME_MAX_GUESS);
	name_max = NAME_MAX_GUESS;
  }
  sstrncpy(dir->dirent.d_name, dir->ent[i], name_max + 1);
  /*
  ** XXX Isn't d_reclen supposed to be the length of the entire structure,
  ** XXX not just the name string?
  */
  dir->dirent.d_reclen = strlen(dir->dirent.d_name);
  
  dir->curent++;
  return &dir->dirent;
}

static int my_closedir(fsdir_t *f, void *_dir)
{
  testdir_t *dir = (testdir_t*)_dir;

  free(dir->path);
  free(dir);

  return 0;
}

static int notdir()
{ errno = ENOTDIR; return -1; }

static int html_open(fsdir_t *f, const char *path, int flags)
{
  int ret;

  ret = f->parent->open(f->parent,path,flags);
  if(ret >= 0)
    f->private = 0;

  return ret;
}

static int html_close(fsdir_t *f, int fd)
{
  f->private = 0;
  return f->parent->close(f->parent,fd);
}

static int html_read(fsdir_t *f, int fd, char *buf, size_t size)
{
  int ret;
  char *cp,*ptr;
  int intag = (int)f->private;
  size_t len;

  ret = f->parent->read(f,fd,buf,size);
  if(ret >= 0) {
    len = ret;
    for(cp = buf, ptr = buf; len; len--, cp++) {
      if(intag) {
        if(*cp == '>')
          intag--;
        ret--;
        continue;
      } else {
        if(*cp == '<') {
          intag++;
          ret--;
          continue;
        }
        *ptr++ = *cp;
      }
    }

    *ptr = '\0';
    f->private = (void*)intag;
  }

  return ret;
}

static int html_write(fsdir_t *f, int fd, const char *buf, size_t size)
{ return f->parent->write(f,fd,buf,size); }

static int html_hit(fsdir_t *f, const char *path, int op)
{
  fsdir_t *newfs;
  struct stat sbuf;
  char buf[MAXPATHLEN] = {'\0'};

  if(op == FSOP_OPEN) {
    if(fs_resolve_path(path,buf,sizeof(buf)) == -1)
      return -1;

    if(fs_stat(buf,&sbuf) == -1)
      return -1;

    if(!S_ISREG(sbuf.st_mode))
      return 0;

    newfs = fs_register(buf);
    newfs->chdir = (int(*)(fsdir_t*,const char*))&notdir;
    newfs->open = &html_open;
    newfs->close = &html_close;
    newfs->read = &html_read;
    newfs->write = &html_write;
    return 1;
  }

  return 0;
}

static int test_init()
{
  fsmatch_t *fm;

  fm = fs_register_match("*.htm*",FSOP_FILEMASK);
  fm->dir_hit = fm->file_hit = html_hit;

  return 0;
}

module test_module = {
  NULL,NULL,
  0x20,
  "test",
  NULL,
  NULL,
  NULL,
  test_init,NULL
};
