#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <errno.h>

#include "main.h"
#include "qsodata.h"
#include "remlog.h"
#include "trx.h"

struct qsodata {
	char call[64];
	char band[64];
	char txrst[64];
	char rxrst[64];
	char name[64];
	char qth[64];
	char notes[256];

	time_t time;
};

static struct qsodata qso;

/* ---------------------------------------------------------------------- */

static msgtype msgbuf;

static void log_msg_clear(void)
{
	msgbuf.mtext[0] = 0;
}

static void log_msg_add(char *type, char *data)
{
	char entry[100];

	snprintf(entry, sizeof(entry), "%s:%s%c", type, data, LOG_MSEPARATOR);
	entry[sizeof(entry) - 1] = 0;

	if (strlen(msgbuf.mtext) + strlen(entry) + 1 > LOG_MSG_LEN) {
		g_print("log message too long, entry dropped: %s\n", entry);
		return;
	}

	strcat(msgbuf.mtext, entry);
}

static int log_msg_send(void)
{
	int msqid, len;

	if ((msqid = msgget(LOG_MKEY, 0666 | IPC_CREAT)) == -1) {
		g_print("msgget: %s\n", strerror(errno));
		return -1;
	}

	msgbuf.mtype = LOG_MTYPE;

	/* allow for the NUL */
	len = strlen(msgbuf.mtext) + 1;

	if (msgsnd(msqid, &msgbuf, len, IPC_NOWAIT) < 0) {
		g_print("msgsnd: %m\n");
		return -1;
	}

#if 0
	g_print("msg sent: (%02ld) '%s'\n", msgbuf.mtype, msgbuf.mtext);
#endif

	return 0;
}

/* ---------------------------------------------------------------------- */

void clear_qsodata(void)
{
	set_qsocall("");
	set_qsotxrst("");
	set_qsorxrst("");
	set_qsoname("");
	set_qsoqth("");
	set_qsonotes("");

	memset(&qso, 0, sizeof(struct qsodata));
}

void log_qsodata(void)
{
	char date[32], time[32], *mode;
	struct tm *tm;

	get_qsocall();
	get_qsoband();
	get_qsotxrst();
	get_qsorxrst();
	get_qsoname();
	get_qsoqth();
	get_qsonotes();

	g_strup(qso.call);

	tm = gmtime(&qso.time);
	strftime(date, sizeof(date), "%d %b %Y", tm);
	strftime(time, sizeof(time), "%H%M", tm);

	mode = trx_get_mode_name();

	log_msg_clear();
	log_msg_add("program",	"gMFSK v" VERSION);
	log_msg_add("version",	LOG_MVERSION);
	log_msg_add("date",	date);
	log_msg_add("time",	time);
	log_msg_add("call",	qso.call);
	log_msg_add("mhz",	qso.band);
	log_msg_add("mode",	mode);
	log_msg_add("tx",	qso.txrst);
	log_msg_add("rx",	qso.rxrst);
	log_msg_add("name",	qso.name);
	log_msg_add("qth",	qso.qth);
	log_msg_add("notes",	qso.notes);
	log_msg_send();

	clear_qsodata();
}

/* ---------------------------------------------------------------------- */

void set_qsotime(void)
{
	if (qso.time == 0) {
		time(&qso.time);
	}
}

/* ---------------------------------------------------------------------- */

static void set_entry(const gchar *name, gchar *str)
{
	gpointer p;

	p = gtk_object_get_data(GTK_OBJECT(appwindow), name);
	gtk_entry_set_text(GTK_ENTRY(p), str);	
}

void set_qsocall(char *str)
{
	set_entry("qsocallentry", str);
}

void set_qsoband(char *str)
{
	set_entry("qsobandentry", str);
}

void set_qsotxrst(char *str)
{
	set_entry("qsotxrstentry", str);
}

void set_qsorxrst(char *str)
{
	set_entry("qsorxrstentry", str);
}

void set_qsoname(char *str)
{
	set_entry("qsonameentry", str);
}

void set_qsoqth(char *str)
{
	set_entry("qsoqthentry", str);
}

void set_qsonotes(char *str)
{
	set_entry("qsonotesentry", str);
}

/* ---------------------------------------------------------------------- */

static gchar *get_entry(const gchar *name)
{
	gpointer p;

	p = gtk_object_get_data(GTK_OBJECT(appwindow), name);
	return gtk_entry_get_text(GTK_ENTRY(p));
}

#define safecopy(d,s)	{ strncpy((d),(s),sizeof(d)); (d)[sizeof(d)-1]=0; }

char *get_qsocall(void)
{
	gchar *p;

	p = get_entry("qsocallentry");
	safecopy(qso.call, p);
	return qso.call;
}

char *get_qsoband(void)
{
	gchar *p;

	p = get_entry("qsobandentry");
	safecopy(qso.band, p);
	return qso.band;
}

char *get_qsotxrst(void)
{
	gchar *p;

	p = get_entry("qsotxrstentry");
	safecopy(qso.txrst, p);
	return qso.txrst;
}

char *get_qsorxrst(void)
{
	gchar *p;

	p = get_entry("qsorxrstentry");
	safecopy(qso.rxrst, p);
	return qso.rxrst;
}

char *get_qsoname(void)
{
	gchar *p;

	p = get_entry("qsonameentry");
	safecopy(qso.name, p);
	return qso.name;
}

char *get_qsoqth(void)
{
	gchar *p;

	p = get_entry("qsoqthentry");
	safecopy(qso.qth, p);
	return qso.qth;
}

char *get_qsonotes(void)
{
	gchar *p;

	p = get_entry("qsonotesentry");
	safecopy(qso.notes, p);
	return qso.notes;
}

/* ---------------------------------------------------------------------- */

